/*
 * drivers/net/phy/marvell.c
 *
 * Driver for Marvell PHYs
 *
 * Author: Andy Fleming
 *
 * Copyright (c) 2004 Freescale Semiconductor, Inc.
 *
 * Copyright (c) 2013 Michael Stapelberg <michael@stapelberg.de>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/spinlock.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/mii.h>
#include <linux/mdio.h>
#include <linux/ethtool.h>
#include <linux/phy.h>
#include <linux/marvell_phy.h>
#include <linux/of.h>

#include <linux/io.h>
#include <asm/irq.h>
#include <linux/uaccess.h>
#include <linux/leds.h>

#define MII_MARVELL_PHY_PAGE		22

#define MII_M1011_IEVENT		0x13
#define MII_M1011_IEVENT_CLEAR		0x0000

#define MII_M1011_IMASK			0x12
#define MII_M1011_IMASK_INIT		0x6400
#define MII_M1011_IMASK_CLEAR		0x0000

#define MII_M1011_PHY_SCR		0x10
#define MII_M1011_PHY_SCR_AUTO_CROSS	0x0060

#define MII_M1145_PHY_EXT_SR		0x1b
#define MII_M1145_PHY_EXT_CR		0x14
#define MII_M1145_RGMII_RX_DELAY	0x0080
#define MII_M1145_RGMII_TX_DELAY	0x0002

#define MII_M1145_HWCFG_MODE_SGMII_NO_CLK	0x4
#define MII_M1145_HWCFG_MODE_MASK		0xf
#define MII_M1145_HWCFG_FIBER_COPPER_AUTO	0x8000

#define MII_M1111_PHY_LED_CONTROL	0x18
#define MII_M1111_PHY_LED_DIRECT	0x4100
#define MII_M1111_PHY_LED_COMBINE	0x411c
#define MII_M1111_PHY_EXT_CR		0x14
#define MII_M1111_RX_DELAY		0x80
#define MII_M1111_TX_DELAY		0x2
#define MII_M1111_PHY_EXT_SR		0x1b

#define MII_M1111_HWCFG_MODE_MASK		0xf
#define MII_M1111_HWCFG_MODE_COPPER_RGMII	0xb
#define MII_M1111_HWCFG_MODE_FIBER_RGMII	0x3
#define MII_M1111_HWCFG_MODE_SGMII_NO_CLK	0x4
#define MII_M1111_HWCFG_MODE_COPPER_RTBI	0x9
#define MII_M1111_HWCFG_FIBER_COPPER_AUTO	0x8000
#define MII_M1111_HWCFG_FIBER_COPPER_RES	0x2000

#define MII_M1111_COPPER		0
#define MII_M1111_FIBER			1

#define MII_88E1121_PHY_MSCR_PAGE	2
#define MII_88E1121_PHY_MSCR_REG	21
#define MII_88E1121_PHY_MSCR_RX_DELAY	BIT(5)
#define MII_88E1121_PHY_MSCR_TX_DELAY	BIT(4)
#define MII_88E1121_PHY_MSCR_DELAY_MASK	(~(0x3 << 4))

#define MII_88E1318S_PHY_MSCR1_REG	16
#define MII_88E1318S_PHY_MSCR1_PAD_ODD	BIT(6)

/* Copper Specific Interrupt Enable Register */
#define MII_88E1318S_PHY_CSIER                              0x12
/* WOL Event Interrupt Enable */
#define MII_88E1318S_PHY_CSIER_WOL_EIE                      BIT(7)

/* LED Timer Control Register */
#define MII_88E1318S_PHY_LED_PAGE                           0x03
#define MII_88E1318S_PHY_LED_TCR                            0x12
#define MII_88E1318S_PHY_LED_TCR_FORCE_INT                  BIT(15)
#define MII_88E1318S_PHY_LED_TCR_INTn_ENABLE                BIT(7)
#define MII_88E1318S_PHY_LED_TCR_INT_ACTIVE_LOW             BIT(11)

/* Magic Packet MAC address registers */
#define MII_88E1318S_PHY_MAGIC_PACKET_WORD2                 0x17
#define MII_88E1318S_PHY_MAGIC_PACKET_WORD1                 0x18
#define MII_88E1318S_PHY_MAGIC_PACKET_WORD0                 0x19

#define MII_88E1318S_PHY_WOL_PAGE                           0x11
#define MII_88E1318S_PHY_WOL_CTRL                           0x10
#define MII_88E1318S_PHY_WOL_CTRL_CLEAR_WOL_STATUS          BIT(12)
#define MII_88E1318S_PHY_WOL_CTRL_MAGIC_PACKET_MATCH_ENABLE BIT(14)

#define MII_88E1121_PHY_LED_CTRL	16
#define MII_88E1121_PHY_LED_PAGE	3
#define MII_88E1121_PHY_LED_DEF		0x0030

#define MII_M1011_PHY_STATUS		0x11
#define MII_M1011_PHY_STATUS_1000	0x8000
#define MII_M1011_PHY_STATUS_100	0x4000
#define MII_M1011_PHY_STATUS_SPD_MASK	0xc000
#define MII_M1011_PHY_STATUS_FULLDUPLEX	0x2000
#define MII_M1011_PHY_STATUS_RESOLVED	0x0800
#define MII_M1011_PHY_STATUS_LINK	0x0400

#define MII_M1116R_CONTROL_REG_MAC	21

#define MII_88E1510_CSCR1	16
#define MII_88E1510_CSCR1_DNSHIFT_CNT_MASK	0x7000
#define MII_88E1510_CSCR1_DNSHIFT_CNT_SHIFT	12
#define MII_88E1510_CSCR1_DNSHIFT_ENB		0x800

#define MII_88E1510_MSCR1	16
#define MII_88E1510_MSCR1_CLK125_DIS	BIT(1)

#define MII_88E1510_MSCR2	21
#define MII_88E1510_MSCR2_DEF_SPEED_MSB (1 << 6)
#define MII_88E1510_MSCR2_DEF_SPEED_LSB (1 << 13)

#define MII_88E1510_LED_FCR	16
#define MII_LFCR_LED3(x)	((x) << 12)
#define MII_LFCR_LED2(x)	((x) << 8)
#define MII_LFCR_LED1(x)	((x) << 4)
#define MII_LFCR_LED0(x)	((x) << 0)
#define MII_88E1510_LFCR_LED2_DISABLE	8
#define MII_88E1510_LFCR_LED2_FORCE_OFF	8
#define MII_88E1510_LFCR_LED2_LINK_10	6
#define MII_88E1510_LFCR_LED1_FORCE_OFF 8
#define MII_88E1510_LFCR_LED1_LINK_100	6
#define MII_88E1510_LFCR_LED1_ACT_BLINK	4
#define MII_88E1510_LFCR_LED0_FORCE_HIZ 10
#define MII_88E1510_LFCR_LED0_FORCE_OFF 8
#define MII_88E1510_LFCR_LED0_LINK		0

#define MII_88E1510_LED_PCR	17
#define MII_88E1510_LED_PCR_LED2_AL	(0 << 4)
#define MII_88E1510_LED_PCR_LED1_AL	(0 << 2)
#define MII_88E1510_LED_PCR_LED0_AL	(0 << 0)

#define MII_88E1510_LED_TCR	18
#define MII_88E1510_LED_TCR_PS_170MS	(4 << 12)
#define MII_88E1510_LED_TCR_BR_84MS		(1 << 8)

#define MII_88E1510_ROICO	24
#define MII_88E1510_ROICO_RESTART_CAL	BIT(15)
#define MII_88E1510_ROICO_CAL_DONE		BIT(14)

#define MII_88E1510_ROIT	25
#define MII_88E1510_ROIT_CAL_TARG_MASK		0x7
#define MII_88E1510_ROIT_CAL_TARG_54_6_OHM	0x2

#define MII_88E1510_COPPER_TEST_PAGE	6
#define MII_88E1510_CHECKER_CNTL		18
#define MII_88E1510_ENABLE_STUB_TEST	BIT(3)

#define MII_M3019_LED_PSR	22
#define MII_M3019_LED_PSR_LINK		5
#define MII_M3019_LED_PSR_ACT_BLINK	11

#define MII_M3019_PLCR	24
#define MII_M3019_PLCR_PS_170MS	(4 << 12)
#define MII_M3019_PLCR_BR_84MS	(1 << 9)
#define MII_M3019_PLCR_L1SS_100	(1 << 3)

#define MII_M3019_PMLO	25
#define MII_M3019_PMLO_LED0_ACTIVELOW	(0 << 12)
#define MII_M3019_PMLO_LED1_ACTIVELOW	(0 << 13)

#define MII_M3019_PHY_SCR2	28
#define MII_M3019_PHY_SCR2_CLASS_A	(1 << 0)

#define MII_M3019_TEST_MODE_SEL	29
#define MII_M3019_TEST_MODE_SEL_OUTPUT_IMP_CAL_OVERRIDE	0xA
#define MII_M3019_TEST_MODE_SEL_OUTPUT_IMP_TARGET		0xB

#define MII_M3019_OICO	30
#define MII_M3019_OICO_IMP_P_80_OHM	(0 << 4)
#define MII_M3019_OICO_IMP_N_80_OHM	0
#define MII_M3019_OICO_IMP_CALIBRATION_DONE		(1 << 14)
#define MII_M3019_OICO_IMP_RESTART_CALIBRATION	(1 << 15)

/* phyg28lp */
#define MII_phyg28lp_MSCR2_DEF_SPEED_MASK	(7 << 0)
#define MII_phyg28lp_MSCR2_DEF_SPEED		(4 << 0)
#define MII_phyg28lp_LFCR_LED_FORCE_OFF		8
#define MII_phyg28lp_LED_PCR_LED3_AL		(0 << 6)

MODULE_DESCRIPTION("Marvell PHY driver");
MODULE_AUTHOR("Andy Fleming");
MODULE_LICENSE("GPL");

static int marvell_ack_interrupt(struct phy_device *phydev)
{
	int err;

	/* Clear the interrupts by reading the reg */
	err = phy_read(phydev, MII_M1011_IEVENT);

	if (err < 0)
		return err;

	return 0;
}

static int marvell_config_intr(struct phy_device *phydev)
{
	int err;

	if (phydev->interrupts == PHY_INTERRUPT_ENABLED)
		err = phy_write(phydev, MII_M1011_IMASK, MII_M1011_IMASK_INIT);
	else
		err = phy_write(phydev, MII_M1011_IMASK, MII_M1011_IMASK_CLEAR);

	return err;
}

static int marvell_config_aneg(struct phy_device *phydev)
{
	int err;

	/* The Marvell PHY has an errata which requires
	 * that certain registers get written in order
	 * to restart autonegotiation */
	err = phy_write(phydev, MII_BMCR, BMCR_RESET);

	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1d, 0x1f);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1e, 0x200c);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1d, 0x5);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1e, 0);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1e, 0x100);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M1011_PHY_SCR,
			MII_M1011_PHY_SCR_AUTO_CROSS);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M1111_PHY_LED_CONTROL,
			MII_M1111_PHY_LED_DIRECT);
	if (err < 0)
		return err;

	err = genphy_config_aneg(phydev);
	if (err < 0)
		return err;

	if (phydev->autoneg != AUTONEG_ENABLE) {
		int bmcr;

		/*
		 * A write to speed/duplex bits (that is performed by
		 * genphy_config_aneg() call above) must be followed by
		 * a software reset. Otherwise, the write has no effect.
		 */
		bmcr = phy_read(phydev, MII_BMCR);
		if (bmcr < 0)
			return bmcr;

		err = phy_write(phydev, MII_BMCR, bmcr | BMCR_RESET);
		if (err < 0)
			return err;
	}

	return 0;
}

static ssize_t marvell_show_reg(struct device *dev, struct device_attribute *attr, char *buf) {
	struct phy_device *phydev = container_of(dev, struct phy_device, dev);
	unsigned int *reg = phydev->priv;
	u16 value = 0;

	value = phy_read(phydev, *reg);

	return snprintf(buf, PAGE_SIZE, "0x%04x\n", value);
}

static ssize_t marvell_store_reg(struct device *dev, struct device_attribute *attr, const char *buf, size_t count) {
	const char hex_chars[] = "0123456789xX";
	struct phy_device *phydev = container_of(dev, struct phy_device, dev);
	unsigned int *reg = phydev->priv;
	size_t first_char = 0;
	char *p_value = NULL;
	char *p_end = NULL;
	int err;
	u16 value = 0;

	first_char = strcspn(buf, hex_chars);
	*reg = (u16)simple_strtoul(&buf[first_char], &p_value, 0);

	/* buf contains garbage */
	if ((*reg == 0) && (&buf[first_char] == p_value))
		return count;

	first_char = strcspn(p_value, hex_chars);
	value = (u16)simple_strtoul(&p_value[first_char], &p_end, 0);

	/* value is specified so this is a write */
	if ((value != 0) || (&p_value[first_char] != p_end)) {
		err = phy_write(phydev, *reg, value);
		if (err < 0)
			return err;
	}

	return count;
}

static DEVICE_ATTR(reg, (S_IRUSR | S_IWUSR), marvell_show_reg, marvell_store_reg);

static int marvell_phy_probe(struct phy_device *phydev) {
	int err;

	if (!phydev->priv) {
		phydev->priv = devm_kzalloc(&phydev->dev, sizeof(unsigned int), GFP_KERNEL);
		if (!phydev->priv)
			return -ENOMEM;
	}

	err = device_create_file(&phydev->dev, &dev_attr_reg);
	if (err) {
		dev_err(&phydev->dev, "device_create_file failed. %d\n", err);
	}
	else {
		err = sysfs_create_link(&phydev->dev.parent->kobj, &phydev->dev.kobj, "phy");
		if (err)
			dev_err(&phydev->dev, "sysfs_create_link failed. %d\n", err);
	}

	return 0;
}

static void marvell_phy_remove(struct phy_device *phydev) {
	sysfs_remove_link(&phydev->dev.parent->kobj, "phy");
	device_remove_file(&phydev->dev, &dev_attr_reg);
}

#ifdef CONFIG_OF_MDIO
/*
 * Set and/or override some configuration registers based on the
 * marvell,reg-init property stored in the of_node for the phydev.
 *
 * marvell,reg-init = <reg-page reg mask value>,...;
 *
 * There may be one or more sets of <reg-page reg mask value>:
 *
 * reg-page: which register bank to use.
 * reg: the register.
 * mask: if non-zero, ANDed with existing register value.
 * value: ORed with the masked value and written to the regiser.
 *
 */
static int marvell_of_reg_init(struct phy_device *phydev)
{
	const __be32 *paddr;
	int len, i, saved_page, current_page, page_changed, ret;

	if (!phydev->dev.of_node)
		return 0;

	paddr = of_get_property(phydev->dev.of_node, "marvell,reg-init", &len);

	if (!paddr || len < (4 * sizeof(*paddr)))
		return 0;

	saved_page = phy_read(phydev, MII_MARVELL_PHY_PAGE);
	if (saved_page < 0)
		return saved_page;
	page_changed = 0;
	current_page = saved_page;

	ret = 0;
	len /= sizeof(*paddr);
	for (i = 0; i < len - 3; i += 4) {
		u16 reg_page = be32_to_cpup(paddr + i);
		u16 reg = be32_to_cpup(paddr + i + 1);
		u16 mask = be32_to_cpup(paddr + i + 2);
		u16 val_bits = be32_to_cpup(paddr + i + 3);
		int val;

		if (reg_page != current_page) {
			current_page = reg_page;
			page_changed = 1;
			ret = phy_write(phydev, MII_MARVELL_PHY_PAGE, reg_page);
			if (ret < 0)
				goto err;
		}

		val = 0;
		if (mask) {
			val = phy_read(phydev, reg);
			if (val < 0) {
				ret = val;
				goto err;
			}
			val &= mask;
		}
		val |= val_bits;

		ret = phy_write(phydev, reg, val);
		if (ret < 0)
			goto err;

	}
err:
	if (page_changed) {
		i = phy_write(phydev, MII_MARVELL_PHY_PAGE, saved_page);
		if (ret == 0)
			ret = i;
	}
	return ret;
}
#else
static int marvell_of_reg_init(struct phy_device *phydev)
{
	return 0;
}
#endif /* CONFIG_OF_MDIO */

static int m88e1121_config_aneg(struct phy_device *phydev)
{
	int err, oldpage, mscr;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE,
			MII_88E1121_PHY_MSCR_PAGE);
	if (err < 0)
		return err;

	if ((phydev->interface == PHY_INTERFACE_MODE_RGMII) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)) {

		mscr = phy_read(phydev, MII_88E1121_PHY_MSCR_REG) &
			MII_88E1121_PHY_MSCR_DELAY_MASK;

		if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID)
			mscr |= (MII_88E1121_PHY_MSCR_RX_DELAY |
				 MII_88E1121_PHY_MSCR_TX_DELAY);
		else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID)
			mscr |= MII_88E1121_PHY_MSCR_RX_DELAY;
		else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)
			mscr |= MII_88E1121_PHY_MSCR_TX_DELAY;

		err = phy_write(phydev, MII_88E1121_PHY_MSCR_REG, mscr);
		if (err < 0)
			return err;
	}

	phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);

	err = phy_write(phydev, MII_BMCR, BMCR_RESET);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M1011_PHY_SCR,
			MII_M1011_PHY_SCR_AUTO_CROSS);
	if (err < 0)
		return err;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	phy_write(phydev, MII_MARVELL_PHY_PAGE, MII_88E1121_PHY_LED_PAGE);
	phy_write(phydev, MII_88E1121_PHY_LED_CTRL, MII_88E1121_PHY_LED_DEF);
	phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);

	err = genphy_config_aneg(phydev);

	return err;
}

static int m88e1318_config_aneg(struct phy_device *phydev)
{
	int err, oldpage, mscr;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE,
			MII_88E1121_PHY_MSCR_PAGE);
	if (err < 0)
		return err;

	mscr = phy_read(phydev, MII_88E1318S_PHY_MSCR1_REG);
	mscr |= MII_88E1318S_PHY_MSCR1_PAD_ODD;

	err = phy_write(phydev, MII_88E1318S_PHY_MSCR1_REG, mscr);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return m88e1121_config_aneg(phydev);
}


static int m88e1510_phy_reset(struct phy_device *phydev)
{
	int err, oldpage, val;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	/* Reset the PHY */
	val = phy_read(phydev, MII_BMCR);
	val |= BMCR_RESET;

	err = phy_write(phydev, MII_BMCR, val);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_BMCR);
	while (val & BMCR_RESET) {
		msleep(1);
		val = phy_read(phydev, MII_BMCR);
	}

	/* Work around an errata */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0xfa);
	if (err < 0)
		return err;

	err = phy_write(phydev, 25, 0);
	if (err < 0)
		return err;

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return 0;
}

static int m88e1510_init_leds(struct phy_device *phydev)
{
	int err, oldpage, val;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 3);
	if (err < 0)
		return err;

	val = MII_LFCR_LED2(MII_88E1510_LFCR_LED2_LINK_10);
	val |= MII_LFCR_LED1(MII_88E1510_LFCR_LED1_LINK_100);
	val |= MII_LFCR_LED0(MII_88E1510_LFCR_LED0_FORCE_HIZ);
	err = phy_write(phydev, MII_88E1510_LED_FCR, val);
	if (err < 0)
		return err;

	val = MII_88E1510_LED_PCR_LED2_AL;
	val |= MII_88E1510_LED_PCR_LED1_AL;
	val |= MII_88E1510_LED_PCR_LED0_AL;
	err = phy_write(phydev, MII_88E1510_LED_PCR, val);
	if (err < 0)
		return err;

	val = MII_88E1510_LED_TCR_PS_170MS;
	val |= MII_88E1510_LED_TCR_BR_84MS;
	err = phy_write(phydev, MII_88E1510_LED_TCR, val);
	if (err < 0)
		return err;

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return 0;
}

static int m88e1510_config_init(struct phy_device *phydev)
{
	int err, oldpage, val;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	/* 2.3.2.9.5.1 Additional PHY initialization */
	phy_write(phydev, MII_MARVELL_PHY_PAGE, 0xff);
	phy_write(phydev, 0x11, 0x214b);
	phy_write(phydev, 0x10, 0x2144);
	phy_write(phydev, 0x11, 0x0c28);
	phy_write(phydev, 0x10, 0x2146);
	phy_write(phydev, 0x11, 0xb233);
	phy_write(phydev, 0x10, 0x214d);
	phy_write(phydev, 0x11, 0xcc0c);
	phy_write(phydev, 0x10, 0x2159);
	phy_write(phydev, 0x16, 0xfb);
	phy_write(phydev, 0x07, 0xc00d);

	/* Adjust link pulse */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0xfc);
	if (err < 0)
		return err;

	val = phy_read(phydev, 1);
	val |= 0x8000;

	err = phy_write(phydev, 1, val);
	if (err < 0)
		return err;

	/* Disable CLK125 Output */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 2);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_88E1510_MSCR1);
	val |= MII_88E1510_MSCR1_CLK125_DIS;

	err = phy_write(phydev, MII_88E1510_MSCR1, val);
	if (err < 0)
		return err;

	/* Set the target output impedence */
	val = phy_read(phydev, MII_88E1510_ROIT);
	val &= ~MII_88E1510_ROIT_CAL_TARG_MASK;
	val |= MII_88E1510_ROIT_CAL_TARG_54_6_OHM;

	err = phy_write(phydev, MII_88E1510_ROIT, val);
	if (err < 0)
		return err;

	val = MII_88E1510_ROICO_RESTART_CAL;
	err = phy_write(phydev, MII_88E1510_ROICO, val);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_88E1510_ROICO);
	while (!(val & MII_88E1510_ROICO_CAL_DONE)) {
		msleep(1);
		val = phy_read(phydev, MII_88E1510_ROICO);
	}

	/* Setup RX and TX delays  and default MAC speed */
	val = phy_read(phydev, MII_88E1510_MSCR2);

	if ((phydev->interface == PHY_INTERFACE_MODE_RGMII) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)) {

		val &= MII_88E1121_PHY_MSCR_DELAY_MASK;

		if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID)
			val |= (MII_88E1121_PHY_MSCR_RX_DELAY |
				 MII_88E1121_PHY_MSCR_TX_DELAY);
		else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID)
			val |= MII_88E1121_PHY_MSCR_RX_DELAY;
		else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)
			val |= MII_88E1121_PHY_MSCR_TX_DELAY;
	}

	val &= ~(MII_88E1510_MSCR2_DEF_SPEED_MSB | MII_88E1510_MSCR2_DEF_SPEED_LSB);

	err = phy_write(phydev, MII_88E1510_MSCR2, val);
	if (err < 0)
		return err;

	/* Enable Downshifting from 1Gbs */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_88E1510_CSCR1);
	val &= ~MII_88E1510_CSCR1_DNSHIFT_CNT_MASK;
	val |= (7 << MII_88E1510_CSCR1_DNSHIFT_CNT_SHIFT);
	val |= MII_88E1510_CSCR1_DNSHIFT_ENB;

	err = phy_write(phydev, MII_88E1510_CSCR1, val);
	if (err < 0)
		return err;

	/* 2.3.2.9.4 Enabling 802.az EEE */
	phy_write_mmd_indirect(phydev, MDIO_AN_EEE_ADV, MDIO_MMD_AN, phydev->addr, (MDIO_EEE_100TX | MDIO_EEE_1000T));
#if 0
	/* Don't enable EEE Buffer. You must use slave mode therefore the MAC of 6270 supports EEE. */
	phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x12);
	val = phy_read(phydev, 0x00);
	phy_write(phydev, 0x00, val | 0x01);
#endif

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	m88e1510_phy_reset(phydev);

	return 0;
}

static int m88e1510_config_aneg(struct phy_device *phydev)
{
	int err;

	err = m88e1318_config_aneg(phydev);
	if (err < 0)
		return err;

	m88e1510_init_leds(phydev);

	return marvell_of_reg_init(phydev);
}

void m88e1510_led_set(struct led_classdev *led_cdev, enum led_brightness brightness) {
	struct phy_device *phydev = container_of(led_cdev->dev->parent, struct phy_device, dev);
	int err, oldpage, ret, cnt;
	u16 val;

	dev_err(&phydev->dev, "led_set start\n");
	
	for(cnt=0; cnt<100; cnt++)
	{
		ret = mutex_trylock(&phydev->lock);
		if (ret != 0) break;
		
		msleep(1);
	}
	
	if (cnt>=100)
	{
		dev_err(&phydev->dev, "led_set failed : can't get mutex\n");
		return;
	}
	
	dev_err(&phydev->dev, "led_set (a)\n");
	
	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);
	
	dev_err(&phydev->dev, "led_set (b)\n");

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, MII_88E1318S_PHY_LED_PAGE);
	if (err < 0)
	{
		dev_err(&phydev->dev, "led_set failed : phy wr err\n");
		goto err;
	}

	dev_err(&phydev->dev, "led_set (c)\n");
	
	val = phy_read(phydev, MII_88E1510_LED_FCR);
	val &= ~MII_LFCR_LED0(0xf);
	val |= MII_LFCR_LED0(MII_88E1510_LFCR_LED0_FORCE_HIZ);

	dev_err(&phydev->dev, "led_set (d)\n");
	
	if (strstr(led_cdev->name, "activity")) {
		val &= ~MII_LFCR_LED1(0xf);

		if (brightness == 0 /*LED_OFF*/)
			val |= MII_LFCR_LED1(MII_88E1510_LFCR_LED1_FORCE_OFF);
		else
			val |= MII_LFCR_LED1(MII_88E1510_LFCR_LED1_LINK_100);
	}
	else {
		val &= ~MII_LFCR_LED2(0xf);

		if (brightness == 0 /*LED_OFF*/)
			val |= MII_LFCR_LED2(MII_88E1510_LFCR_LED2_FORCE_OFF);
		else
			val |= MII_LFCR_LED2(MII_88E1510_LFCR_LED2_LINK_10);
	}

	err = phy_write(phydev, MII_88E1510_LED_FCR, val);
	if (err < 0)
	{
		dev_err(&phydev->dev, "led_set failed : phy wr err\n");
		goto err;
	}
	
	dev_err(&phydev->dev, "led_set (e)\n");

	phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	
	dev_err(&phydev->dev, "led_set (f)\n");
	
err:
	mutex_unlock(&phydev->lock);
}

struct m88e1510_priv {
	unsigned int reg;
	unsigned int loopback;
	char led_link_name[16];
	char led_activity_name[16];
	struct led_classdev led_link;
	struct led_classdev led_activity;
};

static ssize_t m88e1510_loopback_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct phy_device *phydev = container_of(dev, struct phy_device, dev);
	struct m88e1510_priv *priv = phydev->priv;

	return snprintf(buf, PAGE_SIZE, "%d\n", priv->loopback);
};

static int m88e1510_loopback_enable(struct phy_device *phydev, int enable)
{
	int err, oldpage;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, MII_88E1510_COPPER_TEST_PAGE);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_88E1510_CHECKER_CNTL, enable ? MII_88E1510_ENABLE_STUB_TEST : 0);
	if (err < 0)
		return err;

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return 0;
}

static ssize_t m88e1510_loopback_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct phy_device *phydev = container_of(dev, struct phy_device, dev);
	struct m88e1510_priv *priv = phydev->priv;

	priv->loopback = !!(int)simple_strtol(buf, NULL, 0);

	m88e1510_loopback_enable(phydev, priv->loopback);

	return count;
}

DEVICE_ATTR(loopback, (S_IRUGO | S_IWUSR), m88e1510_loopback_show, m88e1510_loopback_store);

static int m88e1510_phy_probe(struct phy_device *phydev) {
	int err;
	struct m88e1510_priv *priv;

	phydev->priv = priv = devm_kzalloc(&phydev->dev, sizeof(*priv), GFP_KERNEL);
	if (priv == NULL)
		return -ENOMEM;

	err = marvell_phy_probe(phydev);
	if (err)
		return err;

	snprintf(priv->led_link_name, sizeof(priv->led_link_name), "link");
	priv->led_link.name = priv->led_link_name;
	priv->led_link.brightness = 1;
	priv->led_link.brightness_set = m88e1510_led_set;
	err = led_classdev_register(&phydev->dev, &priv->led_link);
	if (err)
		dev_err(&phydev->dev, "led_classdev_register failed rc = %d for %s\n", err, priv->led_link_name);

	snprintf(priv->led_activity_name, sizeof(priv->led_activity_name), "activity");
	priv->led_activity.name = priv->led_activity_name;
	priv->led_activity.brightness = 1;
	priv->led_activity.brightness_set = m88e1510_led_set;
	err = led_classdev_register(&phydev->dev, &priv->led_activity);
	if (err)
		dev_err(&phydev->dev, "led_classdev_register failed rc = %d for %s\n", err, priv->led_activity_name);

	if (device_create_file(&phydev->dev, &dev_attr_loopback))
		dev_err(&phydev->dev, "device_create_file failed");

	m88e1510_init_leds(phydev);

	return 0;
}

static void m88e1510_phy_remove(struct phy_device *phydev) {
	struct m88e1510_priv *priv = phydev->priv;

	device_remove_file(&phydev->dev, &dev_attr_loopback);
	led_classdev_unregister(&priv->led_link);
	led_classdev_unregister(&priv->led_activity);

	marvell_phy_remove(phydev);
}

static int phyg28lp_config_init(struct phy_device *phydev)
{
	int err, oldpage, val;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	/* Set default MAC speed */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 2);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_88E1510_MSCR2);
	val &= ~MII_phyg28lp_MSCR2_DEF_SPEED_MASK;
	val |= MII_phyg28lp_MSCR2_DEF_SPEED;

	err = phy_write(phydev, MII_88E1510_MSCR2, val);
	if (err < 0)
		return err;

	/* Enable Downshifting from 1Gbs */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_88E1510_CSCR1);
	val &= ~MII_88E1510_CSCR1_DNSHIFT_CNT_MASK;
	val |= (7 << MII_88E1510_CSCR1_DNSHIFT_CNT_SHIFT);
	val |= MII_88E1510_CSCR1_DNSHIFT_ENB;

	err = phy_write(phydev, MII_88E1510_CSCR1, val);
	if (err < 0)
		return err;

	/*enable 100/1000BaseTx EEE*/
	val = phy_read_mmd_indirect(phydev, MDIO_AN_EEE_ADV,
			MDIO_MMD_AN, phydev->addr);
	val |= MDIO_EEE_100TX;
	val |= MDIO_EEE_1000T;
	phy_write_mmd_indirect(phydev, MDIO_AN_EEE_ADV,
			MDIO_MMD_AN, phydev->addr, val);

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	m88e1510_phy_reset(phydev);

	return 0;
}

static int phyg28lp_config_aneg(struct phy_device *phydev)
{
	int err, oldpage, val;

	err = m88e1318_config_aneg(phydev);
	if (err < 0)
		return err;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 3);
	if (err < 0)
		return err;

	val = (MII_LFCR_LED3(MII_phyg28lp_LFCR_LED_FORCE_OFF) |
			MII_LFCR_LED2(MII_88E1510_LFCR_LED0_LINK) |
			MII_LFCR_LED1(MII_phyg28lp_LFCR_LED_FORCE_OFF) |
			MII_LFCR_LED0(MII_88E1510_LFCR_LED1_ACT_BLINK));
	err = phy_write(phydev, MII_88E1510_LED_FCR, val);
	if (err < 0)
		return err;

	val = (MII_phyg28lp_LED_PCR_LED3_AL |
			MII_88E1510_LED_PCR_LED2_AL |
			MII_88E1510_LED_PCR_LED1_AL |
			MII_88E1510_LED_PCR_LED0_AL);
	err = phy_write(phydev, MII_88E1510_LED_PCR, val);
	if (err < 0)
		return err;

	val = (MII_88E1510_LED_TCR_PS_170MS | MII_88E1510_LED_TCR_BR_84MS);
	err = phy_write(phydev, MII_88E1510_LED_TCR, val);
	if (err < 0)
		return err;

	/* Return to the previous page */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return 0;
}

static int m88e3019_config_init(struct phy_device *phydev)
{
	int err, val;

	/* Setup the LEDs */
	val = MII_LFCR_LED1(MII_M3019_LED_PSR_ACT_BLINK) | MII_LFCR_LED0(MII_M3019_LED_PSR_LINK);
	err = phy_write(phydev, MII_M3019_LED_PSR, val);
	if (err < 0)
		return err;

	val = MII_M3019_PLCR_PS_170MS | MII_M3019_PLCR_BR_84MS | MII_M3019_PLCR_L1SS_100;
	err = phy_write(phydev, MII_M3019_PLCR, val);
	if (err < 0)
		return err;

	val = MII_M3019_PMLO_LED0_ACTIVELOW | MII_M3019_PMLO_LED1_ACTIVELOW;
	err = phy_write(phydev, MII_M3019_PMLO, val);
	if (err < 0)
		return err;

	/* Force MDI mode */
	val = phy_read(phydev, MII_M1011_PHY_SCR);
	val &= 0xffcf;
	err = phy_write(phydev, MII_M1011_PHY_SCR, val);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_M3019_PHY_SCR2);
	val |= MII_M3019_PHY_SCR2_CLASS_A;
	err = phy_write(phydev, MII_M3019_PHY_SCR2, val);
	if (err < 0)
		return err;

	/* Set the target output impedence */
	err = phy_write(phydev, MII_M3019_TEST_MODE_SEL, MII_M3019_TEST_MODE_SEL_OUTPUT_IMP_TARGET);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M3019_OICO, (MII_M3019_OICO_IMP_P_80_OHM | MII_M3019_OICO_IMP_N_80_OHM));
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M3019_TEST_MODE_SEL, MII_M3019_TEST_MODE_SEL_OUTPUT_IMP_CAL_OVERRIDE);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M3019_OICO, MII_M3019_OICO_IMP_RESTART_CALIBRATION);
	if (err < 0)
		return err;

	val = phy_read(phydev, MII_M3019_OICO);
	while (!(val & MII_M3019_OICO_IMP_CALIBRATION_DONE)) {
		msleep(1);
		val = phy_read(phydev, MII_M3019_OICO);
	}

	/* Reset the PHY */
	val = phy_read(phydev, MII_BMCR);
	val |= BMCR_RESET;

	err = phy_write(phydev, MII_BMCR, val);
	if (err < 0)
		return err;

	return 0;
}

static int m88e1116r_config_init(struct phy_device *phydev)
{
	int temp;
	int err;

	temp = phy_read(phydev, MII_BMCR);
	temp |= BMCR_RESET;
	err = phy_write(phydev, MII_BMCR, temp);
	if (err < 0)
		return err;

	mdelay(500);

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0);
	if (err < 0)
		return err;

	temp = phy_read(phydev, MII_M1011_PHY_SCR);
	temp |= (7 << 12);	/* max number of gigabit attempts */
	temp |= (1 << 11);	/* enable downshift */
	temp |= MII_M1011_PHY_SCR_AUTO_CROSS;
	err = phy_write(phydev, MII_M1011_PHY_SCR, temp);
	if (err < 0)
		return err;

	/* Setup RX and TX delays */
	if ((phydev->interface == PHY_INTERFACE_MODE_RGMII) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID) ||
		(phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)) {
		err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 2);
		if (err < 0)
			return err;
		temp = phy_read(phydev, MII_M1116R_CONTROL_REG_MAC);
		temp |= (1 << 5);
		temp |= (1 << 4);
		err = phy_write(phydev, MII_M1116R_CONTROL_REG_MAC, temp);
		if (err < 0)
			return err;
		err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0);
		if (err < 0)
			return err;
	}

	temp = phy_read(phydev, MII_BMCR);
	temp |= BMCR_RESET;
	err = phy_write(phydev, MII_BMCR, temp);
	if (err < 0)
		return err;

	mdelay(500);

	return 0;
}

static int m88e1111_config_init(struct phy_device *phydev)
{
	int err;
	int temp;

	if ((phydev->interface == PHY_INTERFACE_MODE_RGMII) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID) ||
	    (phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)) {

		temp = phy_read(phydev, MII_M1111_PHY_EXT_CR);
		if (temp < 0)
			return temp;

		if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) {
			temp |= (MII_M1111_RX_DELAY | MII_M1111_TX_DELAY);
		} else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID) {
			temp &= ~MII_M1111_TX_DELAY;
			temp |= MII_M1111_RX_DELAY;
		} else if (phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID) {
			temp &= ~MII_M1111_RX_DELAY;
			temp |= MII_M1111_TX_DELAY;
		}

		err = phy_write(phydev, MII_M1111_PHY_EXT_CR, temp);
		if (err < 0)
			return err;

		temp = phy_read(phydev, MII_M1111_PHY_EXT_SR);
		if (temp < 0)
			return temp;

		temp &= ~(MII_M1111_HWCFG_MODE_MASK);

		if (temp & MII_M1111_HWCFG_FIBER_COPPER_RES)
			temp |= MII_M1111_HWCFG_MODE_FIBER_RGMII;
		else
			temp |= MII_M1111_HWCFG_MODE_COPPER_RGMII;

		err = phy_write(phydev, MII_M1111_PHY_EXT_SR, temp);
		if (err < 0)
			return err;
	}

	if (phydev->interface == PHY_INTERFACE_MODE_SGMII) {
		temp = phy_read(phydev, MII_M1111_PHY_EXT_SR);
		if (temp < 0)
			return temp;

		temp &= ~(MII_M1111_HWCFG_MODE_MASK);
		temp |= MII_M1111_HWCFG_MODE_SGMII_NO_CLK;
		temp |= MII_M1111_HWCFG_FIBER_COPPER_AUTO;

		err = phy_write(phydev, MII_M1111_PHY_EXT_SR, temp);
		if (err < 0)
			return err;
	}

	if (phydev->interface == PHY_INTERFACE_MODE_RTBI) {
		temp = phy_read(phydev, MII_M1111_PHY_EXT_CR);
		if (temp < 0)
			return temp;
		temp |= (MII_M1111_RX_DELAY | MII_M1111_TX_DELAY);
		err = phy_write(phydev, MII_M1111_PHY_EXT_CR, temp);
		if (err < 0)
			return err;

		temp = phy_read(phydev, MII_M1111_PHY_EXT_SR);
		if (temp < 0)
			return temp;
		temp &= ~(MII_M1111_HWCFG_MODE_MASK | MII_M1111_HWCFG_FIBER_COPPER_RES);
		temp |= 0x7 | MII_M1111_HWCFG_FIBER_COPPER_AUTO;
		err = phy_write(phydev, MII_M1111_PHY_EXT_SR, temp);
		if (err < 0)
			return err;

		/* soft reset */
		err = phy_write(phydev, MII_BMCR, BMCR_RESET);
		if (err < 0)
			return err;
		do
			temp = phy_read(phydev, MII_BMCR);
		while (temp & BMCR_RESET);

		temp = phy_read(phydev, MII_M1111_PHY_EXT_SR);
		if (temp < 0)
			return temp;
		temp &= ~(MII_M1111_HWCFG_MODE_MASK | MII_M1111_HWCFG_FIBER_COPPER_RES);
		temp |= MII_M1111_HWCFG_MODE_COPPER_RTBI | MII_M1111_HWCFG_FIBER_COPPER_AUTO;
		err = phy_write(phydev, MII_M1111_PHY_EXT_SR, temp);
		if (err < 0)
			return err;
	}

	err = marvell_of_reg_init(phydev);
	if (err < 0)
		return err;

	return phy_write(phydev, MII_BMCR, BMCR_RESET);
}

static int m88e1118_config_aneg(struct phy_device *phydev)
{
	int err;

	err = phy_write(phydev, MII_BMCR, BMCR_RESET);
	if (err < 0)
		return err;

	err = phy_write(phydev, MII_M1011_PHY_SCR,
			MII_M1011_PHY_SCR_AUTO_CROSS);
	if (err < 0)
		return err;

	err = genphy_config_aneg(phydev);
	return 0;
}

static int m88e1118_config_init(struct phy_device *phydev)
{
	int err;

	/* Change address */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x0002);
	if (err < 0)
		return err;

	/* Enable 1000 Mbit */
	err = phy_write(phydev, 0x15, 0x1070);
	if (err < 0)
		return err;

	/* Change address */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x0003);
	if (err < 0)
		return err;

	/* Adjust LED Control */
	if (phydev->dev_flags & MARVELL_PHY_M1118_DNS323_LEDS)
		err = phy_write(phydev, 0x10, 0x1100);
	else
		err = phy_write(phydev, 0x10, 0x021e);
	if (err < 0)
		return err;

	err = marvell_of_reg_init(phydev);
	if (err < 0)
		return err;

	/* Reset address */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x0);
	if (err < 0)
		return err;

	return phy_write(phydev, MII_BMCR, BMCR_RESET);
}

static int m88e1149_config_init(struct phy_device *phydev)
{
	int err;

	/* Change address */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x0002);
	if (err < 0)
		return err;

	/* Enable 1000 Mbit */
	err = phy_write(phydev, 0x15, 0x1048);
	if (err < 0)
		return err;

	err = marvell_of_reg_init(phydev);
	if (err < 0)
		return err;

	/* Reset address */
	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x0);
	if (err < 0)
		return err;

	return phy_write(phydev, MII_BMCR, BMCR_RESET);
}

static int m88e1145_config_init(struct phy_device *phydev)
{
	int err;

	/* Take care of errata E0 & E1 */
	err = phy_write(phydev, 0x1d, 0x001b);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1e, 0x418f);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1d, 0x0016);
	if (err < 0)
		return err;

	err = phy_write(phydev, 0x1e, 0xa2da);
	if (err < 0)
		return err;

	if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID) {
		int temp = phy_read(phydev, MII_M1145_PHY_EXT_CR);
		if (temp < 0)
			return temp;

		temp |= (MII_M1145_RGMII_RX_DELAY | MII_M1145_RGMII_TX_DELAY);

		err = phy_write(phydev, MII_M1145_PHY_EXT_CR, temp);
		if (err < 0)
			return err;

		if (phydev->dev_flags & MARVELL_PHY_M1145_FLAGS_RESISTANCE) {
			err = phy_write(phydev, 0x1d, 0x0012);
			if (err < 0)
				return err;

			temp = phy_read(phydev, 0x1e);
			if (temp < 0)
				return temp;

			temp &= 0xf03f;
			temp |= 2 << 9;	/* 36 ohm */
			temp |= 2 << 6;	/* 39 ohm */

			err = phy_write(phydev, 0x1e, temp);
			if (err < 0)
				return err;

			err = phy_write(phydev, 0x1d, 0x3);
			if (err < 0)
				return err;

			err = phy_write(phydev, 0x1e, 0x8000);
			if (err < 0)
				return err;
		}
	}

	if (phydev->interface == PHY_INTERFACE_MODE_SGMII) {
		int temp = phy_read(phydev, MII_M1145_PHY_EXT_SR);
		if (temp < 0)
			return temp;

		temp &= ~MII_M1145_HWCFG_MODE_MASK;
		temp |= MII_M1145_HWCFG_MODE_SGMII_NO_CLK;
		temp |= MII_M1145_HWCFG_FIBER_COPPER_AUTO;

		err = phy_write(phydev, MII_M1145_PHY_EXT_SR, temp);
		if (err < 0)
			return err;
	}

	err = marvell_of_reg_init(phydev);
	if (err < 0)
		return err;

	return 0;
}

/* marvell_read_status
 *
 * Generic status code does not detect Fiber correctly!
 * Description:
 *   Check the link, then figure out the current state
 *   by comparing what we advertise with what the link partner
 *   advertises.  Start by checking the gigabit possibilities,
 *   then move on to 10/100.
 */
static int marvell_read_status(struct phy_device *phydev)
{
	int adv;
	int err;
	int lpa;
	int status = 0;

	/* Update the link, but return if there
	 * was an error */
	err = genphy_update_link(phydev);
	if (err)
		return err;

	if (AUTONEG_ENABLE == phydev->autoneg) {
		status = phy_read(phydev, MII_M1011_PHY_STATUS);
		if (status < 0)
			return status;

		lpa = phy_read(phydev, MII_LPA);
		if (lpa < 0)
			return lpa;

		adv = phy_read(phydev, MII_ADVERTISE);
		if (adv < 0)
			return adv;

		lpa &= adv;

		if (status & MII_M1011_PHY_STATUS_FULLDUPLEX)
			phydev->duplex = DUPLEX_FULL;
		else
			phydev->duplex = DUPLEX_HALF;

		status = status & MII_M1011_PHY_STATUS_SPD_MASK;
		phydev->pause = phydev->asym_pause = 0;

		switch (status) {
		case MII_M1011_PHY_STATUS_1000:
			phydev->speed = SPEED_1000;
			break;

		case MII_M1011_PHY_STATUS_100:
			phydev->speed = SPEED_100;
			break;

		default:
			phydev->speed = SPEED_10;
			break;
		}

		if (phydev->duplex == DUPLEX_FULL) {
			phydev->pause = lpa & LPA_PAUSE_CAP ? 1 : 0;
			phydev->asym_pause = lpa & LPA_PAUSE_ASYM ? 1 : 0;
		}
	} else {
		int bmcr = phy_read(phydev, MII_BMCR);

		if (bmcr < 0)
			return bmcr;

		if (bmcr & BMCR_FULLDPLX)
			phydev->duplex = DUPLEX_FULL;
		else
			phydev->duplex = DUPLEX_HALF;

		if (bmcr & BMCR_SPEED1000)
			phydev->speed = SPEED_1000;
		else if (bmcr & BMCR_SPEED100)
			phydev->speed = SPEED_100;
		else
			phydev->speed = SPEED_10;

		phydev->pause = phydev->asym_pause = 0;
	}

	return 0;
}

static int m88e1121_did_interrupt(struct phy_device *phydev)
{
	int imask;

	imask = phy_read(phydev, MII_M1011_IEVENT);

	if (imask & MII_M1011_IMASK_INIT)
		return 1;

	return 0;
}

static void m88e1318_get_wol(struct phy_device *phydev, struct ethtool_wolinfo *wol)
{
	wol->supported = WAKE_MAGIC;
	wol->wolopts = 0;

	if (phy_write(phydev, MII_MARVELL_PHY_PAGE,
		      MII_88E1318S_PHY_WOL_PAGE) < 0)
		return;

	if (phy_read(phydev, MII_88E1318S_PHY_WOL_CTRL) &
	    MII_88E1318S_PHY_WOL_CTRL_MAGIC_PACKET_MATCH_ENABLE)
		wol->wolopts |= WAKE_MAGIC;

	if (phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x00) < 0)
		return;
}

static int m88e1318_set_wol(struct phy_device *phydev, struct ethtool_wolinfo *wol)
{
	int err, oldpage, temp;

	oldpage = phy_read(phydev, MII_MARVELL_PHY_PAGE);

	if (wol->wolopts & WAKE_MAGIC) {
		/* Explicitly switch to page 0x00, just to be sure */
		err = phy_write(phydev, MII_MARVELL_PHY_PAGE, 0x00);
		if (err < 0)
			return err;

		/* Enable the WOL interrupt */
		temp = phy_read(phydev, MII_88E1318S_PHY_CSIER);
		temp |= MII_88E1318S_PHY_CSIER_WOL_EIE;
		err = phy_write(phydev, MII_88E1318S_PHY_CSIER, temp);
		if (err < 0)
			return err;

		err = phy_write(phydev, MII_MARVELL_PHY_PAGE,
				MII_88E1318S_PHY_LED_PAGE);
		if (err < 0)
			return err;

		/* Setup LED[2] as interrupt pin (active low) */
		temp = phy_read(phydev, MII_88E1318S_PHY_LED_TCR);
		temp &= ~MII_88E1318S_PHY_LED_TCR_FORCE_INT;
		temp |= MII_88E1318S_PHY_LED_TCR_INTn_ENABLE;
		temp |= MII_88E1318S_PHY_LED_TCR_INT_ACTIVE_LOW;
		err = phy_write(phydev, MII_88E1318S_PHY_LED_TCR, temp);
		if (err < 0)
			return err;

		err = phy_write(phydev, MII_MARVELL_PHY_PAGE,
				MII_88E1318S_PHY_WOL_PAGE);
		if (err < 0)
			return err;

		/* Store the device address for the magic packet */
		err = phy_write(phydev, MII_88E1318S_PHY_MAGIC_PACKET_WORD2,
				((phydev->attached_dev->dev_addr[5] << 8) |
				 phydev->attached_dev->dev_addr[4]));
		if (err < 0)
			return err;
		err = phy_write(phydev, MII_88E1318S_PHY_MAGIC_PACKET_WORD1,
				((phydev->attached_dev->dev_addr[3] << 8) |
				 phydev->attached_dev->dev_addr[2]));
		if (err < 0)
			return err;
		err = phy_write(phydev, MII_88E1318S_PHY_MAGIC_PACKET_WORD0,
				((phydev->attached_dev->dev_addr[1] << 8) |
				 phydev->attached_dev->dev_addr[0]));
		if (err < 0)
			return err;

		/* Clear WOL status and enable magic packet matching */
		temp = phy_read(phydev, MII_88E1318S_PHY_WOL_CTRL);
		temp |= MII_88E1318S_PHY_WOL_CTRL_CLEAR_WOL_STATUS;
		temp |= MII_88E1318S_PHY_WOL_CTRL_MAGIC_PACKET_MATCH_ENABLE;
		err = phy_write(phydev, MII_88E1318S_PHY_WOL_CTRL, temp);
		if (err < 0)
			return err;
	} else {
		err = phy_write(phydev, MII_MARVELL_PHY_PAGE,
				MII_88E1318S_PHY_WOL_PAGE);
		if (err < 0)
			return err;

		/* Clear WOL status and disable magic packet matching */
		temp = phy_read(phydev, MII_88E1318S_PHY_WOL_CTRL);
		temp |= MII_88E1318S_PHY_WOL_CTRL_CLEAR_WOL_STATUS;
		temp &= ~MII_88E1318S_PHY_WOL_CTRL_MAGIC_PACKET_MATCH_ENABLE;
		err = phy_write(phydev, MII_88E1318S_PHY_WOL_CTRL, temp);
		if (err < 0)
			return err;
	}

	err = phy_write(phydev, MII_MARVELL_PHY_PAGE, oldpage);
	if (err < 0)
		return err;

	return 0;
}

static struct phy_driver marvell_drivers[] = {
	{
		.phy_id = MARVELL_PHY_ID_88E1101,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1101",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_aneg = &marvell_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1112,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1112",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1111_config_init,
		.config_aneg = &marvell_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1111,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1111",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1111_config_init,
		.config_aneg = &marvell_config_aneg,
		.read_status = &marvell_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1118,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1118",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1118_config_init,
		.config_aneg = &m88e1118_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = {.owner = THIS_MODULE,},
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1121R,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1121R",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_aneg = &m88e1121_config_aneg,
		.read_status = &marvell_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.did_interrupt = &m88e1121_did_interrupt,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1318S,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1318S",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_aneg = &m88e1318_config_aneg,
		.read_status = &marvell_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.did_interrupt = &m88e1121_did_interrupt,
		.get_wol = &m88e1318_get_wol,
		.set_wol = &m88e1318_set_wol,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1145,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1145",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1145_config_init,
		.config_aneg = &marvell_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1149R,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1149R",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1149_config_init,
		.config_aneg = &m88e1118_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1240,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1240",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1111_config_init,
		.config_aneg = &marvell_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1116R,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1116R",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.config_init = &m88e1116r_config_init,
		.config_aneg = &genphy_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E1510,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E1510",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.soft_reset = &m88e1510_phy_reset,
		.config_init = &m88e1510_config_init,
		.config_aneg = &m88e1510_config_aneg,
		.probe = &m88e1510_phy_probe,
		.remove = &m88e1510_phy_remove,
		.read_status = &marvell_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.did_interrupt = &m88e1121_did_interrupt,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_88E3019,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell 88E3019",
		.features = PHY_BASIC_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.probe = &marvell_phy_probe,
		.remove = &marvell_phy_remove,
		.config_init = &m88e3019_config_init,
		.config_aneg = &marvell_config_aneg,
		.read_status = &genphy_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.driver = { .owner = THIS_MODULE },
	},
	{
		.phy_id = MARVELL_PHY_ID_PHYG28LP,
		.phy_id_mask = MARVELL_PHY_ID_MASK,
		.name = "Marvell PHYG28LP",
		.features = PHY_GBIT_FEATURES,
		.flags = PHY_HAS_INTERRUPT,
		.soft_reset = &m88e1510_phy_reset,
		.config_init = &phyg28lp_config_init,
		.config_aneg = &phyg28lp_config_aneg,
		.probe = &m88e1510_phy_probe,
		.remove = &m88e1510_phy_remove,
		.read_status = &marvell_read_status,
		.ack_interrupt = &marvell_ack_interrupt,
		.config_intr = &marvell_config_intr,
		.did_interrupt = &m88e1121_did_interrupt,
		.resume = &genphy_resume,
		.suspend = &genphy_suspend,
		.driver = { .owner = THIS_MODULE },
	},
};

static int __init marvell_init(void)
{
	return phy_drivers_register(marvell_drivers,
		 ARRAY_SIZE(marvell_drivers));
}

static void __exit marvell_exit(void)
{
	phy_drivers_unregister(marvell_drivers,
		 ARRAY_SIZE(marvell_drivers));
}

module_init(marvell_init);
module_exit(marvell_exit);

static struct mdio_device_id __maybe_unused marvell_tbl[] = {
	{ MARVELL_PHY_ID_88E1101, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1112, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1111, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1118, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1121R, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1145, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1149R, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1240, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1318S, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1116R, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E1510, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_88E3019, MARVELL_PHY_ID_MASK },
	{ MARVELL_PHY_ID_PHYG28LP, MARVELL_PHY_ID_MASK},
	{ }
};

MODULE_DEVICE_TABLE(mdio, marvell_tbl);
