/*
 * McKinley 5 Memory Controller
 *
 * Copyright (c) 2014 Lexmark International Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/platform_device.h>
#include <linux/genalloc.h>
#include <asm/cacheflush.h>

#ifdef CONFIG_EDAC_MM_EDAC
#include <linux/edac.h>
#include "../edac/edac_core.h"
#endif

#define MC_STATUS	0x4
#define  MC_IDLE	(1 << 0)
#define  RRB_EMPTY	(1 << 1)
#define  ROB_EMPTY	(1 << 2)
#define  WCB_EMPTY	(1 << 3)
#define  MC_IDLE_MASK	(MC_IDLE | RRB_EMPTY | ROB_EMPTY | WCB_EMPTY)

#define DRAM_STATUS		0x8
#define  NUM_CS			8
#define  INITIALIZED_MASK	0x1
#define  SELF_REFRESH_MASK	0x4

#define USER_COMMAND_0		0x20
#define  ALL_BANKS_MASK		0x1f000000
#define  ENTER_SELF_REFRESH	(0x40 | ALL_BANKS_MASK)
#define  EXIT_SELF_REFRESH	(0x80 | ALL_BANKS_MASK)
#define  WCB_DRAIN		(0x02 | ALL_BANKS_MASK)

#define MMAP0	0x200
#define MMAP1	0x204
#define MMAP2	0x208
#define  CS_VALID	0x1
#define  AREA_LENGTH_MASK	0x1f
#define  AREA_LENGTH_SHIFT	16

#define DRAM_CONFIG_1		0x300
#define  DRAM_TYPE_MASK		0xf
#define  DRAM_TYPE_SHIFT	4

struct mckinley_idle {
#ifdef CONFIG_MCKINLEY5_ARM_SELF_REFRESH
	struct gen_pool *sram_pool;
	void *sram_vaddr;
	void *sram_execaddr;
	unsigned long sram_stack;
#endif
	struct platform_device *pdev;
	struct resource *res;
	void __iomem *mc_iomem;
	int multi_cs;
	struct dentry *dbg_file;
};

#ifdef CONFIG_MCKINLEY5_ARM_SELF_REFRESH

/* Overkill */
#define SRAM_STACK_SIZE SZ_1K

#define __sram_cpuidle __attribute__((section("sram_cpuidle"))) noinline notrace

extern char __start_sram_cpuidle[];
extern char __stop_sram_cpuidle[];
#define sram_cpuidle_size ((unsigned long) (__stop_sram_cpuidle - __start_sram_cpuidle))
#define sram_offset(sym) ((unsigned long)&sym - (unsigned long)__start_sram_cpuidle)

/* SRAM code start */

static inline int in_self_refresh(u32 dram_status)
{
	int i;

	for (i = 0; i < NUM_CS; i++) {
		if (dram_status & INITIALIZED_MASK) {
			if (!(dram_status & SELF_REFRESH_MASK))
				return 0;
		}
		dram_status >>= 4;
	}

	return 1;
}

typedef unsigned long (*wfi_t)(void __iomem *mc_iomem);

unsigned long __sram_cpuidle sram_wfi(void __iomem *mc_iomem)
{
	unsigned long dram_status;
	unsigned long tries = 5;

	/* Drain the write combine buffer */
	writel(WCB_DRAIN, mc_iomem + USER_COMMAND_0);

	/* Wait for the controller to become idle */
	while ((readl(mc_iomem + MC_STATUS) & MC_IDLE_MASK) != MC_IDLE_MASK)
		;

	/* Loop to make sure we are actually in self-refresh before entering WFI */
	do {
		writel(ENTER_SELF_REFRESH, mc_iomem + USER_COMMAND_0);
		dram_status = readl_relaxed(mc_iomem + DRAM_STATUS);
	} while (!in_self_refresh(dram_status) && --tries);

	wfi();

	writel(EXIT_SELF_REFRESH, mc_iomem + USER_COMMAND_0);

	return 0;
}

/* SRAM code end */

static void populate_tlb(void __iomem *start, size_t size)
{
	void __iomem *addr = (void *)PAGE_ALIGN((unsigned long)start);

	for (; addr < (start + size); addr += PAGE_SIZE) {
		readl_relaxed(addr);
	}
}

/*
 * This function runs from DRAM with a stack in SRAM.  current() and friends
 * aren't valid here, so lots of kernel functions are off limits.
 */
static int notrace mckinley_enter_self_refresh(void *arg)
{
	struct mckinley_idle *data = arg;
	wfi_t idle_ptr = data->sram_execaddr + sram_offset(sram_wfi);
	unsigned long dram_status;

	/*
	 * Populate TLB for code and stack
	 *
	 * This assumes there are enough TLB entries to cover our stack and
	 * code just in case we cross a page boundary after entering
	 * self-refresh. The consequence of accessing memory in self-refresh is
	 * only automatically exiting self-refresh, so this does not have to be
	 * bulletproof to prevent a crash or hang.
	 */
	populate_tlb(data->sram_execaddr, sram_cpuidle_size);
	populate_tlb((void *)data->sram_stack, SRAM_STACK_SIZE);
	rmb();

	/* Jump to SRAM */
	dram_status = idle_ptr(data->mc_iomem);

	return dram_status;
}

extern int call_with_stack(int (*fn)(void *), void *arg, void *sp);

int mckinley_do_idle(struct mckinley_idle *data)
{
	int rc = 0;
	unsigned long stack_ptr =
		ALIGN(data->sram_stack + SRAM_STACK_SIZE, 8);

	if (data->multi_cs)
		wfi();
	else
		rc = call_with_stack(mckinley_enter_self_refresh, data, (void *)stack_ptr);

	return rc;
}

struct mckinley_idle *of_get_mckinley_idle(struct device_node *np,
	const char *propname, int index)
{
	struct device_node *idle_np = NULL;
	struct platform_device *idle_pdev;
	int ret;

	idle_np = of_parse_phandle(np, propname, 0);
	if (!idle_np) {
		ret = -ENODEV;
		goto err;
	}
	/*
	 * Grabs a reference to the device which must be released by
	 * mckinley_idle_put
	 */
	idle_pdev = of_find_device_by_node(idle_np);
	if (!idle_pdev) {
		ret = -EPROBE_DEFER;
		goto err;
	}

	of_node_put(idle_np);
	return platform_get_drvdata(idle_pdev);
err:
	of_node_put(idle_np);
	return ERR_PTR(ret);
}

void mckinley_idle_put(struct mckinley_idle *data)
{
	put_device(&data->pdev->dev);
}

static int arm_self_refresh_remove(struct platform_device *pdev)
{
	struct mckinley_idle *data = platform_get_drvdata(pdev);

	if (data->sram_execaddr)
		iounmap(data->sram_execaddr);
	if (data->sram_stack)
		gen_pool_free(data->sram_pool, data->sram_stack, SRAM_STACK_SIZE);
	if (data->sram_vaddr)
		gen_pool_free(data->sram_pool, (unsigned long) data->sram_vaddr,
				sram_cpuidle_size);
	return 0;
}

static int __init arm_self_refresh_probe(struct platform_device *pdev)
{
	struct mckinley_idle *data = platform_get_drvdata(pdev);
	struct device_node *np = pdev->dev.of_node;
	dma_addr_t sram_paddr;
	int ret;

	data->sram_pool = of_get_named_gen_pool(np, "sram", 0);
	if (!data->sram_pool) {
		dev_err(&pdev->dev, "sram not available\n");
		ret = -ENOMEM;
		goto err;
	}

	data->sram_vaddr = gen_pool_dma_alloc(data->sram_pool,
					      sram_cpuidle_size, &sram_paddr);
	if (!data->sram_vaddr) {
		dev_err(&pdev->dev, "unable to alloc sram\n");
		ret = -ENOMEM;
		goto err;
	}
	dev_dbg(&pdev->dev, "SRAM idle loop physical address %pK\n", (void *)sram_paddr);

	/*
	 * Set as uncached because mmio-sram also has an uncached mapping.
	 * Having multiple maps with inconsistent cacheability is invalid.
	 */
	data->sram_execaddr = __arm_ioremap_exec(sram_paddr, sram_cpuidle_size, false);
	if (!data->sram_execaddr) {
		dev_err(&pdev->dev, "unable to map sram\n");
		ret = -ENOMEM;
		goto err;
	}

	data->sram_stack = gen_pool_alloc(data->sram_pool, SRAM_STACK_SIZE);
	if (!data->sram_stack) {
		dev_err(&pdev->dev, "unable to allocate sram stack\n");
		ret = -ENOMEM;
		goto err;
	}

	/*
	 * Initialize the return address (cpu_resume) and copy the idle function to SRAM
	 */
	memcpy(data->sram_vaddr, __start_sram_cpuidle, sram_cpuidle_size);
	flush_icache_range((unsigned long)data->sram_vaddr,
			   (unsigned long)data->sram_vaddr + sram_cpuidle_size);

	return 0;
err:
	arm_self_refresh_remove(pdev);
	return ret;
}

#else
static int arm_self_refresh_remove(struct platform_device *pdev) {}
static int __init arm_self_refresh_probe(struct platform_device *pdev)
{
	return 0;
}
#endif

static void *mckinley_dbg_start(struct seq_file *s, loff_t *pos)
{
	struct mckinley_idle *priv = s->private;

	if (((*pos) * 0x10) >= resource_size(priv->res))
		return NULL;

	return pos;
}

static void *mckinley_dbg_next(struct seq_file *s, void *v, loff_t *pos)
{
	struct mckinley_idle *priv = s->private;

	*pos = *pos + 1;

	if (((*pos) * 0x10) >= resource_size(priv->res))
		return NULL;

	return pos;
}

static void mckinley_dbg_stop(struct seq_file *s, void *v)
{
}

static int mckinley_dbg_show(struct seq_file *s, void *v)
{
	struct mckinley_idle *priv = s->private;
	void *__iomem reg = priv->mc_iomem + (*(loff_t *)v * 0x10);

	seq_printf(s, "%#08x: %08x %08x %08x %08x\n",
			(u32)(priv->res->start + (*(loff_t *)v * 0x10)),
			readl(reg),
			readl(reg + 4),
			readl(reg + 8),
			readl(reg + 0xc));

	return 0;
}

static const struct seq_operations mckinley_dbg_ops = {
	.start = mckinley_dbg_start,
	.next  = mckinley_dbg_next,
	.stop  = mckinley_dbg_stop,
	.show  = mckinley_dbg_show,
};

static int mckinley_dbg_open(struct inode *inode, struct file *file)
{
	int rc;
	struct seq_file *s;

	rc = seq_open(file, &mckinley_dbg_ops);
	if (rc)
		return rc;

	s = file->private_data;
	s->private = inode->i_private;

	return 0;
}

static const struct file_operations mckinley_dbg_file_fops = {
	.owner   = THIS_MODULE,
	.open    = mckinley_dbg_open,
	.read    = seq_read,
	.llseek  = seq_lseek,
	.release = seq_release
};

static inline unsigned int __init mmap_size_mb(u32 val)
{
	val = (val >> AREA_LENGTH_SHIFT) & AREA_LENGTH_MASK;

	if (WARN_ON(val < 7))
		return 0;
	return 8 << (val - 7);
}

static unsigned int __init mckinley_cs_size_mb(struct mckinley_idle *data, int cs)
{
	u32 reg;

	switch (cs) {
	case 0:
		reg = readl_relaxed(data->mc_iomem + MMAP0);
		break;
	case 1:
		reg = readl_relaxed(data->mc_iomem + MMAP1);
		break;
	case 2:
		reg = readl_relaxed(data->mc_iomem + MMAP2);
		break;
	default:
		BUG();
		return 0;
	}

	if (!(reg & CS_VALID))
		return 0;

	return mmap_size_mb(reg);
}

#ifdef CONFIG_EDAC_MM_EDAC
static enum mem_type __init mckinley_mem_type(struct mckinley_idle *data)
{
	u32 val = readl_relaxed(data->mc_iomem + DRAM_CONFIG_1);

	val = val >> DRAM_TYPE_SHIFT & DRAM_TYPE_MASK;

	switch (val) {
	case 0x9:
		return MEM_DDR2;
	case 0xa:
	case 0x2:
		return MEM_DDR3;
	case 0x3:
		return MEM_DDR4;
	default:
		return MEM_UNKNOWN;
	}
}

static int __init mckinley_edac_probe(struct platform_device *pdev)
{
	struct mckinley_idle *data = platform_get_drvdata(pdev);
	struct edac_mc_layer layers[1];
	struct mem_ctl_info *mci;
	struct dimm_info *dimm;
	enum mem_type mem_type;
	int rc;
	int i;

	layers[0].type = EDAC_MC_LAYER_CHIP_SELECT;
	layers[0].size = 3;
	layers[0].is_virt_csrow = true;

	mci = edac_mc_alloc(0, ARRAY_SIZE(layers), layers, 0);
	if (!mci)
		return -ENOMEM;

	mci->pdev = &pdev->dev;
	mci->edac_ctl_cap = EDAC_FLAG_NONE;
	mci->edac_cap = mci->edac_ctl_cap;
	mci->mod_name = "mckinley5";
	mci->mod_ver = "1";
	mci->ctl_name = dev_name(&pdev->dev);
	mci->scrub_mode = SCRUB_NONE;
	mci->dev_name = dev_name(&pdev->dev);

	mem_type = mckinley_mem_type(data);
	for (i = 0; i < mci->tot_dimms; i++) {
		BUG_ON(i >= layers[0].size);
		dimm = mci->dimms[i];
		dimm->nr_pages = mckinley_cs_size_mb(data, i) << (20 - PAGE_SHIFT);
		dimm->mtype = mem_type;
		dimm->edac_mode = EDAC_NONE;
	}

	rc = edac_mc_add_mc(mci);
	if (rc)
		dev_err(&pdev->dev, "edac_mc_add_mc failed %d\n", rc);

	return rc;
}

static int mckinley_edac_remove(struct platform_device *pdev)
{
	struct mem_ctl_info *mci;

	mci = edac_mc_del_mc(&pdev->dev);
	if (!mci)
		return -ENODEV;

	edac_mc_free(mci);

	return 0;
}
#else
static int __init mckinley_edac_probe(struct platform_device *pdev)
{
	return 0;
}

static int mckinley_edac_remove(struct platform_device *pdev)
{
	return 0;
}
#endif

static int __exit mckinley_remove(struct platform_device *pdev)
{
	struct mckinley_idle *data = platform_get_drvdata(pdev);

	debugfs_remove(data->dbg_file);
	arm_self_refresh_remove(pdev);
	mckinley_edac_remove(pdev);

	return 0;
}

static int __init mckinley_probe(struct platform_device *pdev)
{
	struct mckinley_idle *data;
	int tmp;
	int ret;

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;
	platform_set_drvdata(pdev, data);
	data->pdev = pdev;

	data->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!data->res)
		return -EINVAL;

	data->mc_iomem = devm_ioremap_resource(&pdev->dev, data->res);
	if (IS_ERR(data->mc_iomem))
		return PTR_ERR(data->mc_iomem);

	tmp = !!mckinley_cs_size_mb(data, 0);
	tmp += !!mckinley_cs_size_mb(data, 1);
	tmp += !!mckinley_cs_size_mb(data, 2);
	if (tmp > 1)
		data->multi_cs = 1;

	mckinley_edac_probe(pdev);

	ret = arm_self_refresh_probe(pdev);
	if (ret)
		mckinley_edac_remove(pdev);

	data->dbg_file = debugfs_create_file("mckinley_reg", S_IRUGO, NULL, data, &mckinley_dbg_file_fops);

	return ret;
}

static const struct of_device_id of_mckinley_table[] = {
	{ .compatible = "marvell,mckinley5" },
	{}
};

static struct platform_driver mckinley_driver = {
	.remove = __exit_p(mckinley_remove),
	.driver = {
		.name = "mckinley5",
		.owner = THIS_MODULE,
		.of_match_table = of_mckinley_table,
	},
};

static int __init mckinley_init(void)
{
	return platform_driver_probe(&mckinley_driver, mckinley_probe);
}
subsys_initcall(mckinley_init);

static void __exit mckinley_exit(void)
{
	platform_driver_unregister(&mckinley_driver);
}
module_exit(mckinley_exit);
