/*
 * dro-pegmatite.c - Driver for pegmatite DRO
 *
 */

#include <linux/module.h>
#include <linux/init.h> 
#include <linux/slab.h>
#include <linux/hwmon.h>
#include <linux/sysfs.h>
#include <linux/hwmon-sysfs.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/delay.h>

/* Provides settings for DRO mode of voltage control (vs. Fixed Voltage mode) */
#define AVS_CONTROL_CONF 0x00
#define DROMODE1 0x10
#define DROMODE2 0x14

/* Intermediate, Max, and Min DRO count Values */
#define DROSTATUS1 0x2C
#define DROSTATUS2 0x30

#define INT_DRO_COUNT_MASK 0xffff
#define AVS_ENABLE_MASK 0x3

enum { SHOW_DRO };

struct dro_pegmatite_data {
	struct device *hwmon_dev;
	void __iomem *dro_ioaddr;
};

/*
 *Sysfs stuff
 */
static int32_t _get_dro_value(void __iomem *ioaddr) {
	uint32_t val = 0;

	if (ioaddr == NULL){
		return -1;
	}

	val = readl(ioaddr + AVS_CONTROL_CONF);
	val |= AVS_ENABLE_MASK;
	writel(val, ioaddr + AVS_CONTROL_CONF);

	msleep(200);

	val = readl(ioaddr + DROSTATUS1);
	val &= INT_DRO_COUNT_MASK;

	val = readl(ioaddr + AVS_CONTROL_CONF);
	val &= ~AVS_ENABLE_MASK; 
	writel(val, ioaddr + AVS_CONTROL_CONF);

	return val;
};

static ssize_t show_dro(struct device *dev, struct device_attribute *devattr, char *buf)
{
	struct dro_pegmatite_data *data = dev_get_drvdata(dev);
	void __iomem *ioaddr = data->dro_ioaddr;

	return snprintf(buf, PAGE_SIZE, "%d\n", _get_dro_value(ioaddr));
}

static SENSOR_DEVICE_ATTR(dro, S_IRUGO, show_dro, NULL, SHOW_DRO);

static struct attribute *dro_pegmatite_attributes[] = {
	&sensor_dev_attr_dro.dev_attr.attr,
	NULL
};

static const struct attribute_group dro_pegmatite_group = {
	.attrs = dro_pegmatite_attributes
};

static int dro_pegmatite_probe(struct platform_device *pdev)
{
	struct dro_pegmatite_data *data;
	struct resource *dro_res;
	int err = 0;

	dro_res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if(!dro_res) {
		return -ENODEV;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if(!data) {
		return -ENOMEM;
	}

	data->dro_ioaddr = devm_ioremap_resource(&pdev->dev, dro_res);
	if(IS_ERR(data->dro_ioaddr)) {
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, data);
	err = sysfs_create_group(&pdev->dev.kobj, &dro_pegmatite_group);
	data->hwmon_dev = hwmon_device_register(&pdev->dev);
	if (IS_ERR(data->hwmon_dev)) {
		err = PTR_ERR(data->hwmon_dev);
		dev_err(&pdev->dev, "Class registration failed (%d)\n",
			err);
		return -ENOMEM;
	}

	return 0;
}

static int dro_pegmatite_remove(struct platform_device *pdev)
{
	sysfs_remove_group(&pdev->dev.kobj, &dro_pegmatite_group);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static const struct of_device_id dro_of_match[] = {
	{ .compatible = "marvell,pegmatite-dro", },
	{},
};
MODULE_DEVICE_TABLE(of, dro_of_match);

static struct platform_driver dro_pegmatite_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "pegmatite-dro",
		.of_match_table = dro_of_match,
	},
	.probe = dro_pegmatite_probe,
	.remove = dro_pegmatite_remove,
};
module_platform_driver(dro_pegmatite_driver);

MODULE_ALIAS("platform:pegmatite-dro");
MODULE_DESCRIPTION("pegmatite CPU dro counter");
MODULE_LICENSE("GPL");
