/*
 *  linux/arch/arm/mach-mv61x0/scatterhelp.c
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * Provides basic scatterlist buffer support. For examples of more sophisticated
 * scatter-gather buffer management, see drivers/media/video/videobuf-dma-sg.c.
 * May need to move this into a library eventually.
 */
#include <linux/export.h>
#include <linux/dma-mapping.h>
#include <linux/err.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/scatterlist.h>
#include <linux/platform_data/mv61_cdma.h>
 
/**
 * dmabuf_vmalloc_to_sg - create scatterlist for a buffer obtained from vmalloc
 * @buf: virtual channel control structure
 * 
 * Assumes buffer is virtually contiguous but probably not physically contiguous.
 */
enum dmabuf_malloc_type {
	BUF_IS_KMALLOC,
	BUF_IS_VMALLOC,
};	

static int dmabuf_vkmalloc_to_sg(struct dmabuf_sg *buf, enum dmabuf_malloc_type buftype)
{
	char 	*addr 		= buf->virt;
	unsigned long count	= buf->bytes;
	struct	scatterlist *sglist;
	struct	page *pg;
	int	max_pages 	= (buf->bytes >> PAGE_SHIFT) + 2; /* allow for offset */
	int	list_size 	= max_pages * sizeof(*sglist);
	int 	i;
	int	err = 0;
	
	buf->sg = NULL;
	buf->sg_ents = 0;
	
/* 
 * Scatterlists must fit in a page, not going to deal with chained scatterlists here.
 * Limits max transfer size to (PAGE_SIZE^2)/sizeof( a scatterlist entry), which is 
 * still a very big transfer.
 * For huge transfers or buffers that are not virtually contiguous, see 
 * sg_chain in include/linux/scatterlist.h.
 */

	if (list_size > PAGE_SIZE)
		return -EINVAL;
	
	sglist = kzalloc(list_size, GFP_KERNEL);
	if(!sglist)
		return -ENOMEM;
	memset(sglist, 0, list_size);
	sg_init_table(sglist, max_pages);

	/* break buffer into pages and create scatterlist entry for each */
	for (i = 0; i < max_pages; i++) {
		unsigned int offset, length;

		if(!count)
			break;

		offset = (unsigned int)addr & ~PAGE_MASK;
		length = PAGE_SIZE - offset;
		if (length > count)
			length = count;
		if(buftype == BUF_IS_VMALLOC)
			pg = vmalloc_to_page(addr);
		else /* BUF_IS_KMALLOC */
			pg = virt_to_page(addr);
		
		if (pg) {
			
			BUG_ON(PageHighMem(pg)); 
			
			sg_set_page(&sglist[i], pg, length, offset);
		} else {
			goto err_out;
		}

		addr += length;
		count -= length;
	}
	BUG_ON(count);
	
	/* update the end mark to the actual number of pages */
	buf->sg_ents = i;
	sg_mark_end(&sglist[i ? i - 1 : 0]);
	buf->sg = sglist;
	
	return 0;

err_out:
	vfree(sglist);
	buf->sg = NULL;
	buf->sg_ents = 0;
	return err;
	 	
}

int dmabuf_vmalloc_to_sg(struct dmabuf_sg *buf)
{
	if(!virt_addr_valid(buf->virt) && is_vmalloc_addr(buf->virt))
		return dmabuf_vkmalloc_to_sg(buf, BUF_IS_VMALLOC);
	else
		return -EINVAL;
}
EXPORT_SYMBOL(dmabuf_vmalloc_to_sg);

int dmabuf_kmalloc_to_sg(struct dmabuf_sg *buf)
{
	if(virt_addr_valid(buf->virt) && !is_vmalloc_addr(buf->virt))
		return dmabuf_vkmalloc_to_sg(buf, BUF_IS_KMALLOC);
	else
		return -EINVAL;
	
}
EXPORT_SYMBOL(dmabuf_kmalloc_to_sg);
