/*
 * Driver for the Marvell 61xx CDMA Controller
 *
 * Derived from drivers/dma/dw_dmac.c
 *
 * Copyright (C) 2007-2008 Atmel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/delay.h>
#include <linux/dmaengine.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/jiffies.h>
#include <asm/atomic.h>
#include "cdma.h"
/*
 * This supports the Marvell Central DMA Controller used in the 
 * Marvell 88PA61x0, which does not support descriptor writeback.
 */

/*----------------------------------------------------------------------*/
/**
 * mv61vc_first_active - get first active transaction queued for channel 
 * @mv61vc: virtual channel control structure
 *
 * Call with mv61vc->lock held.
 */
struct mv61_desc *mv61vc_first_active(struct mv61_vdma_chan *mv61vc)
{
	if(list_empty(&mv61vc->active_list))
		return NULL;
	return list_entry(mv61vc->active_list.next, struct mv61_desc, desc_node);
}

/**
 * mv61_tx_list_next - get next subtransaction
 * @desc: top transaction control structure
 *
 * Only called from irq handler.
 *
 * Call with mv61vc->lock held.
 */
bool mv61_tx_list_next(struct mv61_desc *desc)
{
	/* Only called from irq handler. Nothing else will touch the 
	 * active pointer after it is initialized.
	 */
	
	
	if(list_is_last(&desc->active->tx_list,	&desc->tx_list)) {
		/* desc->active is already pointing to last subtrans */
		desc->active = NULL;
		return 0;
	} else {
		desc->active = list_first_entry(&desc->active->tx_list, 
						struct mv61_desc, tx_list);
		return 1;
	}
}

/**
 * mv61_retire_tx - move a transaction descriptor tree to completed list
 * @mv61vc: virtual channel control structure
 * 
 * Call with mv61vc->lock held.
 */
void mv61_retire_tx(struct mv61_vdma_chan *mv61vc)
{
	//dev_printk(KERN_CRIT, chan2dev(&mv61vc->chan), "mv61_retire_tx:%d\n", __LINE__);
	if(!list_empty(&mv61vc->active_list))
		list_move_tail(mv61vc->active_list.next, &mv61vc->complete_list);
}

/**
 * mv61_link_get - request a transaction descriptor from the slab allocator
 * @mv61vc: virtual channel control structure
 * 
 * The descriptors may have been reused, so reinitialize the members that
 * may have been overwritten.
 *
 * This can sleep. Don't call from interrupt context or within a spin_lock.     
 */
struct mv61_chain *mv61_link_get(struct mv61_vdma_chan *mv61vc)
{
	struct mv61_vdma	*mv61v = mv61vc->mv61v;
	struct mv61_chain	*link;
	
	link = kmem_cache_alloc(mv61v->mv61p->chain_cachep, (GFP_DMA | GFP_KERNEL));
	if(link) {

		/* map it now to get the physical address */
		link->phys = dma_map_single(chan2parent(&mv61vc->chan), &link->lli,
				sizeof(link->lli), DMA_TO_DEVICE);
		
		/* sync for cpu so it can be initialized */
		dma_sync_single_for_cpu(chan2parent(&mv61vc->chan),
				link->phys, sizeof(link->lli),
				DMA_TO_DEVICE);

		link->lli.SrcAddr = (dma_addr_t)NULL;
		link->lli.DestAddr = (dma_addr_t)NULL;
		link->lli.OwnLength = 0;
		link->lli.NextCtrl = (DESCRIPTOR_NEXTCTRL_INT_MASK | DESCRIPTOR_NEXTCTRL_STOP_MASK);
	
		link->synced = 0;
		__dev_vdbg(chan2dev(&mv61vc->chan), "mv61_link_get: created link "
						"at virt %p, phys %p\n", 
						(void *)link, 
						(void *)link->phys);
	}
	
	return link;	
}

/**
 * mv61_desc_get - request a transaction descriptor from the slab allocator
 * @mv61vc: virtual channel control structure
 * 
 * The descriptors may have been reused, so reinitialize the members that
 * may have been overwritten.     
 *
 * The all_chains list head is in top struct mv61_dma, but it is never touched 
 * in the irq handler, only the tasklet. Really just need atomic list ops.
 *
 * This can sleep. Don't call from interrupt context or within a spin_lock.     
 */
struct mv61_desc *mv61_desc_get(struct mv61_vdma_chan *mv61vc)
{
	struct mv61_vdma	*mv61v = mv61vc->mv61v;
	struct mv61_desc 	*desc;
	unsigned long flags;
/* 
 * may need to revist this to reduce the kmem_cache_alloc() calls for performance,
 * but for now just get it working.
 */
	desc = kmem_cache_alloc(mv61v->mv61p->desc_cachep, GFP_KERNEL);
	
	if (desc) {
		dma_async_tx_descriptor_init(&desc->txd, NULL);
		INIT_LIST_HEAD(&desc->desc_node);
		INIT_LIST_HEAD(&desc->tx_list);
		INIT_LIST_HEAD(&desc->tx_list);
		INIT_LIST_HEAD(&desc->subchain);
		INIT_LIST_HEAD(&desc->all_chains_node);
		desc->txd.cookie = 0;
		desc->txd.flags = DMA_CTRL_ACK;
		desc->txd.phys = (dma_addr_t)NULL;
		desc->txd.chan = &mv61vc->chan;
		desc->txd.tx_submit = mv61vc_tx_submit;
		desc->txd.callback = NULL;
		desc->txd.callback_param = NULL;
		desc->paused_link = NULL;
		desc->chains = 0;
		desc->links = 0;
		desc->len = 0;
		desc->sublen = 0;
		desc->active = desc;
		
		desc->lli.SrcAddr = (dma_addr_t)NULL;
		desc->lli.DestAddr = (dma_addr_t)NULL;
		desc->lli.OwnLength = 0;
		desc->lli.NextCtrl = (DESCRIPTOR_NEXTCTRL_INT_MASK | DESCRIPTOR_NEXTCTRL_STOP_MASK);
		desc->lli_phys = dma_map_single(chan2parent(&mv61vc->chan), &desc->lli,
				sizeof(desc->lli), DMA_TO_DEVICE);
		dma_sync_single_for_cpu(chan2parent(&mv61vc->chan), desc->lli_phys,
				sizeof(desc->lli), DMA_TO_DEVICE);
		
		spin_lock_irqsave(&mv61v->mv61p->all_chains_lock, flags);
		list_add_tail(&desc->all_chains_node, &mv61v->mv61p->all_chains);
		spin_unlock_irqrestore(&mv61v->mv61p->all_chains_lock, flags);
		
	}
	
	return desc;
	
}

/**
 * mv61_link_put - return a single hardware descriptor to the slab allocator
 * @mv61v: this top virtual dma control instance
 * @chan: dmaengine API channel control structure
 * @link: hardware linked list descriptor
 * 
 * Some of the links may already be unmapped if the channel was paused.
 */
void mv61_link_put(struct mv61_vdma *mv61v, struct dma_chan *chan, 
							struct mv61_chain *link)
{
	list_del(&link->chain_node);
	dma_unmap_single(chan2parent(chan), link->phys, sizeof(link->lli),
								DMA_TO_DEVICE);
	kmem_cache_free(mv61v->mv61p->chain_cachep, link);
}

/**
 * mv61_chain_put - return a single transaction subchain to the slab allocator
 * @mv61v: this top virtual dma control instance
 * @desc: top level transaction control structure
 *
 * The all_chains list head is in top struct mv61_dma, but it is never touched 
 * in the irq handler, only the tasklet. Really just need atomic list ops.
 * The kernel does provide smp-safe versions with internal spin_locks, but they 
 * still aren't bottom-half-safe, so just handle it here.
 */
void mv61_chain_put(struct mv61_vdma *mv61v, struct mv61_desc *desc)
{
	struct mv61_chain *link;
	unsigned long flags;
	while (!list_empty(&desc->subchain)) {
		link = list_entry(desc->subchain.next, struct mv61_chain,
								chain_node);
		mv61_link_put(mv61v, desc->txd.chan, link);
	}
	list_del(&desc->tx_list);
	
	spin_lock_irqsave(&mv61v->mv61p->all_chains_lock, flags);
	list_del(&desc->all_chains_node);
	spin_unlock_irqrestore(&mv61v->mv61p->all_chains_lock, flags);
	
	dma_unmap_single(chan2parent(desc->txd.chan), desc->lli_phys,
				sizeof(desc->lli), DMA_TO_DEVICE);
	
	kmem_cache_free(mv61v->mv61p->desc_cachep, desc);
}

/**
 * mv61_desc_put - return a transaction descriptor tree to the slab allocator
 * @desc: top level transaction control structure
 *
 * TODO: Implement support for the DMA_CTRL_ACK descriptor purgatory.
 */
void mv61_desc_put(struct mv61_desc *desc)
{
	struct mv61_vdma	*mv61v = ddev_to_mv61_vdma(desc->txd.chan->device);
	
/* 
 * may need to revist this to reduce the kmem_cache_alloc() calls for performance,
 * but for now just get it working
 */
	if (desc) {
		struct mv61_desc *child, *_child;

		list_for_each_entry_safe(child, _child, &desc->tx_list, tx_list) {
			mv61_chain_put(mv61v, child);
		}
		mv61_chain_put(mv61v, desc);
	}
}

/**
 * mv61vc_assign_cookie - 
 * @mv61vc: virtual channel control structure
 * @desc: top level transaction control structure
 *
 * Call with mv61vc->lock held and irqs disabled.
 */
dma_cookie_t
mv61vc_assign_cookie(struct mv61_vdma_chan *mv61vc, struct mv61_desc *desc)
{
	dma_cookie_t cookie = mv61vc->chan.cookie;

	if (++cookie < 0)
		cookie = 1;

	mv61vc->chan.cookie = cookie;
	desc->txd.cookie = cookie;

	return cookie;
}

/*
 * mv61vc_prep_slave_lli - populate hw portion of list entry for peripheral
 * @lli: reg values used by hardware for one block of a transfer
 * @mv61vc: virtual dma channel
 * @direction: dma direction defined by linux/dma_mapping.h
 * @flowcontrol: CDMA_CFG_FLOWCTRL value
 * @mem: physical address from scatterlist entry 
 * @len: length in bytes for this piece of transfer
 * @offset: cumulative total length for previous descriptors
 */
int mv61vc_prep_slave_lli(struct mv61_lli *lli, 
					struct mv61_vdma_chan *mv61vc,
					enum dma_transfer_direction direction,
					enum mv61_dma_flow_ctrl	flowcontrol,
					u32 mem, u32 len, size_t offset) 
{
	dma_addr_t * src = NULL;
	dma_addr_t * dest = NULL;

	src = &lli->SrcAddr;
	dest = &lli->DestAddr;

	switch (direction) {
	case DMA_DEV_TO_MEM:
		*src = mv61vc->def.SrcAddr;
		*dest = mem;
		break;
	case DMA_MEM_TO_MEM: /* for dma from sg test */
		*src = (u32)mv61vc->def.SrcAddr + offset;
		*dest = mem;
		break;
	case DMA_MEM_TO_DEV:
		*dest = mv61vc->def.DestAddr;
		*src = mem;
		break;
	default:
		goto err_cfg;
		break;
	}
	
	if(!*dest || !*src)
		goto err_null_mem;
	
	lli->OwnLength = len & DESCRIPTOR_OWNLENGTH_LENGTH_MASK; /* Bits 15:0 are the length */
	lli->OwnLength |= DESCRIPTOR_OWNLENGTH_OWN_MASK; /* Set the own bit to dma */
	
	if((flowcontrol == MV61_DMA_MEMORY_TO_MEMORY) && !dmaengine_check_align(
							MV61_MEMCPY_ALIGN,
							*src,
							*dest,
							len)) {   
		dev_err(chan2dev(&mv61vc->chan), "unaligned access\n");
		goto err_cfg;
	}
	
	return 0;
	
err_null_mem:
	dev_err(chan2dev(&mv61vc->chan), "not enough descriptors available\n");
err_cfg:
	return -EINVAL;
}

/**
 * mv61_clear_pchannel - restore channel to its reset state
 * @mv61pc: physical channel control structure
 * 
 * Call with mv61p->biglock held.
 */
int mv61_clear_pchannel(struct mv61_pdma_chan *mv61pc)
{
	struct mv61_pdma_chan_regs *pcregs = mv61pc->ch_regs;
	
	writel(0, &pcregs->CFG);
	writel(0, &pcregs->FillValue);
	writel(0, &pcregs->intEn);
	writel(0, &pcregs->TimerControl);
	writel(0, &pcregs->CDR);
	writel(0, &pcregs->Control);

	writel(~(CDMA_INTACK_RESERVED1_MASK), &pcregs->intAck);
	return 0;
}

/**
 * mv61_vpmap_v_to_p - get physical channel assigned to this virtual channel
 * @mv61vc: virtual channel control structure
 *
 * Call with mv61p->biglock held.
 */
struct mv61_pdma_chan *mv61_vpmap_v_to_p(struct mv61_vdma_chan *mv61vc)
{
	struct mv61_vdma	*mv61v = mv61vc->mv61v;
	struct mv61_dma_vpmap	*vpmap = mv61v->mv61p->vpmap;
	int			vindex;
	
	vindex = vpmap->voffset[mv61v->vtype] + mv61vc->chan.chan_id;

	return(vpmap->v_to_p[vindex]);
}
 
/**
 * mv61_vpmap_p_to_v - get virtual channel assigned to this physical channel
 * @mv61pc: physical channel control structure
 *
 * Call with mv61p->biglock held.
 */
struct mv61_vdma_chan *mv61_vpmap_p_to_v(struct mv61_pdma_chan *mv61pc)
{
	struct mv61_dma_vpmap	*vpmap = mv61pc->mv61p->vpmap;

	return(vpmap->p_to_v[mv61pc->index]);
}
 
/**
 * mv61_vpmap_pair - assign virtual channel <--> physical channel
 * @mv61vc: virtual channel control structure
 * @mv61pc: physical channel control structure
 *
 * This function only assigns the channel structure pointers. The sharing status
 * flags are handled by the dispatcher.
 *
 * Call with mv61p->biglock held.
 */
struct mv61_vdma_chan *mv61_vpmap_pair(struct mv61_vdma_chan *mv61vc, 
						struct mv61_pdma_chan *mv61pc)
{
	struct mv61_dma_vpmap	*vpmap = mv61pc->mv61p->vpmap;
	struct mv61_vdma	*mv61v = mv61vc->mv61v;
	int			vindex;
	
	vindex = vpmap->voffset[mv61v->vtype] + mv61vc->chan.chan_id;
	
	vpmap->p_to_v[mv61pc->index]	= mv61vc;
	vpmap->v_to_p[vindex]		= mv61pc;
	
	return(0);
}
 
/**
 * mv61_vpmap_unpair - unassign virtual channel <--> physical channel
 * @mv61vc: virtual channel control structure
 * @mv61pc: physical channel control structure
 *
 * This function only assigns the channel structure pointers. The sharing status
 * flags are handled by the dispatcher.
 *
 * Call with mv61p->biglock held.
 */
struct mv61_vdma_chan *mv61_vpmap_unpair(struct mv61_vdma_chan *mv61vc, 
						struct mv61_pdma_chan *mv61pc)
{
	struct mv61_dma_vpmap	*vpmap = mv61pc->mv61p->vpmap;
	struct mv61_vdma	*mv61v = mv61vc->mv61v;
	int			vindex;
	
	vindex = vpmap->voffset[mv61v->vtype] + mv61vc->chan.chan_id;
	
	vpmap->p_to_v[mv61pc->index]	= NULL;
	vpmap->v_to_p[vindex]		= NULL;
	
	return(0);
}
 
/**
 * mv61vc_pause_dump - dump vchan data saved when pausing
 * @link: virtual address of descriptor to dump
 */
void mv61vc_pause_dump(struct mv61_vdma_chan *mv61vc)
{
	printk(KERN_NOTICE "mv61vc pause data @%p:\n", mv61vc);
	SHOWVARW(mv61vc->active_list.next);
	SHOWVARW(mv61vc->status);
	SHOWVARW(mv61vc->residue);
	SHOWVARW(mv61vc->hwstat.Status);
	SHOWVARW(mv61vc->hwstat.CPR);
	SHOWVARW(mv61vc->hwstat.CDR);

	SHOWVARW(mv61vc->hwstat.Control);
}

/**
 * mv61_link_dump - dump a single chained descriptor
 * @link: virtual address of link descriptor to dump
 * @desc: virtual address of subchain's transaction descriptor
 */
void mv61_link_dump(struct mv61_chain *link, struct mv61_desc *desc)
{
	int synced;
	struct device * dev = chan2parent(desc->txd.chan);
	
	if(!desc || ! link)
		return;
		
	synced = link->synced;
	
	if(synced)/* sync for cpu so it can be accessed */
		dma_sync_single_for_cpu(dev,
				link->phys, sizeof(link->lli),
				DMA_TO_DEVICE);

	printk(KERN_NOTICE "mv61_chain descriptor @%p:\n", link);

	SHOWVARW(link->lli.SrcAddr);
	SHOWVARW(link->lli.DestAddr);
	SHOWVARW(link->lli.OwnLength);
	SHOWVARW(link->lli.NextCtrl);
	SHOWVARWA(link->chain_node.next);
	SHOWVARWA(link->chain_node.prev);
	SHOWVARW(link->phys);
	
	if(synced) /* restore to original state */
		dma_sync_single_for_device(chan2parent(desc->txd.chan),
				link->phys, sizeof(link->lli),
				DMA_TO_DEVICE);
}
/**
 * mv61_desc_dump - dump a single transaction descriptor
 * @desc: virtual address of descriptor to dump
 */
void mv61_desc_dump(struct mv61_desc *desc, char *note)
{
	printk(KERN_NOTICE "mv61_desc descriptor %s@%p:\n", note, desc);
	
	SHOWVARW(desc->lli.SrcAddr);
	SHOWVARW(desc->lli.DestAddr);
	SHOWVARW(desc->lli.OwnLength);
	SHOWVARW(desc->lli.NextCtrl);
	SHOWVARWA(desc->subchain.next);
	SHOWVARWA(desc->subchain.prev);
	SHOWVARW(desc->status);
	SHOWVARW(desc->txregs.CFG);
	SHOWVARW(desc->txregs.FillValue);
	SHOWVARW(desc->txregs.intEn);
	SHOWVARW(desc->txregs.TimerControl);
	SHOWVARW(desc->txd.cookie);
	SHOWVARW(desc->txd.flags);
	SHOWVARW(desc->txd.phys);
	SHOWVARW(desc->txd.chan);
	SHOWVARW(desc->txd.tx_submit);
	SHOWVARW(desc->txd.callback);
	SHOWVARW(desc->txd.callback_param);
	SHOWVARW(desc->len);
	SHOWVARWA(desc->desc_node.next);
	SHOWVARWA(desc->desc_node.prev);
	SHOWVARWA(desc->tx_list.next);
	SHOWVARWA(desc->tx_list.prev);
	SHOWVARW(desc->active);
	SHOWVARW(desc->chains);
	SHOWVARW(desc->links);
	SHOWVARW(desc->paused_link);
}

/**
 * mv61_chain_dump - dump a descriptor subchain
 * @desc: top level transaction control structure
 */
void mv61_chain_dump(struct mv61_desc *desc)
{
	struct mv61_chain *link, *_link;
	
	if(!desc)
		return;
	mv61_desc_dump(desc, "");	
	if(list_empty(&desc->subchain))
		return;
	list_for_each_entry_safe(link, _link, &desc->subchain, chain_node) {
		mv61_link_dump(link, desc);
	}
}

/**
 * mv61_tx_dump - dump a transaction descriptor tree
 * @desc: top level transaction control structure
 */
void mv61_tx_dump(struct mv61_desc *desc)
{
	struct mv61_desc *child, *_child;

	if (desc) {
		mv61_chain_dump(desc);

		list_for_each_entry_safe(child, _child, &desc->tx_list, tx_list) {
			mv61_chain_dump(child);
		}
	}
}

void mv61_dump_pchan(struct mv61_pdma_chan_regs *pcregs)
{
	unsigned int tmp;
	
	printk(KERN_NOTICE "mv61_dump_pchan\n");
	tmp = readl(&pcregs->CFG);
	printk(KERN_NOTICE "CFG@%p = 0x%08x\n", &pcregs->CFG, tmp);
	tmp = readl(&pcregs->Control);
	printk(KERN_NOTICE "Control@%p = 0x%08x\n", &pcregs->Control, tmp);
	tmp = readl(&pcregs->Status);
	printk(KERN_NOTICE "Status@%p = 0x%08x\n", &pcregs->Status, tmp);
	tmp = readl(&pcregs->CPR);
	printk(KERN_NOTICE "CPR@%p = 0x%08x\n", &pcregs->CPR, tmp);
	tmp = readl(&pcregs->CDR);
	printk(KERN_NOTICE "CDR@%p = 0x%08x\n", &pcregs->CDR, tmp);
	tmp = readl(&pcregs->CNDAR);
	printk(KERN_NOTICE "CNDAR@%p = 0x%08x\n", &pcregs->CNDAR, tmp);
	tmp = readl(&pcregs->FillValue);
	printk(KERN_NOTICE "FillValue@%p = 0x%08x\n", &pcregs->FillValue, tmp);
	tmp = readl(&pcregs->intEn);
	printk(KERN_NOTICE "intEn@%p = 0x%08x\n", &pcregs->intEn, tmp);
	tmp = readl(&pcregs->intPend);
	printk(KERN_NOTICE "intPend@%p = 0x%08x\n", &pcregs->intPend, tmp);
	tmp = readl(&pcregs->intAck);
	printk(KERN_NOTICE "intAck@%p = 0x%08x\n", &pcregs->intAck, tmp);
	tmp = readl(&pcregs->intForce);
	printk(KERN_NOTICE "intForce@%p = 0x%08x\n", &pcregs->intForce, tmp);
	tmp = readl(&pcregs->TimerControl);
	printk(KERN_NOTICE "TimerControl@%p = 0x%08x\n", &pcregs->TimerControl, tmp);
	tmp = readl(&pcregs->TimeOutStat);
	printk(KERN_NOTICE "TimeOutStat@%p = 0x%08x\n", &pcregs->TimeOutStat, tmp);
	tmp = readl(&pcregs->CRBAR);
	printk(KERN_NOTICE "CRBAR@%p = 0x%08x\n", &pcregs->CRBAR, tmp);
	tmp = readl(&pcregs->CRBLR);
	printk(KERN_NOTICE "CRBLR@%p = 0x%08x\n", &pcregs->CRBLR, tmp);
	tmp = readl(&pcregs->CWBAR);
	printk(KERN_NOTICE "CWBAR@%p = 0x%08x\n", &pcregs->CWBAR, tmp);
	tmp = readl(&pcregs->CWBLR);
	printk(KERN_NOTICE "CWBLR@%p = 0x%08x\n", &pcregs->CWBLR, tmp);
	tmp = readl(&pcregs->CWBRR);
	printk(KERN_NOTICE "CWBRR@%p = 0x%08x\n", &pcregs->CWBRR, tmp);
	tmp = readl(&pcregs->CSRR);
	printk(KERN_NOTICE "CSRR@%p = 0x%08x\n", &pcregs->CSRR, tmp);
	tmp = readl(&pcregs->CSRLI);
	printk(KERN_NOTICE "CSRLI@%p = 0x%08x\n", &pcregs->CSRLI, tmp);
	tmp = readl(&pcregs->CSRUI);
	printk(KERN_NOTICE "CSRUI@%p = 0x%08x\n", &pcregs->CSRUI, tmp);
	tmp = readl(&pcregs->CRSL);
	printk(KERN_NOTICE "CRSL@%p = 0x%08x\n", &pcregs->CRSL, tmp);
	tmp = readl(&pcregs->CRSU);
	printk(KERN_NOTICE "CRSU@%p = 0x%08x\n", &pcregs->CRSU, tmp);
	tmp = readl(&pcregs->CAFL);
	printk(KERN_NOTICE "CAFL@%p = 0x%08x\n", &pcregs->CAFL, tmp);
	tmp = readl(&pcregs->CAFU);
	printk(KERN_NOTICE "CAFU@%p = 0x%08x\n", &pcregs->CAFU, tmp);
}

void mv61_cdma_dumpall(struct mv61_dma *mv61p)
{
	int i;
	 
	printk(KERN_NOTICE "mv61_cdma_dumpall\n");
	if(mv61p->CDMAInt) {
		unsigned int status;
		status = readl(mv61p->CDMAInt);
		printk(KERN_NOTICE "interrupt: status=0x%x\n", status);
	}

	for (i = 0; i < mv61p->pchannels; i++) {	
		mv61_dump_pchan((struct mv61_pdma_chan_regs *)mv61p->ch_regs[i]);
	}
}

