/*
 * Driver for the Marvell 61xx CDMA Controller
 *
 * Derived from drivers/dma/dw_dmac.c
 *
 * Copyright (C) 2007-2008 Atmel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/delay.h>
#include <linux/dmaengine.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/jiffies.h>
#include <asm/atomic.h>
#include "cdma.h"
/*
 * This supports the Marvell Central DMA Controller used in the 
 * Marvell 88PA61x0, which does not support descriptor writeback.
 */

/*----------------------------------------------------------------------*/
/**
 * mv61vc_resume_dma - resume a paused virtual channel
 * @mv61vc: virtual channel control structure
 * @flowcontrol: transfer type
 *
 * Call with mv61p->biglock held and irqs disabled.
 * Call with mv61vc->lock held and irqs disabled.
 *
 */
int mv61vc_resume_dma(struct mv61_vdma_chan *mv61vc, enum mv61_dma_flow_ctrl	
								flowcontrol)
{
	struct mv61_pdma_chan		*mv61pc;
	struct mv61_pdma_chan_regs 	*pcregs;
	struct mv61_desc	*topdesc;
	struct mv61_desc	*activedesc;
	struct mv61_lli		*lli;
	struct mv61_desc	dummydesc;
	u32 tmp;
	int memsrc;
	int transferred = 0;

	mv61pc = mv61_vpmap_v_to_p(mv61vc);
	if(!mv61pc)
		return -EINVAL;
	
	pcregs = mv61pc->ch_regs;
	if(!pcregs)
		return -EINVAL;

	topdesc = mv61vc_first_active(mv61vc);
	if(!topdesc) /* nothing to do */
		return 0;

	if(dumppausedesc)
		mv61_desc_dump(topdesc, "topdesc");

	activedesc = topdesc->active;
	if(!activedesc)
		return -EFAULT;
		
	if(dumppausedesc && (activedesc != topdesc))
		mv61_desc_dump(activedesc, "activedesc");

	/* determine which register image to get addresses from */
	if(activedesc->paused_link)
		lli = &activedesc->paused_link->lli; /* originally auto-loaded */
	else
		lli = &activedesc->lli; /* originally direct-loaded by software */
	
	if(!lli)
		return -EFAULT;
	
	memset((void *)&dummydesc, 0, sizeof(struct mv61_desc));
	
	memsrc = ((flowcontrol == MV61_DMA_MEMORY_TO_PERIPHERAL) ||
			(flowcontrol == MV61_DMA_MEMORY_TO_MEMORY)) ? 1 : 0; 
	
	/* 
	 * Determine the residue for the link descriptor that was in progress.
	 * The same link may have been paused and restarted before, so use its original transfer
	 * size, not the current request size.
	 */
	transferred = (lli->OwnLength & DESCRIPTOR_OWNLENGTH_LENGTH_MASK) - readl(&pcregs->CWBRR);
	
	dummydesc.txregs.CFG 		= activedesc->txregs.CFG;
	dummydesc.txregs.FillValue	= activedesc->txregs.FillValue;
	dummydesc.txregs.intEn		= activedesc->txregs.intEn;
	dummydesc.txregs.TimerControl	= activedesc->txregs.TimerControl;
	
	tmp = mv61vc->hwstat.Control;
	
	dummydesc.lli.OwnLength = readl(&pcregs->CWBRR) & DESCRIPTOR_OWNLENGTH_LENGTH_MASK;

	/*
	 * For non-incrementing addresses, or source addresses in memory,
	 * AddrStat may point beyond the actual address.
	 */
	if(tmp & CDMA_CONTROL_DESTADDRINC_MASK)
		dummydesc.lli.DestAddr = lli->DestAddr + transferred;
	else
		dummydesc.lli.DestAddr = lli->DestAddr;

	if(tmp & CDMA_CONTROL_SRCADDRINC_MASK)
		dummydesc.lli.SrcAddr = lli->SrcAddr + transferred;
	else
		dummydesc.lli.SrcAddr = lli->SrcAddr;

	if(dumppausedesc)
		mv61_desc_dump(&dummydesc, "dummydesc");
	
	/* don't need to update the cookie for a restart */
	return mv61vc_dostart(mv61vc, &dummydesc);
}

/**
 * mv61vc_check_residue - clean up paused virtual channel and check bytes remaining
 * @mv61vc: virtual channel control structure
 *
 * This will walk the descriptor chain of the transaction that was in
 * progress when the channel was paused. Subchain links that
 * are completed or in progress will not be auto-loaded again, so they
 * will be unmapped.
 *
 * Pausing is currently supported for VDMA_OWNED channels only, but
 * this function is written to support all channels types in case it is needed
 * later.
 *
 * Call with mv61p->biglock held and irqs disabled.
 * Call with mv61vc->lock held and irqs disabled.
 *
 */
int mv61vc_check_residue(struct mv61_vdma_chan *mv61vc)
{
	struct mv61_pdma_chan		*mv61pc;
	struct mv61_pdma_chan_regs 	*pcregs;
	struct mv61_desc	*topdesc;
	struct mv61_desc	*activedesc;
	struct mv61_desc	*desc = NULL;
	struct mv61_chain 	*prevlink = NULL;
	int 			transferred = 0;
	struct mv61_lli		*lli = NULL;

	__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d\n", __func__, __LINE__);

	if(mv61vc->status == DMA_IN_PROGRESS)
		return -EBUSY;
	
	topdesc = mv61vc_first_active(mv61vc);
	if(!topdesc) /* nothing to do */
		return 0;

	activedesc = topdesc->active;
	if(!activedesc)
		return -EFAULT;

	mv61pc = mv61_vpmap_v_to_p(mv61vc);
	if(!mv61pc)
		return -EINVAL;

	pcregs = mv61pc->ch_regs;
	if(!pcregs)
		return -EINVAL;

	/* tally the completed subchains and find the first incomplete subchain */
	if(activedesc == topdesc) {
		desc = topdesc;
		__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d\n", __func__, __LINE__);
	}
	else {
		struct mv61_desc *child, *_child;
		
		transferred += topdesc->sublen;
		__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d: transferred tally = %d\n",
						__func__, __LINE__, transferred);
		
		list_for_each_entry_safe(child, _child, &topdesc->tx_list, tx_list) {
			if(activedesc == child) {
				__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d\n",
								__func__, __LINE__);
				/* this descriptor is still in progress */
				desc = child;
				break;
			}
			else {
				
				/* this descriptor is complete */
				transferred += child->sublen;
				__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d: transferred "
							"tally = %d\n",
							__func__, __LINE__, transferred);
			}
		}
	}
	
	/* desc should now point to the first potentially incomplete subchain */
	if(!desc) {
		dev_err(chan2dev(&mv61vc->chan), "%s:%d:error:desc null\n",
							__func__, __LINE__);
		return -EFAULT;
	}
	
	desc->paused_link = NULL;
	prevlink = NULL;
		
	/* tally the completed links in the subchain and  find the first incomplete link */
	if(!list_empty(&desc->subchain)) {
		struct mv61_chain *link, *_link;

		list_for_each_entry_safe(link, _link, &desc->subchain, chain_node) {

			if(link->phys == mv61vc->hwstat.CDR) {
				__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d\n",
								__func__, __LINE__);
				/* Found first link that has not been auto-loaded yet */
				break;
			}
			else {
				/*
				 * Since current link was autoloaded, previous link
				 * was completed.
				 *
				 * Even if a paused link was originally autoloaded,
				 * an adjusted  dummy descriptor must be directly 
				 * loaded to restart it. The dummy is then 
				 * discarded. The dummy will change the addresses
				 * and transfer size, but will restore the original 
				 * lli pointer in the LLIR register..
				 *
				 * For residue calculation, the dummy descriptor is
				 * ignored (it ceases to exist after restarting).
				 * Everything is determined by the original 
				 * transaction descriptor chain.
				 */
				
				if(prevlink) {
					/* 
					 * Previous link was autoloaded
					 * member of struct mv61_chain.
					 */
					lli = &prevlink->lli;
				}
				else {
					/* 
					 * Previous link was directly loaded 
					 * member of struct mv61_desc (top
					 * of a subchain).
					 */
					lli = &desc->lli;
				}
				if(!lli) {
					dev_err(chan2dev(&mv61vc->chan),
							"%s:%d:error:lli null\n",
							__func__, __LINE__);
					
					return -EFAULT;
				}	

				transferred += lli->OwnLength & DESCRIPTOR_OWNLENGTH_LENGTH_MASK;

				__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d: "
							"transferred tally"
							" = %d\n", __func__,
							__LINE__, transferred);
			}
			/*
			 * This link has already been auto-loaded. Sync for cpu
			 * so the register fields can be accessed.
			 */
			if(link->synced) {
				__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d\n",
								__func__, __LINE__);
				dma_sync_single_for_cpu(chan2parent(desc->txd.chan), 
							link->phys, sizeof(link->lli),
							DMA_TO_DEVICE);
				link->synced = 0;
			}

			prevlink = link;
		}
		
	}

	desc->paused_link = prevlink;

	if(desc->paused_link)
		lli = &desc->paused_link->lli; /* originally auto-loaded */
	else
		lli = &desc->lli; /* originally direct-loaded by software */
	
	if(!lli)
		return -EFAULT;
		
	/* 
	 * Determine the residue for the link descriptor that was in progress.
	 * The same link may have been paused and restarted before, so use its 
	 * original transfer size, not the current request size.
	 */
	transferred += (lli->OwnLength & DESCRIPTOR_OWNLENGTH_LENGTH_MASK) - readl(&pcregs->CWBRR);
	
	__dev_vrdbg(chan2dev(&mv61vc->chan), "%s:%d: transferred tally = %d\n",
						 __func__, __LINE__, transferred);
	if(transferred > topdesc->len)
		return -EIO;;
				
	return (topdesc->len - transferred);
}

/**
 * mv61vc_check_residue_running - attempt to check bytes remaining while running
 * @mv61vc: virtual channel control structure
 *
 * This is only supported for a single-descriptor transaction. It is
 * inherently race-prone, so the results are only likely to be meaningful
 * if the peripheral is idle.
 *
 * For destination in memory, some received data could still be sitting in
 * the fifo. Not sure what to do about it.  
 *
 * Call with mv61p->biglock held and irqs disabled.
 * Call with mv61vc->lock held and irqs disabled.
 *
 */
int mv61vc_check_residue_running(struct mv61_vdma_chan *mv61vc)
{
	struct mv61_pdma_chan		*mv61pc;
	struct mv61_pdma_chan_regs 	*pcregs;
	struct mv61_desc		*desc;
	struct mv61_vdma		*mv61v = mv61vc->mv61v;
	int ret = 0;

	desc = mv61vc_first_active(mv61vc);
	if(!desc)
		return 0;
		
	if((desc->chains > 1) || (desc->links > 1)) {
		dev_err(chan2dev(&mv61vc->chan), "%s:%d: not supported for this transaction\n", __func__, __LINE__);
		return -EINVAL;
	}
	
	if(mv61v->vtype != MV61_VDMA_OWNED) {
		dev_err(chan2dev(&mv61vc->chan), "%s:%d: not supported for this channel\n", __func__, __LINE__);
		return -EINVAL;
	}
			
	mv61pc = mv61_vpmap_v_to_p(mv61vc);
	if(!mv61pc)
		return -EINVAL;
	
	pcregs = mv61pc->ch_regs;
	if(!pcregs)
		return -EINVAL;
	
	ret = readl(&pcregs->CWBRR);
	
	return ret;
}
