/*
 * Device Tree support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/io.h>
#include <linux/irqchip.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/slab.h>
#include <asm/cacheflush.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>
#include <asm/smp_plat.h>
#include <asm/suspend.h>
#include "apmu.h"
#include "ciu.h"
#include "apb_config.h"

extern void pegmatite_secondary_startup(void);
extern void pegmatite_smp_jump(void);
extern unsigned long pegmatite_smp_jump_table;
extern unsigned long pegmatite_smp_jump_size;

extern void __ref pegmatite_cpu_die(unsigned int cpu);
extern int pegmatite_cpus_kill(const cpumask_t *cpus);
extern void pegmatite_cpu_resume_entry(void);

/*
 * Write pen_release in a way that is guaranteed to be visible to all
 * observers, irrespective of whether they're taking part in coherency
 * or not.  This is necessary for the hotplug code to work reliably.
 */
static void write_pen_release(int val)
{
	pen_release = val;
	smp_wmb();
	__cpuc_flush_dcache_area((void *)&pen_release, sizeof(pen_release));
	outer_clean_range(__pa(&pen_release), __pa(&pen_release + 1));
}

static DEFINE_SPINLOCK(boot_lock);
static unsigned long *pegmatite_boot_addr;

/*
 * Boot a CPU to "address"
 */
void pegmatite_boot_cpus_to(const cpumask_t *cpus, unsigned long address)
{
	int cpu;

	if(is_fpga()) {
		int *a53address_addr;
		a53address_addr = ioremap_nocache(0xfd00006c, 4);
		writel(0x00000003, a53address_addr);
		iounmap(a53address_addr);
	}

        // mike patch to always give cpu0 a boot vector.
	writel(address, &pegmatite_boot_addr[0]);
        // mike patch to always give cpu0 boot from SQU
        setup_boot_from_squ(0);
        // end patch
	for_each_cpu(cpu, cpus)
		writel(address, &pegmatite_boot_addr[cpu]);
	__cpuc_clean_dcache_area((void *)pegmatite_boot_addr, sizeof(pegmatite_boot_addr[0]) * num_present_cpus());

	/* Configure the CIU so the core's reset vector is the SQU */
	for_each_cpu(cpu, cpus)
		setup_boot_from_squ(cpu);

	/* Bring the core out of reset */
	pegmatite_boot_cpus(cpus);
}

/*
 * platform-specific code to suspend a CPU
 *
 * Called with IRQs disabled on the dying CPU. The CPU can be resumed with
 * pegmatite_cpu_resume() from another CPU.
 */
void pegmatite_cpu_suspend(void)
{
	cpu_suspend(0, (int(*)(unsigned long))pegmatite_cpu_die);
}
EXPORT_SYMBOL(pegmatite_cpu_suspend);

/*
 * platform-specific code to resume a CPU
 */
void pegmatite_cpus_resume(const cpumask_t *cpus)
{
	pegmatite_boot_cpus_to(cpus, virt_to_phys(&pegmatite_cpu_resume_entry));
}

static int pegmatite_boot_secondary(unsigned int cpu, struct task_struct *idle)
{
	unsigned long timeout;

	/*
	 * Set synchronisation state between this boot processor
	 * and the secondary one
	 */
	spin_lock(&boot_lock);

	write_pen_release(cpu_logical_map(cpu));

	pegmatite_boot_cpus_to(cpumask_of(cpu), virt_to_phys(&pegmatite_secondary_startup));

	timeout = jiffies + (1 * HZ);
	while (time_before(jiffies, timeout)) {
		smp_rmb();
		if (pen_release == -1)
			break;

		udelay(10);
	}

	/*
	 * now the secondary core is starting up let it run its
	 * calibrations, then wait for it to finish
	 */
	spin_unlock(&boot_lock);

	return pen_release != -1 ? -ENOSYS : 0;
}

static void pegmatite_secondary_init(unsigned int cpu)
{
	/*
	 * let the primary processor know we're out of the
	 * pen, then head off into the C entry point
	 */
	write_pen_release(-1);

	/*
	 * Synchronise with the boot thread.
	 */
	spin_lock(&boot_lock);
	spin_unlock(&boot_lock);
}

static void __init pegmatite_smp_prepare_cpus(unsigned int max_cpus)
{
	struct device_node *node;
	struct resource res;
	int ret;
	int cpu;
	void __iomem *squ_addr = NULL;

	node = of_find_compatible_node(NULL, NULL, "marvell,pegmatite-smpboot-sram");
	if (!node) {
		pr_err("%s: could not find sram dt node\n", __func__);
		goto err;
	}
	ret = of_address_to_resource(node, 0, &res);
	if (ret < 0) {
		pr_err("%s: could not get address for node %s\n",
		       __func__, node->full_name);
		goto err;
	}

	if (resource_size(&res) < pegmatite_smp_jump_size) {
		pr_err("%s: invalid sram reservation\n", __func__);
		goto err;
	}

	/*
	 * pegmatite_smp_jump includes the instructions needed to get us from
	 * the A53's reset vector to pegmatite_secondary_startup.
	 *
	 * The jump address is the 4 bytes immediately after it, referenced by
	 * pegmatite_smp_jump_address.
	 *
	 * pegmatite_smp_jump and the address of pegmatite_secondary_startup
	 * are copied to the cpu's reset vector at 0xd1000000.  This address
	 * is the first bank of the SQU.
	 *
	 * Write the address of pegmatite_secondary_startup before copying the
	 * section containing the code and the address.
	 *
	 */

	squ_addr = of_iomap(node, 0);
	if (!squ_addr)
		goto err;

	pegmatite_boot_addr = kzalloc(
		sizeof(pegmatite_boot_addr[0]) * num_present_cpus(),
		GFP_KERNEL);
	if (!pegmatite_boot_addr) {
		pr_err("Failed to allocate CPU jump table\n");
		goto err;
	}

	pegmatite_smp_jump_table = virt_to_phys(pegmatite_boot_addr);

	/* Copy the jump instructions from pegmatite_smp_jump to the SQU */
	memcpy(squ_addr, &pegmatite_smp_jump, pegmatite_smp_jump_size);
	wmb();
	iounmap(squ_addr);

	return;
err:
	if (squ_addr)
		iounmap(squ_addr);

	for_each_present_cpu(cpu) {
		if (cpu == smp_processor_id())
			continue;
		set_cpu_present(cpu, 0);
		pr_warn("%s: Disabling SMP\n", __func__);
	}
}

static int pegmatite_cpu_kill(unsigned int cpu)
{
	return pegmatite_cpus_kill(cpumask_of(cpu));
}

struct smp_operations pegmatite_smp_ops __initdata = {
	.smp_prepare_cpus	= pegmatite_smp_prepare_cpus,
	.smp_secondary_init	= pegmatite_secondary_init,
	.smp_boot_secondary	= pegmatite_boot_secondary,
#ifdef CONFIG_HOTPLUG_CPU
	.cpu_die		= pegmatite_cpu_die,
	.cpu_kill               = pegmatite_cpu_kill,
#endif
};

