/*
 * AP PMU support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 * The APMU provides various registers to control Power Management
 * functions related to the Application Processing Subsystem of the SoC
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/io.h>
#include <linux/smp.h>
#include <linux/delay.h>
#include <asm/smp_plat.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/cpumask.h>
#include "apb_config.h"

static void __iomem *apmu_base;
static unsigned int apmu_clkrstgen_cpu_misc;
static unsigned int apmu_cpuclkstatus;

#define APMU_REV0_A 0x158
#define APMU_REV0_B 0x164
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_A 0x338
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_B 0x568
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK 0x10000
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK 0x40000
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK(x) (1 << (x + 4))
#define APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK(x) (1 << x)

#define APMU_MISC_A53COREPWRISLAND_PICR(x) (0x8 * x)
#define APMU_MISC_A53COREPWRISLAND_PISR(x) (0x8*x + 0x4)

#define APMU_MISC_GPUPWRISLAND_PICR 0x30
#define APMU_MISC_GPUPWRISLAND_PISR 0x34

#define APMU_ISOLATION 0
#define APMU_ISOLATION_MASK(x) (x << APMU_ISOLATION)
#define APMU_ISOLATION_ENABLE_MASK (1 << APMU_ISOLATION)
#define ISOLATION_ENABLE 0
#define ISOLATION_DISABLE 1

#define APMU_BULKPWRUP 4
#define APMU_BULKPWRUP_MASK(x) (x << APMU_BULKPWRUP)
#define POWER_RAMP_UP 1
#define SWITCH_FULL_ON 3

#define APMU_DISTPWRUP 8
#define APMU_DISTPWRUP_MASK(x) (x << APMU_DISTPWRUP)
#define SWITCH_OFF 0
#define SWITCH_ON 1

#define APMU_DBGPWRDUP 9
#define APMU_DBGPWRDUP_MASK(x) (x << APMU_DBGPWRDUP)
#define DBGIND_OFF 0
#define DBGIND_ON 1

#define TIMEOUT 10

#define APMU_CPUCLKSTATUS_A 0x340
#define APMU_CPUCLKSTATUS_B 0x570
#define APMU_CPUCLKSTATUS_STANDBYWFI_MASK(x) (1 << (x + 4))

static const struct of_device_id of_apmu_table[] = {
	{.compatible = "marvell,pegmatite-apmu"},
	{ /* end of list */ },
};

int is_reva(void) {
	if(readl(apmu_base + APMU_REV0_A) == 0x00010002)
		return 1;
	else
		return 0;
}

#ifdef CONFIG_SMP
static void pegmatite_core_powerup(const cpumask_t *cpus)
{
	u32 val;
	int i, cpu;

	/* Read the AP CPU Misc Control register */
	val = readl(apmu_base + apmu_clkrstgen_cpu_misc);

	/* Make sure resets are asserted */
	for_each_cpu(cpu, cpus) {
		val &= ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK(cpu);
		val &= ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK(cpu);
	}
	writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

	/* Read the Power Island Control register */
	for_each_cpu(cpu, cpus) {
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));

		/* Clear the bulk power up field */
		val &= ~APMU_BULKPWRUP_MASK(SWITCH_FULL_ON);

		/* Set bulk power for power ramp up */
		val |= APMU_BULKPWRUP_MASK(POWER_RAMP_UP);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	udelay(20);

	/* Poll the power ramp up bit in the Power Island Status register */
	for_each_cpu(cpu, cpus) {
		i = 0;
		while((readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PISR(cpu)) & APMU_BULKPWRUP_MASK(POWER_RAMP_UP)) != APMU_BULKPWRUP_MASK(POWER_RAMP_UP)) {
			i++;
			if(i == TIMEOUT) {
				pr_warn("%s: Timeout reached waiting on CPU%u ramp up\n",
					__func__, cpu);
				break;
			}
		}
	}

	for_each_cpu(cpu, cpus) {
		/* Set bulk power for switch full on */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val |= APMU_BULKPWRUP_MASK(SWITCH_FULL_ON);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));

		/* Set the DISTPWRUP field to 0x0 (for RevA, RevB is inverted) */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		if(is_reva()) {
			val &= ~(APMU_DISTPWRUP_MASK(SWITCH_ON));
		} else {
			val |= APMU_DISTPWRUP_MASK(SWITCH_ON);
		}
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	/* Poll the switch full on bits in the Power Island Status register */
	for_each_cpu(cpu, cpus) {
		i = 0;
		while((readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PISR(cpu)) & APMU_BULKPWRUP_MASK(SWITCH_FULL_ON)) != APMU_BULKPWRUP_MASK(SWITCH_FULL_ON)) {
			i++;
			if(i == TIMEOUT) {
				pr_warn("%s: Timeout reached waiting on CPU%u switch full on\n",
					__func__, cpu);
				break;
			}
		}
	}

	/* Disable the isolation buffer */
	for_each_cpu(cpu, cpus) {
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val |= APMU_ISOLATION_MASK(ISOLATION_DISABLE);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	/* Poll isolation bit in the Power Island Status register */
	for_each_cpu(cpu, cpus) {
		i = 0;
		while((readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PISR(cpu)) & APMU_ISOLATION_MASK(ISOLATION_DISABLE)) != APMU_ISOLATION_MASK(ISOLATION_DISABLE)) {
			i++;
			if(i == TIMEOUT) {
				pr_warn("%s: Timeout reached waiting on CPU%u de-isolation\n",
					__func__, cpu);
				break;
			}
		}
	}

	/* Set Debug Power Up indicator */
	for_each_cpu(cpu, cpus) {
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val |= APMU_DBGPWRDUP_MASK(DBGIND_ON);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	udelay(1);
}

void pegmatite_cores_powerdown(const cpumask_t *cpus)
{
	int cpu;
	int val;
	int i;

        /* Read the AP CPU Misc Control register */
        val = readl(apmu_base + apmu_clkrstgen_cpu_misc);
        /* Make sure resets are asserted */
        for_each_cpu(cpu, cpus) {
                val &= ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK(cpu);
                val &= ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK(cpu);
        }
        writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

        for_each_cpu(cpu, cpus) {
		/* Clear Debug Power Up indicator */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val &= ~APMU_DBGPWRDUP_MASK(DBGIND_ON);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));

		/* Enable the Isolation buffer */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val &= ~APMU_ISOLATION_ENABLE_MASK;
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	for_each_cpu(cpu, cpus) {
		/* Poll isolation bit in the power island status register */
		i = 0;
		while((readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PISR(cpu)) & APMU_ISOLATION_ENABLE_MASK) != ISOLATION_ENABLE) {
			i++;
			if(i == TIMEOUT) {
				pr_warn("%s: Timeout reached waiting on CPU%u isolation\n",
					__func__, cpu);
				break;
			}
		}
	}

	for_each_cpu(cpu, cpus) {
		/* Clear the bulk power up field */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		val &= ~APMU_BULKPWRUP_MASK(SWITCH_FULL_ON);
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));

		/* Set the DISTPWRUP field to 0x1 (for RevA, RevB is inverted) */
		val = readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
		if(is_reva()) {
			val |= APMU_DISTPWRUP_MASK(SWITCH_ON);
		} else {
			val &= ~(APMU_DISTPWRUP_MASK(SWITCH_ON));
		}
		writel(val, apmu_base + APMU_MISC_A53COREPWRISLAND_PICR(cpu));
	}

	for_each_cpu(cpu, cpus) {
		/* Poll the switch full on bits in the Power Island Status register */
		i = 0;
		while((readl(apmu_base + APMU_MISC_A53COREPWRISLAND_PISR(cpu)) & APMU_BULKPWRUP_MASK(SWITCH_FULL_ON)) != SWITCH_OFF) {
			i++;
			if(i == TIMEOUT) {
				pr_warn("%s: Timeout reached waiting on CPU%u switch full on\n",
					__func__, cpu);
				break;
			}
		}
	}
}

int pegmatite_check_wfi(const cpumask_t *cpus)
{
	int cpu;
	u32 val;

	/* Read the CPU Clock Status register */
	val = readl(apmu_base + apmu_cpuclkstatus);

	for_each_cpu(cpu, cpus) {
		if (!(val & APMU_CPUCLKSTATUS_STANDBYWFI_MASK(cpu)))
			return 0;
	}
	return 1;
}

int pegmatite_boot_cpus(const cpumask_t *cpus)
{
	u32 val;
	int cpu;

	if (!apmu_base) {
		pr_warn("Can't boot CPU. APMU is uninitialized\n");
		return 1;
	}

	/*
	 * Power up the core
	 */
	if(!is_fpga()) {
		pegmatite_core_powerup(cpus);
	}

	/*
	 * Read the AP CPU Misc Control register
	 */
	val = readl(apmu_base + apmu_clkrstgen_cpu_misc);

	/* Deassert L2 reset if necessary */
	if (!(val & APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK)) {
		val |= APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK;
		writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

		/*
		 * The L2 reset needs to be deasserted 25 clock cycles before any other
		 * resets in this register.  We'll do 250us.
		 */
		udelay(250);
	}

	/* Debug reset */
	val |= APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK;
	writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

	/* cpuX power on reset */
	for_each_cpu(cpu, cpus)
		val |= APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK(cpu);
	writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

	/* cpuX reset */
	for_each_cpu(cpu, cpus)
		val |= APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK(cpu);
	writel(val, apmu_base + apmu_clkrstgen_cpu_misc);

	return 0;
}
#endif

static int __init pegmatite_apmu_init(void)
{
	struct device_node *np;

	np = of_find_matching_node(NULL, of_apmu_table);
	if (!np)
		return 0;

	pr_info("Initializing Pegmatite APMU\n");
	apmu_base = of_iomap(np, 0);

	/*
	 * Check for the RevA version at the RevA location.  If that doesn't
	 * match, assume RevB.
	 */
	if(is_reva()) {
		apmu_clkrstgen_cpu_misc = APMU_CLKRSTGEN_APCPUMISCCONTROL_A;
		apmu_cpuclkstatus = APMU_CPUCLKSTATUS_A;
	} else {
		apmu_clkrstgen_cpu_misc = APMU_CLKRSTGEN_APCPUMISCCONTROL_B;
		apmu_cpuclkstatus = APMU_CPUCLKSTATUS_B;
	}

	return 0;
}

early_initcall(pegmatite_apmu_init);
