/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/
#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/of_device.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/semaphore.h>
#include <linux/list.h>
#include <linux/kthread.h>
#include <linux/time.h>
#include "ipc_api.h"

MODULE_LICENSE("Dual BSD/GPL");


char *gdbgveripcdevice="ipc detect signal port automatically, build time "__DATE__":"__TIME__;
/***********************************************************************************************************/
/***********************************************************************************************************/
/***********************************************************************************************************/

struct dbgLogBuf
{
	unsigned char d1;
	unsigned short d2;
	unsigned char d3;
	unsigned int d4;
	struct timeval tv;
};
#define DBGLOGBUF_SIZ 2048
static struct dbgLogBuf sDbgLogBuf[DBGLOGBUF_SIZ];

static int sDbgLogWPos = 0;
static int sDbgLogCnt = 0;

static int getNextPos(int inpos)
{
	int ret;
	if (inpos >= DBGLOGBUF_SIZ - 1)
	{
		ret = 0;
	} else {
		ret = inpos + 1;
	}
	return ret;
}

void addDbgLog(char ind1, unsigned short ind2, char ind3, unsigned int ind4)
{
#if 0
	if (sDbgLogCnt > DBGLOGBUF_SIZ -10) {
		return;
	}
#endif	
	sDbgLogBuf[sDbgLogWPos].d1 = ind1;
	sDbgLogBuf[sDbgLogWPos].d2 = ind2;
	sDbgLogBuf[sDbgLogWPos].d3 = ind3;
	sDbgLogBuf[sDbgLogWPos].d4 = ind4;
	
	do_gettimeofday( &(sDbgLogBuf[sDbgLogWPos].tv) );
	
	sDbgLogWPos = getNextPos(sDbgLogWPos);
	sDbgLogCnt++;
}

EXPORT_SYMBOL(addDbgLog);

#if 1
#define _IPC_MEMLOG_MUST addDbgLog
#define _IPC_MEMLOG_1TIME(fmt, ...)
#define _IPC_MEMLOG_DBG(fmt, ...)
#else
#define _IPC_MEMLOG_MUST addDbgLog
#define _IPC_MEMLOG_1TIME addDbgLog
#define _IPC_MEMLOG_DBG addDbgLog
#endif

#undef ENB_PRINTKLOG
#undef ENB_PRINTKLOG_INT
#undef ENB_PRINTKLOG_SND

const char *getPortName(uint8_t port_number)
{
	const char *retstr;
	switch(port_number)
	{
	case 1:
		retstr="leci";
		break;
	case 2:
		retstr="rsci";
		break;
	case 3:
		retstr="fax ";
		break;
	case 4: /* mnakamura */
		retstr="wdog";
		break;
	case 0x8:
	case 0x9:
	case 0xa:
	case 0xb:
	case 0xc:
	case 0xd:
	case 0xe:
	case 0xf:
		retstr="sign";
		break;
	default:
		retstr="unkn";
		break;
	}
	return retstr;
}

const char *getDatatypeName(uint8_t data_type)
{
	const char *retstr;
	switch(data_type)
	{
	case 0:
		retstr="sign";
		break;
	case 1:
		retstr="leci";
		break;
	case 2:
		retstr="rsci";
		break;
	case 3:
		retstr="fax ";
		break;
	case 4: /* mnakamura */
		retstr="wdog";
		break;
	default:
		retstr="unkn";
		break;
	}
	return retstr;
}

static void ipc_printDbgLog(void)
{
	int i;
	int firstpos, num;
	int pos;

	printk(KERN_EMERG "wpos=%d, total=%d\n", sDbgLogWPos, sDbgLogCnt );

	if (sDbgLogCnt < DBGLOGBUF_SIZ) {
		firstpos = 0;
		num = sDbgLogCnt;
	} else {
		firstpos = sDbgLogWPos;
		num = DBGLOGBUF_SIZ;
	}

	for(i=0, pos=firstpos; i<num; i++, pos=getNextPos(pos))
	{
		if ( (sDbgLogBuf[pos].d1 == 'R') && (sDbgLogBuf[pos].d2 == 1) ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(port_number=%d %s) read() start :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getPortName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4 );
		} else if ( (sDbgLogBuf[pos].d1 == 'R') && (sDbgLogBuf[pos].d2 == 2) ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(port_number=%d %s) read() return 0x%8x :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getPortName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d4, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		} else if ( (sDbgLogBuf[pos].d1 == 'W') && (sDbgLogBuf[pos].d2 == 1) ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(port_number=%d %s) write() *buf=0x%8x :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getPortName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d4, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		} else if ( (sDbgLogBuf[pos].d1 == 'W') && (sDbgLogBuf[pos].d2 == 2) ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(port_number=%d %s) write() count=%d start :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getPortName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d4, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		} else if ( sDbgLogBuf[pos].d1 == 'i' ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(data_type  =%d %s) A53 recv 0x%8x :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getDatatypeName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d4, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		} else if ( (sDbgLogBuf[pos].d1 == 'w') && (sDbgLogBuf[pos].d2 == 1) ) {
			printk(KERN_EMERG "%d.%d:[%3d]=(data_type  =%d %s) A53 send 0x%8x :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d3, getDatatypeName(sDbgLogBuf[pos].d3), sDbgLogBuf[pos].d4, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		} else {
			printk(KERN_EMERG "%d.%d:[%3d]=printdbglog :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].tv.tv_sec, sDbgLogBuf[pos].tv.tv_usec, pos, sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
		}
//		printk(KERN_EMERG " :(%c, %4d, %3d, %8x)\n", sDbgLogBuf[pos].d1, sDbgLogBuf[pos].d2, sDbgLogBuf[pos].d3 ,sDbgLogBuf[pos].d4);
	}
}

EXPORT_SYMBOL(ipc_printDbgLog);

/***********************************************************************************************************/
/***********************************************************************************************************/
/***********************************************************************************************************/

#define _FILE_NAME_ "ipc_driver.c"

#define IPC_NAME "ipc"
#define IPC_COMPATIBILITY_NAME "mrvl,IPC"

#define PREFIX IPC_NAME ": "

#define ENTER() pr_debug(PREFIX "ENTER %s\n", __FUNCTION__)
#define EXIT()  pr_debug(PREFIX "EXIT  %s:%d\n", __FUNCTION__, __LINE__)

#define IIR_ACK_SHIFT ( 9 )
#define IIR_ACK_MASK  ( 0x03 << IIR_ACK_SHIFT )

#define ACK_MSG_PROCESSED ( 0x3 )
#define ACK_MSG_DISCARDED ( 0x2 )

#define IIR_CMD_SHIFT ( 8 )
#define IIR_CMD_MASK  ( 1 << IIR_CMD_SHIFT )

#define IIR_PORT_SHIFT ( 0 )
#define IIR_PORT_MASK  ( 0xFF << IIR_PORT_SHIFT )

typedef struct IPC0_REGS_s
{
  volatile uint32_t IPC_ISRR;  ///< 0x0 [R]: IPC_ISRR
  volatile uint32_t IPC_WDR_0;  ///< 0x4 [W]: Write Data Register 0
  volatile uint32_t IPC_WDR_1;  ///< 0x8 [W]: Write Data Register 1
  volatile uint32_t IPC_ISRW;  ///< 0xc [W]: Interrupt Set Register Write
  volatile uint32_t IPC_ICR;  ///< 0x10 [W]: Interrupt Clear Register
  volatile uint32_t IPC_IIR;  ///< 0x14 [R]: Interrupt Identification Register
  volatile uint32_t IPC_RDR_0;  ///< 0x18 [R]: Read Data Register 0
  volatile uint32_t IPC_RDR_1;  ///< 0x1c [R]: Read Data Register 1
  volatile uint32_t IPC_MAJ_MID_REV;  ///< 0x20 [R]: Revision (Major and Mid) Register
  volatile uint32_t IPC_CFG_REV;  ///< 0x24 [R]: Revision (Configuration) Register
  volatile uint32_t IPC_DUMMY;  ///< 0x28 [W]: Dummy Register
} IPC0_REGS_t, IPC1_REGS_t; /* omo */

/* end of header file */

struct ipc_port_config_s;

typedef struct
{
    struct platform_device *pdev;
    char         dev_name[21];
    uint32_t     instance_id;
    IPC0_REGS_t *regs;
    uint32_t     int_num;
    int          open_count;
    struct ipc_port_config_s *open_ports;
//    struct semaphore tx_ready_sem;
//    struct semaphore tx_done_sem;
    uint8_t      ack_type;
} ipc_device_config_t;

typedef struct ipc_port_config_s
{
    struct list_head     list;
    ipc_device_config_t *ipc_device;
    uint8_t              port_number;
    ipc_recv_callback    recv_callback;
    void                *user_param;
} ipc_port_config_t;

static ipc_port_config_t *sFirstPort = NULL;

uint8_t ipc_get_port_number(ipc_drvr_handle handle)
{
	uint8_t ret;
	ipc_port_config_t *port = ( ipc_port_config_t * )handle;
	ret = port->port_number;
	
#ifdef ENB_PRINTKLOG
	printk("sai:%s(%d)%s:%x\n", _FILE_NAME_, __LINE__, __func__, ret);
#endif
	return ret;
}
EXPORT_SYMBOL(ipc_get_port_number);

//use for temporary buffer(early times)
#define TMP_BUF_MAX 32
static int sTmpBufNum=0;
struct ST_TMPBUF {
	uint32_t data;
	uint8_t data_type;
};
static struct ST_TMPBUF sTmpBuf[TMP_BUF_MAX];
static int sDonePutTmpCmd=0;

static int num_ipc_devices = 0;
static ipc_device_config_t *ipc_devices;

static struct semaphore list_sem;
static int g_flag_send;
static wait_queue_head_t g_wqh_send;
static struct semaphore g_tx_ready_sem;
static struct semaphore g_tx_done_sem;
static struct workqueue_struct *ipc_workqueue;

/* omo */
static struct semaphore g_tx_write_sem;
static IPC1_REGS_t *ipc1_regs;
typedef struct
{
    struct work_struct delayed_work;
    ipc_device_config_t *device;
    uint8_t   data_type;
    uint32_t   cmd;
    uint16_t  len;
    void     *buffer;
} recv_data_t;

static recv_data_t recv_data[8];
static recv_data_t recv_data_forack[8];

static unsigned int sAckIntCnt=0;
static unsigned int sCmdIntCnt=0;

uint32_t ipc_get_num_devices( void )
{
    return num_ipc_devices;
}
EXPORT_SYMBOL(ipc_get_num_devices);

const char *ipc_get_device_name( uint32_t device_index )
{
    if (device_index < ipc_get_num_devices() )
    {
        return ipc_devices[device_index].dev_name;
    }
    return NULL;
}
EXPORT_SYMBOL(ipc_get_device_name);

static bool device_is_valid( ipc_device_config_t *device )
{
    if ( ( device != NULL ) &&
         ( device->instance_id < ipc_get_num_devices() ) &&
         ( &ipc_devices[device->instance_id] == device ) 
       )
    {
        return true;
    }
    return false;
}

static bool port_is_valid( ipc_port_config_t *port )
{
    if ( ( port != NULL ) && 
         ( port->ipc_device != NULL )
       )
    {
        return device_is_valid( port->ipc_device );
    }
    return false;
}

static int gRcvSigCnt=0;

static ipc_port_config_t *find_device_port( ipc_device_config_t *device, uint8_t port_number )
{
    ipc_port_config_t *port = NULL;
    ipc_port_config_t *temp;

    ENTER();

    if ( !device_is_valid( device ) )
    {
        return NULL;
    }
	
    down( &list_sem );

    list_for_each_entry( temp, &device->open_ports->list, list )
    {
        if ( temp->port_number == port_number )
        {
            pr_debug("find_device_port found %d:%d\n", device->instance_id, port_number);
            port = temp;
            break;
        }
        else
        {
            pr_debug("(nonmatching device %d:%d)\n", device->instance_id, temp->port_number);
        }
    }

    up( &list_sem );

    EXIT();

    return port;
}

static int sACKWQCnt=0;
static int sCMDWQCnt=0;

static void non_isr_recv( struct work_struct *work)
{
    recv_data_t *data = container_of( work, recv_data_t, delayed_work );
//sai	uint8_t ack_type = ACK_MSG_DISCARDED;
	uint32_t  p1;
	
	int ackFlag=0;
	
	p1 = data->cmd;

_IPC_MEMLOG_1TIME('n', __LINE__, data->data_type, data->len); //data->len = seq id
    ENTER();

    if ( device_is_valid( data->device ) )
    {
        ipc_port_config_t *port;

_IPC_MEMLOG_DBG('n', __LINE__, data->data_type, p1);
    	if (p1 == 0xff31ff31) {
    		ackFlag=1;
_IPC_MEMLOG_DBG('n', __LINE__, data->data_type, p1);
		    down(&g_tx_done_sem);
_IPC_MEMLOG_1TIME('n', __LINE__, data->data_type, p1);
			g_flag_send = 1;
			wake_up(&g_wqh_send);
			up(&g_tx_done_sem);
    		goto finish;
    	}
   		ackFlag=0;
    	
//        port = find_device_port(data->device, data->data_type); //ugh! parameter2 is wrong
        port = sFirstPort;
		
_IPC_MEMLOG_DBG('n', __LINE__, data->data_type, p1);
        if ( port_is_valid(port) )
        {
            void *buffer_va = NULL;
            pr_debug("Port %d, rx cmd %d, buffer 0x%p, len %d\n", data->data_type, p1, data->buffer, data->len);

//sai			ack_type = ACK_MSG_PROCESSED;

//sai			if ((data->buffer != NULL) && (data->len > 0))
            {
//sai				request_mem_region((uint32_t)data->buffer, data->len, IPC_NAME);
//sai				buffer_va = ioremap((uint32_t)data->buffer, data->len);
            	
            	//pass stored data for recv_callback() 
            	int tmpbufcnt=0;
            	if (sDonePutTmpCmd == 0) {
            		sDonePutTmpCmd=1;
	            	while(tmpbufcnt < sTmpBufNum)
	            	{
	            		uint8_t data_type = (uint8_t)((p1 >> 28) & 0x00000007);//datatype_to_port(p1); //iir & IIR_PORT_MASK; ugh! 0: 1: 2: 3: /* mnakamura */
		                port->recv_callback(port, port->user_param, sTmpBuf[tmpbufcnt].data_type, sTmpBuf[tmpbufcnt].data, buffer_va, data->len);
	            		tmpbufcnt++;
	            	}
            	}
            	
            	
            	if ( !(data->data_type==0) || !(gRcvSigCnt==0) )
            	{
	                port->recv_callback(port, port->user_param, data->data_type, p1, buffer_va, data->len);
            	}
if (data->data_type == 0) {
	gRcvSigCnt++;
}

//sai				iounmap(buffer_va); 
//sai				release_mem_region((uint32_t)data->buffer, data->len);
                }
//sai			else
//sai			{
//sai				port->recv_callback(port, port->user_param, port->port_number, p1, data->buffer, data->len);
//sai			}
        }
        else
        {
//sai			ack_type = ACK_MSG_DISCARDED;
_IPC_MEMLOG_MUST('n', __LINE__, 0, 0);
			if ( !(data->data_type==0) || !(gRcvSigCnt==0) )
			{
				if (sTmpBufNum < TMP_BUF_MAX) {
					//store data to temporary buffer, later pass to recv_callback() when port is build
					sTmpBuf[sTmpBufNum].data = p1;
					sTmpBuf[sTmpBufNum].data_type = data->data_type;
					sTmpBufNum++;
				}
			}
if (data->data_type == 0) {
	gRcvSigCnt++;
}
            pr_debug("<CLOSED> Port %d, rx cmd %d, buffer 0x%p, len %d\n", data->data_type, p1, data->buffer, data->len);
        }
		
		{
			uint32_t senddata;
			
			senddata = p1 & 0xfeffffff;
			senddata = senddata | 0x80000000 ;
				
_IPC_MEMLOG_DBG('n', __LINE__, data->data_type, senddata);
	    	down(&g_tx_write_sem); /* omo */
			iowrite32(senddata, &data->device->regs->IPC_WDR_1);
			iowrite32(0x1, &data->device->regs->IPC_ISRW);
			/* omo */
			{
				while(1) {
				    uint32_t iirtmp, iirtmp2;
				    iowrite32(0, &ipc1_regs->IPC_DUMMY);
				    iirtmp = ioread32(&ipc1_regs->IPC_IIR);
					iirtmp2 = iirtmp & 0x1;
					if (iirtmp2 == 0) break;
				}
			}
			
	    	up(&g_tx_write_sem); /* omo */
		}
    } else {
		printk(KERN_EMERG "sai:%s(%d)%s():ERROR invalid port\n", _FILE_NAME_, __LINE__, __func__);
   		ipc_printDbgLog();
    	while(1);
    }

//sai	iowrite32( ( ( uint32_t )ack_type ) << IIR_ACK_SHIFT, &data->device->regs->IPC_ISRW);
finish:
_IPC_MEMLOG_DBG('n', __LINE__, data->data_type, 0);
	if (ackFlag == 0) {
    	sCMDWQCnt--;
	} else {
    	sACKWQCnt--;
	}
    EXIT();
}

static unsigned short sACKWQSeqid=0x8000;
static unsigned short sCMDWQSeqid=0x0000;

static irqreturn_t irq_handler(int irq, void *dev_id)
{
    uint32_t iir;
    ipc_device_config_t *device = ( ipc_device_config_t * )dev_id;
    int ret;
	uint32_t  p1=0,p2=0;

_IPC_MEMLOG_DBG('a', __LINE__, 0, 0);
    iowrite32(0, &device->regs->IPC_DUMMY);
    iir = ioread32(&device->regs->IPC_IIR);
//_IPC_MEMLOG('i', __LINE__, 0xff, iir);

    if ( iir & 0x000000ff ) //if ( iir & IIR_ACK_MASK )
    {
    	unsigned int recv_data_forack_idx;
    	recv_data_forack_idx = sAckIntCnt & 0x00000007;
#ifdef ENB_PRINTKLOG_INT
		printk(KERN_EMERG "sai:%s(%d)%s:INTACK\n", _FILE_NAME_, __LINE__, __func__);
#endif
		p1 = ioread32(&device->regs->IPC_RDR_0);
_IPC_MEMLOG_DBG('a', __LINE__, 0xff, p1);

        device->ack_type = ( iir & IIR_ACK_MASK ) >> IIR_ACK_SHIFT;

    	if (sACKWQCnt >= 2) {
			printk(KERN_EMERG "sai:%s(%d)%s():ERROR ACKWQCNT\n", _FILE_NAME_, __LINE__, __func__);
    	}
    	sACKWQCnt++;
        memset(&(recv_data_forack[recv_data_forack_idx]), 0, sizeof(recv_data_t)); //fill 0 "1 recv_data_t" area
        INIT_WORK( &(recv_data_forack[recv_data_forack_idx].delayed_work), non_isr_recv );

        recv_data_forack[recv_data_forack_idx].cmd         = 0xff31ff31;
        recv_data_forack[recv_data_forack_idx].len         = sACKWQSeqid++;
        recv_data_forack[recv_data_forack_idx].buffer      = (char *)0;
    	recv_data_forack[recv_data_forack_idx].data_type = 0;
        recv_data_forack[recv_data_forack_idx].device      = device;
    	
_IPC_MEMLOG_1TIME('a', __LINE__, 0, (unsigned int)(recv_data_forack[recv_data_forack_idx].len));
        iowrite32(iir & 0x000000ff, &device->regs->IPC_ICR);
    	
        ret = queue_work(ipc_workqueue, &(recv_data_forack[recv_data_forack_idx].delayed_work));
_IPC_MEMLOG_DBG('a', __LINE__, 0, p1);
    	sAckIntCnt++;
    }

    if ( iir & IIR_CMD_MASK )
    {
    	unsigned int recv_data_idx;
    	recv_data_idx = sCmdIntCnt & 0x00000007;
    	
        p1 = ioread32(&device->regs->IPC_RDR_1);
		
    	if (sCMDWQCnt >= 2) {
			printk(KERN_EMERG "sai:%s(%d)%s():ERROR CMDWQCNT\n", _FILE_NAME_, __LINE__, __func__);
    	}
    	sCMDWQCnt++;
        memset(&(recv_data[recv_data_idx]), 0, sizeof(recv_data_t)); //fill 0 "1 recv_data_t" area
        INIT_WORK( &(recv_data[recv_data_idx].delayed_work), non_isr_recv );

        recv_data[recv_data_idx].cmd         = p1;
        recv_data[recv_data_idx].len         = sCMDWQSeqid++;
        recv_data[recv_data_idx].buffer      = (char *)p2;
    	recv_data[recv_data_idx].data_type = (uint8_t)((p1 >> 28) & 0x00000007);//datatype_to_port(p1); //iir & IIR_PORT_MASK; ugh! 0: 1: 2: 3: /* mnakamura */
        recv_data[recv_data_idx].device      = device;
		
_IPC_MEMLOG_MUST('i', __LINE__, recv_data[recv_data_idx].data_type, p1);
    	
#ifdef ENB_PRINTKLOG_INT
		printk(KERN_EMERG "sai:%s(%d)%s:INTCMD %x\n", _FILE_NAME_, __LINE__, __func__, p1);
#endif
        iowrite32(IIR_CMD_MASK, &device->regs->IPC_ICR);

        ret = queue_work(ipc_workqueue, &(recv_data[recv_data_idx].delayed_work));
        sCmdIntCnt++;
    }
    iowrite32(0, &device->regs->IPC_DUMMY);
    iir = ioread32(&device->regs->IPC_IIR);
	if(iir){
//		printk(KERN_EMERG "#### DETECT 0x%08X!!\n", iir);
		_IPC_MEMLOG_MUST('i', __LINE__, 0xff, iir); /* omo */
	}

_IPC_MEMLOG_DBG('a', __LINE__, 0, p1);
    return IRQ_HANDLED;
}

ipc_drvr_handle  ipc_attach( uint32_t device_index, uint8_t port_number, ipc_recv_callback recv_callback, void *user_param )
{
    ipc_device_config_t *device = NULL;
    ipc_port_config_t *port = NULL;

    ENTER();
    if ( device_index >= ipc_get_num_devices() ||
         ipc_devices == NULL )
    {
        EXIT();
        return NULL;
    }

    device = &ipc_devices[ device_index ];

    if ( find_device_port( device, port_number) == NULL )
    {
        down(&list_sem);

        port = kmalloc( sizeof( ipc_port_config_t ), GFP_KERNEL );
        if ( port != NULL )
        {
            memset(port, 0, sizeof(ipc_port_config_t));

            port->ipc_device    = device;
            port->port_number   = port_number;
            port->recv_callback = recv_callback;
            port->user_param    = user_param;

            list_add_tail(&port->list, &device->open_ports->list);

            if ( device->open_count == 0 )
            {
                int retval;
                pr_debug(PREFIX "first port (%d:%d) being opened, attach ISR\n", device_index, port_number);
                retval = request_irq(port->ipc_device->int_num, irq_handler, 0, IPC_NAME, device );
            	
            }
            device->open_count++;
        	
        	if (sFirstPort == NULL) {
        		if ( (0 < port_number) && (port_number < 15) ) {
        			sFirstPort=port;
        		}
        	}

        }
        up(&list_sem);
    }

    EXIT();
    return port;
}
EXPORT_SYMBOL(ipc_attach);

ipc_error_type_t ipc_detach( ipc_drvr_handle handle )
{
    ipc_port_config_t   *port = ( ipc_port_config_t * )handle;
    bool                 found_port = false;
    ipc_port_config_t   *test_port;
    ipc_device_config_t *device;
    struct list_head    *cur, *q;

    ENTER();

    if ( !port_is_valid(port) )
    {
        return e_IPC_ERROR;
    }

    device = port->ipc_device;

    down(&list_sem);

    list_for_each_safe(cur, q, &device->open_ports->list)
    {
        test_port = list_entry(cur, ipc_port_config_t, list);
        if ( test_port->port_number == port->port_number )
        {
            found_port = true;

            list_del( cur );
            kfree( test_port );

            device->open_count--;
            if ( device->open_count == 0 )
            {
                pr_debug(PREFIX "last port (%d) being being closed, free ISR\n", device->instance_id);
                free_irq( device->int_num, device );
            }

            break;
        }
    }

    up(&list_sem);

    if ( found_port )
    {
        return e_IPC_SUCCESS;
    }

    EXIT();
    return e_IPC_ERROR;
}
EXPORT_SYMBOL(ipc_detach);

ipc_error_type_t ipc_send(ipc_drvr_handle handle, uint8_t command, void *buffer, uint16_t length, uint8_t data_type)
{
    ipc_port_config_t *port = ( ipc_port_config_t * )handle;
    ipc_device_config_t *device = NULL;
    ipc_error_type_t result;

    ENTER();

    if ( !port_is_valid( port ) )
    {
        EXIT();
        return e_IPC_ERROR;
    }

    device = port->ipc_device;

_IPC_MEMLOG_DBG('w', __LINE__, data_type, *((uint32_t *)buffer) );
    down(&g_tx_ready_sem);
_IPC_MEMLOG_DBG('w', __LINE__, data_type, *((uint32_t *)buffer) );

    {
        uint32_t senddata;
        senddata = *( uint32_t *) buffer;
        senddata = senddata | 0x01000000 ;
        senddata = senddata | 0x80000000 ;
    	senddata = senddata | (((uint32_t)data_type) << 28); //port_to_datatype(port->port_number) ;
#ifdef ENB_PRINTKLOG_SND
		printk(KERN_EMERG "sai:%s(%d)%s: senddata=%x\n", _FILE_NAME_, __LINE__, __func__, senddata);
#endif
_IPC_MEMLOG_MUST('w', 1, data_type, senddata);
//	    down(&g_tx_done_sem);
    	down(&g_tx_write_sem); /* omo */
        iowrite32( senddata , &device->regs->IPC_WDR_0);
        iowrite32( IIR_CMD_MASK, &device->regs->IPC_ISRW);
    	up(&g_tx_write_sem); /* omo */
//		wait_event_timeout(g_wqh_send, (g_flag_send != 0), 1);
_IPC_MEMLOG_1TIME('w', __LINE__, data_type, *((uint32_t *)buffer) );
		wait_event(g_wqh_send, (g_flag_send != 0));
_IPC_MEMLOG_1TIME('w', __LINE__, data_type, *((uint32_t *)buffer) );
		g_flag_send = 0;
//	    up(&g_tx_done_sem);
_IPC_MEMLOG_DBG('w', __LINE__, data_type, *((uint32_t *)buffer) );
    }
    

    if ( device->ack_type == ACK_MSG_PROCESSED )
    {
        result = e_IPC_SUCCESS;
    }
    else if ( device->ack_type == ACK_MSG_DISCARDED )
    {
        result = e_IPC_NO_LISTENER;
    }
    else
    {
        result = e_IPC_ERROR;
    }

_IPC_MEMLOG_DBG('w', 2, data_type, *( uint32_t *) buffer);
    up(&g_tx_ready_sem);
_IPC_MEMLOG_DBG('w', __LINE__, data_type, *((uint32_t *)buffer) );

    EXIT();
    return result;
}
EXPORT_SYMBOL(ipc_send);

static int ipc_platform_probe(struct platform_device *pdev)
{
    int retval = 0;
    int irq;
    struct resource *reg_addr;
    struct device_node *node = pdev->dev.of_node;
    struct property *prop;
    char *name;
    uint32_t dev_id;
    IPC0_REGS_t *regs;

    ENTER();
 
    irq = platform_get_irq(pdev, 0);
    if (irq < 0)
    {
        dev_err(&pdev->dev, PREFIX "platform_get_irq failed\n");
        EXIT();
        return -ENXIO;
    }

    reg_addr = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    if (!reg_addr)
    {
        dev_err(&pdev->dev, "platform_get_resource failed\n");
        EXIT();
        return -ENXIO;
    }

    prop = of_find_property(node, "device_name", NULL);
    name = (char *)of_prop_next_string(prop, NULL);

    dev_dbg(&pdev->dev, "'%s' registers at addr=0x%X, size=0x%X, irq=%d\n",
           name, reg_addr->start, resource_size(reg_addr), irq);

    if (!request_mem_region(reg_addr->start, resource_size(reg_addr), IPC_NAME))
    {
        dev_err(&pdev->dev, "request_mem_region failed\n");
        retval = -EBUSY;
    }
    regs = ioremap(reg_addr->start, resource_size(reg_addr));
    if (!regs)
    {
        dev_err(&pdev->dev, "ioremap failed\n");
        retval = -ENOMEM;
	}
	/* omo */
	ipc1_regs = ioremap(0xd401d400, sizeof(IPC1_REGS_t));
    if (!ipc1_regs)
    {
    	printk(KERN_EMERG "## ioremap failed:%d\n", __LINE__);
        dev_err(&pdev->dev, "ioremap failed\n");
        retval = -ENOMEM;
    }

    down( &list_sem );

    dev_id = num_ipc_devices;
    num_ipc_devices++;
    ipc_devices = krealloc( ipc_devices, sizeof( ipc_device_config_t ) * num_ipc_devices, GFP_KERNEL );

    ipc_devices[ dev_id ].instance_id = dev_id;
    strcpy( ipc_devices[ dev_id ].dev_name, name );
    ipc_devices[ dev_id ].int_num = irq;
    ipc_devices[ dev_id ].open_count = 0;
    ipc_devices[ dev_id ].pdev = pdev;
    ipc_devices[ dev_id ].regs = regs;

    ipc_devices[ dev_id ].open_ports = kmalloc( sizeof( ipc_port_config_t ), GFP_KERNEL );
    INIT_LIST_HEAD( &ipc_devices[ dev_id ].open_ports->list );

//    sema_init( &ipc_devices[ dev_id ].tx_done_sem,  0 );
//    sema_init( &ipc_devices[ dev_id ].tx_ready_sem, 1 );

    up( &list_sem );

    EXIT();
    return retval;
}


static int ipc_platform_remove(struct platform_device *pdev)
{
    ENTER();
/* 
    TODO - clean up on exit
 
    struct resource *reg_addr;
    int irq;

    ENTER();

    iounmap(regs); 
    reg_addr = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    release_mem_region(reg_addr->start, resource_size(reg_addr));

    irq = platform_get_irq(pdev, 0);
    free_irq(irq, NULL); 

*/
    EXIT();
    return 0;
}

static int ipc_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    ENTER();

    EXIT();
    return 0;
}

static int ipc_platform_resume(struct platform_device *pdev)
{
    ENTER();

    EXIT();
    return 0;
}

static struct platform_device_id mrvl_ipc_driver_ids[] = {
    {
        .name		= IPC_NAME,
    },
	{ },
};
MODULE_DEVICE_TABLE(platform, mrvl_ipc_driver_ids);

static const struct of_device_id mrvl_ipc_dt_match[] = {
    { .compatible = IPC_COMPATIBILITY_NAME, },
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_ipc_dt_match);

static struct platform_driver ipc_platform_driver =
{
    .probe    = ipc_platform_probe,
    .remove   = ipc_platform_remove,
    .suspend  = ipc_platform_suspend,
    .resume   = ipc_platform_resume,
    .id_table = mrvl_ipc_driver_ids,
    .driver   = {
        .name  = IPC_NAME,
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_ipc_dt_match),
    }
};

static int ipc_driver_init(void)
{
    int retval;

    ENTER();

    pr_debug(PREFIX "loading driver\n");

	g_flag_send = 0;
   	init_waitqueue_head(&g_wqh_send);
    sema_init( &list_sem, 1 );
    sema_init( &g_tx_done_sem, 1 );
    sema_init( &g_tx_ready_sem, 1 );
    sema_init( &g_tx_write_sem, 1 ); /* omo */

    num_ipc_devices = 0;
    ipc_devices = NULL;

    ipc_workqueue = create_workqueue(IPC_NAME "_wq");

    retval = platform_driver_register(&ipc_platform_driver);
    if (retval)
    {
        // do any needed cleanup
        retval = pr_err(PREFIX "%s: error registering platform driver\n",__func__);
    }
    else
    {
        pr_debug(PREFIX "platform registration complete\n");
    }
	
    EXIT();
    return retval;
}

static void ipc_driver_exit(void)
{
    ENTER();

    platform_driver_unregister(&ipc_platform_driver);  

    pr_debug(PREFIX "removed IPC driver\n");

    EXIT();
}

module_init(ipc_driver_init);
module_exit(ipc_driver_exit);

