/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

/**
 * \file mesg_queue.c
 *
 * \brief Generic Message Queue utils
 *
 *
 */

#include <linux/export.h>
#include <linux/time.h>
#include <linux/jiffies.h>
#include "mesg_queue.h"

#undef DEBUG
#ifdef DEBUG
#define dbg(args...) printk(args)
#else
#define dbg(args...)
#endif

void mesgq_free(mesg_queue_t *mesgq)
{
    BUG_ON( mesgq == NULL );
    kfifo_free( &(mesgq->mesg_queue_fifo));
}
EXPORT_SYMBOL(mesgq_free);

int mesgq_create(mesg_queue_t *mesgq, size_t mesg_sz, unsigned int depth)
{
    int retcode;

    BUG_ON( mesgq == NULL );
    BUG_ON( mesg_sz == 0 );
    BUG_ON( depth == 0 );

    spin_lock_init( &(mesgq->mesg_queue_lock) );
    sema_init( &(mesgq->mesg_queue_sem), 0 );
    mesgq->mesg_sz = mesg_sz;

    //kfifo size must be a power of 2.  refer to kfifo documentation.
    retcode = kfifo_alloc( &(mesgq->mesg_queue_fifo), roundup_pow_of_two(depth*mesg_sz), GFP_KERNEL );
    //printk( KERN_INFO "kfifo_alloc() retcode=%d\n", retcode );
    BUG_ON( retcode != 0 );
    return retcode;
}
EXPORT_SYMBOL(mesgq_create);

int mesgq_wait_for_msg( mesg_queue_t *mesgq, void *msg, unsigned long int timeout_in_usec )
{
    int retcode;
    unsigned long jiffies;
    struct timeval tv_timeout;

    BUG_ON( mesgq == NULL );

    dbg( "%s\n", __FUNCTION__ );

    if( timeout_in_usec == MESGQ_WAIT_FOREVER ) {
        retcode = down_interruptible( &mesgq->mesg_queue_sem );
    }
    else { 
        tv_timeout.tv_sec = 0;
        tv_timeout.tv_usec = timeout_in_usec;

        jiffies = timeval_to_jiffies( &tv_timeout );

        retcode = down_timeout( &mesgq->mesg_queue_sem, jiffies );
    }

    dbg( "%s down retcode=%d\n", __FUNCTION__, retcode );
    if( retcode != 0 ) {
        BUG_ON( retcode != -ETIME );
        return retcode;
    }

    memset( msg, 0, mesgq->mesg_sz);
    retcode = kfifo_out_spinlocked( &(mesgq->mesg_queue_fifo), msg, mesgq->mesg_sz, &(mesgq->mesg_queue_lock) );
    BUG_ON(retcode != mesgq->mesg_sz);

    dbg( "%s kfifo_out retcode=%d\n", __FUNCTION__, retcode );

    return 0;
}
EXPORT_SYMBOL(mesgq_wait_for_msg);

int mesgq_send_nowait( mesg_queue_t *mesgq, void *msg) {
    int retcode;

    retcode = kfifo_in_spinlocked( &(mesgq->mesg_queue_fifo), msg, mesgq->mesg_sz, &(mesgq->mesg_queue_lock) );
    BUG_ON( retcode != mesgq->mesg_sz );

    dbg( "%s kfifo_in retcode=%d\n", __FUNCTION__, retcode );

    up( &(mesgq->mesg_queue_sem) );

    return 0;
}
EXPORT_SYMBOL(mesgq_send_nowait);

