/*
 * ============================================================================
 * Copyright (c) 2012-2013 Marvell International, Ltd. All Rights Reserved
 *
 *                         Marvell Confidential
 * ============================================================================
 */

#include "dma_buffer.h"
#include "dma_alloc_api.h"
#include "memAPI.h"

#ifndef MIN
#define MIN(a, b) ( (a) > (b) ? (b) : (a) )
#endif

struct BigBuffer_s *dma_buffer_freefunc( struct BigBuffer_s *bb )
{
    if ( bb ) 
    {
        memFree( bb->data );
        if (bb->dma_alloc) 
        {
            if (bb->dma_alloc->direction < 0 ) // recover from missing unmap calls by user. 
            {
                dma_alloc_unmap_single( bb->dma_alloc, bb->dma_alloc->direction * -1 );
            }
            dma_free( bb->dma_alloc );
            memFree( bb->dma_alloc );
        }
        memFree( bb );
  }
  return 0;  
}

struct BigBuffer_s *BigBuffer_convert_to_dma_buffer( struct BigBuffer_s *bb )
{
  if (bb->dma_alloc == 0)
  {
    bb->dma_alloc = dma_alloc( bb->datalen );
    dma_alloc_write( bb->dma_alloc, bb->data, bb->datalen );
    memFree( bb->data );
    bb->data = 0;
    bb->freeFunc = dma_buffer_freefunc;
  }
  return bb;
}

struct BigBuffer_s *dma_buffer_malloc( mlimiter_t *limiter, int32_t len )
{
  ASSERT(len > 0);  // lengths of 0 do "bad things" later when passed to routines like dma_map_single

  if (!limiter)
    limiter = mlimiter_by_name("print");

  struct BigBuffer_s *bb = (struct BigBuffer_s *)MEM_MALLOC_LIMITED(limiter, sizeof(struct BigBuffer_s));
  if (bb)
  {
      bb->datalen = len;
      bb->dma_alloc = dma_alloc( bb->datalen );
      bb->data = 0;
      bb->freeFunc = dma_buffer_freefunc;
      ATInitNode(&bb->listNode); 
  }
  return bb;
}

struct BigBuffer_s *dma_buffer_adopt( void *data, int32_t len )
{  
  mlimiter_t *limiter = mlimiter_by_name("print");
  struct BigBuffer_s *bb = (struct BigBuffer_s *)MEM_MALLOC_LIMITED(limiter, sizeof(struct BigBuffer_s));
  if (bb)
  {
      bb->dma_alloc = dma_alloc( len );
      bb->data = 0;
      bb->datalen = len;
      bb->freeFunc = dma_buffer_freefunc;
      ATInitNode(&bb->listNode); 
      dma_alloc_write( bb->dma_alloc, data, len );
      memFree( data );
  }
  return bb;
}

// This doesn't free the original.
struct BigBuffer_s *dma_buffer_copy_from( void *data, int32_t len )
{  
  mlimiter_t *limiter = mlimiter_by_name("print");
  struct BigBuffer_s *bb = (struct BigBuffer_s *)MEM_MALLOC_LIMITED(limiter, sizeof(struct BigBuffer_s));
  if (bb)
  {
      bb->dma_alloc = dma_alloc( len );
      bb->data = 0;
      bb->datalen = len;
      bb->freeFunc = dma_buffer_freefunc;
      ATInitNode(&bb->listNode); 
      dma_alloc_write( bb->dma_alloc, data, len );
  }
  return bb;
}

struct BigBuffer_s *dma_buffer_realloc( struct BigBuffer_s *bb, int32_t len )
{
    if ( bb && bb->data )
    {
        int32_t copy = MIN( bb->datalen, len );

        bb->dma_alloc = dma_alloc( len );
        bb->datalen = len;
        dma_alloc_write( bb->dma_alloc, bb->data, copy );
        memFree(bb->data);
        bb->data = 0;
    }
    else if ( bb && bb->dma_alloc )
    {
        struct dma_alloc_s *nda = dma_alloc( len );
        ASSERT(len <= bb->datalen);
        int32_t copy = MIN( bb->datalen, len );
        dma_alloc_mmap_forcpu( bb->dma_alloc );
        dma_alloc_write( nda, bb->dma_alloc->v_addr, copy );
        dma_alloc_unmmap_forcpu( bb->dma_alloc );
        memFree( dma_free( bb->dma_alloc ) );
        bb->dma_alloc = nda;
        bb->datalen = len;
    }
    return bb;
}

void *dma_buffer_map_single( struct BigBuffer_s *bb, int direction )
{
    if ( bb->data )
    {
        ASSERT( bb->data == 0 ); 
    }
    else
    {
        dma_alloc_map_single( bb->dma_alloc, direction );   
    }
    return (void *)bb->dma_alloc->hw_addr;
}

void *dma_buffer_unmap_single( struct BigBuffer_s *bb, int direction )
{
    if ( bb->data )
    {
        ASSERT( bb->data == 0 ); 
    }
    else
    {
        dma_alloc_unmap_single( bb->dma_alloc, direction );   
    }
    return (void *)bb->dma_alloc->hw_addr;
}

void * dma_buffer_mmap_forcpu( struct BigBuffer_s *bb )
{
    if ( bb->data )
        return bb->data;
    ASSERT( bb->dma_alloc );
    if ( !bb->dma_alloc->hw_addr || bb->dma_alloc->hw_addr == 0xdeadbeef ) {  
	/// need a hw_address to mmap to userspace, can't directly map kernel address 
	dma_buffer_map_single( bb, DMA_FROM_DEVICE );
	dma_buffer_unmap_single( bb, DMA_FROM_DEVICE );
    }
    dma_alloc_mmap_forcpu( bb->dma_alloc );
    return bb->dma_alloc->v_addr;
}

void * dma_buffer_unmmap_forcpu( struct BigBuffer_s *bb )
{
  if ( bb->data == 0 && bb->dma_alloc )
    dma_alloc_unmmap_forcpu( bb->dma_alloc );
  return 0;
}


struct BigBuffer_s * dma_buffer_from_kernel( struct dma_alloc_s *kernel_copy_to_user_dma_alloc )
{
    struct mlimiter_s *limiter = mlimiter_by_name("print");
    ASSERT(kernel_copy_to_user_dma_alloc && kernel_copy_to_user_dma_alloc->kv_addr);
    struct BigBuffer_s *bb = (struct BigBuffer_s *)MEM_MALLOC_LIMITED(limiter, sizeof(struct BigBuffer_s));
    if (bb) {
	bb->datalen = kernel_copy_to_user_dma_alloc->len;
	bb->dma_alloc = (struct dma_alloc_s *)MEM_MALLOC_LIMITED(limiter, sizeof(struct dma_alloc_s));
	if ( !bb->dma_alloc )
	    return 0;
	memcpy(bb->dma_alloc, kernel_copy_to_user_dma_alloc, sizeof(struct dma_alloc_s));
	bb->data = 0;
	bb->freeFunc = dma_buffer_freefunc;
	ATInitNode(&bb->listNode); 
	dma_alloc_track_add(bb->dma_alloc);
    }
    return bb;
}
