/* 
 *
 * ============================================================================
 * Copyright (c) 2008-2014   Marvell Semiconductor, Inc. All Rights Reserved
 *
 *                         Marvell Confidential
 * ============================================================================
 */
/** 
 * \file BigBuffer.h
 * 
 * \brief The purpose of this class is to have a polymorphic free function.
 *
 * Handle to a big allocation allows subclasses to have different behaviors
 * caching, static buffers etc, base class does malloc free behavior.
 *
 */

#ifndef BIGBUFFER_H
#define BIGBUFFER_H

#include <stdint.h>
#include "list.h"
#include "mlimiter.h"

#ifdef __cplusplus
extern "C" {
#endif

struct BigBuffer_s 
{   
/** Macro for members of the structure must be used by subclasses.  This way the
 * order of the members is preserved. */
#define BigBuffer_T_BaseClassMembers \
    uint8_t * data; /*!< Pointer to a big continous memory chunk, clients shouldn't change the address.*/\
    int32_t datalen; /*!< Length in bytes of the data, clients shouldn't change the length.*/ \
    ATLISTENTRY listNode; /*!< Linked list node for the client's use. Client must remove from lists prior to freeing.*/\
    struct BigBuffer_s *(*freeFunc)(struct BigBuffer_s *bigBuffer) /*!< Overloadable free function. */
    struct dma_alloc_s *dma_alloc; /*!< NULL on threadx, */
    /** Client is in no way obligated to use the listNode, but its there if they
     * want to insert this into a list. */
    BigBuffer_T_BaseClassMembers;
};

/** 
 * \brief Constructor does a malloc to get you a big block of memory with a
 * handle (struct BigBuffer_s) on top of it.
 *
 * \param[in] limiter 0 is default, non zero means a heap limiter is installed.
 * \param[in] len Number of bytes to allocate.
 *
 * \returns null on failure.
 * \returns Big Buffer pointer otherwise.
 */
struct BigBuffer_s * BigBuffer_Malloc( mlimiter_t *limiter, int32_t len );

/**
 * \brief Free the BigBuffer and the big chunk of ram it points to.
 *
 * Uses internal freeFunc so the behaviour may be from a derived class.
 *
 * Usage:   ptr = BigBuffer_Free( ptr );
 *
 * \param[in] bigBuffer Buffer to be freed.
 *
 * \return Always 0 / null.
 *
 */
struct BigBuffer_s * BigBuffer_Free( struct BigBuffer_s *bigBuffer );

/// Construct wrapping object adopting the deletion rights of the
/// passed in data.    
struct BigBuffer_s *BigBuffer_adopt(void *data, int32_t len);

/// realloc a big buffer 
/// normally used to shrink a decompression buffer
struct BigBuffer_s *BigBuffer_realloc( struct BigBuffer_s *bb, int32_t new_length );


#ifdef __cplusplus
}
#endif

#endif // define BIGBUFFER_H
