/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

// simple user space app to link to pie_convenience functions
// and use them to pass a pie_handle from user to kernel space
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <fcntl.h>  // for open
#include <stdlib.h>  // for free
#include <string.h>  // for strncmp

#include "pie_handle_if.h"
#include "pie_if.h"
#include "pie_driverlib_if.h"
#include "pie_convenience_if.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_driverlib_if.h"
#include "pic_convenience_if.h"

#define PATH_TO_PICHANDLE0 "/sys/devices/f95f0000.pic0/pic_handle_revcheck"
#define PATH_TO_PICHANDLE1 "/sys/devices/f95f4000.pic1/pic_handle_revcheck"
#define PATH_TO_PIC_REVCHECK_STATUS "/sys/module/picdriver/parameters/revcheck_status"

#define PATH_TO_PIEHANDLE "/sys/devices/f9400000.pie/pie_handle_revcheck"
#define PATH_TO_PIE_REVCHECK_STATUS "/sys/module/piedriver/parameters/revcheck_status"

void pie_request_revcheck(struct pie_handle_t *pie_handle, char *revcheck_status)
{
    char path[(sizeof PATH_TO_PIEHANDLE) + 1];
    int fd,  i;
    int retval;
    int pie_handle_size;
    uint8_t *pie_rover_handle;

    fd = open(PATH_TO_PIEHANDLE, O_WRONLY);
    if (fd < 0)
    {
	error_print("Failed to open pie_handle for writing! fd=%d\n", fd);
	return;
    }
    pie_handle_size = pie_get_handle_size(pie_handle);
    pie_rover_handle = (uint8_t *) pie_handle;
    do
    {
        retval = write(fd, pie_rover_handle, pie_handle_size);
        if (retval < 0)    
        {
            error_print("Failed to write value! return=%d\n", retval);
            return;
        }
        pie_handle_size -= retval;
        pie_rover_handle += retval;
    } while(pie_handle_size > 0);

    // now read back the value from the module parameter to see if we
    // passed or failed our rev check
    fd = open(PATH_TO_PIE_REVCHECK_STATUS, O_RDONLY);
    if (fd < 0)
    {
	error_print("Failed to open revcheck_status for read! fd=%d\n", fd);
	return;
    }
    retval = read(fd, revcheck_status, 20); 
    if (retval < 0)    
    {
        error_print("Failed to read revcheck status! retval=%d\n", retval);
        return;
    }
    return;
}

// note - h/w rev is assumed to be the same between pic0 and pic1 - so just using 0 here
void pic_request_revcheck(struct pic_handle_t *pic_handle, char *revcheck_status)
{
    char path[(sizeof PATH_TO_PICHANDLE0) + 1];
    int fd,  i;
    int retval;
    int pic_handle_size;
    uint8_t *pic_rover_handle;

    fd = open(PATH_TO_PICHANDLE0, O_WRONLY);
    if (fd < 0)
    {
	error_print("Failed to open pic_handle for writing! fd=%d\n", fd);
	return;
    }
    pic_handle_size = pic_get_handle_size(pic_handle);
    pic_rover_handle = (uint8_t *) pic_handle;
    do
    {
        retval = write(fd, pic_rover_handle, pic_handle_size);
        if (retval < 0)    
        {
            error_print("Failed to write value! return=%d\n", retval);
            return;
        }
        pic_handle_size -= retval;
        pic_rover_handle += retval;
    } while(pic_handle_size > 0);

    // now read back the value from the module parameter to see if we
    // passed or failed our rev check
    fd = open(PATH_TO_PIC_REVCHECK_STATUS, O_RDONLY);
    if (fd < 0)
    {
	error_print("Failed to open revcheck_status for read! fd=%d\n", fd);
	return;
    }
    retval = read(fd, revcheck_status, 20); 
    if (retval < 0)    
    {
        error_print("Failed to read revcheck status! retval=%d\n", retval);
        return;
    }
    return;
}

main()
{
    struct pie_handle_t *pie_handle;
    struct pic_handle_t *pic_handle;    
    char revcheck_status[24];
    uint32_t *tmp_ptr; int i; // remove
    uint8_t gap[4];  // for testing that we can set regs

    printf("In the main app, ready to create a pie_handle, and a pic_handle\n");
    pie_handle = pie_create_new_default_handle();
    pic_handle = pic_create_new_default_handle();

    printf("handles created, ready to dump the common regs *from the handles*\n");
    pic_common_dump_handle_regs(pic_handle);
    pie_common_dump_handle_regs(pie_handle);

// these next few lines create a bunch of output.  Uncomment for debug if you wish    
//    printf("Now ready to dump all the regs in the pie and pic handles\n");
//    pic_dump_handle_regs(pic_handle, 3);
//    pie_dump_handle_regs(pie_handle, 3);

    // this section checks to make sure we are reading and writing the shadow regs in the handle
    printf("now printing chipgap\n");
    pic_chipgap_dump_handle_regs(pic_handle);
    printf("now changing common and chipgap, just to see if we really can write/read shadow regs\n");
    pic_enable_common_irqs(pic_handle, NULL, true); // enable all irqs
    gap[0] = 0xAA; gap[1] = 0x55; gap[2] = 0xBE; gap[3] = 0xEB;
    pic_chipgap_set_INSERT0(pic_handle, gap);
    gap[0] = 0x42; gap[1] = 0x24; gap[2] = 0xEE; gap[3] = 0xFF;
    pic_chipgap_set_INSERT3(pic_handle, gap);
    printf("common and chipgap should now be changed\n");
    pic_common_dump_handle_regs(pic_handle);
    pic_chipgap_dump_handle_regs(pic_handle);

    printf("print xyscale, then set a bunch of the regs, and then print them out again\n");
    pie_xyscale_dump_handle_regs(pie_handle);
    pie_xyscale_set_values(pie_handle, 6, 1, 1, 0x342A5, 0xDE, 0xAD, 0xBE, 0x55AA5, 0x10101,
                           0xF0F0F, 0x50505, 0xFF0FF, 0x10F01, 0x77377, 1, 0xA);
    pie_xyscale_dump_handle_regs(pie_handle);

    // Sleeping to let the serial buffer flush
    sleep(1);

    // Finally, check to make sure we have hardware compatible with our driver
    // Yes, it's the revcheck functionality!
    pie_request_revcheck(pie_handle, revcheck_status);

    // -1 since the string terminating null doesn't come across sysfs
    printf("Pie revcheck\n");
    if (strncmp(revcheck_status, "passed", (sizeof("passed") -1)) == 0)
        printf("PIE Revcheck passed\n");
    else if (strncmp(revcheck_status, "error", (sizeof("error") -1)) == 0)
        printf("PIE Revcheck error\n");
    else
        printf("PIE revcheck didn't pass, Revcheck %s\n", revcheck_status);


    printf("Pic revcheck\n");
    pic_request_revcheck(pic_handle, revcheck_status);

    // -1 since the string terminating null doesn't come across sysfs
    if (strncmp(revcheck_status, "passed", (sizeof("passed") -1)) == 0)
        printf("PIC Revcheck passed\n");
    else if (strncmp(revcheck_status, "error", (sizeof("error") -1)) == 0)
        printf("PIC Revcheck error\n");
    else
        printf("PIC revcheck didn't pass, Revcheck %s\n", revcheck_status);
    
    // at the end, it is required to free the memory for the user space pie_handle (kernel frees its own)
    free(pie_handle);
    free(pic_handle);    
}
