/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
 ******************************************************************************
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>

#include "descriptor.h"
#include "pip_if.h"
#include "pip_handle.h"
#include "pip_convenience_if.h"
#include "PIP_TOP_regmasks.h"
#include "Zx_regmasks.h"

#define WIDTH 32
#define HEIGHT 32

static DECLARE_COMPLETION(dma_interrupts_complete);

static int odma_interrupt_callback(struct pip_odma_interrupt_info *info,
                                   void *unnused)
{
   printk("odma instance=%d int_array=0x%08X ( %s%s%s%s%s%s%s%s%s%s%s)\n",
       info->instance, info->int_array,
       info->bad_rresp ? "bad_rresp " : "",
       info->bad_bresp ? "bad_bresp " : "",
       info->soft_rst_cmpl ? "soft_rst_cmpl " : "",
       info->eoi ? "eoi " : "",
       info->dir_err ? "dir_err " : "",
       info->chg_line_align_err ? "chg_line_align_err " : "",
       info->eol_align_err ? "eol_align_err " : "",
       info->eoi_align_err ? "eoi_align_err " : "",
       info->eoi_err ? "eoi_err " : "",
       info->xfer_end ? "xfer_end " : "",
       info->own ? "own " : "");

   if (info->bad_rresp || info->bad_bresp || info->dir_err ||
       info->chg_line_align_err || info->eol_align_err ||
       info->eoi_align_err || info->eoi_err)
   {
      printk("Got PIP ODMA error interrupt: 0x%08X\n", info->int_array);
      return -1;
   }

   if (info->xfer_end)
   {
       printk("got xfer end!\n");
       complete(&dma_interrupts_complete);
   }

   return 0;
}

static int zx_interrupt_callback(struct pip_zx_interrupt_info *info,
                                 void *unnused)
{
   printk("zx int_array=0x%08X ( %s%s%s%s%s)\n",
       info->int_array,
       info->soft_rst_cmpl ? "soft_rst_cmpl " : "",
       info->illegalcmd_err ? "illegalcmd_err " : "",
       info->eoi ? "eoi " : "",
       info->xfer_end ? "xfer_end " : "",
       info->own ? "own " : "");

   if (info->illegalcmd_err)
   {
      printk("ZX illegal command");
   }

   return 0;
}

static int allocate_buffers(struct odma_axi_descriptor_info *odma_descriptors,
                            struct zx_descriptor_info *zx_descriptors,
                            uint8_t **inputbuf,
                            dma_addr_t *inputbuf_phys,
                            uint8_t **outputbuf,
                            dma_addr_t *outputbuf_phys)
{
    int rc, i;

    rc = odma_axi_descriptor_create_list(odma_descriptors, NULL, 8);
    if (rc)
    {
        printk("creating odma axi descriptors failed: %d\n", rc);
        return rc;
    }
    rc = zx_descriptor_create_list(zx_descriptors, NULL, 8, 32);
    if (rc)
    {
        printk("creating zx descriptors failed: %d\n", rc);
        return rc;
    }

    *inputbuf = dma_alloc_coherent(NULL, HEIGHT * WIDTH, inputbuf_phys, GFP_KERNEL);
    if (*inputbuf == NULL)
    {
        printk("creating dma buf failed\n");
        return -ENOMEM;
    }
    for (i = 0; i < HEIGHT * WIDTH; i++)
        (*inputbuf)[i] = i; // fill with test data

    *outputbuf = dma_alloc_coherent(NULL, HEIGHT * WIDTH, outputbuf_phys, GFP_KERNEL);
    if (*outputbuf == NULL)
    {
        printk("creating dma buf failed\n");
        return -ENOMEM;
    }
    for (i = 0; i < HEIGHT * WIDTH; i++)
        (*outputbuf)[i] = 0x99; // fill with garbage
    return 0;
}

static void free_buffers(struct odma_axi_descriptor_info *odma_descriptors,
                         struct zx_descriptor_info *zx_descriptors,
                         uint8_t *inputbuf,
                         dma_addr_t inputbuf_phys,
                         uint8_t *outputbuf,
                         dma_addr_t outputbuf_phys)
{
    odma_axi_descriptor_destroy_list(odma_descriptors);
    zx_descriptor_destroy_list(zx_descriptors);
    dma_free_coherent(NULL, HEIGHT * WIDTH, inputbuf, inputbuf_phys);
    dma_free_coherent(NULL, HEIGHT * WIDTH, outputbuf, outputbuf_phys);
}

static int perform_dma_band(struct pip_handle_t *handle,
                            struct odma_axi_descriptor_info *odma_descriptors,
                            struct zx_descriptor_info *zx_descriptors,
                            uint8_t *inputbuf,
                            dma_addr_t inputbuf_phys,
                            uint8_t *outputbuf,
                            dma_addr_t outputbuf_phys)
{
    struct odma_axi_descriptor *odma_desc;
    struct zx_descriptor *zx_desc;
    uint32_t *zx_commands;

    odma_desc = &odma_descriptors->descriptors[0];
    odma_desc->ctrl = (ODMA_AXI_DESCRIPTOR_CONFIG_BLOCK_OWN |
                       ODMA_AXI_DESCRIPTOR_CONFIG_INTR_ON_FINISH |
                       ODMA_AXI_DESCRIPTOR_CONFIG_STOP_ON_FINISH |
                       ODMA_AXI_DESCRIPTOR_CONFIG_IMAGE_START |
                       ODMA_AXI_DESCRIPTOR_CONFIG_IMAGE_END);
    odma_desc->source = outputbuf_phys;
    odma_desc->length = HEIGHT * WIDTH;
    odma_desc->lines = HEIGHT;
    odma_desc->width = WIDTH;
    odma_desc->last_line = 0xFFFFFFF0;
    odma_desc->source2 = 0;
    odma_desc->next = 0;

    zx_commands = zx_descriptor_get_command_buffer(zx_descriptors);
    zx_commands[0] = (((WIDTH * HEIGHT - 1) << ZX_CMD_LENGTH_STRETCHBLT_SHIFT) |
                      ZX_CMD_HEADER_STRETCHBLT8W);
    zx_commands[1] = inputbuf_phys;
    zx_commands[2] = 1 << 24; // scale
    zx_commands[3] = 1 << 24; // error

    zx_desc = &zx_descriptors->descriptors[0];
    zx_desc->ctrl = (ZX_DESCRIPTOR_CTRL_INTR_ON_FINISH |
                     ZX_DESCRIPTOR_CTRL_STOP_ON_FINISH |
                     ZX_DESCRIPTOR_CTRL_BLOCK_OWN);
    zx_desc->source = zx_descriptor_get_command_buffer_phys_addr(zx_descriptors);
    BUG_ON(zx_desc->source & 0x3F); // must be 64-byte aligned
    zx_desc->length = 4 * sizeof(uint32_t);
    zx_desc->next = 0;

    wmb(); // flush descriptors before telling hw to read them

    odma_axi_descriptor_print_chain(odma_descriptors);
    pip_start_odma(3, odma_axi_descriptor_get_phys_addr(odma_descriptors, 0));

    zx_descriptor_print_chain(zx_descriptors, 4);
    pip_start_zx(zx_descriptor_get_phys_addr(zx_descriptors, 0));

    return 0;
}

static int run_test(void)
{
    struct pip_handle_t *handle;
    int rc;
    uint32_t reg_val;
    struct odma_axi_descriptor_info odma_descriptors;
    struct zx_descriptor_info zx_descriptors;
    uint8_t *inputbuf, *outputbuf;
    dma_addr_t inputbuf_phys, outputbuf_phys;

    rc = allocate_buffers(&odma_descriptors, &zx_descriptors, &inputbuf,
                          &inputbuf_phys, &outputbuf, &outputbuf_phys);
    if (rc)
    {
        printk("Could not allocate buffers: %d\n", rc);
        return rc;
    }

    pip_do_reset();
    msleep(100);

    handle = pip_create_new_default_handle();

    rc = pip_do_revcheck(handle);
    if (rc < 0)
    {
        printk("pip revcheck failed: %d\n", rc);
        goto out;
    }

    pip_register_odma_callback(handle, odma_interrupt_callback, NULL);
    pip_register_zx_callback(handle, zx_interrupt_callback, NULL);

    // Configure Top
    reg_val = 0;
    reg_val = PIP_CFG_RM_REPLACE_VAL(reg_val, 13);  //RM mode 13
    handle->top->CFG = reg_val;
    reg_val = 0;
    reg_val = PIP_LPC_CFG_IND_LPC_REPLACE_VAL(reg_val, 1);
    handle->top->LPC_CFG = reg_val;
    reg_val = 0;
    reg_val = PIP_PIP_CTL_OUTCTL_REPLACE_VAL(reg_val, 1 << 3); // use odma 3
    reg_val = PIP_PIP_CTL_IE0_REPLACE_VAL(reg_val, 1);
    handle->top->PIP_CTL = reg_val;

    // Configure ZX
    reg_val = handle->zx->ZCR;
    reg_val = ZX_ZCR_OUTSEL3_REPLACE_VAL(reg_val, 1);
    reg_val = ZX_ZCR_OUTSEL2_REPLACE_VAL(reg_val, 2);
    reg_val = ZX_ZCR_OUTSEL1_REPLACE_VAL(reg_val, 3);
    reg_val = ZX_ZCR_OUTSEL0_REPLACE_VAL(reg_val, 0);
    reg_val = ZX_ZCR_ENABLE_REPLACE_VAL(reg_val, 1);
    reg_val = ZX_ZCR_BURSTLENGTH_REPLACE_VAL(reg_val, 1); // 64-byte bursts
    reg_val = ZX_ZCR_COLORMODE_REPLACE_VAL(reg_val, 0); // 8bpp
    handle->zx->ZCR = reg_val;
    handle->zx->ZLLR = WIDTH - 1;
    handle->zx->ZIHR = HEIGHT - 1;
    pip_enable_zx_irq(NULL);

    // Configure ODMA
    reg_val = 0;
    reg_val = ODMA0_CFG_LINE_REV_REPLACE_VAL(reg_val, 0); // left-to-right
    reg_val = ODMA0_CFG_IN_WIDTH_REPLACE_VAL(reg_val, 3); // 8bpp
    reg_val = ODMA0_CFG_BURST_LEN_REPLACE_VAL(reg_val, 2);
    reg_val = ODMA0_CFG_ENABLE_REPLACE_VAL(reg_val, 1);
    reg_val = ODMA0_CFG_LINE_REV_REPLACE_VAL(reg_val, 0);
    handle->odma[3]->cfg = reg_val;
    handle->odma[3]->line_size = WIDTH;
    pip_enable_odma_irq(3, NULL);

    pip_do_configure(handle);
    pip_dump();

    perform_dma_band(handle, &odma_descriptors, &zx_descriptors, inputbuf,
                     inputbuf_phys, outputbuf, outputbuf_phys);

    wait_for_completion_interruptible_timeout(&dma_interrupts_complete, 5*HZ);

    if (memcmp(inputbuf, outputbuf, HEIGHT * WIDTH) != 0)
    {
        pip_dump();
        print_hex_dump(KERN_ERR, "input  ", DUMP_PREFIX_OFFSET, 16, 1, inputbuf, HEIGHT * WIDTH, false);
        print_hex_dump(KERN_ERR, "output ", DUMP_PREFIX_OFFSET, 16, 1, outputbuf, HEIGHT * WIDTH, false);
        printk("ERROR: Buffer comparison failed.\n");
        rc = -1;
    }
    else
    {
        printk("The buffers matched. Test passed!\n");
        rc = 0;
    }

out:
    free_buffers(&odma_descriptors, &zx_descriptors, inputbuf, inputbuf_phys,
                 outputbuf, outputbuf_phys);
    return rc;
}

// start of functions callable by sysfs
// sysfs interface definitions
ssize_t run_test_set(struct device *dev, struct device_attribute *attr,
                     const char *buf, size_t count)
{
    switch(buf[0])
    {
    case 'a':
        printk("\ndumping pip registers\n");
        pip_dump();
        break;
    case 't':
        run_test();
        break;
    default:
        printk("\nThere is no test triggered by character ->%s<-\n",buf);
    }
    return count;
}


char *usagebuf="\
a=basic reg read \n\
t=run dma test\n";
    
ssize_t run_test_get(struct device *dev, struct device_attribute *attr,
                     char *buf)
{
    return scnprintf(buf, PAGE_SIZE, usagebuf);
}

// register sysfs functions for test code
DEVICE_ATTR(piptest, S_IWUSR | S_IRUGO, run_test_get, run_test_set);

static struct attribute *piptest_attrs[] = {
    &dev_attr_piptest.attr,
    NULL,
};

struct attribute_group pip_verification_attrgrp = {
    .name = "piptest",
    .attrs = piptest_attrs,
};

// end of register sysfs functions for test code

struct kobject *kobj;

static int __init pip_verification_init(void)
{
    int retcode;
    
    // create a sysfs interface
    kobj = kobject_create_and_add("pip_verification", firmware_kobj);
    retcode = sysfs_create_group(kobj, &pip_verification_attrgrp);    
    if( retcode != 0 )
    {
        printk( KERN_ERR "%s sysfs_create_file icetest failed retcode=%d\n",
                __func__, retcode );
    }
    return 0;
}
module_init(pip_verification_init);

static void __exit pip_verification_exit(void)
{
    sysfs_remove_group(kobj, &pip_verification_attrgrp);
    kobject_put(kobj);  // tell the system it can release the object
    printk(KERN_ERR "PIP Verification: device remove complete\n");
}
module_exit(pip_verification_exit);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell PIP verification Kernel Module");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_Feb_12");
