/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/module.h>  // for S_IWUSR and S_IRUGO
#include <linux/cdev.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>  // for msleep/ssleep

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"
#include "icetest_if.h"

#include "descriptor.h"
#include "pic_verification.h"

// number of ICETest IDMA bytes per sample is 4 (32 bits define an input pixel)
#define NUM_ICETEST_BPS 4
// number of PIC ODMA bytes per sample is 2 (16 bits in an output pixel)
#define NUM_PICODMA_BPS 2

// to use more memory, you may have to increase the coherent pool by changing the
// u-boot command line to add "coherent_pool=2M" or some amount of memory larger
// than the default 256KB

// NOTE: the number of bytes per line must be an integral number of 32 bit words
// or we will get EOL alignment errors
#define DEFAULT_NUM_BYTES_PER_LINE_OUTPUT 512

//#define NUM_LINES_PER_DESC 32
// for color, the NUM_LINES_PER_DESC and/or num_bytes_per_line_output need to be
// divisible by 3
#define NUM_LINES_PER_DESC 18
#define NUM_DESC 5

struct pic_verification_data picv_object;  // the pic verification object

// global variable to be changed as needed
static int global_num_bytes_per_line_output = DEFAULT_NUM_BYTES_PER_LINE_OUTPUT;
static bool dma_test_has_run = false;

static unsigned int use_picrate_val; // a debug thing - if non-zero, use the picrate and pacing

// globals to keep track of which interrupts we've received
bool pic_end_of_image_interrupt;
bool pic_error_interrupt;
bool icetest_eoi_int;
bool icetest_fin_int;

// forward declaration of module parm set functions
static void set_pictest_status(ulong status);
static void finalize_pictest_status(void);

static void pictest_debug_callback(struct pic_common_ints *intstruct);

static void pictest_interrupt_callback(struct pic_output_dma_interrupt_info *intstruct);

int create_output_buffer(int bufsize)
{
    dma_addr_t buffer_phys_addr;
    uint32_t *buffer_virt_addr;

//    bufsize *= 4; // * 4 for debugging - can see if someone wrote beyond original buffer end

    buffer_virt_addr = dma_alloc_coherent(NULL, bufsize, &buffer_phys_addr, GFP_DMA); 
    printk("%s just called dma_alloc_coherent %d, 0x%X\n", __func__, bufsize, buffer_phys_addr);
    if (buffer_virt_addr == 0)
    {
        set_pictest_status(PIC_TEST_BUF_ERR);
        return -1;
    }
    // put pattern in memory to avoid accidently thinking we got output data
    memset(buffer_virt_addr, 0x55, bufsize);  
    // data buffers have to be 4 byte aligned
    BUG_ON((buffer_phys_addr & 0x3) != 0);
    // store away the info on the outputbuffer
    picv_object.output_data = (uint16_t *)buffer_virt_addr;
    picv_object.output_data_phys = buffer_phys_addr;
    picv_object.out_bufsize = bufsize;
    printk("output buffer physaddr=0x%p, virtaddr=0x%p\n", (void *) buffer_phys_addr, (void *) buffer_virt_addr); 
    return 0;
}

int create_input_buffer(int bufsize)
{
    dma_addr_t buffer_phys_addr;
    uint32_t *buffer_virt_addr;

//    bufsize *= 4; // * 4 for debugging - can see if someone wrote beyond original buffer end

    buffer_virt_addr = dma_alloc_coherent(NULL, bufsize, &buffer_phys_addr, GFP_DMA); 
    printk("%s just called dma_alloc_coherent %d, 0x%X\n", __func__, bufsize, buffer_phys_addr);
    if (buffer_virt_addr == 0)
    {
        set_pictest_status(PIC_TEST_BUF_ERR);
        return -1;
    }
    // data buffers have to be 4 byte aligned
    BUG_ON((buffer_phys_addr & 0x3) != 0);
    // store away the info on the inputbuffer
    picv_object.input_data = buffer_virt_addr;
    picv_object.input_data_phys = buffer_phys_addr;
    picv_object.in_bufsize = bufsize;
    printk("input buffer physaddr=0x%p, virtaddr=0x%p\n", (void *) buffer_phys_addr, (void *) buffer_virt_addr); 
    return 0;
}

int create_icetest_dma_descriptors(int num_descriptors, int data_xfer_size)
{
    DMA_descriptor *descriptor;
    dma_addr_t phys_addr;
    uint32_t flags; // flags to set for the descriptor
    int retval, i;
    
    retval = create_descriptor_list(num_descriptors,
                                    old_4word_axi,
                                    &descriptor, &phys_addr,
                                    NULL);
    printk("Input descriptor virt_addr = 0x%p, phys_addr = 0x%X\n",
           descriptor, phys_addr);
    if (retval != 0)
    {
        set_pictest_status(PIC_TEST_DESC_ERR);
        return retval;
    }
    picv_object.icetestdma_descriptor = descriptor;
    picv_object.icetestdma_descriptor_phys = phys_addr;
    picv_object.num_input_descriptors = num_descriptors;
    
    for (i=0;i<num_descriptors;i++)
    {
        flags = 0;

        /* still need to implement someday - make sure if you have a last data buffer that is smaller
           than the other fixed size buffers (since it's the last one), that we can pass smaller size */
        if (i == 0)
        {
            flags |= DMA_SOI;
        }
        // if last descriptor, calculate actual data size FIXME
        if (i == (num_descriptors - 1))
        {
            //data_size = full_data_size - num_descriptor * data_xfer_size; FIXME
            flags |= DMA_EOI | DMA_STOP_WHEN_DONE | DMA_INT_WHEN_DONE;
        }
        // printk("going to set descriptor %d starting at virt addr 0x%p to point at phys_addr 0x%X\n",
        //         i, descriptor, picv_object.input_data_phys);
        set_descriptor(descriptor, i, data_xfer_size, picv_object.input_data_phys, flags);
    }
    return retval;
}

int create_output_dma_descriptors(int num_descriptors, int data_xfer_size,
                                  DMA_descriptor **descriptor, dma_addr_t *phys_addr,
                                  dma_addr_t buffer_location)
{
    int retval, i;
    uint32_t flags;

    printk("%s:getting ready to call create_descriptor_list num_desc=%d\n",
           __func__, num_descriptors);
    retval = create_descriptor_list(num_descriptors,
                                    old_4word_axi,
                                    descriptor, phys_addr,
                                    NULL);
    printk("Output descriptor virt_addr = 0x%p, phys_addr = 0x%X\n",
           *descriptor, *phys_addr);    
    if (retval != 0)
    {
        set_pictest_status(PIC_TEST_DESC_ERR);
        return retval;
    }

    for (i=0;i<num_descriptors;i++)
    {
        /* still need to implement someday - make sure if you have a last data buffer that is smaller
           than the other fixed size buffers (since it's the last one), that we can pass smaller size */
        flags = 0;
        
        if (i == 0)
        {
            flags |= DMA_SOI;
        }
        // if last descriptor, calculate actual data size FIXME
        if (i == (num_descriptors - 1))
        {
            //data_size = full_data_size - num_descriptor * data_xfer_size; FIXME
            flags |= DMA_EOI | DMA_STOP_WHEN_DONE | DMA_INT_WHEN_DONE;
        }

        printk("going to set descriptor %d starting at virt addr 0x%p to point at physical buffer addr 0x%X\n",
               i, *descriptor, buffer_location);
        set_descriptor(*descriptor, i, data_xfer_size, buffer_location, flags);
    }
    return retval;
}

// create output buffers, create output descriptors, setup regs, start the dmas
static int setup_pic_output_dma(bool eatPICData, uint32_t linesize,
                                int num_pixels_per_desc)
{
    int bufsize, descsize, retval;
    DMA_descriptor *descriptor;
    dma_addr_t phys_addr, buffer_location;
    int i, num_colors, num_descriptors, data_xfer_size;
    struct pic_handle_t *pic_handle = NULL;
    int pic_instance = 0; // FIXME hard-coding pic 0.

    pic_handle = pic_create_new_default_handle();

    // setup the interrupt callback for end of image interrupt
    pic_odma_register_callback_dma_channel(pic_handle, pictest_interrupt_callback, NULL, 0);
    pic_odma_register_callback_dma_channel(pic_handle, pictest_interrupt_callback, NULL, 1);
    pic_odma_register_callback_dma_channel(pic_handle, pictest_interrupt_callback, NULL, 2);
    pic_common_register_callback(pic_handle, pictest_debug_callback, NULL);

    if (eatPICData)
    {
        printk("%s - eating all pic data \n",__func__);
        pic_output_dma_set_disable_mode(pic_handle, PIC_OUT_DMA_DISCARD_DATA);
        retval = 0;
        goto done;  // don't need to set up descriptors when throwing out data
    }

    num_colors = picv_object.num_colors;

    pic_output_dma_set_disable_mode(pic_handle, PIC_OUT_DMA_PROCESS_DATA);

    printk("%s - setting up the output DMA now\n",__func__);

    num_descriptors = NUM_DESC;
    picv_object.num_output_descriptors = num_descriptors;

    // set up which output DMA channels trigger the PIC common interrupt
    pic_output_dma_set_common_int(pic_handle, 7, false);
    
    // set up memory for the dma to write to
    descsize = num_pixels_per_desc * NUM_PICODMA_BPS;
    bufsize = num_descriptors * descsize;

    retval = create_output_buffer(bufsize);
    if (retval != 0)
    {
        printk("ERROR!!! 0 return from create_output_buffer in %s\n",__func__);
        goto done;
    }
    printk("Getting ready to call create_output_dma_descriptors\n");
    // ack all pic output dma ints (NULL = all) - direct hardware call

    pic_do_clear_all_irqs(pic_instance);

    for (i=0; i < num_colors;i++)
    {
        data_xfer_size = descsize/num_colors;
        buffer_location = picv_object.output_data_phys + i * (bufsize/NUM_PICWDMA_CHAN);
        retval = create_output_dma_descriptors(num_descriptors, data_xfer_size, &descriptor,
                                               &phys_addr, buffer_location);
        if (retval != 0)
        {
            printk("ERROR!!!  Bad return creating output_dma_descriptors!!!\n");
            set_pictest_status(PIC_TEST_DESC_ERR);
            goto done;
        }
        picv_object.device_odma_descriptor[i] = descriptor;
        picv_object.device_odma_descriptor_phys[i] = phys_addr; 

        pic_output_dma_channel_set_burst_len(pic_handle, DMA_8_BEATS, i);
        pic_output_dma_channel_set_enable_dma(pic_handle, DMA_ENABLE, i);
        pic_output_dma_channel_set_input_data_width(pic_handle, DMA_WIDTH_16, i);
        
        pic_output_dma_channel_set_linesize(pic_handle, linesize, i);

        // enable all pic output dma ints (NULL = all)
        pic_enable_output_dma_channel_irqs(pic_handle, NULL, true, i);
    }

    pic_enable_common_irqs(pic_handle, NULL, true);
    // set the pic routing table
    pic_output_dma_set_routing(pic_handle, 0, 2, 1, 0);

    // now that all the channels are set up, send down the pic_handle
    pic_do_configure(pic_handle, pic_instance);

    printk("haven't started pic odma yet, here's the debug\n");
    pic_dump_output_dma_regs(pic_instance);
    printk("Dumping the descriptors starting at dma_addr_t 0x%X\n",
           picv_object.device_odma_descriptor_phys[0]);
    dump_descriptors(picv_object.device_odma_descriptor[0], picv_object.num_output_descriptors); 

    for (i=0; i<num_colors;i++)
    {
        printk("Now starting pic odma, channel %d, writing phys descriptor address 0x%X....\n",
               i, picv_object.device_odma_descriptor_phys[i]);

        pic_start_output_dma(pic_instance, i, picv_object.device_odma_descriptor_phys[i]);
    }
done:
    if (pic_handle)
        pic_do_free_handle(pic_handle);
    return retval;
}

#define btoa(x) ((x)?"true":"false")


static void icetest_callback(struct icetest_interrupt_info *is)
{
    printk("\nicetest interrupt!!! Values are:\n");
    printk("overflow_int=%s, dma_int=%s, underflow_int=%s\n",
           btoa(is->overflow_int), btoa(is->dma_int),
           btoa(is->underflow_int));
    if (is->overflow_int)
    {
        printk("ERROR, overflow int set\n");
        BUG_ON(0);
    }
    if (is->underflow_int)
    {
        printk("ERROR, underflow int set\n");
        BUG_ON(0);
    }
}

static void icetest_idma_callback(struct icetest_idma_interrupt_info *is)
{
    struct icetest_testconfig_info info;

//    printk("\nicetest idma interrupt!!! Values are:\n");
//    printk("rresp_int=%s, bresp_int=%s, rst_int=%s, eoi_int=%s, fin_int=%s, who_int=%s\n",
//           btoa(is->rresp_int), btoa(is->bresp_int),
//           btoa(is->rst_int),  btoa(is->eoi_int),  btoa(is->fin_int),  btoa(is->who_int));
    if (is->who_int)
    {
        printk("ERROR, who int set\n");
        BUG_ON(0);
    }
    if (is->rresp_int)
    {
        printk("ERROR, rresp int set\n");
        BUG_ON(0);
    }
    if (is->bresp_int)
    {
        printk("ERROR, bresp int set\n");
        BUG_ON(0);
    }
    if (is->eoi_int)
    {
        icetest_eoi_int = true;
    }
    if (is->fin_int)
    {
        icetest_fin_int = true;
    }
    

    // TEST for replicating what Dave's icefile test is doing - normal mode and softreset in the testconfig reg
    memset(&info, 0, sizeof(struct icetest_testconfig_info));  // set all fields invalid

    info.icetest_enable_valid = true;
    info.softreset_active_valid = true;

    /* 0==normal mode  1==test mode */
    info.icetest_enable = 0;
    /* 0==reset inactive  1==reset active */
    info.softreset_active = 1;

    icetest_set_config(&info);

}

static bool pic_interrupt_test_complete;
static bool icetest_interrupt_test_complete;

static void interrupt_test_pic_callback(struct pic_common_ints *is)
{
    printk("\npic interrupt occurred!  Interrupts values are: \n");
    printk("skew=%s, skew_odma=%s, af_wdma_common=%s, af_dma2=%s\n",
           btoa(is->skew), btoa(is->skew_odma), btoa(is->af_wdma_common),
           btoa(is->af_dma2));
    printk("af_dma1=%s, af_dma0=%s, wdma_common=%s, wdma2=%s, wdma1=%s\n",
           btoa(is->af_dma1), btoa(is->af_dma0), btoa(is->wdma_common),
           btoa(is->wdma2), btoa(is->wdma1));
    printk("wdma0=%s, ps_esd=%s, idma_2d=%s\n",btoa(is->wdma0),
           btoa(is->ps_esd), btoa(is->idma_2d));

    // Yes, we are assuming that all the interrupts come in at the same time.....
    if ((is->skew) && (is->skew_odma) && (is->af_wdma_common) &&
        (is->af_dma2) && (is->af_dma1) && (is->af_dma0) &&
        (is->wdma_common) && (is->wdma2) && (is->wdma1) &&
        (is->wdma0) && (is->ps_esd) && (is->idma_2d))
        pic_interrupt_test_complete = true;
}

static void interrupt_test_icetest_callback(struct icetest_interrupt_info *is)
{
    printk("\nicetest interrupt!!! Values are:\n");
    printk("overflow_int=%s, dma_int=%s, underflow_int=%s\n",
           btoa(is->overflow_int), btoa(is->dma_int),
           btoa(is->underflow_int));
    // Yes, we are assuming that all the interrupts come in at the same time.....
    // and we don't look for dma_int, since it can't get set unless we set the
    // lo level dma interrupts....
    if ((is->overflow_int)  && (is->underflow_int))
        icetest_interrupt_test_complete = true;
}

// set all interrupts in pic_common
static void interrupt_test(uint32_t picinstance)
{
    struct pic_handle_t *pic_handle;

    printk("SANDRA, REWRITE THIS TEST TO FUNCTION LIKE PIE's INTERRUPT TEST!!!!!!!!!!!!!!!!!! FIXME\n");
    pic_handle = pic_create_new_default_handle();

    pic_interrupt_test_complete = false;
    icetest_interrupt_test_complete = false;
    // register a callback for the pic_common interrupt
    pic_common_register_callback(pic_handle, interrupt_test_pic_callback, NULL);

    // force all common interrupts and output dma interrupts
    printk("setting and enabling the interrupts in the pic_handle\n");
    pic_force_common_irqs(pic_handle, NULL, true);
    pic_force_output_dma_irqs(pic_handle, NULL, true, 0);
    pic_force_output_dma_irqs(pic_handle, NULL, true, 1);
    pic_force_output_dma_irqs(pic_handle, NULL, true, 2);    

    pic_enable_common_irqs(pic_handle, NULL, true); // enable all interrupts
    pic_enable_output_dma_channel_irqs(pic_handle, NULL, true, 0); // enable all interrupts
    pic_enable_output_dma_channel_irqs(pic_handle, NULL, true, 1); // enable all interrupts
    pic_enable_output_dma_channel_irqs(pic_handle, NULL, true, 2); // enable all interrupts        

    pic_do_configure(pic_handle, picinstance);

    // now for icetest
    // register callback for the icetest common interrupt
    icetest_register_callback(interrupt_test_icetest_callback);
    // disable all interrupts
    icetest_disable_irq(NULL);  

    // now force all common interrupts
    printk("setting the interrupts for icetest\n");
    icetest_set_irq(NULL);   
    
    printk("now enabling the interrupts for icetest\n");
    icetest_enable_irq(NULL);

    while ((pic_interrupt_test_complete == false) || (icetest_interrupt_test_complete == false))
    {
        ssleep(1);
        printk("Waiting for pic interrupt test to complete\n");
        if ((pic_interrupt_test_complete == false) || (icetest_interrupt_test_complete == false))
        {
            printk("We didn't pass, but if you see that all pic interrupts were true at some point\n");
            printk("and icetest overflow and underflow interrupts were true, then it really did pass\n");
            return;
        }
    }
    printk("Interrupt test succeeded!! all expected interrupts were seen\n");

    pic_do_free_handle(pic_handle);
}

static void bypass_all_pic_subblocks(void)
{
    printk("stub - all blocks assumed to be bypassed at startup %s\n",__FUNCTION__);
}

static void bypass_cisx(void)
{
    printk("stub - all blocks assumed to be bypassed at startup %s\n",__FUNCTION__);
}

// free the descriptors and the data that they point to

void cleanup_input_buffers(void)
{
    dma_free_coherent(NULL, picv_object.in_bufsize,
                      picv_object.input_data,
                      picv_object.input_data_phys);
}

void cleanup_output_buffers(void)
{
    dma_free_coherent(NULL, picv_object.out_bufsize,
                      picv_object.output_data,
                      picv_object.output_data_phys);
}

void cleanup_icetest_descriptors(void)
{
    destroy_descriptors(picv_object.num_input_descriptors,
                        picv_object.icetestdma_descriptor,
                        picv_object.icetestdma_descriptor_phys, NULL);
}

void cleanup_output_descriptors(void)
{
    int i;
    for (i=0;i<picv_object.num_colors;i++)
        destroy_descriptors(picv_object.num_output_descriptors,
                            picv_object.device_odma_descriptor[i], 
                            picv_object.device_odma_descriptor_phys[i], NULL);
}

void print_output_data(uint32_t *input_data, uint16_t *output_data,
                       int num_pixels_per_desc, int num_pixels_per_line)
{
    int i, mismatch, zero, curr_color;
    uint32_t index, color_index[NUM_PICWDMA_CHAN];
    int bufsize_pixels;

    bufsize_pixels = picv_object.num_output_descriptors * num_pixels_per_desc;

    for (i=0; i < picv_object.num_colors; i++)
    {
        color_index[i] = i * (bufsize_pixels/NUM_PICWDMA_CHAN);
    }
    
    mismatch=0;
    zero=0;
    curr_color = 0; // start comparing with channel 0 - red
    index = color_index[curr_color];
    for (i=0; i < bufsize_pixels; i++) 
    {
        if (output_data[index] != (input_data[i] & 0xFFFF))
        {
            if (output_data[i] == 0)
                zero++;
            else
                mismatch++;
            if (mismatch < 20)
                printk("mismatch: od[0x%X]=0x%X, id[0x%X]=0x%X\n",i,output_data[i],i,input_data[i]);
            
        }
        index++;
        if ((index % num_pixels_per_line) == 0)
        {
            // we've finished this line of color, increment it
            color_index[curr_color] += num_pixels_per_line;
            // move to the next color
            curr_color = (curr_color + 1) % picv_object.num_colors;
            index = color_index[curr_color];
        }
    }
    printk("There were %d inappropriate 0 data, and %d mismatches\n",zero,mismatch);
    if ((zero == 0) && (mismatch == 0))
        printk("The data compared - The test HAS SUCCESSFULLY COMPLETED!!\n");
    else
    {
        set_pictest_status(PIC_TEST_COMPARE_ERR);
        printk("Failure!!!!!!!!!!\n");
    }
}

// setup icetest to feed the pic block - don't start dmas yet....
static void setup_icetest_block(void)
{
    struct icetest_testconfig_info info;

    memset(&info, 0, sizeof(struct icetest_testconfig_info)); // set all bits invalid
    
    info.pic_rate_valid = true;
    info.disable_pacing_valid = true;
    info.softreset_active_valid = true;
    info.icetest_enable_valid = true;
    info.icetest_mode_select_valid = true;
    
    if (use_picrate_val < 256)
    {
        info.pic_rate = use_picrate_val;
        info.disable_pacing = IGNORE_PIC_READY;
    }
    else
    {
        info.pic_rate = 0;  
        info.disable_pacing = USE_PIC_READY;
    }

    info.softreset_active = 0;
    info.icetest_enable = ICETEST_ENABLE;
    info.icetest_mode_select = ICETEST_SEL_CISX;  // drive data into the top of CISX
    icetest_set_config(&info);

    // clear, then enable all ICETest block interrupts
    icetest_clear_irq(NULL); // clear all ints
    icetest_register_callback(icetest_callback); // attach callback for common icetest block
    icetest_register_callback_idma(icetest_idma_callback); // attach callback for idmaicetest callback
    
    icetest_enable_irq(NULL);
    icetest_dump_regs();
}

static int setup_icetest_idma(uint32_t linesize_pixels, int num_pixels_per_desc)
{
    int i, colornum, total_num_pixels;
    int bufsize, descsize;
    int retval;
    uint32_t *input_buffer;
    uint8_t current_color;
    
    descsize = num_pixels_per_desc * NUM_ICETEST_BPS;
    bufsize = NUM_DESC * descsize;

    retval = create_input_buffer(bufsize);
    // create a buffer of data to send to PIC - creating a pixel at a time
    if (retval != 0)
    {
        printk("ERROR!!! 0 return from create_input_buffer in %s\n",__FUNCTION__);
        return retval; 
    }

    if (picv_object.num_colors == 1)
        current_color = PIC_CBI_MEVEN;
    else
        current_color = PIC_CBI_CEVEN_0;  // starting with red (FIXME, won't work with dual)
    
    input_buffer = picv_object.input_data;
    total_num_pixels = num_pixels_per_desc * NUM_DESC;
    for (i=0; i < total_num_pixels; i++)
    {
        input_buffer[i] = PIC_INDATA_DATA(i);  

        // put the color tag in the metadata part of the pixel
        input_buffer[i] |= PIC_INDATA_COLOR(current_color); // FIXME - won't work for dual (M/CODD)

        // leave monochrome as is, but for red, green and blue, mark each pixel in the buffer
        // so we can see by the output which buffer is which
        if (current_color == PIC_CBI_CEVEN_0)
            // mark red pixels with a 1 in pixels 14 and 15
            input_buffer[i] |=  (1 << 14);
        if (current_color == PIC_CBI_CEVEN_1)
            // mark green pixels with a 2 in pixels 14 and 15
            input_buffer[i] |=  (2 << 14);
        if (current_color == PIC_CBI_CEVEN_2)
            // mark blue pixels with a 3 in pixels 14 and 15
            input_buffer[i] |=  (3 << 14);

        // if it's the end of the line, tag it, then switch color
        if ((i+1) % linesize_pixels == 0)
        {
            input_buffer[i] |= PIC_INDATA_DATATYPE(PIC_CBI_EOL);
            if (current_color < PIC_CBI_MEVEN)  // if we are color, not mono, switch to next color
            {
                current_color++;
                current_color %= PIC_CBI_MEVEN; // only allow 0, 1, 2
            }
        }
        
        // ignoring the exposure (EXP field)
        // not doing a CRC test, so ignoring the LINECRC field
        // not testing dual scan (only using 1 PIC), so ignoring the TAGOUT field
        // FIXME - do a test for both PICs
    }
    // Tag first and last pixel mono, or Red, Green, and Blue 
    // FIXME - won't work for dual
    for (colornum=0; colornum < picv_object.num_colors; colornum++)
    {
        int firstpixel, lastpixel;
        
        // tag first pixel of every color as start of image (line)
        firstpixel = linesize_pixels * colornum;
        input_buffer[firstpixel] |= PIC_INDATA_DATATYPE(PIC_CBI_SOL);
        printk("first pixel at %d, value = 0x%X\n",
               firstpixel, input_buffer[firstpixel]);

        // tag the last pixel of every color as end of image
        lastpixel = (total_num_pixels - 1) - (colornum * linesize_pixels);
        input_buffer[lastpixel] |= PIC_INDATA_DATATYPE(PIC_CBI_EOI);
        printk("Last pixel at %d, value = 0x%X\n", lastpixel, input_buffer[lastpixel]);
    }
    printk("input data starts at virtaddr 0x%p, physaddr 0x%X\n",
           input_buffer, picv_object.input_data_phys);

    // create descriptors, store the first descriptor's physical address
    retval = create_icetest_dma_descriptors(NUM_DESC, descsize); // icetestdma descriptors need bytes for xfer length
    if (retval != 0)
    {
        printk("ERROR!!!  Bad return creating idma_descriptors\n");
        // cleanup the input buffer we just created
        cleanup_input_buffers();
        return retval;
    }

    // Set up the ICE_TEST_IDMA_REGS
    // cfg reg
    // set Burst Length,  output data width, enable,replicate
    icetest_set_idma_burstlen(DMA_8_BEATS);
    icetest_set_idma_dma_width(DMA_WIDTH_32);
    icetest_set_idma_enable(DMA_ENABLE);    
    icetest_set_idma_replicate(0);
    
    // this outputs an EOL at the end of the line width - set to our data length?
    // this is in number of samples
    icetest_set_idma_linewidth(linesize_pixels); 
    icetest_clear_idma_irqs(NULL); // clear all interrupts
    icetest_enable_idma_irqs(NULL);  // enable all interrupts
    return 0;
}

static void pictest_debug_callback(struct pic_common_ints *intstruct)
{
    printk("%s called - ints_set=0x%X\n", __func__, intstruct->int_array);
}

static void pictest_interrupt_callback(struct pic_output_dma_interrupt_info *intstruct)
{
    printk(" %s called, ints_set=0x%X\n", __func__, intstruct->int_array);

    printk("We really ought to check pic_instance and chan_num in intstruct to see if they are what we expect!!!!!!\n");
    
    if ((intstruct->own) || (intstruct->eoi_err) || (intstruct->eoi_align_err) ||
        (intstruct->eol_align_err) || (intstruct->chg_line_align_err) || (intstruct->dir_err))
    {
        printk("ERROR!! error interrupt! debug_reg=0x%X\n", intstruct->int_array);
        pic_error_interrupt = true;
    }

    if (intstruct->eoi)
    {
        printk("End of image received\n");
        pic_end_of_image_interrupt = true;
    }
}

// this is the main test
int run_dma_test(bool eatPICData, bool color,
                 int num_bytes_per_line_output, int num_pixels_per_desc,
                 int num_pixels_per_line)
{
    int i;

    if (color)
        picv_object.num_colors = NUM_PICWDMA_CHAN;
    else
        picv_object.num_colors = 1;

    pic_end_of_image_interrupt = false;
    pic_error_interrupt = false;
    icetest_fin_int = false;
    icetest_eoi_int = false;
    printk("Executing the main test (Data from ICETest through PIC to RAM and compare)\n");

    // for odma, linesize is in bytes - setup odma and start
    if (setup_pic_output_dma(eatPICData, num_bytes_per_line_output, num_pixels_per_desc) != 0)
    {
        printk("ERROR setting up pic wdma, EXITING TEST\n");
        return -1;
    }
    setup_icetest_block();
    // for idma, linesize is in pixels - we have 4 byte pixels, generating NUM_PICODMA_BPS size output pixels
    if (setup_icetest_idma(num_pixels_per_line, num_pixels_per_desc) != 0)
    {
        printk("ERROR: setting up icetest_idma - EXITING TEST\n");
        // cleanup the output descriptors and buffers
        cleanup_output_buffers();
        cleanup_output_descriptors();
        
        return -1;
    }
    bypass_cisx();
    bypass_all_pic_subblocks();
#if 0
    printk("going to dump the icetest idma\n");
    for (i=0;i<picv_object.num_colors;i++)
    {
        printk("pic WDMA descriptors channel %d\n", i);
        dump_descriptors(picv_object.device_odma_descriptor[i],
                         picv_object.num_output_descriptors);
    }
#endif        
    
    pic_dump(0);
    pic_dump(1);

    icetest_dump_regs();
    icetest_dump_idma_regs();
    
    printk("icetest descriptors\n");
    dump_descriptors(picv_object.icetestdma_descriptor,
                      picv_object.num_input_descriptors);
    printk("\nStarting ICETest dma at 0x%X\n",
           picv_object.icetestdma_descriptor_phys);
    icetest_start_idma(picv_object.icetestdma_descriptor_phys);
//    pic_dump_common_regs(0);
//    pic_dump_common_regs(1);
//    pic_dump_output_dma_regs(0);
//    pic_dump_output_dma_regs(1);
//    icetest_dump_regs();
//    icetest_dump_idma_regs();
    // now wait until we get end of image interrupt back
    if (!eatPICData)
    {
        while ((pic_end_of_image_interrupt == false) && (pic_error_interrupt == false))
        {
            printk("Waiting for PIC end of image interrupt (or pic error)\n");
            ssleep(1);
        }
    }
    while ((icetest_eoi_int == false) || (icetest_fin_int == false))
    {
        printk("Waiting for icetest end of image and finished interrupts, eoi=%s, fin=%s\n",
               btoa(icetest_eoi_int), btoa(icetest_fin_int));
        ssleep(1);
    }
    
    printk("icetest IDMA descriptors\n");
    dump_descriptors(picv_object.icetestdma_descriptor,
                     picv_object.num_input_descriptors); 
    if (!eatPICData)
    {
        printk("pic WDMA descriptors\n");
        for (i=0;i<picv_object.num_colors;i++)
        {
            dump_descriptors(picv_object.device_odma_descriptor[i],
                             picv_object.num_output_descriptors);
        }
        dma_test_has_run = true;
    }
    else
    {
        printk("eating pic data, no descriptors to output\n");
    }
    
    // don't compare the output data if we didn't run anything through
    if (!eatPICData)
        print_output_data(picv_object.input_data, picv_object.output_data, num_pixels_per_desc, num_pixels_per_line);
    else
        printk("Not comparing the PIC output data when in eat PIC data mode\n");
    
    if (pic_error_interrupt == true)
    {
        printk("ERROR!!! Even if the data did compare, we have error interrupts!!  FAIL!!!!\n");
        set_pictest_status(PIC_TEST_BAD_INTERRUPT);
    }
    
    // free the memory buffers, then delete the descriptors that point to them
    cleanup_input_buffers();
    cleanup_output_buffers();
    // cleanup the icetest descriptors
    cleanup_icetest_descriptors();
    // cleanup the output descriptors
    cleanup_output_descriptors();
    return 0;
}

// start of functions callable by sysfs
// sysfs interface definitions
ssize_t run_test_set(struct device *dev, struct device_attribute *attr,
                     const char *buf, size_t count)
{
    char *mybuf;
    uint32_t uval32;
    int val;
    char command[32];
    static bool eatPICData=false;
    int i;
    uint32_t *data_buf;
    uint16_t *data16_buf;
    int num_pixels_per_line;
    int num_pixels_per_desc;
    int temp_bytes_per_line_output;
    const bool mono_test=false;
    const bool color_test=true;

    // before anything else, make sure the number of bytes in the line is an integral number
    // of 32 bit words to avoid EOL alignment errors
    BUG_ON(global_num_bytes_per_line_output % 4 != 0);

    set_pictest_status(TEST_IN_PROGRESS);

    // NOTE: The number of bytes per line on output is specified,
    // so we compute number of pixels per line, which will be the
    // same for input and output.
    num_pixels_per_line = global_num_bytes_per_line_output/NUM_PICODMA_BPS;
    // set the value for the number of pixels per descriptor
    num_pixels_per_desc = num_pixels_per_line * NUM_LINES_PER_DESC;
    
    mybuf = (char *) buf;
    
    // take out the newline in the buffer
    if (buf[count-1] == '\n')
    {
        mybuf[count-1] = '\0';
    }
    switch(mybuf[0])
    {
    case 'a':
        printk("\ndumping pic registers\n");
        pic_dump(0);
        pic_dump(1);
        printk("dumping icetest registers\n");
        icetest_dump_regs();
        icetest_dump_idma_regs();
        break;
    case 'd':
        if (!dma_test_has_run)
        {
            printk("\nCan't dump the descriptors until the dma test has run\n");
            break;
        }
        printk("\ndumping all icetest and pic regs and descriptors\n");
        icetest_dump_regs();
        icetest_dump_idma_regs();
        printk("icetest IDMA descriptors at address 0x%p\n",
               picv_object.icetestdma_descriptor);
        dump_descriptors(picv_object.icetestdma_descriptor,
                         picv_object.num_input_descriptors);
        pic_dump(0);
        pic_dump(1);
        
        for (i=0;i<picv_object.num_colors;i++)
        {
            printk("pic output descriptors at address 0x%p\n",
                   picv_object.device_odma_descriptor[i]);
            dump_descriptors(picv_object.device_odma_descriptor[i],
                             picv_object.num_output_descriptors);
        }
        break;
    case 'e':
        sscanf(mybuf,"%s%d", command, &uval32);
        if (uval32 == 0)
        {
            eatPICData = false;
            printk("\nnormal PIC WDMA - don't eat the data\n");
        }
        else
        {
            eatPICData = true;
            printk("\nPIC WDMA will now eat all the data\n");
        }
        break;
    case 'i':
        // interrupt test
        sscanf(mybuf,"%s%x", command, &uval32);
        if (uval32 > 1)
            printk("\nERROR!  no instance > 1 supported in the interrupt test\n");
        else
            interrupt_test(uval32);
        break;
    case 'm':
        if (!dma_test_has_run)
        {
            printk("\nCan't dump the descriptors until the dma test has run\n");
            break;
        }
        // dump the memory for data in and data out
        printk("\ninput data");
        data_buf = picv_object.input_data;
        for (i=0;(i < num_pixels_per_desc * NUM_DESC);i++)
        {
            if (i%4 == 0)
                printk("\n");
            printk("id[0x%X]=0x%X ",i, data_buf[i]);
        }
        printk("\noutput data");
        data16_buf = picv_object.output_data;
        for (i=0;(i < num_pixels_per_desc * NUM_DESC);i++)
        {
            if (i%4 == 0)
                printk("\n");
            printk("od[0x%X]=0x%X ",i, data16_buf[i]);
        }
        break;
    case 'n':
        // set the number of pixels per line (we compute and store as bytes per line output)
        sscanf(mybuf,"%s%d", command, &uval32);
        if (uval32 == 0)
        {
            printk("\nThe current number of pixels per line is %d\n",
                   global_num_bytes_per_line_output/NUM_PICODMA_BPS);
            break;
        }
        if (uval32 > 100000)
        {
            printk("Error, number of pixels per line must be < 100000\n");
            break;
        }
        temp_bytes_per_line_output = uval32 * NUM_PICODMA_BPS;
        if (temp_bytes_per_line_output % 4 != 0)
            printk("\nERROR! number of pixels per line needs to be aligned\n");
        else
            global_num_bytes_per_line_output = temp_bytes_per_line_output;
        break;
    case 'p':
        // set the PICRate in ICETest - if this is set, don't disable pacing in ICETest
        // set back to 256 to use handshaking (the normal case)
        printk("\nreturn value from sscanf was %d\n",sscanf(mybuf,"%s%d",command,&val));
        printk("Setting picrate to %d\n",val);
        use_picrate_val = val;
        break;
    case 'r':
        printk("\nclearing reset for pic, icetest and icetestdma\n");
        pic_do_reset(0);
        pic_do_reset(1);
        icetest_soft_reset(0);
        icetest_soft_reset(1);        
        icetest_idma_soft_reset(0);
        icetest_idma_soft_reset(1);        
        break;
    case 's':
        printk("\nresetting pic, icetest and icetestdma\n");
        pic_do_reset(0);
        pic_do_reset(1);
        icetest_soft_reset(1);
        icetest_idma_soft_reset(1);
        break;
    case 't':
        // here's the real test
        run_dma_test(eatPICData, mono_test, global_num_bytes_per_line_output,
                     num_pixels_per_desc, num_pixels_per_line);  // run mono test
        if (num_pixels_per_desc % NUM_PICWDMA_CHAN != 0)
        {
            printk("for a color test, the number of pixels per descriptor must be divisible by %d\n",
                   NUM_PICWDMA_CHAN);
            printk("%d is not; the color test will not run - EXITING NOW\n", num_pixels_per_desc);
            break;
        }

        printk("*************** MONO TEST COMPLETE *******************\n");
        printk("*************** STARTING COLOR TEST *******************\n");
        run_dma_test(eatPICData, color_test, global_num_bytes_per_line_output,
                     num_pixels_per_desc, num_pixels_per_line);  // run color test         

        finalize_pictest_status();
        break;
        
    default:
        printk("\nThere is no test triggered by character ->%s<-\n",buf);
    }
    
    return count;
}


char *usagebuf=" \n\
a=basic reg read \n\
d=dump all regs and descriptors  \n\
e 1=set PIC ODMA to throw away data  \n\
e 0=reset PIC ODMA to process data  \n\
i <pic block#>=interrupt test - set pic common  \n\
m=dump memory buffers  \n\
n <num_pixels>=set base10 number of pixels per line (set to 0 to see current value)\n \
p <picrate>=set picrate in ICETEST - set back to 256 to use handshaking (normal case)   \n\
s=reset PIC, icetest and icetest dmas  (warning, execute the \"r\" command after this!)\n\
r=clear reset for \"s\"  \n\
t=run dma test\n";
    
ssize_t run_test_get(struct device *dev, struct device_attribute *attr,
                        char *buf)
{
    return scnprintf(buf, PAGE_SIZE, usagebuf);
}

// register sysfs functions for test code
DEVICE_ATTR(pictest, S_IWUSR | S_IRUGO, run_test_get, run_test_set);

static struct attribute *pictest_attrs[] = {
    &dev_attr_pictest.attr,
    NULL,
};

struct attribute_group pic_verification_attrgrp = {
    .name = "pictest",
    .attrs = pictest_attrs,
};

// end of register sysfs functions for test code

// module parameter - easy way to communicate status to user space
static ulong pictest_status=TEST_LOADED;
module_param(pictest_status, ulong, S_IRUSR);

void set_pictest_status(ulong status)
{
    pictest_status = status;
    printk("Just set the pictest status to %lu\n", status);
}

// if we've come to the end of all the tests, and we didn't log any errors
// (which we know since test status would be TEST_IN_PROGRESS), then set
// the status to be PIC_TEST_PASS.
void finalize_pictest_status(void)
{
    if (pictest_status == TEST_IN_PROGRESS)
    {
        set_pictest_status(PIC_TEST_PASS);
    }
}

struct kobject *kobj;

static int __init pic_verification_init(void)
{
    int retcode;
    
    use_picrate_val = 256;
    // create a sysfs interface
    kobj = kobject_create_and_add("pic_verification", firmware_kobj);
    retcode = sysfs_create_group(kobj, &pic_verification_attrgrp);    
    if( retcode != 0 )
    {
        printk( KERN_ERR "%s sysfs_create_file icetest failed retcode=%d\n",
                __func__, retcode );
    }
    return 0;
}
module_init(pic_verification_init);

static void __exit pic_verification_exit(void)
{
    sysfs_remove_group(kobj, &pic_verification_attrgrp);
    kobject_put(kobj);  // tell the system it can release the object
    printk(KERN_ERR "PIC Verification: device remove complete\n");
}
module_exit(pic_verification_exit);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell PIC verification Kernel Module");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_Feb_12");

