#! /usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2014, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

#

import sys

###############################################################
#### Tweak this stuff to your needs:
###############################################################

# If the file contains things you don't want, consider making a copy of the original and deleting
# the stuff you don't want
regmasks_file = 'asic/G2/include/CISX_int_regmasks.h'

# used as common prefix for a few pieces of generated code
block_name = 'cisx_int'

# Start by enabling just show_register_info until you get parse_mask_line correct,
# and then enable print_structs/print_getters/print_setters/print_func_ptrs
# as wanted, pasting into your code
show_register_info    = False # Used to check if the script is parsing things correctly
print_structs         = False
print_getters         = False
print_setters         = False
print_func_ptrs       = False
print_tl_getter       = True
print_tl_setter       = True
print_struct_defaults = False

def findnth(source, target, n): 
    num = 0 
    start = -1 
    while num < n: 
        start = source.find(target, start+1) 
        if start == -1: return -1 
        num += 1 
    return start 

def replacenth(source, old, new, n): 
    p = findnth(source, old, n) 
    if n == -1: return source 
    return source[:p] + new + source[p+len(old):] 

# It's up to you to do a little parsing. You should be able to do minor tweaks on this.
def parse_mask_line(line):
    """
    line -- a regmasks.h line that shows the MASK field, just like the hardware team produces them

    Return a tuple of this form: (reg_name, field_name, mask) -- as strings.

    If you don't want to use this line, return (None, None, None).

    Ex: line is '#define  SCAN_SCFG2_AFEWIDTH_MASK 0xc000000'
            return ('SCFG2', 'AFEWIDTH', '0xc000000')

    Ex: line is '#define  SCAN_TOP_CTL_LVDSAFE1_LINESYNC_SEL_MASK 0x60'
            return ('CTL', 'LVDSAFE1_LINESYNC_SEL', '0xc000000')

    Ex: line is '#define  LVDSAFE1_REV0_MAJ_MASK 0xffff0000'
            return (None, None, None) # Don't want to generate for this one
    """

    # Brave people use regexes. I am not a brave man.

    # strip #define and _MASK
    line = line.replace("#define  ", "").replace("_MASK", "")

    reg_name, mask = line.split(" ")

    if reg_name.startswith("CISX_INT_"):
        reg_name = reg_name.replace("CISX_INT_", "", 1) # Strip prefix
        #print " reg_name: " + reg_name

        if reg_name.startswith("ODMA_TBL_"):
            reg_name = replacenth(reg_name, "_", " ", 3)
        elif reg_name.startswith("TBL_CNT"):
            reg_name = replacenth(reg_name, "_", " ", 2)
        elif reg_name.startswith("COLOR_OUT_TBL_"):
            reg_name = replacenth(reg_name, "_", " ", 4)
        else:
            reg_name = replacenth(reg_name, "_", " ", 1)

        #print " reg_name: " + reg_name
        reg_name, field_name = reg_name.split(" ")
        return reg_name, field_name, mask
    else:
        return (None, None, None)


###############################################################
##### Begin stuff you shouldn't have to care about. Theoretically.
###############################################################

def hex_to_bin(mask):
    scale = 16 ## equals to hexadecimal
    num_of_bits = 32
    return bin(int(mask, scale))[2:].zfill(num_of_bits)

class Field():
    """
    Represents a field within a register
    """
    def __init__(self, field_name, mask):
        self.field_name = field_name
        self.mask = mask
        self.bitcount = self.count_ones(mask)

    def __str__(self):
        return self.field_name.rjust(12) + ": " + str(self.bitcount) + " bits"

    def count_ones(self, mask):
        bin_str = hex_to_bin(mask)
        return bin_str.count('1')

    def get_ctype(self):
        if self.bitcount >= 1 and self.bitcount <= 8:
            return "uint8_t"
        elif self.bitcount >= 9 and self.bitcount <= 16:
            return "uint16_t"
        elif self.bitcount >= 17 and self.bitcount <= 32:
            return "uint32_t"
        else:
            print "Invalid bitcount: " + self.bitcount + "! Aborting!"
            sys.exit(-1)

    def safe_field_name(self):
        keywords = [ 'auto', 'break', 'case', 'char', 'const',
                     'continue', 'default', 'do', 'double',
                     'else', 'enum', 'extern', 'float', 'for',
                     'goto', 'if', 'int', 'long', 'register',
                     'return', 'short', 'signed', 'sizeof', 'static',
                     'struct', 'switch', 'typedef', 'union',
                     'unsigned', 'void', 'volatile', 'while' ]

        if self.field_name.lower() in keywords:
            return "field_" + self.field_name.lower()
        else:
            return self.field_name.lower()


class Register():
    """
    Represents a full 32-bit register, with all of it's fields
    """
    def __init__(self, name):
        self.name = name
        self.field_names = []   # Use a list to preserve order
        self.fields      = {}   # Use a dict to contain Field objects

    def __str__(self):
        ret = "[reg " + self.name
        for f in self.field_names: # Don't use valid_fields so we can see 'reserved' fields
            ret += "\n   " + str(self.fields[f])
        ret += "]"
        return ret

    def addfield(self, field_name, mask):
        self.field_names.append(field_name)
        self.fields[field_name] = Field(field_name, mask)

    def all_fields(self):
        return [self.fields[f] for f in self.field_names]

    def valid_fields(self):
        return [self.fields[f] for f in self.field_names if 'reserved' not in f.lower()]

    def generate_struct(self):
        ret = "struct %s_%s_reg\n{\n" % (block_name, self.name)
        for f in self.valid_fields():
            spaced_ctype      = f.get_ctype().ljust(8)
            spaced_field_name = (f.safe_field_name() + ';').ljust(12)
            ret += "    %s %s  // %s bits\n" % (spaced_ctype, spaced_field_name, f.bitcount)
        ret += "\n"
        for f in self.valid_fields():
            ret += "    bool " + f.safe_field_name() + "_valid;\n"
        ret += "};\n"
        return ret

    def generate_getter(self):
        ret  = "void get_%s_%s(%sDeviceHandle *device_data, struct %s_%s_reg *cfg)\n{\n" \
                   % (block_name, self.name, block_name, block_name, self.name)
        ret += "    uint32_t reg;\n\n"
        ret += "    reg = %sRead(%s);\n" % (block_name, self.name)
        for f in self.valid_fields():
            ret += "    cfg->%s = CISX_INT_%s_%s_MASK_SHIFT(reg);\n" % (f.safe_field_name(), self.name, f.field_name)
        ret += "}\n"
        return ret

    def generate_setter(self):
        ret  = "void set_%s_%s(%sDeviceHandle *device_data, struct %s_%s_reg *cfg)\n{\n" \
                   % (block_name, self.name, block_name, block_name, self.name)
        ret += "    uint32_t reg;\n\n"
        ret += "    PROTECT_REG_ACCESS;\n"
        ret += "    reg = %sRead(%s);\n" % (block_name, self.name)
        for f in self.valid_fields():
            ret += "    if(cfg->%s_valid)\n" % f.safe_field_name()
            ret += "        reg = CISX_INT_%s_%s_REPLACE_VAL(reg, cfg->%s);\n" % (self.name, f.field_name, f.safe_field_name())
        ret += "    %sWrite(%s, reg);\n" % (block_name, self.name)
        ret += "    UNPROTECT_REG_ACCESS;\n"
        ret += "}\n"
        return ret

    def generate_funcptrs(self):
        ret  = "    void (*get_%s_%s) (%sDeviceHandle *device_data, struct %s_%s_reg *cfg);\n" \
                   % (block_name, self.name, block_name, block_name, self.name)
        ret += "    void (*set_%s_%s) (%sDeviceHandle *device_data, struct %s_%s_reg *cfg);\n" \
                   % (block_name, self.name, block_name, block_name, self.name)
        return ret

    def generate_toplevel_getters(self):
        ret  = "void %s_get_%s(uint8_t cisx_instance, struct %s_%s_reg *cfg)\n{\n" \
                % (block_name, self.name, block_name, self.name)
        ret += "    %s_RETRIEVE_handle_ft;\n\n" % block_name.upper()
        #ret += "    BUG_ON(ft->get_%s_%s == NULL);\n" \
        #            % (block_name, self.name)
        ret += "    ft->get_%s_%s(int_handle, cfg);\n" \
                    % (block_name, self.name)
        ret += "}\n"
        ret += "EXPORT_SYMBOL(%s_get_%s);\n" % (block_name, self.name)
        return ret

    def generate_toplevel_setters(self):
        ret  = "void %s_set_%s(uint8_t cisx_instance, struct %s_%s_reg *cfg)\n{\n" \
                % (block_name, self.name, block_name, self.name)
        ret += "    %s_RETRIEVE_handle_ft;\n\n" % block_name.upper()
        #ret += "    BUG_ON(ft->set_%s_%s == NULL);\n" \
        #            % (block_name, self.name)
        ret += "    ft->set_%s_%s(int_handle, cfg);\n" \
                    % (block_name, self.name)
        ret += "}\n"
        ret += "EXPORT_SYMBOL(%s_set_%s);\n" % (block_name, self.name)
        return ret

    def generate_default_struct(self, reset_hex):
        binary_register = hex_to_bin(reset_hex)

        bits_so_far = 0
        ret =  "// structure based on default hex %s\n" % reset_hex
        ret += "static struct %s_%s_reg %s_default = {\n" % (block_name, self.name, self.name.lower())
        for f in self.all_fields():
            binary_field = binary_register[bits_so_far:bits_so_far+f.bitcount]
            bits_so_far += f.bitcount
            hex_field = hex(int(binary_field, 2))

            comment_str = ""
            extra_padding = '  '
            if 'reserved' in f.safe_field_name():
                comment_str = "//"
                extra_padding = ''
            ret += "    %s.%s %s= %s // 0b%s\n" % \
                        (comment_str, f.safe_field_name().ljust(14), extra_padding, (hex_field + ',').ljust(8), binary_field)
        ret += "\n"
        for f in self.valid_fields():
            ret += "    ." + (f.safe_field_name() + "_valid").ljust(16) + " = true,\n"
        ret += "};\n"
        return ret


def create_struct_defaults(regs):
    defaults = [
        ('SCFG1',  '0x0000f000'),  # SCAN_CFG1_R
        ('SCFG2',  '0x00000000'),  # SCAN_CFG2_R
        ('SCFG3',  '0x00000000'),  # SCAN_CFG3_R
        ('SCTRL',  '0x00000000'),  # SCAN_CTRL_R
        ('SCANX',  '0x00000000'),  # SCAN_X_R
        ('SCYCLE', '0x00000000'),  # SCAN_CYCLE_R
        ('STCFG1', '0x00ff00ff'),  # SCAN_START_CFG1_R
        ('STCFG2', '0x00ff00ff'),  # SCAN_START_CFG2_R
        ('STCFG3', '0xffffffff'),  # SCAN_START_CFG3_R
        ('STCFG4', '0xffffffff'),  # SCAN_START_CFG4_R
        ('CCFG',   '0x00ffffff'),  # SCAN_CLMP_CFG_R
        ('SCLK1',  '0xffffffff'),  # SCAN_SENS_CLK1_R
        ('SCLK2',  '0xffffffff'),  # SCAN_SENS_CLK2_R
        ('SCLK3',  '0xffffffff'),  # SCAN_SENS_CLK3_R
        ('SCLK4',  '0xffffffff'),  # SCAN_SENS_CLK4_R
        ('SCLK5',  '0xffffffff'),  # SCAN_SENS_CLK5_R
        ('SCLK6',  '0xffffffff'),  # SCAN_SENS_CLK6_R
        ('ACLK1',  '0xffffffff'),  # SCAN_AFE_CLK1_R
        ('ACLK2',  '0xffffffff'),  # SCAN_AFE_CLK2_R
        ('ACLK3',  '0xffffffff'),  # SCAN_AFE_CLK3_R
        ('ACLK4',  '0xffffffff'),  # SCAN_AFE_CLK4_R
        ('ACLK5',  '0xffffffff'),  # SCAN_AFE_CLK5_R
        ('LDATA10', '0xff00ff00'),  # SCAN_AFE_LDATA1_R
        ('LDATA11', '0xff00ff00'),  # SCAN_AFE_LDATA1_R
        ('LDATA20', '0xff00ff00'),  # SCAN_AFE_LDATA2_R
        ('LDATA21', '0xff00ff00'),  # SCAN_AFE_LDATA2_R
        ('LDATA30', '0xff00ff00'),  # SCAN_AFE_LDATA3_R
        ('LDATA31', '0xff00ff00'),  # SCAN_AFE_LDATA3_R
        ('LDATA40', '0xff00ff00'),  # SCAN_AFE_LDATA4_R
        ('LDATA41', '0xff00ff00'),  # SCAN_AFE_LDATA4_R
        ('LDATA50', '0xff00ff00'),  # SCAN_AFE_LDATA5_R
        ('LDATA51', '0xff00ff00'),  # SCAN_AFE_LDATA5_R
        ('LDATA60', '0xff00ff00'),  # SCAN_AFE_LDATA6_R
        ('LDATA61', '0xff00ff00'),  # SCAN_AFE_LDATA6_R
        ('PSEQ1',  '0x00000000'),  # SCAN_AFE_PSEQ1_R
        ('PSEQ2',  '0x00000000'),  # SCAN_AFE_PSEQ2_R
        ('BPWM',   '0x0fff0000'),  # SCAN_BULB_PWM_R
        ('CFGARB', '0x00000001'),  # SCAN_CFGARB_R
    ]
    for reg_name, reset_hex in defaults:
        print regs[reg_name].generate_default_struct(reset_hex)

def main():
    f = open(regmasks_file)

    # This list will preserve the correct order (which the dict below doesn't)
    reg_names = [ ]

    # key = reg_name, val = Register object
    regs = {}

    while True:
        line = f.readline()
        if not line: break;
        line = line.strip()

        # Only look for the lines with "MASK " (note the space to not clash with MASK_SHIFT)
        if "MASK " in line:
            reg_name, field_name, mask = parse_mask_line(line)

            if reg_name and field_name and mask:

                if reg_name not in reg_names:
                    reg_names.append(reg_name)
                    # Create the register
                    regs[reg_name] = Register(reg_name)

                # Register exists, now add the field
                regs[reg_name].addfield(field_name, mask)



    # Iterate across reg_names to preserve order
    for reg in reg_names:
        if show_register_info:
            print "-" * 15
            print regs[reg]
        if print_structs:
            print regs[reg].generate_struct()
        if print_getters:
            print regs[reg].generate_getter()
        if print_setters:
            print regs[reg].generate_setter()
        if print_func_ptrs:
            print regs[reg].generate_funcptrs()
        if print_tl_getter:
            print regs[reg].generate_toplevel_getters()
        if print_tl_setter:
            print regs[reg].generate_toplevel_setters()
    if print_struct_defaults:
        create_struct_defaults(regs)

    print "NOTE: This is dumb. It generates getters and setters, even where the register may be read only or write only. You have to clean it up."
    print "NOTE: It also does not generate interrupts correctly"

if __name__ == "__main__":
    main()
