/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

// driver for the scanblk common subblock

#include <linux/irqreturn.h> // needed for irqreturn_t and IRQ_HANDLED
#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON
#include <linux/io.h> // for ioread/write32 (Read/Write macros)
#include <linux/export.h> // for EXPORT_SYMBOL
#include <linux/delay.h>
#include <linux/string.h>  // for memset

#include "scanblk_if.h"
#include "scanblk_data.h"
#include "scanblk_common.h"
#include "scanblk_driver.h"

#include "SCAN_TOP_regstructs.h"
#include "SCAN_TOP_regmasks.h"

/*****************************************************
 *********** RESET REGISTERS *************************
 *****************************************************/


// structure based on default hex 0x0000f000
static struct scanblk_SCFG1_cfg scfg1_default = {
    .p1clken          = 0x0,     // 0b0
    .p2clken          = 0x0,     // 0b0
    .rsclken          = 0x0,     // 0b0
    .cpclken          = 0x0,     // 0b0
    .mclken           = 0x0,     // 0b0
    .vsclken          = 0x0,     // 0b0
    .clclken          = 0x0,     // 0b0
    .lclmpen          = 0x0,     // 0b0
    //.reserved1      = 0x0,     // 0b000
    .bulben           = 0x0,     // 0b0
    .ssmen            = 0x0,     // 0b0000
    .smpol            = 0xf,     // 0b1111
    .p1apol           = 0x0,     // 0b0
    //.reserved2      = 0x0,     // 0b0
    .sptog            = 0x0,     // 0b0
    .sppol            = 0x0,     // 0b0
    .scpol            = 0x0,     // 0b0000
    .acpol            = 0x0,     // 0b0000

    .p1clken_valid    = true,
    .p2clken_valid    = true,
    .rsclken_valid    = true,
    .cpclken_valid    = true,
    .mclken_valid     = true,
    .vsclken_valid    = true,
    .clclken_valid    = true,
    .lclmpen_valid    = true,
    .bulben_valid     = true,
    .ssmen_valid      = true,
    .smpol_valid      = true,
    .p1apol_valid     = true,
    .sptog_valid      = true,
    .sppol_valid      = true,
    .scpol_valid      = true,
    .acpol_valid      = true,
};

// structure based on default hex 0x00000000
static struct scanblk_SCFG2_cfg scfg2_default = {
    .cmode            = 0x0,     // 0b0
    .stype            = 0x0,     // 0b0
    .ssel             = 0x0,     // 0b00
    .afewidth         = 0x0,     // 0b00
    //.reserved1      = 0x0,     // 0b0000
    .leden1           = 0x0,     // 0b000
    .leden0           = 0x0,     // 0b000
    //.reserved2      = 0x0,     // 0b00000000
    .pixper           = 0x0,     // 0b00000000

    .cmode_valid      = true,
    .stype_valid      = true,
    .ssel_valid       = true,
    .afewidth_valid   = true,
    .leden1_valid     = true,
    .leden0_valid     = true,
    .pixper_valid     = true,
};

// structure based on default hex 0x00000000
static struct scanblk_SCFG3_cfg scfg3_default = {
    .p1rise           = 0x0,     // 0b00
    .p1fall           = 0x0,     // 0b00
    .p2rise           = 0x0,     // 0b00
    .p2fall           = 0x0,     // 0b00
    .rsrise           = 0x0,     // 0b00
    .rsfall           = 0x0,     // 0b00
    .cprise           = 0x0,     // 0b00
    .cpfall           = 0x0,     // 0b00
    .mcrise           = 0x0,     // 0b00
    .mcfall           = 0x0,     // 0b00
    .vsrise           = 0x0,     // 0b00
    .vsfall           = 0x0,     // 0b00
    .cd1rise          = 0x0,     // 0b00
    .cd1fall          = 0x0,     // 0b00
    .sprise           = 0x0,     // 0b00
    .spfall           = 0x0,     // 0b00

    .p1rise_valid     = true,
    .p1fall_valid     = true,
    .p2rise_valid     = true,
    .p2fall_valid     = true,
    .rsrise_valid     = true,
    .rsfall_valid     = true,
    .cprise_valid     = true,
    .cpfall_valid     = true,
    .mcrise_valid     = true,
    .mcfall_valid     = true,
    .vsrise_valid     = true,
    .vsfall_valid     = true,
    .cd1rise_valid    = true,
    .cd1fall_valid    = true,
    .sprise_valid     = true,
    .spfall_valid     = true,
};

// structure based on default hex 0x00000000
static struct scanblk_SCTRL_cfg sctrl_default = {
    //.reserved1      = 0x0,     // 0b000000000000000000000000
    .umsen            = 0x0,     // 0b0
    .cmdqen           = 0x0,     // 0b0
    .termcmd          = 0x0,     // 0b0
    .sreset           = 0x0,     // 0b0
    //.reserved2      = 0x0,     // 0b0
    .scmdie           = 0x0,     // 0b0
    .run              = 0x0,     // 0b0
    .scanen           = 0x0,     // 0b0

    .umsen_valid      = true,
    .cmdqen_valid     = true,
    .termcmd_valid    = true,
    .sreset_valid     = true,
    .scmdie_valid     = true,
    .run_valid        = true,
    .scanen_valid     = true,
};

// structure based on default hex 0x00000000
static struct scanblk_SCANX_cfg scanx_default = {
    .expval           = 0x0,     // 0b0000000000000000
    .scanx            = 0x0,     // 0b0000000000000000

    .expval_valid     = true,
    .scanx_valid      = true,
};

// structure based on default hex 0x00000000
static struct scanblk_SCYCLE_cfg scycle_default = {
    //.reserved1      = 0x0,     // 0b00000000000000000000000000
    .scycle           = 0x0,     // 0b000000

    .scycle_valid     = true,
};

// structure based on default hex 0x00ff00ff
static struct scanblk_STCFG1_cfg stcfg1_default = {
    .spgen            = 0x0,     // 0b0
    .sp2ctrl          = 0x0,     // 0b000
    .mskextd          = 0x0,     // 0b0000
    .ssme1            = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00000000
    .ssme2            = 0xff,    // 0b11111111

    .spgen_valid      = true,
    .sp2ctrl_valid    = true,
    .mskextd_valid    = true,
    .ssme1_valid      = true,
    .ssme2_valid      = true,
};

// structure based on default hex 0x00ff00ff
static struct scanblk_STCFG2_cfg stcfg2_default = {
    .spe1_2           = 0x0,     // 0b00000000
    .spe1             = 0xff,    // 0b11111111
    .spe2_2           = 0x0,     // 0b00000000
    .spe2             = 0xff,    // 0b11111111

    .spe1_2_valid     = true,
    .spe1_valid       = true,
    .spe2_2_valid     = true,
    .spe2_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_STCFG3_cfg stcfg3_default = {
    .aph1e1           = 0xff,    // 0b11111111
    .aph1e2           = 0xff,    // 0b11111111
    .aph1e3           = 0xff,    // 0b11111111
    .aph1e4           = 0xff,    // 0b11111111

    .aph1e1_valid     = true,
    .aph1e2_valid     = true,
    .aph1e3_valid     = true,
    .aph1e4_valid     = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_STCFG4_cfg stcfg4_default = {
    .aph1e5           = 0xff,    // 0b11111111
    .aph1e6           = 0xff,    // 0b11111111
    .aph1e7           = 0xff,    // 0b11111111
    .aph1e8           = 0xff,    // 0b11111111

    .aph1e5_valid     = true,
    .aph1e6_valid     = true,
    .aph1e7_valid     = true,
    .aph1e8_valid     = true,
};

// structure based on default hex 0x00ffffff
static struct scanblk_CCFG_cfg ccfg_default = {
    .cpmode           = 0x0,     // 0b0
    .clmode           = 0x0,     // 0b0
    .clsel            = 0x0,     // 0b0
    //.reserved1      = 0x0,     // 0b00000
    .cle1             = 0xff,    // 0b11111111
    .ccle2            = 0xff,    // 0b11111111
    .acle2            = 0xff,    // 0b11111111

    .cpmode_valid     = true,
    .clmode_valid     = true,
    .clsel_valid      = true,
    .cle1_valid       = true,
    .ccle2_valid      = true,
    .acle2_valid      = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK1_cfg sclk1_default = {
    .p1e1             = 0xff,    // 0b11111111
    .p1e2             = 0xff,    // 0b11111111
    .p1e3             = 0xff,    // 0b11111111
    .p1e4             = 0xff,    // 0b11111111

    .p1e1_valid       = true,
    .p1e2_valid       = true,
    .p1e3_valid       = true,
    .p1e4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK2_cfg sclk2_default = {
    .p1e5             = 0xff,    // 0b11111111
    .p1e6             = 0xff,    // 0b11111111
    .p1e7             = 0xff,    // 0b11111111
    .p1e8             = 0xff,    // 0b11111111

    .p1e5_valid       = true,
    .p1e6_valid       = true,
    .p1e7_valid       = true,
    .p1e8_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK3_cfg sclk3_default = {
    .p2e1             = 0xff,    // 0b11111111
    .p2e2             = 0xff,    // 0b11111111
    .p2e3             = 0xff,    // 0b11111111
    .p2e4             = 0xff,    // 0b11111111

    .p2e1_valid       = true,
    .p2e2_valid       = true,
    .p2e3_valid       = true,
    .p2e4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK4_cfg sclk4_default = {
    .p2e5             = 0xff,    // 0b11111111
    .p2e6             = 0xff,    // 0b11111111
    .p2e7             = 0xff,    // 0b11111111
    .p2e8             = 0xff,    // 0b11111111

    .p2e5_valid       = true,
    .p2e6_valid       = true,
    .p2e7_valid       = true,
    .p2e8_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK5_cfg sclk5_default = {
    .rse1             = 0xff,    // 0b11111111
    .rse2             = 0xff,    // 0b11111111
    .rse3             = 0xff,    // 0b11111111
    .rse4             = 0xff,    // 0b11111111

    .rse1_valid       = true,
    .rse2_valid       = true,
    .rse3_valid       = true,
    .rse4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_SCLK6_cfg sclk6_default = {
    .cpe1             = 0xff,    // 0b11111111
    .cpe2             = 0xff,    // 0b11111111
    .cpe3             = 0xff,    // 0b11111111
    .cpe4             = 0xff,    // 0b11111111

    .cpe1_valid       = true,
    .cpe2_valid       = true,
    .cpe3_valid       = true,
    .cpe4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_ACLK1_cfg aclk1_default = {
    .mce1             = 0xff,    // 0b11111111
    .mce2             = 0xff,    // 0b11111111
    .mce3             = 0xff,    // 0b11111111
    .mce4             = 0xff,    // 0b11111111

    .mce1_valid       = true,
    .mce2_valid       = true,
    .mce3_valid       = true,
    .mce4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_ACLK2_cfg aclk2_default = {
    .mce5             = 0xff,    // 0b11111111
    .mce6             = 0xff,    // 0b11111111
    .mce7             = 0xff,    // 0b11111111
    .mce8             = 0xff,    // 0b11111111

    .mce5_valid       = true,
    .mce6_valid       = true,
    .mce7_valid       = true,
    .mce8_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_ACLK3_cfg aclk3_default = {
    .mce9             = 0xff,    // 0b11111111
    .mce10            = 0xff,    // 0b11111111
    .mce11            = 0xff,    // 0b11111111
    .mce12            = 0xff,    // 0b11111111

    .mce9_valid       = true,
    .mce10_valid      = true,
    .mce11_valid      = true,
    .mce12_valid      = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_ACLK4_cfg aclk4_default = {
    .vse1             = 0xff,    // 0b11111111
    .vse2             = 0xff,    // 0b11111111
    .vse3             = 0xff,    // 0b11111111
    .vse4             = 0xff,    // 0b11111111

    .vse1_valid       = true,
    .vse2_valid       = true,
    .vse3_valid       = true,
    .vse4_valid       = true,
};

// structure based on default hex 0xffffffff
static struct scanblk_ACLK5_cfg aclk5_default = {
    .cd1e1            = 0xff,    // 0b11111111
    .cd1e2            = 0xff,    // 0b11111111
    .cd1e3            = 0xff,    // 0b11111111
    .cd1e4            = 0xff,    // 0b11111111

    .cd1e1_valid      = true,
    .cd1e2_valid      = true,
    .cd1e3_valid      = true,
    .cd1e4_valid      = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA10_cfg ldata10_default = {
    .ds1              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc1              = 0x0,     // 0b000000
    .ds2              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc2              = 0x0,     // 0b000000

    .ds1_valid        = true,
    .dc1_valid        = true,
    .ds2_valid        = true,
    .dc2_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA11_cfg ldata11_default = {
    .ds1              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc1              = 0x0,     // 0b000000
    .ds2              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc2              = 0x0,     // 0b000000

    .ds1_valid        = true,
    .dc1_valid        = true,
    .ds2_valid        = true,
    .dc2_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA20_cfg ldata20_default = {
    .ds3              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc3              = 0x0,     // 0b000000
    .ds4              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc4              = 0x0,     // 0b000000

    .ds3_valid        = true,
    .dc3_valid        = true,
    .ds4_valid        = true,
    .dc4_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA21_cfg ldata21_default = {
    .ds3              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc3              = 0x0,     // 0b000000
    .ds4              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc4              = 0x0,     // 0b000000

    .ds3_valid        = true,
    .dc3_valid        = true,
    .ds4_valid        = true,
    .dc4_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA30_cfg ldata30_default = {
    .ds5              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc5              = 0x0,     // 0b000000
    .ds6              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc6              = 0x0,     // 0b000000

    .ds5_valid        = true,
    .dc5_valid        = true,
    .ds6_valid        = true,
    .dc6_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA31_cfg ldata31_default = {
    .ds5              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc5              = 0x0,     // 0b000000
    .ds6              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc6              = 0x0,     // 0b000000

    .ds5_valid        = true,
    .dc5_valid        = true,
    .ds6_valid        = true,
    .dc6_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA40_cfg ldata40_default = {
    .ds7              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc7              = 0x0,     // 0b000000
    .ds8              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc8              = 0x0,     // 0b000000

    .ds7_valid        = true,
    .dc7_valid        = true,
    .ds8_valid        = true,
    .dc8_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA41_cfg ldata41_default = {
    .ds7              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc7              = 0x0,     // 0b000000
    .ds8              = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc8              = 0x0,     // 0b000000

    .ds7_valid        = true,
    .dc7_valid        = true,
    .ds8_valid        = true,
    .dc8_valid        = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA50_cfg ldata50_default = {
    .ds9              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc9              = 0x0,     // 0b000000
    .ds10             = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc10             = 0x0,     // 0b000000

    .ds9_valid        = true,
    .dc9_valid        = true,
    .ds10_valid       = true,
    .dc10_valid       = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA51_cfg ldata51_default = {
    .ds9              = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc9              = 0x0,     // 0b000000
    .ds10             = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc10             = 0x0,     // 0b000000

    .ds9_valid        = true,
    .dc9_valid        = true,
    .ds10_valid       = true,
    .dc10_valid       = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA60_cfg ldata60_default = {
    .ds11             = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc11             = 0x0,     // 0b000000
    .ds12             = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc12             = 0x0,     // 0b000000

    .ds11_valid       = true,
    .dc11_valid       = true,
    .ds12_valid       = true,
    .dc12_valid       = true,
};

// structure based on default hex 0xff00ff00
static struct scanblk_LDATA61_cfg ldata61_default = {
    .ds11             = 0xff,    // 0b11111111
    //.reserved1      = 0x0,     // 0b00
    .dc11             = 0x0,     // 0b000000
    .ds12             = 0xff,    // 0b11111111
    //.reserved2      = 0x0,     // 0b00
    .dc12             = 0x0,     // 0b000000

    .ds11_valid       = true,
    .dc11_valid       = true,
    .ds12_valid       = true,
    .dc12_valid       = true,
};

// structure based on default hex 0x00000000
static struct scanblk_PSEQ1_cfg pseq1_default = {
    //.reserved1      = 0x0,     // 0b00
    .pseq1            = 0x0,     // 0b000000
    //.reserved2      = 0x0,     // 0b00
    .pseq2            = 0x0,     // 0b000000
    //.reserved3      = 0x0,     // 0b00
    .pseq3            = 0x0,     // 0b000000
    //.reserved4      = 0x0,     // 0b00
    .pseq4            = 0x0,     // 0b000000

    .pseq1_valid      = true,
    .pseq2_valid      = true,
    .pseq3_valid      = true,
    .pseq4_valid      = true,
};

// structure based on default hex 0x00000000
static struct scanblk_PSEQ2_cfg pseq2_default = {
    //.reserved1      = 0x0,     // 0b000000000000000000
    .pseq5            = 0x0,     // 0b000000
    //.reserved2      = 0x0,     // 0b00
    .pseq6            = 0x0,     // 0b000000

    .pseq5_valid      = true,
    .pseq6_valid      = true,
};

// structure based on default hex 0x0fff0000
static struct scanblk_BPWM_cfg bpwm_default = {
    //.reserved1      = 0x0,     // 0b0000
    .bpwmper          = 0xfff,   // 0b111111111111
    //.reserved2      = 0x0,     // 0b0000
    .bpwmdty          = 0x0,     // 0b000000000000

    .bpwmper_valid    = true,
    .bpwmdty_valid    = true,
};

// structure based on default hex 0x00000001
static struct scanblk_CFGARB_cfg cfgarb_default = {
    .channel          = 0x0,     // 0b000000
    //.reserved1      = 0x0,     // 0b000000
    .extenable        = 0x0,     // 0b0000
    //.reserved2      = 0x0,     // 0b00000000000000
    .parenable        = 0x1,     // 0b01

    .channel_valid    = true,
    .extenable_valid  = true,
    .parenable_valid  = true,
};

uint32_t get_clk_speed(scanblkDeviceHandle *device_data)
{
    return device_data->clock_speed;
}

void legacy_scif_led_dump(scanblkDeviceHandle *device_data)
{
    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LED00", scanblkRead(LED00),
            "LED10", scanblkRead(LED10),
            "LED20", scanblkRead(LED20)
        );
    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LPWMCfg0", scanblkRead(LPWMConfig0),
            "LPWM00",   scanblkRead(LPWM00),
            "LPWM10",   scanblkRead(LPWM10),
            "LPWM20",   scanblkRead(LPWM20)
        );
    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LED01", scanblkRead(LED01),
            "LED11", scanblkRead(LED11),
            "LED21", scanblkRead(LED21)
        );
    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LPWMCfg1", scanblkRead(LPWMConfig1),
            "LPWM01",   scanblkRead(LPWM01),
            "LPWM11",   scanblkRead(LPWM11),
            "LPWM21",   scanblkRead(LPWM21)
        );
}

/*
 * ballen -- keep this in it's exact format so we can use
 * magic python scripts to generate graphs
 */
void legacy_scif_dump(scanblkDeviceHandle *device_data)
{
    uint32_t scan_clock_mhz;

    scan_clock_mhz = (get_clk_speed(device_data)/1000000);
    debug_print( "scif_dump scan_clock_mhz=%d\n", scan_clock_mhz );

    debug_print("SCFG1=0x%08x ",  scanblkRead(SCFG1));
    debug_print("SCFG2=0x%08x ",  scanblkRead(SCFG2));
    debug_print("SCFG3=0x%08x ",  scanblkRead(SCFG3));
    debug_print("SCTRL=0x%08x",  scanblkRead(SCTRL));
    debug_print("\n");
    debug_print("SCMD=0x%08x ",   scanblkRead(SCMD));
    debug_print("SSTAT=0x%08x ",  scanblkRead(SSTAT));
    debug_print("SIACK=0x%08x ", 0);  // WRITE ONLY REG!
    debug_print("SCANX=0x%08x",  scanblkRead(SCANX));
    debug_print("\n");
    debug_print("SCYCLE=0x%08x ", scanblkRead(SCYCLE));
    debug_print("STCFG1=0x%08x ", scanblkRead(STCFG1));
    debug_print("STCFG2=0x%08x ", scanblkRead(STCFG2));
    debug_print("STCFG3=0x%08x", scanblkRead(STCFG3));
    debug_print("\n");
    debug_print("STCFG4=0x%08x ", scanblkRead(STCFG4));
    debug_print("CCFG=0x%08x ",   scanblkRead(CCFG));
    debug_print("SCLK1=0x%08x ",  scanblkRead(SCLK1));
    debug_print("SCLK2=0x%08x",  scanblkRead(SCLK2));
    debug_print("\n");
    debug_print("SCLK3=0x%08x ",  scanblkRead(SCLK3));
    debug_print("SCLK4=0x%08x ",  scanblkRead(SCLK4));
    debug_print("SCLK5=0x%08x ",  scanblkRead(SCLK5));
    debug_print("SCLK6=0x%08x",  scanblkRead(SCLK6));
    debug_print("\n");
    debug_print("ACLK1=0x%08x ",  scanblkRead(ACLK1));
    debug_print("ACLK2=0x%08x ",  scanblkRead(ACLK2));
    debug_print("ACLK3=0x%08x ",  scanblkRead(ACLK3));
    debug_print("ACLK4=0x%08x",  scanblkRead(ACLK4));
    debug_print("\n");
    debug_print("ACLK5=0x%08x ",  scanblkRead(ACLK5));
    debug_print("ACLK6=0x%08x ",  scanblkRead(ACLK6));
    debug_print("BPWM=0x%08x ",   scanblkRead(BPWM));
    debug_print("CFGARB=0x%08x", scanblkRead(CFGARB));
    debug_print("\n");
    debug_print("MCFG=0x%08x",   scanblkRead(MCFG));
    debug_print("\n");
    debug_print("AFEPC=0x%08x",  scanblkRead(AFEPC));
    debug_print("\n");

    /* davep 16-Jul-2012 ; as of this writing, our current platforms have three
     * separate LDATA settings
     */

    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LDATA10", scanblkRead(LDATA10),
            "LDATA20", scanblkRead(LDATA20),
            "LDATA30", scanblkRead(LDATA30),
            "LDATA40", scanblkRead(LDATA40),
            "LDATA50", scanblkRead(LDATA50),
            "LDATA60", scanblkRead(LDATA60) );

    debug_print( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LDATA11", scanblkRead(LDATA11),
            "LDATA21", scanblkRead(LDATA21),
            "LDATA31", scanblkRead(LDATA31),
            "LDATA41", scanblkRead(LDATA41),
            "LDATA51", scanblkRead(LDATA51),
            "LDATA61", scanblkRead(LDATA61) );

    debug_print( "%8s=0x%08x %8s=0x%08x\n",
            "PSEQ1", scanblkRead(PSEQ1),
            "PSEQ2", scanblkRead(PSEQ2) );

    /* davep 16-Jul-2012 ; LEDs continue to diverge so pushed into separate
     * function
     */
    legacy_scif_led_dump(device_data);
}

void dump_regs(scanblkDeviceHandle *device_data)
{
    debug_print("SCANBLK reg dump\n");

    debug_print("SCFG1=0x%X\n",        scanblkRead(SCFG1));
    debug_print("SCFG2=0x%X\n",        scanblkRead(SCFG2));
    debug_print("SCFG3=0x%X\n",        scanblkRead(SCFG3));
    debug_print("SCTRL=0x%X\n",        scanblkRead(SCTRL));
    debug_print("SCMD=0x%X\n",         scanblkRead(SCMD));
    debug_print("SSTAT=0x%X\n",        scanblkRead(SSTAT));
    debug_print("SIACK=0x%X\n",        scanblkRead(SIACK));
    debug_print("SCANX=0x%X\n",        scanblkRead(SCANX));
    debug_print("SCYCLE=0x%X\n",       scanblkRead(SCYCLE));
    debug_print("STCFG1=0x%X\n",       scanblkRead(STCFG1));
    debug_print("STCFG2=0x%X\n",       scanblkRead(STCFG2));
    debug_print("STCFG3=0x%X\n",       scanblkRead(STCFG3));
    debug_print("STCFG4=0x%X\n",       scanblkRead(STCFG4));
    debug_print("CCFG=0x%X\n",         scanblkRead(CCFG));
    debug_print("SCLK1=0x%X\n",        scanblkRead(SCLK1));
    debug_print("SCLK2=0x%X\n",        scanblkRead(SCLK2));
    debug_print("SCLK3=0x%X\n",        scanblkRead(SCLK3));
    debug_print("SCLK4=0x%X\n",        scanblkRead(SCLK4));
    debug_print("SCLK5=0x%X\n",        scanblkRead(SCLK5));
    debug_print("SCLK6=0x%X\n",        scanblkRead(SCLK6));
    debug_print("ACLK1=0x%X\n",        scanblkRead(ACLK1));
    debug_print("ACLK2=0x%X\n",        scanblkRead(ACLK2));
    debug_print("ACLK3=0x%X\n",        scanblkRead(ACLK3));
    debug_print("ACLK4=0x%X\n",        scanblkRead(ACLK4));
    debug_print("ACLK5=0x%X\n",        scanblkRead(ACLK5));
    debug_print("ACLK6=0x%X\n",        scanblkRead(ACLK6));
    debug_print("LDATA10=0x%X\n",      scanblkRead(LDATA10));
    debug_print("LDATA11=0x%X\n",      scanblkRead(LDATA11));
    debug_print("LDATA20=0x%X\n",      scanblkRead(LDATA20));
    debug_print("LDATA21=0x%X\n",      scanblkRead(LDATA21));
    debug_print("LDATA30=0x%X\n",      scanblkRead(LDATA30));
    debug_print("LDATA31=0x%X\n",      scanblkRead(LDATA31));
    debug_print("LDATA40=0x%X\n",      scanblkRead(LDATA40));
    debug_print("LDATA41=0x%X\n",      scanblkRead(LDATA41));
    debug_print("LDATA50=0x%X\n",      scanblkRead(LDATA50));
    debug_print("LDATA51=0x%X\n",      scanblkRead(LDATA51));
    debug_print("LDATA60=0x%X\n",      scanblkRead(LDATA60));
    debug_print("LDATA61=0x%X\n",      scanblkRead(LDATA61));
    debug_print("PSEQ1=0x%X\n",        scanblkRead(PSEQ1));
    debug_print("PSEQ2=0x%X\n",        scanblkRead(PSEQ2));
    debug_print("BPWM=0x%X\n",         scanblkRead(BPWM));
    debug_print("MCFG=0x%X\n",         scanblkRead(MCFG));
    debug_print("MotorSyncMin=0x%X\n", scanblkRead(MotorSyncMin));
    debug_print("AFEPC=0x%X\n",        scanblkRead(AFEPC));
    debug_print("CFGARB=0x%X\n",       scanblkRead(CFGARB));
    debug_print("LED00=0x%X\n",        scanblkRead(LED00));
    debug_print("LED01=0x%X\n",        scanblkRead(LED01));
    debug_print("LED10=0x%X\n",        scanblkRead(LED10));
    debug_print("LED11=0x%X\n",        scanblkRead(LED11));
    debug_print("LED20=0x%X\n",        scanblkRead(LED20));
    debug_print("LED21=0x%X\n",        scanblkRead(LED21));
    debug_print("LPWMConfig0=0x%X\n",  scanblkRead(LPWMConfig0));
    debug_print("LPWMConfig1=0x%X\n",  scanblkRead(LPWMConfig1));
    debug_print("LPWM00=0x%X\n",       scanblkRead(LPWM00));
    debug_print("LPWM01=0x%X\n",       scanblkRead(LPWM01));
    debug_print("LPWM10=0x%X\n",       scanblkRead(LPWM10));
    debug_print("LPWM11=0x%X\n",       scanblkRead(LPWM11));
    debug_print("LPWM20=0x%X\n",       scanblkRead(LPWM20));
    debug_print("LPWM21=0x%X\n",       scanblkRead(LPWM21));
    debug_print("EXCFG=0x%X\n",        scanblkRead(EXCFG));
    debug_print("OFIE=0x%X\n",         scanblkRead(OFIE));
    debug_print("OFIP=0x%X\n",         scanblkRead(OFIP));
    debug_print("OFIA=0x%X\n",         scanblkRead(OFIA));
    debug_print("OFIF=0x%X\n",         scanblkRead(OFIF));
    debug_print("REV0=0x%X\n",         scanblkRead(REV0));
    debug_print("REV1=0x%X\n",         scanblkRead(REV1));

    debug_print("-------\n");
    legacy_scif_dump(device_data);
}

static void get_scanblk_SCFG1(scanblkDeviceHandle *device_data, struct scanblk_SCFG1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCFG1);
    cfg->p1clken = SCAN_SCFG1_P1CLKEN_MASK_SHIFT(reg);
    cfg->p2clken = SCAN_SCFG1_P2CLKEN_MASK_SHIFT(reg);
    cfg->rsclken = SCAN_SCFG1_RSCLKEN_MASK_SHIFT(reg);
    cfg->cpclken = SCAN_SCFG1_CPCLKEN_MASK_SHIFT(reg);
    cfg->mclken = SCAN_SCFG1_MCLKEN_MASK_SHIFT(reg);
    cfg->vsclken = SCAN_SCFG1_VSCLKEN_MASK_SHIFT(reg);
    cfg->clclken = SCAN_SCFG1_CLCLKEN_MASK_SHIFT(reg);
    cfg->lclmpen = SCAN_SCFG1_LCLMPEN_MASK_SHIFT(reg);
    cfg->bulben = SCAN_SCFG1_BULBEN_MASK_SHIFT(reg);
    cfg->ssmen = SCAN_SCFG1_SSMEN_MASK_SHIFT(reg);
    cfg->smpol = SCAN_SCFG1_SMPOL_MASK_SHIFT(reg);
    cfg->p1apol = SCAN_SCFG1_P1APOL_MASK_SHIFT(reg);
    cfg->sptog = SCAN_SCFG1_SPTOG_MASK_SHIFT(reg);
    cfg->sppol = SCAN_SCFG1_SPPOL_MASK_SHIFT(reg);
    cfg->scpol = SCAN_SCFG1_SCPOL_MASK_SHIFT(reg);
    cfg->acpol = SCAN_SCFG1_ACPOL_MASK_SHIFT(reg);
}

static void set_scanblk_SCFG1(scanblkDeviceHandle *device_data, struct scanblk_SCFG1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCFG1);
    if(cfg->p1clken_valid)
        reg = SCAN_SCFG1_P1CLKEN_REPLACE_VAL(reg, cfg->p1clken);
    if(cfg->p2clken_valid)
        reg = SCAN_SCFG1_P2CLKEN_REPLACE_VAL(reg, cfg->p2clken);
    if(cfg->rsclken_valid)
        reg = SCAN_SCFG1_RSCLKEN_REPLACE_VAL(reg, cfg->rsclken);
    if(cfg->cpclken_valid)
        reg = SCAN_SCFG1_CPCLKEN_REPLACE_VAL(reg, cfg->cpclken);
    if(cfg->mclken_valid)
        reg = SCAN_SCFG1_MCLKEN_REPLACE_VAL(reg, cfg->mclken);
    if(cfg->vsclken_valid)
        reg = SCAN_SCFG1_VSCLKEN_REPLACE_VAL(reg, cfg->vsclken);
    if(cfg->clclken_valid)
        reg = SCAN_SCFG1_CLCLKEN_REPLACE_VAL(reg, cfg->clclken);
    if(cfg->lclmpen_valid)
        reg = SCAN_SCFG1_LCLMPEN_REPLACE_VAL(reg, cfg->lclmpen);
    if(cfg->bulben_valid)
        reg = SCAN_SCFG1_BULBEN_REPLACE_VAL(reg, cfg->bulben);
    if(cfg->ssmen_valid)
        reg = SCAN_SCFG1_SSMEN_REPLACE_VAL(reg, cfg->ssmen);
    if(cfg->smpol_valid)
        reg = SCAN_SCFG1_SMPOL_REPLACE_VAL(reg, cfg->smpol);
    if(cfg->p1apol_valid)
        reg = SCAN_SCFG1_P1APOL_REPLACE_VAL(reg, cfg->p1apol);
    if(cfg->sptog_valid)
        reg = SCAN_SCFG1_SPTOG_REPLACE_VAL(reg, cfg->sptog);
    if(cfg->sppol_valid)
        reg = SCAN_SCFG1_SPPOL_REPLACE_VAL(reg, cfg->sppol);
    if(cfg->scpol_valid)
        reg = SCAN_SCFG1_SCPOL_REPLACE_VAL(reg, cfg->scpol);
    if(cfg->acpol_valid)
        reg = SCAN_SCFG1_ACPOL_REPLACE_VAL(reg, cfg->acpol);
    scanblkWrite(SCFG1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCFG2(scanblkDeviceHandle *device_data, struct scanblk_SCFG2_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCFG2);
    cfg->cmode = SCAN_SCFG2_CMODE_MASK_SHIFT(reg);
    cfg->stype = SCAN_SCFG2_STYPE_MASK_SHIFT(reg);
    cfg->ssel = SCAN_SCFG2_SSEL_MASK_SHIFT(reg);
    cfg->afewidth = SCAN_SCFG2_AFEWIDTH_MASK_SHIFT(reg);
    cfg->leden1 = SCAN_SCFG2_LEDEN1_MASK_SHIFT(reg);
    cfg->leden0 = SCAN_SCFG2_LEDEN0_MASK_SHIFT(reg);
    cfg->pixper = SCAN_SCFG2_PIXPER_MASK_SHIFT(reg);
}

static void set_scanblk_SCFG2(scanblkDeviceHandle *device_data, struct scanblk_SCFG2_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCFG2);
    if(cfg->cmode_valid)
        reg = SCAN_SCFG2_CMODE_REPLACE_VAL(reg, cfg->cmode);
    if(cfg->stype_valid)
        reg = SCAN_SCFG2_STYPE_REPLACE_VAL(reg, cfg->stype);
    if(cfg->ssel_valid)
        reg = SCAN_SCFG2_SSEL_REPLACE_VAL(reg, cfg->ssel);
    if(cfg->afewidth_valid)
        reg = SCAN_SCFG2_AFEWIDTH_REPLACE_VAL(reg, cfg->afewidth);
    if(cfg->leden1_valid)
        reg = SCAN_SCFG2_LEDEN1_REPLACE_VAL(reg, cfg->leden1);
    if(cfg->leden0_valid)
        reg = SCAN_SCFG2_LEDEN0_REPLACE_VAL(reg, cfg->leden0);
    if(cfg->pixper_valid)
        reg = SCAN_SCFG2_PIXPER_REPLACE_VAL(reg, cfg->pixper);
    scanblkWrite(SCFG2, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCFG3(scanblkDeviceHandle *device_data, struct scanblk_SCFG3_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCFG3);
    cfg->p1rise = SCAN_SCFG3_P1RISE_MASK_SHIFT(reg);
    cfg->p1fall = SCAN_SCFG3_P1FALL_MASK_SHIFT(reg);
    cfg->p2rise = SCAN_SCFG3_P2RISE_MASK_SHIFT(reg);
    cfg->p2fall = SCAN_SCFG3_P2FALL_MASK_SHIFT(reg);
    cfg->rsrise = SCAN_SCFG3_RSRISE_MASK_SHIFT(reg);
    cfg->rsfall = SCAN_SCFG3_RSFALL_MASK_SHIFT(reg);
    cfg->cprise = SCAN_SCFG3_CPRISE_MASK_SHIFT(reg);
    cfg->cpfall = SCAN_SCFG3_CPFALL_MASK_SHIFT(reg);
    cfg->mcrise = SCAN_SCFG3_MCRISE_MASK_SHIFT(reg);
    cfg->mcfall = SCAN_SCFG3_MCFALL_MASK_SHIFT(reg);
    cfg->vsrise = SCAN_SCFG3_VSRISE_MASK_SHIFT(reg);
    cfg->vsfall = SCAN_SCFG3_VSFALL_MASK_SHIFT(reg);
    cfg->cd1rise = SCAN_SCFG3_CD1RISE_MASK_SHIFT(reg);
    cfg->cd1fall = SCAN_SCFG3_CD1FALL_MASK_SHIFT(reg);
    cfg->sprise = SCAN_SCFG3_SPRISE_MASK_SHIFT(reg);
    cfg->spfall = SCAN_SCFG3_SPFALL_MASK_SHIFT(reg);
}

static void set_scanblk_SCFG3(scanblkDeviceHandle *device_data, struct scanblk_SCFG3_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCFG3);
    if(cfg->p1rise_valid)
        reg = SCAN_SCFG3_P1RISE_REPLACE_VAL(reg, cfg->p1rise);
    if(cfg->p1fall_valid)
        reg = SCAN_SCFG3_P1FALL_REPLACE_VAL(reg, cfg->p1fall);
    if(cfg->p2rise_valid)
        reg = SCAN_SCFG3_P2RISE_REPLACE_VAL(reg, cfg->p2rise);
    if(cfg->p2fall_valid)
        reg = SCAN_SCFG3_P2FALL_REPLACE_VAL(reg, cfg->p2fall);
    if(cfg->rsrise_valid)
        reg = SCAN_SCFG3_RSRISE_REPLACE_VAL(reg, cfg->rsrise);
    if(cfg->rsfall_valid)
        reg = SCAN_SCFG3_RSFALL_REPLACE_VAL(reg, cfg->rsfall);
    if(cfg->cprise_valid)
        reg = SCAN_SCFG3_CPRISE_REPLACE_VAL(reg, cfg->cprise);
    if(cfg->cpfall_valid)
        reg = SCAN_SCFG3_CPFALL_REPLACE_VAL(reg, cfg->cpfall);
    if(cfg->mcrise_valid)
        reg = SCAN_SCFG3_MCRISE_REPLACE_VAL(reg, cfg->mcrise);
    if(cfg->mcfall_valid)
        reg = SCAN_SCFG3_MCFALL_REPLACE_VAL(reg, cfg->mcfall);
    if(cfg->vsrise_valid)
        reg = SCAN_SCFG3_VSRISE_REPLACE_VAL(reg, cfg->vsrise);
    if(cfg->vsfall_valid)
        reg = SCAN_SCFG3_VSFALL_REPLACE_VAL(reg, cfg->vsfall);
    if(cfg->cd1rise_valid)
        reg = SCAN_SCFG3_CD1RISE_REPLACE_VAL(reg, cfg->cd1rise);
    if(cfg->cd1fall_valid)
        reg = SCAN_SCFG3_CD1FALL_REPLACE_VAL(reg, cfg->cd1fall);
    if(cfg->sprise_valid)
        reg = SCAN_SCFG3_SPRISE_REPLACE_VAL(reg, cfg->sprise);
    if(cfg->spfall_valid)
        reg = SCAN_SCFG3_SPFALL_REPLACE_VAL(reg, cfg->spfall);
    scanblkWrite(SCFG3, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCTRL(scanblkDeviceHandle *device_data, struct scanblk_SCTRL_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCTRL);
    cfg->umsen = SCAN_SCTRL_UMSEN_MASK_SHIFT(reg);
    cfg->cmdqen = SCAN_SCTRL_CMDQEN_MASK_SHIFT(reg);
    cfg->termcmd = SCAN_SCTRL_TERMCMD_MASK_SHIFT(reg);
    cfg->sreset = SCAN_SCTRL_SRESET_MASK_SHIFT(reg);
    cfg->scmdie = SCAN_SCTRL_SCMDIE_MASK_SHIFT(reg);
    cfg->run = SCAN_SCTRL_RUN_MASK_SHIFT(reg);
    cfg->scanen = SCAN_SCTRL_SCANEN_MASK_SHIFT(reg);
}

static void set_scanblk_SCTRL_worker(scanblkDeviceHandle *device_data, struct scanblk_SCTRL_cfg *cfg)
{
    uint32_t reg;

    // WARNING: this worker function has no register access protection, it must be
    //           protected by the caller!
    reg = scanblkRead(SCTRL);
    if(cfg->umsen_valid)
        reg = SCAN_SCTRL_UMSEN_REPLACE_VAL(reg, cfg->umsen);
    if(cfg->cmdqen_valid)
        reg = SCAN_SCTRL_CMDQEN_REPLACE_VAL(reg, cfg->cmdqen);
    if(cfg->termcmd_valid)
        reg = SCAN_SCTRL_TERMCMD_REPLACE_VAL(reg, cfg->termcmd);
    if(cfg->sreset_valid)
        reg = SCAN_SCTRL_SRESET_REPLACE_VAL(reg, cfg->sreset);
    if(cfg->scmdie_valid)
        reg = SCAN_SCTRL_SCMDIE_REPLACE_VAL(reg, cfg->scmdie);
    if(cfg->run_valid)
        reg = SCAN_SCTRL_RUN_REPLACE_VAL(reg, cfg->run);
    if(cfg->scanen_valid)
        reg = SCAN_SCTRL_SCANEN_REPLACE_VAL(reg, cfg->scanen);
    scanblkWrite(SCTRL, reg);
}

static void set_scanblk_SCTRL_thread(scanblkDeviceHandle *device_data, struct scanblk_SCTRL_cfg *cfg)
{
    // NOTE: For use from thread context only!
    PROTECT_REG_ACCESS;
    set_scanblk_SCTRL_worker(device_data, cfg);
    UNPROTECT_REG_ACCESS;
}

static void set_scanblk_SCTRL_isr(scanblkDeviceHandle *device_data, struct scanblk_SCTRL_cfg *cfg)
{
    unsigned long flags;

    // NOTE: For use from interrupt context only!
    PROTECT_INTREG_ACCESS_IRQ;
    set_scanblk_SCTRL_worker(device_data, cfg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static void get_scanblk_SCMD(scanblkDeviceHandle *device_data, struct scanblk_SCMD_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCMD);
    cfg->eoi = SCAN_SCMD_EOI_MASK_SHIFT(reg);
    cfg->sdata = SCAN_SCMD_SDATA_MASK_SHIFT(reg);
    cfg->sptogval = SCAN_SCMD_SPTOGVAL_MASK_SHIFT(reg);
    cfg->sclkper = SCAN_SCMD_SCLKPER_MASK_SHIFT(reg);
    cfg->scany = SCAN_SCMD_SCANY_MASK_SHIFT(reg);
}

static void set_scanblk_SCMD_worker(scanblkDeviceHandle *device_data, struct scanblk_SCMD_cfg *cfg)
{
    uint32_t reg;

    // WARNING: this worker function has no register access protection, it must be
    //           protected by the caller!
    reg = scanblkRead(SCMD);
    if(cfg->eoi_valid)
        reg = SCAN_SCMD_EOI_REPLACE_VAL(reg, cfg->eoi);
    if(cfg->sdata_valid)
        reg = SCAN_SCMD_SDATA_REPLACE_VAL(reg, cfg->sdata);
    if(cfg->sptogval_valid)
        reg = SCAN_SCMD_SPTOGVAL_REPLACE_VAL(reg, cfg->sptogval);
    if(cfg->sclkper_valid)
        reg = SCAN_SCMD_SCLKPER_REPLACE_VAL(reg, cfg->sclkper);
    if(cfg->scany_valid)
        reg = SCAN_SCMD_SCANY_REPLACE_VAL(reg, cfg->scany);
    scanblkWrite(SCMD, reg);
}

static void set_scanblk_SCMD_isr(scanblkDeviceHandle *device_data, struct scanblk_SCMD_cfg *cfg)
{
    unsigned long flags;

    // NOTE: For use from interrupt context only!
    PROTECT_INTREG_ACCESS_IRQ;
    set_scanblk_SCMD_worker(device_data, cfg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static void set_scanblk_SCMD_thread(scanblkDeviceHandle *device_data, struct scanblk_SCMD_cfg *cfg)
{
    // NOTE: For use from thread context only!
    PROTECT_REG_ACCESS;
    set_scanblk_SCMD_worker(device_data, cfg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SSTAT(scanblkDeviceHandle *device_data, struct scanblk_SSTAT_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SSTAT);
    
    BUG_ON(cfg == NULL);
    cfg->scanyr = SCAN_SSTAT_SCANYR_MASK_SHIFT(reg);
    cfg->extintstat = SCAN_SSTAT_EXTINTSTAT_MASK_SHIFT(reg);
    cfg->unexpsync = SCAN_SSTAT_UNEXPSYNC_MASK_SHIFT(reg);
    cfg->cmdstat = SCAN_SSTAT_CMDSTAT_MASK_SHIFT(reg);
    cfg->qempty = SCAN_SSTAT_QEMPTY_MASK_SHIFT(reg);
    cfg->qfull = SCAN_SSTAT_QFULL_MASK_SHIFT(reg);
    cfg->qovr = SCAN_SSTAT_QOVR_MASK_SHIFT(reg);
    cfg->iovr = SCAN_SSTAT_IOVR_MASK_SHIFT(reg);
    cfg->scmdcomp = SCAN_SSTAT_SCMDCOMP_MASK_SHIFT(reg);
}

static void set_scanblk_SIACK(scanblkDeviceHandle *device_data, struct scanblk_irqs *irqstruct)
{
    uint32_t reg = 0;

    reg = SCAN_SIACK_UMSACK_REPLACE_VAL(reg, irqstruct->unexp_mtr_sync);
    reg = SCAN_SIACK_QOVRACK_REPLACE_VAL(reg, irqstruct->queue_overrun);
    reg = SCAN_SIACK_CIOVRACK_REPLACE_VAL(reg, irqstruct->cmd_int_overrun);
    reg = SCAN_SIACK_SCMDACK_REPLACE_VAL(reg, irqstruct->scan_cmd_complete);

    scanblkWrite(SIACK, reg);
}

// clear all interrupts, return set ones in irqstruct
static void clear_scan_irqs(scanblkDeviceHandle *device_data, struct scanblk_irqs *irqstruct)
{
    unsigned long flags;
    struct scanblk_SSTAT_cfg cfg;
    struct scanblk_irqs irq;

    if (irqstruct == NULL)
        irqstruct = &irq;

    PROTECT_INTREG_ACCESS_IRQ;
    get_scanblk_SSTAT(device_data, &cfg);
    if (cfg.unexpsync)
    {
        irqstruct->unexp_mtr_sync = true;
    }
    if (cfg.qovr)
    {
        irqstruct->queue_overrun = true;
    }
    if (cfg.iovr)
    {
        irqstruct->cmd_int_overrun= true;
    }
    if (cfg.scmdcomp)
    {
        irqstruct->scan_cmd_complete = true;
    }
    set_scanblk_SIACK(device_data, irqstruct);  // ack the interrupts
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static void get_scanblk_SCANX(scanblkDeviceHandle *device_data, struct scanblk_SCANX_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCANX);
    cfg->expval = SCAN_SCANX_EXPVAL_MASK_SHIFT(reg);
    cfg->scanx = SCAN_SCANX_SCANX_MASK_SHIFT(reg);
}

static void set_scanblk_SCANX(scanblkDeviceHandle *device_data, struct scanblk_SCANX_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCANX);
    if(cfg->expval_valid)
        reg = SCAN_SCANX_EXPVAL_REPLACE_VAL(reg, cfg->expval);
    if(cfg->scanx_valid)
        reg = SCAN_SCANX_SCANX_REPLACE_VAL(reg, cfg->scanx);
    scanblkWrite(SCANX, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCYCLE(scanblkDeviceHandle *device_data, struct scanblk_SCYCLE_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCYCLE);
    cfg->scycle = SCAN_SCYCLE_SCYCLE_MASK_SHIFT(reg);
}

static void set_scanblk_SCYCLE(scanblkDeviceHandle *device_data, struct scanblk_SCYCLE_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCYCLE);
    if(cfg->scycle_valid)
        reg = SCAN_SCYCLE_SCYCLE_REPLACE_VAL(reg, cfg->scycle);
    scanblkWrite(SCYCLE, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_STCFG1(scanblkDeviceHandle *device_data, struct scanblk_STCFG1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(STCFG1);
    cfg->spgen = SCAN_STCFG1_SPGEN_MASK_SHIFT(reg);
    cfg->sp2ctrl = SCAN_STCFG1_SP2CTRL_MASK_SHIFT(reg);
    cfg->mskextd = SCAN_STCFG1_MSKEXTD_MASK_SHIFT(reg);
    cfg->ssme1 = SCAN_STCFG1_SSME1_MASK_SHIFT(reg);
    cfg->ssme2 = SCAN_STCFG1_SSME2_MASK_SHIFT(reg);
}

static void set_scanblk_STCFG1(scanblkDeviceHandle *device_data, struct scanblk_STCFG1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(STCFG1);
    if(cfg->spgen_valid)
        reg = SCAN_STCFG1_SPGEN_REPLACE_VAL(reg, cfg->spgen);
    if(cfg->sp2ctrl_valid)
        reg = SCAN_STCFG1_SP2CTRL_REPLACE_VAL(reg, cfg->sp2ctrl);
    if(cfg->mskextd_valid)
        reg = SCAN_STCFG1_MSKEXTD_REPLACE_VAL(reg, cfg->mskextd);
    if(cfg->ssme1_valid)
        reg = SCAN_STCFG1_SSME1_REPLACE_VAL(reg, cfg->ssme1);
    if(cfg->ssme2_valid)
        reg = SCAN_STCFG1_SSME2_REPLACE_VAL(reg, cfg->ssme2);
    scanblkWrite(STCFG1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_STCFG2(scanblkDeviceHandle *device_data, struct scanblk_STCFG2_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(STCFG2);
    cfg->spe1_2 = SCAN_STCFG2_SPE1_2_MASK_SHIFT(reg);
    cfg->spe1 = SCAN_STCFG2_SPE1_MASK_SHIFT(reg);
    cfg->spe2_2 = SCAN_STCFG2_SPE2_2_MASK_SHIFT(reg);
    cfg->spe2 = SCAN_STCFG2_SPE2_MASK_SHIFT(reg);
}

static void set_scanblk_STCFG2(scanblkDeviceHandle *device_data, struct scanblk_STCFG2_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(STCFG2);
    if(cfg->spe1_2_valid)
        reg = SCAN_STCFG2_SPE1_2_REPLACE_VAL(reg, cfg->spe1_2);
    if(cfg->spe1_valid)
        reg = SCAN_STCFG2_SPE1_REPLACE_VAL(reg, cfg->spe1);
    if(cfg->spe2_2_valid)
        reg = SCAN_STCFG2_SPE2_2_REPLACE_VAL(reg, cfg->spe2_2);
    if(cfg->spe2_valid)
        reg = SCAN_STCFG2_SPE2_REPLACE_VAL(reg, cfg->spe2);
    scanblkWrite(STCFG2, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_STCFG3(scanblkDeviceHandle *device_data, struct scanblk_STCFG3_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(STCFG3);
    cfg->aph1e1 = SCAN_STCFG3_APH1E1_MASK_SHIFT(reg);
    cfg->aph1e2 = SCAN_STCFG3_APH1E2_MASK_SHIFT(reg);
    cfg->aph1e3 = SCAN_STCFG3_APH1E3_MASK_SHIFT(reg);
    cfg->aph1e4 = SCAN_STCFG3_APH1E4_MASK_SHIFT(reg);
}

static void set_scanblk_STCFG3(scanblkDeviceHandle *device_data, struct scanblk_STCFG3_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(STCFG3);
    if(cfg->aph1e1_valid)
        reg = SCAN_STCFG3_APH1E1_REPLACE_VAL(reg, cfg->aph1e1);
    if(cfg->aph1e2_valid)
        reg = SCAN_STCFG3_APH1E2_REPLACE_VAL(reg, cfg->aph1e2);
    if(cfg->aph1e3_valid)
        reg = SCAN_STCFG3_APH1E3_REPLACE_VAL(reg, cfg->aph1e3);
    if(cfg->aph1e4_valid)
        reg = SCAN_STCFG3_APH1E4_REPLACE_VAL(reg, cfg->aph1e4);
    scanblkWrite(STCFG3, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_STCFG4(scanblkDeviceHandle *device_data, struct scanblk_STCFG4_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(STCFG4);
    cfg->aph1e5 = SCAN_STCFG4_APH1E5_MASK_SHIFT(reg);
    cfg->aph1e6 = SCAN_STCFG4_APH1E6_MASK_SHIFT(reg);
    cfg->aph1e7 = SCAN_STCFG4_APH1E7_MASK_SHIFT(reg);
    cfg->aph1e8 = SCAN_STCFG4_APH1E8_MASK_SHIFT(reg);
}

static void set_scanblk_STCFG4(scanblkDeviceHandle *device_data, struct scanblk_STCFG4_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(STCFG4);
    if(cfg->aph1e5_valid)
        reg = SCAN_STCFG4_APH1E5_REPLACE_VAL(reg, cfg->aph1e5);
    if(cfg->aph1e6_valid)
        reg = SCAN_STCFG4_APH1E6_REPLACE_VAL(reg, cfg->aph1e6);
    if(cfg->aph1e7_valid)
        reg = SCAN_STCFG4_APH1E7_REPLACE_VAL(reg, cfg->aph1e7);
    if(cfg->aph1e8_valid)
        reg = SCAN_STCFG4_APH1E8_REPLACE_VAL(reg, cfg->aph1e8);
    scanblkWrite(STCFG4, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_CCFG(scanblkDeviceHandle *device_data, struct scanblk_CCFG_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(CCFG);
    cfg->cpmode = SCAN_CCFG_CPMODE_MASK_SHIFT(reg);
    cfg->clmode = SCAN_CCFG_CLMODE_MASK_SHIFT(reg);
    cfg->clsel = SCAN_CCFG_CLSEL_MASK_SHIFT(reg);
    cfg->cle1 = SCAN_CCFG_CLE1_MASK_SHIFT(reg);
    cfg->ccle2 = SCAN_CCFG_CCLE2_MASK_SHIFT(reg);
    cfg->acle2 = SCAN_CCFG_ACLE2_MASK_SHIFT(reg);
}

static void set_scanblk_CCFG(scanblkDeviceHandle *device_data, struct scanblk_CCFG_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(CCFG);
    if(cfg->cpmode_valid)
        reg = SCAN_CCFG_CPMODE_REPLACE_VAL(reg, cfg->cpmode);
    if(cfg->clmode_valid)
        reg = SCAN_CCFG_CLMODE_REPLACE_VAL(reg, cfg->clmode);
    if(cfg->clsel_valid)
        reg = SCAN_CCFG_CLSEL_REPLACE_VAL(reg, cfg->clsel);
    if(cfg->cle1_valid)
        reg = SCAN_CCFG_CLE1_REPLACE_VAL(reg, cfg->cle1);
    if(cfg->ccle2_valid)
        reg = SCAN_CCFG_CCLE2_REPLACE_VAL(reg, cfg->ccle2);
    if(cfg->acle2_valid)
        reg = SCAN_CCFG_ACLE2_REPLACE_VAL(reg, cfg->acle2);
    scanblkWrite(CCFG, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK1(scanblkDeviceHandle *device_data, struct scanblk_SCLK1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK1);
    cfg->p1e1 = SCAN_SCLK1_P1E1_MASK_SHIFT(reg);
    cfg->p1e2 = SCAN_SCLK1_P1E2_MASK_SHIFT(reg);
    cfg->p1e3 = SCAN_SCLK1_P1E3_MASK_SHIFT(reg);
    cfg->p1e4 = SCAN_SCLK1_P1E4_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK1(scanblkDeviceHandle *device_data, struct scanblk_SCLK1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK1);
    if(cfg->p1e1_valid)
        reg = SCAN_SCLK1_P1E1_REPLACE_VAL(reg, cfg->p1e1);
    if(cfg->p1e2_valid)
        reg = SCAN_SCLK1_P1E2_REPLACE_VAL(reg, cfg->p1e2);
    if(cfg->p1e3_valid)
        reg = SCAN_SCLK1_P1E3_REPLACE_VAL(reg, cfg->p1e3);
    if(cfg->p1e4_valid)
        reg = SCAN_SCLK1_P1E4_REPLACE_VAL(reg, cfg->p1e4);
    scanblkWrite(SCLK1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK2(scanblkDeviceHandle *device_data, struct scanblk_SCLK2_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK2);
    cfg->p1e5 = SCAN_SCLK2_P1E5_MASK_SHIFT(reg);
    cfg->p1e6 = SCAN_SCLK2_P1E6_MASK_SHIFT(reg);
    cfg->p1e7 = SCAN_SCLK2_P1E7_MASK_SHIFT(reg);
    cfg->p1e8 = SCAN_SCLK2_P1E8_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK2(scanblkDeviceHandle *device_data, struct scanblk_SCLK2_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK2);
    if(cfg->p1e5_valid)
        reg = SCAN_SCLK2_P1E5_REPLACE_VAL(reg, cfg->p1e5);
    if(cfg->p1e6_valid)
        reg = SCAN_SCLK2_P1E6_REPLACE_VAL(reg, cfg->p1e6);
    if(cfg->p1e7_valid)
        reg = SCAN_SCLK2_P1E7_REPLACE_VAL(reg, cfg->p1e7);
    if(cfg->p1e8_valid)
        reg = SCAN_SCLK2_P1E8_REPLACE_VAL(reg, cfg->p1e8);
    scanblkWrite(SCLK2, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK3(scanblkDeviceHandle *device_data, struct scanblk_SCLK3_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK3);
    cfg->p2e1 = SCAN_SCLK3_P2E1_MASK_SHIFT(reg);
    cfg->p2e2 = SCAN_SCLK3_P2E2_MASK_SHIFT(reg);
    cfg->p2e3 = SCAN_SCLK3_P2E3_MASK_SHIFT(reg);
    cfg->p2e4 = SCAN_SCLK3_P2E4_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK3(scanblkDeviceHandle *device_data, struct scanblk_SCLK3_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK3);
    if(cfg->p2e1_valid)
        reg = SCAN_SCLK3_P2E1_REPLACE_VAL(reg, cfg->p2e1);
    if(cfg->p2e2_valid)
        reg = SCAN_SCLK3_P2E2_REPLACE_VAL(reg, cfg->p2e2);
    if(cfg->p2e3_valid)
        reg = SCAN_SCLK3_P2E3_REPLACE_VAL(reg, cfg->p2e3);
    if(cfg->p2e4_valid)
        reg = SCAN_SCLK3_P2E4_REPLACE_VAL(reg, cfg->p2e4);
    scanblkWrite(SCLK3, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK4(scanblkDeviceHandle *device_data, struct scanblk_SCLK4_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK4);
    cfg->p2e5 = SCAN_SCLK4_P2E5_MASK_SHIFT(reg);
    cfg->p2e6 = SCAN_SCLK4_P2E6_MASK_SHIFT(reg);
    cfg->p2e7 = SCAN_SCLK4_P2E7_MASK_SHIFT(reg);
    cfg->p2e8 = SCAN_SCLK4_P2E8_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK4(scanblkDeviceHandle *device_data, struct scanblk_SCLK4_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK4);
    if(cfg->p2e5_valid)
        reg = SCAN_SCLK4_P2E5_REPLACE_VAL(reg, cfg->p2e5);
    if(cfg->p2e6_valid)
        reg = SCAN_SCLK4_P2E6_REPLACE_VAL(reg, cfg->p2e6);
    if(cfg->p2e7_valid)
        reg = SCAN_SCLK4_P2E7_REPLACE_VAL(reg, cfg->p2e7);
    if(cfg->p2e8_valid)
        reg = SCAN_SCLK4_P2E8_REPLACE_VAL(reg, cfg->p2e8);
    scanblkWrite(SCLK4, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK5(scanblkDeviceHandle *device_data, struct scanblk_SCLK5_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK5);
    cfg->rse1 = SCAN_SCLK5_RSE1_MASK_SHIFT(reg);
    cfg->rse2 = SCAN_SCLK5_RSE2_MASK_SHIFT(reg);
    cfg->rse3 = SCAN_SCLK5_RSE3_MASK_SHIFT(reg);
    cfg->rse4 = SCAN_SCLK5_RSE4_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK5(scanblkDeviceHandle *device_data, struct scanblk_SCLK5_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK5);
    if(cfg->rse1_valid)
        reg = SCAN_SCLK5_RSE1_REPLACE_VAL(reg, cfg->rse1);
    if(cfg->rse2_valid)
        reg = SCAN_SCLK5_RSE2_REPLACE_VAL(reg, cfg->rse2);
    if(cfg->rse3_valid)
        reg = SCAN_SCLK5_RSE3_REPLACE_VAL(reg, cfg->rse3);
    if(cfg->rse4_valid)
        reg = SCAN_SCLK5_RSE4_REPLACE_VAL(reg, cfg->rse4);
    scanblkWrite(SCLK5, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_SCLK6(scanblkDeviceHandle *device_data, struct scanblk_SCLK6_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(SCLK6);
    cfg->cpe1 = SCAN_SCLK6_CPE1_MASK_SHIFT(reg);
    cfg->cpe2 = SCAN_SCLK6_CPE2_MASK_SHIFT(reg);
    cfg->cpe3 = SCAN_SCLK6_CPE3_MASK_SHIFT(reg);
    cfg->cpe4 = SCAN_SCLK6_CPE4_MASK_SHIFT(reg);
}

static void set_scanblk_SCLK6(scanblkDeviceHandle *device_data, struct scanblk_SCLK6_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(SCLK6);
    if(cfg->cpe1_valid)
        reg = SCAN_SCLK6_CPE1_REPLACE_VAL(reg, cfg->cpe1);
    if(cfg->cpe2_valid)
        reg = SCAN_SCLK6_CPE2_REPLACE_VAL(reg, cfg->cpe2);
    if(cfg->cpe3_valid)
        reg = SCAN_SCLK6_CPE3_REPLACE_VAL(reg, cfg->cpe3);
    if(cfg->cpe4_valid)
        reg = SCAN_SCLK6_CPE4_REPLACE_VAL(reg, cfg->cpe4);
    scanblkWrite(SCLK6, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK1(scanblkDeviceHandle *device_data, struct scanblk_ACLK1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK1);
    cfg->mce1 = SCAN_ACLK1_MCE1_MASK_SHIFT(reg);
    cfg->mce2 = SCAN_ACLK1_MCE2_MASK_SHIFT(reg);
    cfg->mce3 = SCAN_ACLK1_MCE3_MASK_SHIFT(reg);
    cfg->mce4 = SCAN_ACLK1_MCE4_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK1(scanblkDeviceHandle *device_data, struct scanblk_ACLK1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK1);
    if(cfg->mce1_valid)
        reg = SCAN_ACLK1_MCE1_REPLACE_VAL(reg, cfg->mce1);
    if(cfg->mce2_valid)
        reg = SCAN_ACLK1_MCE2_REPLACE_VAL(reg, cfg->mce2);
    if(cfg->mce3_valid)
        reg = SCAN_ACLK1_MCE3_REPLACE_VAL(reg, cfg->mce3);
    if(cfg->mce4_valid)
        reg = SCAN_ACLK1_MCE4_REPLACE_VAL(reg, cfg->mce4);
    scanblkWrite(ACLK1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK2(scanblkDeviceHandle *device_data, struct scanblk_ACLK2_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK2);
    cfg->mce5 = SCAN_ACLK2_MCE5_MASK_SHIFT(reg);
    cfg->mce6 = SCAN_ACLK2_MCE6_MASK_SHIFT(reg);
    cfg->mce7 = SCAN_ACLK2_MCE7_MASK_SHIFT(reg);
    cfg->mce8 = SCAN_ACLK2_MCE8_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK2(scanblkDeviceHandle *device_data, struct scanblk_ACLK2_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK2);
    if(cfg->mce5_valid)
        reg = SCAN_ACLK2_MCE5_REPLACE_VAL(reg, cfg->mce5);
    if(cfg->mce6_valid)
        reg = SCAN_ACLK2_MCE6_REPLACE_VAL(reg, cfg->mce6);
    if(cfg->mce7_valid)
        reg = SCAN_ACLK2_MCE7_REPLACE_VAL(reg, cfg->mce7);
    if(cfg->mce8_valid)
        reg = SCAN_ACLK2_MCE8_REPLACE_VAL(reg, cfg->mce8);
    scanblkWrite(ACLK2, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK3(scanblkDeviceHandle *device_data, struct scanblk_ACLK3_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK3);
    cfg->mce9 = SCAN_ACLK3_MCE9_MASK_SHIFT(reg);
    cfg->mce10 = SCAN_ACLK3_MCE10_MASK_SHIFT(reg);
    cfg->mce11 = SCAN_ACLK3_MCE11_MASK_SHIFT(reg);
    cfg->mce12 = SCAN_ACLK3_MCE12_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK3(scanblkDeviceHandle *device_data, struct scanblk_ACLK3_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK3);
    if(cfg->mce9_valid)
        reg = SCAN_ACLK3_MCE9_REPLACE_VAL(reg, cfg->mce9);
    if(cfg->mce10_valid)
        reg = SCAN_ACLK3_MCE10_REPLACE_VAL(reg, cfg->mce10);
    if(cfg->mce11_valid)
        reg = SCAN_ACLK3_MCE11_REPLACE_VAL(reg, cfg->mce11);
    if(cfg->mce12_valid)
        reg = SCAN_ACLK3_MCE12_REPLACE_VAL(reg, cfg->mce12);
    scanblkWrite(ACLK3, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK6(scanblkDeviceHandle *device_data, struct scanblk_ACLK6_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK6);
    cfg->mce13 = SCAN_ACLK6_MCE13_MASK_SHIFT(reg);
    cfg->mce14 = SCAN_ACLK6_MCE14_MASK_SHIFT(reg);
    cfg->mce15 = SCAN_ACLK6_MCE15_MASK_SHIFT(reg);
    cfg->mce16 = SCAN_ACLK6_MCE16_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK6(scanblkDeviceHandle *device_data, struct scanblk_ACLK6_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK6);
    if(cfg->mce13_valid)
        reg = SCAN_ACLK6_MCE13_REPLACE_VAL(reg, cfg->mce13);
    if(cfg->mce14_valid)
        reg = SCAN_ACLK6_MCE14_REPLACE_VAL(reg, cfg->mce14);
    if(cfg->mce15_valid)
        reg = SCAN_ACLK6_MCE15_REPLACE_VAL(reg, cfg->mce15);
    if(cfg->mce16_valid)
        reg = SCAN_ACLK6_MCE16_REPLACE_VAL(reg, cfg->mce16);
    scanblkWrite(ACLK6, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK4(scanblkDeviceHandle *device_data, struct scanblk_ACLK4_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK4);
    cfg->vse1 = SCAN_ACLK4_VSE1_MASK_SHIFT(reg);
    cfg->vse2 = SCAN_ACLK4_VSE2_MASK_SHIFT(reg);
    cfg->vse3 = SCAN_ACLK4_VSE3_MASK_SHIFT(reg);
    cfg->vse4 = SCAN_ACLK4_VSE4_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK4(scanblkDeviceHandle *device_data, struct scanblk_ACLK4_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK4);
    if(cfg->vse1_valid)
        reg = SCAN_ACLK4_VSE1_REPLACE_VAL(reg, cfg->vse1);
    if(cfg->vse2_valid)
        reg = SCAN_ACLK4_VSE2_REPLACE_VAL(reg, cfg->vse2);
    if(cfg->vse3_valid)
        reg = SCAN_ACLK4_VSE3_REPLACE_VAL(reg, cfg->vse3);
    if(cfg->vse4_valid)
        reg = SCAN_ACLK4_VSE4_REPLACE_VAL(reg, cfg->vse4);
    scanblkWrite(ACLK4, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_ACLK5(scanblkDeviceHandle *device_data, struct scanblk_ACLK5_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(ACLK5);
    cfg->cd1e1 = SCAN_ACLK5_CD1E1_MASK_SHIFT(reg);
    cfg->cd1e2 = SCAN_ACLK5_CD1E2_MASK_SHIFT(reg);
    cfg->cd1e3 = SCAN_ACLK5_CD1E3_MASK_SHIFT(reg);
    cfg->cd1e4 = SCAN_ACLK5_CD1E4_MASK_SHIFT(reg);
}

static void set_scanblk_ACLK5(scanblkDeviceHandle *device_data, struct scanblk_ACLK5_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(ACLK5);
    if(cfg->cd1e1_valid)
        reg = SCAN_ACLK5_CD1E1_REPLACE_VAL(reg, cfg->cd1e1);
    if(cfg->cd1e2_valid)
        reg = SCAN_ACLK5_CD1E2_REPLACE_VAL(reg, cfg->cd1e2);
    if(cfg->cd1e3_valid)
        reg = SCAN_ACLK5_CD1E3_REPLACE_VAL(reg, cfg->cd1e3);
    if(cfg->cd1e4_valid)
        reg = SCAN_ACLK5_CD1E4_REPLACE_VAL(reg, cfg->cd1e4);
    scanblkWrite(ACLK5, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA10(scanblkDeviceHandle *device_data, struct scanblk_LDATA10_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA10);
    cfg->ds1 = SCAN_LDATA10_DS1_MASK_SHIFT(reg);
    cfg->dc1 = SCAN_LDATA10_DC1_MASK_SHIFT(reg);
    cfg->ds2 = SCAN_LDATA10_DS2_MASK_SHIFT(reg);
    cfg->dc2 = SCAN_LDATA10_DC2_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA10(scanblkDeviceHandle *device_data, struct scanblk_LDATA10_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA10);
    if(cfg->ds1_valid)
        reg = SCAN_LDATA10_DS1_REPLACE_VAL(reg, cfg->ds1);
    if(cfg->dc1_valid)
        reg = SCAN_LDATA10_DC1_REPLACE_VAL(reg, cfg->dc1);
    if(cfg->ds2_valid)
        reg = SCAN_LDATA10_DS2_REPLACE_VAL(reg, cfg->ds2);
    if(cfg->dc2_valid)
        reg = SCAN_LDATA10_DC2_REPLACE_VAL(reg, cfg->dc2);
    scanblkWrite(LDATA10, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA11(scanblkDeviceHandle *device_data, struct scanblk_LDATA11_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA11);
    cfg->ds1 = SCAN_LDATA11_DS1_MASK_SHIFT(reg);
    cfg->dc1 = SCAN_LDATA11_DC1_MASK_SHIFT(reg);
    cfg->ds2 = SCAN_LDATA11_DS2_MASK_SHIFT(reg);
    cfg->dc2 = SCAN_LDATA11_DC2_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA11(scanblkDeviceHandle *device_data, struct scanblk_LDATA11_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA11);
    if(cfg->ds1_valid)
        reg = SCAN_LDATA11_DS1_REPLACE_VAL(reg, cfg->ds1);
    if(cfg->dc1_valid)
        reg = SCAN_LDATA11_DC1_REPLACE_VAL(reg, cfg->dc1);
    if(cfg->ds2_valid)
        reg = SCAN_LDATA11_DS2_REPLACE_VAL(reg, cfg->ds2);
    if(cfg->dc2_valid)
        reg = SCAN_LDATA11_DC2_REPLACE_VAL(reg, cfg->dc2);
    scanblkWrite(LDATA11, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA20(scanblkDeviceHandle *device_data, struct scanblk_LDATA20_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA20);
    cfg->ds3 = SCAN_LDATA20_DS3_MASK_SHIFT(reg);
    cfg->dc3 = SCAN_LDATA20_DC3_MASK_SHIFT(reg);
    cfg->ds4 = SCAN_LDATA20_DS4_MASK_SHIFT(reg);
    cfg->dc4 = SCAN_LDATA20_DC4_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA20(scanblkDeviceHandle *device_data, struct scanblk_LDATA20_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA20);
    if(cfg->ds3_valid)
        reg = SCAN_LDATA20_DS3_REPLACE_VAL(reg, cfg->ds3);
    if(cfg->dc3_valid)
        reg = SCAN_LDATA20_DC3_REPLACE_VAL(reg, cfg->dc3);
    if(cfg->ds4_valid)
        reg = SCAN_LDATA20_DS4_REPLACE_VAL(reg, cfg->ds4);
    if(cfg->dc4_valid)
        reg = SCAN_LDATA20_DC4_REPLACE_VAL(reg, cfg->dc4);
    scanblkWrite(LDATA20, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA21(scanblkDeviceHandle *device_data, struct scanblk_LDATA21_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA21);
    cfg->ds3 = SCAN_LDATA21_DS3_MASK_SHIFT(reg);
    cfg->dc3 = SCAN_LDATA21_DC3_MASK_SHIFT(reg);
    cfg->ds4 = SCAN_LDATA21_DS4_MASK_SHIFT(reg);
    cfg->dc4 = SCAN_LDATA21_DC4_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA21(scanblkDeviceHandle *device_data, struct scanblk_LDATA21_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA21);
    if(cfg->ds3_valid)
        reg = SCAN_LDATA21_DS3_REPLACE_VAL(reg, cfg->ds3);
    if(cfg->dc3_valid)
        reg = SCAN_LDATA21_DC3_REPLACE_VAL(reg, cfg->dc3);
    if(cfg->ds4_valid)
        reg = SCAN_LDATA21_DS4_REPLACE_VAL(reg, cfg->ds4);
    if(cfg->dc4_valid)
        reg = SCAN_LDATA21_DC4_REPLACE_VAL(reg, cfg->dc4);
    scanblkWrite(LDATA21, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA30(scanblkDeviceHandle *device_data, struct scanblk_LDATA30_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA30);
    cfg->ds5 = SCAN_LDATA30_DS5_MASK_SHIFT(reg);
    cfg->dc5 = SCAN_LDATA30_DC5_MASK_SHIFT(reg);
    cfg->ds6 = SCAN_LDATA30_DS6_MASK_SHIFT(reg);
    cfg->dc6 = SCAN_LDATA30_DC6_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA30(scanblkDeviceHandle *device_data, struct scanblk_LDATA30_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA30);
    if(cfg->ds5_valid)
        reg = SCAN_LDATA30_DS5_REPLACE_VAL(reg, cfg->ds5);
    if(cfg->dc5_valid)
        reg = SCAN_LDATA30_DC5_REPLACE_VAL(reg, cfg->dc5);
    if(cfg->ds6_valid)
        reg = SCAN_LDATA30_DS6_REPLACE_VAL(reg, cfg->ds6);
    if(cfg->dc6_valid)
        reg = SCAN_LDATA30_DC6_REPLACE_VAL(reg, cfg->dc6);
    scanblkWrite(LDATA30, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA31(scanblkDeviceHandle *device_data, struct scanblk_LDATA31_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA31);
    cfg->ds5 = SCAN_LDATA31_DS5_MASK_SHIFT(reg);
    cfg->dc5 = SCAN_LDATA31_DC5_MASK_SHIFT(reg);
    cfg->ds6 = SCAN_LDATA31_DS6_MASK_SHIFT(reg);
    cfg->dc6 = SCAN_LDATA31_DC6_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA31(scanblkDeviceHandle *device_data, struct scanblk_LDATA31_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA31);
    if(cfg->ds5_valid)
        reg = SCAN_LDATA31_DS5_REPLACE_VAL(reg, cfg->ds5);
    if(cfg->dc5_valid)
        reg = SCAN_LDATA31_DC5_REPLACE_VAL(reg, cfg->dc5);
    if(cfg->ds6_valid)
        reg = SCAN_LDATA31_DS6_REPLACE_VAL(reg, cfg->ds6);
    if(cfg->dc6_valid)
        reg = SCAN_LDATA31_DC6_REPLACE_VAL(reg, cfg->dc6);
    scanblkWrite(LDATA31, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA40(scanblkDeviceHandle *device_data, struct scanblk_LDATA40_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA40);
    cfg->ds7 = SCAN_LDATA40_DS7_MASK_SHIFT(reg);
    cfg->dc7 = SCAN_LDATA40_DC7_MASK_SHIFT(reg);
    cfg->ds8 = SCAN_LDATA40_DS8_MASK_SHIFT(reg);
    cfg->dc8 = SCAN_LDATA40_DC8_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA40(scanblkDeviceHandle *device_data, struct scanblk_LDATA40_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA40);
    if(cfg->ds7_valid)
        reg = SCAN_LDATA40_DS7_REPLACE_VAL(reg, cfg->ds7);
    if(cfg->dc7_valid)
        reg = SCAN_LDATA40_DC7_REPLACE_VAL(reg, cfg->dc7);
    if(cfg->ds8_valid)
        reg = SCAN_LDATA40_DS8_REPLACE_VAL(reg, cfg->ds8);
    if(cfg->dc8_valid)
        reg = SCAN_LDATA40_DC8_REPLACE_VAL(reg, cfg->dc8);
    scanblkWrite(LDATA40, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA41(scanblkDeviceHandle *device_data, struct scanblk_LDATA41_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA41);
    cfg->ds7 = SCAN_LDATA41_DS7_MASK_SHIFT(reg);
    cfg->dc7 = SCAN_LDATA41_DC7_MASK_SHIFT(reg);
    cfg->ds8 = SCAN_LDATA41_DS8_MASK_SHIFT(reg);
    cfg->dc8 = SCAN_LDATA41_DC8_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA41(scanblkDeviceHandle *device_data, struct scanblk_LDATA41_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA41);
    if(cfg->ds7_valid)
        reg = SCAN_LDATA41_DS7_REPLACE_VAL(reg, cfg->ds7);
    if(cfg->dc7_valid)
        reg = SCAN_LDATA41_DC7_REPLACE_VAL(reg, cfg->dc7);
    if(cfg->ds8_valid)
        reg = SCAN_LDATA41_DS8_REPLACE_VAL(reg, cfg->ds8);
    if(cfg->dc8_valid)
        reg = SCAN_LDATA41_DC8_REPLACE_VAL(reg, cfg->dc8);
    scanblkWrite(LDATA41, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA50(scanblkDeviceHandle *device_data, struct scanblk_LDATA50_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA50);
    cfg->ds9 = SCAN_LDATA50_DS9_MASK_SHIFT(reg);
    cfg->dc9 = SCAN_LDATA50_DC9_MASK_SHIFT(reg);
    cfg->ds10 = SCAN_LDATA50_DS10_MASK_SHIFT(reg);
    cfg->dc10 = SCAN_LDATA50_DC10_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA50(scanblkDeviceHandle *device_data, struct scanblk_LDATA50_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA50);
    if(cfg->ds9_valid)
        reg = SCAN_LDATA50_DS9_REPLACE_VAL(reg, cfg->ds9);
    if(cfg->dc9_valid)
        reg = SCAN_LDATA50_DC9_REPLACE_VAL(reg, cfg->dc9);
    if(cfg->ds10_valid)
        reg = SCAN_LDATA50_DS10_REPLACE_VAL(reg, cfg->ds10);
    if(cfg->dc10_valid)
        reg = SCAN_LDATA50_DC10_REPLACE_VAL(reg, cfg->dc10);
    scanblkWrite(LDATA50, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA51(scanblkDeviceHandle *device_data, struct scanblk_LDATA51_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA51);
    cfg->ds9 = SCAN_LDATA51_DS9_MASK_SHIFT(reg);
    cfg->dc9 = SCAN_LDATA51_DC9_MASK_SHIFT(reg);
    cfg->ds10 = SCAN_LDATA51_DS10_MASK_SHIFT(reg);
    cfg->dc10 = SCAN_LDATA51_DC10_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA51(scanblkDeviceHandle *device_data, struct scanblk_LDATA51_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA51);
    if(cfg->ds9_valid)
        reg = SCAN_LDATA51_DS9_REPLACE_VAL(reg, cfg->ds9);
    if(cfg->dc9_valid)
        reg = SCAN_LDATA51_DC9_REPLACE_VAL(reg, cfg->dc9);
    if(cfg->ds10_valid)
        reg = SCAN_LDATA51_DS10_REPLACE_VAL(reg, cfg->ds10);
    if(cfg->dc10_valid)
        reg = SCAN_LDATA51_DC10_REPLACE_VAL(reg, cfg->dc10);
    scanblkWrite(LDATA51, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA60(scanblkDeviceHandle *device_data, struct scanblk_LDATA60_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA60);
    cfg->ds11 = SCAN_LDATA60_DS11_MASK_SHIFT(reg);
    cfg->dc11 = SCAN_LDATA60_DC11_MASK_SHIFT(reg);
    cfg->ds12 = SCAN_LDATA60_DS12_MASK_SHIFT(reg);
    cfg->dc12 = SCAN_LDATA60_DC12_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA60(scanblkDeviceHandle *device_data, struct scanblk_LDATA60_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA60);
    if(cfg->ds11_valid)
        reg = SCAN_LDATA60_DS11_REPLACE_VAL(reg, cfg->ds11);
    if(cfg->dc11_valid)
        reg = SCAN_LDATA60_DC11_REPLACE_VAL(reg, cfg->dc11);
    if(cfg->ds12_valid)
        reg = SCAN_LDATA60_DS12_REPLACE_VAL(reg, cfg->ds12);
    if(cfg->dc12_valid)
        reg = SCAN_LDATA60_DC12_REPLACE_VAL(reg, cfg->dc12);
    scanblkWrite(LDATA60, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LDATA61(scanblkDeviceHandle *device_data, struct scanblk_LDATA61_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LDATA61);
    cfg->ds11 = SCAN_LDATA61_DS11_MASK_SHIFT(reg);
    cfg->dc11 = SCAN_LDATA61_DC11_MASK_SHIFT(reg);
    cfg->ds12 = SCAN_LDATA61_DS12_MASK_SHIFT(reg);
    cfg->dc12 = SCAN_LDATA61_DC12_MASK_SHIFT(reg);
}

static void set_scanblk_LDATA61(scanblkDeviceHandle *device_data, struct scanblk_LDATA61_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LDATA61);
    if(cfg->ds11_valid)
        reg = SCAN_LDATA61_DS11_REPLACE_VAL(reg, cfg->ds11);
    if(cfg->dc11_valid)
        reg = SCAN_LDATA61_DC11_REPLACE_VAL(reg, cfg->dc11);
    if(cfg->ds12_valid)
        reg = SCAN_LDATA61_DS12_REPLACE_VAL(reg, cfg->ds12);
    if(cfg->dc12_valid)
        reg = SCAN_LDATA61_DC12_REPLACE_VAL(reg, cfg->dc12);
    scanblkWrite(LDATA61, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_PSEQ1(scanblkDeviceHandle *device_data, struct scanblk_PSEQ1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(PSEQ1);
    cfg->pseq1 = SCAN_PSEQ1_PSEQ1_MASK_SHIFT(reg);
    cfg->pseq2 = SCAN_PSEQ1_PSEQ2_MASK_SHIFT(reg);
    cfg->pseq3 = SCAN_PSEQ1_PSEQ3_MASK_SHIFT(reg);
    cfg->pseq4 = SCAN_PSEQ1_PSEQ4_MASK_SHIFT(reg);
}

static void set_scanblk_PSEQ1(scanblkDeviceHandle *device_data, struct scanblk_PSEQ1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(PSEQ1);
    if(cfg->pseq1_valid)
        reg = SCAN_PSEQ1_PSEQ1_REPLACE_VAL(reg, cfg->pseq1);
    if(cfg->pseq2_valid)
        reg = SCAN_PSEQ1_PSEQ2_REPLACE_VAL(reg, cfg->pseq2);
    if(cfg->pseq3_valid)
        reg = SCAN_PSEQ1_PSEQ3_REPLACE_VAL(reg, cfg->pseq3);
    if(cfg->pseq4_valid)
        reg = SCAN_PSEQ1_PSEQ4_REPLACE_VAL(reg, cfg->pseq4);
    scanblkWrite(PSEQ1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_PSEQ2(scanblkDeviceHandle *device_data, struct scanblk_PSEQ2_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(PSEQ2);
    cfg->pseq5 = SCAN_PSEQ2_PSEQ5_MASK_SHIFT(reg);
    cfg->pseq6 = SCAN_PSEQ2_PSEQ6_MASK_SHIFT(reg);
}

static void set_scanblk_PSEQ2(scanblkDeviceHandle *device_data, struct scanblk_PSEQ2_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(PSEQ2);
    if(cfg->pseq5_valid)
        reg = SCAN_PSEQ2_PSEQ5_REPLACE_VAL(reg, cfg->pseq5);
    if(cfg->pseq6_valid)
        reg = SCAN_PSEQ2_PSEQ6_REPLACE_VAL(reg, cfg->pseq6);
    scanblkWrite(PSEQ2, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_BPWM(scanblkDeviceHandle *device_data, struct scanblk_BPWM_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(BPWM);
    cfg->bpwmper = SCAN_BPWM_BPWMPER_MASK_SHIFT(reg);
    cfg->bpwmdty = SCAN_BPWM_BPWMDTY_MASK_SHIFT(reg);
}

static void set_scanblk_BPWM(scanblkDeviceHandle *device_data, struct scanblk_BPWM_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(BPWM);
    if(cfg->bpwmper_valid)
        reg = SCAN_BPWM_BPWMPER_REPLACE_VAL(reg, cfg->bpwmper);
    if(cfg->bpwmdty_valid)
        reg = SCAN_BPWM_BPWMDTY_REPLACE_VAL(reg, cfg->bpwmdty);
    scanblkWrite(BPWM, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_MCFG(scanblkDeviceHandle *device_data, struct scanblk_MCFG_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(MCFG);
    cfg->psmode = SCAN_MCFG_PSMODE_MASK_SHIFT(reg);
    cfg->pscfg = SCAN_MCFG_PSCFG_MASK_SHIFT(reg);
    cfg->pscount = SCAN_MCFG_PSCOUNT_MASK_SHIFT(reg);
    cfg->lsmode = SCAN_MCFG_LSMODE_MASK_SHIFT(reg);
    cfg->mtype = SCAN_MCFG_MTYPE_MASK_SHIFT(reg);
}

static void set_scanblk_MCFG(scanblkDeviceHandle *device_data, struct scanblk_MCFG_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(MCFG);
    if(cfg->psmode_valid)
        reg = SCAN_MCFG_PSMODE_REPLACE_VAL(reg, cfg->psmode);
    if(cfg->pscfg_valid)
        reg = SCAN_MCFG_PSCFG_REPLACE_VAL(reg, cfg->pscfg);
    if(cfg->pscount_valid)
        reg = SCAN_MCFG_PSCOUNT_REPLACE_VAL(reg, cfg->pscount);
    if(cfg->lsmode_valid)
        reg = SCAN_MCFG_LSMODE_REPLACE_VAL(reg, cfg->lsmode);
    if(cfg->mtype_valid)
        reg = SCAN_MCFG_MTYPE_REPLACE_VAL(reg, cfg->mtype);
    scanblkWrite(MCFG, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_MOTORSYNCMIN(scanblkDeviceHandle *device_data, struct scanblk_MOTORSYNCMIN_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(MotorSyncMin);
    cfg->msmin = SCAN_MOTORSYNCMIN_MSMIN_MASK_SHIFT(reg);
}

static void set_scanblk_MOTORSYNCMIN(scanblkDeviceHandle *device_data, struct scanblk_MOTORSYNCMIN_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(MotorSyncMin);
    if(cfg->msmin_valid)
        reg = SCAN_MOTORSYNCMIN_MSMIN_REPLACE_VAL(reg, cfg->msmin);
    scanblkWrite(MotorSyncMin, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_AFEPC(scanblkDeviceHandle *device_data, struct scanblk_AFEPC_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(AFEPC);
    cfg->field_auto = SCAN_AFEPC_AUTO_MASK_SHIFT(reg);
}

static void set_scanblk_AFEPC(scanblkDeviceHandle *device_data, struct scanblk_AFEPC_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(AFEPC);
    if(cfg->field_auto_valid)
        reg = SCAN_AFEPC_AUTO_REPLACE_VAL(reg, cfg->field_auto);
    scanblkWrite(AFEPC, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_CFGARB(scanblkDeviceHandle *device_data, struct scanblk_CFGARB_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(CFGARB);
    cfg->channel = SCAN_CFGARB_CHANNEL_MASK_SHIFT(reg);
    cfg->extenable = SCAN_CFGARB_EXTENABLE_MASK_SHIFT(reg);
    cfg->parenable = SCAN_CFGARB_PARENABLE_MASK_SHIFT(reg);
}

static void set_scanblk_CFGARB(scanblkDeviceHandle *device_data, struct scanblk_CFGARB_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(CFGARB);
    if(cfg->channel_valid)
        reg = SCAN_CFGARB_CHANNEL_REPLACE_VAL(reg, cfg->channel);
    if(cfg->extenable_valid)
        reg = SCAN_CFGARB_EXTENABLE_REPLACE_VAL(reg, cfg->extenable);
    if(cfg->parenable_valid)
        reg = SCAN_CFGARB_PARENABLE_REPLACE_VAL(reg, cfg->parenable);
    scanblkWrite(CFGARB, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED00(scanblkDeviceHandle *device_data, struct scanblk_LED00_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED00);
    cfg->led0pos = SCAN_LED00_LED0POS_MASK_SHIFT(reg);
    cfg->led0neg = SCAN_LED00_LED0NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED00(scanblkDeviceHandle *device_data, struct scanblk_LED00_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED00);
    if(cfg->led0pos_valid)
        reg = SCAN_LED00_LED0POS_REPLACE_VAL(reg, cfg->led0pos);
    if(cfg->led0neg_valid)
        reg = SCAN_LED00_LED0NEG_REPLACE_VAL(reg, cfg->led0neg);
    scanblkWrite(LED00, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED01(scanblkDeviceHandle *device_data, struct scanblk_LED01_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED01);
    cfg->led0pos = SCAN_LED01_LED0POS_MASK_SHIFT(reg);
    cfg->led0neg = SCAN_LED01_LED0NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED01(scanblkDeviceHandle *device_data, struct scanblk_LED01_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED01);
    if(cfg->led0pos_valid)
        reg = SCAN_LED01_LED0POS_REPLACE_VAL(reg, cfg->led0pos);
    if(cfg->led0neg_valid)
        reg = SCAN_LED01_LED0NEG_REPLACE_VAL(reg, cfg->led0neg);
    scanblkWrite(LED01, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED10(scanblkDeviceHandle *device_data, struct scanblk_LED10_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED10);
    cfg->led1pos = SCAN_LED10_LED1POS_MASK_SHIFT(reg);
    cfg->led1neg = SCAN_LED10_LED1NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED10(scanblkDeviceHandle *device_data, struct scanblk_LED10_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED10);
    if(cfg->led1pos_valid)
        reg = SCAN_LED10_LED1POS_REPLACE_VAL(reg, cfg->led1pos);
    if(cfg->led1neg_valid)
        reg = SCAN_LED10_LED1NEG_REPLACE_VAL(reg, cfg->led1neg);
    scanblkWrite(LED10, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED11(scanblkDeviceHandle *device_data, struct scanblk_LED11_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED11);
    cfg->led1pos = SCAN_LED11_LED1POS_MASK_SHIFT(reg);
    cfg->led1neg = SCAN_LED11_LED1NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED11(scanblkDeviceHandle *device_data, struct scanblk_LED11_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED11);
    if(cfg->led1pos_valid)
        reg = SCAN_LED11_LED1POS_REPLACE_VAL(reg, cfg->led1pos);
    if(cfg->led1neg_valid)
        reg = SCAN_LED11_LED1NEG_REPLACE_VAL(reg, cfg->led1neg);
    scanblkWrite(LED11, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED20(scanblkDeviceHandle *device_data, struct scanblk_LED20_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED20);
    cfg->led2pos = SCAN_LED20_LED2POS_MASK_SHIFT(reg);
    cfg->led2neg = SCAN_LED20_LED2NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED20(scanblkDeviceHandle *device_data, struct scanblk_LED20_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED20);
    if(cfg->led2pos_valid)
        reg = SCAN_LED20_LED2POS_REPLACE_VAL(reg, cfg->led2pos);
    if(cfg->led2neg_valid)
        reg = SCAN_LED20_LED2NEG_REPLACE_VAL(reg, cfg->led2neg);
    scanblkWrite(LED20, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LED21(scanblkDeviceHandle *device_data, struct scanblk_LED21_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LED21);
    cfg->led2pos = SCAN_LED21_LED2POS_MASK_SHIFT(reg);
    cfg->led2neg = SCAN_LED21_LED2NEG_MASK_SHIFT(reg);
}

static void set_scanblk_LED21(scanblkDeviceHandle *device_data, struct scanblk_LED21_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LED21);
    if(cfg->led2pos_valid)
        reg = SCAN_LED21_LED2POS_REPLACE_VAL(reg, cfg->led2pos);
    if(cfg->led2neg_valid)
        reg = SCAN_LED21_LED2NEG_REPLACE_VAL(reg, cfg->led2neg);
    scanblkWrite(LED21, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWMCONFIG0(scanblkDeviceHandle *device_data, struct scanblk_LPWMCONFIG0_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWMConfig0);
    cfg->lpwmper = SCAN_LPWMCONFIG0_LPWMPER_MASK_SHIFT(reg);
    cfg->ledpol = SCAN_LPWMCONFIG0_LEDPOL_MASK_SHIFT(reg);
}

static void set_scanblk_LPWMCONFIG0(scanblkDeviceHandle *device_data, struct scanblk_LPWMCONFIG0_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWMConfig0);
    if(cfg->lpwmper_valid)
        reg = SCAN_LPWMCONFIG0_LPWMPER_REPLACE_VAL(reg, cfg->lpwmper);
    if(cfg->ledpol_valid)
        reg = SCAN_LPWMCONFIG0_LEDPOL_REPLACE_VAL(reg, cfg->ledpol);
    scanblkWrite(LPWMConfig0, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWMCONFIG1(scanblkDeviceHandle *device_data, struct scanblk_LPWMCONFIG1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWMConfig1);
    cfg->lpwmper = SCAN_LPWMCONFIG1_LPWMPER_MASK_SHIFT(reg);
    cfg->ledpol = SCAN_LPWMCONFIG1_LEDPOL_MASK_SHIFT(reg);
}

static void set_scanblk_LPWMCONFIG1(scanblkDeviceHandle *device_data, struct scanblk_LPWMCONFIG1_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWMConfig1);
    if(cfg->lpwmper_valid)
        reg = SCAN_LPWMCONFIG1_LPWMPER_REPLACE_VAL(reg, cfg->lpwmper);
    if(cfg->ledpol_valid)
        reg = SCAN_LPWMCONFIG1_LEDPOL_REPLACE_VAL(reg, cfg->ledpol);
    scanblkWrite(LPWMConfig1, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM00(scanblkDeviceHandle *device_data, struct scanblk_LPWM00_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM00);
    cfg->led0on = SCAN_LPWM00_LED0ON_MASK_SHIFT(reg);
    cfg->led0off = SCAN_LPWM00_LED0OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM00(scanblkDeviceHandle *device_data, struct scanblk_LPWM00_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM00);
    if(cfg->led0on_valid)
        reg = SCAN_LPWM00_LED0ON_REPLACE_VAL(reg, cfg->led0on);
    if(cfg->led0off_valid)
        reg = SCAN_LPWM00_LED0OFF_REPLACE_VAL(reg, cfg->led0off);
    scanblkWrite(LPWM00, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM01(scanblkDeviceHandle *device_data, struct scanblk_LPWM01_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM01);
    cfg->led0on = SCAN_LPWM01_LED0ON_MASK_SHIFT(reg);
    cfg->led0off = SCAN_LPWM01_LED0OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM01(scanblkDeviceHandle *device_data, struct scanblk_LPWM01_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM01);
    if(cfg->led0on_valid)
        reg = SCAN_LPWM01_LED0ON_REPLACE_VAL(reg, cfg->led0on);
    if(cfg->led0off_valid)
        reg = SCAN_LPWM01_LED0OFF_REPLACE_VAL(reg, cfg->led0off);
    scanblkWrite(LPWM01, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM10(scanblkDeviceHandle *device_data, struct scanblk_LPWM10_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM10);
    cfg->led1on = SCAN_LPWM10_LED1ON_MASK_SHIFT(reg);
    cfg->led1off = SCAN_LPWM10_LED1OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM10(scanblkDeviceHandle *device_data, struct scanblk_LPWM10_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM10);
    if(cfg->led1on_valid)
        reg = SCAN_LPWM10_LED1ON_REPLACE_VAL(reg, cfg->led1on);
    if(cfg->led1off_valid)
        reg = SCAN_LPWM10_LED1OFF_REPLACE_VAL(reg, cfg->led1off);
    scanblkWrite(LPWM10, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM11(scanblkDeviceHandle *device_data, struct scanblk_LPWM11_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM11);
    cfg->led1on = SCAN_LPWM11_LED1ON_MASK_SHIFT(reg);
    cfg->led1off = SCAN_LPWM11_LED1OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM11(scanblkDeviceHandle *device_data, struct scanblk_LPWM11_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM11);
    if(cfg->led1on_valid)
        reg = SCAN_LPWM11_LED1ON_REPLACE_VAL(reg, cfg->led1on);
    if(cfg->led1off_valid)
        reg = SCAN_LPWM11_LED1OFF_REPLACE_VAL(reg, cfg->led1off);
    scanblkWrite(LPWM11, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM20(scanblkDeviceHandle *device_data, struct scanblk_LPWM20_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM20);
    cfg->led2on = SCAN_LPWM20_LED2ON_MASK_SHIFT(reg);
    cfg->led2off = SCAN_LPWM20_LED2OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM20(scanblkDeviceHandle *device_data, struct scanblk_LPWM20_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM20);
    if(cfg->led2on_valid)
        reg = SCAN_LPWM20_LED2ON_REPLACE_VAL(reg, cfg->led2on);
    if(cfg->led2off_valid)
        reg = SCAN_LPWM20_LED2OFF_REPLACE_VAL(reg, cfg->led2off);
    scanblkWrite(LPWM20, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_LPWM21(scanblkDeviceHandle *device_data, struct scanblk_LPWM21_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(LPWM21);
    cfg->led2on = SCAN_LPWM21_LED2ON_MASK_SHIFT(reg);
    cfg->led2off = SCAN_LPWM21_LED2OFF_MASK_SHIFT(reg);
}

static void set_scanblk_LPWM21(scanblkDeviceHandle *device_data, struct scanblk_LPWM21_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(LPWM21);
    if(cfg->led2on_valid)
        reg = SCAN_LPWM21_LED2ON_REPLACE_VAL(reg, cfg->led2on);
    if(cfg->led2off_valid)
        reg = SCAN_LPWM21_LED2OFF_REPLACE_VAL(reg, cfg->led2off);
    scanblkWrite(LPWM21, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_EXCFG(scanblkDeviceHandle *device_data, struct scanblk_EXCFG_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(EXCFG);
    cfg->expenable = SCAN_EXCFG_EXPENABLE_MASK_SHIFT(reg);
    cfg->expmask = SCAN_EXCFG_EXPMASK_MASK_SHIFT(reg);
    cfg->expdiv = SCAN_EXCFG_EXPDIV_MASK_SHIFT(reg);
}

static void set_scanblk_EXCFG(scanblkDeviceHandle *device_data, struct scanblk_EXCFG_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(EXCFG);
    if(cfg->expenable_valid)
        reg = SCAN_EXCFG_EXPENABLE_REPLACE_VAL(reg, cfg->expenable);
    if(cfg->expmask_valid)
        reg = SCAN_EXCFG_EXPMASK_REPLACE_VAL(reg, cfg->expmask);
    if(cfg->expdiv_valid)
        reg = SCAN_EXCFG_EXPDIV_REPLACE_VAL(reg, cfg->expdiv);
    scanblkWrite(EXCFG, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_OFIE(scanblkDeviceHandle *device_data, struct scanblk_OFIE_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(OFIE);
    cfg->eovf_ie = SCAN_OFIE_EOVF_IE_MASK_SHIFT(reg);
    cfg->iovf_ie = SCAN_OFIE_IOVF_IE_MASK_SHIFT(reg);
}

static void set_scanblk_OFIE(scanblkDeviceHandle *device_data, struct scanblk_OFIE_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(OFIE);
    if(cfg->eovf_ie_valid)
        reg = SCAN_OFIE_EOVF_IE_REPLACE_VAL(reg, cfg->eovf_ie);
    if(cfg->iovf_ie_valid)
        reg = SCAN_OFIE_IOVF_IE_REPLACE_VAL(reg, cfg->iovf_ie);
    scanblkWrite(OFIE, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_OFIP(scanblkDeviceHandle *device_data, struct scanblk_OFIP_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(OFIP);
    cfg->eovf_ip = SCAN_OFIP_EOVF_IP_MASK_SHIFT(reg);
    cfg->iovf_ip = SCAN_OFIP_IOVF_IP_MASK_SHIFT(reg);
}

static void set_scanblk_OFIA(scanblkDeviceHandle *device_data, struct scanblk_OFIA_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(OFIA);
    if(cfg->eovf_ia_valid)
        reg = SCAN_OFIA_EOVF_IA_REPLACE_VAL(reg, cfg->eovf_ia);
    if(cfg->iovf_ia_valid)
        reg = SCAN_OFIA_IOVF_IA_REPLACE_VAL(reg, cfg->iovf_ia);
    scanblkWrite(OFIA, reg);
    UNPROTECT_REG_ACCESS;
}

static void set_scanblk_OFIF(scanblkDeviceHandle *device_data, struct scanblk_OFIF_cfg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = scanblkRead(OFIF);
    if(cfg->eovf_if_valid)
        reg = SCAN_OFIF_EOVF_IF_REPLACE_VAL(reg, cfg->eovf_if);
    if(cfg->iovf_if_valid)
        reg = SCAN_OFIF_IOVF_IF_REPLACE_VAL(reg, cfg->iovf_if);
    scanblkWrite(OFIF, reg);
    UNPROTECT_REG_ACCESS;
}

static void get_scanblk_REV0(scanblkDeviceHandle *device_data, struct scanblk_REV0_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(REV0);
    cfg->cfg = SCAN_REV0_CFG_MASK_SHIFT(reg);
    cfg->maj = SCAN_REV0_MAJ_MASK_SHIFT(reg);
}

static void get_scanblk_REV1(scanblkDeviceHandle *device_data, struct scanblk_REV1_cfg *cfg)
{
    uint32_t reg;

    reg = scanblkRead(REV1);
    cfg->mid = SCAN_REV1_MID_MASK_SHIFT(reg);
}


static void register_irqcallback (scanblkDeviceHandle *device_data, scanblk_irqcallback_fcn fcn)
{
    device_data->interrupt_callback = fcn;
}

static void enable_int (scanblkDeviceHandle *device_data, struct scanblk_irqs *irqstruct)
{
    struct scanblk_SCTRL_cfg cfg;
    memset(&cfg, 0x0, sizeof(struct scanblk_SCTRL_cfg));

    if(irqstruct->unexp_mtr_sync)
    {
        cfg.umsen = 1;
        cfg.umsen_valid = true;
    }
    if(irqstruct->queue_overrun)
    {
        /* I think nothing needs to be done here. */
    }
    if(irqstruct->cmd_int_overrun)
    {
        /* I think nothing needs to be done here. */
    }
    if(irqstruct->scan_cmd_complete)
    {
        cfg.scmdie = 1;
        cfg.scmdie_valid = true;
    }
    set_scanblk_SCTRL_thread(device_data, &cfg);
}

static void get_enabled_irqs(scanblkDeviceHandle *device_data, struct scanblk_irqs *irqstruct)
{
    struct scanblk_SCTRL_cfg cfg;

    get_scanblk_SCTRL(device_data, &cfg);
    
    if (cfg.umsen)
    {
        irqstruct->unexp_mtr_sync = true;
    }
    if (cfg.scmdie)
    {
        irqstruct->scan_cmd_complete = true;
    }
}

static void disable_int (scanblkDeviceHandle *device_data, struct scanblk_irqs *irqstruct)
{
    struct scanblk_SCTRL_cfg cfg;
    memset(&cfg, 0x0, sizeof(struct scanblk_SCTRL_cfg));
    
    if(irqstruct->unexp_mtr_sync)
    {
        cfg.umsen_valid = true;
    }
    if(irqstruct->queue_overrun)
    {
        /* I think nothing needs to be done here. */
    }
    if(irqstruct->cmd_int_overrun)
    {
        /* I think nothing needs to be done here. */
    }
    if(irqstruct->scan_cmd_complete)
    {
        cfg.scmdie_valid = true;
    }
    set_scanblk_SCTRL_thread(device_data, &cfg);
}


void soft_reset(scanblkDeviceHandle *device_data)
{
    struct scanblk_SCTRL_cfg sctrl;
    struct scanblk_SCFG1_cfg scfg1;

    /* shut down all clocks, interrupts, and commands first */
    memset(&sctrl, 0x0, sizeof(sctrl));
    sctrl.umsen_valid   = true;
    sctrl.cmdqen_valid  = true;
    sctrl.termcmd_valid = true;
    sctrl.sreset_valid  = true;
    sctrl.scmdie_valid  = true;
    sctrl.run_valid     = true;
    sctrl.scanen_valid  = true;
    set_scanblk_SCTRL_thread(device_data, &sctrl);

    /* ack any and all pending interrupts */
    clear_scan_irqs(device_data, NULL);

    /* disable all clocks */
    memset(&scfg1, 0x0, sizeof(scfg1));
    scfg1.p1clken_valid = true;
    scfg1.p2clken_valid = true;
    scfg1.rsclken_valid = true;
    scfg1.cpclken_valid = true;
    scfg1.mclken_valid  = true;
    scfg1.vsclken_valid = true;
    scfg1.clclken_valid = true;
    scfg1.lclmpen_valid = true;
    scfg1.bulben_valid  = true;
    scfg1.ssmen_valid   = true;
    scfg1.smpol_valid   = true;
    scfg1.p1apol_valid  = true;
    scfg1.sptog_valid   = true;
    scfg1.sppol_valid   = true;
    scfg1.scpol_valid   = true;
    scfg1.acpol_valid   = true;
    set_scanblk_SCFG1(device_data, &scfg1);

    /* Scan block soft reset */
    sctrl.sreset       = 1;
    sctrl.sreset_valid = true;
    set_scanblk_SCTRL_thread(device_data, &sctrl);
    udelay(500);
    sctrl.sreset       = 0;
    sctrl.sreset_valid = true;
    set_scanblk_SCTRL_thread(device_data, &sctrl);

    /****************
     * set all registers to their default power-on values.
     ****************/
    set_scanblk_SCFG1        (device_data,   &scfg1_default);
    set_scanblk_SCFG2        (device_data,   &scfg2_default);
    set_scanblk_SCFG3        (device_data,   &scfg3_default);
    set_scanblk_SCTRL_thread (device_data,   &sctrl_default);
    set_scanblk_SCANX   (device_data,   &scanx_default);
    set_scanblk_SCYCLE  (device_data,  &scycle_default);
    set_scanblk_STCFG1  (device_data,  &stcfg1_default);
    set_scanblk_STCFG2  (device_data,  &stcfg2_default);
    set_scanblk_STCFG3  (device_data,  &stcfg3_default);
    set_scanblk_STCFG4  (device_data,  &stcfg4_default);
    set_scanblk_CCFG    (device_data,    &ccfg_default);
    set_scanblk_SCLK1   (device_data,   &sclk1_default);
    set_scanblk_SCLK2   (device_data,   &sclk2_default);
    set_scanblk_SCLK3   (device_data,   &sclk3_default);
    set_scanblk_SCLK4   (device_data,   &sclk4_default);
    set_scanblk_SCLK5   (device_data,   &sclk5_default);
    set_scanblk_SCLK6   (device_data,   &sclk6_default);
    set_scanblk_ACLK1   (device_data,   &aclk1_default);
    set_scanblk_ACLK2   (device_data,   &aclk2_default);
    set_scanblk_ACLK3   (device_data,   &aclk3_default);
    set_scanblk_ACLK4   (device_data,   &aclk4_default);
    set_scanblk_ACLK5   (device_data,   &aclk5_default);
    set_scanblk_LDATA10 (device_data, &ldata10_default);
    set_scanblk_LDATA11 (device_data, &ldata11_default);
    set_scanblk_LDATA20 (device_data, &ldata20_default);
    set_scanblk_LDATA21 (device_data, &ldata21_default);
    set_scanblk_LDATA30 (device_data, &ldata30_default);
    set_scanblk_LDATA31 (device_data, &ldata31_default);
    set_scanblk_LDATA40 (device_data, &ldata40_default);
    set_scanblk_LDATA41 (device_data, &ldata41_default);
    set_scanblk_LDATA50 (device_data, &ldata50_default);
    set_scanblk_LDATA51 (device_data, &ldata51_default);
    set_scanblk_LDATA60 (device_data, &ldata60_default);
    set_scanblk_LDATA61 (device_data, &ldata61_default);
    set_scanblk_PSEQ1   (device_data,   &pseq1_default);
    set_scanblk_PSEQ2   (device_data,   &pseq2_default);
    set_scanblk_BPWM    (device_data,    &bpwm_default);
    set_scanblk_CFGARB  (device_data,  &cfgarb_default);
}

void register_override(scanblkDeviceHandle *device_data, char * regname, uint32_t value)
{
    debug_print("%s: regname=<%s>, val=0x%08x\n", __func__, regname, value);

         if(strcmp(regname, "SCFG1")        == 0) { scanblkWrite(SCFG1,        value); }
    else if(strcmp(regname, "SCFG2")        == 0) { scanblkWrite(SCFG2,        value); }
    else if(strcmp(regname, "SCFG3")        == 0) { scanblkWrite(SCFG3,        value); }
    else if(strcmp(regname, "SCTRL")        == 0) { scanblkWrite(SCTRL,        value); }
    else if(strcmp(regname, "SCMD")         == 0) { scanblkWrite(SCMD,         value); }
    else if(strcmp(regname, "SSTAT")        == 0) { scanblkWrite(SSTAT,        value); }
    else if(strcmp(regname, "SIACK")        == 0) { scanblkWrite(SIACK,        value); }
    else if(strcmp(regname, "SCANX")        == 0) { scanblkWrite(SCANX,        value); }
    else if(strcmp(regname, "SCYCLE")       == 0) { scanblkWrite(SCYCLE,       value); }
    else if(strcmp(regname, "STCFG1")       == 0) { scanblkWrite(STCFG1,       value); }
    else if(strcmp(regname, "STCFG2")       == 0) { scanblkWrite(STCFG2,       value); }
    else if(strcmp(regname, "STCFG3")       == 0) { scanblkWrite(STCFG3,       value); }
    else if(strcmp(regname, "STCFG4")       == 0) { scanblkWrite(STCFG4,       value); }
    else if(strcmp(regname, "CCFG")         == 0) { scanblkWrite(CCFG,         value); }
    else if(strcmp(regname, "SCLK1")        == 0) { scanblkWrite(SCLK1,        value); }
    else if(strcmp(regname, "SCLK2")        == 0) { scanblkWrite(SCLK2,        value); }
    else if(strcmp(regname, "SCLK3")        == 0) { scanblkWrite(SCLK3,        value); }
    else if(strcmp(regname, "SCLK4")        == 0) { scanblkWrite(SCLK4,        value); }
    else if(strcmp(regname, "SCLK5")        == 0) { scanblkWrite(SCLK5,        value); }
    else if(strcmp(regname, "SCLK6")        == 0) { scanblkWrite(SCLK6,        value); }
    else if(strcmp(regname, "ACLK1")        == 0) { scanblkWrite(ACLK1,        value); }
    else if(strcmp(regname, "ACLK2")        == 0) { scanblkWrite(ACLK2,        value); }
    else if(strcmp(regname, "ACLK3")        == 0) { scanblkWrite(ACLK3,        value); }
    else if(strcmp(regname, "ACLK4")        == 0) { scanblkWrite(ACLK4,        value); }
    else if(strcmp(regname, "ACLK5")        == 0) { scanblkWrite(ACLK5,        value); }
    else if(strcmp(regname, "ACLK6")        == 0) { scanblkWrite(ACLK6,        value); }
    else if(strcmp(regname, "LDATA10")      == 0) { scanblkWrite(LDATA10,      value); }
    else if(strcmp(regname, "LDATA11")      == 0) { scanblkWrite(LDATA11,      value); }
    else if(strcmp(regname, "LDATA20")      == 0) { scanblkWrite(LDATA20,      value); }
    else if(strcmp(regname, "LDATA21")      == 0) { scanblkWrite(LDATA21,      value); }
    else if(strcmp(regname, "LDATA30")      == 0) { scanblkWrite(LDATA30,      value); }
    else if(strcmp(regname, "LDATA31")      == 0) { scanblkWrite(LDATA31,      value); }
    else if(strcmp(regname, "LDATA40")      == 0) { scanblkWrite(LDATA40,      value); }
    else if(strcmp(regname, "LDATA41")      == 0) { scanblkWrite(LDATA41,      value); }
    else if(strcmp(regname, "LDATA50")      == 0) { scanblkWrite(LDATA50,      value); }
    else if(strcmp(regname, "LDATA51")      == 0) { scanblkWrite(LDATA51,      value); }
    else if(strcmp(regname, "LDATA60")      == 0) { scanblkWrite(LDATA60,      value); }
    else if(strcmp(regname, "LDATA61")      == 0) { scanblkWrite(LDATA61,      value); }
    else if(strcmp(regname, "PSEQ1")        == 0) { scanblkWrite(PSEQ1,        value); }
    else if(strcmp(regname, "PSEQ2")        == 0) { scanblkWrite(PSEQ2,        value); }
    else if(strcmp(regname, "BPWM")         == 0) { scanblkWrite(BPWM,         value); }
    else if(strcmp(regname, "MCFG")         == 0) { scanblkWrite(MCFG,         value); }
    else if(strcmp(regname, "MotorSyncMin") == 0) { scanblkWrite(MotorSyncMin, value); }
    else if(strcmp(regname, "AFEPC")        == 0) { scanblkWrite(AFEPC,        value); }
    else if(strcmp(regname, "CFGARB")       == 0) { scanblkWrite(CFGARB,       value); }
    else if(strcmp(regname, "LED00")        == 0) { scanblkWrite(LED00,        value); }
    else if(strcmp(regname, "LED01")        == 0) { scanblkWrite(LED01,        value); }
    else if(strcmp(regname, "LED10")        == 0) { scanblkWrite(LED10,        value); }
    else if(strcmp(regname, "LED11")        == 0) { scanblkWrite(LED11,        value); }
    else if(strcmp(regname, "LED20")        == 0) { scanblkWrite(LED20,        value); }
    else if(strcmp(regname, "LED21")        == 0) { scanblkWrite(LED21,        value); }
    else if(strcmp(regname, "LPWMConfig0")  == 0) { scanblkWrite(LPWMConfig0,  value); }
    else if(strcmp(regname, "LPWMConfig1")  == 0) { scanblkWrite(LPWMConfig1,  value); }
    else if(strcmp(regname, "LPWM00")       == 0) { scanblkWrite(LPWM00,       value); }
    else if(strcmp(regname, "LPWM01")       == 0) { scanblkWrite(LPWM01,       value); }
    else if(strcmp(regname, "LPWM10")       == 0) { scanblkWrite(LPWM10,       value); }
    else if(strcmp(regname, "LPWM11")       == 0) { scanblkWrite(LPWM11,       value); }
    else if(strcmp(regname, "LPWM20")       == 0) { scanblkWrite(LPWM20,       value); }
    else if(strcmp(regname, "LPWM21")       == 0) { scanblkWrite(LPWM21,       value); }
    else if(strcmp(regname, "EXCFG")        == 0) { scanblkWrite(EXCFG,        value); }
    else if(strcmp(regname, "OFIE")         == 0) { scanblkWrite(OFIE,         value); }
    else if(strcmp(regname, "OFIP")         == 0) { scanblkWrite(OFIP,         value); }
    else if(strcmp(regname, "OFIA")         == 0) { scanblkWrite(OFIA,         value); }
    else if(strcmp(regname, "OFIF")         == 0) { scanblkWrite(OFIF,         value); }
    else if(strcmp(regname, "REV0")         == 0) { scanblkWrite(REV0,         value); }
    else if(strcmp(regname, "REV1")         == 0) { scanblkWrite(REV1,         value); }
    else { debug_print("%s: No matching register %s, ignoring!\n", __func__, regname); }
}

struct scanblk_common_function_struct scanblk_functions =
{
    .version = 0,

    .register_irqcallback = register_irqcallback,
    .enable_int           = enable_int,
    .get_enabled_ints     = get_enabled_irqs,
    .disable_int          = disable_int,
    .clear_scan_irqs      = clear_scan_irqs,
    .dump_regs            = dump_regs,
    .soft_reset           = soft_reset,
    .register_override    = register_override,
    .get_clk_speed        = get_clk_speed,

    /* Getters on the left, setters on the right -- one per register */
    .get_scanblk_SCFG1        = get_scanblk_SCFG1,        .set_scanblk_SCFG1        = set_scanblk_SCFG1,
    .get_scanblk_SCFG2        = get_scanblk_SCFG2,        .set_scanblk_SCFG2        = set_scanblk_SCFG2,
    .get_scanblk_SCFG3        = get_scanblk_SCFG3,        .set_scanblk_SCFG3        = set_scanblk_SCFG3,
    .get_scanblk_SCTRL        = get_scanblk_SCTRL,        .set_scanblk_SCTRL        = set_scanblk_SCTRL_thread,
                                                          .set_scanblk_SCTRL_isr    = set_scanblk_SCTRL_isr,
    .get_scanblk_SCMD         = get_scanblk_SCMD,         .set_scanblk_SCMD         = set_scanblk_SCMD_thread,
                                                          .set_scanblk_SCMD_isr     = set_scanblk_SCMD_isr,
    .get_scanblk_SSTAT        = get_scanblk_SSTAT,        /* read only -- no setter */
    /* write only -- no getter */                         .set_scanblk_SIACK        = set_scanblk_SIACK,
    .get_scanblk_SCANX        = get_scanblk_SCANX,        .set_scanblk_SCANX        = set_scanblk_SCANX,
    .get_scanblk_SCYCLE       = get_scanblk_SCYCLE,       .set_scanblk_SCYCLE       = set_scanblk_SCYCLE,
    .get_scanblk_STCFG1       = get_scanblk_STCFG1,       .set_scanblk_STCFG1       = set_scanblk_STCFG1,
    .get_scanblk_STCFG2       = get_scanblk_STCFG2,       .set_scanblk_STCFG2       = set_scanblk_STCFG2,
    .get_scanblk_STCFG3       = get_scanblk_STCFG3,       .set_scanblk_STCFG3       = set_scanblk_STCFG3,
    .get_scanblk_STCFG4       = get_scanblk_STCFG4,       .set_scanblk_STCFG4       = set_scanblk_STCFG4,
    .get_scanblk_CCFG         = get_scanblk_CCFG,         .set_scanblk_CCFG         = set_scanblk_CCFG,
    .get_scanblk_SCLK1        = get_scanblk_SCLK1,        .set_scanblk_SCLK1        = set_scanblk_SCLK1,
    .get_scanblk_SCLK2        = get_scanblk_SCLK2,        .set_scanblk_SCLK2        = set_scanblk_SCLK2,
    .get_scanblk_SCLK3        = get_scanblk_SCLK3,        .set_scanblk_SCLK3        = set_scanblk_SCLK3,
    .get_scanblk_SCLK4        = get_scanblk_SCLK4,        .set_scanblk_SCLK4        = set_scanblk_SCLK4,
    .get_scanblk_SCLK5        = get_scanblk_SCLK5,        .set_scanblk_SCLK5        = set_scanblk_SCLK5,
    .get_scanblk_SCLK6        = get_scanblk_SCLK6,        .set_scanblk_SCLK6        = set_scanblk_SCLK6,
    .get_scanblk_ACLK1        = get_scanblk_ACLK1,        .set_scanblk_ACLK1        = set_scanblk_ACLK1,
    .get_scanblk_ACLK2        = get_scanblk_ACLK2,        .set_scanblk_ACLK2        = set_scanblk_ACLK2,
    .get_scanblk_ACLK3        = get_scanblk_ACLK3,        .set_scanblk_ACLK3        = set_scanblk_ACLK3,
    .get_scanblk_ACLK4        = get_scanblk_ACLK4,        .set_scanblk_ACLK4        = set_scanblk_ACLK4,
    .get_scanblk_ACLK5        = get_scanblk_ACLK5,        .set_scanblk_ACLK5        = set_scanblk_ACLK5,
    .get_scanblk_ACLK6        = get_scanblk_ACLK6,        .set_scanblk_ACLK6        = set_scanblk_ACLK6,
    .get_scanblk_LDATA10      = get_scanblk_LDATA10,      .set_scanblk_LDATA10      = set_scanblk_LDATA10,
    .get_scanblk_LDATA11      = get_scanblk_LDATA11,      .set_scanblk_LDATA11      = set_scanblk_LDATA11,
    .get_scanblk_LDATA20      = get_scanblk_LDATA20,      .set_scanblk_LDATA20      = set_scanblk_LDATA20,
    .get_scanblk_LDATA21      = get_scanblk_LDATA21,      .set_scanblk_LDATA21      = set_scanblk_LDATA21,
    .get_scanblk_LDATA30      = get_scanblk_LDATA30,      .set_scanblk_LDATA30      = set_scanblk_LDATA30,
    .get_scanblk_LDATA31      = get_scanblk_LDATA31,      .set_scanblk_LDATA31      = set_scanblk_LDATA31,
    .get_scanblk_LDATA40      = get_scanblk_LDATA40,      .set_scanblk_LDATA40      = set_scanblk_LDATA40,
    .get_scanblk_LDATA41      = get_scanblk_LDATA41,      .set_scanblk_LDATA41      = set_scanblk_LDATA41,
    .get_scanblk_LDATA50      = get_scanblk_LDATA50,      .set_scanblk_LDATA50      = set_scanblk_LDATA50,
    .get_scanblk_LDATA51      = get_scanblk_LDATA51,      .set_scanblk_LDATA51      = set_scanblk_LDATA51,
    .get_scanblk_LDATA60      = get_scanblk_LDATA60,      .set_scanblk_LDATA60      = set_scanblk_LDATA60,
    .get_scanblk_LDATA61      = get_scanblk_LDATA61,      .set_scanblk_LDATA61      = set_scanblk_LDATA61,
    .get_scanblk_PSEQ1        = get_scanblk_PSEQ1,        .set_scanblk_PSEQ1        = set_scanblk_PSEQ1,
    .get_scanblk_PSEQ2        = get_scanblk_PSEQ2,        .set_scanblk_PSEQ2        = set_scanblk_PSEQ2,
    .get_scanblk_BPWM         = get_scanblk_BPWM,         .set_scanblk_BPWM         = set_scanblk_BPWM,
    .get_scanblk_MCFG         = get_scanblk_MCFG,         .set_scanblk_MCFG         = set_scanblk_MCFG,
    .get_scanblk_MOTORSYNCMIN = get_scanblk_MOTORSYNCMIN, .set_scanblk_MOTORSYNCMIN = set_scanblk_MOTORSYNCMIN,
    .get_scanblk_AFEPC        = get_scanblk_AFEPC,        .set_scanblk_AFEPC        = set_scanblk_AFEPC,
    .get_scanblk_CFGARB       = get_scanblk_CFGARB,       .set_scanblk_CFGARB       = set_scanblk_CFGARB,
    .get_scanblk_LED00        = get_scanblk_LED00,        .set_scanblk_LED00        = set_scanblk_LED00,
    .get_scanblk_LED01        = get_scanblk_LED01,        .set_scanblk_LED01        = set_scanblk_LED01,
    .get_scanblk_LED10        = get_scanblk_LED10,        .set_scanblk_LED10        = set_scanblk_LED10,
    .get_scanblk_LED11        = get_scanblk_LED11,        .set_scanblk_LED11        = set_scanblk_LED11,
    .get_scanblk_LED20        = get_scanblk_LED20,        .set_scanblk_LED20        = set_scanblk_LED20,
    .get_scanblk_LED21        = get_scanblk_LED21,        .set_scanblk_LED21        = set_scanblk_LED21,
    .get_scanblk_LPWMCONFIG0  = get_scanblk_LPWMCONFIG0,  .set_scanblk_LPWMCONFIG0  = set_scanblk_LPWMCONFIG0,
    .get_scanblk_LPWMCONFIG1  = get_scanblk_LPWMCONFIG1,  .set_scanblk_LPWMCONFIG1  = set_scanblk_LPWMCONFIG1,
    .get_scanblk_LPWM00       = get_scanblk_LPWM00,       .set_scanblk_LPWM00       = set_scanblk_LPWM00,
    .get_scanblk_LPWM01       = get_scanblk_LPWM01,       .set_scanblk_LPWM01       = set_scanblk_LPWM01,
    .get_scanblk_LPWM10       = get_scanblk_LPWM10,       .set_scanblk_LPWM10       = set_scanblk_LPWM10,
    .get_scanblk_LPWM11       = get_scanblk_LPWM11,       .set_scanblk_LPWM11       = set_scanblk_LPWM11,
    .get_scanblk_LPWM20       = get_scanblk_LPWM20,       .set_scanblk_LPWM20       = set_scanblk_LPWM20,
    .get_scanblk_LPWM21       = get_scanblk_LPWM21,       .set_scanblk_LPWM21       = set_scanblk_LPWM21,
    .get_scanblk_EXCFG        = get_scanblk_EXCFG,        .set_scanblk_EXCFG        = set_scanblk_EXCFG,
    .get_scanblk_OFIE         = get_scanblk_OFIE,         .set_scanblk_OFIE         = set_scanblk_OFIE,
    .get_scanblk_OFIP         = get_scanblk_OFIP,         /* read only -- no setter */
    /* write only -- no getter */                         .set_scanblk_OFIA         = set_scanblk_OFIA,
    /* write only -- no getter */                         .set_scanblk_OFIF         = set_scanblk_OFIF,
    .get_scanblk_REV0         = get_scanblk_REV0,         /* read only -- no setter */
    .get_scanblk_REV1         = get_scanblk_REV1,         /* read only -- no setter */
};

void scanblk_common_init(struct device_data *device_data)
{
    device_data->fcn_tbl = &scanblk_functions;
    register_scanblk_subblock(scanblk_common, device_data);
    spin_lock_init(&(device_data->int_spinlock));
    spin_lock_init(&(device_data->reg_spinlock));
    device_data->interrupt_callback = NULL;
}
EXPORT_SYMBOL(scanblk_common_init);

void scanblk_common_exit(struct device_data *device_data)
{
    unregister_scanblk_subblock(scanblk_common);
}
EXPORT_SYMBOL(scanblk_common_exit);

/*************************************
 * scanblk_platform_irq: interrupt handler
 * @irq:  irq number
 * @pdev: interrupt source
 *
 * This function returns IRQ_HANDLED if the IRQ has been handled
 * This is an ISR don't trace, use attribute interface instead
 **************************************/
///  NOTE - This function runs in interrupt context - no long operations allowed
irqreturn_t scanblk_platform_irq(int irq, void *pdev)
{
    struct scanblk_irqs interrupts;
    struct device_data *device_data = pdev;

    //debug_print("%s:%d reached\n", __FUNCTION__, __LINE__);
    memset(&interrupts, 0x0, sizeof(interrupts));

    clear_scan_irqs(device_data, &interrupts);   
    device_data->interrupt_callback(&interrupts);

    return IRQ_HANDLED;
}

