/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/of_irq.h>
#include <linux/of_device.h>
#include <linux/of_address.h>
#include <linux/dma-mapping.h>

#include "scanblk_if.h"
#include "scanblk_data.h"
#include "scanblk_common_pltfm.h"

struct device *global_dev;

// structure to store state of block during suspend/resume
struct device_suspend_context_struct
{
    int placeholder;  // what needs to be stored in suspend for this block?  FIXME TODO
};

extern void scanblk_dump_regs(void);
// functions only used for debug and bringup - not needed in normal driver use
static ssize_t debug_test_write1(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
    struct device_data *device_data;
    device_data = dev_get_drvdata(dev);
    scanblk_dump_regs();
    return count;
}
DEVICE_ATTR(dumpregs, S_IWUSR | S_IRUGO, NULL, debug_test_write1);

extern void scanblk_register_override(char * regname, uint32_t value);
static ssize_t debug_register_override(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
    char reg_name[32];
    char val_buf[32];
    struct device_data *device_data;
    int ret, i, j;
    unsigned long val;

    device_data = dev_get_drvdata(dev);

    memset(reg_name, 0x0, sizeof(reg_name));
    memset(val_buf, 0x0, sizeof(val_buf));

    /* Grab the register to set, stick it in reg_name */
    for(i=0; i < (sizeof(reg_name)-1) && i < count; i++) {
        if(buf[i] == ' ') {
            break;
        }
        reg_name[i] = buf[i];
    }

    /* Hopefully we have a space here to grab the value string */
    if(buf[i] == ' ') {
        while(buf[i] == ' ') {
            i++;
        }
        for(j = 0; j < (sizeof(val_buf)-1) && i < count; j++, i++) {
            val_buf[j] = buf[i];
        }

        ret = kstrtoul(val_buf, 0, &val);
        if(ret == 0) {
            scanblk_register_override(reg_name, val);
        }
    }
    else {
        debug_print("Did you forget the arg? write in the format of \"reg_name 0x1234\"\n");
    }

    return count;
}
DEVICE_ATTR(ovr, S_IWUSR | S_IRUGO, NULL, debug_register_override);

static const struct attribute *scanblk_attrs[] = {
    &dev_attr_dumpregs.attr,
    &dev_attr_ovr.attr,
    NULL,
};

// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
static struct device_data *init_subblock(char *subblock_name,
                                         void (*init_func)(struct device_data *),
                                         void (*exit_func)(struct device_data *),
                                         int irq, uint32_t clock_speed,
                                         struct device_node *parent_node,
                                         struct list_head *device_data_list)
{
    struct resource res;
    struct device_node *child_node;
    struct device_data *device_data;
    int retval;

    // now this is for each child
    child_node = of_get_child_by_name(parent_node, subblock_name);
    if (child_node == NULL)
    {
        debug_print("no child by name of %s\n", subblock_name);
        return NULL; // no child by that name, just return
    }

    retval = of_address_to_resource(child_node, 0, &res);
    if (retval < 0)
    {
        dev_err(global_dev, "%s can't get resource for node, returned 0x%X\n", subblock_name, retval);
        goto early_failure;
    }

    device_data = (struct device_data *)kzalloc(sizeof(struct device_data), GFP_KERNEL);
    if (!device_data)
    {
        dev_err(global_dev, "kzalloc failed\n");
        goto early_failure;
    }

    if (!request_mem_region(res.start, resource_size(&res), subblock_name))
    {
        dev_err(global_dev, "%s request_mem_region failed: %s\n", subblock_name, __func__);
        goto request_mem_region_failed;
    }

    device_data->virt_addr = of_iomap(child_node, 0); // get virtual address
    if (device_data->virt_addr == NULL)
    {
        dev_err(global_dev, "%s of_iomap failed: %s\n", subblock_name, __func__);
        device_data->virt_addr = 0;
        goto iomap_failed;
    }

    device_data->phys_addr   = res.start;
    device_data->addr_size   = resource_size(&res);
    device_data->irq         = irq;
    device_data->exit_func   = exit_func;
    device_data->clock_speed = clock_speed;

    // if we are a subblock that has its own IRQ (the calling function knows this),
    // then map the interrupt service routine
    if (irq > -1)
    {
        printk("%s: irq=%d, IRQF_ONESHOT=%d\n", __func__, irq, IRQF_ONESHOT);
        //retval = request_threaded_irq(irq, NULL,
        //                              scanblk_platform_irq,
        //                              IRQF_ONESHOT, "SCANBLK", device_data);
        retval = request_irq(irq, scanblk_platform_irq,
                             IRQF_SHARED, "SCANBLK", device_data);
        if (retval)
        {
            dev_err(global_dev, "request_irq failed. ret=%d\n", retval);
            goto request_irq_failed;
        }
    }

    // add the device_data to the linked list
    list_add((struct list_head *)&device_data->linked_list, device_data_list);

    init_func(device_data);

    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return device_data;

request_irq_failed:
    device_data->irq = 0;

    iounmap(device_data->virt_addr);
iomap_failed:
    release_mem_region(device_data->phys_addr, device_data->addr_size);
request_mem_region_failed:
    kfree(device_data);
early_failure:
    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return NULL;
}


static LIST_HEAD(device_data_list);

static int scanblk_platform_probe(struct platform_device *pdev)
{
    int retval, irq, ignore;
    struct device_data *device_data;
    struct device_node *parent_node = pdev->dev.of_node;
    uint32_t clock_speed;

    printk("Entered %s\n", __func__);

    // initialize the head ptr for the whole linked list of device_data
    global_dev = &pdev->dev; // for error print statements (dev_err)

    if (!parent_node)
    {
        dev_err(global_dev, "Could not find device node %s\n", "scan");
        return -ENODEV;
    }

    // get the irq from the parent node (scan has 1 interrupt)
    irq = irq_of_parse_and_map(parent_node, 0);
    if (irq < 0)
    {
        dev_err(global_dev, "platform_get_irq failed\n");
        retval = -ENXIO;
        goto early_failure;
    }
    debug_print("Probe in %s: irq = %d\n", __func__, irq);

    retval = of_property_read_u32(parent_node, "clock-frequency", &clock_speed);
    if(retval != 0) {
        dev_err(global_dev, "%s: Could not find 'clock-frequency' in device tree!\n", __func__);
    }
    printk("%s: clock_speed=%d\n", __func__, clock_speed);

    // scan has many subblocks - parse them all out of the device tree, allocate device_data
    // for each of them, put that data on the device_data_list, and call their
    // respective init functions in the main driver code

    device_data = init_subblock("scan-core", scanblk_common_init, scanblk_common_exit, irq,
                                clock_speed, parent_node, &device_data_list);

    platform_set_drvdata(pdev, &device_data_list); // store our data in the device struct

    // for debugging and bringup to read registers- not needed for everyday driver work
    ignore = sysfs_create_files(&(pdev->dev.kobj), scanblk_attrs);

    return 0;

early_failure:
    return retval;
}


static int scanblk_platform_remove(struct platform_device *pdev)
{
    // get linked list of device_datas
    struct list_head *device_data_list_ptr = platform_get_drvdata(pdev); 
    struct device_data *device_data, *dd_next; // dd_next is used by list function/macro

    // ballen TODO -- this doesn't seem to be iterating across anything. Something's wrong.
    list_for_each_entry_safe(device_data, dd_next, device_data_list_ptr, linked_list)
    {
        // items were added to the list LIFO, and we remove FIFO
        list_del(&device_data->linked_list);
        device_data->exit_func(device_data);  // call exit function for driver subblock
        iounmap(device_data->virt_addr);
        release_mem_region(device_data->phys_addr, device_data->addr_size);
        if (device_data->irq > 0)
        {
            free_irq(device_data->irq, device_data);
        }
        kfree(device_data);
    }
    // tell scanblk.c we're closing down now.  It would be called once for each scanblk instance... if we had more than 1
    scanblk_exit();

    //kfree(device_data_list_ptr); Can't free this? I'm confused if it's statically allocated. Macros....
    
    platform_set_drvdata(pdev, NULL);
    sysfs_remove_files(&(pdev->dev.kobj), scanblk_attrs);
    debug_print("%s: complete\n",__func__);
    return 0;
}

static int scanblk_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    if(device_data == NULL)
    {
        debug_print(KERN_ERR "%s: device_data is null! Aborting!\n", __func__);
        return -1;
    }

    debug_print(KERN_WARNING "%s event = %d\n", __func__, state.event);

    // allocate suspend context structure
    device_data->device_suspend_context = kmalloc(sizeof(struct device_suspend_context_struct), GFP_KERNEL);
    if (device_data->device_suspend_context == NULL)
    {
        return -1;
    }

    return (0);
}

static int scanblk_platform_resume(struct platform_device *pdev)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    debug_print(KERN_WARNING "%s\n",__func__);

    if (device_data->device_suspend_context == NULL)
    {
        debug_print(KERN_ERR "%s: invalid suspend context\n",__func__);
        return -1;
    }

    // restore block context here TODO FIXME what to do?

    // free the suspend context
    kfree(device_data->device_suspend_context);
    device_data->device_suspend_context = NULL;
    return(0);
}

#ifdef CONFIG_OF
static const struct of_device_id mrvl_scanblk_dt_match[] = {
    { .compatible = "ipg2scan"},
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_scanblk_dt_match);
#endif

static struct platform_driver scanblk_platform_driver =
{
    .probe   = scanblk_platform_probe,
    .remove  = scanblk_platform_remove,
    .suspend = scanblk_platform_suspend,
    .resume  = scanblk_platform_resume,
    .driver  = {
        .name           = "scanblk_block",
        .owner          = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_scanblk_dt_match),
    }
};

module_platform_driver(scanblk_platform_driver);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell scanblk subblock driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_May_15");

