/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/export.h>  // for EXPORT_SYMBOL
#include <linux/slab.h>    // for memory allocation

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for pie_full_subblock_list enum
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pogo_dma_driver.h"
#include "pogo_dma.h"
#include "pie_driver.h"
#include "pogo_dma_if.h"
#include "pie_driver_odma.h"
#include "pie_depogoizer_if.h"
#include "pie_pogoizer_if.h"
#include "pie_otpogoizer_if.h"
#include "pie_otdepogoizer_if.h"
#include "pie_pogo_idma_if.h"
#include "pie_pogo_odma_if.h"
#include "pie_pogo_otidma_if.h"
#include "pie_pogo_otodma_if.h"

#include "pie_driverlib_if.h"  // for print macro

// NOTE:  when creating new subblocks for pogo, each must register with us
// using register_pogo_device_data()

// note that this code is dependent on the pogo_subblocks enum maintaining the order
// of multiple instances of the type.  e.g. pogo_idma_udma is followed
// by pogo_idma_core after a space for the number of dma instances

static enum pogo_subblocks get_actual_subblock(enum pogo_subblocks subblock, int instance)
{
    enum pogo_subblocks actual_subblock;

    actual_subblock = subblock;
    if (subblock == pogo_idma_udma)
    {
        if ((instance < 0) || (instance > 3))
        {
            error_print(KERN_ERR "ERROR: instance %d illegal for pogo_idma_udma\n",instance);
            return pogo_idma_udma;
        }
        actual_subblock = pogo_idma_udma + instance;
    }
    else if (subblock == pogo_idma_core)
    {
        if ((instance < 0) || (instance > 3))
        {
            error_print(KERN_ERR "ERROR: instance %d illegal for pogo_idma_core\n",instance);
            return pogo_idma_core;
        }
        actual_subblock = pogo_idma_core + instance;
    }
    return actual_subblock;
}

#define CHECK4NULL(pointer)                                             \
    if (pointer == NULL)                                                \
    {                                                                   \
        print("ERROR %s, pointer %s is NULL\n", __func__,               \
              #pointer);                                                \
        BUG();                                                          \
    }

// return the private data and the function table for the requested pogo_subblock
#define POGO_RETRIEVE_function_table_data(pogo_subblock, ft, device_data_requested, pogo_handle, instance) \
    actual_subblock = get_actual_subblock(pogo_subblock, instance);     \
    CHECK4NULL(pogo_object->priv_data[actual_subblock]);                \
    pogo_handle = pogo_object->priv_data[actual_subblock];              \
    device_data_requested = pogo_handle->device_data;                   \
    ft = pogo_handle->fcn_tbl;                                          \
    BUG_ON(device_data_requested == NULL);                              \
    BUG_ON(ft == NULL);

#define PIE_POGO_RETRIEVE_function_table_data(pogo_subblock, function_struct, pogo_data, handle) \
    pogo_data *device_data;                                             \
    struct function_struct *ft;                                         \
    handle *pogo_handle;                                                \
    enum pogo_subblocks actual_subblock;                                \
                                                                        \
    POGO_RETRIEVE_function_table_data(pogo_subblock, ft, device_data, pogo_handle, 0);

#define PIE_IDMA_RETRIEVE_function_table_data(instance)                 \
    struct pie_pogo_idma_udma_function_struct *ft_iu;                   \
    struct pie_pogo_idma_core_function_struct *ft_ic;                   \
    piePogoIDMAData *device_data_iu = NULL;                             \
    piePogoIDMAData *device_data_ic = NULL;                             \
    enum pogo_subblocks actual_subblock;                                \
    piePogoIDMAUDMADeviceHandle *pogo_udma_handle;                      \
    piePogoIDMACoreDeviceHandle *pogo_core_handle;                      \
                                                                        \
    POGO_RETRIEVE_function_table_data(pogo_idma_udma, ft_iu, device_data_iu, pogo_udma_handle, instance); \
    POGO_RETRIEVE_function_table_data(pogo_idma_core, ft_ic, device_data_ic, pogo_core_handle, instance);


#define PIE_ODMA_RETRIEVE_function_table_data                           \
    struct pie_pogo_odma_udma_function_struct *ft_ou;                   \
    struct pie_pogo_odma_core_function_struct *ft_oc;                   \
    piePogoODMAData *device_data_ou = NULL;                             \
    piePogoODMAData *device_data_oc = NULL;                             \
    enum pogo_subblocks actual_subblock;                                \
    piePogoODMAUDMADeviceHandle *pogo_udma_handle;                      \
    piePogoODMACoreDeviceHandle *pogo_core_handle;                      \
                                                                        \
    POGO_RETRIEVE_function_table_data(pogo_odma_udma, ft_ou, device_data_ou, pogo_udma_handle, 0); \
    POGO_RETRIEVE_function_table_data(pogo_odma_core, ft_oc, device_data_oc, pogo_core_handle, 0);

#define PIE_OT_IDMA_RETRIEVE_function_table_data                        \
    struct pie_pogo_otidma_udma_function_struct *ft_iu;                 \
    struct pie_pogo_otidma_core_function_struct *ft_ic;                 \
    piePogoOTIDMAData *device_data_iu = NULL;                           \
    piePogoOTIDMAData *device_data_ic = NULL;                           \
    enum pogo_subblocks actual_subblock;                                \
    piePogoOTIDMAUDMADeviceHandle *pogo_otudma_handle;                  \
    piePogoOTIDMACoreDeviceHandle *pogo_otcore_handle;                  \
                                                                        \
    POGO_RETRIEVE_function_table_data(pogo_ot_idma_udma, ft_iu, device_data_iu, pogo_otudma_handle, 0); \
    POGO_RETRIEVE_function_table_data(pogo_ot_idma_core, ft_ic, device_data_ic, pogo_otcore_handle, 0);

#define PIE_OT_ODMA_RETRIEVE_function_table_data                        \
    struct pie_pogo_otodma_udma_function_struct *ft_ou;                 \
    struct pie_pogo_otodma_core_function_struct *ft_oc;                 \
    piePogoOTODMAData *device_data_ou = NULL;                           \
    piePogoOTODMAData *device_data_oc = NULL;                           \
    enum pogo_subblocks actual_subblock;                                \
    piePogoOTODMAUDMADeviceHandle *pogo_otudma_handle;                  \
    piePogoOTODMACoreDeviceHandle *pogo_otcore_handle;                  \
                                                                        \
    POGO_RETRIEVE_function_table_data(pogo_ot_odma_udma, ft_ou, device_data_ou, pogo_otudma_handle, 0); \
    POGO_RETRIEVE_function_table_data(pogo_ot_odma_core, ft_oc, device_data_oc, pogo_otcore_handle, 0);

void register_pogo_subblock(enum pogo_subblocks subblock, void *pogo_subblock_handle, int instance)
{
    pogoDeviceHandle *pogo_device;
    enum pogo_subblocks actual_subblock;

    actual_subblock = get_actual_subblock(subblock, instance);

    pogo_device = get_pogo_device();

    pogo_device->device_data->priv_data[actual_subblock] = pogo_subblock_handle;

    if (actual_subblock == pogo_ot_idma_core)
        debug_print("ot_idma_core, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);
        
    if (actual_subblock == pogo_ot_idma_udma)
        debug_print("ot_idma_udma, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);

    if (actual_subblock == pogo_ot_odma_core)
        debug_print("ot_odma_core, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);
        
    if (actual_subblock == pogo_ot_odma_udma)
        debug_print("ot_odma_udma, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);
    
    if (actual_subblock == pogo_odma_core)
        debug_print("odma_core, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);
    if (actual_subblock == pogo_odma_udma)
        debug_print("odma_udma, device_data=0x%p, stored 0x%p at 0x%p\n",
                    pogo_device->device_data, pogo_device->device_data->priv_data[actual_subblock],
                    &pogo_device->device_data->priv_data[actual_subblock]);

    debug_print("Block %d registered with %s\n",actual_subblock, __func__);
}
EXPORT_SYMBOL(register_pogo_subblock);

void *unregister_pogo_subblock(enum pogo_subblocks subblock, int instance)
{
    enum pogo_subblocks actual_subblock;
    pogoDeviceHandle *pogo_device;
    void *pogo_subblock_handle;

    actual_subblock = get_actual_subblock(subblock, instance);

    pogo_device = get_pogo_device();

    pogo_subblock_handle = pogo_device->device_data->priv_data[actual_subblock];
    pogo_device->device_data->priv_data[actual_subblock] = NULL;

    debug_print("Block %d unregistered with %s\n",actual_subblock, __func__);
    return pogo_subblock_handle;
}
EXPORT_SYMBOL(unregister_pogo_subblock);



/////////////////////////////////
// for debug - dump registers
/////////////////////////////////

void dump_pogo_ot_idma_only_regs(piePogoData *pogo_object)
{
    PIE_OT_IDMA_RETRIEVE_function_table_data;
   
    ft_iu->dump_pogo_otidma_udma_regs(device_data_iu);
    ft_ic->dump_pogo_otidma_core_regs(device_data_ic);
    
}

void dump_pogo_idma_only_regs(piePogoData *pogo_object, uint32_t channel_mask)
{
    int i;
    
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        if (channel_mask & 0x01)
        {
            PIE_IDMA_RETRIEVE_function_table_data(i);
            
            ft_iu->dump_pogo_idma_udma_regs(device_data_iu);
            ft_ic->dump_pogo_idma_core_regs(device_data_ic);
        }
        channel_mask = channel_mask >> 1;
    }
}

void dump_pogo_idma_regs(piePogoData *pogo_object, uint32_t channel_mask)
{
    dump_pogo_idma_only_regs(pogo_object, channel_mask);
    dump_pogo_ot_idma_only_regs(pogo_object);
}

void dump_pogo_odma_only_regs(piePogoData *pogo_object)
{
    PIE_ODMA_RETRIEVE_function_table_data;
   
    ft_ou->dump_pogo_odma_udma_regs(device_data_ou);
    ft_oc->dump_pogo_odma_core_regs(device_data_oc);
    
}

void dump_pogo_ot_odma_only_regs(piePogoData *pogo_object)
{
    PIE_OT_ODMA_RETRIEVE_function_table_data;
   
    ft_ou->dump_pogo_otodma_udma_regs(device_data_ou);
    ft_oc->dump_pogo_otodma_core_regs(device_data_oc);
    
}

void dump_pogo_odma_regs(piePogoData *pogo_object)
{
    dump_pogo_odma_only_regs(pogo_object);
    dump_pogo_ot_odma_only_regs(pogo_object);
}

void dump_pogoizer_regs(piePogoData *pogo_object)
{
    PIE_POGO_RETRIEVE_function_table_data(pogoizer, pie_pogoizer_function_struct,
                                          piePogoizerData, piePogoizerDeviceHandle);

    ft->dump_pie_pogoizer_regs(device_data);
}

void dump_otpogoizer_regs(piePogoData *pogo_object)
{
    PIE_POGO_RETRIEVE_function_table_data(otpogoizer, pie_otpogoizer_function_struct,
                                          pieOTPogoizerData, pieOTPogoizerDeviceHandle);

    ft->dump_pie_otpogoizer_regs(device_data);
}

void dump_pogo_regs(piePogoData *pogo_object)
{
    dump_pogoizer_regs(pogo_object);
    dump_otpogoizer_regs(pogo_object);
}

void dump_otdepogoizer_regs(piePogoData *pogo_object)
{
    PIE_POGO_RETRIEVE_function_table_data(otdepogoizer, pie_otdepogoizer_function_struct,
                                          pieOTDepogoizerData, pieOTDepogoizerDeviceHandle);

    ft->dump_pie_otdepogoizer_regs(device_data);
}

void dump_depogoizer_regs(piePogoData *pogo_object)
{
    PIE_POGO_RETRIEVE_function_table_data(depogoizer, pie_depogoizer_function_struct,
                                          pieDepogoizerData, pieDepogoizerDeviceHandle);

    ft->dump_pie_depogoizer_regs(device_data);
}

void dump_depogo_regs(piePogoData *pogo_object)
{
    dump_depogoizer_regs(pogo_object);
    dump_otdepogoizer_regs(pogo_object);
}

void pogo_dma_dump_all_regs(piePogoData *pogo_object)
{
    dump_pogo_regs(pogo_object);
    dump_depogo_regs(pogo_object);

    dump_pogo_odma_regs(pogo_object);
    dump_pogo_idma_regs(pogo_object, ALL_PIE_POGO_IDMA_CHANNELS);
}


///  NOTE - This function runs in interrupt context - no long operations allowed
void handle_pogo_odma_only_irqs(piePogoData *pogo_object)
{
    PIE_ODMA_RETRIEVE_function_table_data;

    // handle any pie ODMA interrupts
    ft_ou->handle_pogo_odma_udma_irqs(device_data_ou);
    ft_oc->handle_pogo_odma_core_irqs(device_data_oc);
}

///  NOTE - This function runs in interrupt context - no long operations allowed
void handle_pogo_ot_odma_only_irqs(piePogoData *pogo_object)
{
    PIE_OT_ODMA_RETRIEVE_function_table_data;

    // handle any pie ODMA interrupts
    ft_ou->handle_pogo_otodma_udma_irqs(device_data_ou);
    ft_oc->handle_pogo_otodma_core_irqs(device_data_oc);
}

///  NOTE - This function runs in interrupt context - no long operations allowed
void handle_pogo_odma_irqs(piePogoData *pogo_object)
{
    handle_pogo_odma_only_irqs(pogo_object);
    handle_pogo_ot_odma_only_irqs(pogo_object);
}

void handle_pogo_ot_idma_only_irqs(piePogoData *pogo_object)
{
    PIE_OT_IDMA_RETRIEVE_function_table_data;

    // handle andy pie IDMA interrupts
    ft_iu->handle_pogo_otidma_udma_irqs(device_data_iu);
    ft_ic->handle_pogo_otidma_core_irqs(device_data_ic);    
}

void handle_pogo_idma_only_irqs(piePogoData *pogo_object)
{
    int channel;

    // handle IDMA ints for each channel
    for (channel=0;channel<NUM_PIE_POGO_IDMA_CHANNELS;channel++)
    {
        PIE_IDMA_RETRIEVE_function_table_data(channel)
        
        ft_iu->handle_pogo_idma_udma_irqs(device_data_iu);
        ft_ic->handle_pogo_idma_core_irqs(device_data_ic);
    }
}

///  NOTE - This function runs in interrupt context - no long operations allowed
void handle_pogo_idma_irqs(piePogoData *pogo_object)
{
    handle_pogo_idma_only_irqs(pogo_object);
    handle_pogo_ot_idma_only_irqs(pogo_object);
}

void clear_pogo_odma_only_irqs(piePogoData *pogo_object)
{
    PIE_ODMA_RETRIEVE_function_table_data;
    
    ft_ou->clear_pogo_odma_udma_irqs(device_data_ou, NULL);
    ft_oc->clear_pogo_odma_core_irqs(device_data_oc, NULL);
}

void clear_pogo_ot_odma_only_irqs(piePogoData *pogo_object)
{
    PIE_OT_ODMA_RETRIEVE_function_table_data;
    
    ft_ou->clear_pogo_otodma_udma_irqs(device_data_ou, NULL);
    ft_oc->clear_pogo_otodma_core_irqs(device_data_oc, NULL);
}

void clear_pogo_odma_irqs(piePogoData *pogo_object)
{
    clear_pogo_odma_only_irqs(pogo_object);
    clear_pogo_ot_odma_only_irqs(pogo_object);    
}

void clear_pogo_ot_idma_only_irqs(piePogoData *pogo_object)
{
    PIE_OT_IDMA_RETRIEVE_function_table_data;
    
    ft_iu->clear_pogo_otidma_udma_irqs(device_data_iu, NULL);
    ft_ic->clear_pogo_otidma_core_irqs(device_data_ic, NULL);
}

void clear_pogo_idma_only_irqs(piePogoData *pogo_object)
{
    int channel;
    
    // clear ints
    for (channel=0;channel<NUM_PIE_POGO_IDMA_CHANNELS;channel++)
    {
        PIE_IDMA_RETRIEVE_function_table_data(channel);
        // IDMA
        ft_iu->clear_pogo_idma_udma_irqs(device_data_iu, NULL);
        ft_ic->clear_pogo_idma_core_irqs(device_data_ic, NULL);
    }
}

void clear_pogo_idma_irqs(piePogoData *pogo_object)
{
    clear_pogo_idma_only_irqs(pogo_object);
    clear_pogo_ot_idma_only_irqs(pogo_object);    
}

bool idma_enabled(piePogoData *pogo_object, uint8_t channum)
{
    PIE_IDMA_RETRIEVE_function_table_data(channum);

    return(ft_iu->udma_enabled(device_data_iu));
}

bool odma_enabled(piePogoData *pogo_object)
{
    PIE_ODMA_RETRIEVE_function_table_data;

    return(ft_ou->udma_enabled(device_data_ou));
}

bool idma_idle(piePogoData *pogo_object, uint8_t channum)
{
    PIE_IDMA_RETRIEVE_function_table_data(channum);

    return(ft_iu->udma_idle(device_data_iu));
}

bool odma_idle(piePogoData *pogo_object)
{
    PIE_ODMA_RETRIEVE_function_table_data;

    return(ft_ou->udma_idle(device_data_ou));
}


// caller passes in the pogo_fmt_type and an array of
// physical pointers to each of the descriptors for the idmas
void start_pie_pogo_idma(piePogoData *pogo_object, dma_addr_t phys_desc,
                         uint8_t channum)
{
    PIE_IDMA_RETRIEVE_function_table_data(channum);
    
    debug_print("%s: Before starting IDMA\n",__func__);

    // do we need to do alignment checks here?  32byte aligned for best speed
        
    // start the dma
    ft_iu->start_pogo_idma_udma(device_data_iu, (uint32_t *)phys_desc);
    debug_print("started IDMA %d at descriptor phys addr 0x%X\n", channum, phys_desc);

}

void start_pie_pogo_odma(piePogoData *pogo_object, dma_addr_t phys_desc)
{
    PIE_ODMA_RETRIEVE_function_table_data;

#ifdef DURING_DEBUG
    // dump all the regs and descriptors
    //dump_pogo_odma_regs(pogo_object);
    debug_print("starting ODMA\n");    
#endif
    
    // do we need to do alignment checks here?  32byte aligned for best speed
    ft_ou->start_pogo_odma_udma(device_data_ou, (uint32_t *) phys_desc);
    
#ifdef DURING_DEBUG
    debug_print("Output ODMA just started at physaddress 0x%X\n", phys_desc);
    //dump_pogo_odma_regs(pogo_object);
#endif
}

void start_pie_pogo_ot_idma(piePogoData *pogo_object, dma_addr_t phys_desc)
{
    PIE_OT_IDMA_RETRIEVE_function_table_data;

#ifdef DURING_DEBUG
    // dump all the regs and descriptors
    //dump_pogo_idma_regs(pogo_object);
    debug_print("starting OT IDMA\n");    
#endif
    
    // do we need to do alignment checks here?  32byte aligned for best speed
    ft_iu->start_pogo_otidma_udma(device_data_iu, (uint32_t *) phys_desc);
    
#ifdef DURING_DEBUG
    debug_print("Output OT IDMA just started at physaddress 0x%X\n", phys_desc);
    //dump_pogo_idma_regs(pogo_object);
#endif
}


void start_pie_pogo_ot_odma(piePogoData *pogo_object, dma_addr_t phys_desc)
{
    PIE_OT_ODMA_RETRIEVE_function_table_data;

#ifdef DURING_DEBUG
    // dump all the regs and descriptors
    //dump_pogo_odma_regs(pogo_object);
    debug_print("starting OT ODMA\n");    
#endif
    
    // do we need to do alignment checks here?  32byte aligned for best speed
    ft_ou->start_pogo_otodma_udma(device_data_ou, (uint32_t *) phys_desc);
    
#ifdef DURING_DEBUG
    debug_print("Output OT DMA just started at physaddress 0x%X\n", phys_desc);
    //dump_pogo_odma_regs(pogo_object);
#endif
}

static int pie_revcheck(piePogoData *pogo_object,
                        struct pie_handle_t *pie_handle)
{
    int i;
    int retval = 0;

    if (pogo_object->priv_data[pogoizer] == NULL)
    {
        return -1; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(pogoizer, pie_pogoizer_function_struct,
                                              piePogoizerData, piePogoizerDeviceHandle);

        retval = ft->pie_revcheck(device_data, pie_handle);
        if (retval != 0)
            return retval;
    }
    if (pogo_object->priv_data[depogoizer] == NULL)
    {
        return -1; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(depogoizer, pie_depogoizer_function_struct,
                                              pieDepogoizerData, pieDepogoizerDeviceHandle);
        
        retval = ft->pie_revcheck(device_data, pie_handle);
        if (retval != 0)
            return retval;
    }
    if (pogo_object->priv_data[otpogoizer] == NULL)
    {
        return -1; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otpogoizer, pie_otpogoizer_function_struct,
                                              pieOTPogoizerData, pieOTPogoizerDeviceHandle);

        retval = ft->pie_revcheck(device_data, pie_handle);
        if (retval != 0)
            return retval;
    }
    if (pogo_object->priv_data[otdepogoizer] == NULL)
    {
        return -1; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otdepogoizer, pie_otdepogoizer_function_struct,
                                              pieOTDepogoizerData, pieOTDepogoizerDeviceHandle);

        retval = ft->pie_revcheck(device_data, pie_handle);
        if (retval != 0)
            return retval;
    }
    
    // revision check pogo odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_odma_core] == NULL))
    {
        return -1; // objects don't exist
    }
    else
    {
        PIE_ODMA_RETRIEVE_function_table_data;

        retval = ft_ou->pie_revcheck(device_data_ou, pie_handle);
        if (retval != 0)
            return retval;
        
        retval = ft_oc->pie_revcheck(device_data_oc, pie_handle);
        if (retval != 0)
            return retval;
    }
    // revision check pogo idma subblocks (udma and core for all channels)
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        if ((pogo_object->priv_data[pogo_idma_udma+i] == NULL) ||
            (pogo_object->priv_data[pogo_idma_core+i] == NULL))
        {
            continue; // objects for channel i don't exist
        }
        else
        {
            PIE_IDMA_RETRIEVE_function_table_data(i);
            
            retval = ft_iu->pie_revcheck(device_data_iu, pie_handle, i);
            if (retval != 0)
                return retval;
            retval = ft_ic->pie_revcheck(device_data_ic, pie_handle, i);
            if (retval != 0)
                return retval;
        }
    }
    // revision check pogo OT idma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_idma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_idma_core] == NULL))
    {
        return -1; // objects don't exist
    }
    else
    {
        PIE_OT_IDMA_RETRIEVE_function_table_data;

        retval = ft_iu->pie_revcheck(device_data_iu, pie_handle);
        if (retval != 0)
            return retval;
        
        retval = ft_ic->pie_revcheck(device_data_ic, pie_handle);
        if (retval != 0)
            return retval;
    }

    // revision check pogo OT odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_odma_core] == NULL))
    {
        return -1; // objects don't exist
    }
    else
    {
        PIE_OT_ODMA_RETRIEVE_function_table_data;

        retval = ft_ou->pie_revcheck(device_data_ou, pie_handle);
        if (retval != 0)
            return retval;
        
        retval = ft_oc->pie_revcheck(device_data_oc, pie_handle);
        if (retval != 0)
            return retval;
    }
    
    return retval;
}

static void pie_reset(piePogoData *pogo_object)
{
    int i;

    if (pogo_object->priv_data[pogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(pogoizer, pie_pogoizer_function_struct,
                                              piePogoizerData, piePogoizerDeviceHandle);
        ft->pie_reset(device_data);
    }
    if (pogo_object->priv_data[depogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(depogoizer, pie_depogoizer_function_struct,
                                              pieDepogoizerData, pieDepogoizerDeviceHandle);
        
        ft->pie_reset(device_data);
    }
    if (pogo_object->priv_data[otpogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otpogoizer, pie_otpogoizer_function_struct,
                                              pieOTPogoizerData, pieOTPogoizerDeviceHandle);
        ft->pie_reset(device_data);
    }
    if (pogo_object->priv_data[otdepogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otdepogoizer, pie_otdepogoizer_function_struct,
                                              pieOTDepogoizerData, pieOTDepogoizerDeviceHandle);
        
        ft->pie_reset(device_data);
    }

    // reset pogo odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_ODMA_RETRIEVE_function_table_data;

        ft_ou->pie_reset(device_data_ou);
        ft_oc->pie_reset(device_data_oc);
    }
    // reset pogo idma subblocks (udma and core for all channels)
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        if ((pogo_object->priv_data[pogo_idma_udma+i] == NULL) ||
            (pogo_object->priv_data[pogo_idma_core+i] == NULL))
        {
            continue; // objects for channel i don't exist
        }
        else
        {
            PIE_IDMA_RETRIEVE_function_table_data(i);
            
            ft_iu->pie_reset(device_data_iu);
            ft_ic->pie_reset(device_data_ic);
        }
    }

    // reset pogo OT idma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_idma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_idma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_IDMA_RETRIEVE_function_table_data;

        ft_iu->pie_reset(device_data_iu);
        ft_ic->pie_reset(device_data_ic);
    }

    // reset pogo OT odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_ODMA_RETRIEVE_function_table_data;

        ft_ou->pie_reset(device_data_ou);
        ft_oc->pie_reset(device_data_oc);
    }
    
}


void pie_configure(piePogoData *pogo_object, struct pie_handle_t *pie_handle)
{
    int i;

    // configure pogo
    if (pogo_object->priv_data[pogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(pogoizer, pie_pogoizer_function_struct,
                                              piePogoizerData, piePogoizerDeviceHandle);
        
        ft->pie_configure(device_data, pie_handle);
    }
    // configure depogo
    if (pogo_object->priv_data[depogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(depogoizer, pie_depogoizer_function_struct,
                                              pieDepogoizerData, pieDepogoizerDeviceHandle);
        
        ft->pie_configure(device_data, pie_handle);
    }
    // configure OT pogo
    if (pogo_object->priv_data[otpogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otpogoizer, pie_otpogoizer_function_struct,
                                              pieOTPogoizerData, pieOTPogoizerDeviceHandle);
        
        ft->pie_configure(device_data, pie_handle);
    }
    // configure OT depogo
    if (pogo_object->priv_data[otdepogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otdepogoizer, pie_otdepogoizer_function_struct,
                                              pieOTDepogoizerData, pieOTDepogoizerDeviceHandle);
        
        ft->pie_configure(device_data, pie_handle);
    }
    // configure pogo odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_ODMA_RETRIEVE_function_table_data;

        ft_oc->pie_configure(device_data_oc, pie_handle);
        ft_ou->pie_configure(device_data_ou, pie_handle);
    }
    // configure pogo idma subblocks (udma and core for all channels)
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        if ((pogo_object->priv_data[pogo_idma_udma+i] == NULL) ||
            (pogo_object->priv_data[pogo_idma_core+i] == NULL))
        {
            continue; // objects for channel i don't exist
        }
        else
        {
            PIE_IDMA_RETRIEVE_function_table_data(i);

            ft_ic->pie_configure(device_data_ic, pie_handle, i);
            ft_iu->pie_configure(device_data_iu, pie_handle, i);
        }
    }

    // configure pogo OT idma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_idma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_idma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_IDMA_RETRIEVE_function_table_data;

        ft_ic->pie_configure(device_data_ic, pie_handle);
        ft_iu->pie_configure(device_data_iu, pie_handle);
    }

    // configure pogo OT odma subblocks (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_ODMA_RETRIEVE_function_table_data;

        ft_oc->pie_configure(device_data_oc, pie_handle);
        ft_ou->pie_configure(device_data_ou, pie_handle);
    }
    
}

static void pie_get_current(piePogoData *pogo_object, struct pie_handle_t *pie_handle)
{
    int i;

    // get current pogoizer regs
    if (pogo_object->priv_data[pogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(pogoizer, pie_pogoizer_function_struct,
                                              piePogoizerData, piePogoizerDeviceHandle);
        
        ft->pie_get_current(device_data, pie_handle);
    }
    // get_current depogo regs
    if (pogo_object->priv_data[depogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(depogoizer, pie_depogoizer_function_struct,
                                              pieDepogoizerData, pieDepogoizerDeviceHandle);
        
        ft->pie_get_current(device_data, pie_handle);
    }
    // get current OT pogoizer regs
    if (pogo_object->priv_data[otpogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otpogoizer, pie_otpogoizer_function_struct,
                                              pieOTPogoizerData, pieOTPogoizerDeviceHandle);
        
        ft->pie_get_current(device_data, pie_handle);
    }
    // get_current OT depogo regs
    if (pogo_object->priv_data[otdepogoizer] == NULL)
    {
        return; // object doesn't exist
    }
    else
    {
        PIE_POGO_RETRIEVE_function_table_data(otdepogoizer, pie_otdepogoizer_function_struct,
                                              pieOTDepogoizerData, pieOTDepogoizerDeviceHandle);
        
        ft->pie_get_current(device_data, pie_handle);
    }
    // get_current pogo odma regs (both udma and core)
    if ((pogo_object->priv_data[pogo_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_ODMA_RETRIEVE_function_table_data;

        ft_oc->pie_get_current(device_data_oc, pie_handle);
        ft_ou->pie_get_current(device_data_ou, pie_handle);
    }
    // get_current pogo idma regs (udma and core for all channels)
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        if ((pogo_object->priv_data[pogo_idma_udma+i] == NULL) ||
            (pogo_object->priv_data[pogo_idma_core+i] == NULL))
        {
            continue; // objects for channel i don't exist
        }
        else
        {
            PIE_IDMA_RETRIEVE_function_table_data(i);
            
            ft_ic->pie_get_current(device_data_ic, pie_handle, i);
            ft_iu->pie_get_current(device_data_iu, pie_handle, i);
        }
    }
    
    // get_current pogo OT idma regs (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_idma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_idma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_IDMA_RETRIEVE_function_table_data;

        ft_ic->pie_get_current(device_data_ic, pie_handle);
        ft_iu->pie_get_current(device_data_iu, pie_handle);
    }
    
    // get_current pogo OT odma regs (both udma and core)
    if ((pogo_object->priv_data[pogo_ot_odma_udma] == NULL) ||
        (pogo_object->priv_data[pogo_ot_odma_core] == NULL))
    {
        return; // objects don't exist
    }
    else
    {
        PIE_OT_ODMA_RETRIEVE_function_table_data;

        ft_oc->pie_get_current(device_data_oc, pie_handle);
        ft_ou->pie_get_current(device_data_ou, pie_handle);
    }
    
}

void pie_get_pogo_sizes_array(uint32_t *subblock_array, 
                              void *input_object)
{
    enum pie_full_subblock_list i;
    int x;
    piePogoData *pogo_object;
    
    pogo_object = (piePogoData *) input_object;
    i = pogoizer_index;
    if (pogo_object->priv_data[pogoizer] != NULL)
    {
        subblock_array[i] = sizeof(POGOIZER_REGS_t);
    }

    i = depogoizer_index;
    if (pogo_object->priv_data[depogoizer] != NULL)
    {
        subblock_array[i] = sizeof(DEPOGOIZER_REGS_t);
    }
    i = otpogoizer_index;
    if (pogo_object->priv_data[otpogoizer] != NULL)
    {
        subblock_array[i] = sizeof(OTPOGOIZER_REGS_t);
    }

    i = otdepogoizer_index;
    if (pogo_object->priv_data[otdepogoizer] != NULL)
    {
        subblock_array[i] = sizeof(OTDEPOGOIZER_REGS_t);
    }

    i = pogo_odma_udma_index;
    if (pogo_object->priv_data[pogo_odma_udma] != NULL)
    {
        subblock_array[i] = sizeof(POGO_ODMA_POGO_ODMA_UDMA_REGS_t);
    }

    i = pogo_odma_core_index;
    if (pogo_object->priv_data[pogo_odma_core] != NULL)
    {
        subblock_array[i] = sizeof(POGO_ODMA_POGO_ODMA_CORE_REGS_t);
    }

    // make sure the pogo idma indices are set up as we expected
    if (((pogo_idma0_udma_index + 1) != pogo_idma1_udma_index) ||
        ((pogo_idma0_udma_index + NUM_PIE_POGO_IDMA_CHANNELS) != pogo_idma0_core_index))
    {
        error_print("%s: pogo idma index in pie_full_subblock_list is incorrect!!!\n", __func__);
        BUG();
    }
    
    for (x=0;x<NUM_PIE_POGO_IDMA_CHANNELS;x++)
    {
        i = pogo_idma0_udma_index + x;
        if (pogo_object->priv_data[pogo_idma_udma+x] != NULL)
            subblock_array[i] = sizeof(POGO_IDMA0_POGO_IDMA_UDMA_REGS_t);
        
        i = pogo_idma0_core_index + x;
        if (pogo_object->priv_data[pogo_idma_core+x] != NULL)
            subblock_array[i] = sizeof(POGO_IDMA0_POGO_IDMA_CORE_REGS_t);
    }
    // now the OT idmas
    i = ot_pogo_idma_core_index;
    if (pogo_object->priv_data[pogo_ot_idma_core] != NULL)
    {
        subblock_array[i] = sizeof(OT_POGO_IDMA_POGO_IDMA_CORE_REGS_t);
    }
    i = ot_pogo_idma_udma_index;
    if (pogo_object->priv_data[pogo_ot_idma_udma] != NULL)
    {
        subblock_array[i] = sizeof(OT_POGO_IDMA_POGO_IDMA_UDMA_REGS_t);
    }
    // now the OT odmas
    i = pogo_otodma_core_index;
    if (pogo_object->priv_data[pogo_ot_odma_core] != NULL)
    {
        subblock_array[i] = sizeof(POGO_OTODMA_POGO_ODMA_CORE_REGS_t);
    }
    i = pogo_otodma_udma_index;
    if (pogo_object->priv_data[pogo_ot_odma_udma] != NULL)
    {
        subblock_array[i] = sizeof(POGO_OTODMA_POGO_ODMA_UDMA_REGS_t);
    }
}


struct pie_pogo_dma_function_struct pie_pogo_dma_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,

    .pogo_dma_dump_all_regs = pogo_dma_dump_all_regs,
    
    .handle_pogo_odma_irqs = handle_pogo_odma_irqs,
    .handle_pogo_idma_irqs = handle_pogo_idma_irqs,

    .clear_pogo_odma_irqs = clear_pogo_odma_irqs,
    .clear_pogo_idma_irqs = clear_pogo_idma_irqs,

    .idma_idle = idma_idle,
    .odma_idle = odma_idle,
    .idma_enabled = idma_enabled,
    .odma_enabled = odma_enabled,
    .start_pie_pogo_idma = start_pie_pogo_idma,
    .start_pie_pogo_odma = start_pie_pogo_odma,

    .start_pie_pogo_ot_idma = start_pie_pogo_ot_idma,
    .start_pie_pogo_ot_odma = start_pie_pogo_ot_odma,
    
    .pie_get_pogo_sizes_array = pie_get_pogo_sizes_array,
};

int pogo_dma_init(void)
{
    pogoDeviceHandle *pogo_handle; 
    int retval = 0;
    int i;

    pogo_handle = allocate_memory(sizeof(pogoDeviceHandle), GFP_KERNEL);
    pogo_handle->device_data = allocate_memory(sizeof(piePogoData), GFP_KERNEL);    
    pogo_handle->fcn_tbl = &pie_pogo_dma_functions;
    // NULL out the pointers in the device data array so I know who has properly registered
    for (i=0;i<max_pogo_subblocks;i++)
        pogo_handle->device_data->priv_data[i] = NULL;
    
    register_pie_subblock(pogo_dma, pogo_handle);
    return retval;
}

void pogo_dma_exit(void)
{
    pogoDeviceHandle *pogo_handle;
    
    pogo_handle = unregister_pie_subblock(pogo_dma);
    free_memory(pogo_handle->device_data);
    free_memory(pogo_handle);
}
