/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_ycc2rgb.h"
#include "pie_ycc2rgb_if.h"

#include "pie_driverlib_if.h"  // for print macros


void dump_ycc2rgb_regs(ycc2rgbData *device_data)
{
    print("PIE YCC2RGB Registers\n");
    print("CCR=0x%X\n", ycc2rgbRead(CCR));
    print("REV0=0x%X\n", ycc2rgbRead(REV0));
    print("REV1=0x%X\n", ycc2rgbRead(REV1));
    print("MCR0=0x%X\n", ycc2rgbRead(MCR0));
    print("MCR1=0x%X\n", ycc2rgbRead(MCR1));
    print("MCR2=0x%X\n", ycc2rgbRead(MCR2));
    print("MCR3=0x%X\n", ycc2rgbRead(MCR3));
    print("MCR4=0x%X\n", ycc2rgbRead(MCR4));
    print("MCR5=0x%X\n", ycc2rgbRead(MCR5));
    print("MCR6=0x%X\n", ycc2rgbRead(MCR6));
    print("MCR7=0x%X\n", ycc2rgbRead(MCR7));    
    print("MCR8=0x%X\n", ycc2rgbRead(MCR8));
}

static int pie_revcheck(ycc2rgbData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = Y2ES_REV0_MAJ_MASK_SHIFT(ycc2rgbRead(REV0));
    pieh_rev = Y2ES_REV0_MAJ_MASK_SHIFT(pie_handle->pie_ycc2rgb->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        
        return -1;
    }
    else
        return 0;
}

static void pie_reset(ycc2rgbData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = ycc2rgbRead(CCR);
    reg = Y2ES_CCR_BYPASSALL_REPLACE_VAL(reg, 1);  // bypass to do a reset
    ycc2rgbWrite(CCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(ycc2rgbData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for ycc2rgb

    PROTECT_REG_ACCESS;
    // disable (bypass) the block
    ycc2rgbWrite(CCR, (pie_handle->pie_ycc2rgb->CCR & PIE_CSC_BYPASS));
    // write the CCR (contains enable bit) last
    // REV0 and REV1 are Read only
    ycc2rgbWrite(MCR0, pie_handle->pie_ycc2rgb->MCR0);
    ycc2rgbWrite(MCR1, pie_handle->pie_ycc2rgb->MCR1);
    ycc2rgbWrite(MCR2, pie_handle->pie_ycc2rgb->MCR2);
    ycc2rgbWrite(MCR3, pie_handle->pie_ycc2rgb->MCR3);
    ycc2rgbWrite(MCR4, pie_handle->pie_ycc2rgb->MCR4);
    ycc2rgbWrite(MCR5, pie_handle->pie_ycc2rgb->MCR5);
    ycc2rgbWrite(MCR6, pie_handle->pie_ycc2rgb->MCR6);
    ycc2rgbWrite(MCR7, pie_handle->pie_ycc2rgb->MCR7);
    ycc2rgbWrite(MCR8, pie_handle->pie_ycc2rgb->MCR8);
    // now write the CCR (with the enable unmasked)
    ycc2rgbWrite(CCR, pie_handle->pie_ycc2rgb->CCR);
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(ycc2rgbData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_ycc2rgb->CCR  = ycc2rgbRead(CCR);
    pie_handle->pie_ycc2rgb->REV0 = ycc2rgbRead(REV0);
    pie_handle->pie_ycc2rgb->REV1 = ycc2rgbRead(REV1);
    pie_handle->pie_ycc2rgb->MCR0 = ycc2rgbRead(MCR0);
    pie_handle->pie_ycc2rgb->MCR1 = ycc2rgbRead(MCR1);
    pie_handle->pie_ycc2rgb->MCR2 = ycc2rgbRead(MCR2);
    pie_handle->pie_ycc2rgb->MCR3 = ycc2rgbRead(MCR3);
    pie_handle->pie_ycc2rgb->MCR4 = ycc2rgbRead(MCR4);
    pie_handle->pie_ycc2rgb->MCR5 = ycc2rgbRead(MCR5);
    pie_handle->pie_ycc2rgb->MCR6 = ycc2rgbRead(MCR6);
    pie_handle->pie_ycc2rgb->MCR7 = ycc2rgbRead(MCR7);
    pie_handle->pie_ycc2rgb->MCR8 = ycc2rgbRead(MCR8);
}

struct pie_ycc2rgb_function_struct pie_ycc2rgb_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_ycc2rgb_regs  = dump_ycc2rgb_regs,
};

void pie_ycc2rgb_init(ycc2rgbData *device_data)
{
    ycc2rgbDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(ycc2rgbDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_ycc2rgb_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(ycc2rgb, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // ycc2rgb has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE YCC2RGB LOADED .....\n");
}
EXPORT_SYMBOL(pie_ycc2rgb_init);


void pie_ycc2rgb_exit(ycc2rgbData *device_data)
{
    ycc2rgbDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(ycc2rgb);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_ycc2rgb_exit);

