/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pie_xyscaler.h"
#include "pie_xyscaler_if.h"

#include "pie_driverlib_if.h"  // for print macros


void dump_xyscaler_regs(xyscaleData *device_data)
{
    print("PIE XYSCALER Registers\n");
    print("CSR=0x%X\n",  xyscalerRead(CSR));
    print("CCR=0x%X\n",  xyscalerRead(CCR));
    print("CFR1=0x%X\n", xyscalerRead(CFR1));
    print("CFR2=0x%X\n", xyscalerRead(CFR2));
    print("HSR=0x%X\n",  xyscalerRead(HSR));
    print("VSR=0x%X\n",  xyscalerRead(VSR));
    print("HSMR=0x%X\n", xyscalerRead(HSMR));
    print("VSMR=0x%X\n", xyscalerRead(VSMR));
    print("CXR=0x%X\n",  xyscalerRead(CXR));
    print("CYR=0x%X\n",  xyscalerRead(CYR));
    print("HCPR=0x%X\n", xyscalerRead(HCPR));
    print("VCPR=0x%X\n", xyscalerRead(VCPR));
    print("XOR=0x%X\n",  xyscalerRead(XOR));
    print("REV0=0x%X\n", xyscalerRead(REV0));
    print("REV1=0x%X\n", xyscalerRead(REV1));
}

static int pie_revcheck(xyscaleData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0 = XYSCALE_REV0_MAJ_MASK_SHIFT(xyscalerRead(REV0));
    pieh_rev = XYSCALE_REV0_MAJ_MASK_SHIFT(pie_handle->pie_xyscale->REV0);

    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(xyscaleData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = xyscalerRead(CSR);
    reg = XYSCALE_CSR_B_REPLACE_VAL(reg, 1);  // bypass to do a reset
    xyscalerWrite(CSR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(xyscaleData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for xyscaler

    // NOTE: we need to put the xyscale block into bypass before
    // writing any of the other R/W xyscale registers. Additionally,
    // we need to make sure the rest of the CSR bits are set before
    // the bypass is cleared.  According to the ASIC doc:

    //    Therefore, when rounding, two sequential writes must be
    //    written to the Configuration Register, one with the bits set
    //    and bypass bit 0 set as well, the second with bypass reset
    //    to zero to start functional operation.

    // To take care of that, we write the CSR register bits while
    // forcing the block into bypass at the start of the function.
    // Then we write all of the other registers.  Then we do the 2nd
    // write to the CSR with the original value (if the user requested
    // bypass, we stay in bypass, but if the user wanted it enabled,
    // it would happen now)

    // Also, it is necessary to write the write-only registers *after*
    // the block is enabled (out of bypass).  So the final order is:
    // 1) write the CSR register with bypass forced to 1
    // 2) write the R/W registers
    // 3) write the CSR register normally (we assume the user turned off bypass)
    // 4) write the Write-only registers.
    
    PROTECT_REG_ACCESS;
    xyscalerWrite(CSR, pie_handle->pie_xyscale->CSR | XYSCALE_CSR_B_MASK); // force block into bypass

    xyscalerWrite(CCR, pie_handle->pie_xyscale->CCR);
    xyscalerWrite(CFR1, pie_handle->pie_xyscale->CFR1);
    xyscalerWrite(CFR2, pie_handle->pie_xyscale->CFR2);
    xyscalerWrite(HSR, pie_handle->pie_xyscale->HSR);
    xyscalerWrite(VSR, pie_handle->pie_xyscale->VSR);
    xyscalerWrite(HSMR, pie_handle->pie_xyscale->HSMR);
    xyscalerWrite(VSMR, pie_handle->pie_xyscale->VSMR);
    // CXR and CYR are Read only
    // HCPR, VCPR, XOR, REV0, and REV1 are Read only

    // write CSR last since it can take the block out of bypass
    xyscalerWrite(CSR, pie_handle->pie_xyscale->CSR);

    // these write only registers must only be written when we are enabled
    xyscalerWrite(VSPR, pie_handle->pie_xyscale->VSPR);
    xyscalerWrite(XTR, pie_handle->pie_xyscale->XTR);
    xyscalerWrite(YTR, pie_handle->pie_xyscale->YTR);
    
    UNPROTECT_REG_ACCESS;

}

static void pie_get_current(xyscaleData *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t leftovers = pie_handle->pie_xyscale_partial_pixel_pos;

    pie_handle->pie_xyscale->CSR =    xyscalerRead(CSR);
    pie_handle->pie_xyscale->CCR =    xyscalerRead(CCR);
    pie_handle->pie_xyscale->CFR1 =    xyscalerRead(CFR1);
    pie_handle->pie_xyscale->CFR2 =    xyscalerRead(CFR2);
    pie_handle->pie_xyscale->HSR =    xyscalerRead(HSR);
    pie_handle->pie_xyscale->VSR =    xyscalerRead(VSR);
    pie_handle->pie_xyscale->HSMR =    xyscalerRead(HSMR);
    pie_handle->pie_xyscale->VSMR =    xyscalerRead(VSMR);
    // XTR and YTR are Write only
    pie_handle->pie_xyscale->CXR =    xyscalerRead(CXR);
    pie_handle->pie_xyscale->CYR =    xyscalerRead(CYR);
    // VSPR is Write only
    pie_handle->pie_xyscale->HCPR =    xyscalerRead(HCPR);
    pie_handle->pie_xyscale->VCPR =    xyscalerRead(VCPR);
    pie_handle->pie_xyscale->XOR =    xyscalerRead(XOR);
    pie_handle->pie_xyscale->REV0 =    xyscalerRead(REV0);
    pie_handle->pie_xyscale->REV1 =    xyscalerRead(REV1);

    if (XYSCALE_CSR_LM_MASK_SHIFT(pie_handle->pie_xyscale->CSR))
    {
        if (XYSCALE_CYR_DY_MASK_SHIFT(leftovers) == leftovers)
        {
            pie_handle->pie_xyscale->CYR = leftovers;
        }
    }
    else
    {
        if (XYSCALE_VCPR_P_MASK_SHIFT(leftovers) == leftovers)
        {
            pie_handle->pie_xyscale->VCPR = leftovers;
        }
    }
    pie_handle->pie_xyscale_partial_pixel_pos = 0xFFFFFFFF;
}

struct pie_xyscaler_function_struct pie_xyscaler_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,
    
    .dump_xyscaler_regs  = dump_xyscaler_regs,
};

void pie_xyscaler_init(xyscaleData *device_data)
{
    xyscaleDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(xyscaleDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_xyscaler_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(xyscaler, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // pd has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("XYSCALER_INIT LOADED.........\n");
}
EXPORT_SYMBOL(pie_xyscaler_init);


void pie_xyscaler_exit(xyscaleData *device_data)
{
    xyscaleDeviceHandle *pie_device_handle;
    
    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(xyscaler);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_xyscaler_exit);
