/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"  
#include "pie_driver.h"
#include "pie_xycscaler.h"
#include "pie_xycscaler_if.h"

#include "pie_driverlib_if.h"   // for print macro

void dump_xycscaler_regs(xy_c_scaleData *device_data)
{
    print("PIE XYCSCALER Registers\n");
    print("CSR=0x%X\n",  xycscalerRead(CSR));
    print("CLCR=0x%X\n",  xycscalerRead(CLCR));
    print("CCR=0x%X\n",  xycscalerRead(CCR));
    print("CFR1=0x%X\n",  xycscalerRead(CFR1));
    print("CFR2=0x%X\n",  xycscalerRead(CFR2));
    print("HSR=0x%X\n",  xycscalerRead(HSR));
    print("VSR=0x%X\n",  xycscalerRead(VSR));
    print("CXR=0x%X\n",  xycscalerRead(CXR));
    print("CYR=0x%X\n",  xycscalerRead(CYR));
    print("XOR=0x%X\n",  xycscalerRead(XOR));
    print("SR=0x%X\n",  xycscalerRead(SR));
    print("REV0=0x%X\n",  xycscalerRead(REV0));
    print("REV1=0x%X\n",  xycscalerRead(REV1));
}

static int pie_revcheck(xy_c_scaleData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = XYCSCALE_REV0_MAJ_MASK_SHIFT(xycscalerRead(REV0));
    pieh_rev = XYCSCALE_REV0_MAJ_MASK_SHIFT(pie_handle->pie_xy_c_scale->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(xy_c_scaleData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = xycscalerRead(CSR);
    reg = XYCSCALE_CSR_B_REPLACE_VAL(reg, 1);  // bypass to do a reset
    xycscalerWrite(CSR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(xy_c_scaleData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for xy_c_scaler

    // NOTE: we need to put the xycscale block into bypass before
    // writing any of the other R/W xycscale registers. Additionally,
    // we need to make sure the rest of the CSR bits are set before
    // the bypass is cleared. 

    // To take care of that, we write the CSR register bits while
    // forcing the block into bypass at the start of the function.
    // Then we write all of the other registers.  Then we do the 2nd
    // write to the CSR with the original value (if the user requested
    // bypass, we stay in bypass, but if the user wanted it enabled,
    // it would happen now)

    // Also, it is necessary to write the write-only registers *after*
    // the block is enabled (out of bypass).  So the final order is:
    // 1) write the CSR register with bypass forced to 1
    // 2) write the R/W registers
    // 3) write the CSR register normally (we assume the user turned off bypass)
    // 4) write the Write-only registers.

    PROTECT_REG_ACCESS;
    xycscalerWrite(CSR, pie_handle->pie_xy_c_scale->CSR | XYCSCALE_CSR_B_MASK); // force block into bypass
    // CLCR is Read only
    xycscalerWrite(CCR, pie_handle->pie_xy_c_scale->CCR);
    xycscalerWrite(CFR1, pie_handle->pie_xy_c_scale->CFR1);
    xycscalerWrite(CFR2, pie_handle->pie_xy_c_scale->CFR2);
    xycscalerWrite(HSR, pie_handle->pie_xy_c_scale->HSR);
    xycscalerWrite(VSR, pie_handle->pie_xy_c_scale->VSR);
    // CXR, CYR, and XOR are Read only
    xycscalerWrite(SR, pie_handle->pie_xy_c_scale->SR);
    // REV0, and REV1 are Read only

    // write the config reg last as requested by user
    // (bypass needs to be written after R/W regs)
    // enable should be written here
    xycscalerWrite(CSR, pie_handle->pie_xy_c_scale->CSR);

    // these write only registers must only be written when we are enabled
    xycscalerWrite(XTR, pie_handle->pie_xy_c_scale->XTR);
    xycscalerWrite(YTR, pie_handle->pie_xy_c_scale->YTR);

    UNPROTECT_REG_ACCESS;
    

}

static void pie_get_current(xy_c_scaleData *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t leftovers = pie_handle->pie_xy_c_scale_partial_pixel_pos;

    pie_handle->pie_xy_c_scale->CSR = xycscalerRead(CSR);
    pie_handle->pie_xy_c_scale->CLCR = xycscalerRead(CLCR);
    pie_handle->pie_xy_c_scale->CCR = xycscalerRead(CCR);
    pie_handle->pie_xy_c_scale->CFR1 = xycscalerRead(CFR1);
    pie_handle->pie_xy_c_scale->CFR2 = xycscalerRead(CFR2);
    pie_handle->pie_xy_c_scale->HSR = xycscalerRead(HSR);
    pie_handle->pie_xy_c_scale->VSR = xycscalerRead(VSR);
    // XTR and YTR are Write only
    pie_handle->pie_xy_c_scale->CXR = xycscalerRead(CXR);
    pie_handle->pie_xy_c_scale->CYR = xycscalerRead(CYR);
    pie_handle->pie_xy_c_scale->XOR = xycscalerRead(XOR);
    pie_handle->pie_xy_c_scale->SR = xycscalerRead(SR);
    pie_handle->pie_xy_c_scale->REV0 = xycscalerRead(REV0);
    pie_handle->pie_xy_c_scale->REV1 = xycscalerRead(REV1);

    if (XYCSCALE_CYR_DY_MASK_SHIFT(leftovers) == leftovers)
    {
        pie_handle->pie_xy_c_scale->CYR = leftovers;
    }
    pie_handle->pie_xy_c_scale_partial_pixel_pos = 0xFFFFFFFF;
}

struct pie_xycscaler_function_struct pie_xycscaler_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,

    .dump_xycscaler_regs  = dump_xycscaler_regs,
};

void pie_xycscaler_init(xy_c_scaleData *device_data)
{
    xy_c_scaleDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(xy_c_scaleDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_xycscaler_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(xycscaler, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // pd has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("XYCSCALER IS LOADED.....\n");
}
EXPORT_SYMBOL(pie_xycscaler_init);


void pie_xycscaler_exit(xy_c_scaleData *device_data)
{
    xy_c_scaleDeviceHandle *pie_device_handle;
    
    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(xycscaler);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_xycscaler_exit);
