/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_tcns.h"
#include "pie_tcns_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_tcns_regs(tcnsData *device_data)
{
    print("PIE TCNS Registers\n");
    print("TCNSCR=0x%X\n", tcnsRead(TCNSCR));
    print("REV0=0x%X\n", tcnsRead(REV0));
    print("REV1=0x%X\n", tcnsRead(REV1));
}

// lut 1 is the main lookup table (tcnslut)
// lut 2 is the alternate look up table (tcnslut2)
void dump_tcns_luts(tcnsData *device_data, int lut_num)
{
    int i;
    uint32_t reg, reg_orig, reg_LUT;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg_orig = tcnsRead(TCNSCR);
    reg = TCNSENSE_TCNSCR_C_REPLACE_VAL(reg_orig, 1); // enable writing with 1
    tcnsWrite(TCNSCR, reg);

    if (lut_num == 1)
        print("PIE Main TCNS Y Cr Cb LUT entries");
    else if (lut_num == 2)
        print("PIE Alternate TCNS Y Cr Cb LUT entries");
    else
    {
        error_print("Error: only support for TCNSLUT and TCSNLUT2\n");
        return;
    }
        
    for (i=0;i<PIE_TCNS_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        if (lut_num == 1)
        {
            reg_LUT = tcnsReadArray(TCNSLUT, i);
            print("0x%X ", TCNSENSE_TCNSLUT_Y_MASK_SHIFT(reg_LUT));
            print("0x%X ", TCNSENSE_TCNSLUT_CR_MASK_SHIFT(reg_LUT));
            print("0x%X ", TCNSENSE_TCNSLUT_CB_MASK_SHIFT(reg_LUT));
        }
        else
        {
            reg_LUT = tcnsReadArray(TCNSLUT2, i);
            print("0x%X ", TCNSENSE_TCNSLUT2_Y_MASK_SHIFT(reg_LUT));
            print("0x%X ", TCNSENSE_TCNSLUT2_CR_MASK_SHIFT(reg_LUT));
            print("0x%X ", TCNSENSE_TCNSLUT2_CB_MASK_SHIFT(reg_LUT));
        }
    }

    // write back the original reg value for tcns config reg
    tcnsWrite(TCNSCR, reg_orig);
    UNPROTECT_REG_ACCESS;
}
    
static int pie_revcheck(tcnsData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = TCNSENSE_REV0_MAJ_MASK_SHIFT(tcnsRead(REV0));
    pieh_rev = TCNSENSE_REV0_MAJ_MASK_SHIFT(pie_handle->pie_tcns->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(tcnsData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = tcnsRead(TCNSCR);
    reg = TCNSENSE_TCNSCR_B_REPLACE_VAL(reg, PIE_CSC_BYPASS);  // bypass to do a reset
    tcnsWrite(TCNSCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(tcnsData *device_data,
                          struct pie_handle_t *pie_handle)
{
    int i;
    // There is no callback to configure for tcns
    
    PROTECT_REG_ACCESS;

    // NOTE: to program the LUTs the CPU access bit must
    // be set first.  Then all the LUTs can be written,
    // then the CPU access bit will be cleared.

    // set CPU access bit before programming LUTs
    tcnsWrite(TCNSCR, (pie_handle->pie_tcns->TCNSCR | TCNSENSE_TCNSCR_C_MASK));
    // REV0 and REV1 are Read only
    for (i=0;i<PIE_TCNS_LUT_ENTRIES;i++)
    {
        tcnsWriteArray(TCNSLUT, pie_handle->pie_tcns->TCNSLUT[i], i);
        tcnsWriteArray(TCNSLUT2, pie_handle->pie_tcns->TCNSLUT2[i], i);
    }
    
    // now write the TCNSCR (with the cpu_access normal)
    tcnsWrite(TCNSCR, pie_handle->pie_tcns->TCNSCR);
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(tcnsData *device_data,
                            struct pie_handle_t *pie_handle)
{
    int i;
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg = tcnsRead(TCNSCR);
    pie_handle->pie_tcns->TCNSCR    = reg;
        
    tcnsWrite(TCNSCR, (reg | TCNSENSE_TCNSCR_C_MASK));
    
    pie_handle->pie_tcns->REV0      = tcnsRead(REV0);              
    pie_handle->pie_tcns->REV1      = tcnsRead(REV1);              

    for (i=0;i<PIE_TCNS_LUT_ENTRIES;i++)
    {
        pie_handle->pie_tcns->TCNSLUT[i] = tcnsReadArray(TCNSLUT, i);
        pie_handle->pie_tcns->TCNSLUT2[i] = tcnsReadArray(TCNSLUT2, i);
    }
    // now write the TCNSCR (with the cpu_access normal)
    tcnsWrite(TCNSCR, reg);
    UNPROTECT_REG_ACCESS;
}

struct pie_tcns_function_struct pie_tcns_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_tcns_regs        = dump_tcns_regs,
    .dump_tcns_luts        = dump_tcns_luts,    
};

void pie_tcns_init(tcnsData *device_data)
{
    tcnsDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(tcnsDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_tcns_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(tcns, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // tcns has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE TCNS LOADED .....\n");
}
EXPORT_SYMBOL(pie_tcns_init);


void pie_tcns_exit(tcnsData *device_data)
{
    // unregister with the parent
    tcnsDeviceHandle *pie_device_handle;

    pie_device_handle = unregister_pie_subblock(tcns);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_tcns_exit);

