/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_rgb2ycc.h"
#include "pie_rgb2ycc_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_rgb2ycc_regs(rgb2yccData *device_data)
{
    print("PIE RGB2YCC Registers\n");
    print("CCR=0x%X\n", rgb2yccRead(CCR));
    print("REV0=0x%X\n", rgb2yccRead(REV0));
    print("REV1=0x%X\n", rgb2yccRead(REV1));
    print("MCR0=0x%X\n", rgb2yccRead(MCR0));
    print("MCR1=0x%X\n", rgb2yccRead(MCR1));
    print("MCR2=0x%X\n", rgb2yccRead(MCR2));
    print("MCR3=0x%X\n", rgb2yccRead(MCR3));
    print("MCR4=0x%X\n", rgb2yccRead(MCR4));
    print("MCR5=0x%X\n", rgb2yccRead(MCR5));
    print("MCR6=0x%X\n", rgb2yccRead(MCR6));
    print("MCR7=0x%X\n", rgb2yccRead(MCR7));    
    print("MCR8=0x%X\n", rgb2yccRead(MCR8));
}

static int pie_revcheck(rgb2yccData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = ES2Y_REV0_MAJ_MASK_SHIFT(rgb2yccRead(REV0));
    pieh_rev = ES2Y_REV0_MAJ_MASK_SHIFT(pie_handle->pie_rgb2ycc->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(rgb2yccData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = rgb2yccRead(CCR);
    reg = ES2Y_CCR_BYPASSALL_REPLACE_VAL(reg, 1);  // bypass to do a reset
    rgb2yccWrite(CCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(rgb2yccData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for rgb2ycc

    PROTECT_REG_ACCESS;
    // disable (bypass) the block
    rgb2yccWrite(CCR, (pie_handle->pie_rgb2ycc->CCR & PIE_CSC_BYPASS));
    // write the CCR (contains enable bit) last
    // REV0 and REV1 are Read only
    rgb2yccWrite(MCR0, pie_handle->pie_rgb2ycc->MCR0);
    rgb2yccWrite(MCR1, pie_handle->pie_rgb2ycc->MCR1);
    rgb2yccWrite(MCR2, pie_handle->pie_rgb2ycc->MCR2);
    rgb2yccWrite(MCR3, pie_handle->pie_rgb2ycc->MCR3);
    rgb2yccWrite(MCR4, pie_handle->pie_rgb2ycc->MCR4);
    rgb2yccWrite(MCR5, pie_handle->pie_rgb2ycc->MCR5);
    rgb2yccWrite(MCR6, pie_handle->pie_rgb2ycc->MCR6);
    rgb2yccWrite(MCR7, pie_handle->pie_rgb2ycc->MCR7);
    rgb2yccWrite(MCR8, pie_handle->pie_rgb2ycc->MCR8);
    // now write the CCR (with the enable unmasked)
    rgb2yccWrite(CCR, pie_handle->pie_rgb2ycc->CCR);
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(rgb2yccData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_rgb2ycc->CCR  = rgb2yccRead(CCR);
    pie_handle->pie_rgb2ycc->REV0 = rgb2yccRead(REV0);
    pie_handle->pie_rgb2ycc->REV1 = rgb2yccRead(REV1);
    pie_handle->pie_rgb2ycc->MCR0 = rgb2yccRead(MCR0);
    pie_handle->pie_rgb2ycc->MCR1 = rgb2yccRead(MCR1);
    pie_handle->pie_rgb2ycc->MCR2 = rgb2yccRead(MCR2);
    pie_handle->pie_rgb2ycc->MCR3 = rgb2yccRead(MCR3);
    pie_handle->pie_rgb2ycc->MCR4 = rgb2yccRead(MCR4);
    pie_handle->pie_rgb2ycc->MCR5 = rgb2yccRead(MCR5);
    pie_handle->pie_rgb2ycc->MCR6 = rgb2yccRead(MCR6);
    pie_handle->pie_rgb2ycc->MCR7 = rgb2yccRead(MCR7);
    pie_handle->pie_rgb2ycc->MCR8 = rgb2yccRead(MCR8);
}

struct pie_rgb2ycc_function_struct pie_rgb2ycc_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_rgb2ycc_regs  = dump_rgb2ycc_regs,
};

void pie_rgb2ycc_init(rgb2yccData *device_data)
{
    rgb2yccDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(rgb2yccDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_rgb2ycc_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(rgb2ycc, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // rgb2ycc has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE RGB2YCC LOADED .....\n");
}
EXPORT_SYMBOL(pie_rgb2ycc_init);


void pie_rgb2ycc_exit(rgb2yccData *device_data)
{
    rgb2yccDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(rgb2ycc);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_rgb2ycc_exit);

