/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_rgb2esrgb.h"
#include "pie_rgb2esrgb_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_rgb2esrgb_regs(rgb2esrgbData *device_data)
{
    print("PIE RGB2ESRGB Registers\n");
    print("REV0=0x%X\n", rgb2esrgbRead(REV0));
    print("REV1=0x%X\n", rgb2esrgbRead(REV1));
    print("LUT_CR=0x%X\n", rgb2esrgbRead(LUT_CR));
    print("EXTRAREG0=0x%X\n", rgb2esrgbRead(EXTRAREG0));
    print("EXTRAREG1=0x%X\n", rgb2esrgbRead(EXTRAREG1));
    print("EXTRAREG2=0x%X\n", rgb2esrgbRead(EXTRAREG2));    
}

void dump_rgb2esrgb_luts(rgb2esrgbData *device_data)
{
    int i;
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg = rgb2esrgbRead(LUT_CR);
    rgb2esrgbWrite(LUT_CR, (reg | LUT1D_RGB2ESRGB_LUT_CR_CPU_ACCESS_MASK));
    
    print("PIE RGB2ESRGB Red LUT entries");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);            
        print("0x%X ", rgb2esrgbReadArray(LUT0, i));
    }

    print("\nGreen LUT entries ");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        
        print("0x%X ", rgb2esrgbReadArray(LUT1, i));
    }

    print("\nBlue LUT entries  ");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        print("0x%X ", rgb2esrgbReadArray(LUT2, i));
    }
    rgb2esrgbWrite(LUT_CR, reg);
    UNPROTECT_REG_ACCESS;
}

static int pie_revcheck(rgb2esrgbData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = LUT1D_RGB2ESRGB_REV0_MAJ_MASK_SHIFT(rgb2esrgbRead(REV0));
    pieh_rev = LUT1D_RGB2ESRGB_REV0_MAJ_MASK_SHIFT(pie_handle->pie_rgb2esrgb->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(rgb2esrgbData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = rgb2esrgbRead(LUT_CR);
    reg = LUT1D_RGB2ESRGB_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(reg, PIE_CSC_BYPASS);  // bypass to do a reset
    rgb2esrgbWrite(LUT_CR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(rgb2esrgbData *device_data,
                          struct pie_handle_t *pie_handle)
{
    int i;
    // There is no callback to configure for rgb2esrgb
    
    PROTECT_REG_ACCESS;

    rgb2esrgbWrite(EXTRAREG0, pie_handle->pie_rgb2esrgb->EXTRAREG0);
    rgb2esrgbWrite(EXTRAREG1, pie_handle->pie_rgb2esrgb->EXTRAREG1);
    rgb2esrgbWrite(EXTRAREG2, pie_handle->pie_rgb2esrgb->EXTRAREG2);


    // NOTE: to program the LUTs the CPU access bit must
    // be set first.  Then all the LUTs can be written,
    // then the CPU access bit will be cleared.

    // set CPU access bit before programming LUTs
    rgb2esrgbWrite(LUT_CR, (pie_handle->pie_rgb2esrgb->LUT_CR | LUT1D_RGB2ESRGB_LUT_CR_CPU_ACCESS_MASK));
    // REV0 and REV1 are Read only
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        rgb2esrgbWriteArray(LUT0, pie_handle->pie_rgb2esrgb->LUT0[i], i);
        rgb2esrgbWriteArray(LUT1, pie_handle->pie_rgb2esrgb->LUT1[i], i);
        rgb2esrgbWriteArray(LUT2, pie_handle->pie_rgb2esrgb->LUT2[i], i);
    }
    
    // now write the LUT_CR (with the cpu_access normal)
    rgb2esrgbWrite(LUT_CR, pie_handle->pie_rgb2esrgb->LUT_CR);
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(rgb2esrgbData *device_data,
                            struct pie_handle_t *pie_handle)
{
    int i;
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg = rgb2esrgbRead(LUT_CR);
    pie_handle->pie_rgb2esrgb->LUT_CR    = reg;

    rgb2esrgbWrite(LUT_CR, (reg | LUT1D_RGB2ESRGB_LUT_CR_CPU_ACCESS_MASK));
    
    pie_handle->pie_rgb2esrgb->REV0      = rgb2esrgbRead(REV0);              
    pie_handle->pie_rgb2esrgb->REV1      = rgb2esrgbRead(REV1);              
    pie_handle->pie_rgb2esrgb->EXTRAREG0 = rgb2esrgbRead(EXTRAREG0);
    pie_handle->pie_rgb2esrgb->EXTRAREG1 = rgb2esrgbRead(EXTRAREG1);
    pie_handle->pie_rgb2esrgb->EXTRAREG2 = rgb2esrgbRead(EXTRAREG2);
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        pie_handle->pie_rgb2esrgb->LUT0[i] = rgb2esrgbReadArray(LUT0, i);
        pie_handle->pie_rgb2esrgb->LUT1[i] = rgb2esrgbReadArray(LUT1, i);
        pie_handle->pie_rgb2esrgb->LUT2[i] = rgb2esrgbReadArray(LUT2, i);
    }
    // now write the LUT_CR (with the cpu_access normal)
    rgb2esrgbWrite(LUT_CR, reg);
    UNPROTECT_REG_ACCESS;
}

struct pie_rgb2esrgb_function_struct pie_rgb2esrgb_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_rgb2esrgb_regs  = dump_rgb2esrgb_regs,
    .dump_rgb2esrgb_luts  = dump_rgb2esrgb_luts,
};

void pie_rgb2esrgb_init(rgb2esrgbData *device_data)
{
    rgb2esrgbDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(rgb2esrgbDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_rgb2esrgb_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(rgb2esrgb, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // rgb2esrgb has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE RGB2ESRGB LOADED .....\n");
}
EXPORT_SYMBOL(pie_rgb2esrgb_init);


void pie_rgb2esrgb_exit(rgb2esrgbData *device_data)
{
    rgb2esrgbDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(rgb2esrgb);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_rgb2esrgb_exit);

