/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>  // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>  // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t
#include <linux/delay.h>  // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pogo_dma_driver.h"
#include "pie_pogo_otodma.h"
#include "pie_pogo_otodma_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_pogo_otodma_udma_regs(struct device_data *device_data)
{
    print("Dump PIE POGO OT ODMA UDMA Registers\n");
    print("UCR=0x%X\n", piePogoOTODMAUDMARead(UCR));
    print("USR=0x%X\n", piePogoOTODMAUDMARead(USR));
    print("UPR=0x%X\n", piePogoOTODMAUDMARead(UPR));
    print("UIER=0x%X\n", piePogoOTODMAUDMARead(UIER));
    print("UIPR=0x%X\n", piePogoOTODMAUDMARead(UIPR));
    print("UDR=0x%X\n", piePogoOTODMAUDMARead(UDR));
    print("UBAR=0x%X\n", piePogoOTODMAUDMARead(UBAR));
    print("UBLR=0x%X\n", piePogoOTODMAUDMARead(UBLR));
    print("UBRR=0x%X\n", piePogoOTODMAUDMARead(UBRR));
    print("UTR0=0x%X\n", piePogoOTODMAUDMARead(UTR0));
    print("UTR1=0x%X\n", piePogoOTODMAUDMARead(UTR1));
}

void dump_pogo_otodma_core_regs(struct device_data *device_data)
{
    print("Dump PIE POGO OT ODMA core ODMA regs)\n");
    print("OIER=0x%X\n",piePogoOTODMACoreRead(OIER));
    print("OIPR=0x%X\n",piePogoOTODMACoreRead(OIPR));
    print("OLWR=0x%X\n",piePogoOTODMACoreRead(OLWR));
    print("OCNTRL=0x%X\n",piePogoOTODMACoreRead(OCNTRL));
    print("OTR0=0x%X\n",piePogoOTODMACoreRead(OTR0));
    print("OTR1=0x%X\n",piePogoOTODMACoreRead(OTR1));    
    print("OTR2=0x%X\n",piePogoOTODMACoreRead(OTR2));
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_otodma_udma_irqs_uint32(struct device_data *device_data,
                                                 uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoOTODMAUDMARead(UIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoOTODMAUDMAWrite(UICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// irqstruct - if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_otodma_udma_irqs(struct device_data *device_data,
                                 struct odma_interrupt_info *irqstruct)
{
    uint32_t reg, ints_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_pogo_otodma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    ints_to_clear = pie_otodma_convert_irqstruct_to_update_udma_reg(irqstruct, 0, true);
    reg = clear_pogo_otodma_udma_irqs_uint32(device_data, ints_to_clear);
    
    // set fields for all ints that were set
    pie_otodma_convert_udma_uintarray_to_irqstruct(irqstruct, reg);
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_otodma_core_irqs_uint32(struct device_data *device_data,
                                                   uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoOTODMACoreRead(OIPR);
    piePogoOTODMACoreWrite(OICR, int_val);
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// irqstruct - if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_otodma_core_irqs(struct device_data *device_data,
                                 struct odma_interrupt_info *irqstruct)
{
    uint32_t reg, ints_to_clear;
    
    if (irqstruct == NULL)
    {
        reg = clear_pogo_otodma_core_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    ints_to_clear = pie_otodma_convert_irqstruct_to_update_core_reg(irqstruct, 0, true);
    reg = clear_pogo_otodma_core_irqs_uint32(device_data, ints_to_clear);

    // set fields for all ints that were set
    pie_otodma_convert_core_uintarray_to_irqstruct(irqstruct, reg);
}

// this function runs in interrupt context - no long operations
void handle_pogo_otodma_udma_irqs(struct device_data *device_data)
{
    uint32_t int_val;
    struct odma_interrupt_info irqstruct;    

    int_val = clear_pogo_otodma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    if (int_val != 0)
    {
        pie_otodma_convert_udma_uintarray_to_irqstruct(&irqstruct, int_val);
        // invalidate the core parts
        irqstruct.EndOfStrip = false;
        irqstruct.LengthErr = false;
        
        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

// this function runs in interrupt context - no long operations
void handle_pogo_otodma_core_irqs(struct device_data *device_data)
{
    uint32_t int_val;
    struct odma_interrupt_info irqstruct;    

    int_val = clear_pogo_otodma_core_irqs_uint32(device_data, 0xFFFFFFFF);
    if (int_val != 0)
    {
        pie_otodma_convert_core_uintarray_to_irqstruct(&irqstruct, int_val);
        // invalidate the udma parts of the structure
        irqstruct.Desc = false;
        irqstruct.ClearComplete = false;
        irqstruct.Own = false;
        irqstruct.OutOfRangeErr = false;
        irqstruct.RRespErr = false;
        irqstruct.BRespErr = false;

        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback);
        }
    }
}

void start_pogo_otodma_udma(struct device_data *device_data, uint32_t *phys_desc)
{
    piePogoOTODMAUDMAWrite(UDR, phys_desc);
}

/////////////////////////////////////
// udma functions
/////////////////////////////////////

static int ot_udma_revcheck(struct device_data *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = POGO_OTODMA_POGO_ODMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(piePogoOTODMAUDMARead(UTR0));
    pieh_rev = POGO_OTODMA_POGO_ODMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(pie_handle->pie_otpogo_odma_udma->UTR0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void ot_udma_reset(struct device_data *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = piePogoOTODMAUDMARead(UCR);
    reg = POGO_OTODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 0); // put udma in reset
    piePogoOTODMAUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;

    msleep(5);  // wait for the reset to occur

    PROTECT_REG_ACCESS;
    reg = piePogoOTODMAUDMARead(UCR);
    reg = POGO_OTODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 1); // take udma out of reset
    piePogoOTODMAUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;
}

// NOTE: we do not write the UDR, since that is not a configuration
// register, but rather starts the DMA.  Be sure to call the
// function write_pogo_otodma_descriptor() to do that after configure
static void ot_udma_configure(struct device_data *device_data,
                              struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_otodma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_otodma_udma_callback_data;
    
    // USR and UPR are Read only
    piePogoOTODMAUDMAWrite(UICR, pie_handle->pie_otpogo_odma_udma->UICR); // irq clear before enable
    piePogoOTODMAUDMAWrite(UIER, pie_handle->pie_otpogo_odma_udma->UIER);
    // UIPR is Read only
    piePogoOTODMAUDMAWrite(UIFR, pie_handle->pie_otpogo_odma_udma->UIFR);
    // configure does NOT write the UDR - it would start the dma going
    // UBAR, UBLR, UBRR, UTR0, and UTR1 are Read only

    // just as a good policy, take the subblock out of soft reset as the last thing
    piePogoOTODMAUDMAWrite(UCR, pie_handle->pie_otpogo_odma_udma->UCR);
}

static void get_ot_udma_current(struct device_data *device_data,
                                struct pie_handle_t *pie_handle)
{
    pie_handle->pie_otpogo_odma_udma->UCR = piePogoOTODMAUDMARead(UCR);
    pie_handle->pie_otpogo_odma_udma->USR = piePogoOTODMAUDMARead(USR);
    pie_handle->pie_otpogo_odma_udma->UPR = piePogoOTODMAUDMARead(UPR);
    pie_handle->pie_otpogo_odma_udma->UIER = piePogoOTODMAUDMARead(UIER);
    pie_handle->pie_otpogo_odma_udma->UIPR = piePogoOTODMAUDMARead(UIPR);
    // UICR and UIFR are Write only
    pie_handle->pie_otpogo_odma_udma->UDR = piePogoOTODMAUDMARead(UDR);
    pie_handle->pie_otpogo_odma_udma->UBAR = piePogoOTODMAUDMARead(UBAR);
    pie_handle->pie_otpogo_odma_udma->UBLR = piePogoOTODMAUDMARead(UBLR);
    pie_handle->pie_otpogo_odma_udma->UBRR = piePogoOTODMAUDMARead(UBRR);
    pie_handle->pie_otpogo_odma_udma->UTR0 = piePogoOTODMAUDMARead(UTR0);
    pie_handle->pie_otpogo_odma_udma->UTR1 = piePogoOTODMAUDMARead(UTR1);
}

/////////////////////////////////////
// core functions
/////////////////////////////////////

static int ot_core_revcheck(struct device_data *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = POGO_OTODMA_POGO_ODMA_CORE_OTR1_TAGMAJ_MASK_SHIFT(piePogoOTODMACoreRead(OTR1));
    pieh_rev = POGO_OTODMA_POGO_ODMA_CORE_OTR1_TAGMAJ_MASK_SHIFT(pie_handle->pie_otpogo_odma_core->OTR1);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void ot_core_reset(struct device_data *device_data)
{
    return; // nothing to do for odma_core subblock reset
}

static void ot_core_configure(struct device_data *device_data,
                              struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_otodma_core_callback;
    device_data->interrupt_callback_data = pie_handle->pie_otodma_core_callback_data;
    
    piePogoOTODMACoreWrite(OICR, pie_handle->pie_otpogo_odma_core->OICR); // clear irq before enable
    // OIPR is Read only
    piePogoOTODMACoreWrite(OIER, pie_handle->pie_otpogo_odma_core->OIER);
    piePogoOTODMACoreWrite(OIFR, pie_handle->pie_otpogo_odma_core->OIFR);
    piePogoOTODMACoreWrite(OLWR, pie_handle->pie_otpogo_odma_core->OLWR);
    piePogoOTODMACoreWrite(OCNTRL, pie_handle->pie_otpogo_odma_core->OCNTRL);    
    // OTR0, OTR1, and OTR2 are Read only
}


static void get_ot_core_current(struct device_data *device_data,
                             struct pie_handle_t *pie_handle)
{
    pie_handle->pie_otpogo_odma_core->OIER = piePogoOTODMACoreRead(OIER);
    pie_handle->pie_otpogo_odma_core->OIPR = piePogoOTODMACoreRead(OIPR);
    // OICR and OIFR are Write only
    pie_handle->pie_otpogo_odma_core->OLWR = piePogoOTODMACoreRead(OLWR);
    pie_handle->pie_otpogo_odma_core->OCNTRL = piePogoOTODMACoreRead(OCNTRL);
    pie_handle->pie_otpogo_odma_core->OTR0 = piePogoOTODMACoreRead(OTR0);
    pie_handle->pie_otpogo_odma_core->OTR1 = piePogoOTODMACoreRead(OTR1);
    pie_handle->pie_otpogo_odma_core->OTR2 = piePogoOTODMACoreRead(OTR2);
}

struct pie_pogo_otodma_udma_function_struct pie_pogo_otodma_udma_functions =
{
    .pie_reset = ot_udma_reset,
    .pie_configure = ot_udma_configure,
    .pie_get_current = get_ot_udma_current,
    .pie_revcheck = ot_udma_revcheck,

    .dump_pogo_otodma_udma_regs = dump_pogo_otodma_udma_regs,

    .handle_pogo_otodma_udma_irqs = handle_pogo_otodma_udma_irqs,
    .clear_pogo_otodma_udma_irqs = clear_pogo_otodma_udma_irqs,

    .start_pogo_otodma_udma = start_pogo_otodma_udma,
};

struct pie_pogo_otodma_core_function_struct pie_pogo_otodma_core_functions =
{
    .pie_reset = ot_core_reset,
    .pie_configure = ot_core_configure,
    .pie_get_current = get_ot_core_current,    
    .pie_revcheck = ot_core_revcheck,

    .dump_pogo_otodma_core_regs = dump_pogo_otodma_core_regs,

    .handle_pogo_otodma_core_irqs = handle_pogo_otodma_core_irqs,

    .clear_pogo_otodma_core_irqs = clear_pogo_otodma_core_irqs,
};

// odma register/unregister
void pie_pogo_otodma_udma_init(struct device_data *device_data)
{
    piePogoOTODMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoOTODMAUDMADeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_otodma_udma_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_ot_odma_udma, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIE OTODMA UDMA LOADED .....\n");    
}
EXPORT_SYMBOL(pie_pogo_otodma_udma_init);

void pie_pogo_otodma_udma_exit(struct device_data *device_data)
{
    piePogoOTODMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = unregister_pogo_subblock(pogo_ot_odma_udma, 0);
    free_memory(pie_device_handle);
    debug_print("PIE OTODMA UDMA EXIT .....\n");
}
EXPORT_SYMBOL(pie_pogo_otodma_udma_exit);

void pie_pogo_otodma_core_init(struct device_data *device_data)
{
    piePogoOTODMACoreDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoOTODMACoreDeviceHandle), GFP_KERNEL);
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_otodma_core_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_ot_odma_core, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;    
    pie_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIE OTODMA CORE LOADED .....\n");
}
EXPORT_SYMBOL(pie_pogo_otodma_core_init);

void pie_pogo_otodma_core_exit(void)
{
    piePogoOTODMACoreDeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_ot_odma_core, 0);
    free_memory(pie_device_handle);
    debug_print("PIE OTODMA CORE EXIT .....\n");
}
EXPORT_SYMBOL(pie_pogo_otodma_core_exit);

