/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie Read/Write macro)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t
#include <linux/delay.h>      // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pie_pogo_otidma.h"
#include "pie_pogo_otidma_if.h"
#include "pogo_dma_driver.h"

#include "pie_driverlib_if.h"

void dump_pogo_otidma_udma_regs(piePogoOTIDMAData *device_data)
{
    print("print the OT UDMA IDMA regs)\n");
    print("UCR=0x%X\n", piePogoOTIdmaUDMARead(UCR));
    print("USR=0x%X\n", piePogoOTIdmaUDMARead(USR));
    print("UPR=0x%X\n", piePogoOTIdmaUDMARead(UPR));
    print("UIER=0x%X\n", piePogoOTIdmaUDMARead(UIER));
    print("UIPR=0x%X\n", piePogoOTIdmaUDMARead(UIPR));
    print("UDR=0x%X\n", piePogoOTIdmaUDMARead(UDR));
    print("UBAR=0x%X\n", piePogoOTIdmaUDMARead(UBAR));
    print("UBLR=0x%X\n", piePogoOTIdmaUDMARead(UBLR));
    print("UBRR=0x%X\n", piePogoOTIdmaUDMARead(UBRR));
    print("UTR0=0x%X\n", piePogoOTIdmaUDMARead(UTR0));
    print("UTR1=0x%X\n", piePogoOTIdmaUDMARead(UTR1));
}

void dump_pogo_otidma_core_regs(piePogoOTIDMAData *device_data)
{
    print("(now the core OT IDMA regs)\n");
    print("ICR=0x%X\n", piePogoOTIdmaCoreRead(ICR));
    print("IIER=0x%X\n", piePogoOTIdmaCoreRead(IIER));
    print("IIPR=0x%X\n", piePogoOTIdmaCoreRead(IIPR));
    print("ILWR=0x%X\n", piePogoOTIdmaCoreRead(ILWR));
    print("IRHR=0x%X\n", piePogoOTIdmaCoreRead(IRHR));
    print("ITR0=0x%X\n", piePogoOTIdmaCoreRead(ITR0));
    print("ITR1=0x%X\n", piePogoOTIdmaCoreRead(ITR1));
    print("ITR2=0x%X\n", piePogoOTIdmaCoreRead(ITR2));
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_otidma_udma_irqs_uint32(piePogoOTIDMAData *device_data,
                                                   uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoOTIdmaUDMARead(UIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoOTIdmaUDMAWrite(UICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

//  if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_otidma_udma_irqs(piePogoOTIDMAData *device_data,
                                 struct idma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_pogo_otidma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    irqs_to_clear = pie_otidma_convert_irqstruct_to_update_udma_reg(irqstruct, 0, true);
    reg = clear_pogo_otidma_udma_irqs_uint32(device_data, irqs_to_clear);
    
    // set fields for all irqs that were set
    pie_otidma_convert_udma_uintarray_to_irqstruct(irqstruct, reg);
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_otidma_core_irqs_uint32(piePogoOTIDMAData *device_data,
                                                   uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoOTIdmaCoreRead(IIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoOTIdmaCoreWrite(IICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}
          
//  if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_otidma_core_irqs(piePogoOTIDMAData *device_data,
                                 struct idma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;
    
    if (irqstruct == NULL)
    {
        reg = clear_pogo_otidma_core_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    irqs_to_clear = pie_otidma_convert_irqstruct_to_update_core_reg(irqstruct, 0, true);
    reg = clear_pogo_otidma_core_irqs_uint32(device_data, irqs_to_clear);

    // set fields for all irqs that were set
    pie_otidma_convert_core_uintarray_to_irqstruct(irqstruct, reg);
}

// this function runs in interrupt context - no long operations
void handle_pogo_otidma_udma_irqs(piePogoOTIDMAData *device_data)
{
    uint32_t int_val;
    struct idma_interrupt_info irqstruct;
    
    int_val = clear_pogo_otidma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    if (int_val != 0)
    {
        pie_otidma_convert_udma_uintarray_to_irqstruct(&irqstruct, int_val);

        // invalidate the idma_core parts of the structure
        irqstruct.EndOfStrip = false; 
        irqstruct.LengthErr = false;
        
        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

// this function runs in interrupt context - no long operations
void handle_pogo_otidma_core_irqs(piePogoOTIDMAData *device_data)
{
    uint32_t int_val;
    struct idma_interrupt_info irqstruct;

    int_val = clear_pogo_otidma_core_irqs_uint32(device_data, 0xFFFFFFFF);
    if (int_val != 0)
    {
        pie_otidma_convert_core_uintarray_to_irqstruct(&irqstruct, int_val);

        // invalidate the udma parts of the structure
        irqstruct.Desc = false;
        irqstruct.ClearComplete = false;
        irqstruct.Own = false;
        irqstruct.OutOfRangeErr = false;
        irqstruct.RRespErr = false;
        irqstruct.BRespErr = false;

        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback_data);
        }
    }
}

void start_pogo_otidma_udma(piePogoOTIDMAData *device_data, uint32_t *phys_desc)
{
    piePogoOTIdmaUDMAWrite(UDR, phys_desc);
}

/////////////////////////////////////
// udma functions
/////////////////////////////////////

static int ot_udma_revcheck(piePogoOTIDMAData *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = OT_POGO_IDMA_POGO_IDMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(piePogoOTIdmaUDMARead(UTR0));
    pieh_rev = OT_POGO_IDMA_POGO_IDMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(pie_handle->pie_otpogo_idma_udma->UTR0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void ot_udma_reset(piePogoOTIDMAData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = piePogoOTIdmaUDMARead(UCR);
    reg = OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 0); // put udma in reset
    piePogoOTIdmaUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;

    msleep(5);  // wait for the reset to occur

    PROTECT_REG_ACCESS;
    reg = piePogoOTIdmaUDMARead(UCR);
    reg = OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 1); // take udma out of reset
    piePogoOTIdmaUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;
}


// NOTE: we do not write the UDR, since that is not a configuration
// register, but rather starts the DMA.  Be sure to call the
// function write_pogo_otidma_descriptor() to do that after configure
static void ot_udma_configure(piePogoOTIDMAData *device_data,
                              struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_otidma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_otidma_udma_callback_data;    

    // USR and UPR are Read only
    piePogoOTIdmaUDMAWrite(UICR, pie_handle->pie_otpogo_idma_udma->UICR); // irq clear before enable
    piePogoOTIdmaUDMAWrite(UIER, pie_handle->pie_otpogo_idma_udma->UIER);
    // UIPR is Read only
    piePogoOTIdmaUDMAWrite(UIFR, pie_handle->pie_otpogo_idma_udma->UIFR);
    // configure does NOT write the UDR - it would start the dma going
    // UBAR, UBLR, UBRR, UTR0, and UTR1 are Read only

    // just as a good policy, take the subblock out of soft reset as the last thing
    piePogoOTIdmaUDMAWrite(UCR, pie_handle->pie_otpogo_idma_udma->UCR);    
}

static void ot_get_udma_current(piePogoOTIDMAData *device_data,
                                struct pie_handle_t *pie_handle)
{
    pie_handle->pie_otpogo_idma_udma->UCR = piePogoOTIdmaUDMARead(UCR);
    pie_handle->pie_otpogo_idma_udma->USR = piePogoOTIdmaUDMARead(USR);
    pie_handle->pie_otpogo_idma_udma->UPR = piePogoOTIdmaUDMARead(UPR);
    pie_handle->pie_otpogo_idma_udma->UIER = piePogoOTIdmaUDMARead(UIER);
    pie_handle->pie_otpogo_idma_udma->UIPR = piePogoOTIdmaUDMARead(UIPR);
    // UICR and UIFR are Write only
    pie_handle->pie_otpogo_idma_udma->UDR = piePogoOTIdmaUDMARead(UDR);
    pie_handle->pie_otpogo_idma_udma->UBAR = piePogoOTIdmaUDMARead(UBAR);
    pie_handle->pie_otpogo_idma_udma->UBLR = piePogoOTIdmaUDMARead(UBLR);
    pie_handle->pie_otpogo_idma_udma->UBRR = piePogoOTIdmaUDMARead(UBRR);
    pie_handle->pie_otpogo_idma_udma->UTR0 = piePogoOTIdmaUDMARead(UTR0);
    pie_handle->pie_otpogo_idma_udma->UTR1 = piePogoOTIdmaUDMARead(UTR1);
}

/////////////////////////////////////
// core functions
/////////////////////////////////////

static int ot_core_revcheck(piePogoOTIDMAData *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0= OT_POGO_IDMA_POGO_IDMA_CORE_ITR1_TAGMAJ_MASK_SHIFT(piePogoOTIdmaCoreRead(ITR1));
    pieh_rev = OT_POGO_IDMA_POGO_IDMA_CORE_ITR1_TAGMAJ_MASK_SHIFT(pie_handle->pie_otpogo_idma_core->ITR1);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        
        return -1;
    }
    else
        return 0;
}


static void ot_core_reset(piePogoOTIDMAData *device_data)
{
    return; // nothing to do for idma_core subblock reset
}

static void ot_core_configure(piePogoOTIDMAData *device_data,
                              struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_otidma_core_callback;
    device_data->interrupt_callback_data = pie_handle->pie_otidma_core_callback_data;
    
    piePogoOTIdmaCoreWrite(ICR, pie_handle->pie_otpogo_idma_core->ICR);
    piePogoOTIdmaCoreWrite(IICR, pie_handle->pie_otpogo_idma_core->IICR); // clear irq before enable
    // IIPR is Read only
    piePogoOTIdmaCoreWrite(IIER, pie_handle->pie_otpogo_idma_core->IIER);
    piePogoOTIdmaCoreWrite(IIFR, pie_handle->pie_otpogo_idma_core->IIFR);
    piePogoOTIdmaCoreWrite(ILWR, pie_handle->pie_otpogo_idma_core->ILWR);
    piePogoOTIdmaCoreWrite(IRHR, pie_handle->pie_otpogo_idma_core->IRHR);
    // ITR0, ITR1, and ITR2 are Read only
}

static void ot_get_core_current(piePogoOTIDMAData *device_data,
                                struct pie_handle_t *pie_handle)
               
{
    pie_handle->pie_otpogo_idma_core->ICR = piePogoOTIdmaCoreRead(ICR);
    pie_handle->pie_otpogo_idma_core->IIER = piePogoOTIdmaCoreRead(IIER);
    pie_handle->pie_otpogo_idma_core->IIPR = piePogoOTIdmaCoreRead(IIPR);
    // IICR and IIFR are Write only
    pie_handle->pie_otpogo_idma_core->ILWR = piePogoOTIdmaCoreRead(ILWR);
    pie_handle->pie_otpogo_idma_core->IRHR = piePogoOTIdmaCoreRead(IRHR);
    pie_handle->pie_otpogo_idma_core->ITR0 = piePogoOTIdmaCoreRead(ITR0);
    pie_handle->pie_otpogo_idma_core->ITR1 = piePogoOTIdmaCoreRead(ITR1);
    pie_handle->pie_otpogo_idma_core->ITR2 = piePogoOTIdmaCoreRead(ITR2);
}

struct pie_pogo_otidma_udma_function_struct pie_pogo_otidma_udma_functions =
{
    .pie_reset = ot_udma_reset,
    .pie_configure = ot_udma_configure,
    .pie_get_current = ot_get_udma_current,
    .pie_revcheck = ot_udma_revcheck,

    .dump_pogo_otidma_udma_regs = dump_pogo_otidma_udma_regs,

    .handle_pogo_otidma_udma_irqs = handle_pogo_otidma_udma_irqs,
    .clear_pogo_otidma_udma_irqs = clear_pogo_otidma_udma_irqs,

    .start_pogo_otidma_udma = start_pogo_otidma_udma,
};

struct pie_pogo_otidma_core_function_struct pie_pogo_otidma_core_functions =
{
    .pie_reset = ot_core_reset,
    .pie_configure = ot_core_configure,
    .pie_get_current = ot_get_core_current,
    .pie_revcheck = ot_core_revcheck,

    .dump_pogo_otidma_core_regs = dump_pogo_otidma_core_regs,

    .handle_pogo_otidma_core_irqs = handle_pogo_otidma_core_irqs,
    .clear_pogo_otidma_core_irqs = clear_pogo_otidma_core_irqs,
};

// idma register/unregister
void pie_pogo_otidma_udma_init(piePogoOTIDMAData *device_data)
{
    piePogoOTIDMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoOTIDMAUDMADeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_otidma_udma_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_ot_idma_udma, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIE OTIDMA UDMA LOADED .....\n");
}
EXPORT_SYMBOL(pie_pogo_otidma_udma_init);

void pie_pogo_otidma_udma_exit(piePogoOTIDMAData *device_data)
{
    piePogoOTIDMAUDMADeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_ot_idma_udma, 0);
    free_memory(pie_device_handle);
    debug_print("PIE OTIDMA UDMA EXIT .....\n");
}
EXPORT_SYMBOL(pie_pogo_otidma_udma_exit);

void pie_pogo_otidma_core_init(piePogoOTIDMAData *device_data)
{
    piePogoOTIDMACoreDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoOTIDMACoreDeviceHandle), GFP_KERNEL);
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_otidma_core_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_ot_idma_core, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIE OTIDMA CORE LOADED .....\n");
}
EXPORT_SYMBOL(pie_pogo_otidma_core_init);

void pie_pogo_otidma_core_exit(piePogoOTIDMAData *device_data)
{
    piePogoOTIDMACoreDeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_ot_idma_core, 0);
    free_memory(pie_device_handle);
    debug_print("PIE OTIDMA CORE EXIT .....\n");
}
EXPORT_SYMBOL(pie_pogo_otidma_core_exit);

