/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>  // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>  // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t
#include <linux/delay.h>  // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pogo_dma_driver.h"
#include "pie_pogo_odma.h"
#include "pie_pogo_odma_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_pogo_odma_udma_regs(struct device_data *device_data)
{
    print("Dump PIE POGO ODMA UDMA Registers\n");
    print("UCR=0x%X\n", piePogoODMAUDMARead(UCR));
    print("USR=0x%X\n", piePogoODMAUDMARead(USR));
    print("UPR=0x%X\n", piePogoODMAUDMARead(UPR));
    print("UIER=0x%X\n", piePogoODMAUDMARead(UIER));
    print("UIPR=0x%X\n", piePogoODMAUDMARead(UIPR));
    print("UDR=0x%X\n", piePogoODMAUDMARead(UDR));
    print("UBAR=0x%X\n", piePogoODMAUDMARead(UBAR));
    print("UBLR=0x%X\n", piePogoODMAUDMARead(UBLR));
    print("UBRR=0x%X\n", piePogoODMAUDMARead(UBRR));
    print("UTR0=0x%X\n", piePogoODMAUDMARead(UTR0));
    print("UTR1=0x%X\n", piePogoODMAUDMARead(UTR1));
}

void dump_pogo_odma_core_regs(struct device_data *device_data)
{
    print("Dump PIE POGO ODMA core ODMA regs)\n");
    print("OIER=0x%X\n",piePogoODMACoreRead(OIER));
    print("OIPR=0x%X\n",piePogoODMACoreRead(OIPR));
    print("OLWR=0x%X\n",piePogoODMACoreRead(OLWR));
    print("OCNTRL=0x%X\n",piePogoODMACoreRead(OCNTRL));
    print("OTR0=0x%X\n",piePogoODMACoreRead(OTR0));
    print("OTR1=0x%X\n",piePogoODMACoreRead(OTR1));    
    print("OTR2=0x%X\n",piePogoODMACoreRead(OTR2));
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_odma_udma_irqs_uint32(struct device_data *device_data,
                                                 uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoODMAUDMARead(UIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoODMAUDMAWrite(UICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// irqstruct - if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_odma_udma_irqs(struct device_data *device_data,
                               struct odma_interrupt_info *irqstruct)
{
    uint32_t reg, ints_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_pogo_odma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    ints_to_clear = pie_odma_convert_irqstruct_to_update_udma_reg(irqstruct, 0, true);
    reg = clear_pogo_odma_udma_irqs_uint32(device_data, ints_to_clear);
    
    // set fields for all ints that were set
    pie_odma_convert_udma_uintarray_to_irqstruct(irqstruct, reg);
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_odma_core_irqs_uint32(struct device_data *device_data,
                                                 uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoODMACoreRead(OIPR);
    piePogoODMACoreWrite(OICR, int_val);
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// irqstruct - if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_odma_core_irqs(struct device_data *device_data,
                               struct odma_interrupt_info *irqstruct)
{
    uint32_t reg, ints_to_clear;
    
    if (irqstruct == NULL)
    {
        reg = clear_pogo_odma_core_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    ints_to_clear = pie_odma_convert_irqstruct_to_update_core_reg(irqstruct, 0, true);
    reg = clear_pogo_odma_core_irqs_uint32(device_data, ints_to_clear);

    // set fields for all ints that were set
    pie_odma_convert_core_uintarray_to_irqstruct(irqstruct, reg);
}

// this function runs in interrupt context - no long operations
void handle_pogo_odma_udma_irqs(struct device_data *device_data)
{
    uint32_t int_val;
    struct odma_interrupt_info irqstruct;    
    
    int_val = clear_pogo_odma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    if (int_val != 0)
    {
        pie_odma_convert_udma_uintarray_to_irqstruct(&irqstruct, int_val);
        // invalidate the core parts
        irqstruct.EndOfStrip = false;
        irqstruct.LengthErr = false;
        
        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

// this function runs in interrupt context - no long operations
void handle_pogo_odma_core_irqs(struct device_data *device_data)
{
    uint32_t int_val;
    struct odma_interrupt_info irqstruct;    

    int_val = clear_pogo_odma_core_irqs_uint32(device_data, 0xFFFFFFFF);
    if (int_val != 0)
    {
        pie_odma_convert_core_uintarray_to_irqstruct(&irqstruct, int_val);
        // invalidate the udma parts of the structure
        irqstruct.Desc = false;
        irqstruct.ClearComplete = false;
        irqstruct.Own = false;
        irqstruct.OutOfRangeErr = false;
        irqstruct.RRespErr = false;
        irqstruct.BRespErr = false;

        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback);
        }
    }
}

void start_pogo_odma_udma(struct device_data *device_data, uint32_t *phys_desc)
{
    piePogoODMAUDMAWrite(UDR, phys_desc);
}

/////////////////////////////////////
// udma functions
/////////////////////////////////////

///// SPECIAL UDMA function needed for the dma system - need to be able to check if dma block is enabled
static bool udma_enabled(struct device_data *device_data)
{
    uint32_t reg;
    uint8_t soft_reset;
    bool enabled;

    enabled = false;
    reg = piePogoODMAUDMARead(USR);
    soft_reset = POGO_ODMA_POGO_ODMA_UDMA_USR_CLEAR_MASK_SHIFT(reg);
    if (soft_reset == 0)
    {
        enabled = true;
    }
    return enabled;
}

///// SPECIAL UDMA function needed for the dma system - need to be able to check if status is idle or not
static bool udma_idle(struct device_data *device_data)
{
    uint32_t reg;
    uint8_t busy;
    bool idle;

    idle = false;
    reg = piePogoODMAUDMARead(USR);
    busy = POGO_ODMA_POGO_ODMA_UDMA_USR_BUSY_MASK_SHIFT(reg);
    if (busy == 0)
    {
        idle = true;
    }
    return idle;
}

static int udma_revcheck(struct device_data *device_data,
                         struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = POGO_ODMA_POGO_ODMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(piePogoODMAUDMARead(UTR0));
    pieh_rev = POGO_ODMA_POGO_ODMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(pie_handle->pie_pogo_odma_udma->UTR0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void udma_reset(struct device_data *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = piePogoODMAUDMARead(UCR);
    reg = POGO_ODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 0); // put udma in reset
    piePogoODMAUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;

    msleep(5);  // wait for the reset to occur

    PROTECT_REG_ACCESS;
    reg = piePogoODMAUDMARead(UCR);
    reg = POGO_ODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 1); // take udma out of reset
    piePogoODMAUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;
}

// NOTE: we do not write the UDR, since that is not a configuration
// register, but rather starts the DMA.  Be sure to call the
// function write_pogo_odma_descriptor() to do that after configure
static void udma_configure(struct device_data *device_data,
                           struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_odma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_odma_udma_callback_data;
    
    // USR and UPR are Read only
    piePogoODMAUDMAWrite(UICR, pie_handle->pie_pogo_odma_udma->UICR); // irq clear before enable
    piePogoODMAUDMAWrite(UIER, pie_handle->pie_pogo_odma_udma->UIER);
    // UIPR is Read only
    piePogoODMAUDMAWrite(UIFR, pie_handle->pie_pogo_odma_udma->UIFR);
    // configure does NOT write the UDR - it would start the dma going
    // UBAR, UBLR, UBRR, UTR0, and UTR1 are Read only

    // just as a good policy, take the subblock out of soft reset as the last thing
    piePogoODMAUDMAWrite(UCR, pie_handle->pie_pogo_odma_udma->UCR);
}

static void get_udma_current(struct device_data *device_data,
                             struct pie_handle_t *pie_handle)
{
    pie_handle->pie_pogo_odma_udma->UCR = piePogoODMAUDMARead(UCR);
    pie_handle->pie_pogo_odma_udma->USR = piePogoODMAUDMARead(USR);
    pie_handle->pie_pogo_odma_udma->UPR = piePogoODMAUDMARead(UPR);
    pie_handle->pie_pogo_odma_udma->UIER = piePogoODMAUDMARead(UIER);
    pie_handle->pie_pogo_odma_udma->UIPR = piePogoODMAUDMARead(UIPR);
    // UICR and UIFR are Write only
    pie_handle->pie_pogo_odma_udma->UDR = piePogoODMAUDMARead(UDR);
    pie_handle->pie_pogo_odma_udma->UBAR = piePogoODMAUDMARead(UBAR);
    pie_handle->pie_pogo_odma_udma->UBLR = piePogoODMAUDMARead(UBLR);
    pie_handle->pie_pogo_odma_udma->UBRR = piePogoODMAUDMARead(UBRR);
    pie_handle->pie_pogo_odma_udma->UTR0 = piePogoODMAUDMARead(UTR0);
    pie_handle->pie_pogo_odma_udma->UTR1 = piePogoODMAUDMARead(UTR1);
}

/////////////////////////////////////
// core functions
/////////////////////////////////////

static int core_revcheck(struct device_data *device_data,
                         struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = POGO_ODMA_POGO_ODMA_CORE_OTR1_TAGMAJ_MASK_SHIFT(piePogoODMACoreRead(OTR1));
    pieh_rev = POGO_ODMA_POGO_ODMA_CORE_OTR1_TAGMAJ_MASK_SHIFT(pie_handle->pie_pogo_odma_core->OTR1);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void core_reset(struct device_data *device_data)
{
    return; // nothing to do for odma_core subblock reset
}

static void core_configure(struct device_data *device_data,
                           struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_odma_core_callback;
    device_data->interrupt_callback_data = pie_handle->pie_odma_core_callback_data;
    
    piePogoODMACoreWrite(OICR, pie_handle->pie_pogo_odma_core->OICR); // clear irq before enable
    // OIPR is Read only
    piePogoODMACoreWrite(OIER, pie_handle->pie_pogo_odma_core->OIER);
    piePogoODMACoreWrite(OIFR, pie_handle->pie_pogo_odma_core->OIFR);
    piePogoODMACoreWrite(OLWR, pie_handle->pie_pogo_odma_core->OLWR);
    piePogoODMACoreWrite(OCNTRL, pie_handle->pie_pogo_odma_core->OCNTRL);    
    // OTR0, OTR1, and OTR2 are Read only
}


static void get_core_current(struct device_data *device_data,
                             struct pie_handle_t *pie_handle)
{
    pie_handle->pie_pogo_odma_core->OIER = piePogoODMACoreRead(OIER);
    pie_handle->pie_pogo_odma_core->OIPR = piePogoODMACoreRead(OIPR);
    // OICR and OIFR are Write only
    pie_handle->pie_pogo_odma_core->OLWR = piePogoODMACoreRead(OLWR);
    pie_handle->pie_pogo_odma_core->OCNTRL = piePogoODMACoreRead(OCNTRL);
    pie_handle->pie_pogo_odma_core->OTR0 = piePogoODMACoreRead(OTR0);
    pie_handle->pie_pogo_odma_core->OTR1 = piePogoODMACoreRead(OTR1);
    pie_handle->pie_pogo_odma_core->OTR2 = piePogoODMACoreRead(OTR2);
}

struct pie_pogo_odma_udma_function_struct pie_pogo_odma_udma_functions =
{
    .pie_reset = udma_reset,
    .pie_configure = udma_configure,
    .pie_get_current = get_udma_current,
    .pie_revcheck = udma_revcheck,

    .dump_pogo_odma_udma_regs = dump_pogo_odma_udma_regs,

    .handle_pogo_odma_udma_irqs = handle_pogo_odma_udma_irqs,
    .clear_pogo_odma_udma_irqs = clear_pogo_odma_udma_irqs,

    .udma_idle = udma_idle,
    .udma_enabled = udma_enabled,    
    .start_pogo_odma_udma = start_pogo_odma_udma,
};

struct pie_pogo_odma_core_function_struct pie_pogo_odma_core_functions =
{
    .pie_reset = core_reset,
    .pie_configure = core_configure,
    .pie_get_current = get_core_current,    
    .pie_revcheck = core_revcheck,

    .dump_pogo_odma_core_regs = dump_pogo_odma_core_regs,

    .handle_pogo_odma_core_irqs = handle_pogo_odma_core_irqs,

    .clear_pogo_odma_core_irqs = clear_pogo_odma_core_irqs,

};

// odma register/unregister
void pie_pogo_odma_udma_init(struct device_data *device_data)
{
    piePogoODMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoODMAUDMADeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_odma_udma_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_odma_udma, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
}
EXPORT_SYMBOL(pie_pogo_odma_udma_init);

void pie_pogo_odma_udma_exit(struct device_data *device_data)
{
    piePogoODMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = unregister_pogo_subblock(pogo_odma_udma, 0);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_pogo_odma_udma_exit);

void pie_pogo_odma_core_init(struct device_data *device_data)
{
    piePogoODMACoreDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoODMACoreDeviceHandle), GFP_KERNEL);
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_odma_core_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_odma_core, pie_device_handle, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;    
    pie_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pie_pogo_odma_core_init);

void pie_pogo_odma_core_exit(void)
{
    piePogoODMACoreDeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_odma_core, 0);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_pogo_odma_core_exit);

