/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie Read/Write macro)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t
#include <linux/delay.h>      // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pie_pogo_idma.h"
#include "pie_pogo_idma_if.h"
#include "pogo_dma_driver.h"

#include "pie_driverlib_if.h"


void dump_pogo_idma_udma_regs(piePogoIDMAData *device_data)
{
    print("print the UDMA IDMA regs)\n");
    print("UCR=0x%X\n", piePogoIdmaUDMARead(UCR));
    print("USR=0x%X\n", piePogoIdmaUDMARead(USR));
    print("UPR=0x%X\n", piePogoIdmaUDMARead(UPR));
    print("UIER=0x%X\n", piePogoIdmaUDMARead(UIER));
    print("UIPR=0x%X\n", piePogoIdmaUDMARead(UIPR));
    print("UDR=0x%X\n", piePogoIdmaUDMARead(UDR));
    print("UBAR=0x%X\n", piePogoIdmaUDMARead(UBAR));
    print("UBLR=0x%X\n", piePogoIdmaUDMARead(UBLR));
    print("UBRR=0x%X\n", piePogoIdmaUDMARead(UBRR));
    print("UTR0=0x%X\n", piePogoIdmaUDMARead(UTR0));
    print("UTR1=0x%X\n", piePogoIdmaUDMARead(UTR1));
}

void dump_pogo_idma_core_regs(piePogoIDMAData *device_data)
{
    print("(now the core IDMA regs)\n");
    print("ICR=0x%X\n", piePogoIdmaCoreRead(ICR));
    print("IIER=0x%X\n", piePogoIdmaCoreRead(IIER));
    print("IIPR=0x%X\n", piePogoIdmaCoreRead(IIPR));
    print("ILWR=0x%X\n", piePogoIdmaCoreRead(ILWR));
    print("IRHR=0x%X\n", piePogoIdmaCoreRead(IRHR));
    print("ITR0=0x%X\n", piePogoIdmaCoreRead(ITR0));
    print("ITR1=0x%X\n", piePogoIdmaCoreRead(ITR1));
    print("ITR2=0x%X\n", piePogoIdmaCoreRead(ITR2));
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_idma_udma_irqs_uint32(piePogoIDMAData *device_data,
                                                 uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoIdmaUDMARead(UIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoIdmaUDMAWrite(UICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

//  if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_idma_udma_irqs(piePogoIDMAData *device_data,
                               struct idma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_pogo_idma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    irqs_to_clear = pie_idma_convert_irqstruct_to_update_udma_reg(irqstruct, 0, true);
    reg = clear_pogo_idma_udma_irqs_uint32(device_data, irqs_to_clear);
    
    // set fields for all irqs that were set
    pie_idma_convert_udma_uintarray_to_irqstruct(irqstruct, reg);
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_pogo_idma_core_irqs_uint32(piePogoIDMAData *device_data,
                                                 uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = piePogoIdmaCoreRead(IIPR);
    int_val &= reg;
    if (int_val)
    {
        piePogoIdmaCoreWrite(IICR, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}
          
//  if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_pogo_idma_core_irqs(piePogoIDMAData *device_data,
                               struct idma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;
    
    if (irqstruct == NULL)
    {
        reg = clear_pogo_idma_core_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    irqs_to_clear = pie_idma_convert_irqstruct_to_update_core_reg(irqstruct, 0, true);
    reg = clear_pogo_idma_core_irqs_uint32(device_data, irqs_to_clear);

    // set fields for all irqs that were set
    pie_idma_convert_core_uintarray_to_irqstruct(irqstruct, reg);
}

// this function runs in interrupt context - no long operations
void handle_pogo_idma_udma_irqs(piePogoIDMAData *device_data)
{
    uint32_t int_val;
    struct idma_interrupt_info irqstruct;
    
    int_val = clear_pogo_idma_udma_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    if (int_val != 0)
    {
        pie_idma_convert_udma_uintarray_to_irqstruct(&irqstruct, int_val);
        irqstruct.instance = device_data->submodinstance;

        // invalidate the idma_core parts of the structure
        irqstruct.EndOfStrip = false; 
        irqstruct.LengthErr = false;
        
        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

// this function runs in interrupt context - no long operations
void handle_pogo_idma_core_irqs(piePogoIDMAData *device_data)
{
    uint32_t int_val;
    struct idma_interrupt_info irqstruct;

    int_val = clear_pogo_idma_core_irqs_uint32(device_data, 0xFFFFFFFF);
    if (int_val != 0)
    {
        pie_idma_convert_core_uintarray_to_irqstruct(&irqstruct, int_val);
        irqstruct.instance = device_data->submodinstance;

        // invalidate the udma parts of the structure
        irqstruct.Desc = false;
        irqstruct.ClearComplete = false;
        irqstruct.Own = false;
        irqstruct.OutOfRangeErr = false;
        irqstruct.RRespErr = false;
        irqstruct.BRespErr = false;

        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback_data);
        }
    }
}

void start_pogo_idma_udma(piePogoIDMAData *device_data, uint32_t *phys_desc)
{
    piePogoIdmaUDMAWrite(UDR, phys_desc);
}

/////////////////////////////////////
// udma functions
/////////////////////////////////////

///// SPECIAL UDMA function needed for the dma system - need to be able to check if dma block is enabled
static bool udma_enabled(piePogoIDMAData *device_data)
{
    uint32_t reg;
    uint8_t soft_reset;
    bool enabled;

    enabled = false;
    reg = piePogoIdmaUDMARead(USR);
    soft_reset = POGO_IDMA0_POGO_IDMA_UDMA_USR_CLEAR_MASK_SHIFT(reg);
    if (soft_reset == 0)
    {
        enabled = true;
    }
    return enabled;
}

///// SPECIAL UDMA function needed for the dma system - need to be able to check if status is idle or not
static bool udma_idle(piePogoIDMAData *device_data)
{
    uint32_t reg;
    uint8_t busy;
    bool idle;

    idle = false;
    reg = piePogoIdmaUDMARead(USR);
    busy = POGO_IDMA0_POGO_IDMA_UDMA_USR_BUSY_MASK_SHIFT(reg);
    if (busy == 0)
    {
        idle = true;
    }
    return idle;
}

static int udma_revcheck(piePogoIDMAData *device_data,
                         struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = POGO_IDMA0_POGO_IDMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(piePogoIdmaUDMARead(UTR0));
    pieh_rev = POGO_IDMA0_POGO_IDMA_UDMA_UTR0_TAGMAJ_MASK_SHIFT(pie_handle->pie_pogo_idma_udma[idma_num]->UTR0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void udma_reset(piePogoIDMAData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = piePogoIdmaUDMARead(UCR);
    reg = POGO_IDMA0_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 0); // put udma in reset
    piePogoIdmaUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;

    msleep(5);  // wait for the reset to occur

    PROTECT_REG_ACCESS;
    reg = piePogoIdmaUDMARead(UCR);
    reg = POGO_IDMA0_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(reg, 1); // take udma out of reset
    piePogoIdmaUDMAWrite(UCR, reg);
    UNPROTECT_REG_ACCESS;
}


// NOTE: we do not write the UDR, since that is not a configuration
// register, but rather starts the DMA.  Be sure to call the
// function write_pogo_idma_descriptor() to do that after configure
static void udma_configure(piePogoIDMAData *device_data,
                           struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    device_data->interrupt_callback = pie_handle->pie_idma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_idma_udma_callback_data;    

    // USR and UPR are Read only
    piePogoIdmaUDMAWrite(UICR, pie_handle->pie_pogo_idma_udma[idma_num]->UICR); // irq clear before enable
    piePogoIdmaUDMAWrite(UIER, pie_handle->pie_pogo_idma_udma[idma_num]->UIER);
    // UIPR is Read only
    piePogoIdmaUDMAWrite(UIFR, pie_handle->pie_pogo_idma_udma[idma_num]->UIFR);
    // configure does NOT write the UDR - it would start the dma going
    // UBAR, UBLR, UBRR, UTR0, and UTR1 are Read only

    // just as a good policy, take the subblock out of soft reset as the last thing
    piePogoIdmaUDMAWrite(UCR, pie_handle->pie_pogo_idma_udma[idma_num]->UCR);    
}

static void get_udma_current(piePogoIDMAData *device_data,
                             struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    pie_handle->pie_pogo_idma_udma[idma_num]->UCR = piePogoIdmaUDMARead(UCR);
    pie_handle->pie_pogo_idma_udma[idma_num]->USR = piePogoIdmaUDMARead(USR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UPR = piePogoIdmaUDMARead(UPR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UIER = piePogoIdmaUDMARead(UIER);
    pie_handle->pie_pogo_idma_udma[idma_num]->UIPR = piePogoIdmaUDMARead(UIPR);
    // UICR and UIFR are Write only
    pie_handle->pie_pogo_idma_udma[idma_num]->UDR = piePogoIdmaUDMARead(UDR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UBAR = piePogoIdmaUDMARead(UBAR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UBLR = piePogoIdmaUDMARead(UBLR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UBRR = piePogoIdmaUDMARead(UBRR);
    pie_handle->pie_pogo_idma_udma[idma_num]->UTR0 = piePogoIdmaUDMARead(UTR0);
    pie_handle->pie_pogo_idma_udma[idma_num]->UTR1 = piePogoIdmaUDMARead(UTR1);
}

/////////////////////////////////////
// core functions
/////////////////////////////////////

static int core_revcheck(piePogoIDMAData *device_data,
                         struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0= POGO_IDMA0_POGO_IDMA_CORE_ITR1_TAGMAJ_MASK_SHIFT(piePogoIdmaCoreRead(ITR1));
    pieh_rev = POGO_IDMA0_POGO_IDMA_CORE_ITR1_TAGMAJ_MASK_SHIFT(pie_handle->pie_pogo_idma_core[idma_num]->ITR1);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        
        return -1;
    }
    else
        return 0;
}


static void core_reset(piePogoIDMAData *device_data)
{
    return; // nothing to do for idma_core subblock reset
}

static void core_configure(piePogoIDMAData *device_data,
                           struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    device_data->interrupt_callback = pie_handle->pie_idma_core_callback;
    device_data->interrupt_callback_data = pie_handle->pie_idma_core_callback_data;
    
    piePogoIdmaCoreWrite(ICR, pie_handle->pie_pogo_idma_core[idma_num]->ICR);
    piePogoIdmaCoreWrite(IICR, pie_handle->pie_pogo_idma_core[idma_num]->IICR); // clear irq before enable
    // IIPR is Read only
    piePogoIdmaCoreWrite(IIER, pie_handle->pie_pogo_idma_core[idma_num]->IIER);
    piePogoIdmaCoreWrite(IIFR, pie_handle->pie_pogo_idma_core[idma_num]->IIFR);
    piePogoIdmaCoreWrite(ILWR, pie_handle->pie_pogo_idma_core[idma_num]->ILWR);
    piePogoIdmaCoreWrite(IRHR, pie_handle->pie_pogo_idma_core[idma_num]->IRHR);
    // ITR0, ITR1, and ITR2 are Read only
}

static void get_core_current(piePogoIDMAData *device_data,
                             struct pie_handle_t *pie_handle,
                             uint8_t idma_num)
{
    pie_handle->pie_pogo_idma_core[idma_num]->ICR = piePogoIdmaCoreRead(ICR);
    pie_handle->pie_pogo_idma_core[idma_num]->IIER = piePogoIdmaCoreRead(IIER);
    pie_handle->pie_pogo_idma_core[idma_num]->IIPR = piePogoIdmaCoreRead(IIPR);
    // IICR and IIFR are Write only
    pie_handle->pie_pogo_idma_core[idma_num]->ILWR = piePogoIdmaCoreRead(ILWR);
    pie_handle->pie_pogo_idma_core[idma_num]->IRHR = piePogoIdmaCoreRead(IRHR);
    pie_handle->pie_pogo_idma_core[idma_num]->ITR0 = piePogoIdmaCoreRead(ITR0);
    pie_handle->pie_pogo_idma_core[idma_num]->ITR1 = piePogoIdmaCoreRead(ITR1);
    pie_handle->pie_pogo_idma_core[idma_num]->ITR2 = piePogoIdmaCoreRead(ITR2);
}

struct pie_pogo_idma_udma_function_struct pie_pogo_idma_udma_functions =
{
    .pie_reset = udma_reset,
    .pie_configure = udma_configure,
    .pie_get_current = get_udma_current,
    .pie_revcheck = udma_revcheck,

    .dump_pogo_idma_udma_regs = dump_pogo_idma_udma_regs,

    .udma_idle = udma_idle,
    .udma_enabled = udma_enabled,
    .handle_pogo_idma_udma_irqs = handle_pogo_idma_udma_irqs,
    .clear_pogo_idma_udma_irqs = clear_pogo_idma_udma_irqs,

    .start_pogo_idma_udma = start_pogo_idma_udma,
};

struct pie_pogo_idma_core_function_struct pie_pogo_idma_core_functions =
{
    .pie_reset = core_reset,
    .pie_configure = core_configure,
    .pie_get_current = get_core_current,
    .pie_revcheck = core_revcheck,

    .dump_pogo_idma_core_regs = dump_pogo_idma_core_regs,

    .handle_pogo_idma_core_irqs = handle_pogo_idma_core_irqs,
    .clear_pogo_idma_core_irqs = clear_pogo_idma_core_irqs,
};

// idma register/unregister
void pie_pogo_idma_udma_init(piePogoIDMAData *device_data)
{
    piePogoIDMAUDMADeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoIDMAUDMADeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_idma_udma_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_idma_udma, pie_device_handle, device_data->submodinstance);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
}
EXPORT_SYMBOL(pie_pogo_idma_udma_init);

void pie_pogo_idma_udma_exit(piePogoIDMAData *device_data)
{
    piePogoIDMAUDMADeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_idma_udma, device_data->submodinstance);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_pogo_idma_udma_exit);

void pie_pogo_idma_core_init(piePogoIDMAData *device_data)
{
    piePogoIDMACoreDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(piePogoIDMACoreDeviceHandle), GFP_KERNEL);
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_pogo_idma_core_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(pogo_idma_core, pie_device_handle, pie_device_handle->device_data->submodinstance);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
}
EXPORT_SYMBOL(pie_pogo_idma_core_init);

void pie_pogo_idma_core_exit(piePogoIDMAData *device_data)
{
    piePogoIDMACoreDeviceHandle *pie_device_handle;
    
    pie_device_handle = unregister_pogo_subblock(pogo_idma_core, device_data->submodinstance);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_pogo_idma_core_exit);

