/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t
#include <linux/delay.h>     // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_otmarb.h"
#include "pie_otmarb_if.h"
#include "pie_driverlib_if.h"  // for print macros



// nothing for OTMarb to do on a reset
static void pie_reset(pieOTMarbData *device_data)
{
    return;
}

static void pie_configure(pieOTMarbData *device_data, struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_otmarb_callback;
    device_data->interrupt_callback_data = pie_handle->pie_otmarb_callback_data;
    
    pieOTMarbWrite(Config, pie_handle->pie_otmarb->Config);
    pieOTMarbWrite(BGColor, pie_handle->pie_otmarb->BGColor);
    pieOTMarbWrite(IntEnable, pie_handle->pie_otmarb->IntEnable);
    pieOTMarbWrite(IntAck, pie_handle->pie_otmarb->IntAck);    
    //  IntStat is Read only
    pieOTMarbWrite(IntTest, pie_handle->pie_otmarb->IntTest);
    //  REV0 and REV1 are read only
}

static void pie_get_current(pieOTMarbData *device_data, struct pie_handle_t *pie_handle)
{
    pie_handle->pie_otmarb->Config = pieOTMarbRead(Config);
    pie_handle->pie_otmarb->BGColor = pieOTMarbRead(BGColor);
    pie_handle->pie_otmarb->IntEnable = pieOTMarbRead(IntEnable);
    // IntAck is write only
    pie_handle->pie_otmarb->IntStat = pieOTMarbRead(IntStat);
    // IntTest is write only    
    pie_handle->pie_otmarb->REV0 = pieOTMarbRead(REV0);
    pie_handle->pie_otmarb->REV1 = pieOTMarbRead(REV1);
}

static int pie_revcheck(pieOTMarbData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = OTMARB_REV0_MAJ_MASK_SHIFT(pieOTMarbRead(REV0));
    pieh_rev = OTMARB_REV0_MAJ_MASK_SHIFT(pie_handle->pie_otmarb->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

void pie_dump_regs(pieOTMarbData *device_data)
{
    print("PIE OTMARB Registers\n");
    print("Config=0x%X\n",pieOTMarbRead(Config));
    print("BGColor=0x%X\n",pieOTMarbRead(BGColor));
    print("IntEnable=0x%X\n",pieOTMarbRead(IntEnable));
    // IntAck is write only
    print("IntStat=0x%X\n",pieOTMarbRead(IntStat));
    // IntTest is write only
    print("REV0=0x%X\n",pieOTMarbRead(REV0));
    print("REV1=0x%X\n",pieOTMarbRead(REV1));
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_otmarb_irqs_uint32(pieOTMarbData *device_data,
                                         uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;
    
    PROTECT_INTREG_ACCESS_IRQ;
    reg = pieOTMarbRead(IntStat);
    int_val &= reg;
    if (int_val)
    {
        pieOTMarbWrite(IntAck, int_val);
    }
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// this function runs in interrupt context - no long operations
void handle_otmarb_irqs(pieOTMarbData *device_data)
{
    uint32_t int_val;
    struct otmarb_interrupt_info irqstruct;

    int_val = clear_otmarb_irqs_uint32(device_data, 0xFFFFFFFF);
    if (int_val != 0)
    {
        pie_otmarb_convert_uintarray_to_irqstruct(&irqstruct, int_val);

        // now that we have a list of interrupts that were set, do some kind of processing with that.
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback_data);
        }
    }
}

//  if a valid pointer, return the pending interrupts and clear just those.
//                  if NULL, clear all interrupts
void clear_otmarb_irqs(pieOTMarbData *device_data,
                       struct otmarb_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_otmarb_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    irqs_to_clear = pie_otmarb_convert_irqstruct_to_update_reg(irqstruct, 0, true);
    reg = clear_otmarb_irqs_uint32(device_data, irqs_to_clear);
    
    // set fields for all irqs that were set
    pie_otmarb_convert_uintarray_to_irqstruct(irqstruct, reg);
}

struct pie_otmarb_function_struct pie_otmarb_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,

    .pie_dump_regs = pie_dump_regs,

    .handle_otmarb_irqs = handle_otmarb_irqs,
    .clear_otmarb_irqs = clear_otmarb_irqs
};

void pie_otmarb_init(pieOTMarbData *device_data)
{
    otmarbDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(otmarbDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_otmarb_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(otmarb, pie_device_handle);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));    
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIE OTMARB LOADED .....\n");
}
EXPORT_SYMBOL(pie_otmarb_init);

void pie_otmarb_exit(pieOTMarbData *device_data)
{
    otmarbDeviceHandle *pie_device_handle;
    
    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(otmarb);
    free_memory(pie_device_handle);
    debug_print("PIE OTMARB EXIT .....\n");    
}
EXPORT_SYMBOL(pie_otmarb_exit);
