/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_ngadjust.h"
#include "pie_ngadjust_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_ngadjust_regs(ngadjustData *device_data)
{
    print("PIE NGAdjust Registers\n");
    print("Config =0x%X\n",  ngadjustRead(Config));
    print("REV0   =0x%X\n",  ngadjustRead(REV0));
    print("REV1   =0x%X\n",  ngadjustRead(REV1));

    // Note: not dumping LUTAccess here (see dump_ngadjust_lut)
}

void dump_ngadjust_lut(ngadjustData *device_data)
{
    int i;
    uint32_t reg, reg_orig, reg_LUT;
    
    PROTECT_REG_ACCESS;

    // Set CPU access bit before reading LUTs
    reg_orig = ngadjustRead(Config);
    reg = NGADJUST_CONFIG_CPU_ACCESS_REPLACE_VAL(reg_orig, 1);
    ngadjustWrite(Config, reg);

    print("PIE NGAdjust LUT entries (C1,C2):\n");
        
    for (i=0; i < PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        if (i%8 == 0)
        {
            print("\n%04d-%04d: ", i, i+7);
        }
        reg_LUT = ngadjustReadArray(LUTAccess, i);
        print("(0x%02X,", NGADJUST_LUTACCESS_C1_MASK_SHIFT(reg_LUT));
        print("0x%02X) ", NGADJUST_LUTACCESS_C2_MASK_SHIFT(reg_LUT));
    }

    print("\n");

    // Write back the original value for the config reg
    ngadjustWrite(Config, reg_orig);
    UNPROTECT_REG_ACCESS;
}

static int pie_revcheck(ngadjustData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = NGADJUST_REV0_MAJ_MASK_SHIFT(ngadjustRead(REV0));
    pieh_rev = NGADJUST_REV0_MAJ_MASK_SHIFT(pie_handle->pie_ngadjust->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(ngadjustData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = ngadjustRead(Config);
    reg = NGADJUST_CONFIG_BYPASS_REPLACE_VAL(reg, 1);  // bypass to do a reset
    ngadjustWrite(Config, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}
 
static void pie_configure(ngadjustData *device_data,
                          struct pie_handle_t  *pie_handle)
{
    uint32_t reg;
    int      i;

    // There is no callback to configure for ngadjust

    PROTECT_REG_ACCESS;

    // Set CPU access bit before writing LUTs
    reg = NGADJUST_CONFIG_CPU_ACCESS_REPLACE_VAL(pie_handle->pie_ngadjust->Config, 1);
    ngadjustWrite(Config, reg);

    for (i=0; i<PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        ngadjustWriteArray(LUTAccess, pie_handle->pie_ngadjust->LUTAccess[i], i);
    }

    ngadjustWrite(Config, pie_handle->pie_ngadjust->Config);

    // REV0 and REV1 are Read only

    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(ngadjustData *device_data,
                            struct pie_handle_t  *pie_handle)
{
    int i;
    uint32_t reg;

    PROTECT_REG_ACCESS;

    // Note: make sure to grab Config first, will need it when restoring
    //       from LUT access below.
    pie_handle->pie_ngadjust->Config = ngadjustRead(Config);
    pie_handle->pie_ngadjust->REV0   = ngadjustRead(REV0);
    pie_handle->pie_ngadjust->REV1   = ngadjustRead(REV1);

    // Set CPU access bit before reading LUTs
    reg = NGADJUST_CONFIG_CPU_ACCESS_REPLACE_VAL(pie_handle->pie_ngadjust->Config, 1);
    ngadjustWrite(Config, reg);

    for (i=0; i<PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        pie_handle->pie_ngadjust->LUTAccess[i] = ngadjustReadArray(LUTAccess, i);
    }

    // Write back the original value for the config reg
    ngadjustWrite(Config, pie_handle->pie_ngadjust->Config);
    UNPROTECT_REG_ACCESS;
}

struct pie_ngadjust_function_struct pie_ngadjust_functions =
{
    .pie_reset          = pie_reset,
    .pie_configure      = pie_configure,
    .pie_get_current    = pie_get_current,
    .pie_revcheck       = pie_revcheck,
    
    .dump_ngadjust_regs = dump_ngadjust_regs,
    .dump_ngadjust_lut  = dump_ngadjust_lut,
};

void pie_ngadjust_init(ngadjustData *device_data)
{
    ngadjustDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(ngadjustDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_ngadjust_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(ngadjust, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // ngadjust has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE NGADJUST LOADED .....\n");
}
EXPORT_SYMBOL(pie_ngadjust_init);


void pie_ngadjust_exit(ngadjustData *device_data)
{
    ngadjustDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(ngadjust);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_ngadjust_exit);

