/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#ifdef __KERNEL__
#include <linux/kernel.h>  // for uint32_t
#else
#include <stdlib.h>  // for malloc
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <fcntl.h>   // for open
#endif


#include "PIE_regheaders.h"  // for the regmasks
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"      // for fixup_shadow_pointers
#include "pie_if.h"          // for pie irq structures
#include "pie_driverlib_if.h"
//*****************************************************
// interrupt support functions
//*****************************************************

//----------------------------------------------
// set all interrupts in the passed in irqstruct
//----------------------------------------------
void pie_common_set_all_irqstruct(struct pie_common_ints *irqstruct, bool setval)
{
    irqstruct->distort = setval;
    irqstruct->ihi_16x8_short = setval;
    irqstruct->ihi_16x8_tall = setval;
    irqstruct->ihi_3x3_short = setval;
    irqstruct->ihi_3x3_tall = setval;
    irqstruct->pack_mono_wdma = setval;
    irqstruct->dis_map = setval;
    irqstruct->ac_corr_rdma2 = setval;
    irqstruct->ac_corr_rdma1 = setval;
    irqstruct->ac_corr_rdma0 = setval;
    irqstruct->ac_data_rdma2 = setval;
    irqstruct->ac_data_rdma1 = setval;
    irqstruct->ac_data_rdma0 = setval;
    irqstruct->dsmf_short = setval;
    irqstruct->dsmf_tall = setval;
    irqstruct->otmarb = setval;
    irqstruct->ot_wdma = setval;
    irqstruct->wdma = setval;
    irqstruct->ot_rdma = setval;
    irqstruct->rdma2 = setval;
    irqstruct->rdma1 = setval;
    irqstruct->rdma0 = setval;
}

// this works for OT and Data idma
void pie_pogo_set_all_irqstruct_idma(struct idma_interrupt_info *irqstruct, bool setval)
{
    irqstruct->Desc = setval;
    irqstruct->ClearComplete = setval;
    irqstruct->Own = setval;
    irqstruct->OutOfRangeErr = setval;
    irqstruct->RRespErr = setval;
    irqstruct->BRespErr = setval;
    irqstruct->EndOfStrip = setval;
    irqstruct->LengthErr = setval;
}

// this works for OT and Data odma
void pie_pogo_set_all_irqstruct_odma(struct odma_interrupt_info *irqstruct, bool setval)
{
    irqstruct->Desc = setval;
    irqstruct->ClearComplete = setval;
    irqstruct->Own = setval;
    irqstruct->OutOfRangeErr = setval;
    irqstruct->RRespErr = setval;
    irqstruct->BRespErr = setval;
    irqstruct->EndOfStrip = setval;
    irqstruct->LengthErr = setval;
}

void pie_otmarb_set_all_irqstruct(struct otmarb_interrupt_info *irqstruct, bool setval)
{
    irqstruct->DTmismatchErr = setval;
}

//------------------------------------------------------------------------------
// Convert irqstructs to register bits, and update from the passed in reg value.
//
// pass in the current value for the interrupt register, and only the set fields
// in the irqstruct will update it.
// These functions are structured for the case where the interrupt registers all have
// the same bit mappings between enable, pending, and ack, so passing in the type
// of masks to use for conversion is unnecessary. (e.g. parameter operation is not used
// as it is in icetest_common.c)
//------------------------------------------------------------------------------

uint32_t pie_common_convert_irqstruct_to_update_reg(struct pie_common_ints *irqstruct,
                                                    uint32_t pie_common_reg,
                                                    bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;

    if (irqstruct->distort)
        pie_common_reg = PIE_COMMON_IPR_DISTORTINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ihi_16x8_short)
        pie_common_reg = PIE_COMMON_IPR_IHI16X8SHORTINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ihi_16x8_tall)
        pie_common_reg = PIE_COMMON_IPR_IHI16X8TALLINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ihi_3x3_short)
        pie_common_reg = PIE_COMMON_IPR_IHI3X3SHORTINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ihi_3x3_tall)
        pie_common_reg = PIE_COMMON_IPR_IHI3X3TALLINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->pack_mono_wdma)
        pie_common_reg = PIE_COMMON_IPR_PACKMONO_WDMAINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->dis_map)
        pie_common_reg = PIE_COMMON_IPR_DIS_MAPINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_corr_rdma2)
        pie_common_reg = PIE_COMMON_IPR_AC_CORR_RDMA2INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_corr_rdma1)
        pie_common_reg = PIE_COMMON_IPR_AC_CORR_RDMA1INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_corr_rdma0)
        pie_common_reg = PIE_COMMON_IPR_AC_CORR_RDMA0INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_data_rdma2)
        pie_common_reg = PIE_COMMON_IPR_AC_DATA_RDMA2INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_data_rdma1)
        pie_common_reg = PIE_COMMON_IPR_AC_DATA_RDMA1INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ac_data_rdma0)
        pie_common_reg = PIE_COMMON_IPR_AC_DATA_RDMA0INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->dsmf_short)
        pie_common_reg = PIE_COMMON_IPR_DSMFSHORTINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->dsmf_tall)
        pie_common_reg = PIE_COMMON_IPR_DSMFTALLINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->otmarb)
        pie_common_reg = PIE_COMMON_IPR_OTMARBINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ot_wdma)
        pie_common_reg = PIE_COMMON_IPR_OT_WDMAINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->wdma)
        pie_common_reg = PIE_COMMON_IPR_WDMAINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->ot_rdma)
        pie_common_reg = PIE_COMMON_IPR_OT_RDMAINTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->rdma2)
        pie_common_reg = PIE_COMMON_IPR_RDMA2INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->rdma1)
        pie_common_reg = PIE_COMMON_IPR_RDMA1INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    if (irqstruct->rdma0)
        pie_common_reg = PIE_COMMON_IPR_RDMA0INTPEND_REPLACE_VAL(pie_common_reg, set_val);
    
    return pie_common_reg;
}

uint32_t pie_odma_convert_irqstruct_to_update_udma_reg(struct odma_interrupt_info *irqstruct,
                                                       uint32_t udmareg,
                                                       bool  set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->Desc)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_DESC_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->ClearComplete)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_CLEARCOMPLETE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->Own)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_OWN_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->OutOfRangeErr)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_OUTOFRANGE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->RRespErr)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_RRESP_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->BRespErr)
        udmareg = POGO_ODMA_POGO_ODMA_UDMA_UIER_BRESP_REPLACE_VAL(udmareg, set_val);
    
    return udmareg;
}

uint32_t pie_odma_convert_irqstruct_to_update_core_reg(struct odma_interrupt_info *irqstruct,
                                                       uint32_t corereg,
                                                       bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->EndOfStrip)
        corereg = POGO_ODMA_POGO_ODMA_CORE_OIER_EOS_REPLACE_VAL(corereg, set_val);
    if (irqstruct->LengthErr)
        corereg = POGO_ODMA_POGO_ODMA_CORE_OIER_LENGTHERR_REPLACE_VAL(corereg, set_val);

    return corereg;
}

// OT odma functions
uint32_t pie_otodma_convert_irqstruct_to_update_udma_reg(struct odma_interrupt_info *irqstruct,
                                                         uint32_t udmareg,
                                                         bool  set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->Desc)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_DESC_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->ClearComplete)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_CLEARCOMPLETE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->Own)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_OWN_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->OutOfRangeErr)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_OUTOFRANGE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->RRespErr)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_RRESP_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->BRespErr)
        udmareg = POGO_OTODMA_POGO_ODMA_UDMA_UIER_BRESP_REPLACE_VAL(udmareg, set_val);
    
    return udmareg;
}

uint32_t pie_otodma_convert_irqstruct_to_update_core_reg(struct odma_interrupt_info *irqstruct,
                                                         uint32_t corereg,
                                                         bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->EndOfStrip)
        corereg = POGO_OTODMA_POGO_ODMA_CORE_OIER_EOS_REPLACE_VAL(corereg, set_val);
    if (irqstruct->LengthErr)
        corereg = POGO_OTODMA_POGO_ODMA_CORE_OIER_LENGTHERR_REPLACE_VAL(corereg, set_val);

    return corereg;
}

// idma functions
uint32_t pie_idma_convert_irqstruct_to_update_udma_reg(struct idma_interrupt_info *irqstruct,
                                                       uint32_t udmareg,
                                                       bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->Desc)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_DESC_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->ClearComplete)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_CLEARCOMPLETE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->Own)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_OWN_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->OutOfRangeErr)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_OUTOFRANGE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->RRespErr)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_RRESP_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->BRespErr)
        udmareg = POGO_IDMA0_POGO_IDMA_UDMA_UIER_BRESP_REPLACE_VAL(udmareg, set_val);
    
    return udmareg;
}

uint32_t pie_idma_convert_irqstruct_to_update_core_reg(struct idma_interrupt_info *irqstruct,
                                                       uint32_t corereg,
                                                       bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->EndOfStrip)
        corereg = POGO_IDMA0_POGO_IDMA_CORE_IIER_EOS_REPLACE_VAL(corereg, set_val);
    if (irqstruct->LengthErr)
        corereg = POGO_IDMA0_POGO_IDMA_CORE_IIER_LENGTHERR_REPLACE_VAL(corereg, set_val);
    
    return corereg;
}


// OT idma functions
uint32_t pie_otidma_convert_irqstruct_to_update_udma_reg(struct idma_interrupt_info *irqstruct,
                                                         uint32_t udmareg,
                                                         bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->Desc)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_DESC_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->ClearComplete)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_CLEARCOMPLETE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->Own)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_OWN_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->OutOfRangeErr)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_OUTOFRANGE_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->RRespErr)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_RRESP_REPLACE_VAL(udmareg, set_val);
    if (irqstruct->BRespErr)
        udmareg = OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_BRESP_REPLACE_VAL(udmareg, set_val);
    
    return udmareg;
}

uint32_t pie_otidma_convert_irqstruct_to_update_core_reg(struct idma_interrupt_info *irqstruct,
                                                         uint32_t corereg,
                                                         bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;
    
    if (irqstruct->EndOfStrip)
        corereg = OT_POGO_IDMA_POGO_IDMA_CORE_IIER_EOS_REPLACE_VAL(corereg, set_val);
    if (irqstruct->LengthErr)
        corereg = OT_POGO_IDMA_POGO_IDMA_CORE_IIER_LENGTHERR_REPLACE_VAL(corereg, set_val);
    
    return corereg;
}

uint32_t pie_distort_convert_irqstruct_to_update_reg(struct idma_interrupt_info *irqstruct,
                                                     uint32_t reg,
                                                     bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;

    if (irqstruct->Own)
        reg = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIER_OWN_REPLACE_VAL(reg, set_val);
    if (irqstruct->Desc)
        reg = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIER_DESC_REPLACE_VAL(reg, set_val);
    if (irqstruct->ClearComplete)
        reg = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIER_CLEARCOMPLETE_REPLACE_VAL(reg, set_val);
    if (irqstruct->RRespErr)
        reg = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIER_RRESP_REPLACE_VAL(reg, set_val);
    if (irqstruct->BRespErr)
        reg = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIER_BRESP_REPLACE_VAL(reg, set_val);

    return reg;
}

uint32_t pie_otmarb_convert_irqstruct_to_update_reg(struct otmarb_interrupt_info *irqstruct,
                                                    uint32_t reg,
                                                    bool set_true_clear_false)
{
    uint8_t set_val;

    set_val = set_true_clear_false?1:0;

    if (irqstruct->DTmismatchErr)
        reg = OTMARB_INTSTAT_MISMATCHINTPEND_REPLACE_VAL(reg, set_val);

    return reg;
}

//------------------------------------------------------------------------------
// Convert register bits to irqstructs 
//------------------------------------------------------------------------------

void pie_common_convert_uintarray_to_irqstruct(struct pie_common_ints *irqstruct,
                                               uint32_t irqs)
{
    irqstruct->distort = (PIE_COMMON_IPR_DISTORTINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ihi_16x8_short = (PIE_COMMON_IPR_IHI16X8SHORTINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ihi_16x8_tall = (PIE_COMMON_IPR_IHI16X8TALLINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ihi_3x3_short = (PIE_COMMON_IPR_IHI3X3SHORTINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ihi_3x3_tall = (PIE_COMMON_IPR_IHI3X3TALLINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->pack_mono_wdma = (PIE_COMMON_IPR_PACKMONO_WDMAINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->dis_map = (PIE_COMMON_IPR_DIS_MAPINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_corr_rdma2 = (PIE_COMMON_IPR_AC_CORR_RDMA2INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_corr_rdma1 = (PIE_COMMON_IPR_AC_CORR_RDMA1INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_corr_rdma0 = (PIE_COMMON_IPR_AC_CORR_RDMA0INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_data_rdma2 = (PIE_COMMON_IPR_AC_DATA_RDMA2INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_data_rdma1 = (PIE_COMMON_IPR_AC_DATA_RDMA1INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ac_data_rdma0 = (PIE_COMMON_IPR_AC_DATA_RDMA0INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->dsmf_short = (PIE_COMMON_IPR_DSMFSHORTINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->dsmf_tall = (PIE_COMMON_IPR_DSMFTALLINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->otmarb = (PIE_COMMON_IPR_OTMARBINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ot_wdma = (PIE_COMMON_IPR_OT_WDMAINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->wdma = (PIE_COMMON_IPR_WDMAINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->ot_rdma = (PIE_COMMON_IPR_OT_RDMAINTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->rdma2 = (PIE_COMMON_IPR_RDMA2INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->rdma1 = (PIE_COMMON_IPR_RDMA1INTPEND_MASK_SHIFT(irqs) != 0);
    irqstruct->rdma0 = (PIE_COMMON_IPR_RDMA0INTPEND_MASK_SHIFT(irqs) != 0);
}

void pie_odma_convert_udma_uintarray_to_irqstruct(struct odma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
                                                  
{
    irqstruct->Desc = (POGO_ODMA_POGO_ODMA_UDMA_UIER_DESC_MASK_SHIFT(irqs) != 0);
    irqstruct->ClearComplete = (POGO_ODMA_POGO_ODMA_UDMA_UIER_CLEARCOMPLETE_MASK_SHIFT(irqs) != 0);
    irqstruct->Own = (POGO_ODMA_POGO_ODMA_UDMA_UIER_OWN_MASK_SHIFT(irqs) != 0);
    irqstruct->OutOfRangeErr = (POGO_ODMA_POGO_ODMA_UDMA_UIER_OUTOFRANGE_MASK_SHIFT(irqs) != 0);
    irqstruct->RRespErr = (POGO_ODMA_POGO_ODMA_UDMA_UIER_RRESP_MASK_SHIFT(irqs) != 0);
    irqstruct->BRespErr = (POGO_ODMA_POGO_ODMA_UDMA_UIER_BRESP_MASK_SHIFT(irqs) != 0);
}

    
void pie_odma_convert_core_uintarray_to_irqstruct(struct odma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
{
    irqstruct->EndOfStrip = (POGO_ODMA_POGO_ODMA_CORE_OIER_EOS_MASK_SHIFT(irqs) != 0);
    irqstruct->LengthErr = (POGO_ODMA_POGO_ODMA_CORE_OIER_LENGTHERR_MASK_SHIFT(irqs) != 0);
}

// OT odma functions
void pie_otodma_convert_udma_uintarray_to_irqstruct(struct odma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
                                                  
{
    irqstruct->Desc = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_DESC_MASK_SHIFT(irqs) != 0);
    irqstruct->ClearComplete = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_CLEARCOMPLETE_MASK_SHIFT(irqs) != 0);
    irqstruct->Own = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_OWN_MASK_SHIFT(irqs) != 0);
    irqstruct->OutOfRangeErr = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_OUTOFRANGE_MASK_SHIFT(irqs) != 0);
    irqstruct->RRespErr = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_RRESP_MASK_SHIFT(irqs) != 0);
    irqstruct->BRespErr = (POGO_OTODMA_POGO_ODMA_UDMA_UIER_BRESP_MASK_SHIFT(irqs) != 0);
}

void pie_otodma_convert_core_uintarray_to_irqstruct(struct odma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
{
    irqstruct->EndOfStrip = (POGO_OTODMA_POGO_ODMA_CORE_OIER_EOS_MASK_SHIFT(irqs) != 0);
    irqstruct->LengthErr = (POGO_OTODMA_POGO_ODMA_CORE_OIER_LENGTHERR_MASK_SHIFT(irqs) != 0);
}

// idma functions

void pie_idma_convert_udma_uintarray_to_irqstruct(struct idma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
{
    irqstruct->Desc = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_DESC_MASK_SHIFT(irqs) != 0);
    irqstruct->ClearComplete = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_CLEARCOMPLETE_MASK_SHIFT(irqs) != 0);
    irqstruct->Own = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_OWN_MASK_SHIFT(irqs) != 0);
    irqstruct->OutOfRangeErr = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_OUTOFRANGE_MASK_SHIFT(irqs) != 0);
    irqstruct->RRespErr = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_RRESP_MASK_SHIFT(irqs) != 0);
    irqstruct->BRespErr = (POGO_IDMA0_POGO_IDMA_UDMA_UIER_BRESP_MASK_SHIFT(irqs) != 0);
}

void pie_idma_convert_core_uintarray_to_irqstruct(struct idma_interrupt_info *irqstruct,
                                                  uint32_t irqs)
{
    irqstruct->EndOfStrip = (POGO_IDMA0_POGO_IDMA_CORE_IIER_EOS_MASK_SHIFT(irqs) != 0);
    irqstruct->LengthErr = (POGO_IDMA0_POGO_IDMA_CORE_IIER_LENGTHERR_MASK_SHIFT(irqs) != 0);
}

// OT idma functions

void pie_otidma_convert_udma_uintarray_to_irqstruct(struct idma_interrupt_info *irqstruct,
                                                    uint32_t irqs)
{
    irqstruct->Desc = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_DESC_MASK_SHIFT(irqs) != 0);
    irqstruct->ClearComplete = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_CLEARCOMPLETE_MASK_SHIFT(irqs) != 0);
    irqstruct->Own = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_OWN_MASK_SHIFT(irqs) != 0);
    irqstruct->OutOfRangeErr = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_OUTOFRANGE_MASK_SHIFT(irqs) != 0);
    irqstruct->RRespErr = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_RRESP_MASK_SHIFT(irqs) != 0);
    irqstruct->BRespErr = (OT_POGO_IDMA_POGO_IDMA_UDMA_UIER_BRESP_MASK_SHIFT(irqs) != 0);
}

void pie_otidma_convert_core_uintarray_to_irqstruct(struct idma_interrupt_info *irqstruct,
                                                    uint32_t irqs)
{
    irqstruct->EndOfStrip = (OT_POGO_IDMA_POGO_IDMA_CORE_IIER_EOS_MASK_SHIFT(irqs) != 0);
    irqstruct->LengthErr = (OT_POGO_IDMA_POGO_IDMA_CORE_IIER_LENGTHERR_MASK_SHIFT(irqs) != 0);
}

void pie_otmarb_convert_uintarray_to_irqstruct(struct otmarb_interrupt_info *irqstruct,
                                               uint32_t irqs)
{
    irqstruct->DTmismatchErr = (OTMARB_INTSTAT_MISMATCHINTPEND_MASK_SHIFT(irqs) != 0);
}

#define COMPUTE_SHADOW_POINTER(pointer_name, subblock_index)            \
    shadow_size = pie_handle->subblock_size_table[subblock_index];      \
    if (shadow_size == 0)                                               \
    {                                                                   \
        /* this block was never registered - mark it NULL */            \
        pie_handle->pointer_name = NULL;                                \
    }                                                                   \
    else                                                                \
    {                                                                   \
        pie_handle->pointer_name = (void *) shadow_reg_addr;            \
        /* compute the next subblock of shadow registers */             \
        shadow_reg_addr += shadow_size;                                 \
    }

        


// create the pointers into the shadow register space from the subblock sizes table
void fixup_shadow_pointers(struct pie_handle_t *pie_handle)
{
    int shadow_size;  // size in bytes of the subblock shadow registers
    // 8 bit pointer for pointer math since shadow_size is in bytes    
    uint8_t *shadow_reg_addr; // current address of shadow registers

    // Set the shadow_regs pointer to point to the next word address
    // Since we allocated the size of the struct + size of the shadow registers,
    // the structure's shadow_regs pointer is the last word in the structure
    // before the actual registers - so point this at them (pointer math ++ uint32)
    pie_handle->shadow_regs = (uint32_t *)  &(pie_handle->shadow_regs);
    pie_handle->shadow_regs++;  // point at the word after my field    
    // the start of the shadow registers - used by COMPUTE_SHADOW_POINTER
    shadow_reg_addr = (uint8_t *) pie_handle->shadow_regs;

    COMPUTE_SHADOW_POINTER(pie_common, common_index);
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_udma[0], pogo_idma0_udma_index);
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_udma[1], pogo_idma1_udma_index);
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_udma[2], pogo_idma2_udma_index);    
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_core[0], pogo_idma0_core_index);
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_core[1], pogo_idma1_core_index);
    COMPUTE_SHADOW_POINTER(pie_pogo_idma_core[2], pogo_idma2_core_index);
    COMPUTE_SHADOW_POINTER(pie_pogoizer, pogoizer_index);
    COMPUTE_SHADOW_POINTER(pie_depogoizer, depogoizer_index);    
    COMPUTE_SHADOW_POINTER(pie_pogo_odma_udma, pogo_odma_udma_index);    
    COMPUTE_SHADOW_POINTER(pie_pogo_odma_core, pogo_odma_core_index);    
    COMPUTE_SHADOW_POINTER(pie_otpogo_idma_udma, ot_pogo_idma_udma_index);
    COMPUTE_SHADOW_POINTER(pie_otpogo_idma_core, ot_pogo_idma_core_index);
    COMPUTE_SHADOW_POINTER(pie_otpogoizer, otpogoizer_index);
    COMPUTE_SHADOW_POINTER(pie_colorshift, colorshift_index);
    COMPUTE_SHADOW_POINTER(pie_rgb2esrgb, lut1d_rgb2esrgb_index);
    COMPUTE_SHADOW_POINTER(pie_sccsc, sccsc_index);
    COMPUTE_SHADOW_POINTER(pie_rgb2ycc, es2y_index);
    COMPUTE_SHADOW_POINTER(pie_cstats, cstats_index);
    COMPUTE_SHADOW_POINTER(pie_otmarb, otmarb_index);
    COMPUTE_SHADOW_POINTER(pie_ngadjust, ngadjust_index);
    COMPUTE_SHADOW_POINTER(pie_istone, istone_index);
    COMPUTE_SHADOW_POINTER(pie_tcns, tcnsense_index);
    COMPUTE_SHADOW_POINTER(pie_denoise, denoise_index);
    COMPUTE_SHADOW_POINTER(pie_dsmf, dsmf_index);
    COMPUTE_SHADOW_POINTER(pie_decim, decim_index);
    COMPUTE_SHADOW_POINTER(pie_ycc2rgb, y2es_index);    
    COMPUTE_SHADOW_POINTER(pie_xyscale, xyscale_index);
    COMPUTE_SHADOW_POINTER(pie_clippad, clippad_index);    
    COMPUTE_SHADOW_POINTER(pie_otdepogoizer, otdepogoizer_index);
    COMPUTE_SHADOW_POINTER(pie_otpogo_odma_udma, pogo_otodma_udma_index);
    COMPUTE_SHADOW_POINTER(pie_otpogo_odma_core, pogo_otodma_core_index);    
    COMPUTE_SHADOW_POINTER(pie_bde[0], lut1d_bde0_index);
    COMPUTE_SHADOW_POINTER(pie_bde[1], lut1d_bde1_index);
    COMPUTE_SHADOW_POINTER(pie_bde[2], lut1d_bde2_index);        
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_top, antifcor_index);
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_data0, ddma_ac_data0_index);
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_data1, ddma_ac_data1_index);
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_data2, ddma_ac_data2_index);    
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_corr0, ddma_ac_corr0_index);
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_corr1, ddma_ac_corr1_index);
    COMPUTE_SHADOW_POINTER(pie_distort_antifcor_ddma_ac_corr2, ddma_ac_corr2_index);    
    COMPUTE_SHADOW_POINTER(pie_distort_top, distort_index);
    COMPUTE_SHADOW_POINTER(pie_distort_top_map, dismap_index);    
    COMPUTE_SHADOW_POINTER(pie_xy_c_scale, xy_c_scale_index);
    COMPUTE_SHADOW_POINTER(pie_ihi, ihi_index);
    COMPUTE_SHADOW_POINTER(pie_ihi_depogoizer, ihidepogoizer_index);
    COMPUTE_SHADOW_POINTER(pie_ihi_pogo_odma_udma, pogo_ihi_odma_udma_index);
    COMPUTE_SHADOW_POINTER(pie_ihi_pogo_odma_core, pogo_ihi_odma_core_index);
}

// function to get the subblock offsets array from the driver.  If this is called from
// kernel space, it just calls the kernel function directly to get the data.  If this is
// called from user space, it will use the sysfs to get the data
uint32_t *get_subblock_sizes(void)
{
    uint32_t *subblock_array;
    int retval;
    
   // Memory will be allocated for the subblock array.  It needs to be
    // deallocated when pie_handle is created (see pie_create_new_default_handle)

#ifdef __KERNEL__
    
    // since we are in the kernel, we can just call the kernel function
    retval = pie_get_subblock_sizes_array(&subblock_array);
    if (retval == -ENODEV)
    {
        error_print("pie_get_subblock_sizes_array returned No such device - check that the PIE driver loaded.\n");
        BUG_ON(1);
    }
    if (retval == -ENOMEM)
    {
        error_print("%s: can not continue if we cannot allocate memory for the subblock\n", __func__);
        BUG_ON(1);
    }
#else
#define PATH_TO_PIEHANDLE "/sys/devices/f9400000.pie/subblock_sizes_array"
#define PATH_TO_PIEHANDLE64 "/sys/devices/f940000000000000.pie/subblock_sizes_array"
#define BUFSIZE_MAX 255    
    
    // read the data from the sysfs file
    char path[(sizeof PATH_TO_PIEHANDLE) + 1];
    int fd,  i;
    int readret;

    subblock_array = (uint32_t *) allocate_memory(BUFSIZE_MAX, NULL);

    fd = open(PATH_TO_PIEHANDLE, O_RDONLY);
    if (-1 == fd)
    {
        // try to open the 64-bit version
        fd = open(PATH_TO_PIEHANDLE64, O_RDONLY);
        if (-1 == fd)
        {
            error_print("Failed to open pie_handle (to get subblock) for reading!\n");
            return(NULL);
        }
    }
    
    readret = read(fd, subblock_array, BUFSIZE_MAX);
    
    if (-1 == readret)    
    {
	error_print("Failed to read value!\n");
	return(NULL);
    }

    // now read to verify we get a 0 back.  If we don't we have a problem, since
    // we used up our buffer already....
    if (0 != read(fd, subblock_array, BUFSIZE_MAX))
    {
        error_print("%s: warning: sysfs return bigger than expected: first read=%d, 2nd is non_zero\n",
                    __func__, readret);
    }
    close(fd);

#endif
    return subblock_array;
}

