/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h" 
#include "pie_driver.h"
#include "pie_distort.h"
#include "pie_distort_if.h"

#include "pie_driverlib_if.h"  // for print macros

/* ------------ DISTORT ------------ */

static void dump_distort_regs(distortData *device_data)
{
    print("PIE DISTORT Registers\n");
    printDistortReg(Cfg);
    printDistortReg(OutputStripStatus);
    printDistortReg(QueueHsStatus);
    printDistortReg(IntEnable);
    printDistortReg(IntPending);
    printDistortReg(Ch1_ImageDimension);
    printDistortReg(Ch2_ImageDimension);
    printDistortReg(Ch3_ImageDimension);
    printDistortReg(Ch1_DataStartPos_X);
    printDistortReg(Ch2_DataStartPos_X);
    printDistortReg(Ch3_DataStartPos_X);
    printDistortReg(Ch1_DataStartPos_Y);
    printDistortReg(Ch2_DataStartPos_Y);
    printDistortReg(Ch3_DataStartPos_Y);
    printDistortReg(Ch1_FixedRowStepX);
    printDistortReg(Ch2_FixedRowStepX);
    printDistortReg(Ch3_FixedRowStepX);
    printDistortReg(Ch1_FixedRowStepY);
    printDistortReg(Ch2_FixedRowStepY);
    printDistortReg(Ch3_FixedRowStepY);
    printDistortReg(Ch1_FixedColStepX);
    printDistortReg(Ch2_FixedColStepX);
    printDistortReg(Ch3_FixedColStepX);
    printDistortReg(Ch1_FixedColStepY);
    printDistortReg(Ch2_FixedColStepY);
    printDistortReg(Ch3_FixedColStepY);
    printDistortReg(Ch1_BlankValue);
    printDistortReg(Ch2_BlankValue);
    printDistortReg(Ch3_BlankValue);
    printDistortReg(Ch1_HighestLine);
    printDistortReg(Ch2_HighestLine);
    printDistortReg(Ch3_HighestLine);
    printDistortReg(CurrentPosition);
    printDistortReg(PosSelect);
    printDistortReg(REV0);
    printDistortReg(REV1);
    printDistortReg(HCL0);
    printDistortReg(HCH0);
    printDistortReg(HCL1);
    printDistortReg(HCH1);
    printDistortReg(HCL2);
    printDistortReg(HCH2);
    printDistortReg(HCL3);
    printDistortReg(HCH3);
    printDistortReg(HCL4);
    printDistortReg(HCH4);
    printDistortReg(HCL5);
    printDistortReg(HCH5);
    printDistortReg(HCL6);
    printDistortReg(HCH6);
    printDistortReg(HCL7);
    printDistortReg(HCH7);
    printDistortReg(HCL8);
    printDistortReg(HCH8);
    printDistortReg(HCL9);
    printDistortReg(HCH9);
    printDistortReg(HCL10);
    printDistortReg(HCH10);
    printDistortReg(HCL11);
    printDistortReg(HCH11);
    printDistortReg(HCL12);
    printDistortReg(HCH12);
    printDistortReg(HCL13);
    printDistortReg(HCH13);
    printDistortReg(HCL14);
    printDistortReg(HCH14);
    printDistortReg(HCL15);
    printDistortReg(HCH15);
    printDistortReg(VCL0);
    printDistortReg(VCH0);
    printDistortReg(VCL1);
    printDistortReg(VCH1);
    printDistortReg(VCL2);
    printDistortReg(VCH2);
    printDistortReg(VCL3);
    printDistortReg(VCH3);
    printDistortReg(VCL4);
    printDistortReg(VCH4);
    printDistortReg(VCL5);
    printDistortReg(VCH5);
    printDistortReg(VCL6);
    printDistortReg(VCH6);
    printDistortReg(VCL7);
    printDistortReg(VCH7);
    printDistortReg(VCL8);
    printDistortReg(VCH8);
    printDistortReg(VCL9);
    printDistortReg(VCH9);
    printDistortReg(VCL10);
    printDistortReg(VCH10);
    printDistortReg(VCL11);
    printDistortReg(VCH11);
    printDistortReg(VCL12);
    printDistortReg(VCH12);
    printDistortReg(VCL13);
    printDistortReg(VCH13);
    printDistortReg(VCL14);
    printDistortReg(VCH14);
    printDistortReg(VCL15);
    printDistortReg(VCH15);
}

static int pie_revcheck(distortData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0 = DISTORT_TOP_DISTORT_REV0_MAJ_MASK_SHIFT(distortRead(REV0));
    pieh_rev = DISTORT_TOP_DISTORT_REV0_MAJ_MASK_SHIFT(pie_handle->pie_distort_top->REV0);

    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(distortData *device_data)
{
    uint32_t reg;
    PROTECT_REG_ACCESS;
    reg = 0;
    reg = DISTORT_TOP_DISTORT_CFG_BYPASS_REPLACE_VAL(reg, 1);
    distortWrite(Cfg, 1);
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(distortData *device_data,
                          struct pie_handle_t *pie_handle)
{
    uint32_t reg;

    device_data->interrupt_callback = pie_handle->pie_idma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_idma_udma_callback_data;    

    PROTECT_REG_ACCESS;

    distortWrite(Ch1_ImageDimension, pie_handle->pie_distort_top->Ch1_ImageDimension);
    distortWrite(Ch2_ImageDimension, pie_handle->pie_distort_top->Ch2_ImageDimension);
    distortWrite(Ch3_ImageDimension, pie_handle->pie_distort_top->Ch3_ImageDimension);
    distortWrite(Ch1_DataStartPos_X, pie_handle->pie_distort_top->Ch1_DataStartPos_X);
    distortWrite(Ch2_DataStartPos_X, pie_handle->pie_distort_top->Ch2_DataStartPos_X);
    distortWrite(Ch3_DataStartPos_X, pie_handle->pie_distort_top->Ch3_DataStartPos_X);
    distortWrite(Ch1_DataStartPos_Y, pie_handle->pie_distort_top->Ch1_DataStartPos_Y);
    distortWrite(Ch2_DataStartPos_Y, pie_handle->pie_distort_top->Ch2_DataStartPos_Y);
    distortWrite(Ch3_DataStartPos_Y, pie_handle->pie_distort_top->Ch3_DataStartPos_Y);
    distortWrite(Ch1_FixedRowStepX, pie_handle->pie_distort_top->Ch1_FixedRowStepX);
    distortWrite(Ch2_FixedRowStepX, pie_handle->pie_distort_top->Ch2_FixedRowStepX);
    distortWrite(Ch3_FixedRowStepX, pie_handle->pie_distort_top->Ch3_FixedRowStepX);
    distortWrite(Ch1_FixedRowStepY, pie_handle->pie_distort_top->Ch1_FixedRowStepY);
    distortWrite(Ch2_FixedRowStepY, pie_handle->pie_distort_top->Ch2_FixedRowStepY);
    distortWrite(Ch3_FixedRowStepY, pie_handle->pie_distort_top->Ch3_FixedRowStepY);
    distortWrite(Ch1_FixedColStepX, pie_handle->pie_distort_top->Ch1_FixedColStepX);
    distortWrite(Ch2_FixedColStepX, pie_handle->pie_distort_top->Ch2_FixedColStepX);
    distortWrite(Ch3_FixedColStepX, pie_handle->pie_distort_top->Ch3_FixedColStepX);
    distortWrite(Ch1_FixedColStepY, pie_handle->pie_distort_top->Ch1_FixedColStepY);
    distortWrite(Ch2_FixedColStepY, pie_handle->pie_distort_top->Ch2_FixedColStepY);
    distortWrite(Ch3_FixedColStepY, pie_handle->pie_distort_top->Ch3_FixedColStepY);
    distortWrite(Ch1_BlankValue, pie_handle->pie_distort_top->Ch1_BlankValue);
    distortWrite(Ch2_BlankValue, pie_handle->pie_distort_top->Ch2_BlankValue);
    distortWrite(Ch3_BlankValue, pie_handle->pie_distort_top->Ch3_BlankValue);
    distortWrite(PosSelect, pie_handle->pie_distort_top->PosSelect);
    distortWrite(HCL0, pie_handle->pie_distort_top->HCL0);
    distortWrite(HCH0, pie_handle->pie_distort_top->HCH0);
    distortWrite(HCL1, pie_handle->pie_distort_top->HCL1);
    distortWrite(HCH1, pie_handle->pie_distort_top->HCH1);
    distortWrite(HCL2, pie_handle->pie_distort_top->HCL2);
    distortWrite(HCH2, pie_handle->pie_distort_top->HCH2);
    distortWrite(HCL3, pie_handle->pie_distort_top->HCL3);
    distortWrite(HCH3, pie_handle->pie_distort_top->HCH3);
    distortWrite(HCL4, pie_handle->pie_distort_top->HCL4);
    distortWrite(HCH4, pie_handle->pie_distort_top->HCH4);
    distortWrite(HCL5, pie_handle->pie_distort_top->HCL5);
    distortWrite(HCH5, pie_handle->pie_distort_top->HCH5);
    distortWrite(HCL6, pie_handle->pie_distort_top->HCL6);
    distortWrite(HCH6, pie_handle->pie_distort_top->HCH6);
    distortWrite(HCL7, pie_handle->pie_distort_top->HCL7);
    distortWrite(HCH7, pie_handle->pie_distort_top->HCH7);
    distortWrite(HCL8, pie_handle->pie_distort_top->HCL8);
    distortWrite(HCH8, pie_handle->pie_distort_top->HCH8);
    distortWrite(HCL9, pie_handle->pie_distort_top->HCL9);
    distortWrite(HCH9, pie_handle->pie_distort_top->HCH9);
    distortWrite(HCL10, pie_handle->pie_distort_top->HCL10);
    distortWrite(HCH10, pie_handle->pie_distort_top->HCH10);
    distortWrite(HCL11, pie_handle->pie_distort_top->HCL11);
    distortWrite(HCH11, pie_handle->pie_distort_top->HCH11);
    distortWrite(HCL12, pie_handle->pie_distort_top->HCL12);
    distortWrite(HCH12, pie_handle->pie_distort_top->HCH12);
    distortWrite(HCL13, pie_handle->pie_distort_top->HCL13);
    distortWrite(HCH13, pie_handle->pie_distort_top->HCH13);
    distortWrite(HCL14, pie_handle->pie_distort_top->HCL14);
    distortWrite(HCH14, pie_handle->pie_distort_top->HCH14);
    distortWrite(HCL15, pie_handle->pie_distort_top->HCL15);
    distortWrite(HCH15, pie_handle->pie_distort_top->HCH15);
    distortWrite(VCL0, pie_handle->pie_distort_top->VCL0);
    distortWrite(VCH0, pie_handle->pie_distort_top->VCH0);
    distortWrite(VCL1, pie_handle->pie_distort_top->VCL1);
    distortWrite(VCH1, pie_handle->pie_distort_top->VCH1);
    distortWrite(VCL2, pie_handle->pie_distort_top->VCL2);
    distortWrite(VCH2, pie_handle->pie_distort_top->VCH2);
    distortWrite(VCL3, pie_handle->pie_distort_top->VCL3);
    distortWrite(VCH3, pie_handle->pie_distort_top->VCH3);
    distortWrite(VCL4, pie_handle->pie_distort_top->VCL4);
    distortWrite(VCH4, pie_handle->pie_distort_top->VCH4);
    distortWrite(VCL5, pie_handle->pie_distort_top->VCL5);
    distortWrite(VCH5, pie_handle->pie_distort_top->VCH5);
    distortWrite(VCL6, pie_handle->pie_distort_top->VCL6);
    distortWrite(VCH6, pie_handle->pie_distort_top->VCH6);
    distortWrite(VCL7, pie_handle->pie_distort_top->VCL7);
    distortWrite(VCH7, pie_handle->pie_distort_top->VCH7);
    distortWrite(VCL8, pie_handle->pie_distort_top->VCL8);
    distortWrite(VCH8, pie_handle->pie_distort_top->VCH8);
    distortWrite(VCL9, pie_handle->pie_distort_top->VCL9);
    distortWrite(VCH9, pie_handle->pie_distort_top->VCH9);
    distortWrite(VCL10, pie_handle->pie_distort_top->VCL10);
    distortWrite(VCH10, pie_handle->pie_distort_top->VCH10);
    distortWrite(VCL11, pie_handle->pie_distort_top->VCL11);
    distortWrite(VCH11, pie_handle->pie_distort_top->VCH11);
    distortWrite(VCL12, pie_handle->pie_distort_top->VCL12);
    distortWrite(VCH12, pie_handle->pie_distort_top->VCH12);
    distortWrite(VCL13, pie_handle->pie_distort_top->VCL13);
    distortWrite(VCH13, pie_handle->pie_distort_top->VCH13);
    distortWrite(VCL14, pie_handle->pie_distort_top->VCL14);
    distortWrite(VCH14, pie_handle->pie_distort_top->VCH14);
    distortWrite(VCL15, pie_handle->pie_distort_top->VCL15);
    distortWrite(VCH15, pie_handle->pie_distort_top->VCH15);

    /*
     * Cfg must be written last. The DataStart registers (and possibly others)
     * do not take affect if written after Cfg.
     */
    distortWrite(Cfg, pie_handle->pie_distort_top->Cfg);

    /* Enable the Distort interrupts */
    reg = 0;
    reg = DISTORT_TOP_DISTORT_INTENABLE_ENDOFSTRIP_REPLACE_VAL(reg, 1);
    distortWrite(IntEnable, reg);

    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(distortData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_distort_top->Cfg = distortRead(Cfg);
    pie_handle->pie_distort_top->OutputStripStatus = distortRead(OutputStripStatus);
    pie_handle->pie_distort_top->IntEnable = distortRead(IntEnable);
    pie_handle->pie_distort_top->IntPending = distortRead(IntPending);
    pie_handle->pie_distort_top->Ch1_ImageDimension = distortRead(Ch1_ImageDimension);
    pie_handle->pie_distort_top->Ch2_ImageDimension = distortRead(Ch2_ImageDimension);
    pie_handle->pie_distort_top->Ch3_ImageDimension = distortRead(Ch3_ImageDimension);
    pie_handle->pie_distort_top->Ch1_DataStartPos_X = distortRead(Ch1_DataStartPos_X);
    pie_handle->pie_distort_top->Ch2_DataStartPos_X = distortRead(Ch2_DataStartPos_X);
    pie_handle->pie_distort_top->Ch3_DataStartPos_X = distortRead(Ch3_DataStartPos_X);
    pie_handle->pie_distort_top->Ch1_DataStartPos_Y = distortRead(Ch1_DataStartPos_Y);
    pie_handle->pie_distort_top->Ch2_DataStartPos_Y = distortRead(Ch2_DataStartPos_Y);
    pie_handle->pie_distort_top->Ch3_DataStartPos_Y = distortRead(Ch3_DataStartPos_Y);
    pie_handle->pie_distort_top->Ch1_FixedRowStepX = distortRead(Ch1_FixedRowStepX);
    pie_handle->pie_distort_top->Ch2_FixedRowStepX = distortRead(Ch2_FixedRowStepX);
    pie_handle->pie_distort_top->Ch3_FixedRowStepX = distortRead(Ch3_FixedRowStepX);
    pie_handle->pie_distort_top->Ch1_FixedRowStepY = distortRead(Ch1_FixedRowStepY);
    pie_handle->pie_distort_top->Ch2_FixedRowStepY = distortRead(Ch2_FixedRowStepY);
    pie_handle->pie_distort_top->Ch3_FixedRowStepY = distortRead(Ch3_FixedRowStepY);
    pie_handle->pie_distort_top->Ch1_FixedColStepX = distortRead(Ch1_FixedColStepX);
    pie_handle->pie_distort_top->Ch2_FixedColStepX = distortRead(Ch2_FixedColStepX);
    pie_handle->pie_distort_top->Ch3_FixedColStepX = distortRead(Ch3_FixedColStepX);
    pie_handle->pie_distort_top->Ch1_FixedColStepY = distortRead(Ch1_FixedColStepY);
    pie_handle->pie_distort_top->Ch2_FixedColStepY = distortRead(Ch2_FixedColStepY);
    pie_handle->pie_distort_top->Ch3_FixedColStepY = distortRead(Ch3_FixedColStepY);
    pie_handle->pie_distort_top->Ch1_BlankValue = distortRead(Ch1_BlankValue);
    pie_handle->pie_distort_top->Ch2_BlankValue = distortRead(Ch2_BlankValue);
    pie_handle->pie_distort_top->Ch3_BlankValue = distortRead(Ch3_BlankValue);
    pie_handle->pie_distort_top->Ch1_HighestLine = distortRead(Ch1_HighestLine);
    pie_handle->pie_distort_top->Ch2_HighestLine = distortRead(Ch2_HighestLine);
    pie_handle->pie_distort_top->Ch3_HighestLine = distortRead(Ch3_HighestLine);
    pie_handle->pie_distort_top->CurrentPosition = distortRead(CurrentPosition);
    pie_handle->pie_distort_top->PosSelect = distortRead(PosSelect);
    pie_handle->pie_distort_top->REV0 = distortRead(REV0);
    pie_handle->pie_distort_top->REV1 = distortRead(REV1);
    pie_handle->pie_distort_top->HCL0 = distortRead(HCL0);
    pie_handle->pie_distort_top->HCH0 = distortRead(HCH0);
    pie_handle->pie_distort_top->HCL1 = distortRead(HCL1);
    pie_handle->pie_distort_top->HCH1 = distortRead(HCH1);
    pie_handle->pie_distort_top->HCL2 = distortRead(HCL2);
    pie_handle->pie_distort_top->HCH2 = distortRead(HCH2);
    pie_handle->pie_distort_top->HCL3 = distortRead(HCL3);
    pie_handle->pie_distort_top->HCH3 = distortRead(HCH3);
    pie_handle->pie_distort_top->HCL4 = distortRead(HCL4);
    pie_handle->pie_distort_top->HCH4 = distortRead(HCH4);
    pie_handle->pie_distort_top->HCL5 = distortRead(HCL5);
    pie_handle->pie_distort_top->HCH5 = distortRead(HCH5);
    pie_handle->pie_distort_top->HCL6 = distortRead(HCL6);
    pie_handle->pie_distort_top->HCH6 = distortRead(HCH6);
    pie_handle->pie_distort_top->HCL7 = distortRead(HCL7);
    pie_handle->pie_distort_top->HCH7 = distortRead(HCH7);
    pie_handle->pie_distort_top->HCL8 = distortRead(HCL8);
    pie_handle->pie_distort_top->HCH8 = distortRead(HCH8);
    pie_handle->pie_distort_top->HCL9 = distortRead(HCL9);
    pie_handle->pie_distort_top->HCH9 = distortRead(HCH9);
    pie_handle->pie_distort_top->HCL10 = distortRead(HCL10);
    pie_handle->pie_distort_top->HCH10 = distortRead(HCH10);
    pie_handle->pie_distort_top->HCL11 = distortRead(HCL11);
    pie_handle->pie_distort_top->HCH11 = distortRead(HCH11);
    pie_handle->pie_distort_top->HCL12 = distortRead(HCL12);
    pie_handle->pie_distort_top->HCH12 = distortRead(HCH12);
    pie_handle->pie_distort_top->HCL13 = distortRead(HCL13);
    pie_handle->pie_distort_top->HCH13 = distortRead(HCH13);
    pie_handle->pie_distort_top->HCL14 = distortRead(HCL14);
    pie_handle->pie_distort_top->HCH14 = distortRead(HCH14);
    pie_handle->pie_distort_top->HCL15 = distortRead(HCL15);
    pie_handle->pie_distort_top->HCH15 = distortRead(HCH15);
    pie_handle->pie_distort_top->VCL0 = distortRead(VCL0);
    pie_handle->pie_distort_top->VCH0 = distortRead(VCH0);
    pie_handle->pie_distort_top->VCL1 = distortRead(VCL1);
    pie_handle->pie_distort_top->VCH1 = distortRead(VCH1);
    pie_handle->pie_distort_top->VCL2 = distortRead(VCL2);
    pie_handle->pie_distort_top->VCH2 = distortRead(VCH2);
    pie_handle->pie_distort_top->VCL3 = distortRead(VCL3);
    pie_handle->pie_distort_top->VCH3 = distortRead(VCH3);
    pie_handle->pie_distort_top->VCL4 = distortRead(VCL4);
    pie_handle->pie_distort_top->VCH4 = distortRead(VCH4);
    pie_handle->pie_distort_top->VCL5 = distortRead(VCL5);
    pie_handle->pie_distort_top->VCH5 = distortRead(VCH5);
    pie_handle->pie_distort_top->VCL6 = distortRead(VCL6);
    pie_handle->pie_distort_top->VCH6 = distortRead(VCH6);
    pie_handle->pie_distort_top->VCL7 = distortRead(VCL7);
    pie_handle->pie_distort_top->VCH7 = distortRead(VCH7);
    pie_handle->pie_distort_top->VCL8 = distortRead(VCL8);
    pie_handle->pie_distort_top->VCH8 = distortRead(VCH8);
    pie_handle->pie_distort_top->VCL9 = distortRead(VCL9);
    pie_handle->pie_distort_top->VCH9 = distortRead(VCH9);
    pie_handle->pie_distort_top->VCL10 = distortRead(VCL10);
    pie_handle->pie_distort_top->VCH10 = distortRead(VCH10);
    pie_handle->pie_distort_top->VCL11 = distortRead(VCL11);
    pie_handle->pie_distort_top->VCH11 = distortRead(VCH11);
    pie_handle->pie_distort_top->VCL12 = distortRead(VCL12);
    pie_handle->pie_distort_top->VCH12 = distortRead(VCH12);
    pie_handle->pie_distort_top->VCL13 = distortRead(VCL13);
    pie_handle->pie_distort_top->VCH13 = distortRead(VCH13);
    pie_handle->pie_distort_top->VCL14 = distortRead(VCL14);
    pie_handle->pie_distort_top->VCH14 = distortRead(VCH14);
    pie_handle->pie_distort_top->VCL15 = distortRead(VCL15);
    pie_handle->pie_distort_top->VCH15 = distortRead(VCH15);
}

static void handle_distort_irqs(distortData *device_data)
{
    uint32_t int_val;
    uint32_t ch[3];
    unsigned long flags;

    PROTECT_INTREG_ACCESS_IRQ;
    /* get and clear interrupts */
    int_val = distortRead(IntPending);
    distortWrite(IntClear, int_val);
    /* get highest line */
    ch[0] = distortRead(Ch1_HighestLine);
    ch[1] = distortRead(Ch2_HighestLine);
    ch[2] = distortRead(Ch3_HighestLine);
    UNPROTECT_INTREG_ACCESS_IRQ;

    if (device_data->interrupt_callback != NULL)
    {
        struct idma_interrupt_info irqstruct = {0};
        int i;
        irqstruct.instance = device_data->submodinstance;
        irqstruct.EndOfStrip = (int_val & DISTORT_TOP_DISTORT_INTPENDING_ENDOFSTRIP_MASK);
        for (i = 0; i < 3; i++)
        {
            irqstruct.highest_line[i] = DISTORT_TOP_DISTORT_CH1_HIGHESTLINE_HIGHESTLINE_MASK_SHIFT(ch[i]);
            irqstruct.highest_line_experimental[i] = DISTORT_TOP_DISTORT_CH1_HIGHESTLINE_HIGHESTLINEEXP_MASK_SHIFT(ch[i]);
        }
        device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback_data);
    }
}

static void start_pie_distort(distortData *device_data,
                              struct pie_handle_t *pie_handle)
{
    uint32_t reg = pie_handle->pie_distort_top->OutputStripConfig;
    reg = DISTORT_TOP_DISTORT_OUTPUTSTRIPCONFIG_INTAFTER_REPLACE_VAL(reg, 1);
    distortWrite(OutputStripConfig, reg);
}

struct pie_distort_function_struct pie_distort_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,
    
    .handle_distort_irqs = handle_distort_irqs,
    .start_pie_distort = start_pie_distort,
    .dump_distort_regs = dump_distort_regs,
};

void pie_distort_init(distortData *device_data)
{
    distortDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(distortDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_distort_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(distort, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));

    // distort has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("DISTORT_INIT LOADED.........\n");
}
EXPORT_SYMBOL(pie_distort_init);

void pie_distort_exit(distortData *device_data)
{
    distortDeviceHandle *pie_device_handle;

    print("DISTORT_INIT EXIT.........\n");
    pie_device_handle = unregister_pie_subblock(distort);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_distort_exit);



/* ------------ DISTORT DMA ------------ */

/*
 * Cast the instance to DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t since
 * they are all the same anyway.
 */
static DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *get_ddma_regs(struct pie_handle_t *pie_handle, int instance)
{
    switch (instance)
    {
    case 0: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data0;
    case 1: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data1;
    case 2: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data2;
    default: BUG();
    }
}

// this function runs in interrupt context - no long operations
static void handle_distort_idma_irqs(ddmaAcDataData *device_data)
{
    uint32_t int_val;
    unsigned long flags;

    // get and clear interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    int_val = distortDmaRead(UIPR);
    distortDmaWrite(UICR, int_val);
    UNPROTECT_INTREG_ACCESS_IRQ;

    if (device_data->interrupt_callback != NULL)
    {
        struct idma_interrupt_info irqstruct = {0};
        irqstruct.instance = device_data->submodinstance;
        irqstruct.Desc = (int_val & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIPR_DESC_MASK);
        irqstruct.ClearComplete = (int_val & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIPR_CLEARCOMPLETE_MASK);
        irqstruct.Own = (int_val & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIPR_OWN_MASK);
        irqstruct.RRespErr = (int_val & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIPR_RRESP_MASK);
        irqstruct.BRespErr = (int_val & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIPR_BRESP_MASK);
        device_data->interrupt_callback((void *) &irqstruct, device_data->interrupt_callback_data);
    }
}
static void start_pie_distort_idma(ddmaAcDataData *device_data, dma_addr_t phys_addr)
{
    uint32_t reg = distortDmaRead(USR);
    if (reg & DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_USR_BUSY_MASK)
    {
        error_print("%s: Distort IDMA %d is already running\n", __func__, device_data->submodinstance);
        return;
    }
    if (phys_addr & 0x3)
    {
        error_print("%s: Distort IDMA descriptor address %pa is unaligned!\n",
               __func__, &phys_addr);
    }
    distortDmaWrite(UDR, (uint32_t) phys_addr);
}

static int pie_dma_revcheck(ddmaAcDataData *device_data,
                            struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *regs;
    regs = get_ddma_regs(pie_handle, device_data->submodinstance);

    rev0 = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UTR1_TAGMAJ_MASK_SHIFT(distortDmaRead(UTR1));
    pieh_rev = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UTR1_TAGMAJ_MASK_SHIFT(regs->UTR1);

    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_dma_reset(ddmaAcDataData *device_data)
{
    distortDmaWrite(UDR, 0);
}

static void pie_dma_configure(ddmaAcDataData *device_data,
                              struct pie_handle_t *pie_handle)
{
    DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *regs;

    device_data->interrupt_callback = pie_handle->pie_idma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_idma_udma_callback_data;

    /* Enable and reset interrupts. */
    regs = get_ddma_regs(pie_handle, device_data->submodinstance);
    distortDmaWrite(UCR, regs->UCR);
    distortDmaWrite(UIER, regs->UIER);
    distortDmaWrite(UICR, regs->UIER);
    distortDmaWrite(UIFR, regs->UIFR);
    distortDmaWrite(UIWR, regs->UIWR);
}

static void pie_dma_get_current(ddmaAcDataData *device_data,
                                struct pie_handle_t *pie_handle)
{
    DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *regs;
    regs = get_ddma_regs(pie_handle, device_data->submodinstance);
    regs->UCR = distortDmaRead(UCR);
    regs->USR = distortDmaRead(USR);
    regs->UPR = distortDmaRead(UPR);
    regs->UIER = distortDmaRead(UIER);
    // UICR not readable
    // UIFR not readable
    regs->UDR = distortDmaRead(UDR);
    regs->UBAR = distortDmaRead(UBAR);
    regs->UBLR = distortDmaRead(UBLR);
    regs->UIWR = distortDmaRead(UIWR);
    regs->UTR0 = distortDmaRead(UTR0);
    regs->UTR1 = distortDmaRead(UTR1);
    regs->UTR2 = distortDmaRead(UTR2);
}

struct pie_ddma_ac_data_function_struct pie_distort_dma_functions =
{
    .pie_reset = pie_dma_reset,
    .pie_configure = pie_dma_configure,
    .pie_get_current = pie_dma_get_current,
    .pie_revcheck = pie_dma_revcheck,
    
    .start_pie_distort_idma = start_pie_distort_idma,
    .handle_distort_idma_irqs = handle_distort_idma_irqs,
};

static enum pie_internal_subblock get_ddma_subblock(int instance)
{
    switch (instance)
    {
    case 0: return ddma0;
    case 1: return ddma1;
    case 2: return ddma2;
    default: BUG();
    }
}

void pie_ddma_ac_data_init(ddmaAcDataData *device_data)
{
    ddmaAcDataDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(ddmaAcDataDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_distort_dma_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(get_ddma_subblock(device_data->submodinstance), pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));

    // distortdma has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("DISTORT_DMA_INIT LOADED.........\n");
}

void pie_ddma_ac_data_exit(ddmaAcDataData *device_data)
{
    ddmaAcDataDeviceHandle *pie_device_handle;

    print("DISTORT_DMA_INIT EXIT.........\n");
    pie_device_handle = unregister_pie_subblock(get_ddma_subblock(device_data->submodinstance));
    free_memory(pie_device_handle);
}



/* ------------ ANTIFCOR ------------ */

static int pie_antifcor_revcheck(antifcorData *device_data,
                                 struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0 = DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_REV0_MAJ_MASK_SHIFT(antiRead(REV0));
    pieh_rev = DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_REV0_MAJ_MASK_SHIFT(pie_handle->pie_distort_antifcor_top->REV0);

    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_antifcor_reset(antifcorData *device_data)
{
    uint32_t reg;
    reg = antiRead(ANTIFCOR_CFG);

    reg = 0;
    reg = DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_ANTIFCOR_CFG_BYPASS_ALL_REPLACE_VAL(reg, 1);
    antiWrite(ANTIFCOR_CFG, reg);

    reg = antiRead(ANTIFCOR_CFG);
}

static void pie_antifcor_configure(antifcorData *device_data,
                                   struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_idma_udma_callback;
    device_data->interrupt_callback_data = pie_handle->pie_idma_udma_callback_data;

    PROTECT_REG_ACCESS;
    antiWrite(ANTIFCOR_CLIP, pie_handle->pie_distort_antifcor_top->ANTIFCOR_CLIP);
    antiWrite(ANTIFCOR_CFG, pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG);
    UNPROTECT_REG_ACCESS;
}

static void pie_antifcor_get_current(antifcorData *device_data,
                                     struct pie_handle_t *pie_handle)
{
    pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG = antiRead(ANTIFCOR_CFG);
    pie_handle->pie_distort_antifcor_top->ANTIFCOR_CLIP = antiRead(ANTIFCOR_CLIP);
    pie_handle->pie_distort_antifcor_top->REV0 = antiRead(REV0);
    pie_handle->pie_distort_antifcor_top->REV1 = antiRead(REV1);
}

struct pie_antifcor_function_struct pie_antifcor_functions =
{
    .pie_reset = pie_antifcor_reset,
    .pie_configure = pie_antifcor_configure,
    .pie_get_current = pie_antifcor_get_current,
    .pie_revcheck = pie_antifcor_revcheck,
};

void pie_antifcor_init(antifcorData *device_data)
{
    antifcorDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(antifcorDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_antifcor_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(antifcor, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));

    // antifcor has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("ANTIFCOR LOADED.........\n");
}

void pie_antifcor_exit(antifcorData *device_data)
{
    antifcorDeviceHandle *pie_device_handle;

    print("ANTIFCOR EXIT.........\n");
    pie_device_handle = unregister_pie_subblock(antifcor);
    free_memory(pie_device_handle);
}
