/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>         // for ioread/write32 (pie read/write macro)
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/spinlock.h>   // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pogo_dma_driver.h"
#include "pie_depogoizer.h"
#include "pie_depogoizer_if.h"
#include "pie_driverlib_if.h"  // for print macro

void dump_pie_depogoizer_regs(struct device_data *device_data)
{
    print("PIE DEPOGOIZER Registers\n");
    print("Cfg=0x%X\n",  pieDePogoizerRead(Cfg));
    print("REV0=0x%X\n", pieDePogoizerRead(REV0));
    print("REV1=0x%X\n", pieDePogoizerRead(REV1));
}

static int pie_revcheck(struct device_data *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = DEPOGOIZER_REV0_MAJ_MASK_SHIFT(pieDePogoizerRead(REV0));
    pieh_rev = DEPOGOIZER_REV0_MAJ_MASK_SHIFT(pie_handle->pie_depogoizer->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(struct device_data *device_data)
{
    return; // nothing to do for depogoizer subblock reset
}

static void pie_configure(struct device_data *device_data,
                          struct pie_handle_t *pie_handle)
{
    // no callback to configure for depogoizer
    pieDePogoizerWrite(Cfg, pie_handle->pie_depogoizer->Cfg);
    // REV0, and REV1 are Read only
}

static void pie_get_current(struct device_data *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_depogoizer->Cfg = pieDePogoizerRead(Cfg);
    pie_handle->pie_depogoizer->REV0 = pieDePogoizerRead(REV0);
    pie_handle->pie_depogoizer->REV1 = pieDePogoizerRead(REV1);    
}

struct pie_depogoizer_function_struct pie_depogoizer_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck      = pie_revcheck,

    .dump_pie_depogoizer_regs = dump_pie_depogoizer_regs,
};

void pie_depogoizer_init(struct device_data *device_data)
{
    pieDepogoizerDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(pieDepogoizerDeviceHandle), GFP_KERNEL);
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_depogoizer_functions;
    pie_device_handle->device_data = device_data;
    register_pogo_subblock(depogoizer, pie_device_handle, 0);
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pie_depogoizer_init);

void pie_depogoizer_exit(struct device_data *device_data)
{
    pieDepogoizerDeviceHandle *pie_device_handle;

    pie_device_handle = unregister_pogo_subblock(depogoizer, 0);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_depogoizer_exit);



