/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_decim.h"
#include "pie_decim_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_decim_regs(decimData *device_data)
{
    print("PIE DECIM Registers\n");
    print("div=0x%X\n",   decimRead(div));
    print("start=0x%X\n", decimRead(start));
    print("REV0=0x%X\n",  decimRead(REV0));
    print("REV1=0x%X\n",  decimRead(REV1));
}

static int pie_revcheck(decimData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0 = DECIM_REV0_MAJ_MASK_SHIFT(decimRead(REV0));
    pieh_rev = DECIM_REV0_MAJ_MASK_SHIFT(pie_handle->pie_decim->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(decimData *device_data)
{
    // There is neither a reset bit nor a bypass bit for this
    // subblock.  However, writing 0 to the registers passes all
    // the pixels unmodified.

    PROTECT_REG_ACCESS;
    decimWrite(div, 0);
    decimWrite(start, 0);    
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(decimData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for decim

    PROTECT_REG_ACCESS;
    decimWrite(div, pie_handle->pie_decim->div);
    decimWrite(start, pie_handle->pie_decim->start);
    // REV0 and REV1 are Read only
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(decimData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_decim->div   = decimRead(div);
    pie_handle->pie_decim->start = decimRead(start);
    pie_handle->pie_decim->REV0  = decimRead(REV0);
    pie_handle->pie_decim->REV1  = decimRead(REV1);
}

struct pie_decim_function_struct pie_decim_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_decim_regs  = dump_decim_regs,
};

void pie_decim_init(decimData *device_data)
{
    decimDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(decimDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_decim_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(decim, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // decim has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE DECIM LOADED .....\n");
}
EXPORT_SYMBOL(pie_decim_init);


void pie_decim_exit(decimData *device_data)
{
    decimDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(decim);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_decim_exit);

