/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_cstats.h"
#include "pie_cstats_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_cstats_regs(cstatsData *device_data)
{
    print("PIE CSTATS Registers\n");
    print("CFG=0x%X\n",    cstatsRead(CFG));
    print("LM=0x%X\n",     cstatsRead(LM));
    print("RM=0x%X\n",     cstatsRead(RM));
    print("SW=0x%X\n",     cstatsRead(SW));
    print("STW=0x%X\n",    cstatsRead(STW));
    print("TR1=0x%X\n",    cstatsRead(TR1));
    print("TR2=0x%X\n",    cstatsRead(TR2));
    print("SBLK=0x%X\n",   cstatsRead(SBLK));
    print("SGRY=0x%X\n",   cstatsRead(SGRY));
    print("SC=0x%X\n",     cstatsRead(SC));
    // CFGBW is write only
    print("CFGBR=0x%X\n",  cstatsRead(CFGBR));
    print("REV0=0x%X\n",   cstatsRead(REV0));
    print("REV1=0x%X\n",   cstatsRead(REV1));
    // RAM0-RAM3 not dumped here (see dump_cstats_ram)
}

void dump_cstats_ram(cstatsData *device_data, int ram_num)
{
    int      i;
    int      ram_size;
    uint32_t ram_val;

    // Validate ram number, set ram size
    switch (ram_num)
    {
        case 0:
        case 1:
        case 2:
            ram_size = PIE_CSTATS_SRAM_SIZE;
            break;
        case 3:
            ram_size = PIE_CSTATS_NONWHITE_SRAM_SIZE;
            break;
        default:
            error_print("Error: ram num %d out of range.\n", ram_num);
            return;
            break;
    }

    print("PIE CSTATS RAM%d (%d):", ram_num, ram_size);

    for (i=0; i<ram_size; i++)
    {
        // Grab the value from the appropriate RAM
        switch (ram_num)
        {
            case 0:
                ram_val = cstatsReadArray(RAM0, i);
                break;
            case 1:
                ram_val = cstatsReadArray(RAM1, i);
                break;
            case 2:
                ram_val = cstatsReadArray(RAM2, i);
                break;
            case 3:
                ram_val = cstatsReadArray(RAM3, i);
                break;
            default:
                ram_val = 0xdeadbeef;
                break;
        }

        if (i%8 == 0)
        {
            print("\n%03d-%03d: ", i, i+7);
        }
        print("0x%08X ", ram_val);
    }
    print("\n");
}

static int pie_revcheck(cstatsData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = CSTATS_REV0_MAJ_MASK_SHIFT(cstatsRead(REV0));
    pieh_rev = CSTATS_REV0_MAJ_MASK_SHIFT(pie_handle->pie_cstats->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}


static void pie_reset(cstatsData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;

    // Disable the block, and leave it disabled. If someone wants this subblock,
    // they need to enable it
    reg = cstatsRead(CFG);
    reg = CSTATS_CFG_E_REPLACE_VAL(reg, 0);
    cstatsWrite(CFG, reg);

    // Reset the histogram RAMs
    reg = cstatsRead(CFGBR);
    reg = CSTATS_CFGBW_HCLR_REPLACE_VAL(reg, 1);
    cstatsWrite(CFGBW, reg);
    // REVISIT: should we wait here for the reset to complete?

    UNPROTECT_REG_ACCESS;
}

static void pie_configure(cstatsData *device_data,
                          struct pie_handle_t *pie_handle)
{
    uint32_t reg;
    int      i;

    // There is no callback to configure for cstats

    PROTECT_REG_ACCESS;

    // The programmers guide makes a point: do not program the block
    // while its enabled.
    reg = CSTATS_CFG_E_REPLACE_VAL(pie_handle->pie_cstats->CFG, 0);
    cstatsWrite(CFG, reg);

    cstatsWrite(LM,    pie_handle->pie_cstats->LM);
    cstatsWrite(RM,    pie_handle->pie_cstats->RM);
    cstatsWrite(SW,    pie_handle->pie_cstats->SW);
    cstatsWrite(STW,   pie_handle->pie_cstats->STW);
    cstatsWrite(TR1,   pie_handle->pie_cstats->TR1);
    cstatsWrite(TR2,   pie_handle->pie_cstats->TR2);
    cstatsWrite(SBLK,  pie_handle->pie_cstats->SBLK);
    cstatsWrite(SGRY,  pie_handle->pie_cstats->SGRY);
    cstatsWrite(SC,    pie_handle->pie_cstats->SC);
    cstatsWrite(CFGBW, pie_handle->pie_cstats->CFGBW);

    for (i=0; i<PIE_CSTATS_SRAM_SIZE; i++)
    {
        cstatsWriteArray(RAM0, pie_handle->pie_cstats->RAM0[i], i);
        cstatsWriteArray(RAM1, pie_handle->pie_cstats->RAM1[i], i);
        cstatsWriteArray(RAM2, pie_handle->pie_cstats->RAM2[i], i);
    }

    for (i=0; i<PIE_CSTATS_NONWHITE_SRAM_SIZE; i++)
    {
        cstatsWriteArray(RAM3, pie_handle->pie_cstats->RAM3[i], i);
    }

    // Write CFG last (in case it's enabled in pie handle)
    cstatsWrite(CFG, pie_handle->pie_cstats->CFG);

    // CFGBR, REV0, and REV1 are Read only
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(cstatsData *device_data,
                            struct pie_handle_t *pie_handle)
{
    int i;

    pie_handle->pie_cstats->CFG    = cstatsRead(CFG);
    pie_handle->pie_cstats->LM     = cstatsRead(LM);
    pie_handle->pie_cstats->RM     = cstatsRead(RM);
    pie_handle->pie_cstats->SW     = cstatsRead(SW);
    pie_handle->pie_cstats->STW    = cstatsRead(STW);
    pie_handle->pie_cstats->TR1    = cstatsRead(TR1);
    pie_handle->pie_cstats->TR2    = cstatsRead(TR2);
    pie_handle->pie_cstats->SBLK   = cstatsRead(SBLK);
    pie_handle->pie_cstats->SGRY   = cstatsRead(SGRY);
    pie_handle->pie_cstats->SC     = cstatsRead(SC);
    // CFGBW is write only
    pie_handle->pie_cstats->CFGBR  = cstatsRead(CFGBR);
    pie_handle->pie_cstats->REV0   = cstatsRead(REV0);
    pie_handle->pie_cstats->REV1   = cstatsRead(REV1);

    for (i=0; i<PIE_CSTATS_SRAM_SIZE; i++)
    {
        pie_handle->pie_cstats->RAM0[i] = cstatsReadArray(RAM0, i);
        pie_handle->pie_cstats->RAM1[i] = cstatsReadArray(RAM1, i);
        pie_handle->pie_cstats->RAM2[i] = cstatsReadArray(RAM2, i);
    }

    for (i=0; i<PIE_CSTATS_NONWHITE_SRAM_SIZE; i++)
    {
        pie_handle->pie_cstats->RAM3[i] = cstatsReadArray(RAM3, i);
    }

}

struct pie_cstats_function_struct pie_cstats_functions =
{
    .pie_reset         = pie_reset,
    .pie_configure     = pie_configure,
    .pie_get_current   = pie_get_current,
    .pie_revcheck      = pie_revcheck,
    
    .dump_cstats_regs  = dump_cstats_regs,
    .dump_cstats_ram   = dump_cstats_ram,
};

void pie_cstats_init(cstatsData *device_data)
{
    cstatsDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(cstatsDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_cstats_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(cstats, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // cstats has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE CSTATS LOADED .....\n");
}
EXPORT_SYMBOL(pie_cstats_init);


void pie_cstats_exit(cstatsData *device_data)
{
    cstatsDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(cstats);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_cstats_exit);

