/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */


// this file needs to compile in both user and kernel space so that convenience
// functions are available from either a kernel loadable module or an app

#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/string.h>
#include <linux/module.h>
#else
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#define BUG() assert(0)
#endif

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"      // detailed info about the pie handle
#include "pie_if.h"          // for the interrupt info
#include "pie_driver.h"      // pie.c enums and exports 
#include "pie_convenience_if.h"

#include "pie_driverlib_if.h"  // driver utilites and print macros

#define PIE_PTRCHECK_NAME_NR(handle_name, subblockname) \
    BUG_ON(handle_name == NULL);                        \
    if (handle_name->subblockname == NULL)              \
    {                                                   \
        print("NULL subblockname %s\n", #subblockname); \
        return;                                         \
    }

#define PIE_PTRCHECK_NAME_ARRY_NR(handle_name, subblockname, index)     \
    BUG_ON(handle_name == NULL);                                        \
    if (handle_name->subblockname[index] == NULL)                       \
    {                                                                   \
        print("NULL subblockname %s\n", #subblockname);                 \
        return;                                                         \
    }

#define PIE_PTRCHECK_NAME(handle_name, subblockname, retval)     \
    BUG_ON(handle_name == NULL);                                 \
    if (handle_name->subblockname == NULL)                       \
    {                                                            \
        print ("NULL subblockname %s\n", #subblockname);         \
        return retval;                                           \
    }

#define PIE_PTRCHECK(subblockname, retval)                      \
    PIE_PTRCHECK_NAME(pie_handle, subblockname, retval);

#define PIE_PTRCHECK_ARRY_NR(subblockname, index)               \
    PIE_PTRCHECK_NAME_ARRY_NR(pie_handle, subblockname, index);

#define PIE_PTRCHECK_NR(subblockname)                   \
    PIE_PTRCHECK_NAME_NR(pie_handle, subblockname);



///////////////////////////////////////////
// to get the size of the whole pie_handle
///////////////////////////////////////////

int pie_get_handle_size(struct pie_handle_t *pie_handle)
{
    return pie_handle->pie_handle_size;
}


////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// PIE SUBBLOCK FUNCTIONS
// Each subblock needs to define their convenience functions.  
// All subblocks must have:
// 1) get default function
// 2) clone function
// 3) dump function
// 4) set bypass (if applicable)
// 5) (optional) set functions to set various registers
// 6) (optional) interrupt functions - enable, set callback, ...
//
// Note that when you add functions 1-4 (and optional #5) for a new subblock,
// you need to make sure to add the calls to those functions in the
// "pie high level functions"
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// pie common
////////////////////////////////////////////////////////////////////////

static void get_pie_common_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_common->IENR = 0;
    pie_handle->pie_common->IPR = 0;
    pie_handle->pie_common->CBIR = 0;
    
    regval = PIE_COMMON_CTL_IMAGE_RDY_BYP_REPLACE_VAL(0, 0);
    regval = PIE_COMMON_CTL_OT_RDY_BYP_REPLACE_VAL(regval, 1);
    regval = PIE_COMMON_CTL_DB_DEBUG_SEL_REPLACE_VAL(regval, 0);
    regval = PIE_COMMON_CTL_DIS_MAP_POGO_EN_REPLACE_VAL(regval, 0);
    regval = PIE_COMMON_CTL_DISTORT_DMA_EN_REPLACE_VAL(regval, 0);
    regval = PIE_COMMON_CTL_RESERVE_MODE_EN_REPLACE_VAL(regval, 0);
    regval = PIE_COMMON_CTL_PIPEORDER_CONFIG_REPLACE_VAL(regval, 0);
    pie_handle->pie_common->CTL = regval;
    pie_handle->pie_common->DBG = 0x55110000;
    pie_handle->pie_common->REV0 = PIE_COMMON_MAJ_REV0;
    pie_handle->pie_common->REV1 = 0;
}

static void pie_common_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_common);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_common);    

    // clone the callbacks
    dest_pie_handle->pie_common_callback = src_pie_handle->pie_common_callback;
    dest_pie_handle->pie_common_callback_data = src_pie_handle->pie_common_callback_data;

    // clone the registers
    dest_pie_handle->pie_common->IENR = src_pie_handle->pie_common->IENR;
    dest_pie_handle->pie_common->IPR = src_pie_handle->pie_common->IPR;
    dest_pie_handle->pie_common->CBIR = src_pie_handle->pie_common->CBIR;
    dest_pie_handle->pie_common->CTL = src_pie_handle->pie_common->CTL;
    dest_pie_handle->pie_common->REV0 = src_pie_handle->pie_common->REV0;
    dest_pie_handle->pie_common->REV1 = src_pie_handle->pie_common->REV1;
}

void pie_common_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE common regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_common);

    print("IENR=0x%X\n",  pie_handle->pie_common->IENR);
    print("IPR=0x%X\n",  pie_handle->pie_common->IPR);
    print("CBIR=0x%X\n",  pie_handle->pie_common->CBIR);
    print("CTL=0x%X\n",  pie_handle->pie_common->CTL);
    print("REV0=0x%X\n",  pie_handle->pie_common->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_common->REV1);
}
EXPORT_SYMBOL(pie_common_dump_handle_regs);

void pie_common_set_distort_dma(struct pie_handle_t *pie_handle, bool enabled)
{
    PIE_PTRCHECK_NR(pie_common);
    pie_handle->pie_common->CTL = PIE_COMMON_CTL_DISTORT_DMA_EN_REPLACE_VAL(pie_handle->pie_common->CTL, enabled);
}
EXPORT_SYMBOL(pie_common_set_distort_dma);

bool pie_common_get_distort_dma_enabled(struct pie_handle_t *pie_handle)
{
    bool enabled;
    PIE_PTRCHECK(pie_common, 0);
    enabled = PIE_COMMON_CTL_DISTORT_DMA_EN_MASK_SHIFT(pie_handle->pie_common->CTL);

    return enabled;
}
EXPORT_SYMBOL(pie_common_get_distort_dma_enabled);

void pie_common_set_ot_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_common);
    pie_handle->pie_common->CTL = PIE_COMMON_CTL_OT_RDY_BYP_REPLACE_VAL(pie_handle->pie_common->CTL, bypass);
}
EXPORT_SYMBOL(pie_common_set_ot_bypass);

bool pie_common_get_ot_odma_enabled(struct pie_handle_t *pie_handle)
{
    bool enabled;
    PIE_PTRCHECK(pie_common, 0);
    enabled = (PIE_COMMON_CTL_OT_RDY_BYP_MASK_SHIFT(pie_handle->pie_common->CTL) == 0);

    return enabled;
}
EXPORT_SYMBOL(pie_common_get_ot_odma_enabled);

    
//
// pie common interrupt functions
//

// if enable == false, disable requested irqs
// if enable == true, disable requested irqs
// if irqstruct == NULL, then enable/disable all irqs

void pie_enable_common_irqs(struct pie_handle_t *pie_handle,
                            struct pie_common_ints *irqstruct,
                            bool enable)
{
    struct pie_common_ints irq;

    PIE_PTRCHECK_NR(pie_common);
    
    if (irqstruct == NULL)
    {
        pie_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear enable irq bits
    pie_handle->pie_common->IENR =
        pie_common_convert_irqstruct_to_update_reg(irqstruct,
                                                    pie_handle->pie_common->IENR,
                                                    enable);
}
EXPORT_SYMBOL(pie_enable_common_irqs);

// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs

void pie_force_common_irqs(struct pie_handle_t *pie_handle,
                           struct pie_common_ints *irqstruct,
                           bool force)
{
    struct pie_common_ints irq;
    
    PIE_PTRCHECK_NR(pie_common);
    
    if (irqstruct == NULL)
    {
        pie_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
}
EXPORT_SYMBOL(pie_force_common_irqs);

void pie_register_common_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);
    
    pie_handle->pie_common_callback = callback_fcn;
    pie_handle->pie_common_callback_data = callback_data;    
}
EXPORT_SYMBOL(pie_register_common_callback);

    
////////////////////////////////////////////////////////////////////////
// pie pogoizer
////////////////////////////////////////////////////////////////////////

static void get_pogoizer_default(struct pie_handle_t *pie_handle)
{
    // pogoizer subblock

    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_pogoizer->Cfg = 0;
    pie_handle->pie_pogoizer->LPad = 0;
    pie_handle->pie_pogoizer->RPad = 0;
    pie_handle->pie_pogoizer->LVal0 = 0;
    pie_handle->pie_pogoizer->LVal1 = 0;
    pie_handle->pie_pogoizer->LVal2 = 0;
    pie_handle->pie_pogoizer->RVal0 = 0;
    pie_handle->pie_pogoizer->RVal1 = 0;
    pie_handle->pie_pogoizer->RVal2 = 0;
    pie_handle->pie_pogoizer->LTrunc = 0;
    pie_handle->pie_pogoizer->RTrunc = POGOIZER_RTRUNC_LASTCOLUMN_REPLACE_VAL(0, 0xFFFF);
    pie_handle->pie_pogoizer->REV0 = POGOIZER_MAJ_REV;
    pie_handle->pie_pogoizer->REV1 = 0;
}

static void pie_pogoizer_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_pogoizer);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_pogoizer);
    
    dest_pie_handle->pie_pogoizer->Cfg = src_pie_handle->pie_pogoizer->Cfg;
    dest_pie_handle->pie_pogoizer->LPad = src_pie_handle->pie_pogoizer->LPad;   
    dest_pie_handle->pie_pogoizer->RPad = src_pie_handle->pie_pogoizer->RPad;  
    dest_pie_handle->pie_pogoizer->LVal0 = src_pie_handle->pie_pogoizer->LVal0; 
    dest_pie_handle->pie_pogoizer->LVal1 = src_pie_handle->pie_pogoizer->LVal1;
    dest_pie_handle->pie_pogoizer->LVal2 = src_pie_handle->pie_pogoizer->LVal2;
    dest_pie_handle->pie_pogoizer->RVal0 = src_pie_handle->pie_pogoizer->RVal0;
    dest_pie_handle->pie_pogoizer->RVal1 = src_pie_handle->pie_pogoizer->RVal1;
    dest_pie_handle->pie_pogoizer->RVal2 = src_pie_handle->pie_pogoizer->RVal2;
    dest_pie_handle->pie_pogoizer->LTrunc = src_pie_handle->pie_pogoizer->LTrunc;
    dest_pie_handle->pie_pogoizer->RTrunc = src_pie_handle->pie_pogoizer->RTrunc;
    dest_pie_handle->pie_pogoizer->REV0 = src_pie_handle->pie_pogoizer->REV0;
    dest_pie_handle->pie_pogoizer->REV1 = src_pie_handle->pie_pogoizer->REV1;  
}


void pie_pogoizer_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE pogoizer regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_pogoizer);

    print("Cfg=0x%X\n",  pie_handle->pie_pogoizer->Cfg);
    print("LPad=0x%X\n",  pie_handle->pie_pogoizer->LPad);
    print("RPad=0x%X\n",  pie_handle->pie_pogoizer->RPad);
    print("LVal0=0x%X\n",  pie_handle->pie_pogoizer->LVal0);
    print("LVal1=0x%X\n",  pie_handle->pie_pogoizer->LVal1);
    print("LVal2=0x%X\n",  pie_handle->pie_pogoizer->LVal2);    
    print("RVal0=0x%X\n",  pie_handle->pie_pogoizer->RVal0);
    print("RVal1=0x%X\n",  pie_handle->pie_pogoizer->RVal1);
    print("RVal2=0x%X\n",  pie_handle->pie_pogoizer->RVal2);
    print("LTrunc=0x%X\n",  pie_handle->pie_pogoizer->LTrunc);
    print("RTrunc=0x%X\n",  pie_handle->pie_pogoizer->RTrunc);    
    print("REV0=0x%X\n",  pie_handle->pie_pogoizer->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_pogoizer->REV1);     
}
EXPORT_SYMBOL(pie_pogoizer_dump_handle_regs);

int pie_pogoizer_set_config(struct pie_handle_t *pie_handle, uint8_t inputformat,
                            uint8_t pogo_bpp, uint8_t colorswap, 
                            uint8_t scalex, uint8_t scaley)
{
    PIE_PTRCHECK(pie_pogoizer, -1);

    pie_handle->pie_pogoizer->Cfg =
        POGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_pogoizer->Cfg, inputformat);
    pie_handle->pie_pogoizer->Cfg =
        POGOIZER_CFG_COLORSWAP_REPLACE_VAL(pie_handle->pie_pogoizer->Cfg, colorswap);
    pie_handle->pie_pogoizer->Cfg =
        POGOIZER_CFG_BPP_REPLACE_VAL(pie_handle->pie_pogoizer->Cfg, pogo_bpp);
    pie_handle->pie_pogoizer->Cfg =
        POGOIZER_CFG_SCALEX_REPLACE_VAL(pie_handle->pie_pogoizer->Cfg, scalex);
    pie_handle->pie_pogoizer->Cfg =
        POGOIZER_CFG_SCALEY_REPLACE_VAL(pie_handle->pie_pogoizer->Cfg, scaley);

    return 0;     
}
EXPORT_SYMBOL(pie_pogoizer_set_config);

////////////////////////////////////////////////////////////////////////
// pie input dma (udma and core)
////////////////////////////////////////////////////////////////////////

static void get_pogo_idma_udma_default(struct pie_handle_t *pie_handle, int idma_num)
{
    uint32_t regval;
    POGO_IDMA0_POGO_IDMA_UDMA_REGS_t *idma_handle;
    
    idma_handle = pie_handle->pie_pogo_idma_udma[idma_num];
    
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    idma_handle->UCR = 0;
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_DMAPAUSED_REPLACE_VAL(0, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_PAUSEAFTER_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_PAUSEBEFORE_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_INTONFINISH_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_STOPONFINISH_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_CMDBUFFULL_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_CMDBUFEMPTY_REPLACE_VAL(regval, 1);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_DATABUFFULL_REPLACE_VAL(regval, 0);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_DATABUFEMPTY_REPLACE_VAL(regval, 1);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_CLEAR_REPLACE_VAL(regval, 1);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_USR_BUSY_REPLACE_VAL(regval, 0);
    idma_handle->USR = regval;
    regval = POGO_IDMA0_POGO_IDMA_UDMA_UPR_FIFODEPTH_REPLACE_VAL(0, 128);
    regval = POGO_IDMA0_POGO_IDMA_UDMA_UPR_BUSWIDTH_REPLACE_VAL(regval, 1);
    idma_handle->UPR = regval;  // block instantiation parms - user should compare against ASIC vals
    idma_handle->UIER = 0;
    idma_handle->UIPR = 0;
    idma_handle->UICR = 0;
    idma_handle->UIFR = 0;
    idma_handle->UDR = 0;
    idma_handle->UBAR = 0;
    idma_handle->UBLR = 0;
    idma_handle->UBRR = 0;
    idma_handle->UTR0 = POGO_IDMA_UDMA_MAJ_REV;
    idma_handle->UTR1 = 0;
}

static void get_pogo_idma_core_default(struct pie_handle_t *pie_handle, int idma_num)
{
    POGO_IDMA0_POGO_IDMA_CORE_REGS_t *idma_handle;    

    idma_handle = pie_handle->pie_pogo_idma_core[idma_num];
    
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    idma_handle->ICR = 0;
    idma_handle->IIER = 0;
    idma_handle->IIPR = 0;
    idma_handle->IICR = 0;
    idma_handle->IIFR = 0;
    idma_handle->ILWR = 0;
    idma_handle->IRHR = 0;
    idma_handle->ITR0 = 0;
    idma_handle->ITR1 = POGO_IDMA_CORE_MAJ_REV;
    idma_handle->ITR2 = 1;
}

static void pie_input_dma_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle,
                                     uint8_t idma_num)
{
    // PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_pogo_idma_udma); FIXME - add Arry versions of these macros
    // PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_pogo_idma_udma);
    // PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_pogo_idma_core);
    // PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_pogo_idma_core);

    // clone the callbacks
    dest_pie_handle->pie_idma_udma_callback = src_pie_handle->pie_idma_udma_callback;
    dest_pie_handle->pie_idma_udma_callback_data = src_pie_handle->pie_idma_udma_callback_data;

    dest_pie_handle->pie_idma_core_callback = src_pie_handle->pie_idma_core_callback;
    dest_pie_handle->pie_idma_core_callback_data = src_pie_handle->pie_idma_core_callback_data;

    // clone the registers
    dest_pie_handle->pie_pogo_idma_core[idma_num]->ICR = src_pie_handle->pie_pogo_idma_core[idma_num]->ICR;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->IIER = src_pie_handle->pie_pogo_idma_core[idma_num]->IIER;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->IIPR = src_pie_handle->pie_pogo_idma_core[idma_num]->IIPR;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->IICR = src_pie_handle->pie_pogo_idma_core[idma_num]->IICR;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->IIFR = src_pie_handle->pie_pogo_idma_core[idma_num]->IIFR;    
    dest_pie_handle->pie_pogo_idma_core[idma_num]->ILWR = src_pie_handle->pie_pogo_idma_core[idma_num]->ILWR;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->IRHR = src_pie_handle->pie_pogo_idma_core[idma_num]->IRHR;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->ITR0 = src_pie_handle->pie_pogo_idma_core[idma_num]->ITR0;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->ITR1 = src_pie_handle->pie_pogo_idma_core[idma_num]->ITR1;
    dest_pie_handle->pie_pogo_idma_core[idma_num]->ITR2 = src_pie_handle->pie_pogo_idma_core[idma_num]->ITR2;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UCR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UCR;  
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->USR = src_pie_handle->pie_pogo_idma_udma[idma_num]->USR;  
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UPR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UPR;  
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UIER = src_pie_handle->pie_pogo_idma_udma[idma_num]->UIER;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UIPR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UIPR;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UICR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UICR;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UIFR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UIFR;    
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UDR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UDR;  
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UBAR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UBAR;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UBLR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UBLR;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UBRR = src_pie_handle->pie_pogo_idma_udma[idma_num]->UBRR;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UTR0 = src_pie_handle->pie_pogo_idma_udma[idma_num]->UTR0;
    dest_pie_handle->pie_pogo_idma_udma[idma_num]->UTR1 = src_pie_handle->pie_pogo_idma_udma[idma_num]->UTR1;    
}

void pie_input_dma_dump_handle_regs(struct pie_handle_t *pie_handle, uint8_t idma_num)
{
    print("PIE IDMA[%d] core regs (pie_handle)\n", idma_num);

    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_udma, idma_num);
    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_core, idma_num);    

    print("ICR=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->ICR);
    print("IIER=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->IIER);
    print("IIPR=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->IIPR);
    // IICR and IIFR are Write only
    print("ILWR=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->ILWR);
    print("IRHR=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->IRHR);
    print("ITR0=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->ITR0);
    print("ITR1=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->ITR1);
    print("ITR2=0x%X\n", pie_handle->pie_pogo_idma_core[idma_num]->ITR2);
    print("PIE IDMA[%d] udma regs (pie_handle)\n", idma_num);
    print("UCR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UCR);
    print("USR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->USR);
    print("UPR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UPR);
    print("UIER=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UIER);
    print("UIPR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UIPR);
    // UICR and UIFR are Write only
    print("UDR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UDR);
    print("UBAR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UBAR);
    print("UBLR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UBLR);
    print("UBRR=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UBRR);
    print("UTR0=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UTR0);
    print("UTR1=0x%X\n", pie_handle->pie_pogo_idma_udma[idma_num]->UTR1);
}
EXPORT_SYMBOL(pie_input_dma_dump_handle_regs);

void pie_input_dma_set_parms(struct pie_handle_t *pie_handle, uint8_t handshake,
                             uint8_t ownership, uint8_t own_polarity, uint8_t enable,
                             uint32_t linewidth, uint8_t rowheight, int channel_num)
{
    int burstsize, beats, buswidth, fifodepth;
    uint8_t beats_reg_val, burstsize_reg_val, asic_reg_bus_width;

    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_udma, channel_num);
    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_core, channel_num);
    
    // setup the idma udma register structs from the parms passed in

    // beats: if PIE IDMA has a 64 bit wide bus
    //      4 "beats" (transfers) is 64 bits * 4, 32 byte burst
    //  and 8 "beats" (transfers) is 64 bits * 8, 64 byte burst
    // Calculate the right number of beats to give a burst that half fills the fifo
    // (which gives us double buffering into the fifo).

    // when we created the pie_handle, the UPR register was read from the ASIC.  Use
    // that for the ODMA ASIC instantiation parms

    fifodepth = POGO_IDMA0_POGO_IDMA_UDMA_UPR_FIFODEPTH_MASK_SHIFT(pie_handle->pie_pogo_idma_udma[channel_num]->UPR);
    asic_reg_bus_width = POGO_IDMA0_POGO_IDMA_UDMA_UPR_BUSWIDTH_MASK_SHIFT(pie_handle->pie_pogo_idma_udma[channel_num]->UPR);
    
    // for double buffering, we want to make the UDMA burst size 1/2 the size of the fifo
    burstsize = fifodepth/2;
    switch(asic_reg_bus_width)
    {
    case BUS_WIDTH_32_BIT:
        // Bus width is 4 Bytes (32 bits)
        buswidth = 4;
        break;
    case BUS_WIDTH_64_BIT:
        // Bus width is 8 Bytes (64 bits)
        buswidth = 8;
        break;
    default:
        error_print("ERROR, UNSUPPORTED BUS WIDTH: 0x%X\n",asic_reg_bus_width);
        buswidth = 4;
    }
    
    beats = burstsize/buswidth;
    if (beats >= 16)
    {
        beats_reg_val = DMA_16_BEATS;
        beats = 16;
    }
    else if (beats >= 8)
    {
        beats_reg_val = DMA_8_BEATS;
        beats = 8;
    }
    else
    {
        beats_reg_val = DMA_4_BEATS;
        beats = 4;
    }

    burstsize = beats * buswidth; // snap burstsize, if needed
    debug_print("%s: computed burstsize is %d\n", __func__, burstsize);

    // set burst length, enable, etc into UCR
    pie_handle->pie_pogo_idma_udma[channel_num]->UCR =
        POGO_IDMA0_POGO_IDMA_UDMA_UCR_HANDSHAKEENABLE_REPLACE_VAL(pie_handle->pie_pogo_idma_udma[channel_num]->UCR, handshake);
    pie_handle->pie_pogo_idma_udma[channel_num]->UCR =
        POGO_IDMA0_POGO_IDMA_UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(pie_handle->pie_pogo_idma_udma[channel_num]->UCR, ownership);
    pie_handle->pie_pogo_idma_udma[channel_num]->UCR =
        POGO_IDMA0_POGO_IDMA_UDMA_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_pogo_idma_udma[channel_num]->UCR, own_polarity);
    pie_handle->pie_pogo_idma_udma[channel_num]->UCR =
        POGO_IDMA0_POGO_IDMA_UDMA_UCR_BEATS_REPLACE_VAL(pie_handle->pie_pogo_idma_udma[channel_num]->UCR, beats_reg_val);
    pie_handle->pie_pogo_idma_udma[channel_num]->UCR =
        POGO_IDMA0_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_pogo_idma_udma[channel_num]->UCR, enable);
    
    // done calculating UCR parms, now for ICR...
    if (burstsize >= 64)
        burstsize_reg_val = CORE_BURST_SIZE_64;
    else if (burstsize >= 48)
        burstsize_reg_val = CORE_BURST_SIZE_48;
    else if (burstsize >= 32)
        burstsize_reg_val = CORE_BURST_SIZE_32;
    else
        burstsize_reg_val = CORE_BURST_SIZE_16;

    pie_handle->pie_pogo_idma_core[channel_num]->ICR =
        POGO_IDMA0_POGO_IDMA_CORE_ICR_BURSTSIZE_REPLACE_VAL(pie_handle->pie_pogo_idma_core[channel_num]->ICR, burstsize_reg_val);
    
    // now for ILWR and IRHR
    pie_handle->pie_pogo_idma_core[channel_num]->ILWR =
        POGO_IDMA0_POGO_IDMA_CORE_ILWR_LINEWIDTH_REPLACE_VAL(pie_handle->pie_pogo_idma_core[channel_num]->ILWR, linewidth);
    pie_handle->pie_pogo_idma_core[channel_num]->IRHR =
        POGO_IDMA0_POGO_IDMA_CORE_IRHR_ROWHEIGHT_REPLACE_VAL(pie_handle->pie_pogo_idma_core[channel_num]->IRHR, rowheight);
}
EXPORT_SYMBOL(pie_input_dma_set_parms);

//
// pie idma interrupt functions
//

void pie_enable_pogo_idma_irqs(struct pie_handle_t *pie_handle,
                               struct idma_interrupt_info *irqstruct,
                               int channel, bool enable)
{
    struct idma_interrupt_info irq;

    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_udma, channel);
    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_core, channel);

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_idma(&irq, true);
        irqstruct = &irq;
    }
    pie_handle->pie_pogo_idma_udma[channel]->UIER =
        pie_idma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                      pie_handle->pie_pogo_idma_udma[channel]->UIER,
                                                      enable);
    
    pie_handle->pie_pogo_idma_core[channel]->IIER =
        pie_idma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                      pie_handle->pie_pogo_idma_core[channel]->IIER,
                                                      enable);
}
EXPORT_SYMBOL(pie_enable_pogo_idma_irqs);


// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs

void pie_force_pogo_idma_irqs(struct pie_handle_t *pie_handle,
                              struct idma_interrupt_info *irqstruct,
                              bool force, int channel)
{
    struct idma_interrupt_info irq;

    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_udma, channel);
    PIE_PTRCHECK_ARRY_NR(pie_pogo_idma_core, channel);

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_idma(&irq, true);
        irqstruct = &irq;
    }
    // set/clear force irq bits
    pie_handle->pie_pogo_idma_udma[channel]->UIFR = 
        pie_idma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                      pie_handle->pie_pogo_idma_udma[channel]->UIFR,
                                                      force);

    pie_handle->pie_pogo_idma_core[channel]->IIFR =
        pie_idma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                      pie_handle->pie_pogo_idma_core[channel]->IIFR,
                                                      force);

}
EXPORT_SYMBOL(pie_force_pogo_idma_irqs);

void pie_register_idma_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);

    pie_handle->pie_idma_udma_callback = callback_fcn;
    pie_handle->pie_idma_udma_callback_data = callback_data;
    
    pie_handle->pie_idma_core_callback = callback_fcn;
    pie_handle->pie_idma_core_callback_data = callback_data;
}
EXPORT_SYMBOL(pie_register_idma_callback);


////////////////////////////////////////////////////////////////////////
// pie depogoizer
////////////////////////////////////////////////////////////////////////

#define PIE_DEPOGOIZER_MAX_INPUT_LINES  (36)
#define PIE_DEPOGOIZER_MAX_OUTPUT_LINES (16)

static void get_depogoizer_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    // depogoizer subblock
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = DEPOGOIZER_CFG_MONOCHAN_REPLACE_VAL(0, 0);
    regval = DEPOGOIZER_CFG_PIXORDER_REPLACE_VAL(regval, 0);
    regval = DEPOGOIZER_CFG_BPP_REPLACE_VAL(regval, 3);
    regval = DEPOGOIZER_CFG_COLORSWAP_REPLACE_VAL(regval, 0);
    regval = DEPOGOIZER_CFG_FMT_REPLACE_VAL(regval, 0);
    pie_handle->pie_depogoizer->Cfg = regval;        
    pie_handle->pie_depogoizer->REV0 = DEPOGOIZER_MAJ_REV;
    pie_handle->pie_depogoizer->REV1 = 0;
}

static void pie_depogoizer_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_depogoizer);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_depogoizer);

    dest_pie_handle->pie_depogoizer->Cfg = src_pie_handle->pie_depogoizer->Cfg;
    dest_pie_handle->pie_depogoizer->REV0 = src_pie_handle->pie_depogoizer->REV0;        
    dest_pie_handle->pie_depogoizer->REV1 = src_pie_handle->pie_depogoizer->REV1;        
}

void pie_depogoizer_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE depogoizer regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_depogoizer);
    
    print("Cfg=0x%X\n",  pie_handle->pie_depogoizer->Cfg);
    print("REV0=0x%X\n",  pie_handle->pie_depogoizer->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_depogoizer->REV1);    
}
EXPORT_SYMBOL(pie_depogoizer_dump_handle_regs);

int pie_depogoizer_set_config(struct pie_handle_t *pie_handle, uint8_t outputformat,
                              uint8_t pogo_bpp, uint8_t colorswap, uint8_t monochan)
{
    PIE_PTRCHECK(pie_depogoizer, -1);

    // starting with g2, only 8 bpp is supported in PIE depogoizer
    // This is a read-only field, but setting it in the call to this
    // function provides for error checking
    if (pogo_bpp != POGO_8BPP)
    {
        error_print("%s:ERROR, only pogo_bpp of 8BPP (%d) is supported\n", __func__, POGO_8BPP);
        return -1;
    }

    pie_handle->pie_depogoizer->Cfg =
        DEPOGOIZER_CFG_MONOCHAN_REPLACE_VAL(pie_handle->pie_depogoizer->Cfg, monochan);

    pie_handle->pie_depogoizer->Cfg =
        DEPOGOIZER_CFG_COLORSWAP_REPLACE_VAL(pie_handle->pie_depogoizer->Cfg, colorswap);

    pie_handle->pie_depogoizer->Cfg =
        DEPOGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_depogoizer->Cfg, outputformat);

    return 0;     
}
EXPORT_SYMBOL(pie_depogoizer_set_config);


int pie_depogoizer_get_output_lines(struct pie_handle_t *pie_handle, unsigned int *out_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;

    PIE_PTRCHECK(pie_depogoizer, -1);

    lines_in = *out_lines;

    if (cap_at_max && (lines_in > PIE_DEPOGOIZER_MAX_INPUT_LINES))
    {
        // Cap the input lines
        lines_in = PIE_DEPOGOIZER_MAX_INPUT_LINES;
        retVal = -1;
    }

    lines_out = lines_in;

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_out > PIE_DEPOGOIZER_MAX_OUTPUT_LINES))
    {
        lines_out = PIE_DEPOGOIZER_MAX_OUTPUT_LINES;
        retVal = -1;
    }
    else if (lines_out < 1)
    {
        retVal = -1;
    }

    *out_lines = lines_out;

    return retVal;
}
EXPORT_SYMBOL(pie_depogoizer_get_output_lines);

int pie_depogoizer_get_input_lines(struct pie_handle_t *pie_handle, unsigned int *in_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;

    PIE_PTRCHECK(pie_depogoizer, -1);

    lines_out = *in_lines;

    if (cap_at_max && (lines_out > PIE_DEPOGOIZER_MAX_OUTPUT_LINES))
    {
        // Cap the input lines
        lines_out = PIE_DEPOGOIZER_MAX_OUTPUT_LINES;
        retVal = -1;
    }

    lines_in = lines_out;

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_in > PIE_DEPOGOIZER_MAX_INPUT_LINES))
    {
        lines_in = PIE_DEPOGOIZER_MAX_INPUT_LINES;
        retVal = -1;
    }
    else if (lines_in < 1)
    {
        retVal = -1;
    }

    *in_lines = lines_in;

    return retVal;
}
EXPORT_SYMBOL(pie_depogoizer_get_input_lines);

int pie_depogoizer_get_discard_lines(struct pie_handle_t *pie_handle, unsigned int *discard_lines)
{
    PIE_PTRCHECK(pie_depogoizer, -1);
    return 0;
}
EXPORT_SYMBOL(pie_depogoizer_get_discard_lines);

////////////////////////////////////////////////////////////////////////
// pie odma (udma and core)
////////////////////////////////////////////////////////////////////////

static void get_pogo_odma_udma_default(struct pie_handle_t *pie_handle)
{
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_pogo_odma_udma->UCR = 0;
    pie_handle->pie_pogo_odma_udma->USR = 0;
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_DMAPAUSED_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_PAUSEAFTER_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_PAUSEBEFORE_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_INTONFINISH_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_STOPONFINISH_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_CMDBUFFULL_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_CMDBUFEMPTY_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 1);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_DATABUFFULL_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_DATABUFEMPTY_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 1);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_CLEAR_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 1);
    pie_handle->pie_pogo_odma_udma->USR =
        POGO_ODMA_POGO_ODMA_UDMA_USR_BUSY_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->USR, 0);
    pie_handle->pie_pogo_odma_udma->UPR = 0;
    pie_handle->pie_pogo_odma_udma->UPR =
        POGO_ODMA_POGO_ODMA_UDMA_UPR_FIFODEPTH_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UPR, 128);
    pie_handle->pie_pogo_odma_udma->UPR =
        POGO_ODMA_POGO_ODMA_UDMA_UPR_BUSWIDTH_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UPR, 1);
    pie_handle->pie_pogo_odma_udma->UIER = 0;
    pie_handle->pie_pogo_odma_udma->UIPR = 0;
    pie_handle->pie_pogo_odma_udma->UICR = 0;
    pie_handle->pie_pogo_odma_udma->UIFR = 0;
    pie_handle->pie_pogo_odma_udma->UDR = 0;
    pie_handle->pie_pogo_odma_udma->UBAR = 0;
    pie_handle->pie_pogo_odma_udma->UBLR = 0;
    pie_handle->pie_pogo_odma_udma->UBRR = 0;
    pie_handle->pie_pogo_odma_udma->UTR0 = POGO_ODMA_UDMA_MAJ_REV;
    pie_handle->pie_pogo_odma_udma->UTR1 = 0;
}

static void get_pogo_odma_core_default(struct pie_handle_t *pie_handle)
{
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_pogo_odma_core->OIER = 0; 
    pie_handle->pie_pogo_odma_core->OIPR = 0; 
    pie_handle->pie_pogo_odma_core->OICR = 0;
    pie_handle->pie_pogo_odma_core->OIFR = 0; 
    pie_handle->pie_pogo_odma_core->OLWR = 0; 
    pie_handle->pie_pogo_odma_core->OCNTRL = 0;
    pie_handle->pie_pogo_odma_core->OTR0 = 0;
    pie_handle->pie_pogo_odma_core->OTR1 = POGO_ODMA_CORE_MAJ_REV;
    pie_handle->pie_pogo_odma_core->OTR2 = 8;
}

static void pie_output_dma_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_pogo_odma_udma);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_pogo_odma_udma);
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_pogo_odma_core);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_pogo_odma_core);

    // clone the callbacks
    dest_pie_handle->pie_odma_udma_callback = src_pie_handle->pie_odma_udma_callback;
    dest_pie_handle->pie_odma_udma_callback_data = src_pie_handle->pie_odma_udma_callback_data;

    dest_pie_handle->pie_odma_core_callback = src_pie_handle->pie_odma_core_callback;
    dest_pie_handle->pie_odma_core_callback_data = src_pie_handle->pie_odma_core_callback_data;

    // clone the registers
    dest_pie_handle->pie_pogo_odma_core->OIER = src_pie_handle->pie_pogo_odma_core->OIER;
    dest_pie_handle->pie_pogo_odma_core->OIPR = src_pie_handle->pie_pogo_odma_core->OIPR;
    dest_pie_handle->pie_pogo_odma_core->OICR = src_pie_handle->pie_pogo_odma_core->OICR;
    dest_pie_handle->pie_pogo_odma_core->OIFR = src_pie_handle->pie_pogo_odma_core->OIFR;
    dest_pie_handle->pie_pogo_odma_core->OLWR = src_pie_handle->pie_pogo_odma_core->OLWR;
    dest_pie_handle->pie_pogo_odma_core->OCNTRL = src_pie_handle->pie_pogo_odma_core->OCNTRL;
    dest_pie_handle->pie_pogo_odma_core->OTR0 = src_pie_handle->pie_pogo_odma_core->OTR0;
    dest_pie_handle->pie_pogo_odma_core->OTR1 = src_pie_handle->pie_pogo_odma_core->OTR1;
    dest_pie_handle->pie_pogo_odma_core->OTR2 = src_pie_handle->pie_pogo_odma_core->OTR2;

    dest_pie_handle->pie_pogo_odma_udma->UCR = src_pie_handle->pie_pogo_odma_udma->UCR;
    dest_pie_handle->pie_pogo_odma_udma->USR = src_pie_handle->pie_pogo_odma_udma->USR;
    dest_pie_handle->pie_pogo_odma_udma->UPR = src_pie_handle->pie_pogo_odma_udma->UPR;
    dest_pie_handle->pie_pogo_odma_udma->UIER = src_pie_handle->pie_pogo_odma_udma->UIER;
    dest_pie_handle->pie_pogo_odma_udma->UIPR = src_pie_handle->pie_pogo_odma_udma->UIPR;
    dest_pie_handle->pie_pogo_odma_udma->UICR = src_pie_handle->pie_pogo_odma_udma->UICR;
    dest_pie_handle->pie_pogo_odma_udma->UIFR = src_pie_handle->pie_pogo_odma_udma->UIFR;
    dest_pie_handle->pie_pogo_odma_udma->UDR = src_pie_handle->pie_pogo_odma_udma->UDR;
    dest_pie_handle->pie_pogo_odma_udma->UBAR = src_pie_handle->pie_pogo_odma_udma->UBAR;
    dest_pie_handle->pie_pogo_odma_udma->UBLR = src_pie_handle->pie_pogo_odma_udma->UBLR;
    dest_pie_handle->pie_pogo_odma_udma->UBRR = src_pie_handle->pie_pogo_odma_udma->UBRR;
    dest_pie_handle->pie_pogo_odma_udma->UTR0 = src_pie_handle->pie_pogo_odma_udma->UTR0;
    dest_pie_handle->pie_pogo_odma_udma->UTR1 = src_pie_handle->pie_pogo_odma_udma->UTR1;
}

void pie_output_dma_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE ODMA core regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_pogo_odma_udma);
    PIE_PTRCHECK_NR(pie_pogo_odma_core);

    print("OIER=0x%X\n", pie_handle->pie_pogo_odma_core->OIER);
    print("OIPR=0x%X\n", pie_handle->pie_pogo_odma_core->OIPR);
    // OICR and OIFR are Write only
    print("OLWR=0x%X\n", pie_handle->pie_pogo_odma_core->OLWR);
    print("OCNTRL=0x%X\n", pie_handle->pie_pogo_odma_core->OCNTRL);
    print("OTR0=0x%X\n", pie_handle->pie_pogo_odma_core->OTR0);
    print("OTR1=0x%X\n", pie_handle->pie_pogo_odma_core->OTR1);
    print("OTR2=0x%X\n", pie_handle->pie_pogo_odma_core->OTR2);
    print("PIE ODMA udma regs (pie_handle)\n");
    print("UCR=0x%X\n", pie_handle->pie_pogo_odma_udma->UCR);
    print("USR=0x%X\n", pie_handle->pie_pogo_odma_udma->USR);
    print("UPR=0x%X\n", pie_handle->pie_pogo_odma_udma->UPR);
    print("UIER=0x%X\n", pie_handle->pie_pogo_odma_udma->UIER);
    print("UIPR=0x%X\n", pie_handle->pie_pogo_odma_udma->UIPR);
    // UICR and UIFR are Write only
    print("UDR=0x%X\n", pie_handle->pie_pogo_odma_udma->UDR);
    print("UBAR=0x%X\n", pie_handle->pie_pogo_odma_udma->UBAR);
    print("UBLR=0x%X\n", pie_handle->pie_pogo_odma_udma->UBLR);
    print("UBRR=0x%X\n", pie_handle->pie_pogo_odma_udma->UBRR);
    print("UTR0=0x%X\n", pie_handle->pie_pogo_odma_udma->UTR0);
    print("UTR1=0x%X\n", pie_handle->pie_pogo_odma_udma->UTR1);
}
EXPORT_SYMBOL(pie_output_dma_dump_handle_regs);

// a dump_enable value of 1 dumps the data, a value of 0 process data normally through the odma
void pie_odma_set_dump_data(struct pie_handle_t *pie_handle, uint8_t dump_data)
{
    PIE_PTRCHECK_NR(pie_pogo_odma_core);

    pie_handle->pie_pogo_odma_core->OCNTRL =
        POGO_ODMA_POGO_ODMA_CORE_OCNTRL_DUMPENABLE_REPLACE_VAL(pie_handle->pie_pogo_odma_core->OCNTRL, dump_data);
}
EXPORT_SYMBOL(pie_odma_set_dump_data);

void pie_output_dma_set_parms(struct pie_handle_t *pie_handle, uint8_t handshake,
                              uint8_t ownership, uint8_t own_polarity, uint8_t enable,
                              uint32_t linewidth)
{
    int burstsize, beats, buswidth, fifodepth;
    uint8_t beats_reg_val, asic_reg_bus_width;

    PIE_PTRCHECK_NR(pie_pogo_odma_udma);
    PIE_PTRCHECK_NR(pie_pogo_odma_core);
    
    // beats: if PIE ODMA has a 64 bit wide bus
    //      4 "beats" (transfers) is 64 bits * 4, 32 byte burst
    //  and 8 "beats" (transfers) is 64 bits * 8, 64 byte burst
    // Calculate the right number of beats to give a burst that half empties the fifo
    // (which gives us double buffering from the fifo).

    // when we created the pie_handle, the UPR register was read from the ASIC.  Use
    // that for the ODMA ASIC instantiation parms

    fifodepth = POGO_ODMA_POGO_ODMA_UDMA_UPR_FIFODEPTH_MASK_SHIFT(pie_handle->pie_pogo_odma_udma->UPR);
    asic_reg_bus_width = POGO_ODMA_POGO_ODMA_UDMA_UPR_BUSWIDTH_MASK_SHIFT(pie_handle->pie_pogo_odma_udma->UPR);
    
    // for double buffering, we want to make the UDMA burst size 1/2 the size of the fifo
    burstsize = fifodepth/2;
    switch(asic_reg_bus_width)
    {
    case BUS_WIDTH_32_BIT:
        // Bus width is 4 Bytes (32 bits)
        buswidth = 4;
        break;
    case BUS_WIDTH_64_BIT:
        // Bus width is 8 Bytes (64 bits)
        buswidth = 8;
        break;
    default:
        error_print("ERROR, UNSUPPORTED BUS WIDTH: 0x%X\n",asic_reg_bus_width);
        buswidth = 4;
    }
    beats = burstsize/buswidth;
    if (beats >= 16)
    {
        beats_reg_val = DMA_16_BEATS;
        beats = 16;
    }
    else if (beats >= 8)
    {
        beats_reg_val = DMA_8_BEATS;
        beats = 8;
    }
    else
    {
        beats_reg_val = DMA_4_BEATS;
        beats = 4;
    }

    burstsize = beats * buswidth; // snap burstsize, if needed
    debug_print("%s: computed burstsize is %d\n", __func__, burstsize);

    // set burst length, enable, etc into UCR
    pie_handle->pie_pogo_odma_udma->UCR =
        POGO_ODMA_POGO_ODMA_UDMA_UCR_HANDSHAKEENABLE_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UCR, handshake);

    pie_handle->pie_pogo_odma_udma->UCR =
        POGO_ODMA_POGO_ODMA_UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UCR, ownership);

    pie_handle->pie_pogo_odma_udma->UCR =
        POGO_ODMA_POGO_ODMA_UDMA_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UCR, own_polarity);

    pie_handle->pie_pogo_odma_udma->UCR =
        POGO_ODMA_POGO_ODMA_UDMA_UCR_BEATS_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UCR, beats_reg_val);
    
    pie_handle->pie_pogo_odma_udma->UCR =
        POGO_ODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_pogo_odma_udma->UCR, enable);

    // now set OLWR
    pie_handle->pie_pogo_odma_core->OLWR =
        POGO_ODMA_POGO_ODMA_CORE_OLWR_LINEWIDTH_REPLACE_VAL(pie_handle->pie_pogo_odma_core->OLWR, linewidth);
}
EXPORT_SYMBOL(pie_output_dma_set_parms);

//
// pie odma interrupt functions
//

void pie_enable_pogo_odma_irqs(struct pie_handle_t *pie_handle,
                               struct odma_interrupt_info *irqstruct,
                               bool enable)
{
    struct odma_interrupt_info irq;

    PIE_PTRCHECK_NR(pie_pogo_odma_udma);
    PIE_PTRCHECK_NR(pie_pogo_odma_core);    

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_odma(&irq, true);
        irqstruct = &irq;
    }
    pie_handle->pie_pogo_odma_udma->UIER =
        pie_odma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                      pie_handle->pie_pogo_odma_udma->UIER,
                                                      enable);
    pie_handle->pie_pogo_odma_core->OIER =
        pie_odma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                      pie_handle->pie_pogo_odma_core->OIER,
                                                      enable);
}
EXPORT_SYMBOL(pie_enable_pogo_odma_irqs);

// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs

void pie_force_pogo_odma_irqs(struct pie_handle_t *pie_handle,
                              struct odma_interrupt_info *irqstruct,
                              bool force)
{
    struct odma_interrupt_info irq;

    PIE_PTRCHECK_NR(pie_pogo_odma_udma);
    PIE_PTRCHECK_NR(pie_pogo_odma_core);    

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_odma(&irq, true);
        irqstruct = &irq;
    }
    // set/clear force irq bits
    pie_handle->pie_pogo_odma_udma->UIFR = 
        pie_odma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                      pie_handle->pie_pogo_odma_udma->UIFR,
                                                      force);

    pie_handle->pie_pogo_odma_core->OIFR =
        pie_odma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                      pie_handle->pie_pogo_odma_core->OIFR,
                                                      force);
}
EXPORT_SYMBOL(pie_force_pogo_odma_irqs);

void pie_register_odma_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);
    
    pie_handle->pie_odma_udma_callback = callback_fcn;
    pie_handle->pie_odma_udma_callback_data = callback_data;    
    pie_handle->pie_odma_core_callback = callback_fcn;
    pie_handle->pie_odma_core_callback_data = callback_data;    
}
EXPORT_SYMBOL(pie_register_odma_callback);


////////////////////////////////////////////////////////////////////////
// pie ot pogoizer
////////////////////////////////////////////////////////////////////////
static void get_otpogoizer_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    // pogoizer subblock
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = 0;
    regval = OTPOGOIZER_CFG_BPP_REPLACE_VAL(regval, 2);
    regval = OTPOGOIZER_CFG_FMT_REPLACE_VAL(regval, 6);
    pie_handle->pie_otpogoizer->Cfg = regval;
    pie_handle->pie_otpogoizer->LPad = 0;
    pie_handle->pie_otpogoizer->RPad = 0;
    pie_handle->pie_otpogoizer->LVal0 = 0;
    pie_handle->pie_otpogoizer->RVal0 = 0;
    pie_handle->pie_otpogoizer->LTrunc = 0;
    pie_handle->pie_otpogoizer->RTrunc = OTPOGOIZER_RTRUNC_LASTCOLUMN_REPLACE_VAL(0, 0xFFFF);;    
    pie_handle->pie_otpogoizer->REV0 = OTPOGOIZER_MAJ_REV;
    pie_handle->pie_otpogoizer->REV1 = 0;
}

static void pie_otpogoizer_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otpogoizer);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otpogoizer);

    dest_pie_handle->pie_otpogoizer->Cfg = src_pie_handle->pie_otpogoizer->Cfg;
    dest_pie_handle->pie_otpogoizer->LPad = src_pie_handle->pie_otpogoizer->LPad;
    dest_pie_handle->pie_otpogoizer->RPad = src_pie_handle->pie_otpogoizer->RPad;
    dest_pie_handle->pie_otpogoizer->LVal0 = src_pie_handle->pie_otpogoizer->LVal0;
    dest_pie_handle->pie_otpogoizer->RVal0 = src_pie_handle->pie_otpogoizer->RVal0;
    dest_pie_handle->pie_otpogoizer->LTrunc = src_pie_handle->pie_otpogoizer->LTrunc;
    dest_pie_handle->pie_otpogoizer->RTrunc = src_pie_handle->pie_otpogoizer->RTrunc;    
    dest_pie_handle->pie_otpogoizer->REV0 = src_pie_handle->pie_otpogoizer->REV0;        
    dest_pie_handle->pie_otpogoizer->REV1 = src_pie_handle->pie_otpogoizer->REV1;        
}

void pie_otpogoizer_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE OT pogoizer regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_otpogoizer);
    
    print("Cfg=0x%X\n",  pie_handle->pie_otpogoizer->Cfg);
    print("LPad=0x%X\n",  pie_handle->pie_otpogoizer->LPad);
    print("RPad=0x%X\n",  pie_handle->pie_otpogoizer->RPad);
    print("LVal0=0x%X\n",  pie_handle->pie_otpogoizer->LVal0);
    print("RVal0=0x%X\n",  pie_handle->pie_otpogoizer->RVal0);
    print("LTrunc=0x%X\n",  pie_handle->pie_otpogoizer->LTrunc);
    print("RTrunc=0x%X\n",  pie_handle->pie_otpogoizer->RTrunc);    
    print("REV0=0x%X\n",  pie_handle->pie_otpogoizer->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_otpogoizer->REV1);
    
}
EXPORT_SYMBOL(pie_otpogoizer_dump_handle_regs);

int pie_ot_pogoizer_set_config(struct pie_handle_t *pie_handle, uint8_t ot_inputformat,
                               uint8_t pogo_ot_bpp)
{
    PIE_PTRCHECK(pie_otpogoizer, -1);

    pie_handle->pie_otpogoizer->Cfg =
        OTPOGOIZER_CFG_BPP_REPLACE_VAL(pie_handle->pie_otpogoizer->Cfg, pogo_ot_bpp);

    pie_handle->pie_otpogoizer->Cfg =
        OTPOGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_otpogoizer->Cfg, ot_inputformat);

    return 0;     
}
EXPORT_SYMBOL(pie_ot_pogoizer_set_config);

int pie_ot_pogoizer_set_input_format(struct pie_handle_t *pie_handle, uint32_t pogo_ot_bpp,
                                     bool pack_ot_data)
{
    uint32_t config_bpp;
    uint32_t ot_outputformat;
    PIE_PTRCHECK(pie_otpogoizer, -1);

    switch (pogo_ot_bpp)
    {
    case 1: config_bpp = OTPOGO_1BPP; break;
    case 2: config_bpp = OTPOGO_2BPP; break;
    case 4: config_bpp = OTPOGO_4BPP; break;
    case 8: config_bpp = OTPOGO_8BPP; break;
    default: return -EINVAL;
    }

    if (pack_ot_data)
    {
        ot_outputformat = POGO_FMT_1CHAN_PK;
    }
    else
    {
        ot_outputformat = POGO_FMT_1CHAN_UNPK;
    }

    pie_handle->pie_otpogoizer->Cfg =
        OTPOGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_otpogoizer->Cfg, ot_outputformat);

    pie_handle->pie_otpogoizer->Cfg =
        OTPOGOIZER_CFG_BPP_REPLACE_VAL(pie_handle->pie_otpogoizer->Cfg, config_bpp);

    return 0;
}
EXPORT_SYMBOL(pie_ot_pogoizer_set_input_format);

int pie_ot_pogoizer_get_valid_bits_per_pixel(struct pie_handle_t *pie_handle, uint32_t *pogo_ot_bpp)
{
    uint32_t config_bpp;
    PIE_PTRCHECK(pie_otpogoizer, -1);

    config_bpp = OTPOGOIZER_CFG_BPP_MASK_SHIFT(pie_handle->pie_otpogoizer->Cfg);

    switch (config_bpp)
    {
    case OTPOGO_1BPP: *pogo_ot_bpp = 1; break;
    case OTPOGO_2BPP: *pogo_ot_bpp = 2; break;
    case OTPOGO_4BPP: *pogo_ot_bpp = 4; break;
    case OTPOGO_8BPP: *pogo_ot_bpp = 8; break;
    default: return -EINVAL;
    }

    return 0;
}
EXPORT_SYMBOL(pie_ot_pogoizer_get_valid_bits_per_pixel);

int pie_ot_pogoizer_get_bits_per_pixel(struct pie_handle_t *pie_handle, uint32_t *pogo_ot_bpp)
{
    uint32_t config_bpp;
    PIE_PTRCHECK(pie_otpogoizer, -1);

    config_bpp = OTPOGOIZER_CFG_BPP_MASK_SHIFT(pie_handle->pie_otpogoizer->Cfg);

    switch (config_bpp)
    {
    case OTPOGO_1BPP: *pogo_ot_bpp = 1; break;
    case OTPOGO_2BPP: *pogo_ot_bpp = 2; break;
    case OTPOGO_4BPP: *pogo_ot_bpp = 4; break;
    case OTPOGO_8BPP: *pogo_ot_bpp = 8; break;
    default: return -EINVAL;
    }

    if (OTPOGOIZER_CFG_FMT_MASK_SHIFT(pie_handle->pie_otpogoizer->Cfg) == POGO_FMT_1CHAN_UNPK)
    {
        *pogo_ot_bpp = 8;
    }
    return 0;
}
EXPORT_SYMBOL(pie_ot_pogoizer_get_bits_per_pixel);

////////////////////////////////////////////////////////////////////////
// pie OT idma
////////////////////////////////////////////////////////////////////////

static void get_pogo_otidma_udma_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    OT_POGO_IDMA_POGO_IDMA_UDMA_REGS_t *idma_handle;
    
    idma_handle = pie_handle->pie_otpogo_idma_udma;
    
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    idma_handle->UCR = 0;
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_DMAPAUSED_REPLACE_VAL(0, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_PAUSEAFTER_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_PAUSEBEFORE_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_INTONFINISH_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_STOPONFINISH_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_CMDBUFFULL_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_CMDBUFEMPTY_REPLACE_VAL(regval, 1);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_DATABUFFULL_REPLACE_VAL(regval, 0);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_DATABUFEMPTY_REPLACE_VAL(regval, 1);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_CLEAR_REPLACE_VAL(regval, 1);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_USR_BUSY_REPLACE_VAL(regval, 0);
    idma_handle->USR = regval;
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_UPR_FIFODEPTH_REPLACE_VAL(0, 64);
    regval = OT_POGO_IDMA_POGO_IDMA_UDMA_UPR_BUSWIDTH_REPLACE_VAL(regval, 1);
    idma_handle->UPR = regval;  // block instantiation parms - user should compare against ASIC vals
    idma_handle->UIER = 0;
    idma_handle->UIPR = 0;
    idma_handle->UICR = 0;
    idma_handle->UIFR = 0;
    idma_handle->UDR = 0;
    idma_handle->UBAR = 0;
    idma_handle->UBLR = 0;
    idma_handle->UBRR = 0;
    idma_handle->UTR0 = POGO_OTIDMA_UDMA_MAJ_REV;
    idma_handle->UTR1 = 0;
}

static void get_pogo_otidma_core_default(struct pie_handle_t *pie_handle)
{
    OT_POGO_IDMA_POGO_IDMA_CORE_REGS_t *idma_handle;    

    idma_handle = pie_handle->pie_otpogo_idma_core;
    
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    idma_handle->ICR = 0;
    idma_handle->IIER = 0;
    idma_handle->IIPR = 0;
    idma_handle->IICR = 0;
    idma_handle->IIFR = 0;
    idma_handle->ILWR = 0;
    idma_handle->IRHR = 0;
    idma_handle->ITR0 = 0;
    idma_handle->ITR1 = POGO_OTIDMA_CORE_MAJ_REV;
    idma_handle->ITR2 = 1;
}

static void pie_ot_input_dma_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)

{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otpogo_idma_udma); 
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otpogo_idma_udma);
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otpogo_idma_core);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otpogo_idma_core);

    // clone the callbacks
    dest_pie_handle->pie_otidma_udma_callback = src_pie_handle->pie_otidma_udma_callback;
    dest_pie_handle->pie_otidma_udma_callback_data = src_pie_handle->pie_otidma_udma_callback_data;

    dest_pie_handle->pie_otidma_core_callback = src_pie_handle->pie_otidma_core_callback;
    dest_pie_handle->pie_otidma_core_callback_data = src_pie_handle->pie_otidma_core_callback_data;
    
    // clone the registers
    dest_pie_handle->pie_otpogo_idma_core->ICR = src_pie_handle->pie_otpogo_idma_core->ICR;
    dest_pie_handle->pie_otpogo_idma_core->IIER = src_pie_handle->pie_otpogo_idma_core->IIER;
    dest_pie_handle->pie_otpogo_idma_core->IIPR = src_pie_handle->pie_otpogo_idma_core->IIPR;
    dest_pie_handle->pie_otpogo_idma_core->IICR = src_pie_handle->pie_otpogo_idma_core->IICR;
    dest_pie_handle->pie_otpogo_idma_core->IIFR = src_pie_handle->pie_otpogo_idma_core->IIFR;    
    dest_pie_handle->pie_otpogo_idma_core->ILWR = src_pie_handle->pie_otpogo_idma_core->ILWR;
    dest_pie_handle->pie_otpogo_idma_core->IRHR = src_pie_handle->pie_otpogo_idma_core->IRHR;
    dest_pie_handle->pie_otpogo_idma_core->ITR0 = src_pie_handle->pie_otpogo_idma_core->ITR0;
    dest_pie_handle->pie_otpogo_idma_core->ITR1 = src_pie_handle->pie_otpogo_idma_core->ITR1;
    dest_pie_handle->pie_otpogo_idma_core->ITR2 = src_pie_handle->pie_otpogo_idma_core->ITR2;
    dest_pie_handle->pie_otpogo_idma_udma->UCR = src_pie_handle->pie_otpogo_idma_udma->UCR;  
    dest_pie_handle->pie_otpogo_idma_udma->USR = src_pie_handle->pie_otpogo_idma_udma->USR;  
    dest_pie_handle->pie_otpogo_idma_udma->UPR = src_pie_handle->pie_otpogo_idma_udma->UPR;  
    dest_pie_handle->pie_otpogo_idma_udma->UIER = src_pie_handle->pie_otpogo_idma_udma->UIER;
    dest_pie_handle->pie_otpogo_idma_udma->UIPR = src_pie_handle->pie_otpogo_idma_udma->UIPR;
    dest_pie_handle->pie_otpogo_idma_udma->UICR = src_pie_handle->pie_otpogo_idma_udma->UICR;
    dest_pie_handle->pie_otpogo_idma_udma->UIFR = src_pie_handle->pie_otpogo_idma_udma->UIFR;    
    dest_pie_handle->pie_otpogo_idma_udma->UDR = src_pie_handle->pie_otpogo_idma_udma->UDR;  
    dest_pie_handle->pie_otpogo_idma_udma->UBAR = src_pie_handle->pie_otpogo_idma_udma->UBAR;
    dest_pie_handle->pie_otpogo_idma_udma->UBLR = src_pie_handle->pie_otpogo_idma_udma->UBLR;
    dest_pie_handle->pie_otpogo_idma_udma->UBRR = src_pie_handle->pie_otpogo_idma_udma->UBRR;
    dest_pie_handle->pie_otpogo_idma_udma->UTR0 = src_pie_handle->pie_otpogo_idma_udma->UTR0;
    dest_pie_handle->pie_otpogo_idma_udma->UTR1 = src_pie_handle->pie_otpogo_idma_udma->UTR1;    
}

void pie_ot_input_dma_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE OT IDMA core regs (pie_handle)\n");

    PIE_PTRCHECK_NR(pie_otpogo_idma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_idma_core);

    print("ICR=0x%X\n", pie_handle->pie_otpogo_idma_core->ICR);
    print("IIER=0x%X\n", pie_handle->pie_otpogo_idma_core->IIER);
    print("IIPR=0x%X\n", pie_handle->pie_otpogo_idma_core->IIPR);
    // IICR and IIFR are Write only
    print("ILWR=0x%X\n", pie_handle->pie_otpogo_idma_core->ILWR);
    print("IRHR=0x%X\n", pie_handle->pie_otpogo_idma_core->IRHR);
    print("ITR0=0x%X\n", pie_handle->pie_otpogo_idma_core->ITR0);
    print("ITR1=0x%X\n", pie_handle->pie_otpogo_idma_core->ITR1);
    print("ITR2=0x%X\n", pie_handle->pie_otpogo_idma_core->ITR2);
    print("PIE OT IDMA udma regs (pie_handle)\n");
    print("UCR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UCR);
    print("USR=0x%X\n", pie_handle->pie_otpogo_idma_udma->USR);
    print("UPR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UPR);
    print("UIER=0x%X\n", pie_handle->pie_otpogo_idma_udma->UIER);
    print("UIPR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UIPR);
    // UICR and UIFR are Write only
    print("UDR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UDR);
    print("UBAR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UBAR);
    print("UBLR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UBLR);
    print("UBRR=0x%X\n", pie_handle->pie_otpogo_idma_udma->UBRR);
    print("UTR0=0x%X\n", pie_handle->pie_otpogo_idma_udma->UTR0);
    print("UTR1=0x%X\n", pie_handle->pie_otpogo_idma_udma->UTR1);
}
EXPORT_SYMBOL(pie_ot_input_dma_dump_handle_regs);

void pie_ot_input_dma_set_parms(struct pie_handle_t *pie_handle, uint8_t handshake,
                                uint8_t ownership, uint8_t own_polarity, uint8_t enable,
                                uint32_t linewidth, uint8_t rowheight)
{
    int burstsize, beats, buswidth, fifodepth;
    uint8_t beats_reg_val, burstsize_reg_val, asic_reg_bus_width;

    PIE_PTRCHECK_NR(pie_otpogo_idma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_idma_core);
    
    // setup the OT idma udma register structs from the parms passed in

    // beats: if PIE OT IDMA has a 64 bit wide bus
    //      4 "beats" (transfers) is 64 bits * 4, 32 byte burst
    //  and 8 "beats" (transfers) is 64 bits * 8, 64 byte burst
    // Calculate the right number of beats to give a burst that half fills the fifo
    // (which gives us double buffering into the fifo).

    // when we created the pie_handle, the UPR register was read from the ASIC.  Use
    // that for the IDMA ASIC instantiation parms

    fifodepth = OT_POGO_IDMA_POGO_IDMA_UDMA_UPR_FIFODEPTH_MASK_SHIFT(pie_handle->pie_otpogo_idma_udma->UPR);
    asic_reg_bus_width = OT_POGO_IDMA_POGO_IDMA_UDMA_UPR_BUSWIDTH_MASK_SHIFT(pie_handle->pie_otpogo_idma_udma->UPR);
    
    // for double buffering, we want to make the UDMA burst size 1/2 the size of the fifo
    burstsize = fifodepth/2;
    switch(asic_reg_bus_width)
    {
    case BUS_WIDTH_32_BIT:
        // Bus width is 4 Bytes (32 bits)
        buswidth = 4;
        break;
    case BUS_WIDTH_64_BIT:
        // Bus width is 8 Bytes (64 bits)
        buswidth = 8;
        break;
    default:
        error_print("%s:ERROR, UNSUPPORTED BUS WIDTH: 0x%X\n", __func__, asic_reg_bus_width);
        buswidth = 4;
    }
    
    beats = burstsize/buswidth;
    if (beats >= 16)
    {
        beats_reg_val = DMA_16_BEATS;
        beats = 16;
    }
    else if (beats >= 8)
    {
        beats_reg_val = DMA_8_BEATS;
        beats = 8;
    }
    else
    {
        beats_reg_val = DMA_4_BEATS;
        beats = 4;
    }

    burstsize = beats * buswidth; // snap burstsize, if needed
    debug_print("%s: computed burstsize is %d\n", __func__, burstsize);

    // set burst length, enable, etc into UCR
    pie_handle->pie_otpogo_idma_udma->UCR =
        OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_HANDSHAKEENABLE_REPLACE_VAL(pie_handle->pie_otpogo_idma_udma->UCR, handshake);
    pie_handle->pie_otpogo_idma_udma->UCR =
        OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(pie_handle->pie_otpogo_idma_udma->UCR, ownership);
    pie_handle->pie_otpogo_idma_udma->UCR =
        OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_otpogo_idma_udma->UCR, own_polarity);
    pie_handle->pie_otpogo_idma_udma->UCR =
        OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_BEATS_REPLACE_VAL(pie_handle->pie_otpogo_idma_udma->UCR, beats_reg_val);
    pie_handle->pie_otpogo_idma_udma->UCR =
        OT_POGO_IDMA_POGO_IDMA_UDMA_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_otpogo_idma_udma->UCR, enable);
    
    // done calculating UCR parms, now for ICR...
    if (burstsize >= 64)
        burstsize_reg_val = CORE_BURST_SIZE_64;
    else if (burstsize >= 48)
        burstsize_reg_val = CORE_BURST_SIZE_48;
    else if (burstsize >= 32)
        burstsize_reg_val = CORE_BURST_SIZE_32;
    else
        burstsize_reg_val = CORE_BURST_SIZE_16;

    pie_handle->pie_otpogo_idma_core->ICR =
        OT_POGO_IDMA_POGO_IDMA_CORE_ICR_BURSTSIZE_REPLACE_VAL(pie_handle->pie_otpogo_idma_core->ICR, burstsize_reg_val);
    
    // now for ILWR and IRHR
    pie_handle->pie_otpogo_idma_core->ILWR =
        OT_POGO_IDMA_POGO_IDMA_CORE_ILWR_LINEWIDTH_REPLACE_VAL(pie_handle->pie_otpogo_idma_core->ILWR, linewidth);
    pie_handle->pie_otpogo_idma_core->IRHR =
        OT_POGO_IDMA_POGO_IDMA_CORE_IRHR_ROWHEIGHT_REPLACE_VAL(pie_handle->pie_otpogo_idma_core->IRHR, rowheight);
}
EXPORT_SYMBOL(pie_ot_input_dma_set_parms);

//
// pie OT idma interrupt functions
//

void pie_enable_pogo_otidma_irqs(struct pie_handle_t *pie_handle,
                                 struct idma_interrupt_info *irqstruct,
                                 bool enable)
{
    struct idma_interrupt_info irq;

    PIE_PTRCHECK_NR(pie_otpogo_idma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_idma_core);

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_idma(&irq, true);
        irqstruct = &irq;
    }
    pie_handle->pie_otpogo_idma_udma->UIER =
        pie_otidma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                        pie_handle->pie_otpogo_idma_udma->UIER,
                                                        enable);
    
    pie_handle->pie_otpogo_idma_core->IIER =
        pie_otidma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                        pie_handle->pie_otpogo_idma_core->IIER,
                                                        enable);
}
EXPORT_SYMBOL(pie_enable_pogo_otidma_irqs);

void pie_register_ot_idma_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);

    pie_handle->pie_otidma_udma_callback = callback_fcn;
    pie_handle->pie_otidma_udma_callback_data = callback_data;
    
    pie_handle->pie_otidma_core_callback = callback_fcn;
    pie_handle->pie_otidma_core_callback_data = callback_data;
}
EXPORT_SYMBOL(pie_register_ot_idma_callback);



////////////////////////////////////////////////////////////////////////
// pie ot depogoizer
////////////////////////////////////////////////////////////////////////
static void get_otdepogoizer_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    // depogoizer subblock
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = OTDEPOGOIZER_CFG_MONOCHAN_REPLACE_VAL(0, 0);
    regval = OTDEPOGOIZER_CFG_PIXORDER_REPLACE_VAL(regval, 0);
    regval = OTDEPOGOIZER_CFG_BPP_REPLACE_VAL(regval, 3);
    regval = OTDEPOGOIZER_CFG_COLORSWAP_REPLACE_VAL(regval, 0);
    regval = OTDEPOGOIZER_CFG_FMT_REPLACE_VAL(regval, 0);
    pie_handle->pie_otdepogoizer->Cfg = regval;        
    pie_handle->pie_otdepogoizer->REV0 = OTDEPOGOIZER_MAJ_REV;
    pie_handle->pie_otdepogoizer->REV1 = 0;
}

static void pie_otdepogoizer_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otdepogoizer);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otdepogoizer);

    dest_pie_handle->pie_otdepogoizer->Cfg = src_pie_handle->pie_otdepogoizer->Cfg;
    dest_pie_handle->pie_otdepogoizer->REV0 = src_pie_handle->pie_otdepogoizer->REV0;        
    dest_pie_handle->pie_otdepogoizer->REV1 = src_pie_handle->pie_otdepogoizer->REV1;        
}

void pie_otdepogoizer_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE OT depogoizer regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_otdepogoizer);
    
    print("Cfg=0x%X\n",  pie_handle->pie_otdepogoizer->Cfg);
    print("REV0=0x%X\n",  pie_handle->pie_otdepogoizer->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_otdepogoizer->REV1);    
}
EXPORT_SYMBOL(pie_otdepogoizer_dump_handle_regs);

int pie_ot_depogoizer_set_config(struct pie_handle_t *pie_handle, uint8_t ot_outputformat,
                                 uint8_t pogo_ot_bpp)
{
    PIE_PTRCHECK(pie_otdepogoizer, -1);

    pie_handle->pie_otdepogoizer->Cfg =
        OTDEPOGOIZER_CFG_BPP_REPLACE_VAL(pie_handle->pie_otdepogoizer->Cfg, pogo_ot_bpp);

    pie_handle->pie_otdepogoizer->Cfg =
        OTDEPOGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_otdepogoizer->Cfg, ot_outputformat);

    return 0;     
}
EXPORT_SYMBOL(pie_ot_depogoizer_set_config);

int pie_ot_depogoizer_set_output_format(struct pie_handle_t *pie_handle, uint32_t pogo_ot_bpp,
                                        bool pack_ot_data)
{
    uint32_t config_bpp;
    uint32_t ot_outputformat;
    PIE_PTRCHECK(pie_otdepogoizer, -1);

    switch (pogo_ot_bpp)
    {
    case 1: config_bpp = OTPOGO_1BPP; break;
    case 2: config_bpp = OTPOGO_2BPP; break;
    case 4: config_bpp = OTPOGO_4BPP; break;
    case 8: config_bpp = OTPOGO_8BPP; break;
    default: return -EINVAL;
    }

    if (pack_ot_data)
    {
        ot_outputformat = POGO_FMT_1CHAN_PK;
    }
    else
    {
        ot_outputformat = POGO_FMT_1CHAN_UNPK;
    }

    pie_handle->pie_otdepogoizer->Cfg =
        OTDEPOGOIZER_CFG_FMT_REPLACE_VAL(pie_handle->pie_otdepogoizer->Cfg, ot_outputformat);

    pie_handle->pie_otdepogoizer->Cfg =
        OTDEPOGOIZER_CFG_BPP_REPLACE_VAL(pie_handle->pie_otdepogoizer->Cfg, config_bpp);

    return 0;
}
EXPORT_SYMBOL(pie_ot_depogoizer_set_output_format);

int pie_ot_depogoizer_get_valid_bits_per_pixel(struct pie_handle_t *pie_handle, uint32_t *pogo_ot_bpp)
{
    uint32_t config_bpp;
    PIE_PTRCHECK(pie_otdepogoizer, -1);

    config_bpp = OTDEPOGOIZER_CFG_BPP_MASK_SHIFT(pie_handle->pie_otdepogoizer->Cfg);

    switch (config_bpp)
    {
    case OTPOGO_1BPP: *pogo_ot_bpp = 1; break;
    case OTPOGO_2BPP: *pogo_ot_bpp = 2; break;
    case OTPOGO_4BPP: *pogo_ot_bpp = 4; break;
    case OTPOGO_8BPP: *pogo_ot_bpp = 8; break;
    default: return -EINVAL;
    }

    return 0;
}
EXPORT_SYMBOL(pie_ot_depogoizer_get_valid_bits_per_pixel);

int pie_ot_depogoizer_get_bits_per_pixel(struct pie_handle_t *pie_handle, uint32_t *pogo_ot_bpp)
{
    uint32_t config_bpp;
    PIE_PTRCHECK(pie_otdepogoizer, -1);

    config_bpp = OTDEPOGOIZER_CFG_BPP_MASK_SHIFT(pie_handle->pie_otdepogoizer->Cfg);

    switch (config_bpp)
    {
    case OTPOGO_1BPP: *pogo_ot_bpp = 1; break;
    case OTPOGO_2BPP: *pogo_ot_bpp = 2; break;
    case OTPOGO_4BPP: *pogo_ot_bpp = 4; break;
    case OTPOGO_8BPP: *pogo_ot_bpp = 8; break;
    default: return -EINVAL;
    }

    if (OTDEPOGOIZER_CFG_FMT_MASK_SHIFT(pie_handle->pie_otdepogoizer->Cfg) == POGO_FMT_1CHAN_UNPK)
    {
        *pogo_ot_bpp = 8;
    }
    return 0;
}
EXPORT_SYMBOL(pie_ot_depogoizer_get_bits_per_pixel);

////////////////////////////////////////////////////////////////////////
// pie OT odma (udma and core)
////////////////////////////////////////////////////////////////////////

static void get_pogo_otodma_udma_default(struct pie_handle_t *pie_handle)
{
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_otpogo_odma_udma->UCR = 0;
    pie_handle->pie_otpogo_odma_udma->USR = 0;
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_DMAPAUSED_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_PAUSEAFTER_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_PAUSEBEFORE_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_INTONFINISH_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_STOPONFINISH_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_CMDBUFFULL_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_CMDBUFEMPTY_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 1);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_DATABUFFULL_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_DATABUFEMPTY_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 1);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_CLEAR_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 1);
    pie_handle->pie_otpogo_odma_udma->USR =
        POGO_OTODMA_POGO_ODMA_UDMA_USR_BUSY_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->USR, 0);
    pie_handle->pie_otpogo_odma_udma->UPR = 0;
    pie_handle->pie_otpogo_odma_udma->UPR =
        POGO_OTODMA_POGO_ODMA_UDMA_UPR_FIFODEPTH_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UPR, 64);
    pie_handle->pie_otpogo_odma_udma->UPR =
        POGO_OTODMA_POGO_ODMA_UDMA_UPR_BUSWIDTH_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UPR, 1);
    pie_handle->pie_otpogo_odma_udma->UIER = 0;
    pie_handle->pie_otpogo_odma_udma->UIPR = 0;
    pie_handle->pie_otpogo_odma_udma->UICR = 0;
    pie_handle->pie_otpogo_odma_udma->UIFR = 0;
    pie_handle->pie_otpogo_odma_udma->UDR = 0;
    pie_handle->pie_otpogo_odma_udma->UBAR = 0;
    pie_handle->pie_otpogo_odma_udma->UBLR = 0;
    pie_handle->pie_otpogo_odma_udma->UBRR = 0;
    pie_handle->pie_otpogo_odma_udma->UTR0 = POGO_OTODMA_UDMA_MAJ_REV;
    pie_handle->pie_otpogo_odma_udma->UTR1 = 0;
}

static void get_pogo_otodma_core_default(struct pie_handle_t *pie_handle)
{
    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    pie_handle->pie_otpogo_odma_core->OIER = 0; 
    pie_handle->pie_otpogo_odma_core->OIPR = 0; 
    pie_handle->pie_otpogo_odma_core->OICR = 0;
    pie_handle->pie_otpogo_odma_core->OIFR = 0; 
    pie_handle->pie_otpogo_odma_core->OLWR = 0; 
    pie_handle->pie_otpogo_odma_core->OCNTRL = 0;
    pie_handle->pie_otpogo_odma_core->OTR0 = 0;
    pie_handle->pie_otpogo_odma_core->OTR1 = POGO_OTODMA_CORE_MAJ_REV;
    pie_handle->pie_otpogo_odma_core->OTR2 = 8;
}

static void pie_ot_output_dma_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otpogo_odma_udma);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otpogo_odma_udma);
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otpogo_odma_core);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otpogo_odma_core);

    // clone the callbacks
    dest_pie_handle->pie_otodma_udma_callback = src_pie_handle->pie_otodma_udma_callback;
    dest_pie_handle->pie_otodma_udma_callback_data = src_pie_handle->pie_otodma_udma_callback_data;

    dest_pie_handle->pie_otodma_core_callback = src_pie_handle->pie_otodma_core_callback;
    dest_pie_handle->pie_otodma_core_callback_data = src_pie_handle->pie_otodma_core_callback_data;
    
    // clone the registers
    dest_pie_handle->pie_otpogo_odma_core->OIER = src_pie_handle->pie_otpogo_odma_core->OIER;
    dest_pie_handle->pie_otpogo_odma_core->OIPR = src_pie_handle->pie_otpogo_odma_core->OIPR;
    dest_pie_handle->pie_otpogo_odma_core->OICR = src_pie_handle->pie_otpogo_odma_core->OICR;
    dest_pie_handle->pie_otpogo_odma_core->OIFR = src_pie_handle->pie_otpogo_odma_core->OIFR;
    dest_pie_handle->pie_otpogo_odma_core->OLWR = src_pie_handle->pie_otpogo_odma_core->OLWR;
    dest_pie_handle->pie_otpogo_odma_core->OCNTRL = src_pie_handle->pie_otpogo_odma_core->OCNTRL;
    dest_pie_handle->pie_otpogo_odma_core->OTR0 = src_pie_handle->pie_otpogo_odma_core->OTR0;
    dest_pie_handle->pie_otpogo_odma_core->OTR1 = src_pie_handle->pie_otpogo_odma_core->OTR1;
    dest_pie_handle->pie_otpogo_odma_core->OTR2 = src_pie_handle->pie_otpogo_odma_core->OTR2;

    dest_pie_handle->pie_otpogo_odma_udma->UCR = src_pie_handle->pie_otpogo_odma_udma->UCR;
    dest_pie_handle->pie_otpogo_odma_udma->USR = src_pie_handle->pie_otpogo_odma_udma->USR;
    dest_pie_handle->pie_otpogo_odma_udma->UPR = src_pie_handle->pie_otpogo_odma_udma->UPR;
    dest_pie_handle->pie_otpogo_odma_udma->UIER = src_pie_handle->pie_otpogo_odma_udma->UIER;
    dest_pie_handle->pie_otpogo_odma_udma->UIPR = src_pie_handle->pie_otpogo_odma_udma->UIPR;
    dest_pie_handle->pie_otpogo_odma_udma->UICR = src_pie_handle->pie_otpogo_odma_udma->UICR;
    dest_pie_handle->pie_otpogo_odma_udma->UIFR = src_pie_handle->pie_otpogo_odma_udma->UIFR;
    dest_pie_handle->pie_otpogo_odma_udma->UDR = src_pie_handle->pie_otpogo_odma_udma->UDR;
    dest_pie_handle->pie_otpogo_odma_udma->UBAR = src_pie_handle->pie_otpogo_odma_udma->UBAR;
    dest_pie_handle->pie_otpogo_odma_udma->UBLR = src_pie_handle->pie_otpogo_odma_udma->UBLR;
    dest_pie_handle->pie_otpogo_odma_udma->UBRR = src_pie_handle->pie_otpogo_odma_udma->UBRR;
    dest_pie_handle->pie_otpogo_odma_udma->UTR0 = src_pie_handle->pie_otpogo_odma_udma->UTR0;
    dest_pie_handle->pie_otpogo_odma_udma->UTR1 = src_pie_handle->pie_otpogo_odma_udma->UTR1;
}

void pie_ot_output_dma_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE OT ODMA core regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_otpogo_odma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_odma_core);

    print("OIER=0x%X\n", pie_handle->pie_otpogo_odma_core->OIER);
    print("OIPR=0x%X\n", pie_handle->pie_otpogo_odma_core->OIPR);
    // OICR and OIFR are Write only
    print("OLWR=0x%X\n", pie_handle->pie_otpogo_odma_core->OLWR);
    print("OCNTRL=0x%X\n", pie_handle->pie_otpogo_odma_core->OCNTRL);
    print("OTR0=0x%X\n", pie_handle->pie_otpogo_odma_core->OTR0);
    print("OTR1=0x%X\n", pie_handle->pie_otpogo_odma_core->OTR1);
    print("OTR2=0x%X\n", pie_handle->pie_otpogo_odma_core->OTR2);
    print("PIE OT ODMA udma regs (pie_handle)\n");
    print("UCR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UCR);
    print("USR=0x%X\n", pie_handle->pie_otpogo_odma_udma->USR);
    print("UPR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UPR);
    print("UIER=0x%X\n", pie_handle->pie_otpogo_odma_udma->UIER);
    print("UIPR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UIPR);
    // UICR and UIFR are Write only
    print("UDR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UDR);
    print("UBAR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UBAR);
    print("UBLR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UBLR);
    print("UBRR=0x%X\n", pie_handle->pie_otpogo_odma_udma->UBRR);
    print("UTR0=0x%X\n", pie_handle->pie_otpogo_odma_udma->UTR0);
    print("UTR1=0x%X\n", pie_handle->pie_otpogo_odma_udma->UTR1);
}
EXPORT_SYMBOL(pie_ot_output_dma_dump_handle_regs);

void pie_ot_output_dma_set_parms(struct pie_handle_t *pie_handle, uint8_t handshake,
                                 uint8_t ownership, uint8_t own_polarity, uint8_t enable,
                                 uint32_t linewidth)
{
    int burstsize, beats, buswidth, fifodepth;
    uint8_t beats_reg_val, asic_reg_bus_width;

    PIE_PTRCHECK_NR(pie_otpogo_odma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_odma_core);

    // beats: if PIE ODMA has a 64 bit wide bus
    //      4 "beats" (transfers) is 64 bits * 4, 32 byte burst
    //  and 8 "beats" (transfers) is 64 bits * 8, 64 byte burst
    // Calculate the right number of beats to give a burst that half empties the fifo
    // (which gives us double buffering from the fifo).

    // when we created the pie_handle, the UPR register was read from the ASIC.  Use
    // that for the ODMA ASIC instantiation parms

    fifodepth = POGO_OTODMA_POGO_ODMA_UDMA_UPR_FIFODEPTH_MASK_SHIFT(pie_handle->pie_otpogo_odma_udma->UPR);
    asic_reg_bus_width = POGO_OTODMA_POGO_ODMA_UDMA_UPR_BUSWIDTH_MASK_SHIFT(pie_handle->pie_otpogo_odma_udma->UPR);
    
    // for double buffering, we want to make the UDMA burst size 1/2 the size of the fifo
    burstsize = fifodepth/2;
    switch(asic_reg_bus_width)
    {
    case BUS_WIDTH_32_BIT:
        // Bus width is 4 Bytes (32 bits)
        buswidth = 4;
        break;
    case BUS_WIDTH_64_BIT:
        // Bus width is 8 Bytes (64 bits)
        buswidth = 8;
        break;
    default:
        error_print("ERROR, UNSUPPORTED BUS WIDTH: 0x%X\n",asic_reg_bus_width);
        buswidth = 4;
    }
    beats = burstsize/buswidth;
    if (beats >= 16)
    {
        beats_reg_val = DMA_16_BEATS;
        beats = 16;
    }
    else if (beats >= 8)
    {
        beats_reg_val = DMA_8_BEATS;
        beats = 8;
    }
    else
    {
        beats_reg_val = DMA_4_BEATS;
        beats = 4;
    }

    burstsize = beats * buswidth; // snap burstsize, if needed
    debug_print("%s: computed burstsize is %d\n", __func__, burstsize);


    // set burst length, enable, etc into UCR
    pie_handle->pie_otpogo_odma_udma->UCR =
        POGO_OTODMA_POGO_ODMA_UDMA_UCR_HANDSHAKEENABLE_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UCR, handshake);

    pie_handle->pie_otpogo_odma_udma->UCR =
        POGO_OTODMA_POGO_ODMA_UDMA_UCR_OWNWRITEDISABLE_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UCR, ownership);

    pie_handle->pie_otpogo_odma_udma->UCR =
        POGO_OTODMA_POGO_ODMA_UDMA_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UCR, own_polarity);

    pie_handle->pie_otpogo_odma_udma->UCR =
        POGO_OTODMA_POGO_ODMA_UDMA_UCR_BEATS_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UCR, beats_reg_val);
    
    pie_handle->pie_otpogo_odma_udma->UCR =
        POGO_OTODMA_POGO_ODMA_UDMA_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_otpogo_odma_udma->UCR, enable);

    // now set OLWR
    pie_handle->pie_otpogo_odma_core->OLWR =
        POGO_OTODMA_POGO_ODMA_CORE_OLWR_LINEWIDTH_REPLACE_VAL(pie_handle->pie_otpogo_odma_core->OLWR, linewidth);
}
EXPORT_SYMBOL(pie_ot_output_dma_set_parms);

//
// pie OT odma interrupt functions
//

void pie_enable_pogo_otodma_irqs(struct pie_handle_t *pie_handle,
                                 struct odma_interrupt_info *irqstruct,
                                 bool enable)
{
    struct odma_interrupt_info irq;

    PIE_PTRCHECK_NR(pie_otpogo_odma_udma);
    PIE_PTRCHECK_NR(pie_otpogo_odma_core);    

    if (irqstruct == NULL)
    {
        pie_pogo_set_all_irqstruct_odma(&irq, true);  // same irqs as pogo_odma
        irqstruct = &irq;
    }
    pie_handle->pie_otpogo_odma_udma->UIER =
        pie_otodma_convert_irqstruct_to_update_udma_reg(irqstruct,
                                                        pie_handle->pie_otpogo_odma_udma->UIER,
                                                        enable);
    pie_handle->pie_otpogo_odma_core->OIER =
        pie_otodma_convert_irqstruct_to_update_core_reg(irqstruct,
                                                        pie_handle->pie_otpogo_odma_core->OIER,
                                                        enable);
}
EXPORT_SYMBOL(pie_enable_pogo_otodma_irqs);


void pie_register_ot_odma_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);
    
    pie_handle->pie_otodma_udma_callback = callback_fcn;
    pie_handle->pie_otodma_udma_callback_data = callback_data;    
    pie_handle->pie_otodma_core_callback = callback_fcn;
    pie_handle->pie_otodma_core_callback_data = callback_data;    
}
EXPORT_SYMBOL(pie_register_ot_odma_callback);

////////////////////////////////////////////////////////////////////////
// pie bde
////////////////////////////////////////////////////////////////////////

static void get_bde_default(struct pie_handle_t *pie_handle, int channel)
{
    int i;
    
    pie_handle->pie_bde[channel]->REV0 = PIE_BDE_MAJ_REV0;
    pie_handle->pie_bde[channel]->REV1 = 0;
    pie_handle->pie_bde[channel]->LUT_CR = 1;
    
    for (i=0; i<PIE_BDE_MAX_LUT_ENTRIES; i++)
    {
        pie_handle->pie_bde[channel]->LUT0[i] = 0;
    }
}

static void pie_bde_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle, int channel)
{
    int i;
    
    PIE_PTRCHECK_NAME_ARRY_NR(src_pie_handle,  pie_bde, channel);
    PIE_PTRCHECK_NAME_ARRY_NR(dest_pie_handle, pie_bde, channel);
    
    dest_pie_handle->pie_bde[channel]->REV0 = src_pie_handle->pie_bde[channel]->REV0;
    dest_pie_handle->pie_bde[channel]->REV1 = src_pie_handle->pie_bde[channel]->REV1;
    dest_pie_handle->pie_bde[channel]->LUT_CR = src_pie_handle->pie_bde[channel]->LUT_CR;
    for (i=0; i<PIE_BDE_MAX_LUT_ENTRIES; i++)
        dest_pie_handle->pie_bde[channel]->LUT0[i] = src_pie_handle->pie_bde[channel]->LUT0[i];

}

void pie_bde_dump_handle_regs(struct pie_handle_t *pie_handle, int channel)
{
    print("PIE BDE regs (pie_handle)\n");

    PIE_PTRCHECK_ARRY_NR(pie_bde, channel);

    print("REV0      = 0x%X\n", pie_handle->pie_bde[channel]->REV0);
    print("REV1      = 0x%X\n", pie_handle->pie_bde[channel]->REV1);
    print("LUT_CR    = 0x%X\n", pie_handle->pie_bde[channel]->LUT_CR);
}
EXPORT_SYMBOL(pie_bde_dump_handle_regs);

void pie_bde_dump_handle_lut(struct pie_handle_t *pie_handle, int channel)
{
    int i;

    PIE_PTRCHECK_ARRY_NR(pie_bde, channel);

    print("PIE BDE dump LUT (pie_handle) channel %d\n", channel);
    for (i=0;i<PIE_BDE_MAX_LUT_ENTRIES; i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        print("0x%X ", pie_handle->pie_bde[channel]->LUT0[i]);
    }
    print("\n");
}
EXPORT_SYMBOL(pie_bde_dump_handle_lut);

void pie_bde_set_lut(struct pie_handle_t *pie_handle, uint32_t *data, uint32_t num_entries, int channel)
{
    int i;

    PIE_PTRCHECK_ARRY_NR(pie_bde, channel);

    for (i=0;i<num_entries;i++)
    {
        pie_handle->pie_bde[channel]->LUT0[i] =
            DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT0_LUT_VAL_REPLACE_VAL(0,data[i]);
    }
}
EXPORT_SYMBOL(pie_bde_set_lut);

// set/clear all bypass bits - it doesn't cause any problems to always
// set all 3 when only 1 is needed, and can enhance performance
void pie_bde_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    int i;

    for (i=0;i<NUM_BDE_CHANNELS;i++)
    {
        PIE_PTRCHECK_ARRY_NR(pie_bde, i);
        if (bypass == true)
            pie_handle->pie_bde[i]->LUT_CR =
                DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(pie_handle->pie_bde[i]->LUT_CR,
                                                                             PIE_BDE_BYPASS);
        else
            pie_handle->pie_bde[i]->LUT_CR =
                DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(pie_handle->pie_bde[i]->LUT_CR,
                                                                             PIE_BDE_ENABLE);
    }
}
EXPORT_SYMBOL(pie_bde_set_bypass);


////////////////////////////////////////////////////////////////////////
// pie decim (decimator)
////////////////////////////////////////////////////////////////////////

static void get_decim_default(struct pie_handle_t *pie_handle)
{
    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    pie_handle->pie_decim->div   = 0;
    pie_handle->pie_decim->start = 0;
    pie_handle->pie_decim->REV0 = PIE_DECIM_MAJ_REV0;
    pie_handle->pie_decim->REV1 = 0;
}

static void pie_decim_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_decim);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_decim);
    
    dest_pie_handle->pie_decim->div   = src_pie_handle->pie_decim->div;
    dest_pie_handle->pie_decim->start = src_pie_handle->pie_decim->start;    
    dest_pie_handle->pie_decim->REV0  = src_pie_handle->pie_decim->REV0;
    dest_pie_handle->pie_decim->REV1  = src_pie_handle->pie_decim->REV1;
}

void pie_decim_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE DECIM regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_decim);

    print("div   = 0x%X\n",  pie_handle->pie_decim->div);
    print("start = 0x%X\n",  pie_handle->pie_decim->start);
    print("REV0  = 0x%X\n",  pie_handle->pie_decim->REV0);
    print("REV1  = 0x%X\n",  pie_handle->pie_decim->REV1);
}
EXPORT_SYMBOL(pie_decim_dump_handle_regs);

void pie_decim_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_decim);

    // there is no bypass bit, but setting the regs to 0 lets data pass through
    if (bypass == true)
    {
        pie_handle->pie_decim->div = 0;
        pie_handle->pie_decim->start = 0;
    }
}
EXPORT_SYMBOL(pie_decim_set_bypass);

void pie_decim_set_div(struct pie_handle_t *pie_handle, uint8_t div_x, uint8_t div_y)
{
    pie_handle->pie_decim->div =
        DECIM_DIV_DIV_X_REPLACE_VAL(pie_handle->pie_decim->div, div_x);
    pie_handle->pie_decim->div =
        DECIM_DIV_DIV_Y_REPLACE_VAL(pie_handle->pie_decim->div, div_y);
    
}
EXPORT_SYMBOL(pie_decim_set_div);

void pie_decim_set_start(struct pie_handle_t *pie_handle, uint8_t start)
{
    pie_handle->pie_decim->start =
        DECIM_START_Y_START_POS_REPLACE_VAL(pie_handle->pie_decim->start, start);
    
}
EXPORT_SYMBOL(pie_decim_set_start);

////////////////////////////////////////////////////////////////////////
// pie clippad
////////////////////////////////////////////////////////////////////////

static void get_clippad_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = CLIPPAD_CFG_PIXORDER_REPLACE_VAL(regval, 1);
    regval = CLIPPAD_CFG_BYPASS_REPLACE_VAL(regval, 1);
    pie_handle->pie_clippad->Cfg = regval;
    pie_handle->pie_clippad->LPad = 0;
    pie_handle->pie_clippad->RPad = 0;
    pie_handle->pie_clippad->LVal0 = 0;
    pie_handle->pie_clippad->LVal1 = 0;
    pie_handle->pie_clippad->LVal2 = 0;
    pie_handle->pie_clippad->LVal3 = 0;
    pie_handle->pie_clippad->RVal0 = 0;
    pie_handle->pie_clippad->RVal1 = 0;
    pie_handle->pie_clippad->RVal2 = 0;
    pie_handle->pie_clippad->RVal3 = 0;
    pie_handle->pie_clippad->LTrunc = 0;
    pie_handle->pie_clippad->RTrunc = 0xFFFF;
    pie_handle->pie_clippad->LineWidth = 0;
    pie_handle->pie_clippad->CLIPPADPR = 0;
    pie_handle->pie_clippad->REV0 = PIE_CLIPPAD_MAJ_REV0;
    pie_handle->pie_clippad->REV1 = 0;
}

static void pie_clippad_clone_regs(struct pie_handle_t *src_pie_handle,
                                   struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_clippad);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_clippad);
    
    dest_pie_handle->pie_clippad->Cfg = src_pie_handle->pie_clippad->Cfg;
    dest_pie_handle->pie_clippad->LPad = src_pie_handle->pie_clippad->LPad;
    dest_pie_handle->pie_clippad->RPad = src_pie_handle->pie_clippad->RPad;
    dest_pie_handle->pie_clippad->LVal0 = src_pie_handle->pie_clippad->LVal0;
    dest_pie_handle->pie_clippad->LVal1 = src_pie_handle->pie_clippad->LVal1;
    dest_pie_handle->pie_clippad->LVal2 = src_pie_handle->pie_clippad->LVal2;
    dest_pie_handle->pie_clippad->LVal3 = src_pie_handle->pie_clippad->LVal3;
    dest_pie_handle->pie_clippad->RVal0 = src_pie_handle->pie_clippad->RVal0;
    dest_pie_handle->pie_clippad->RVal1 = src_pie_handle->pie_clippad->RVal1;
    dest_pie_handle->pie_clippad->RVal2 = src_pie_handle->pie_clippad->RVal2;
    dest_pie_handle->pie_clippad->RVal3 = src_pie_handle->pie_clippad->RVal3;
    dest_pie_handle->pie_clippad->LTrunc = src_pie_handle->pie_clippad->LTrunc;
    dest_pie_handle->pie_clippad->RTrunc = src_pie_handle->pie_clippad->RTrunc;
    dest_pie_handle->pie_clippad->LineWidth = src_pie_handle->pie_clippad->LineWidth;
    dest_pie_handle->pie_clippad->CLIPPADPR = src_pie_handle->pie_clippad->CLIPPADPR;
    dest_pie_handle->pie_clippad->REV0 = src_pie_handle->pie_clippad->REV0;
    dest_pie_handle->pie_clippad->REV1 = src_pie_handle->pie_clippad->REV1;
}

void pie_clippad_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE CLIPPAD regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_clippad);

    print("Cfg       = 0x%X\n",  pie_handle->pie_clippad->Cfg);
    print("LPad      = 0x%X\n", pie_handle->pie_clippad->LPad);
    print("RPad      = 0x%X\n", pie_handle->pie_clippad->RPad);
    print("LVal0     = 0x%X\n", pie_handle->pie_clippad->LVal0);
    print("LVal1     = 0x%X\n", pie_handle->pie_clippad->LVal1);
    print("LVal2     = 0x%X\n", pie_handle->pie_clippad->LVal2);
    print("LVal3     = 0x%X\n", pie_handle->pie_clippad->LVal3);
    print("RVal0     = 0x%X\n", pie_handle->pie_clippad->RVal0);
    print("RVal1     = 0x%X\n", pie_handle->pie_clippad->RVal1);
    print("RVal2     = 0x%X\n", pie_handle->pie_clippad->RVal2);
    print("RVal3     = 0x%X\n", pie_handle->pie_clippad->RVal3);
    print("LTrunc    = 0x%X\n", pie_handle->pie_clippad->LTrunc);
    print("RTrunc    = 0x%X\n", pie_handle->pie_clippad->RTrunc);
    print("LineWidth = 0x%X\n", pie_handle->pie_clippad->LineWidth);
    print("CLIPPADPR = 0x%X\n", pie_handle->pie_clippad->CLIPPADPR);
    print("REV0      = 0x%X\n", pie_handle->pie_clippad->REV0);
    print("REV1      = 0x%X\n", pie_handle->pie_clippad->REV1);
}
EXPORT_SYMBOL(pie_clippad_dump_handle_regs);

void pie_clippad_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_clippad);

    if (bypass == true)
        pie_handle->pie_clippad->Cfg =
            CLIPPAD_CFG_BYPASS_REPLACE_VAL(pie_handle->pie_clippad->Cfg, 1);
    else
        pie_handle->pie_clippad->Cfg =
            CLIPPAD_CFG_BYPASS_REPLACE_VAL(pie_handle->pie_clippad->Cfg, 0);
}
EXPORT_SYMBOL(pie_clippad_set_bypass);

void pie_clippad_set_pad_left(struct pie_handle_t *pie_handle, bool edgefill, int length)
{
    uint32_t reg;

    reg = pie_handle->pie_clippad->LPad;
    
    if (edgefill)
        reg = CLIPPAD_LPAD_LEFTEDGEFILL_REPLACE_VAL(reg, 1);
    else
        reg = CLIPPAD_LPAD_LEFTEDGEFILL_REPLACE_VAL(reg, 0);

    pie_handle->pie_clippad->LPad =
        CLIPPAD_LPAD_LEFTLEN_REPLACE_VAL(reg, length);
}
EXPORT_SYMBOL(pie_clippad_set_pad_left);

void pie_clippad_set_pad_right(struct pie_handle_t *pie_handle,bool edgefill, int length)
{
    uint32_t reg;

    reg = pie_handle->pie_clippad->RPad;
    
    if (edgefill)
        reg = CLIPPAD_RPAD_RIGHTEDGEFILL_REPLACE_VAL(reg, 1);
    else
        reg = CLIPPAD_RPAD_RIGHTEDGEFILL_REPLACE_VAL(reg, 0);

    pie_handle->pie_clippad->RPad =
        CLIPPAD_RPAD_RIGHTLEN_REPLACE_VAL(reg, length);
}
EXPORT_SYMBOL(pie_clippad_set_pad_right);

void pie_clippad_set_trunc_left(struct pie_handle_t *pie_handle, int column)
{
    uint32_t reg;

    reg = pie_handle->pie_clippad->LTrunc;
    
    pie_handle->pie_clippad->LTrunc =
        CLIPPAD_LTRUNC_FIRSTCOLUMN_REPLACE_VAL(reg, column);
}
EXPORT_SYMBOL(pie_clippad_set_trunc_left);

void pie_clippad_set_trunc_right(struct pie_handle_t *pie_handle, int column)
{
    uint32_t reg;

    reg = pie_handle->pie_clippad->RTrunc;
    
    pie_handle->pie_clippad->RTrunc =
        CLIPPAD_RTRUNC_LASTCOLUMN_REPLACE_VAL(reg, column);
}
EXPORT_SYMBOL(pie_clippad_set_trunc_right);

////////////////////////////////////////////////////////////////////////
// pie rgb2esrgb
////////////////////////////////////////////////////////////////////////

static void get_rgb2esrgb_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    int i;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    pie_handle->pie_rgb2esrgb->REV0 = PIE_RGB2ESRGB_MAJ_REV0;
    pie_handle->pie_rgb2esrgb->REV1 = 0;
    regval = 0;
    regval = LUT1D_RGB2ESRGB_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(regval, 1);
    pie_handle->pie_rgb2esrgb->LUT_CR = regval;
    pie_handle->pie_rgb2esrgb->EXTRAREG0 = 0;
    pie_handle->pie_rgb2esrgb->EXTRAREG1 = 0;
    pie_handle->pie_rgb2esrgb->EXTRAREG2 = 0;
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        pie_handle->pie_rgb2esrgb->LUT0[i] = 0;
        pie_handle->pie_rgb2esrgb->LUT1[i] = 0;
        pie_handle->pie_rgb2esrgb->LUT2[i] = 0;
    }
}

static void pie_rgb2esrgb_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    int i;
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_rgb2esrgb);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_rgb2esrgb);    

    dest_pie_handle->pie_rgb2esrgb->REV0 = src_pie_handle->pie_rgb2esrgb->REV0;
    dest_pie_handle->pie_rgb2esrgb->REV1 = src_pie_handle->pie_rgb2esrgb->REV1;
    dest_pie_handle->pie_rgb2esrgb->LUT_CR = src_pie_handle->pie_rgb2esrgb->LUT_CR;
    dest_pie_handle->pie_rgb2esrgb->EXTRAREG0 = src_pie_handle->pie_rgb2esrgb->EXTRAREG0;
    dest_pie_handle->pie_rgb2esrgb->EXTRAREG1 = src_pie_handle->pie_rgb2esrgb->EXTRAREG1;
    dest_pie_handle->pie_rgb2esrgb->EXTRAREG2 = src_pie_handle->pie_rgb2esrgb->EXTRAREG2;
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        dest_pie_handle->pie_rgb2esrgb->LUT0[i] = src_pie_handle->pie_rgb2esrgb->LUT0[i];
        dest_pie_handle->pie_rgb2esrgb->LUT1[i] = src_pie_handle->pie_rgb2esrgb->LUT1[i];
        dest_pie_handle->pie_rgb2esrgb->LUT2[i] = src_pie_handle->pie_rgb2esrgb->LUT2[i];           
    }    
}

void pie_rgb2esrgb_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_rgb2esrgb);

    print("PIE RGB2ESRGB dump regs (pie_handle)\n");
    print("REV0=0x%X\n", pie_handle->pie_rgb2esrgb->REV0);
    print("REV1=0x%X\n", pie_handle->pie_rgb2esrgb->REV1);
    print("LUT_CR=0x%X\n", pie_handle->pie_rgb2esrgb->LUT_CR);
    print("EXTRAREG0=0x%X\n", pie_handle->pie_rgb2esrgb->EXTRAREG0);
    print("EXTRAREG1=0x%X\n", pie_handle->pie_rgb2esrgb->EXTRAREG1);
    print("EXTRAREG2=0x%X\n", pie_handle->pie_rgb2esrgb->EXTRAREG2);
}
EXPORT_SYMBOL(pie_rgb2esrgb_dump_handle_regs);

void pie_rgb2esrgb_dump_handle_lut(struct pie_handle_t *pie_handle)
{
    int i;
    PIE_PTRCHECK_NR(pie_rgb2esrgb);

    print("PIE RGB2ESRGB dump LUT (pie_handle)\n");
    print("Red LUT entries  ");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);            
        print("0x%X ", pie_handle->pie_rgb2esrgb->LUT0[i]);
    }

    print("\nGreen LUT entries ");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);            
        print("0x%X ", pie_handle->pie_rgb2esrgb->LUT1[i]);
    }

    print("\nBlue LUT entries ");
    for (i=0;i<PIE_RGB2ESRGB_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);            
        print("0x%X ", pie_handle->pie_rgb2esrgb->LUT2[i]);
    }
    print("\n");
}
EXPORT_SYMBOL(pie_rgb2esrgb_dump_handle_lut);

void pie_rgb2esrgb_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_rgb2esrgb);

    if (bypass == true)
        pie_handle->pie_rgb2esrgb->LUT_CR =
            LUT1D_RGB2ESRGB_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(pie_handle->pie_rgb2esrgb->LUT_CR,
                                                             PIE_CSC_BYPASS);
    else
        pie_handle->pie_rgb2esrgb->LUT_CR =
            LUT1D_RGB2ESRGB_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(pie_handle->pie_rgb2esrgb->LUT_CR,
                                                             PIE_CSC_ENABLE);
}
EXPORT_SYMBOL(pie_rgb2esrgb_set_bypass);

// extra reg0 is for the red lut, reg1 for the green lut, reg2 for the blue lut, as expected
void pie_rgb2esrgb_set_extraregs(struct pie_handle_t *pie_handle, int associated_lut, uint32_t regval)
{
    PIE_PTRCHECK_NR(pie_rgb2esrgb);

    if (associated_lut == PIE_CSC_RED_LUT)
    {
        pie_handle->pie_rgb2esrgb->EXTRAREG0 =
            LUT1D_RGB2ESRGB_EXTRAREG0_LUT_VAL_REPLACE_VAL(pie_handle->pie_rgb2esrgb->EXTRAREG0,
                                                          regval);
    }
    if (associated_lut == PIE_CSC_GREEN_LUT)
    {
        pie_handle->pie_rgb2esrgb->EXTRAREG1 =
            LUT1D_RGB2ESRGB_EXTRAREG1_LUT_VAL_REPLACE_VAL(pie_handle->pie_rgb2esrgb->EXTRAREG1,
                                                          regval);
    }
    if (associated_lut ==  PIE_CSC_BLUE_LUT)
    {
        pie_handle->pie_rgb2esrgb->EXTRAREG2 =
            LUT1D_RGB2ESRGB_EXTRAREG2_LUT_VAL_REPLACE_VAL(pie_handle->pie_rgb2esrgb->EXTRAREG2,
                                                          regval);
    }
}
EXPORT_SYMBOL(pie_rgb2esrgb_set_extraregs);

void pie_rgb2esrgb_set_lut(struct pie_handle_t *pie_handle, uint32_t *data, uint32_t num_entries, int lut)
{
    int i;
    PIE_PTRCHECK_NR(pie_rgb2esrgb);
    
    if (lut == PIE_CSC_RED_LUT)
    {
        for (i=0;i<num_entries;i++)
        {
            pie_handle->pie_rgb2esrgb->LUT0[i] = LUT1D_RGB2ESRGB_LUT0_LUT_VAL_REPLACE_VAL(0,data[i]);
        }
    }
    if (lut == PIE_CSC_GREEN_LUT)
    {
        for (i=0;i<num_entries;i++)
        {
            pie_handle->pie_rgb2esrgb->LUT1[i] = LUT1D_RGB2ESRGB_LUT1_LUT_VAL_REPLACE_VAL(0,data[i]);
        }
    }
    if (lut == PIE_CSC_BLUE_LUT)
    {
        for (i=0;i<num_entries;i++)
        {
            pie_handle->pie_rgb2esrgb->LUT2[i] = LUT1D_RGB2ESRGB_LUT2_LUT_VAL_REPLACE_VAL(0,data[i]);
        }
    }
}
EXPORT_SYMBOL(pie_rgb2esrgb_set_lut);

////////////////////////////////////////////////////////////////////////
// pie esrgb2ycc
////////////////////////////////////////////////////////////////////////

static void get_rgb2ycc_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = ES2Y_CCR_BYPASSALL_REPLACE_VAL(regval, 1);
    pie_handle->pie_rgb2ycc->CCR = regval;
    
    pie_handle->pie_rgb2ycc->MCR0 = 0;
    pie_handle->pie_rgb2ycc->MCR1 = 0;
    pie_handle->pie_rgb2ycc->MCR2 = 0;
    pie_handle->pie_rgb2ycc->MCR3 = 0;
    pie_handle->pie_rgb2ycc->MCR4 = 0;
    pie_handle->pie_rgb2ycc->MCR5 = 0;
    pie_handle->pie_rgb2ycc->MCR6 = 0;
    pie_handle->pie_rgb2ycc->MCR7 = 0;
    pie_handle->pie_rgb2ycc->MCR8 = 0;
    
    pie_handle->pie_rgb2ycc->REV0 = PIE_RGB2YCC_MAJ_REV0;
    pie_handle->pie_rgb2ycc->REV1 = 0;
}

static void pie_esrgb2ycc_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_rgb2ycc);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_rgb2ycc);
    
    dest_pie_handle->pie_rgb2ycc->CCR = src_pie_handle->pie_rgb2ycc->CCR;
    dest_pie_handle->pie_rgb2ycc->REV0 = src_pie_handle->pie_rgb2ycc->REV0;
    dest_pie_handle->pie_rgb2ycc->REV1 = src_pie_handle->pie_rgb2ycc->REV1;
    dest_pie_handle->pie_rgb2ycc->MCR0 = src_pie_handle->pie_rgb2ycc->MCR0;
    dest_pie_handle->pie_rgb2ycc->MCR1 = src_pie_handle->pie_rgb2ycc->MCR1;
    dest_pie_handle->pie_rgb2ycc->MCR2 = src_pie_handle->pie_rgb2ycc->MCR2;
    dest_pie_handle->pie_rgb2ycc->MCR3 = src_pie_handle->pie_rgb2ycc->MCR3;
    dest_pie_handle->pie_rgb2ycc->MCR4 = src_pie_handle->pie_rgb2ycc->MCR4;
    dest_pie_handle->pie_rgb2ycc->MCR5 = src_pie_handle->pie_rgb2ycc->MCR5;
    dest_pie_handle->pie_rgb2ycc->MCR6 = src_pie_handle->pie_rgb2ycc->MCR6;
    dest_pie_handle->pie_rgb2ycc->MCR7 = src_pie_handle->pie_rgb2ycc->MCR7;
    dest_pie_handle->pie_rgb2ycc->MCR8 = src_pie_handle->pie_rgb2ycc->MCR8;
}

void pie_esrgb2ycc_dump_coeff(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_rgb2ycc);
    
    print("%s: coefficients are:\n", __func__);
    print("MCR0:0x%X, ", pie_handle->pie_rgb2ycc->MCR0);
    print("MCR1:0x%X, ", pie_handle->pie_rgb2ycc->MCR1);
    print("MCR2:0x%X\n", pie_handle->pie_rgb2ycc->MCR2);
    print("MCR3:0x%X, ", pie_handle->pie_rgb2ycc->MCR3);
    print("MCR4:0x%X, ", pie_handle->pie_rgb2ycc->MCR4);
    print("MCR5:0x%X\n", pie_handle->pie_rgb2ycc->MCR5);
    print("MCR6:0x%X, ", pie_handle->pie_rgb2ycc->MCR6);
    print("MCR7:0x%X, ", pie_handle->pie_rgb2ycc->MCR7);
    print("MCR8:0x%X\n", pie_handle->pie_rgb2ycc->MCR8);
}
EXPORT_SYMBOL(pie_esrgb2ycc_dump_coeff);

void pie_esrgb2ycc_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_rgb2ycc);
    
    print("PIE ESRGB2YCC dump regs (pie_handle)\n");
    print("CCR=0x%X\n", pie_handle->pie_rgb2ycc->CCR);
    print("REV0=0x%X\n", pie_handle->pie_rgb2ycc->REV0);
    print("REV1=0x%X\n", pie_handle->pie_rgb2ycc->REV1);
    print("MCR0=0x%X\n", pie_handle->pie_rgb2ycc->MCR0);
    print("MCR1=0x%X\n", pie_handle->pie_rgb2ycc->MCR1);
    print("MCR2=0x%X\n", pie_handle->pie_rgb2ycc->MCR2);
    print("MCR3=0x%X\n", pie_handle->pie_rgb2ycc->MCR3);
    print("MCR4=0x%X\n", pie_handle->pie_rgb2ycc->MCR4);
    print("MCR5=0x%X\n", pie_handle->pie_rgb2ycc->MCR5);
    print("MCR6=0x%X\n", pie_handle->pie_rgb2ycc->MCR6);
    print("MCR7=0x%X\n", pie_handle->pie_rgb2ycc->MCR7);    
    print("MCR8=0x%X\n", pie_handle->pie_rgb2ycc->MCR8);
}
EXPORT_SYMBOL(pie_esrgb2ycc_dump_handle_regs);

void pie_esrgb2ycc_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_rgb2ycc);

    if (bypass == true)
        pie_handle->pie_rgb2ycc->CCR = ES2Y_CCR_BYPASSALL_REPLACE_VAL(pie_handle->pie_rgb2ycc->CCR,
                                                                      PIE_CSC_BYPASS);
    else
        pie_handle->pie_rgb2ycc->CCR = ES2Y_CCR_BYPASSALL_REPLACE_VAL(pie_handle->pie_rgb2ycc->CCR,
                                                                      PIE_CSC_ENABLE);        
}
EXPORT_SYMBOL(pie_esrgb2ycc_set_bypass);

// set up the coefficients for esRGB2YCrCb 
void pie_esrgb2ycc_set_coeff(struct pie_handle_t *pie_handle, uint32_t *coefficients)
{
    PIE_PTRCHECK_NR(pie_rgb2ycc);
    
    pie_handle->pie_rgb2ycc->MCR0 = coefficients[0];
    pie_handle->pie_rgb2ycc->MCR1 = coefficients[1];
    pie_handle->pie_rgb2ycc->MCR2 = coefficients[2];
    pie_handle->pie_rgb2ycc->MCR3 = coefficients[3];
    pie_handle->pie_rgb2ycc->MCR4 = coefficients[4];
    pie_handle->pie_rgb2ycc->MCR5 = coefficients[5];
    pie_handle->pie_rgb2ycc->MCR6 = coefficients[6];
    pie_handle->pie_rgb2ycc->MCR7 = coefficients[7];
    pie_handle->pie_rgb2ycc->MCR8 = coefficients[8];
}
EXPORT_SYMBOL(pie_esrgb2ycc_set_coeff);

////////////////////////////////////////////////////////////////////////
// pie ycc2esrgb
////////////////////////////////////////////////////////////////////////

static void get_ycc2rgb_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = Y2ES_CCR_BYPASSALL_REPLACE_VAL(regval, 1);
    pie_handle->pie_ycc2rgb->CCR = regval;
    
    pie_handle->pie_ycc2rgb->MCR0 = 0;
    pie_handle->pie_ycc2rgb->MCR1 = 0;
    pie_handle->pie_ycc2rgb->MCR2 = 0;
    pie_handle->pie_ycc2rgb->MCR3 = 0;
    pie_handle->pie_ycc2rgb->MCR4 = 0;
    pie_handle->pie_ycc2rgb->MCR5 = 0;
    pie_handle->pie_ycc2rgb->MCR6 = 0;
    pie_handle->pie_ycc2rgb->MCR7 = 0;
    pie_handle->pie_ycc2rgb->MCR8 = 0;
    
    pie_handle->pie_ycc2rgb->REV0 = PIE_YCC2RGB_MAJ_REV0;
    pie_handle->pie_ycc2rgb->REV1 = 0;
}

static void pie_ycc2esrgb_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_ycc2rgb);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_ycc2rgb);
    
    dest_pie_handle->pie_ycc2rgb->CCR = src_pie_handle->pie_ycc2rgb->CCR;
    dest_pie_handle->pie_ycc2rgb->REV0 = src_pie_handle->pie_ycc2rgb->REV0;
    dest_pie_handle->pie_ycc2rgb->REV1 = src_pie_handle->pie_ycc2rgb->REV1;
    dest_pie_handle->pie_ycc2rgb->MCR0 = src_pie_handle->pie_ycc2rgb->MCR0;
    dest_pie_handle->pie_ycc2rgb->MCR1 = src_pie_handle->pie_ycc2rgb->MCR1;
    dest_pie_handle->pie_ycc2rgb->MCR2 = src_pie_handle->pie_ycc2rgb->MCR2;
    dest_pie_handle->pie_ycc2rgb->MCR3 = src_pie_handle->pie_ycc2rgb->MCR3;
    dest_pie_handle->pie_ycc2rgb->MCR4 = src_pie_handle->pie_ycc2rgb->MCR4;
    dest_pie_handle->pie_ycc2rgb->MCR5 = src_pie_handle->pie_ycc2rgb->MCR5;
    dest_pie_handle->pie_ycc2rgb->MCR6 = src_pie_handle->pie_ycc2rgb->MCR6;
    dest_pie_handle->pie_ycc2rgb->MCR7 = src_pie_handle->pie_ycc2rgb->MCR7;
    dest_pie_handle->pie_ycc2rgb->MCR8 = src_pie_handle->pie_ycc2rgb->MCR8;
}

void pie_ycc2esrgb_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_ycc2rgb);
    
    print("PIE ESYCC2RGB dump regs (pie_handle)\n");
    print("CCR=0x%X\n", pie_handle->pie_ycc2rgb->CCR);
    print("REV0=0x%X\n", pie_handle->pie_ycc2rgb->REV0);
    print("REV1=0x%X\n", pie_handle->pie_ycc2rgb->REV1);
    print("MCR0=0x%X", pie_handle->pie_ycc2rgb->MCR0);
    print("MCR1=0x%X", pie_handle->pie_ycc2rgb->MCR1);
    print("MCR2=0x%X\n", pie_handle->pie_ycc2rgb->MCR2);
    print("MCR3=0x%X", pie_handle->pie_ycc2rgb->MCR3);
    print("MCR4=0x%X", pie_handle->pie_ycc2rgb->MCR4);
    print("MCR5=0x%X\n", pie_handle->pie_ycc2rgb->MCR5);
    print("MCR6=0x%X", pie_handle->pie_ycc2rgb->MCR6);
    print("MCR7=0x%X", pie_handle->pie_ycc2rgb->MCR7);    
    print("MCR8=0x%X\n", pie_handle->pie_ycc2rgb->MCR8);
}
EXPORT_SYMBOL(pie_ycc2esrgb_dump_handle_regs);

void pie_ycc2esrgb_dump_coeff(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_ycc2rgb);
    
    print("%s: coefficients are:\n", __func__);
    print("MCR0:0x%X, ", pie_handle->pie_ycc2rgb->MCR0);
    print("MCR1:0x%X, ", pie_handle->pie_ycc2rgb->MCR1);
    print("MCR2:0x%X\n", pie_handle->pie_ycc2rgb->MCR2);
    print("MCR3:0x%X, ", pie_handle->pie_ycc2rgb->MCR3);
    print("MCR4:0x%X, ", pie_handle->pie_ycc2rgb->MCR4);
    print("MCR5:0x%X\n", pie_handle->pie_ycc2rgb->MCR5);
    print("MCR6:0x%X, ", pie_handle->pie_ycc2rgb->MCR6);
    print("MCR7:0x%X, ", pie_handle->pie_ycc2rgb->MCR7);
    print("MCR8:0x%X\n", pie_handle->pie_ycc2rgb->MCR8);
}
EXPORT_SYMBOL(pie_ycc2esrgb_dump_coeff);

void pie_ycc2esrgb_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_ycc2rgb);

    if (bypass == true)
        pie_handle->pie_ycc2rgb->CCR = Y2ES_CCR_BYPASSALL_REPLACE_VAL(pie_handle->pie_ycc2rgb->CCR,
                                                                      PIE_CSC_BYPASS);
    else
        pie_handle->pie_ycc2rgb->CCR = Y2ES_CCR_BYPASSALL_REPLACE_VAL(pie_handle->pie_ycc2rgb->CCR,
                                                                      PIE_CSC_ENABLE);        
}
EXPORT_SYMBOL(pie_ycc2esrgb_set_bypass);

// set up the coefficients for YCrCb2esRGB
void pie_ycc2esrgb_set_coeff(struct pie_handle_t *pie_handle, uint32_t *coefficients)
{
    PIE_PTRCHECK_NR(pie_ycc2rgb);
    
    pie_handle->pie_ycc2rgb->MCR0 = coefficients[0];
    pie_handle->pie_ycc2rgb->MCR1 = coefficients[1];
    pie_handle->pie_ycc2rgb->MCR2 = coefficients[2];
    pie_handle->pie_ycc2rgb->MCR3 = coefficients[3];
    pie_handle->pie_ycc2rgb->MCR4 = coefficients[4];
    pie_handle->pie_ycc2rgb->MCR5 = coefficients[5];
    pie_handle->pie_ycc2rgb->MCR6 = coefficients[6];
    pie_handle->pie_ycc2rgb->MCR7 = coefficients[7];
    pie_handle->pie_ycc2rgb->MCR8 = coefficients[8];
}
EXPORT_SYMBOL(pie_ycc2esrgb_set_coeff);

////////////////////////////////////////////////////////////////////////
// pie tcns
////////////////////////////////////////////////////////////////////////

static void get_tcns_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    int i;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = TCNSENSE_TCNSCR_B_REPLACE_VAL(regval, PIE_CSC_BYPASS);  // bypass to do a reset
    pie_handle->pie_tcns->TCNSCR = regval;
    pie_handle->pie_tcns->REV0 = PIE_TCNS_MAJ_REV0;
    pie_handle->pie_tcns->REV1 = 0;

    for (i=0;i<PIE_TCNS_LUT_ENTRIES;i++)
    {
        pie_handle->pie_tcns->TCNSLUT[i] = 0;
        pie_handle->pie_tcns->TCNSLUT2[i] = 0;
    }
}

static void pie_tcns_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    int i;

    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_tcns);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_tcns);    

    dest_pie_handle->pie_tcns->TCNSCR = src_pie_handle->pie_tcns->TCNSCR;
    dest_pie_handle->pie_tcns->REV0   = src_pie_handle->pie_tcns->REV0;
    dest_pie_handle->pie_tcns->REV1   = src_pie_handle->pie_tcns->REV1;

    for (i=0; i<PIE_TCNS_LUT_ENTRIES; i++)
    {
        dest_pie_handle->pie_tcns->TCNSLUT[i]  = src_pie_handle->pie_tcns->TCNSLUT[i];
        dest_pie_handle->pie_tcns->TCNSLUT2[i] = src_pie_handle->pie_tcns->TCNSLUT2[i];
    }
}

void pie_tcns_dump_handle_lut(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_tcns);

    print("STUBBED - implement %s\n", __func__);
    // print the lut in the pie struct
}
EXPORT_SYMBOL(pie_tcns_dump_handle_lut);

void pie_tcns_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_tcns);

    print("PIE TCNS dump regs (pie_handle)\n");
    print("TCNSCR =0x%X\n",  pie_handle->pie_tcns->TCNSCR);
    print("REV0   =0x%X\n",  pie_handle->pie_tcns->REV0);
    print("REV1   =0x%X\n",  pie_handle->pie_tcns->REV1);
}
EXPORT_SYMBOL(pie_tcns_dump_handle_regs);

void pie_tcns_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_tcns);

    if (bypass == true)
        pie_handle->pie_tcns->TCNSCR =
            TCNSENSE_TCNSCR_B_REPLACE_VAL(pie_handle->pie_tcns->TCNSCR, PIE_CSC_BYPASS);
    else
        pie_handle->pie_tcns->TCNSCR =
            TCNSENSE_TCNSCR_B_REPLACE_VAL(pie_handle->pie_tcns->TCNSCR, PIE_CSC_ENABLE);
}
EXPORT_SYMBOL(pie_tcns_set_bypass);

void pie_tcns_set_lut(struct pie_handle_t *pie_handle, struct ycc_values *lut, int lut_num)
{
    int i;
    
    PIE_PTRCHECK_NR(pie_tcns);

    for (i=0;i<PIE_TCNS_LUT_ENTRIES;i++)
    {
        if (lut_num == 1)
        {
            pie_handle->pie_tcns->TCNSLUT[i] =
                TCNSENSE_TCNSLUT_Y_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT[i],
                                               lut[i].y_lut_value);
            pie_handle->pie_tcns->TCNSLUT[i] =
                TCNSENSE_TCNSLUT_CR_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT[i],
                                               lut[i].cr_lut_value);
            pie_handle->pie_tcns->TCNSLUT[i] =
                TCNSENSE_TCNSLUT_CB_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT[i],
                                               lut[i].cb_lut_value);
        }
        else if (lut_num == 2)
        {
            pie_handle->pie_tcns->TCNSLUT2[i] =
                TCNSENSE_TCNSLUT_Y_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT2[i],
                                               lut[i].y_lut_value);
            pie_handle->pie_tcns->TCNSLUT2[i] =
                TCNSENSE_TCNSLUT_CR_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT2[i],
                                               lut[i].cr_lut_value);
            pie_handle->pie_tcns->TCNSLUT2[i] =
                TCNSENSE_TCNSLUT_CB_REPLACE_VAL(pie_handle->pie_tcns->TCNSLUT2[i],
                                               lut[i].cb_lut_value);
        }
        else
            error_print("%s bad lut number %d, only 1 and 2 supported\n", __func__, lut_num);
    }
}
EXPORT_SYMBOL(pie_tcns_set_lut);

////////////////////////////////////////////////////////////////////////
// pie sccsc
////////////////////////////////////////////////////////////////////////

static void get_sccsc_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014

    // Note: STATUS not used in pie sccsc instance
    regval = 0;
    pie_handle->pie_sccsc->STATUS  = regval;

    regval = 0;
    regval = SCCSC_CNTRL_BYPASSALL_REPLACE_VAL(regval, 1);
    regval = SCCSC_CNTRL_WME_REPLACE_VAL(regval, 1);
    pie_handle->pie_sccsc->CNTRL   = regval;
    
    // Note: MCClr, LCClr, CACHE_INVLD, CACHE_CNTRL, IDX_MISS_COUNTER, PIXEL_MISS_COUNTER,
    // LATENCY_COUNTER, BASE_ADDR not used in pie sccsc instance
    regval = 0;
    pie_handle->pie_sccsc->MCClr              = regval;
    pie_handle->pie_sccsc->LCClr              = regval;
    pie_handle->pie_sccsc->CACHE_INVLD        = regval;
    pie_handle->pie_sccsc->CACHE_CNTRL        = regval;
    pie_handle->pie_sccsc->IDX_MISS_COUNTER   = regval;
    pie_handle->pie_sccsc->PIXEL_MISS_COUNTER = regval;
    pie_handle->pie_sccsc->LATENCY_COUNTER    = regval;
    pie_handle->pie_sccsc->BASE_ADDR          = regval;

    regval = 0;
    regval = SCCSC_WT_CH0_WT_REPLACE_VAL(regval, 0x3ff);
    regval = SCCSC_WT_CH1_WT_REPLACE_VAL(regval, 0x3ff);
    regval = SCCSC_WT_CH2_WT_REPLACE_VAL(regval, 0x3ff);
    pie_handle->pie_sccsc->WT = regval;

    regval = 0;
    pie_handle->pie_sccsc->WR_REG0 = regval;
    pie_handle->pie_sccsc->WR_REG1 = regval;
    pie_handle->pie_sccsc->WR_REG2 = regval;
    pie_handle->pie_sccsc->KR_REG0 = regval;
    pie_handle->pie_sccsc->KR_REG1 = regval;
    pie_handle->pie_sccsc->KR_REG2 = regval;

    regval = 0;
    regval = SCCSC_RAR_RNW_REPLACE_VAL(regval, 1);
    pie_handle->pie_sccsc->RAR = regval;

    regval = 0;
    pie_handle->pie_sccsc->RDW0 = regval;
    pie_handle->pie_sccsc->RDW1 = regval;
    pie_handle->pie_sccsc->RDW2 = regval;
    pie_handle->pie_sccsc->RDR0 = regval;
    pie_handle->pie_sccsc->RDR1 = regval;
    pie_handle->pie_sccsc->RDR2 = regval;

    // block instantiation parms - user should compare against ASIC vals
    regval = 0;
    regval = SCCSC_PARAMS_CACHELINES_REPLACE_VAL(regval, 0x40);
    regval = SCCSC_PARAMS_NUMOUTS_REPLACE_VAL(regval, 0x6);
    regval = SCCSC_PARAMS_OUTWIDTH_REPLACE_VAL(regval, 0xA);
    regval = SCCSC_PARAMS_INWIDTH_REPLACE_VAL(regval, 0x8);
    pie_handle->pie_sccsc->PARAMS = regval;
    // rev - user should request rev check against asic values
    pie_handle->pie_sccsc->REV = PIE_SCCSC_MAJ_REV;

}

static void pie_sccsc_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_sccsc);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_sccsc);    

    dest_pie_handle->pie_sccsc->CNTRL   = src_pie_handle->pie_sccsc->CNTRL;
    dest_pie_handle->pie_sccsc->WT      = src_pie_handle->pie_sccsc->WT;
    dest_pie_handle->pie_sccsc->WR_REG0 = src_pie_handle->pie_sccsc->WR_REG0;
    dest_pie_handle->pie_sccsc->WR_REG1 = src_pie_handle->pie_sccsc->WR_REG1;
    dest_pie_handle->pie_sccsc->WR_REG2 = src_pie_handle->pie_sccsc->WR_REG2;
    dest_pie_handle->pie_sccsc->KR_REG0 = src_pie_handle->pie_sccsc->KR_REG0;
    dest_pie_handle->pie_sccsc->KR_REG1 = src_pie_handle->pie_sccsc->KR_REG1;
    dest_pie_handle->pie_sccsc->KR_REG2 = src_pie_handle->pie_sccsc->KR_REG2;
    dest_pie_handle->pie_sccsc->RAR     = src_pie_handle->pie_sccsc->RAR;
    dest_pie_handle->pie_sccsc->RDW0    = src_pie_handle->pie_sccsc->RDW0;
    dest_pie_handle->pie_sccsc->RDW1    = src_pie_handle->pie_sccsc->RDW1;
    dest_pie_handle->pie_sccsc->RDW2    = src_pie_handle->pie_sccsc->RDW2;
    dest_pie_handle->pie_sccsc->RDR0    = src_pie_handle->pie_sccsc->RDR0;
    dest_pie_handle->pie_sccsc->RDR1    = src_pie_handle->pie_sccsc->RDR1;
    dest_pie_handle->pie_sccsc->RDR2    = src_pie_handle->pie_sccsc->RDR2;
    dest_pie_handle->pie_sccsc->REV     = src_pie_handle->pie_sccsc->REV;
    dest_pie_handle->pie_sccsc->PARAMS  = src_pie_handle->pie_sccsc->PARAMS;

    // Note: the LUTs are not in register space (so not in the pie handle).
    //       The caller will have to deal with the LUT clone if needed ...
}

void pie_sccsc_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_sccsc);
    
    print("PIE SCCSC dump regs (pie_handle)\n");
    print("CNTRL   =0x%X\n",  pie_handle->pie_sccsc->CNTRL);
    print("WT      =0x%X\n",  pie_handle->pie_sccsc->WT);
    print("WR_REG0 =0x%X\n",  pie_handle->pie_sccsc->WR_REG0);
    print("WR_REG1 =0x%X\n",  pie_handle->pie_sccsc->WR_REG1);
    print("WR_REG2 =0x%X\n",  pie_handle->pie_sccsc->WR_REG2);
    print("KR_REG0 =0x%X\n",  pie_handle->pie_sccsc->KR_REG0);
    print("KR_REG1 =0x%X\n",  pie_handle->pie_sccsc->KR_REG1);
    print("KR_REG2 =0x%X\n",  pie_handle->pie_sccsc->KR_REG2);
    print("RAR     =0x%X\n",  pie_handle->pie_sccsc->RAR);
    print("RDW0    =0x%X\n",  pie_handle->pie_sccsc->RDW0);
    print("RDW1    =0x%X\n",  pie_handle->pie_sccsc->RDW1);
    print("RDW2    =0x%X\n",  pie_handle->pie_sccsc->RDW2);
    print("RDR0    =0x%X\n",  pie_handle->pie_sccsc->RDR0);
    print("RDR1    =0x%X\n",  pie_handle->pie_sccsc->RDR1);
    print("RDR2    =0x%X\n",  pie_handle->pie_sccsc->RDR2);
    print("REV     =0x%X\n",  pie_handle->pie_sccsc->REV);
    print("PARAMS  =0x%X\n",  pie_handle->pie_sccsc->PARAMS);

    // Note: the LUTs are not in register space (so not in the pie handle)
}
EXPORT_SYMBOL(pie_sccsc_dump_handle_regs);

void pie_sccsc_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    if (bypass == true) {
        pie_handle->pie_sccsc->CNTRL =
            SCCSC_CNTRL_BYPASSALL_REPLACE_VAL(pie_handle->pie_sccsc->CNTRL, PIE_CSC_BYPASS);
    }
    else {
        pie_handle->pie_sccsc->CNTRL =
            SCCSC_CNTRL_BYPASSALL_REPLACE_VAL(pie_handle->pie_sccsc->CNTRL, PIE_CSC_ENABLE);
    }
}
EXPORT_SYMBOL(pie_sccsc_set_bypass);

void pie_sccsc_whitemap_enable(struct pie_handle_t *pie_handle, bool enable)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    if (enable == true) {
        pie_handle->pie_sccsc->CNTRL =
            SCCSC_CNTRL_WME_REPLACE_VAL(pie_handle->pie_sccsc->CNTRL, 1);
    }
    else {
        pie_handle->pie_sccsc->CNTRL =
            SCCSC_CNTRL_WME_REPLACE_VAL(pie_handle->pie_sccsc->CNTRL, 0);
    }
}
EXPORT_SYMBOL(pie_sccsc_whitemap_enable);

void pie_sccsc_whitemap_thresh(struct pie_handle_t *pie_handle, uint16_t ch0, uint16_t ch1, uint16_t ch2)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    pie_handle->pie_sccsc->WT = SCCSC_WT_CH0_WT_REPLACE_VAL(pie_handle->pie_sccsc->WT, ch0);
    pie_handle->pie_sccsc->WT = SCCSC_WT_CH1_WT_REPLACE_VAL(pie_handle->pie_sccsc->WT, ch1);
    pie_handle->pie_sccsc->WT = SCCSC_WT_CH2_WT_REPLACE_VAL(pie_handle->pie_sccsc->WT, ch2);
}
EXPORT_SYMBOL(pie_sccsc_whitemap_thresh);

void pie_sccsc_whitemap_response(struct pie_handle_t *pie_handle, uint16_t ch0, uint16_t ch1, uint16_t ch2)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    pie_handle->pie_sccsc->WR_REG0 = SCCSC_WR_REG0_CH0_WR_REPLACE_VAL(pie_handle->pie_sccsc->WR_REG0, ch0);
    pie_handle->pie_sccsc->WR_REG0 = SCCSC_WR_REG0_CH1_WR_REPLACE_VAL(pie_handle->pie_sccsc->WR_REG0, ch1);
    pie_handle->pie_sccsc->WR_REG0 = SCCSC_WR_REG0_CH2_WR_REPLACE_VAL(pie_handle->pie_sccsc->WR_REG0, ch2);
}
EXPORT_SYMBOL(pie_sccsc_whitemap_response);

void pie_sccsc_blackmap_mode(struct pie_handle_t *pie_handle, uint8_t mode)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    pie_handle->pie_sccsc->CNTRL = SCCSC_CNTRL_KME_REPLACE_VAL(pie_handle->pie_sccsc->CNTRL, mode);
}
EXPORT_SYMBOL(pie_sccsc_blackmap_mode);

void pie_sccsc_blackmap_response(struct pie_handle_t *pie_handle, uint16_t ch0, uint16_t ch1, uint16_t ch2)
{
    PIE_PTRCHECK_NR(pie_sccsc);

    pie_handle->pie_sccsc->KR_REG0 = SCCSC_KR_REG0_CH0_KR_REPLACE_VAL(pie_handle->pie_sccsc->KR_REG0, ch0);
    pie_handle->pie_sccsc->KR_REG0 = SCCSC_KR_REG0_CH1_KR_REPLACE_VAL(pie_handle->pie_sccsc->KR_REG0, ch1);
    pie_handle->pie_sccsc->KR_REG0 = SCCSC_KR_REG0_CH2_KR_REPLACE_VAL(pie_handle->pie_sccsc->KR_REG0, ch2);
}
EXPORT_SYMBOL(pie_sccsc_blackmap_response);


////////////////////////////////////////////////////////////////////////
// pie cstats
////////////////////////////////////////////////////////////////////////

static void get_cstats_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    int      i;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = CSTATS_CFG_MCC_REPLACE_VAL(regval, 1);
    pie_handle->pie_cstats->CFG = regval;
    
    regval = 0;
    pie_handle->pie_cstats->LM    = regval;
    pie_handle->pie_cstats->RM    = regval;
    pie_handle->pie_cstats->SW    = regval;
    pie_handle->pie_cstats->STW   = regval;
    pie_handle->pie_cstats->TR1   = regval;
    pie_handle->pie_cstats->TR2   = regval;
    pie_handle->pie_cstats->SBLK  = regval;
    pie_handle->pie_cstats->SGRY  = regval;
    pie_handle->pie_cstats->SC    = regval;
    pie_handle->pie_cstats->CFGBW = regval;
    pie_handle->pie_cstats->CFGBR = regval;

    pie_handle->pie_cstats->REV0 = PIE_CSTATS_MAJ_REV0;
    pie_handle->pie_cstats->REV1 = 0;

    for (i=0; i<PIE_CSTATS_SRAM_SIZE; i++)
    {
        pie_handle->pie_cstats->RAM0[i] = 0;
        pie_handle->pie_cstats->RAM1[i] = 0;
        pie_handle->pie_cstats->RAM2[i] = 0;
    }

    for (i=0; i<PIE_CSTATS_NONWHITE_SRAM_SIZE; i++)
    {
        pie_handle->pie_cstats->RAM3[i] = 0;
    }
}

static void pie_cstats_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    int i;

    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_cstats);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_cstats);    

    dest_pie_handle->pie_cstats->CFG   = src_pie_handle->pie_cstats->CFG;
    dest_pie_handle->pie_cstats->LM    = src_pie_handle->pie_cstats->LM;
    dest_pie_handle->pie_cstats->RM    = src_pie_handle->pie_cstats->RM;
    dest_pie_handle->pie_cstats->SW    = src_pie_handle->pie_cstats->SW;
    dest_pie_handle->pie_cstats->STW   = src_pie_handle->pie_cstats->STW;
    dest_pie_handle->pie_cstats->TR1   = src_pie_handle->pie_cstats->TR1;
    dest_pie_handle->pie_cstats->TR2   = src_pie_handle->pie_cstats->TR2;
    dest_pie_handle->pie_cstats->SBLK  = src_pie_handle->pie_cstats->SBLK;
    dest_pie_handle->pie_cstats->SGRY  = src_pie_handle->pie_cstats->SGRY;
    dest_pie_handle->pie_cstats->SC    = src_pie_handle->pie_cstats->SC;
    dest_pie_handle->pie_cstats->CFGBW = src_pie_handle->pie_cstats->CFGBW;
    dest_pie_handle->pie_cstats->CFGBR = src_pie_handle->pie_cstats->CFGBR;
    dest_pie_handle->pie_cstats->REV0  = src_pie_handle->pie_cstats->REV0;
    dest_pie_handle->pie_cstats->REV1  = src_pie_handle->pie_cstats->REV1;

    for (i=0; i<PIE_CSTATS_SRAM_SIZE; i++)
    {
        dest_pie_handle->pie_cstats->RAM0[i] = src_pie_handle->pie_cstats->RAM0[i];
        dest_pie_handle->pie_cstats->RAM1[i] = src_pie_handle->pie_cstats->RAM1[i];
        dest_pie_handle->pie_cstats->RAM2[i] = src_pie_handle->pie_cstats->RAM2[i];
    }

    for (i=0; i<PIE_CSTATS_NONWHITE_SRAM_SIZE; i++)
    {
        dest_pie_handle->pie_cstats->RAM3[i] = src_pie_handle->pie_cstats->RAM3[i];
    }
}

void pie_cstats_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_cstats);
    
    print("PIE cstats dump regs (pie_handle)\n");
    print("CFG   =0x%X\n",  pie_handle->pie_cstats->CFG);
    print("LM    =0x%X\n",  pie_handle->pie_cstats->LM);
    print("RM    =0x%X\n",  pie_handle->pie_cstats->RM);
    print("SW    =0x%X\n",  pie_handle->pie_cstats->SW);
    print("STW   =0x%X\n",  pie_handle->pie_cstats->STW);
    print("TR1   =0x%X\n",  pie_handle->pie_cstats->TR1);
    print("TR2   =0x%X\n",  pie_handle->pie_cstats->TR2);
    print("SBLK  =0x%X\n",  pie_handle->pie_cstats->SBLK);
    print("SGRY  =0x%X\n",  pie_handle->pie_cstats->SGRY);
    print("SC    =0x%X\n",  pie_handle->pie_cstats->SC);
    print("CFGBW =0x%X\n",  pie_handle->pie_cstats->CFGBW);
    print("CFGBR =0x%X\n",  pie_handle->pie_cstats->CFGBR);
    print("REV0  =0x%X\n",  pie_handle->pie_cstats->REV0);
    print("REV1  =0x%X\n",  pie_handle->pie_cstats->REV1);
}
EXPORT_SYMBOL(pie_cstats_dump_handle_regs);

void pie_cstats_dump_handle_luts(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_cstats);

    print("%s: stubbed\n", __func__);
}
EXPORT_SYMBOL(pie_cstats_dump_handle_luts);

void pie_cstats_set_enable(struct pie_handle_t *pie_handle, bool enable)
{
    PIE_PTRCHECK_NR(pie_cstats);

    if (enable == true) {
        pie_handle->pie_cstats->CFG = CSTATS_CFG_E_REPLACE_VAL(pie_handle->pie_cstats->CFG, 1);
    }
    else {
        pie_handle->pie_cstats->CFG = CSTATS_CFG_E_REPLACE_VAL(pie_handle->pie_cstats->CFG, 0);
    }
}
EXPORT_SYMBOL(pie_cstats_set_enable);

void pie_cstats_set_cmode(struct pie_handle_t *pie_handle, uint8_t cmode)
{
    PIE_PTRCHECK_NR(pie_cstats);

    if( cmode == PIE_CSTATS_CFG_MONO ) {
        pie_handle->pie_cstats->CFG = CSTATS_CFG_MONO_REPLACE_VAL(pie_handle->pie_cstats->CFG, 1);
    }
    else {
        pie_handle->pie_cstats->CFG = CSTATS_CFG_MONO_REPLACE_VAL(pie_handle->pie_cstats->CFG, 0);
    }
}
EXPORT_SYMBOL(pie_cstats_set_cmode);

void pie_cstats_set_margins(struct pie_handle_t *pie_handle, uint32_t left, uint32_t right)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->LM = CSTATS_LM_LM_REPLACE_VAL( pie_handle->pie_cstats->LM, left );
    pie_handle->pie_cstats->RM = CSTATS_RM_RM_REPLACE_VAL( pie_handle->pie_cstats->RM, right );
}
EXPORT_SYMBOL(pie_cstats_set_margins);

void pie_cstats_set_strip_width(struct pie_handle_t *pie_handle, uint32_t width_in_pixels)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->STW = CSTATS_STW_SW_REPLACE_VAL( pie_handle->pie_cstats->STW, width_in_pixels );
}
EXPORT_SYMBOL(pie_cstats_set_strip_width);

void pie_cstats_set_rgb_threshold(struct pie_handle_t *pie_handle, uint32_t value)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->TR2 = CSTATS_TR2_RGBT_REPLACE_VAL( pie_handle->pie_cstats->TR2, value );
}
EXPORT_SYMBOL(pie_cstats_set_rgb_threshold);

void pie_cstats_set_white_threshold(struct pie_handle_t *pie_handle, uint32_t value)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->TR2 = CSTATS_TR2_WT_REPLACE_VAL( pie_handle->pie_cstats->TR2, value );
}
EXPORT_SYMBOL(pie_cstats_set_white_threshold);

void pie_cstats_set_black_threshold(struct pie_handle_t *pie_handle, uint32_t value)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->TR2 = CSTATS_TR2_BT_REPLACE_VAL( pie_handle->pie_cstats->TR2, value );
}
EXPORT_SYMBOL(pie_cstats_set_black_threshold);

void pie_cstats_histogram(struct pie_handle_t *pie_handle, uint8_t which)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->CFG = CSTATS_CFG_EH_REPLACE_VAL( pie_handle->pie_cstats->CFG, which );
}
EXPORT_SYMBOL(pie_cstats_histogram);

void pie_cstats_set_input_mux(struct pie_handle_t *pie_handle, uint8_t which)
{
    PIE_PTRCHECK_NR(pie_cstats);

    pie_handle->pie_cstats->CFG = CSTATS_CFG_IM_REPLACE_VAL( pie_handle->pie_cstats->CFG, which );
}
EXPORT_SYMBOL(pie_cstats_set_input_mux);

static void cstats_set_sram( volatile uint32_t *dst, uint32_t *src, int sram_size )
{
    int i;

    for( i=0 ; i<sram_size ; i++ ) {
        *dst++ = *src++;
    }
}

static void cstats_get_sram( uint32_t *dst, volatile uint32_t *src, int sram_size )
{
    int i;

    for( i=0 ; i<sram_size ; i++ ) {
        *dst++ = *src++;
    }
}

void pie_cstats_get_sram( struct pie_handle_t *pie_handle, int sram_idx, uint32_t sram[], int sram_size )
{
    switch( sram_idx ) {
        case 0 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_get_sram( sram, pie_handle->pie_cstats->RAM0, PIE_CSTATS_SRAM_SIZE );
            break;

        case 1 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_get_sram( sram, pie_handle->pie_cstats->RAM1, PIE_CSTATS_SRAM_SIZE );
            break;

        case 2 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_get_sram( sram, pie_handle->pie_cstats->RAM2, PIE_CSTATS_SRAM_SIZE );
            break;

        case 3 : 
            BUG_ON( sram_size != PIE_CSTATS_NONWHITE_SRAM_SIZE );
            cstats_get_sram( sram, pie_handle->pie_cstats->RAM3, PIE_CSTATS_NONWHITE_SRAM_SIZE );
            break;

        default : 
            error_print("%s invalid sram number = %d\n", __FUNCTION__, sram_idx);
            break;
    }
}
EXPORT_SYMBOL(pie_cstats_get_sram);

void pie_cstats_set_sram( struct pie_handle_t *pie_handle, int sram_idx, uint32_t sram[], int sram_size )
{
    switch( sram_idx ) {
        case 0 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_set_sram( pie_handle->pie_cstats->RAM0, sram, PIE_CSTATS_SRAM_SIZE );
            break;

        case 1 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_set_sram( pie_handle->pie_cstats->RAM1, sram, PIE_CSTATS_SRAM_SIZE );
            break;

        case 2 : 
            BUG_ON( sram_size != PIE_CSTATS_SRAM_SIZE );
            cstats_set_sram( pie_handle->pie_cstats->RAM2, sram, PIE_CSTATS_SRAM_SIZE );
            break;

        case 3 : 
            BUG_ON( sram_size != PIE_CSTATS_NONWHITE_SRAM_SIZE );
            cstats_set_sram( pie_handle->pie_cstats->RAM3, sram, PIE_CSTATS_NONWHITE_SRAM_SIZE );
            break;

        default : 
            error_print("%s invalid sram number = %d\n", __FUNCTION__, sram_idx);
            break;
    }
}
EXPORT_SYMBOL(pie_cstats_set_sram);


////////////////////////////////////////////////////////////////////////
// pie ngadjust
////////////////////////////////////////////////////////////////////////

static void get_ngadjust_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    int i;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014

    regval = 0;
    regval = NGADJUST_CONFIG_BYPASS_REPLACE_VAL(regval, 1);
    regval = NGADJUST_CONFIG_OTENABLE_REPLACE_VAL(regval, 0xffff);
    pie_handle->pie_ngadjust->Config = regval;

    for (i=0; i<PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        pie_handle->pie_ngadjust->LUTAccess[i] = 0;
    }

    // Read the rev from hardware: it lets us know how the block was
    // actually constructed.
    pie_handle->pie_ngadjust->REV0 = PIE_NGADJUST_MAJ_REV0;
    pie_handle->pie_ngadjust->REV1 = 0;
}

static void pie_ngadjust_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    int i;

    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_ngadjust);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_ngadjust);    

    dest_pie_handle->pie_ngadjust->REV0   = src_pie_handle->pie_ngadjust->REV0;
    dest_pie_handle->pie_ngadjust->REV1   = src_pie_handle->pie_ngadjust->REV1;
    dest_pie_handle->pie_ngadjust->Config = src_pie_handle->pie_ngadjust->Config;

    for (i = 0; i < PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        dest_pie_handle->pie_ngadjust->LUTAccess[i] = src_pie_handle->pie_ngadjust->LUTAccess[i];
    }    
}

void pie_ngadjust_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK_NR(pie_ngadjust);
    
    print("PIE NGAdjust dump regs (pie_handle)\n");
    print("Config =0x%X\n",  pie_handle->pie_ngadjust->Config);
    print("REV0   =0x%X\n",  pie_handle->pie_ngadjust->REV0);
    print("REV1   =0x%X\n",  pie_handle->pie_ngadjust->REV1);
}
EXPORT_SYMBOL(pie_ngadjust_dump_handle_regs);

void pie_ngadjust_dump_handle_lut(struct pie_handle_t *pie_handle)
{
    int i;

    PIE_PTRCHECK_NR(pie_ngadjust);

    print("PIE NGAdjust LUT entries (pie_handle):\n");
        
    for (i = 0; i < PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        if (i%8 == 0)
        {
            print("\n%04d-%04d: ", i, i+7);
        }
        print("(0x%04X,", pie_handle->pie_ngadjust->LUTAccess[i]);
    }

    print("\n");
}
EXPORT_SYMBOL(pie_ngadjust_dump_handle_lut);

void pie_ngadjust_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_ngadjust);

    if (bypass == true)
    {
        pie_handle->pie_ngadjust->Config =
            NGADJUST_CONFIG_BYPASS_REPLACE_VAL(pie_handle->pie_ngadjust->Config, 1);
    }
    else
    {
        pie_handle->pie_ngadjust->Config =
            NGADJUST_CONFIG_BYPASS_REPLACE_VAL(pie_handle->pie_ngadjust->Config, 0);
    }
}
EXPORT_SYMBOL(pie_ngadjust_set_bypass);

void pie_ngadjust_set_lut(struct pie_handle_t *pie_handle, uint8_t *c1_lut, uint8_t *c2_lut)
{
    int i;
    
    PIE_PTRCHECK_NR(pie_ngadjust);

    for (i=0; i<PIE_NGADJUST_LUT_ENTRIES; i++)
    {
        pie_handle->pie_ngadjust->LUTAccess[i] = 0;
        pie_handle->pie_ngadjust->LUTAccess[i] =
            NGADJUST_LUTACCESS_C1_REPLACE_VAL(pie_handle->pie_ngadjust->LUTAccess[i],
                                              c1_lut[i]);
        pie_handle->pie_ngadjust->LUTAccess[i] =
            NGADJUST_LUTACCESS_C2_REPLACE_VAL(pie_handle->pie_ngadjust->LUTAccess[i],
                                              c2_lut[i]);
    }
}
EXPORT_SYMBOL(pie_ngadjust_set_lut);

////////////////////////////////////////////////////////////////////////
// pie dsmf
////////////////////////////////////////////////////////////////////////

// get defaults for dsmf
static void get_dsmf_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    int i;

    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = DSMF_DSMF_CTL_BYPASS_ALL_REPLACE_VAL(regval, 1);
    regval = DSMF_DSMF_CTL_EN_POGO_OFFSET_REPLACE_VAL(regval, 1);
    regval = DSMF_DSMF_CTL_ALPHA_FILT_EN_REPLACE_VAL(regval, 1);
    pie_handle->pie_dsmf->DSMF_Ctl = regval;
    pie_handle->pie_dsmf->DSMF_Seg_HV_Proj = 0;
    pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj = 0;
    pie_handle->pie_dsmf->DSMF_Seg_Isolated = 0;
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Text = 0;
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed = 0;
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Text = 0;
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed = 0;
    pie_handle->pie_dsmf->DSMF_Seg_Line = 0;
    pie_handle->pie_dsmf->DSMF_Seg_Line_Diag = 0;
    pie_handle->pie_dsmf->DSMF_SCR1 = 0;
    pie_handle->pie_dsmf->DSMF_SCR2 = 0;
    pie_handle->pie_dsmf->DSMF_TEL1R = 0;
    pie_handle->pie_dsmf->DSMF_TEL2R = 0;
    pie_handle->pie_dsmf->DSMF_TEL3R = 0;
    pie_handle->pie_dsmf->DSMF_EC_COMPARE = 0;
    pie_handle->pie_dsmf->DSMF_EC_WR = 0;
    pie_handle->pie_dsmf->DSMF_EC_OBS = 0;
    pie_handle->pie_dsmf->DSMF_ECI = 0;
    pie_handle->pie_dsmf->DSMF_YVC00_01 = 0;
    pie_handle->pie_dsmf->DSMF_YVC02_03 = 0;
    pie_handle->pie_dsmf->DSMF_YVC04_10 = 0;
    pie_handle->pie_dsmf->DSMF_YVC11_12 = 0;
    pie_handle->pie_dsmf->DSMF_YVC13_14 = 0;
    pie_handle->pie_dsmf->DSMF_YVC20_21 = 0;
    pie_handle->pie_dsmf->DSMF_YVC22_23 = 0;
    pie_handle->pie_dsmf->DSMF_YVC24 = 0;
    pie_handle->pie_dsmf->DSMF_YVC30_31 = 0;
    pie_handle->pie_dsmf->DSMF_YVC32_33 = 0;
    pie_handle->pie_dsmf->DSMF_YVC34_40 = 0;
    pie_handle->pie_dsmf->DSMF_YVC41_42 = 0;
    pie_handle->pie_dsmf->DSMF_YVC43_44 = 0;
    pie_handle->pie_dsmf->DSMF_CVC00_01 = 0;
    pie_handle->pie_dsmf->DSMF_CVC02_03 = 0;
    pie_handle->pie_dsmf->DSMF_CVC04_10 = 0;
    pie_handle->pie_dsmf->DSMF_CVC11_12 = 0;
    pie_handle->pie_dsmf->DSMF_CVC13_14 = 0;
    pie_handle->pie_dsmf->DSMF_CVC20_21 = 0;
    pie_handle->pie_dsmf->DSMF_CVC22_23 = 0;
    pie_handle->pie_dsmf->DSMF_CVC24 = 0;
    pie_handle->pie_dsmf->DSMF_CVC30_31 = 0;
    pie_handle->pie_dsmf->DSMF_CVC32_33 = 0;
    pie_handle->pie_dsmf->DSMF_CVC34_40 = 0;
    pie_handle->pie_dsmf->DSMF_CVC41_42 = 0;
    pie_handle->pie_dsmf->DSMF_CVC43_44 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_MULT = 0;
    pie_handle->pie_dsmf->DSMF_FILT_SHIFT = 0;
    pie_handle->pie_dsmf->DSMF_YVC00_01_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC02_03_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC04_10_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC11_12_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC13_14_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC20_21_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC22_23_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC24_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC30_31_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC32_33_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC34_40_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC41_42_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC43_44_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC00_01_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC02_03_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC04_10_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC11_12_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC13_14_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC20_21_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC22_23_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC24_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC30_31_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC32_33_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC34_40_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC41_42_F1 = 0;
    pie_handle->pie_dsmf->DSMF_CVC43_44_F1 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_MULT_F1 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1 = 0;
    pie_handle->pie_dsmf->DSMF_YVC00_01_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC02_03_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC04_10_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC11_12_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC13_14_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC20_21_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC22_23_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC24_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC30_31_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC32_33_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC34_40_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC41_42_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC43_44_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC00_01_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC02_03_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC04_10_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC11_12_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC13_14_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC20_21_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC22_23_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC24_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC30_31_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC32_33_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC34_40_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC41_42_F2 = 0;
    pie_handle->pie_dsmf->DSMF_CVC43_44_F2 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_MULT_F2 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2 = 0;
    pie_handle->pie_dsmf->DSMF_YVC00_01_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC02_03_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC04_10_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC11_12_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC13_14_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC20_21_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC22_23_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC24_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC30_31_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC32_33_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC34_40_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC41_42_F3 = 0;
    pie_handle->pie_dsmf->DSMF_YVC43_44_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC00_01_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC02_03_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC04_10_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC11_12_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC13_14_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC20_21_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC22_23_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC24_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC30_31_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC32_33_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC34_40_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC41_42_F3 = 0;
    pie_handle->pie_dsmf->DSMF_CVC43_44_F3 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_MULT_F3 = 0;
    pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3 = 0;
    regval = 0;
    regval = DSMF_DSMF_AVG_MASK_AVG_MASK_FIXED_REPLACE_VAL(regval, 0xFF);
    regval = DSMF_DSMF_AVG_MASK_AVG_MASK_SEL_REPLACE_VAL(regval, 0x1);    
    pie_handle->pie_dsmf->DSMF_AVG_MASK = regval;

    for (i=0; i<PIE_DSMF_CTL_LUT_ENTRIES; i++)
    {
        pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i] = 0;
    }

    for (i=0; i<PIE_DSMF_SEG_PROJ_LUT_ENTRIES; i++)
    {
        // ballen Jan 2015
        // This LUT maps an 8 bit Y value to a new Y value. Sane defaults here
        // mean keeping the new Y equal to the old Y
        pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i] = i;
    }
    
    for (i=0; i<PIE_DSMF_AI_LUT_ENTRIES; i++)
    {
        pie_handle->pie_dsmf->DSMF_AI_LUT1[i] = 0;
        pie_handle->pie_dsmf->DSMF_AI_LUT0[i] = 0;
    }
        
    pie_handle->pie_dsmf->REV0 = PIE_DSMF_MAJ_REV0;
    pie_handle->pie_dsmf->REV1 = 0;

}

static void pie_dsmf_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    int i;
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_dsmf);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_dsmf);    

    dest_pie_handle->pie_dsmf->DSMF_Ctl = src_pie_handle->pie_dsmf->DSMF_Ctl;
    dest_pie_handle->pie_dsmf->DSMF_Seg_HV_Proj = src_pie_handle->pie_dsmf->DSMF_Seg_HV_Proj;
    dest_pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj = src_pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj;
    dest_pie_handle->pie_dsmf->DSMF_Seg_Isolated = src_pie_handle->pie_dsmf->DSMF_Seg_Isolated;
    dest_pie_handle->pie_dsmf->DSMF_Seg_3x3_Text = src_pie_handle->pie_dsmf->DSMF_Seg_3x3_Text;
    dest_pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed = src_pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed;
    dest_pie_handle->pie_dsmf->DSMF_Seg_5x5_Text = src_pie_handle->pie_dsmf->DSMF_Seg_5x5_Text;
    dest_pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed = src_pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed;
    dest_pie_handle->pie_dsmf->DSMF_Seg_Line = src_pie_handle->pie_dsmf->DSMF_Seg_Line;
    dest_pie_handle->pie_dsmf->DSMF_Seg_Line_Diag = src_pie_handle->pie_dsmf->DSMF_Seg_Line_Diag;
    dest_pie_handle->pie_dsmf->DSMF_SCR1 = src_pie_handle->pie_dsmf->DSMF_SCR1;
    dest_pie_handle->pie_dsmf->DSMF_SCR2 = src_pie_handle->pie_dsmf->DSMF_SCR2;
    dest_pie_handle->pie_dsmf->DSMF_TEL1R = src_pie_handle->pie_dsmf->DSMF_TEL1R;
    dest_pie_handle->pie_dsmf->DSMF_TEL2R = src_pie_handle->pie_dsmf->DSMF_TEL2R;
    dest_pie_handle->pie_dsmf->DSMF_TEL3R = src_pie_handle->pie_dsmf->DSMF_TEL3R;
    dest_pie_handle->pie_dsmf->DSMF_EC_COMPARE = src_pie_handle->pie_dsmf->DSMF_EC_COMPARE;
    dest_pie_handle->pie_dsmf->DSMF_EC_WR = src_pie_handle->pie_dsmf->DSMF_EC_WR;
    dest_pie_handle->pie_dsmf->DSMF_EC_OBS = src_pie_handle->pie_dsmf->DSMF_EC_OBS;
    dest_pie_handle->pie_dsmf->DSMF_ECI = src_pie_handle->pie_dsmf->DSMF_ECI;
    dest_pie_handle->pie_dsmf->DSMF_YVC00_01 = src_pie_handle->pie_dsmf->DSMF_YVC00_01;
    dest_pie_handle->pie_dsmf->DSMF_YVC02_03 = src_pie_handle->pie_dsmf->DSMF_YVC02_03;
    dest_pie_handle->pie_dsmf->DSMF_YVC04_10 = src_pie_handle->pie_dsmf->DSMF_YVC04_10;
    dest_pie_handle->pie_dsmf->DSMF_YVC11_12 = src_pie_handle->pie_dsmf->DSMF_YVC11_12;
    dest_pie_handle->pie_dsmf->DSMF_YVC13_14 = src_pie_handle->pie_dsmf->DSMF_YVC13_14;
    dest_pie_handle->pie_dsmf->DSMF_YVC20_21 = src_pie_handle->pie_dsmf->DSMF_YVC20_21;
    dest_pie_handle->pie_dsmf->DSMF_YVC22_23 = src_pie_handle->pie_dsmf->DSMF_YVC22_23;
    dest_pie_handle->pie_dsmf->DSMF_YVC24 = src_pie_handle->pie_dsmf->DSMF_YVC24;
    dest_pie_handle->pie_dsmf->DSMF_YVC30_31 = src_pie_handle->pie_dsmf->DSMF_YVC30_31;
    dest_pie_handle->pie_dsmf->DSMF_YVC32_33 = src_pie_handle->pie_dsmf->DSMF_YVC32_33;
    dest_pie_handle->pie_dsmf->DSMF_YVC34_40 = src_pie_handle->pie_dsmf->DSMF_YVC34_40;
    dest_pie_handle->pie_dsmf->DSMF_YVC41_42 = src_pie_handle->pie_dsmf->DSMF_YVC41_42;
    dest_pie_handle->pie_dsmf->DSMF_YVC43_44 = src_pie_handle->pie_dsmf->DSMF_YVC43_44;
    dest_pie_handle->pie_dsmf->DSMF_CVC00_01 = src_pie_handle->pie_dsmf->DSMF_CVC00_01;
    dest_pie_handle->pie_dsmf->DSMF_CVC02_03 = src_pie_handle->pie_dsmf->DSMF_CVC02_03;
    dest_pie_handle->pie_dsmf->DSMF_CVC04_10 = src_pie_handle->pie_dsmf->DSMF_CVC04_10;
    dest_pie_handle->pie_dsmf->DSMF_CVC11_12 = src_pie_handle->pie_dsmf->DSMF_CVC11_12;
    dest_pie_handle->pie_dsmf->DSMF_CVC13_14 = src_pie_handle->pie_dsmf->DSMF_CVC13_14;
    dest_pie_handle->pie_dsmf->DSMF_CVC20_21 = src_pie_handle->pie_dsmf->DSMF_CVC20_21;
    dest_pie_handle->pie_dsmf->DSMF_CVC22_23 = src_pie_handle->pie_dsmf->DSMF_CVC22_23;
    dest_pie_handle->pie_dsmf->DSMF_CVC24 = src_pie_handle->pie_dsmf->DSMF_CVC24;
    dest_pie_handle->pie_dsmf->DSMF_CVC30_31 = src_pie_handle->pie_dsmf->DSMF_CVC30_31;
    dest_pie_handle->pie_dsmf->DSMF_CVC32_33 = src_pie_handle->pie_dsmf->DSMF_CVC32_33;
    dest_pie_handle->pie_dsmf->DSMF_CVC34_40 = src_pie_handle->pie_dsmf->DSMF_CVC34_40;
    dest_pie_handle->pie_dsmf->DSMF_CVC41_42 = src_pie_handle->pie_dsmf->DSMF_CVC41_42;
    dest_pie_handle->pie_dsmf->DSMF_CVC43_44 = src_pie_handle->pie_dsmf->DSMF_CVC43_44;
    dest_pie_handle->pie_dsmf->DSMF_FILT_MULT = src_pie_handle->pie_dsmf->DSMF_FILT_MULT;
    dest_pie_handle->pie_dsmf->DSMF_FILT_SHIFT = src_pie_handle->pie_dsmf->DSMF_FILT_SHIFT;
    dest_pie_handle->pie_dsmf->DSMF_YVC00_01_F1 = src_pie_handle->pie_dsmf->DSMF_YVC00_01_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC02_03_F1 = src_pie_handle->pie_dsmf->DSMF_YVC02_03_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC04_10_F1 = src_pie_handle->pie_dsmf->DSMF_YVC04_10_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC11_12_F1 = src_pie_handle->pie_dsmf->DSMF_YVC11_12_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC13_14_F1 = src_pie_handle->pie_dsmf->DSMF_YVC13_14_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC20_21_F1 = src_pie_handle->pie_dsmf->DSMF_YVC20_21_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC22_23_F1 = src_pie_handle->pie_dsmf->DSMF_YVC22_23_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC24_F1 = src_pie_handle->pie_dsmf->DSMF_YVC24_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC30_31_F1 = src_pie_handle->pie_dsmf->DSMF_YVC30_31_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC32_33_F1 = src_pie_handle->pie_dsmf->DSMF_YVC32_33_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC34_40_F1 = src_pie_handle->pie_dsmf->DSMF_YVC34_40_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC41_42_F1 = src_pie_handle->pie_dsmf->DSMF_YVC41_42_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC43_44_F1 = src_pie_handle->pie_dsmf->DSMF_YVC43_44_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC00_01_F1 = src_pie_handle->pie_dsmf->DSMF_CVC00_01_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC02_03_F1 = src_pie_handle->pie_dsmf->DSMF_CVC02_03_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC04_10_F1 = src_pie_handle->pie_dsmf->DSMF_CVC04_10_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC11_12_F1 = src_pie_handle->pie_dsmf->DSMF_CVC11_12_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC13_14_F1 = src_pie_handle->pie_dsmf->DSMF_CVC13_14_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC20_21_F1 = src_pie_handle->pie_dsmf->DSMF_CVC20_21_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC22_23_F1 = src_pie_handle->pie_dsmf->DSMF_CVC22_23_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC24_F1 = src_pie_handle->pie_dsmf->DSMF_CVC24_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC30_31_F1 = src_pie_handle->pie_dsmf->DSMF_CVC30_31_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC32_33_F1 = src_pie_handle->pie_dsmf->DSMF_CVC32_33_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC34_40_F1 = src_pie_handle->pie_dsmf->DSMF_CVC34_40_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC41_42_F1 = src_pie_handle->pie_dsmf->DSMF_CVC41_42_F1;
    dest_pie_handle->pie_dsmf->DSMF_CVC43_44_F1 = src_pie_handle->pie_dsmf->DSMF_CVC43_44_F1;
    dest_pie_handle->pie_dsmf->DSMF_FILT_MULT_F1 = src_pie_handle->pie_dsmf->DSMF_FILT_MULT_F1;
    dest_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1 = src_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1;
    dest_pie_handle->pie_dsmf->DSMF_YVC00_01_F2 = src_pie_handle->pie_dsmf->DSMF_YVC00_01_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC02_03_F2 = src_pie_handle->pie_dsmf->DSMF_YVC02_03_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC04_10_F2 = src_pie_handle->pie_dsmf->DSMF_YVC04_10_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC11_12_F2 = src_pie_handle->pie_dsmf->DSMF_YVC11_12_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC13_14_F2 = src_pie_handle->pie_dsmf->DSMF_YVC13_14_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC20_21_F2 = src_pie_handle->pie_dsmf->DSMF_YVC20_21_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC22_23_F2 = src_pie_handle->pie_dsmf->DSMF_YVC22_23_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC24_F2 = src_pie_handle->pie_dsmf->DSMF_YVC24_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC30_31_F2 = src_pie_handle->pie_dsmf->DSMF_YVC30_31_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC32_33_F2 = src_pie_handle->pie_dsmf->DSMF_YVC32_33_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC34_40_F2 = src_pie_handle->pie_dsmf->DSMF_YVC34_40_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC41_42_F2 = src_pie_handle->pie_dsmf->DSMF_YVC41_42_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC43_44_F2 = src_pie_handle->pie_dsmf->DSMF_YVC43_44_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC00_01_F2 = src_pie_handle->pie_dsmf->DSMF_CVC00_01_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC02_03_F2 = src_pie_handle->pie_dsmf->DSMF_CVC02_03_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC04_10_F2 = src_pie_handle->pie_dsmf->DSMF_CVC04_10_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC11_12_F2 = src_pie_handle->pie_dsmf->DSMF_CVC11_12_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC13_14_F2 = src_pie_handle->pie_dsmf->DSMF_CVC13_14_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC20_21_F2 = src_pie_handle->pie_dsmf->DSMF_CVC20_21_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC22_23_F2 = src_pie_handle->pie_dsmf->DSMF_CVC22_23_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC24_F2 = src_pie_handle->pie_dsmf->DSMF_CVC24_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC30_31_F2 = src_pie_handle->pie_dsmf->DSMF_CVC30_31_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC32_33_F2 = src_pie_handle->pie_dsmf->DSMF_CVC32_33_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC34_40_F2 = src_pie_handle->pie_dsmf->DSMF_CVC34_40_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC41_42_F2 = src_pie_handle->pie_dsmf->DSMF_CVC41_42_F2;
    dest_pie_handle->pie_dsmf->DSMF_CVC43_44_F2 = src_pie_handle->pie_dsmf->DSMF_CVC43_44_F2;
    dest_pie_handle->pie_dsmf->DSMF_FILT_MULT_F2 = src_pie_handle->pie_dsmf->DSMF_FILT_MULT_F2;
    dest_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2 = src_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2;
    dest_pie_handle->pie_dsmf->DSMF_YVC00_01_F3 = src_pie_handle->pie_dsmf->DSMF_YVC00_01_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC02_03_F3 = src_pie_handle->pie_dsmf->DSMF_YVC02_03_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC04_10_F3 = src_pie_handle->pie_dsmf->DSMF_YVC04_10_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC11_12_F3 = src_pie_handle->pie_dsmf->DSMF_YVC11_12_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC13_14_F3 = src_pie_handle->pie_dsmf->DSMF_YVC13_14_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC20_21_F3 = src_pie_handle->pie_dsmf->DSMF_YVC20_21_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC22_23_F3 = src_pie_handle->pie_dsmf->DSMF_YVC22_23_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC24_F3 = src_pie_handle->pie_dsmf->DSMF_YVC24_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC30_31_F3 = src_pie_handle->pie_dsmf->DSMF_YVC30_31_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC32_33_F3 = src_pie_handle->pie_dsmf->DSMF_YVC32_33_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC34_40_F3 = src_pie_handle->pie_dsmf->DSMF_YVC34_40_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC41_42_F3 = src_pie_handle->pie_dsmf->DSMF_YVC41_42_F3;
    dest_pie_handle->pie_dsmf->DSMF_YVC43_44_F3 = src_pie_handle->pie_dsmf->DSMF_YVC43_44_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC00_01_F3 = src_pie_handle->pie_dsmf->DSMF_CVC00_01_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC02_03_F3 = src_pie_handle->pie_dsmf->DSMF_CVC02_03_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC04_10_F3 = src_pie_handle->pie_dsmf->DSMF_CVC04_10_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC11_12_F3 = src_pie_handle->pie_dsmf->DSMF_CVC11_12_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC13_14_F3 = src_pie_handle->pie_dsmf->DSMF_CVC13_14_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC20_21_F3 = src_pie_handle->pie_dsmf->DSMF_CVC20_21_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC22_23_F3 = src_pie_handle->pie_dsmf->DSMF_CVC22_23_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC24_F3 = src_pie_handle->pie_dsmf->DSMF_CVC24_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC30_31_F3 = src_pie_handle->pie_dsmf->DSMF_CVC30_31_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC32_33_F3 = src_pie_handle->pie_dsmf->DSMF_CVC32_33_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC34_40_F3 = src_pie_handle->pie_dsmf->DSMF_CVC34_40_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC41_42_F3 = src_pie_handle->pie_dsmf->DSMF_CVC41_42_F3;
    dest_pie_handle->pie_dsmf->DSMF_CVC43_44_F3 = src_pie_handle->pie_dsmf->DSMF_CVC43_44_F3;
    dest_pie_handle->pie_dsmf->DSMF_FILT_MULT_F3 = src_pie_handle->pie_dsmf->DSMF_FILT_MULT_F3;
    dest_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3 = src_pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3;
    dest_pie_handle->pie_dsmf->DSMF_AVG_MASK = src_pie_handle->pie_dsmf->DSMF_AVG_MASK;
    // DSMF_FILT_CTL_LUT cloned in loop below
    // DSMF_SEG_PROJ_LUT cloned in loop below
    // DSMF_AI_LUT0 cloned in loop below
    // DSMF_AI_LUT1 cloned in loop below
    dest_pie_handle->pie_dsmf->REV0 = src_pie_handle->pie_dsmf->REV0;
    dest_pie_handle->pie_dsmf->REV1 = src_pie_handle->pie_dsmf->REV1;

    for (i=0;i<PIE_DSMF_CTL_LUT_ENTRIES;i++)
    {
        dest_pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i] = src_pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i];
    }
    for (i=0; i<PIE_DSMF_SEG_PROJ_LUT_ENTRIES; i++)
    {
        dest_pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i] = src_pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i];
    }
    for (i=0;i<PIE_DSMF_AI_LUT_ENTRIES;i++) 
    {    
        dest_pie_handle->pie_dsmf->DSMF_AI_LUT0[i] = src_pie_handle->pie_dsmf->DSMF_AI_LUT0[i];
        dest_pie_handle->pie_dsmf->DSMF_AI_LUT1[i] = src_pie_handle->pie_dsmf->DSMF_AI_LUT1[i];
    }
}

void pie_dsmf_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE DSMF regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_dsmf);

    print("DSMF_Ctl=0x%X\n", pie_handle->pie_dsmf->DSMF_Ctl);
    print("DSMF_Seg_HV_Proj=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_HV_Proj);
    print("DSMF_Seg_Diag_Proj=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj);
    print("DSMF_Seg_Isolated=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_Isolated);
    print("DSMF_Seg_3x3_Text=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_3x3_Text);
    print("DSMF_Seg_3x3_Mixed=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed);
    print("DSMF_Seg_5x5_Text=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_5x5_Text);
    print("DSMF_Seg_5x5_Mixed=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed);
    print("DSMF_Seg_Line=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_Line);
    print("DSMF_Seg_Line_Diag=0x%X\n", pie_handle->pie_dsmf->DSMF_Seg_Line_Diag);
    print("DSMF_SCR1=0x%X\n", pie_handle->pie_dsmf->DSMF_SCR1);
    print("DSMF_SCR2=0x%X\n", pie_handle->pie_dsmf->DSMF_SCR2);
    print("DSMF_TEL1R=0x%X\n", pie_handle->pie_dsmf->DSMF_TEL1R);
    print("DSMF_TEL2R=0x%X\n", pie_handle->pie_dsmf->DSMF_TEL2R);
    print("DSMF_TEL3R=0x%X\n", pie_handle->pie_dsmf->DSMF_TEL3R);
    print("DSMF_EC_COMPARE=0x%X\n", pie_handle->pie_dsmf->DSMF_EC_COMPARE);
    // DSMF_EC_WR is a write only register
    print("DSMF_EC_OBS=0x%X\n", pie_handle->pie_dsmf->DSMF_EC_OBS);
    print("DSMF_ECI=0x%X\n", pie_handle->pie_dsmf->DSMF_ECI);
    print("DSMF_YVC00_01=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC00_01);
    print("DSMF_YVC02_03=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC02_03);
    print("DSMF_YVC04_10=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC04_10);
    print("DSMF_YVC11_12=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC11_12);
    print("DSMF_YVC13_14=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC13_14);
    print("DSMF_YVC20_21=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC20_21);
    print("DSMF_YVC22_23=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC22_23);
    print("DSMF_YVC24=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC24);
    print("DSMF_YVC30_31=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC30_31);
    print("DSMF_YVC32_33=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC32_33);
    print("DSMF_YVC34_40=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC34_40);
    print("DSMF_YVC41_42=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC41_42);
    print("DSMF_YVC43_44=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC43_44);
    print("DSMF_CVC00_01=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC00_01);
    print("DSMF_CVC02_03=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC02_03);
    print("DSMF_CVC04_10=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC04_10);
    print("DSMF_CVC11_12=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC11_12);
    print("DSMF_CVC13_14=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC13_14);
    print("DSMF_CVC20_21=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC20_21);
    print("DSMF_CVC22_23=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC22_23);
    print("DSMF_CVC24=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC24);
    print("DSMF_CVC30_31=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC30_31);
    print("DSMF_CVC32_33=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC32_33);
    print("DSMF_CVC34_40=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC34_40);
    print("DSMF_CVC41_42=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC41_42);
    print("DSMF_CVC43_44=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC43_44);
    print("DSMF_FILT_MULT=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_MULT);
    print("DSMF_FILT_SHIFT=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_SHIFT);
    print("DSMF_YVC00_01_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC00_01_F1);
    print("DSMF_YVC02_03_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC02_03_F1);
    print("DSMF_YVC04_10_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC04_10_F1);
    print("DSMF_YVC11_12_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC11_12_F1);
    print("DSMF_YVC13_14_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC13_14_F1);
    print("DSMF_YVC20_21_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC20_21_F1);
    print("DSMF_YVC22_23_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC22_23_F1);
    print("DSMF_YVC24_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC24_F1);
    print("DSMF_YVC30_31_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC30_31_F1);
    print("DSMF_YVC32_33_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC32_33_F1);
    print("DSMF_YVC34_40_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC34_40_F1);
    print("DSMF_YVC41_42_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC41_42_F1);
    print("DSMF_YVC43_44_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC43_44_F1);
    print("DSMF_CVC00_01_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC00_01_F1);
    print("DSMF_CVC02_03_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC02_03_F1);
    print("DSMF_CVC04_10_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC04_10_F1);
    print("DSMF_CVC11_12_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC11_12_F1);
    print("DSMF_CVC13_14_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC13_14_F1);
    print("DSMF_CVC20_21_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC20_21_F1);
    print("DSMF_CVC22_23_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC22_23_F1);
    print("DSMF_CVC24_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC24_F1);
    print("DSMF_CVC30_31_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC30_31_F1);
    print("DSMF_CVC32_33_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC32_33_F1);
    print("DSMF_CVC34_40_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC34_40_F1);
    print("DSMF_CVC41_42_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC41_42_F1);
    print("DSMF_CVC43_44_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC43_44_F1);
    print("DSMF_FILT_MULT_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_MULT_F1);
    print("DSMF_FILT_SHIFT_F1=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1);
    print("DSMF_YVC00_01_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC00_01_F2);
    print("DSMF_YVC02_03_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC02_03_F2);
    print("DSMF_YVC04_10_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC04_10_F2);
    print("DSMF_YVC11_12_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC11_12_F2);
    print("DSMF_YVC13_14_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC13_14_F2);
    print("DSMF_YVC20_21_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC20_21_F2);
    print("DSMF_YVC22_23_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC22_23_F2);
    print("DSMF_YVC24_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC24_F2);
    print("DSMF_YVC30_31_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC30_31_F2);
    print("DSMF_YVC32_33_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC32_33_F2);
    print("DSMF_YVC34_40_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC34_40_F2);
    print("DSMF_YVC41_42_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC41_42_F2);
    print("DSMF_YVC43_44_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC43_44_F2);
    print("DSMF_CVC00_01_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC00_01_F2);
    print("DSMF_CVC02_03_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC02_03_F2);
    print("DSMF_CVC04_10_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC04_10_F2);
    print("DSMF_CVC11_12_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC11_12_F2);
    print("DSMF_CVC13_14_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC13_14_F2);
    print("DSMF_CVC20_21_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC20_21_F2);
    print("DSMF_CVC22_23_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC22_23_F2);
    print("DSMF_CVC24_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC24_F2);
    print("DSMF_CVC30_31_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC30_31_F2);
    print("DSMF_CVC32_33_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC32_33_F2);
    print("DSMF_CVC34_40_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC34_40_F2);
    print("DSMF_CVC41_42_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC41_42_F2);
    print("DSMF_CVC43_44_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC43_44_F2);
    print("DSMF_FILT_MULT_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_MULT_F2);
    print("DSMF_FILT_SHIFT_F2=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2);
    print("DSMF_YVC00_01_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC00_01_F3);
    print("DSMF_YVC02_03_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC02_03_F3);
    print("DSMF_YVC04_10_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC04_10_F3);
    print("DSMF_YVC11_12_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC11_12_F3);
    print("DSMF_YVC13_14_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC13_14_F3);
    print("DSMF_YVC20_21_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC20_21_F3);
    print("DSMF_YVC22_23_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC22_23_F3);
    print("DSMF_YVC24_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC24_F3);
    print("DSMF_YVC30_31_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC30_31_F3);
    print("DSMF_YVC32_33_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC32_33_F3);
    print("DSMF_YVC34_40_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC34_40_F3);
    print("DSMF_YVC41_42_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC41_42_F3);
    print("DSMF_YVC43_44_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_YVC43_44_F3);
    print("DSMF_CVC00_01_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC00_01_F3);
    print("DSMF_CVC02_03_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC02_03_F3);
    print("DSMF_CVC04_10_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC04_10_F3);
    print("DSMF_CVC11_12_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC11_12_F3);
    print("DSMF_CVC13_14_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC13_14_F3);
    print("DSMF_CVC20_21_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC20_21_F3);
    print("DSMF_CVC22_23_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC22_23_F3);
    print("DSMF_CVC24_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC24_F3);
    print("DSMF_CVC30_31_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC30_31_F3);
    print("DSMF_CVC32_33_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC32_33_F3);
    print("DSMF_CVC34_40_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC34_40_F3);
    print("DSMF_CVC41_42_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC41_42_F3);
    print("DSMF_CVC43_44_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_CVC43_44_F3);
    print("DSMF_FILT_MULT_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_MULT_F3);
    print("DSMF_FILT_SHIFT_F3=0x%X\n", pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3);
    print("DSMF_AVG_MASK=0x%X\n", pie_handle->pie_dsmf->DSMF_AVG_MASK);
    // DSMF_FILT_CTL_LUT not dumped - just doing regs
    // DSMF_SEG_PROJ_LUT not dumped - just doing regs, and it's write only
    // DSMF_AI_LUT0 not dumped - just doing regs, and it's write only
    // DSMF_AI_LUT1 not dumped - just doing regs, and it's write only    
    print("REV0=0x%X\n", pie_handle->pie_dsmf->REV0);
    print("REV1=0x%X\n", pie_handle->pie_dsmf->REV1);    
    
}
EXPORT_SYMBOL(pie_dsmf_dump_handle_regs);

// 3 of the 4 LUTs are write only, but we have these in the pie_handle as shadow LUTs
void pie_dsmf_dump_handle_luts(struct pie_handle_t *pie_handle)
{
    int i;

    PIE_PTRCHECK_NR(pie_dsmf);

    print("PIE DSMF LUT entries\n");

    print("DSMF_FILT_CTL_LUT:\n");
    for (i=0;i<PIE_DSMF_CTL_LUT_ENTRIES;i++) 
        print("0x%X ", pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i]);

    print("\nDSMF_SEG_PROJ_LUT:");
    for (i=0;i<PIE_DSMF_SEG_PROJ_LUT_ENTRIES;i++) 
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        print("0x%X ", pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i]);
    }

    print("\nDSMF_AI_LUT0:");
    for (i=0;i<PIE_DSMF_AI_LUT_ENTRIES;i++) 
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        print("0x%X ", pie_handle->pie_dsmf->DSMF_AI_LUT0[i]);
    }

    print("\nDSMF_AI_LUT1:");
    for (i=0;i<PIE_DSMF_AI_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);
        print("0x%X ", pie_handle->pie_dsmf->DSMF_AI_LUT1[i]);
    }
    print("\n");
}
EXPORT_SYMBOL(pie_dsmf_dump_handle_luts);

void pie_dsmf_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    if (bypass)
    {
        /* Set the bypass bit */
        pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_BYPASS_ALL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, PIE_CSC_BYPASS);
    }
    else
    {
        /* Clear the bypass bit */
        pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_BYPASS_ALL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, PIE_CSC_ENABLE);
    }
}
EXPORT_SYMBOL(pie_dsmf_set_bypass);


void pie_dsmf_setup_ctl(struct pie_handle_t *pie_handle, bool test_mixed, bool test_halftone,
                               bool test_text, bool test_isolated, bool alpha_filter_en,
                               uint8_t filt_idx_sel, uint8_t ot_out_sel)
{
    /* Put in filter in bypass. Stil chews rows, though */
    //pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_BYPASS_FILT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, true);

    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_TEST_MIXED_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, test_mixed);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_TEST_HALFTONE_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, test_halftone);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_TEST_TEXT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, test_text);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_TEST_ISOLATED_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, test_isolated);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_ALPHA_FILT_EN_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, alpha_filter_en);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_FILT_IDX_SEL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, filt_idx_sel);
    pie_handle->pie_dsmf->DSMF_Ctl = DSMF_DSMF_CTL_OT_OUT_SEL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Ctl, ot_out_sel);
}
EXPORT_SYMBOL(pie_dsmf_setup_ctl);

void pie_dsmf_setup_sculptor_config(struct pie_handle_t *pie_handle, int scale_10, int scale_15, int scale_20,
                                    int lut_output_scale, int offset, bool use_erosion, bool use_dilation, bool te_use_5x5_matrix )
{
    pie_handle->pie_dsmf->DSMF_SCR1 = DSMF_DSMF_SCR1_SCALE_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR1, scale_10);
    pie_handle->pie_dsmf->DSMF_SCR1 = DSMF_DSMF_SCR1_SCALE_1_5_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR1, scale_15);
    pie_handle->pie_dsmf->DSMF_SCR1 = DSMF_DSMF_SCR1_SCALE_2_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR1, scale_20);
    pie_handle->pie_dsmf->DSMF_SCR1 = DSMF_DSMF_SCR1_LUT_SCALE_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR1, lut_output_scale);
    pie_handle->pie_dsmf->DSMF_SCR1 = DSMF_DSMF_SCR1_OFFSET_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR1, offset);

    pie_handle->pie_dsmf->DSMF_SCR2 = DSMF_DSMF_SCR2_EN_EROSION_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR2, use_erosion);
    pie_handle->pie_dsmf->DSMF_SCR2 = DSMF_DSMF_SCR2_EN_DILATION_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR2, use_dilation);
    pie_handle->pie_dsmf->DSMF_SCR2 = DSMF_DSMF_SCR2_FIVEXFIVE_TE_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SCR2, te_use_5x5_matrix);
}
EXPORT_SYMBOL(pie_dsmf_setup_sculptor_config);

void pie_dsmf_setup_te(struct pie_handle_t *pie_handle, bool use_mf_te,
    uint32_t te_count_black_min,  uint32_t te_count_white_min,   uint32_t te_count_white_plus_black_min,
    uint32_t te_center_color_max, uint32_t te_intense_color_max, uint32_t te_black_y_max,
    uint32_t te_black_CbCr_max,   uint32_t te_white_y_min,       uint32_t te_white_CbCr_max)
{
    if(use_mf_te) {
        /* TEL1R */
        pie_handle->pie_dsmf->DSMF_TEL1R = DSMF_DSMF_TEL1R_CENTERCMAX_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL1R, te_center_color_max);
        pie_handle->pie_dsmf->DSMF_TEL1R = DSMF_DSMF_TEL1R_COUNTBWMIN_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL1R, te_count_white_plus_black_min);
        pie_handle->pie_dsmf->DSMF_TEL1R = DSMF_DSMF_TEL1R_COUNTWMIN_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL1R, te_count_white_min);
        pie_handle->pie_dsmf->DSMF_TEL1R = DSMF_DSMF_TEL1R_COUNTBMIN_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL1R, te_count_black_min);
        /* TEL2R */
        pie_handle->pie_dsmf->DSMF_TEL2R = DSMF_DSMF_TEL2R_CMAX_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL2R, te_intense_color_max);
        /* TEL3R */
        pie_handle->pie_dsmf->DSMF_TEL3R = DSMF_DSMF_TEL3R_WHITECRCBMAX_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL3R, te_white_CbCr_max);
        pie_handle->pie_dsmf->DSMF_TEL3R = DSMF_DSMF_TEL3R_WHITEYMIN_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL3R, te_white_y_min);
        pie_handle->pie_dsmf->DSMF_TEL3R = DSMF_DSMF_TEL3R_BLACKCRCBMAX_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL3R, te_black_CbCr_max);
        pie_handle->pie_dsmf->DSMF_TEL3R = DSMF_DSMF_TEL3R_BLACKYMAX_REPLACE_VAL(
                                               pie_handle->pie_dsmf->DSMF_TEL3R, te_black_y_max);
    }
}
EXPORT_SYMBOL(pie_dsmf_setup_te);

void pie_dsmf_set_filtctl_lut(struct pie_handle_t *pie_handle, const uint32_t * conv_filter_coeff_sel,
                                    const uint32_t * ai_lut_y_sel, const uint32_t * ai_lut_c_sel,
                                    const uint32_t * unsharp_mask_en, const uint32_t * te_enhance_en,
                                    uint32_t num_entries)
{
    int i;
    
    uint32_t lesser = (num_entries < PIE_DSMF_CTL_LUT_ENTRIES) ? num_entries : PIE_DSMF_CTL_LUT_ENTRIES;
    uint32_t val;

    PIE_PTRCHECK_NR(pie_dsmf);

    for (i=0; i<lesser; i++) {
        val = 0;

        /* Curiously, there don't seem to be pre-generated marcros for these. So, if the chip changes,
         * there will be dragons here. */
        val |= (conv_filter_coeff_sel[i] & 0x3) << 4;
        val |= (ai_lut_y_sel[i] & 0x1) << 3;
        val |= (ai_lut_c_sel[i] & 0x1) << 2;
        val |= (unsharp_mask_en[i] & 0x1) << 1;
        val |= (te_enhance_en[i] & 0x1) << 0;

        pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i] =
            DSMF_DSMF_FILT_CTL_LUT_FILT_CTL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i], val);
    }
}
EXPORT_SYMBOL(pie_dsmf_set_filtctl_lut);


/* Used in verification driver */
void pie_dsmf_set_filtctl_lut_raw(struct pie_handle_t *pie_handle, uint32_t *lut)
{
    int i;
    
    PIE_PTRCHECK_NR(pie_dsmf);

    for (i=0; i<PIE_DSMF_CTL_LUT_ENTRIES; i++) 
        pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i] =
            DSMF_DSMF_FILT_CTL_LUT_FILT_CTL_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_CTL_LUT[i], lut[i]);
}
EXPORT_SYMBOL(pie_dsmf_set_filtctl_lut_raw);

void pie_dsmf_set_seg_proj_lut(struct pie_handle_t *pie_handle, uint32_t *lut)
{
    int i;

    PIE_PTRCHECK_NR(pie_dsmf);

    for (i=0; i<PIE_DSMF_SEG_PROJ_LUT_ENTRIES; i++) 
        pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i] =
            DSMF_DSMF_SEG_PROJ_LUT_D_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_SEG_PROJ_LUT[i], lut[i]);
}
EXPORT_SYMBOL(pie_dsmf_set_seg_proj_lut);

void pie_dsmf_set_ai_lut0(struct pie_handle_t *pie_handle, int8_t *y_lut, uint32_t y_len, int8_t *crcb_lut, uint32_t crcb_len)
{
    int i;
    PIE_PTRCHECK_NR(pie_dsmf);

    if( y_len > PIE_DSMF_AI_LUT_ENTRIES )
        y_len = PIE_DSMF_AI_LUT_ENTRIES;
    if( crcb_len > PIE_DSMF_AI_LUT_ENTRIES )
        crcb_len = PIE_DSMF_AI_LUT_ENTRIES;

    if(y_lut) {
        for (i=0; i < y_len; i++)
        {
            uint8_t val = (uint8_t)y_lut[i]; // trim leading 1s if number is negative (2s complement)
            pie_handle->pie_dsmf->DSMF_AI_LUT0[i] =
                DSMF_DSMF_AI_LUT0_Y_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_AI_LUT0[i], val);
        }
    }
    if(crcb_lut) {
        for(i=0; i < crcb_len; i++) {
            uint8_t val = (uint8_t)crcb_lut[i]; // trim leading 1s if number is negative (2s complement)
            pie_handle->pie_dsmf->DSMF_AI_LUT0[i] =
                DSMF_DSMF_AI_LUT0_C_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_AI_LUT0[i], val);
        }
    }
}
EXPORT_SYMBOL(pie_dsmf_set_ai_lut0);

void pie_dsmf_set_ai_lut1(struct pie_handle_t *pie_handle, int8_t *y_lut, uint32_t y_len, int8_t *crcb_lut, uint32_t crcb_len)
{
    int i;
    PIE_PTRCHECK_NR(pie_dsmf);

    if( y_len > PIE_DSMF_AI_LUT_ENTRIES )
        y_len = PIE_DSMF_AI_LUT_ENTRIES;
    if( crcb_len > PIE_DSMF_AI_LUT_ENTRIES )
        crcb_len = PIE_DSMF_AI_LUT_ENTRIES;

    if(y_lut) {
        for (i=0; i < y_len; i++)
        {
            uint8_t val = (uint8_t)y_lut[i]; // trim leading 1s if number is negative (2s complement)
            pie_handle->pie_dsmf->DSMF_AI_LUT1[i] =
                DSMF_DSMF_AI_LUT1_Y_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_AI_LUT1[i], val);
        }
    }
    if(crcb_lut) {
        for(i=0; i < crcb_len; i++) {
            uint8_t val = (uint8_t)crcb_lut[i]; // trim leading 1s if number is negative (2s complement)
            pie_handle->pie_dsmf->DSMF_AI_LUT1[i] =
                DSMF_DSMF_AI_LUT1_C_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_AI_LUT1[i], val);
        }
    }
}
EXPORT_SYMBOL(pie_dsmf_set_ai_lut1);

static void pie_dsmf_set_varcoeff(volatile uint32_t * coeff_lut_start, const int32_t * coeff_values, uint32_t length)
{
    BUG_ON(length != 25); // hard coded since below values are also hard coded

    // ballen 01/2015
    // This is carefully set to work on any of the Y or CRCB coefficient "luts" (actually registers)
    // for any of the four filters (0 - 3). Some day, if these "luts" change, this code will break
    // badly. In the mean time, we save a whole lot of boilerplate.
    // 
    // I debated using a for loop to clean up these hard coded values, but I decided it was actually
    // more clear to use explicit values -- especially since there is a single, unpaired register
    // in the middle.
    //
    // Final note: I'm manually doing a bitwise AND on the mask inside the REPLACE macro because they
    // aren't graceful about handling negative signed values (the leading 1s in twos complement were
    // not bounded appropriately and were leaking past the register location).

    /* First 7 paired registers: 00_01, 02_03, 04_10, 11_12, 13_14, 20_21, 22_23 registers */
    coeff_lut_start[0]  = DSMF_DSMF_YVC00_01_A_REPLACE_VAL(coeff_lut_start[0],  coeff_values[0] & DSMF_DSMF_YVC00_01_A_MASK);
    coeff_lut_start[0]  = DSMF_DSMF_YVC00_01_B_REPLACE_VAL(coeff_lut_start[0],  coeff_values[1] & DSMF_DSMF_YVC00_01_A_MASK);
    coeff_lut_start[1]  = DSMF_DSMF_YVC02_03_A_REPLACE_VAL(coeff_lut_start[1],  coeff_values[2] & DSMF_DSMF_YVC02_03_A_MASK);
    coeff_lut_start[1]  = DSMF_DSMF_YVC02_03_B_REPLACE_VAL(coeff_lut_start[1],  coeff_values[3] & DSMF_DSMF_YVC02_03_A_MASK);
    coeff_lut_start[2]  = DSMF_DSMF_YVC04_10_A_REPLACE_VAL(coeff_lut_start[2],  coeff_values[4] & DSMF_DSMF_YVC04_10_A_MASK);
    coeff_lut_start[2]  = DSMF_DSMF_YVC04_10_B_REPLACE_VAL(coeff_lut_start[2],  coeff_values[5] & DSMF_DSMF_YVC04_10_A_MASK);
    coeff_lut_start[3]  = DSMF_DSMF_YVC11_12_A_REPLACE_VAL(coeff_lut_start[3],  coeff_values[6] & DSMF_DSMF_YVC11_12_A_MASK);
    coeff_lut_start[3]  = DSMF_DSMF_YVC11_12_B_REPLACE_VAL(coeff_lut_start[3],  coeff_values[7] & DSMF_DSMF_YVC11_12_A_MASK);
    coeff_lut_start[4]  = DSMF_DSMF_YVC13_14_A_REPLACE_VAL(coeff_lut_start[4],  coeff_values[8] & DSMF_DSMF_YVC13_14_A_MASK);
    coeff_lut_start[4]  = DSMF_DSMF_YVC13_14_B_REPLACE_VAL(coeff_lut_start[4],  coeff_values[9] & DSMF_DSMF_YVC13_14_A_MASK);
    coeff_lut_start[5]  = DSMF_DSMF_YVC20_21_A_REPLACE_VAL(coeff_lut_start[5],  coeff_values[10] & DSMF_DSMF_YVC20_21_A_MASK);
    coeff_lut_start[5]  = DSMF_DSMF_YVC20_21_B_REPLACE_VAL(coeff_lut_start[5],  coeff_values[11] & DSMF_DSMF_YVC20_21_A_MASK);
    coeff_lut_start[6]  = DSMF_DSMF_YVC22_23_A_REPLACE_VAL(coeff_lut_start[6],  coeff_values[12] & DSMF_DSMF_YVC22_23_A_MASK);
    coeff_lut_start[6]  = DSMF_DSMF_YVC22_23_B_REPLACE_VAL(coeff_lut_start[6],  coeff_values[13] & DSMF_DSMF_YVC22_23_A_MASK);

    /* Now we have a single register (not paired). Go figure. */
    coeff_lut_start[7]  = DSMF_DSMF_YVC24_A_REPLACE_VAL(   coeff_lut_start[7],  coeff_values[14] & DSMF_DSMF_YVC24_A_MASK);

    /* Final 5 paired registers: 30_31, 32_33, 34_40, 41_42, 43_44 */
    coeff_lut_start[8]  = DSMF_DSMF_YVC30_31_A_REPLACE_VAL(coeff_lut_start[8],  coeff_values[15] & DSMF_DSMF_YVC30_31_A_MASK);
    coeff_lut_start[8]  = DSMF_DSMF_YVC30_31_B_REPLACE_VAL(coeff_lut_start[8],  coeff_values[16] & DSMF_DSMF_YVC30_31_A_MASK);
    coeff_lut_start[9]  = DSMF_DSMF_YVC32_33_A_REPLACE_VAL(coeff_lut_start[9],  coeff_values[17] & DSMF_DSMF_YVC32_33_A_MASK);
    coeff_lut_start[9]  = DSMF_DSMF_YVC32_33_B_REPLACE_VAL(coeff_lut_start[9],  coeff_values[18] & DSMF_DSMF_YVC32_33_A_MASK);
    coeff_lut_start[10] = DSMF_DSMF_YVC34_40_A_REPLACE_VAL(coeff_lut_start[10], coeff_values[19] & DSMF_DSMF_YVC34_40_A_MASK);
    coeff_lut_start[10] = DSMF_DSMF_YVC34_40_B_REPLACE_VAL(coeff_lut_start[10], coeff_values[20] & DSMF_DSMF_YVC34_40_A_MASK);
    coeff_lut_start[11] = DSMF_DSMF_YVC41_42_A_REPLACE_VAL(coeff_lut_start[11], coeff_values[21] & DSMF_DSMF_YVC41_42_A_MASK);
    coeff_lut_start[11] = DSMF_DSMF_YVC41_42_B_REPLACE_VAL(coeff_lut_start[11], coeff_values[22] & DSMF_DSMF_YVC41_42_A_MASK);
    coeff_lut_start[12] = DSMF_DSMF_YVC43_44_A_REPLACE_VAL(coeff_lut_start[12], coeff_values[23] & DSMF_DSMF_YVC43_44_A_MASK);
    coeff_lut_start[12] = DSMF_DSMF_YVC43_44_B_REPLACE_VAL(coeff_lut_start[12], coeff_values[24] &
                                (DSMF_DSMF_YVC43_44_B_MASK >> DSMF_DSMF_YVC43_44_B_SHIFT)); // weird 13 bit pixel. All others are 11

}

void pie_dsmf_set_varcoeff_y(struct pie_handle_t *pie_handle, uint32_t filter_number, const int32_t * y_coefficients, uint32_t length)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    switch(filter_number)
    {
        case 0:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_YVC00_01, y_coefficients, length );
            break;
        case 1:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_YVC00_01_F1, y_coefficients, length );
            break;
        case 2:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_YVC00_01_F2, y_coefficients, length );
            break;
        case 3:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_YVC00_01_F3, y_coefficients, length );
            break;
        default:
            print("%s: Invalid filter_number (%d)!\n", __FUNCTION__, filter_number);
            break;
    }
}
EXPORT_SYMBOL(pie_dsmf_set_varcoeff_y);

void pie_dsmf_set_varcoeff_crcb(struct pie_handle_t *pie_handle, uint32_t filter_number, const int32_t * crcb_coefficients, uint32_t length)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    switch(filter_number)
    {
        case 0:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_CVC00_01, crcb_coefficients, length );
            break;
        case 1:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_CVC00_01_F1, crcb_coefficients, length );
            break;
        case 2:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_CVC00_01_F2, crcb_coefficients, length );
            break;
        case 3:
            pie_dsmf_set_varcoeff( &pie_handle->pie_dsmf->DSMF_CVC00_01_F3, crcb_coefficients, length );
            break;
        default:
            print("%s: Invalid filter_number (%d)!\n", __FUNCTION__, filter_number);
            break;
    }
}
EXPORT_SYMBOL(pie_dsmf_set_varcoeff_crcb);

void pie_dsmf_set_filt_mult(struct pie_handle_t *pie_handle, uint32_t filter_number, uint32_t y_mult, uint32_t crcb_mult)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    switch(filter_number)
    {
        case 0:
            pie_handle->pie_dsmf->DSMF_FILT_MULT =
                DSMF_DSMF_FILT_MULT_Y_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT, y_mult);
            pie_handle->pie_dsmf->DSMF_FILT_MULT =
                DSMF_DSMF_FILT_MULT_C_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT, crcb_mult);
            break;
        case 1:
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F1 =
                DSMF_DSMF_FILT_MULT_F1_Y_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F1, y_mult);
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F1 =
                DSMF_DSMF_FILT_MULT_F1_C_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F1, crcb_mult);
            break;
        case 2:
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F2 =
                DSMF_DSMF_FILT_MULT_F2_Y_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F2, y_mult);
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F2 =
                DSMF_DSMF_FILT_MULT_F2_C_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F2, crcb_mult);
            break;
        case 3:
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F3 =
                DSMF_DSMF_FILT_MULT_F3_Y_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F3, y_mult);
            pie_handle->pie_dsmf->DSMF_FILT_MULT_F3 =
                DSMF_DSMF_FILT_MULT_F3_C_MULT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_MULT_F3, crcb_mult);
            break;
        default:
            print("%s: Invalid filter_number (%d)!\n", __FUNCTION__, filter_number);
            break;
    }
}
EXPORT_SYMBOL(pie_dsmf_set_filt_mult);

void pie_dsmf_set_filt_shift(struct pie_handle_t *pie_handle, uint32_t filter_number, uint32_t y_shift, uint32_t crcb_shift)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    switch(filter_number)
    {
        case 0:
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT =
                DSMF_DSMF_FILT_SHIFT_Y_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT, y_shift);
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT =
                DSMF_DSMF_FILT_SHIFT_C_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT, crcb_shift);
            break;
        case 1:
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1 =
                DSMF_DSMF_FILT_SHIFT_F1_Y_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1, y_shift);
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1 =
                DSMF_DSMF_FILT_SHIFT_F1_C_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F1, crcb_shift);
            break;
        case 2:
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2 =
                DSMF_DSMF_FILT_SHIFT_F2_Y_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2, y_shift);
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2 =
                DSMF_DSMF_FILT_SHIFT_F2_C_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F2, crcb_shift);
            break;
        case 3:
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3 =
                DSMF_DSMF_FILT_SHIFT_F3_Y_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3, y_shift);
            pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3 =
                DSMF_DSMF_FILT_SHIFT_F3_C_SHIFT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_FILT_SHIFT_F3, crcb_shift);
            break;
        default:
            print("%s: Invalid filter_number (%d)!\n", __FUNCTION__, filter_number);
            break;
    }
}
EXPORT_SYMBOL(pie_dsmf_set_filt_shift);

void pie_dsmf_set_proj_thresh(struct pie_handle_t *pie_handle, uint16_t horiz, uint16_t vert, uint16_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_HV_Proj =
        DSMF_DSMF_SEG_HV_PROJ_THRESH_PROJ_HORIZ_X55_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_HV_Proj, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_HV_Proj =
        DSMF_DSMF_SEG_HV_PROJ_THRESH_PROJ_VERT_X55_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_HV_Proj, vert);
    pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj =
        DSMF_DSMF_SEG_DIAG_PROJ_THRESH_PROJ_DIAG_X25_REPLACE_VAL (pie_handle->pie_dsmf->DSMF_Seg_Diag_Proj, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_proj_thresh);

void pie_dsmf_set_isolated_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_Isolated =
        DSMF_DSMF_SEG_ISOLATED_THRESH_ISOLATED_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Isolated, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_Isolated =
        DSMF_DSMF_SEG_ISOLATED_THRESH_ISOLATED_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Isolated, vert);
    pie_handle->pie_dsmf->DSMF_Seg_Isolated =
        DSMF_DSMF_SEG_ISOLATED_THRESH_ISOLATED_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Isolated, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_isolated_thresh);

void pie_dsmf_set_3x3text_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_3x3_Text =
        DSMF_DSMF_SEG_3X3_TEXT_THRESH_3X3_TEXT_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Text, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Text =
        DSMF_DSMF_SEG_3X3_TEXT_THRESH_3X3_TEXT_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Text, vert);
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Text =
        DSMF_DSMF_SEG_3X3_TEXT_THRESH_3X3_TEXT_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Text, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_3x3text_thresh);

void pie_dsmf_set_3x3mixed_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed =
        DSMF_DSMF_SEG_3X3_MIXED_THRESH_3X3_MIXED_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed =
        DSMF_DSMF_SEG_3X3_MIXED_THRESH_3X3_MIXED_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed, vert);
    pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed =
        DSMF_DSMF_SEG_3X3_MIXED_THRESH_3X3_MIXED_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_3x3_Mixed, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_3x3mixed_thresh);

void pie_dsmf_set_5x5text_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_5x5_Text =
        DSMF_DSMF_SEG_5X5_TEXT_THRESH_5X5_TEXT_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Text, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Text =
        DSMF_DSMF_SEG_5X5_TEXT_THRESH_5X5_TEXT_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Text, vert);
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Text =
        DSMF_DSMF_SEG_5X5_TEXT_THRESH_5X5_TEXT_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Text, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_5x5text_thresh);

void pie_dsmf_set_5x5mixed_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed =
        DSMF_DSMF_SEG_5X5_MIXED_THRESH_5X5_MIXED_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed =
        DSMF_DSMF_SEG_5X5_MIXED_THRESH_5X5_MIXED_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed, vert);
    pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed =
        DSMF_DSMF_SEG_5X5_MIXED_THRESH_5X5_MIXED_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_5x5_Mixed, diag);
}
EXPORT_SYMBOL(pie_dsmf_set_5x5mixed_thresh);

void pie_dsmf_set_line_detection_thresh(struct pie_handle_t *pie_handle, uint8_t horiz, uint8_t vert, uint8_t diag, uint8_t diag1, uint8_t diag2)
{
    PIE_PTRCHECK_NR(pie_dsmf);

    pie_handle->pie_dsmf->DSMF_Seg_Line =
        DSMF_DSMF_SEG_LINE_THRESH_LINE_HORIZ_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Line, horiz);
    pie_handle->pie_dsmf->DSMF_Seg_Line =
        DSMF_DSMF_SEG_LINE_THRESH_LINE_VERT_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Line, vert);
    pie_handle->pie_dsmf->DSMF_Seg_Line_Diag =
        DSMF_DSMF_SEG_LINE_DIAG_THRESH_LINE_DIAG_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Line_Diag, diag);
    pie_handle->pie_dsmf->DSMF_Seg_Line_Diag =
        DSMF_DSMF_SEG_LINE_DIAG_THRESH_LINE_DIAG1_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Line_Diag, diag1);
    pie_handle->pie_dsmf->DSMF_Seg_Line_Diag =
        DSMF_DSMF_SEG_LINE_DIAG_THRESH_LINE_DIAG2_REPLACE_VAL(pie_handle->pie_dsmf->DSMF_Seg_Line_Diag, diag2);
}
EXPORT_SYMBOL(pie_dsmf_set_line_detection_thresh);

/*
 * Get the number of pixels eaten by DSMF on a *single* border. For example, the 9x9 FIR filter eats
 * a total of 8 rows, but only 4 on each side.
 */
uint32_t pie_dsmf_get_eaten_border_pixels(struct pie_handle_t *pie_handle)
{
    //TODO -- uncomment this once we actually use the pie_handle
    //PIE_PTRCHECK(pie_dsmf, 0);

    /* TODO -- change this for other modes -- e.g. Average mode (7x7 => 6, 5x5 => 4, 3x3 => 2). See also
     * win_shrink modes, etc. */
    
    // for now, only doing 11x11 pogo window manager that eats 5 rows on top and 5 rows on bottom
    return 5;
}
EXPORT_SYMBOL(pie_dsmf_get_eaten_border_pixels);

////////////////////////////////////////////////////////////////////////
// pie distort 
////////////////////////////////////////////////////////////////////////

static void get_distort_dma_default(struct pie_handle_t *pie_handle, int instance);

static void get_distort_default(struct pie_handle_t *pie_handle)
{
    // All values from 88PA6270RB-Registers.pdf - February 9, 2015
    memset(pie_handle->pie_distort_top, 0, sizeof(*pie_handle->pie_distort_top));

    pie_handle->pie_distort_top->Cfg =
        DISTORT_TOP_DISTORT_CFG_BYPASS_REPLACE_VAL(pie_handle->pie_distort_top->Cfg, 1);

    pie_handle->pie_distort_top->QueueHsStatus =
        DISTORT_TOP_DISTORT_QUEUEHSSTATUS_QUEUEEMPTY_REPLACE_VAL(pie_handle->pie_distort_top->QueueHsStatus, 1);
    pie_handle->pie_distort_top->QueueHsStatus =
        DISTORT_TOP_DISTORT_QUEUEHSSTATUS_QUEUEDEPTH_REPLACE_VAL(pie_handle->pie_distort_top->QueueHsStatus, 8);

    pie_handle->pie_distort_top->REV0 = PIE_DISTORT_MAJ_REV0;
    pie_handle->pie_distort_top->REV1 = 0;

    get_distort_dma_default(pie_handle, 1);
    get_distort_dma_default(pie_handle, 2);
    get_distort_dma_default(pie_handle, 3);
}

static void pie_distort_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_distort_top);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_distort_top);

    memcpy(dest_pie_handle->pie_distort_top, src_pie_handle->pie_distort_top,
           sizeof(*src_pie_handle->pie_distort_top));
}

void pie_distort_set_filter_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Cfg = DISTORT_TOP_DISTORT_CFG_BYPASS_REPLACE_VAL(pie_handle->pie_distort_top->Cfg, bypass);
}
EXPORT_SYMBOL(pie_distort_set_filter_bypass);

bool pie_distort_get_filter_bypass(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK(pie_distort_top, true);

    if (DISTORT_TOP_DISTORT_CFG_BYPASS_MASK_SHIFT(pie_handle->pie_distort_top->Cfg) == 1)
        return true;
    else
        return false;
}
EXPORT_SYMBOL(pie_distort_get_filter_bypass);


void pie_distort_set_enable_colors(struct pie_handle_t *pie_handle, bool color1, bool color2, bool color3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Cfg =
        DISTORT_TOP_DISTORT_CFG_ENCOLOR1_REPLACE_VAL(pie_handle->pie_distort_top->Cfg, color1);
    pie_handle->pie_distort_top->Cfg =
        DISTORT_TOP_DISTORT_CFG_ENCOLOR2_REPLACE_VAL(pie_handle->pie_distort_top->Cfg, color2);
    pie_handle->pie_distort_top->Cfg =
        DISTORT_TOP_DISTORT_CFG_ENCOLOR3_REPLACE_VAL(pie_handle->pie_distort_top->Cfg, color3);
}
EXPORT_SYMBOL(pie_distort_set_enable_colors);

void pie_distort_set_strip_dimensions(struct pie_handle_t *pie_handle, unsigned int width, unsigned int height, unsigned int offset)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->OutputStripConfig =
        DISTORT_TOP_DISTORT_OUTPUTSTRIPCONFIG_STRIPWIDTH_REPLACE_VAL(pie_handle->pie_distort_top->OutputStripConfig, width);
    pie_handle->pie_distort_top->OutputStripConfig =
        DISTORT_TOP_DISTORT_OUTPUTSTRIPCONFIG_STRIPHEIGHT_REPLACE_VAL(pie_handle->pie_distort_top->OutputStripConfig, height);
    pie_handle->pie_distort_top->OutputStripConfig =
        DISTORT_TOP_DISTORT_OUTPUTSTRIPCONFIG_STRIPOFFSET_REPLACE_VAL(pie_handle->pie_distort_top->OutputStripConfig, offset);
}
EXPORT_SYMBOL(pie_distort_set_strip_dimensions);

void pie_distort_set_input_dimensions(struct pie_handle_t *pie_handle, unsigned int width_ch1, unsigned int height_ch1, unsigned int width_ch2, unsigned int height_ch2, unsigned int width_ch3, unsigned int height_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_ImageDimension =
        DISTORT_TOP_DISTORT_CH1_IMAGEDIMENSION_INWIDTH_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_ImageDimension, width_ch1);
    pie_handle->pie_distort_top->Ch1_ImageDimension =
        DISTORT_TOP_DISTORT_CH1_IMAGEDIMENSION_INHEIGHT_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_ImageDimension, height_ch1);
    pie_handle->pie_distort_top->Ch2_ImageDimension =
        DISTORT_TOP_DISTORT_CH2_IMAGEDIMENSION_INWIDTH_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_ImageDimension, width_ch2);
    pie_handle->pie_distort_top->Ch2_ImageDimension =
        DISTORT_TOP_DISTORT_CH2_IMAGEDIMENSION_INHEIGHT_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_ImageDimension, height_ch2);
    pie_handle->pie_distort_top->Ch3_ImageDimension =
        DISTORT_TOP_DISTORT_CH3_IMAGEDIMENSION_INWIDTH_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_ImageDimension, width_ch3);
    pie_handle->pie_distort_top->Ch3_ImageDimension =
        DISTORT_TOP_DISTORT_CH3_IMAGEDIMENSION_INHEIGHT_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_ImageDimension, height_ch3);
}
EXPORT_SYMBOL(pie_distort_set_input_dimensions);

void pie_distort_set_start_position_x(struct pie_handle_t *pie_handle, int whole_ch1, int fractional_ch1, int whole_ch2, int fractional_ch2, int whole_ch3, int fractional_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH1_DATASTARTPOS_X_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_DataStartPos_X, whole_ch1);
    pie_handle->pie_distort_top->Ch1_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH1_DATASTARTPOS_X_F_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_DataStartPos_X, fractional_ch1);
    pie_handle->pie_distort_top->Ch2_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH2_DATASTARTPOS_X_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_DataStartPos_X, whole_ch2);
    pie_handle->pie_distort_top->Ch2_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH2_DATASTARTPOS_X_F_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_DataStartPos_X, fractional_ch2);
    pie_handle->pie_distort_top->Ch3_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH3_DATASTARTPOS_X_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_DataStartPos_X, whole_ch3);
    pie_handle->pie_distort_top->Ch3_DataStartPos_X =
        DISTORT_TOP_DISTORT_CH3_DATASTARTPOS_X_F_POSXY_X_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_DataStartPos_X, fractional_ch3);
}
EXPORT_SYMBOL(pie_distort_set_start_position_x);

void pie_distort_set_start_position_y(struct pie_handle_t *pie_handle, int whole_ch1, int fractional_ch1, int whole_ch2, int fractional_ch2, int whole_ch3, int fractional_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH1_DATASTARTPOS_Y_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_DataStartPos_Y, whole_ch1);
    pie_handle->pie_distort_top->Ch1_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH1_DATASTARTPOS_Y_F_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_DataStartPos_Y, fractional_ch1);
    pie_handle->pie_distort_top->Ch2_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH2_DATASTARTPOS_Y_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_DataStartPos_Y, whole_ch2);
    pie_handle->pie_distort_top->Ch2_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH2_DATASTARTPOS_Y_F_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_DataStartPos_Y, fractional_ch2);
    pie_handle->pie_distort_top->Ch3_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH3_DATASTARTPOS_Y_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_DataStartPos_Y, whole_ch3);
    pie_handle->pie_distort_top->Ch3_DataStartPos_Y =
        DISTORT_TOP_DISTORT_CH3_DATASTARTPOS_Y_F_POSXY_Y_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_DataStartPos_Y, fractional_ch3);
}
EXPORT_SYMBOL(pie_distort_set_start_position_y);

void pie_distort_set_row_step_size_x(struct pie_handle_t *pie_handle, unsigned int step_ch1, unsigned int step_ch2, unsigned int step_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_FixedRowStepX =
        DISTORT_TOP_DISTORT_CH1_FIXEDROWSTEPX_ROWSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_FixedRowStepX, step_ch1);
    pie_handle->pie_distort_top->Ch2_FixedRowStepX =
        DISTORT_TOP_DISTORT_CH2_FIXEDROWSTEPX_ROWSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_FixedRowStepX, step_ch2);
    pie_handle->pie_distort_top->Ch3_FixedRowStepX =
        DISTORT_TOP_DISTORT_CH3_FIXEDROWSTEPX_ROWSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_FixedRowStepX, step_ch3);
}
EXPORT_SYMBOL(pie_distort_set_row_step_size_x);

void pie_distort_set_row_step_size_y(struct pie_handle_t *pie_handle, unsigned int step_ch1, unsigned int step_ch2, unsigned int step_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_FixedRowStepY =
        DISTORT_TOP_DISTORT_CH1_FIXEDROWSTEPY_ROWSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_FixedRowStepY, step_ch1);
    pie_handle->pie_distort_top->Ch2_FixedRowStepY =
        DISTORT_TOP_DISTORT_CH2_FIXEDROWSTEPY_ROWSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_FixedRowStepY, step_ch2);
    pie_handle->pie_distort_top->Ch3_FixedRowStepY =
        DISTORT_TOP_DISTORT_CH3_FIXEDROWSTEPY_ROWSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_FixedRowStepY, step_ch3);
}
EXPORT_SYMBOL(pie_distort_set_row_step_size_y);

void pie_distort_set_col_step_size_x(struct pie_handle_t *pie_handle, unsigned int step_ch1, unsigned int step_ch2, unsigned int step_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_FixedColStepX =
        DISTORT_TOP_DISTORT_CH1_FIXEDCOLSTEPX_COLSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_FixedColStepX, step_ch1);
    pie_handle->pie_distort_top->Ch2_FixedColStepX =
        DISTORT_TOP_DISTORT_CH2_FIXEDCOLSTEPX_COLSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_FixedColStepX, step_ch2);
    pie_handle->pie_distort_top->Ch3_FixedColStepX =
        DISTORT_TOP_DISTORT_CH3_FIXEDCOLSTEPX_COLSTEPX_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_FixedColStepX, step_ch3);
}
EXPORT_SYMBOL(pie_distort_set_col_step_size_x);

void pie_distort_set_col_step_size_y(struct pie_handle_t *pie_handle, unsigned int step_ch1, unsigned int step_ch2, unsigned int step_ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_FixedColStepY =
        DISTORT_TOP_DISTORT_CH1_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_FixedColStepY, step_ch1);
    pie_handle->pie_distort_top->Ch2_FixedColStepY =
        DISTORT_TOP_DISTORT_CH2_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_FixedColStepY, step_ch2);
    pie_handle->pie_distort_top->Ch3_FixedColStepY =
        DISTORT_TOP_DISTORT_CH3_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_FixedColStepY, step_ch3);
}
EXPORT_SYMBOL(pie_distort_set_col_step_size_y);

void pie_distort_set_blank_value(struct pie_handle_t *pie_handle, unsigned int ch1, unsigned int ch2, unsigned int ch3)
{
    PIE_PTRCHECK_NR(pie_distort_top);
    pie_handle->pie_distort_top->Ch1_BlankValue =
        DISTORT_TOP_DISTORT_CH1_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch1_BlankValue, ch1);
    pie_handle->pie_distort_top->Ch2_BlankValue =
        DISTORT_TOP_DISTORT_CH2_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch2_BlankValue, ch2);
    pie_handle->pie_distort_top->Ch3_BlankValue =
        DISTORT_TOP_DISTORT_CH3_FIXEDCOLSTEPY_COLSTEPY_REPLACE_VAL(pie_handle->pie_distort_top->Ch3_BlankValue, ch3);
}
EXPORT_SYMBOL(pie_distort_set_blank_value);

////////////////////////////////////////////////////////////////////////
// pie distort IDMA
////////////////////////////////////////////////////////////////////////

static DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *get_distort_dma(struct pie_handle_t *pie_handle, int instance)
{
    switch (instance) {
    case 1: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data0;
    case 2: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data1;
    case 3: return (DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *) pie_handle->pie_distort_antifcor_ddma_ac_data2;
    default: BUG();
    }
}

static void get_distort_dma_default(struct pie_handle_t *pie_handle, int instance)
{
    DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t *regs;
    regs = get_distort_dma(pie_handle, instance);

    // All values from 88PA6270RB-Registers.pdf - February 9, 2015
    memset(regs, 0, sizeof(*regs));

    regs->USR = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_USR_CLEAR_REPLACE_VAL(regs->USR, 1);
    regs->USR = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_USR_DATABUFEMPTY_REPLACE_VAL(regs->USR, 1);
    regs->USR = DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_USR_CMDBUFEMPTY_REPLACE_VAL(regs->USR, 1);

    regs->UTR1 = PIE_DISTORT_DMA_MAJ_REV;
}

static void pie_distort_dma_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_distort_antifcor_ddma_ac_data0);
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_distort_antifcor_ddma_ac_data1);
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_distort_antifcor_ddma_ac_data2);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_distort_antifcor_ddma_ac_data0);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_distort_antifcor_ddma_ac_data1);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_distort_antifcor_ddma_ac_data2);

    memcpy(dest_pie_handle->pie_distort_antifcor_ddma_ac_data0, src_pie_handle->pie_distort_antifcor_ddma_ac_data0,
           sizeof(*src_pie_handle->pie_distort_antifcor_ddma_ac_data0));
    memcpy(dest_pie_handle->pie_distort_antifcor_ddma_ac_data1, src_pie_handle->pie_distort_antifcor_ddma_ac_data1,
           sizeof(*src_pie_handle->pie_distort_antifcor_ddma_ac_data1));
    memcpy(dest_pie_handle->pie_distort_antifcor_ddma_ac_data2, src_pie_handle->pie_distort_antifcor_ddma_ac_data2,
           sizeof(*src_pie_handle->pie_distort_antifcor_ddma_ac_data2));
}

// note this subblock doesn't yet have a dump function

void pie_distort_idma_set_configuration(struct pie_handle_t *pie_handle,
                                        int instance, bool enable, int beats,
                                        int ownpolarity, int ownwritedisable,
                                        int outputselect, int descawcache,
                                        int descarcache, int dataaxcache)
{
    switch (instance) {
    case 0:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data0);
        pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR, enable);
        pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UCR_BEATS_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR, beats);
        pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR, ownpolarity);
        pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UCR_OUTPUTSELECT_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data0->UCR, outputselect);
        break;
    case 1:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data1);
        pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA1_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR, enable);
        pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA1_UCR_BEATS_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR, beats);
        pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA1_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR, ownpolarity);
        pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA1_UCR_OUTPUTSELECT_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data1->UCR, outputselect);
        break;
    case 2:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data2);
        pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA2_UCR_ENABLE_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR, enable);
        pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA2_UCR_BEATS_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR, beats);
        pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA2_UCR_OWNPOLARITY_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR, ownpolarity);
        pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA2_UCR_OUTPUTSELECT_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data2->UCR, outputselect);
        break;
    default:
        error_print("%s: bad instance=%d\n", __func__, instance);
        return;
    }
}
EXPORT_SYMBOL(pie_distort_idma_set_configuration);

void pie_distort_idma_set_image_width(struct pie_handle_t *pie_handle, int instance, unsigned int line_width_bytes)
{
    switch (instance) {
    case 0:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data0);
        pie_handle->pie_distort_antifcor_ddma_ac_data0->UIWR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_UIWR_WIDTH_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data0->UIWR, line_width_bytes);
        break;
    case 1:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data1);
        pie_handle->pie_distort_antifcor_ddma_ac_data1->UIWR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA1_UIWR_WIDTH_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data1->UIWR, line_width_bytes);
        break;
    case 2:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data2);
        pie_handle->pie_distort_antifcor_ddma_ac_data2->UIWR =
            DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA2_UIWR_WIDTH_REPLACE_VAL(pie_handle->pie_distort_antifcor_ddma_ac_data2->UIWR, line_width_bytes);
        break;
    default:
        error_print("%s: bad instance=%d\n", __func__, instance);
        return;
    }
}
EXPORT_SYMBOL(pie_distort_idma_set_image_width);

//
// pie distort idma interrupt functions
//

void pie_enable_distort_idma_irqs(struct pie_handle_t *pie_handle,
                                  struct idma_interrupt_info *irqstruct,
                                  int channel, bool enable)
{
    uint32_t *reg;
    uint32_t val;

    switch (channel) {
    case 0:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data0);
        reg = (uint32_t *) &pie_handle->pie_distort_antifcor_ddma_ac_data0->UIER;
        break;
    case 1:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data1);
        reg = (uint32_t *) &pie_handle->pie_distort_antifcor_ddma_ac_data1->UIER;
        break;
    case 2:
        PIE_PTRCHECK_NR(pie_distort_antifcor_ddma_ac_data2);
        reg = (uint32_t *) &pie_handle->pie_distort_antifcor_ddma_ac_data2->UIER;
        break;
    default:
        // this case should never happen
        error_print("%s: there is no such channel as %d\n", __func__, channel);
        BUG();
    }

    if (irqstruct == NULL)
    {
        val = enable ? 0xFFFFFFFF : 0;
    }
    else
    {
        val = pie_distort_convert_irqstruct_to_update_reg(irqstruct, *reg, enable);
    }
    *reg = val;
}
EXPORT_SYMBOL(pie_enable_distort_idma_irqs);


////////////////////////////////////////////////////////////////////////
// pie antiflare
////////////////////////////////////////////////////////////////////////

static void get_antifcor_default(struct pie_handle_t *pie_handle)
{
    // All values from 88PA6270RB-Registers.pdf - February 9, 2015
    memset(pie_handle->pie_distort_antifcor_top, 0, sizeof(*pie_handle->pie_distort_antifcor_top));

    pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG =
        DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_ANTIFCOR_CFG_BYPASS_ALL_REPLACE_VAL(pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG, 1);

    pie_handle->pie_distort_antifcor_top->REV0 = PIE_DISTORT_ANTIFCOR_MAJ_REV0;
    pie_handle->pie_distort_antifcor_top->REV1 = 0;
}

static void pie_antifcor_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_distort_antifcor_top);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_distort_antifcor_top);

    memcpy(dest_pie_handle->pie_distort_antifcor_top, src_pie_handle->pie_distort_antifcor_top,
           sizeof(*src_pie_handle->pie_distort_antifcor_top));
}

void pie_antifcor_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_distort_antifcor_top);
    pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG =
        DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_ANTIFCOR_CFG_BYPASS_ALL_REPLACE_VAL(pie_handle->pie_distort_antifcor_top->ANTIFCOR_CFG,
                                                                              bypass);
}
EXPORT_SYMBOL(pie_antifcor_set_bypass);

////////////////////////////////////////////////////////////////////////
// pie XYScale 
////////////////////////////////////////////////////////////////////////

#define PIE_XYSCALE_MAX_INPUT_LINES  (36)
#define PIE_XYSCALE_MAX_OUTPUT_LINES (-1)

static void get_xyscale_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = 0;
    regval = XYSCALE_CSR_RU_REPLACE_VAL(regval, 0);
    regval = XYSCALE_CSR_LM_REPLACE_VAL(regval, 1);
    regval = XYSCALE_CSR_B_REPLACE_VAL(regval, 1);
    pie_handle->pie_xyscale->CSR = regval;
    
    pie_handle->pie_xyscale->CCR = 0;
    pie_handle->pie_xyscale->CFR1 =  0;
    pie_handle->pie_xyscale->CFR2 = 0;
    pie_handle->pie_xyscale->HSR = 0;
    pie_handle->pie_xyscale->VSR = 0;
    pie_handle->pie_xyscale->HSMR = 0;
    pie_handle->pie_xyscale->VSMR = 0;
    pie_handle->pie_xyscale->XTR = 0;
    pie_handle->pie_xyscale->YTR = 0;
    pie_handle->pie_xyscale->CXR = 0;
    pie_handle->pie_xyscale->CYR = 0;
    pie_handle->pie_xyscale->VSPR = 0;
    pie_handle->pie_xyscale->HCPR = XYSCALE_HCPR_P_REPLACE_VAL(0, 0x10000);
    pie_handle->pie_xyscale->VCPR = XYSCALE_VCPR_P_REPLACE_VAL(0, 0x10000);
    pie_handle->pie_xyscale->XOR = 0; // deprecated register
    pie_handle->pie_xyscale->REV0 = PIE_XYSCALE_MAJ_REV0;
    pie_handle->pie_xyscale->REV1 = 0;

    // Mark an invalid value into the partial pixel position
    pie_handle->pie_xyscale_partial_pixel_pos = 0xFFFFFFFF;
}

static void pie_xyscale_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_xyscale);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_xyscale);

    dest_pie_handle->pie_xyscale->CSR = src_pie_handle->pie_xyscale->CSR;  
    dest_pie_handle->pie_xyscale->CCR = src_pie_handle->pie_xyscale->CCR;  
    dest_pie_handle->pie_xyscale->CFR1 = src_pie_handle->pie_xyscale->CFR1; 
    dest_pie_handle->pie_xyscale->CFR2 = src_pie_handle->pie_xyscale->CFR2;
    dest_pie_handle->pie_xyscale->HSR = src_pie_handle->pie_xyscale->HSR; 
    dest_pie_handle->pie_xyscale->VSR = src_pie_handle->pie_xyscale->VSR; 
    dest_pie_handle->pie_xyscale->HSMR = src_pie_handle->pie_xyscale->HSMR;
    dest_pie_handle->pie_xyscale->VSMR = src_pie_handle->pie_xyscale->VSMR;
    dest_pie_handle->pie_xyscale->XTR = src_pie_handle->pie_xyscale->XTR;
    dest_pie_handle->pie_xyscale->YTR = src_pie_handle->pie_xyscale->YTR;
    dest_pie_handle->pie_xyscale->CXR = src_pie_handle->pie_xyscale->CXR; 
    dest_pie_handle->pie_xyscale->CYR = src_pie_handle->pie_xyscale->CYR;
    dest_pie_handle->pie_xyscale->VSPR = src_pie_handle->pie_xyscale->VSPR; 
    dest_pie_handle->pie_xyscale->HCPR = src_pie_handle->pie_xyscale->HCPR;
    dest_pie_handle->pie_xyscale->VCPR = src_pie_handle->pie_xyscale->VCPR;    
    dest_pie_handle->pie_xyscale->XOR = src_pie_handle->pie_xyscale->XOR; 
    dest_pie_handle->pie_xyscale->REV0 = src_pie_handle->pie_xyscale->REV0;
    dest_pie_handle->pie_xyscale->REV1 = src_pie_handle->pie_xyscale->REV1;
}

void pie_xyscale_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE XYSCALER regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_xyscale);    

    print("CSR=0x%X\n",  pie_handle->pie_xyscale->CSR);
    print("CCR=0x%X\n",  pie_handle->pie_xyscale->CCR);
    print("CFR1=0x%X\n", pie_handle->pie_xyscale->CFR1);
    print("CFR2=0x%X\n",  pie_handle->pie_xyscale->CFR2);
    print("HSR=0x%X\n",   pie_handle->pie_xyscale->HSR);
    print("VSR=0x%X\n",   pie_handle->pie_xyscale->VSR);
    print("HSMR=0x%X\n",  pie_handle->pie_xyscale->HSMR);
    print("VSMR=0x%X\n",  pie_handle->pie_xyscale->VSMR);
    print("CXR=0x%X\n",   pie_handle->pie_xyscale->CXR);
    print("CYR=0x%X\n",   pie_handle->pie_xyscale->CYR);
    print("HCPR=0x%X\n",  pie_handle->pie_xyscale->HCPR);
    print("VCPR=0x%X\n",  pie_handle->pie_xyscale->VCPR);
    print("XOR=0x%X\n",   pie_handle->pie_xyscale->XOR);
    print("REV0=0x%X\n",  pie_handle->pie_xyscale->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_xyscale->REV1);
}
EXPORT_SYMBOL(pie_xyscale_dump_handle_regs);

void pie_xyscale_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    if (bypass == true)
        pie_handle->pie_xyscale->CSR = XYSCALE_CSR_B_REPLACE_VAL(pie_handle->pie_xyscale->CSR, PIE_SCALE_BYPASS);
     else
        pie_handle->pie_xyscale->CSR = XYSCALE_CSR_B_REPLACE_VAL(pie_handle->pie_xyscale->CSR, PIE_SCALE_ENABLE);

}
EXPORT_SYMBOL(pie_xyscale_set_bypass);

bool pie_xyscale_get_bypass(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK(pie_xyscale, true);

    if (XYSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xyscale->CSR) == PIE_SCALE_BYPASS)
        return true;
    else
        return false;
}
EXPORT_SYMBOL(pie_xyscale_get_bypass);

void pie_xyscale_get_fw_rev(struct pie_handle_t *pie_handle, uint32_t *version, uint32_t *config)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    // the rev0 register's major field is the only field in rev0 that f/w cares about
    *version = XYSCALE_REV0_MAJ_MASK_SHIFT(pie_handle->pie_xyscale->REV0);
    *config = XYSCALE_REV1_CFG_MASK_SHIFT(pie_handle->pie_xyscale->REV1);
}
EXPORT_SYMBOL(pie_xyscale_get_fw_rev);

// SET the CSR reg

void pie_xyscale_set_roundup(struct pie_handle_t *pie_handle, uint8_t ru)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->CSR = XYSCALE_CSR_RU_REPLACE_VAL(pie_handle->pie_xyscale->CSR, ru);    
}
EXPORT_SYMBOL(pie_xyscale_set_roundup);

void pie_xyscale_set_linear_mode(struct pie_handle_t *pie_handle, uint8_t lm)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->CSR = XYSCALE_CSR_LM_REPLACE_VAL(pie_handle->pie_xyscale->CSR, lm);
}
EXPORT_SYMBOL(pie_xyscale_set_linear_mode);

uint8_t pie_xyscale_get_linear_mode(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK(pie_xyscale, 0);

    return(XYSCALE_CSR_LM_MASK_SHIFT(pie_handle->pie_xyscale->CSR));
}
EXPORT_SYMBOL(pie_xyscale_get_linear_mode);

void pie_xyscale_set_column_cnt(struct pie_handle_t *pie_handle, uint32_t col_cnt)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->CCR = XYSCALE_CCR_CC_REPLACE_VAL(pie_handle->pie_xyscale->CCR, col_cnt);
}
EXPORT_SYMBOL(pie_xyscale_set_column_cnt);

void pie_xyscale_set_replicate_mode(struct pie_handle_t *pie_handle, uint8_t rm)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->CCR = XYSCALE_CCR_RM_REPLACE_VAL(pie_handle->pie_xyscale->CCR, rm);
}
EXPORT_SYMBOL(pie_xyscale_set_replicate_mode);

void pie_xyscale_set_fill_color(struct pie_handle_t *pie_handle, uint8_t fillcolor0, uint8_t fillcolor1,
                                uint8_t fillcolor2)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->CFR2 =  XYSCALE_CFR2_FILL0_REPLACE_VAL(pie_handle->pie_xyscale->CFR2,
                                                                     fillcolor0);
    pie_handle->pie_xyscale->CFR1 =  XYSCALE_CFR1_FILL1_REPLACE_VAL(pie_handle->pie_xyscale->CFR1,
                                                                     fillcolor1);
    pie_handle->pie_xyscale->CFR1 =  XYSCALE_CFR1_FILL2_REPLACE_VAL(pie_handle->pie_xyscale->CFR1,
                                                                     fillcolor2);
}
EXPORT_SYMBOL(pie_xyscale_set_fill_color);

void pie_xyscale_set_scale(struct pie_handle_t *pie_handle, uint32_t x_scale, uint32_t y_scale)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->HSR =  XYSCALE_HSR_S_REPLACE_VAL(pie_handle->pie_xyscale->HSR,
                                                               x_scale);
    pie_handle->pie_xyscale->VSR =  XYSCALE_VSR_S_REPLACE_VAL(pie_handle->pie_xyscale->VSR,
                                                               y_scale);
}
EXPORT_SYMBOL(pie_xyscale_set_scale);

void pie_xyscale_calculate_and_set_scale(struct pie_handle_t *pie_handle, float x_factor, float y_factor)
{
#ifndef __KERNEL__
    unsigned int x_scale, y_scale;
    unsigned int x_edge_pixels = 0, y_edge_pixels = 0;
    unsigned int x_minus_norm = 0, y_minus_norm = 0;
    int x_transpose = 0, y_transpose = 0;

    PIE_PTRCHECK_NR(pie_xyscale);

    if (pie_xyscale_get_linear_mode(pie_handle))
    {
        x_scale = 0x10000 / x_factor;
        y_scale = 0x10000 / y_factor;

        x_minus_norm = x_scale - 0x10000;
        y_minus_norm = y_scale - 0x10000;

        if (x_factor > 1.0)
        {
            x_edge_pixels = (unsigned int)((x_factor-1)/2);
            if ((float)x_edge_pixels < ((x_factor-1)/2))
            {
                x_edge_pixels++;
            }
        }
        if (y_factor > 1.0)
        {
            y_edge_pixels = (unsigned int)((y_factor-1)/2);
            if ((float)y_edge_pixels < ((y_factor-1)/2))
            {
                y_edge_pixels++;
            }
        }

        x_transpose = x_edge_pixels * x_scale - 0x10000/2 + x_scale/2;
        if (x_transpose < 0)
        {
            x_transpose = 0;
        }
        else if (x_transpose > 0xFFFFF)
        {
            x_transpose = 0xFFFFF;
        }

        y_transpose = y_edge_pixels * y_scale - 0x10000/2 + y_scale/2;
        if (y_transpose < 0)
        {
            y_transpose = 0;
        }
        else if (y_transpose > 0xFFFFF)
        {
            y_transpose = 0xFFFFF;
        }
    }
    else
    {
        x_scale = x_factor * 0x10000;
        y_scale = y_factor * 0x10000;
    }
    pie_handle->pie_xyscale->HSR =  XYSCALE_HSR_S_REPLACE_VAL(pie_handle->pie_xyscale->HSR,
                                                              x_scale);
    pie_handle->pie_xyscale->VSR =  XYSCALE_VSR_S_REPLACE_VAL(pie_handle->pie_xyscale->VSR,
                                                              y_scale);

    pie_handle->pie_xyscale->HSR =  XYSCALE_HSR_E_REPLACE_VAL(pie_handle->pie_xyscale->HSR,
                                                              x_edge_pixels);

    pie_handle->pie_xyscale->HSMR =  XYSCALE_HSMR_S_REPLACE_VAL(pie_handle->pie_xyscale->HSMR,
                                                                x_minus_norm);
    pie_handle->pie_xyscale->VSMR =  XYSCALE_VSMR_S_REPLACE_VAL(pie_handle->pie_xyscale->VSMR,
                                                                y_minus_norm);

    pie_handle->pie_xyscale->XTR =  XYSCALE_XTR_DX_REPLACE_VAL(pie_handle->pie_xyscale->XTR,
                                                               x_transpose);
    pie_handle->pie_xyscale->YTR =  XYSCALE_YTR_DY_REPLACE_VAL(pie_handle->pie_xyscale->YTR,
                                                               y_transpose);

    pie_handle->pie_xyscale->CXR =  XYSCALE_CXR_DX_REPLACE_VAL(pie_handle->pie_xyscale->CXR,
                                                               x_transpose);
    pie_handle->pie_xyscale->CYR =  XYSCALE_CYR_DY_REPLACE_VAL(pie_handle->pie_xyscale->CYR,
                                                               y_transpose);
#endif
}
EXPORT_SYMBOL(pie_xyscale_calculate_and_set_scale);

void pie_xyscale_set_edge_pixel_cnt(struct pie_handle_t *pie_handle, uint8_t edge_pixel_cnt)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->HSR =  XYSCALE_HSR_E_REPLACE_VAL(pie_handle->pie_xyscale->HSR,
                                                               edge_pixel_cnt);
}
EXPORT_SYMBOL(pie_xyscale_set_edge_pixel_cnt);

void pie_xyscale_set_minus_scale(struct pie_handle_t *pie_handle, uint32_t x_minus, uint32_t y_minus)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->HSMR =  XYSCALE_HSMR_S_REPLACE_VAL(pie_handle->pie_xyscale->HSMR,
                                                                 x_minus);
    pie_handle->pie_xyscale->VSMR =  XYSCALE_VSMR_S_REPLACE_VAL(pie_handle->pie_xyscale->VSMR,
                                                                 y_minus);
}
EXPORT_SYMBOL(pie_xyscale_set_minus_scale);

void pie_xyscale_set_transpose(struct pie_handle_t *pie_handle, uint32_t x_trans, uint32_t y_trans)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->XTR =  XYSCALE_XTR_DX_REPLACE_VAL(pie_handle->pie_xyscale->XTR,
                                                                x_trans);
    pie_handle->pie_xyscale->YTR =  XYSCALE_YTR_DY_REPLACE_VAL(pie_handle->pie_xyscale->YTR,
                                                                y_trans);
}
EXPORT_SYMBOL(pie_xyscale_set_transpose);

void pie_xyscale_set_y_percentage(struct pie_handle_t *pie_handle, uint32_t yper)
{
    PIE_PTRCHECK_NR(pie_xyscale);

    pie_handle->pie_xyscale->VSPR =  XYSCALE_VSPR_P_REPLACE_VAL(pie_handle->pie_xyscale->VSPR,
                                                                 yper);
}
EXPORT_SYMBOL(pie_xyscale_set_y_percentage);

void pie_xyscale_set_values(struct pie_handle_t *pie_handle, uint8_t ru, uint8_t lm, uint8_t bypass,
                            uint32_t col_cnt, uint8_t fillcolor0, uint8_t fillcolor1,
                            uint8_t fillcolor2, uint32_t x_scale, uint32_t y_scale,
                            uint32_t x_minus, uint32_t y_minus, uint32_t x_trans,
                            uint32_t y_trans, uint32_t yper, uint8_t replicate,
                            uint8_t edge_pix_cnt)
{
    pie_xyscale_set_roundup(pie_handle, ru);
    pie_xyscale_set_linear_mode(pie_handle, lm);
    pie_xyscale_set_bypass(pie_handle, bypass);
    pie_xyscale_set_column_cnt(pie_handle, col_cnt);
    pie_xyscale_set_fill_color(pie_handle, fillcolor0, fillcolor1, fillcolor2);
    pie_xyscale_set_scale(pie_handle, x_scale, y_scale);
    pie_xyscale_set_minus_scale(pie_handle, x_minus, y_minus);
    pie_xyscale_set_transpose(pie_handle, x_trans, y_trans);
    pie_xyscale_set_y_percentage(pie_handle, yper);
    pie_xyscale_set_replicate_mode(pie_handle, replicate);
    pie_xyscale_set_edge_pixel_cnt(pie_handle, edge_pix_cnt);
}
EXPORT_SYMBOL(pie_xyscale_set_values);

void pie_xyscale_update_write_only_regs(struct pie_handle_t *pie_handle)
{
    pie_handle->pie_xyscale->YTR = pie_handle->pie_xyscale->CYR;
    pie_handle->pie_xyscale->VSPR = pie_handle->pie_xyscale->VCPR;
}

int pie_xyscale_get_output_lines(struct pie_handle_t *pie_handle, unsigned int *out_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;
    unsigned int first_line_weight;
    unsigned int scale = 0x10000;
    unsigned int percent = 0;

    PIE_PTRCHECK(pie_xyscale, -1);

    lines_in = *out_lines;

    if (cap_at_max && (lines_in > PIE_XYSCALE_MAX_INPUT_LINES))
    {
        // Cap the input lines
        lines_in = PIE_XYSCALE_MAX_INPUT_LINES;
        retVal = -1;
    }

    if (pie_xyscale_get_bypass(pie_handle) == true)
    {
        lines_out = lines_in;
    }
    else if (pie_xyscale_get_linear_mode(pie_handle) == true)
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xyscale->CYR);

        lines_out = (0x10000 * (lines_in-1) - percent + scale - 1) / scale;

        lines_in = (scale * (lines_out) + percent)/ 0x10000 + 1;
        if (cap_at_max && (lines_in != *out_lines))
        {
            if (lines_in > *out_lines) 
            {
               lines_out--;
            }
            retVal = -1;
        }
    }
    else
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_VCPR_P_MASK_SHIFT(pie_handle->pie_xyscale->VCPR);

        first_line_weight = percent;
        if (first_line_weight == 0x10000)
        {
            first_line_weight = scale;
        }
        lines_out = (scale * (lines_in - 1) + first_line_weight) / 0x10000;

        lines_in = (0x10000 * lines_out + (scale - first_line_weight) + scale - 1) / scale;
        if (cap_at_max && (lines_in != *out_lines))
        {
            if (lines_in > *out_lines) 
            {
               lines_out--;
            }
            retVal = -1;
        }
    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_out > PIE_XYSCALE_MAX_OUTPUT_LINES))
    {
        lines_out = PIE_XYSCALE_MAX_OUTPUT_LINES;
        retVal = -1;
    }
    else if (lines_out < 1)
    {
        retVal = -1;
    }

    *out_lines = lines_out;

    //print("XY GetOut Lines IN:%d Out:%d Perc:%d VSR:%x ret:%d\n", lines_in, lines_out, percent, scale, retVal);
    return retVal;
}
EXPORT_SYMBOL(pie_xyscale_get_output_lines);

int pie_xyscale_get_input_lines(struct pie_handle_t *pie_handle, unsigned int *in_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;
    unsigned int first_line_weight;
    unsigned int scale=0x10000;
    unsigned int percent=0;

    PIE_PTRCHECK(pie_xyscale, -1);

    lines_out = *in_lines;

    if (cap_at_max && (lines_out > PIE_XYSCALE_MAX_OUTPUT_LINES))
    {
        // Cap the input lines
        lines_out = PIE_XYSCALE_MAX_OUTPUT_LINES;
        retVal = -1;
    }

    if (pie_xyscale_get_bypass(pie_handle) == true)
    {
        // Bypass is active, input width matches output width
        lines_in = lines_out;
    }
    else if (pie_xyscale_get_linear_mode(pie_handle) == true)
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xyscale->CYR);
        lines_in = (scale * (lines_out) + percent)/ 0x10000 + 1;

        lines_out = (0x10000 * (lines_in-1) - percent + scale - 1) / scale;
        if (cap_at_max && (lines_out != *in_lines))
        {
            if (lines_out > *in_lines)
            {
               lines_in--;
            }
            retVal = -1;
        }
    }
    else
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_VCPR_P_MASK_SHIFT(pie_handle->pie_xyscale->VCPR);

        first_line_weight = percent;
        if (first_line_weight == 0x10000)
        {
            first_line_weight = scale;
        }
        lines_in = (0x10000 * lines_out + (scale - first_line_weight) + scale - 1) / scale;

        lines_out = (scale * (lines_in - 1) + first_line_weight) / 0x10000;
        if (cap_at_max && (lines_out != *in_lines))
        {
            if (lines_out > *in_lines)
            {
               lines_in--;
            }
            retVal = -1;
        }
    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_in > PIE_XYSCALE_MAX_INPUT_LINES))
    {
        lines_in = PIE_XYSCALE_MAX_INPUT_LINES;
        retVal = -1;
    }
    else if (lines_in < 1)
    {
        retVal = -1;
    }

    *in_lines = lines_in;

    //print("XY GetIn Lines IN:%d Out:%d Perc:%d VSR:%x ret:%d\n", lines_in, lines_out, percent, scale, retVal);
    return retVal;
}
EXPORT_SYMBOL(pie_xyscale_get_input_lines);

int pie_xyscale_get_discard_lines(struct pie_handle_t *pie_handle, unsigned int *discard_lines)
{
    int retVal = 0;
    unsigned int discard = 0;
    unsigned int previous_discard;
    unsigned int first_line_weight;
    unsigned int scale=0x10000;
    unsigned int percent=0;
    unsigned int leftovers=0;

    PIE_PTRCHECK(pie_xyscale, -1);

    /*
     * This function is intended to be called to determine the number of lines that can be discarded based upon what
     * the SUBSEQUENT sub-block has discarded.  We will overwrite the discard lines and send back a return code.
     */

    previous_discard = *discard_lines;

    if (pie_xyscale_get_bypass(pie_handle) == true)
    {
        discard = previous_discard;
    }
    else if (pie_xyscale_get_linear_mode(pie_handle) == true)
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xyscale->CYR);

        discard = (scale * (previous_discard) + percent);

        leftovers = discard & 0xFFFF;
        discard /= 0x10000;

        // Store the partial pixel position for use during pie_get_current
        pie_handle->pie_xyscale_partial_pixel_pos = leftovers;
    }
    else
    {
        scale = XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR);
        percent = XYSCALE_VCPR_P_MASK_SHIFT(pie_handle->pie_xyscale->VCPR);

        first_line_weight = percent;
        if (first_line_weight == 0x10000)
        {
            first_line_weight = scale;
        }

        // Note that we don't round here like we do for input
        discard = (0x10000 * previous_discard + scale - first_line_weight)/scale;
        leftovers = discard * scale + first_line_weight;
        leftovers = leftovers & 0xFFFF;

        // Store the partial pixel position for use during pie_get_current
        pie_handle->pie_xyscale_partial_pixel_pos = leftovers;

    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    *discard_lines = discard;

    //print("XY GetDiscard Prev:%d Scale:%x Percent:%d Disc:%d Leftovers:%d ret:%d\n",
    //      previous_discard, scale, percent, discard, leftovers, retVal);

    return retVal;
}
EXPORT_SYMBOL(pie_xyscale_get_discard_lines);

int pie_xyscale_get_output_width(struct pie_handle_t *pie_handle, unsigned int *out_w)
{
    unsigned int in_w;
    unsigned int column_width;

    PIE_PTRCHECK(pie_xyscale, -1);

    in_w = *out_w;

    if (pie_xyscale_get_bypass(pie_handle) == true)
    {
        // Bypass is active, input width matches output width
        return 0;
    }

    if (pie_xyscale_get_linear_mode(pie_handle) == true)
    {
        *out_w = 0x10000 * in_w / XYSCALE_HSR_S_MASK_SHIFT(pie_handle->pie_xyscale->HSR);
    }
    else
    {
        *out_w = XYSCALE_HSR_S_MASK_SHIFT(pie_handle->pie_xyscale->HSR) * in_w / 0x10000;
    }

    column_width = XYSCALE_CCR_CC_MASK_SHIFT(pie_handle->pie_xyscale->CCR);

    if (column_width)
    {
        *out_w = column_width;
    }

    return 0;
}
EXPORT_SYMBOL(pie_xyscale_get_output_width);

////////////////////////////////////////////////////////////////////////
// pie XY C Scale 
////////////////////////////////////////////////////////////////////////
#define PIE_XYCSCALE_MAX_INPUT_LINES  (36)
#define PIE_XYCSCALE_MAX_OUTPUT_LINES (-1)

static void get_xy_c_scale_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = XYCSCALE_CSR_O_REPLACE_VAL(0, 0);
    regval = XYCSCALE_CSR_O_REPLACE_VAL(regval, 0);
    regval = XYCSCALE_CSR_B_REPLACE_VAL(regval, 1);    
    pie_handle->pie_xy_c_scale->CSR = regval;

    pie_handle->pie_xy_c_scale->CLCR = 0;
    pie_handle->pie_xy_c_scale->CCR = 0;
    pie_handle->pie_xy_c_scale->CFR1 = 0;
    pie_handle->pie_xy_c_scale->CFR2 = 0;
    pie_handle->pie_xy_c_scale->HSR = 0;
    pie_handle->pie_xy_c_scale->VSR = 0;
    pie_handle->pie_xy_c_scale->XTR = 0;
    pie_handle->pie_xy_c_scale->YTR = 0;
    pie_handle->pie_xy_c_scale->CXR = 0;
    pie_handle->pie_xy_c_scale->CYR = 0;
    pie_handle->pie_xy_c_scale->XOR = 0; // unknown, so we just set to 0
    pie_handle->pie_xy_c_scale->SR = 0;
    pie_handle->pie_xy_c_scale->REV0 = PIE_XY_C_SCALE_MAJ_REV0;
    pie_handle->pie_xy_c_scale->REV1 = 0;

    // Mark an invalid value into the partial pixel position
    pie_handle->pie_xy_c_scale_partial_pixel_pos = 0xFFFFFFFF;
}

static void pie_xy_c_scale_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_xy_c_scale);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_xy_c_scale);
    
    dest_pie_handle->pie_xy_c_scale->CSR = src_pie_handle->pie_xy_c_scale->CSR;
    dest_pie_handle->pie_xy_c_scale->CLCR = src_pie_handle->pie_xy_c_scale->CLCR;
    dest_pie_handle->pie_xy_c_scale->CCR = src_pie_handle->pie_xy_c_scale->CCR;  
    dest_pie_handle->pie_xy_c_scale->CFR1 = src_pie_handle->pie_xy_c_scale->CFR1; 
    dest_pie_handle->pie_xy_c_scale->CFR2 = src_pie_handle->pie_xy_c_scale->CFR2;
    dest_pie_handle->pie_xy_c_scale->HSR = src_pie_handle->pie_xy_c_scale->HSR; 
    dest_pie_handle->pie_xy_c_scale->VSR = src_pie_handle->pie_xy_c_scale->VSR;
    dest_pie_handle->pie_xy_c_scale->XTR = src_pie_handle->pie_xy_c_scale->XTR;
    dest_pie_handle->pie_xy_c_scale->YTR = src_pie_handle->pie_xy_c_scale->YTR;     
    dest_pie_handle->pie_xy_c_scale->CXR = src_pie_handle->pie_xy_c_scale->CXR; 
    dest_pie_handle->pie_xy_c_scale->CYR = src_pie_handle->pie_xy_c_scale->CYR; 
    dest_pie_handle->pie_xy_c_scale->XOR = src_pie_handle->pie_xy_c_scale->XOR; 
    dest_pie_handle->pie_xy_c_scale->SR = src_pie_handle->pie_xy_c_scale->SR;   
    dest_pie_handle->pie_xy_c_scale->REV0 = src_pie_handle->pie_xy_c_scale->REV0;
    dest_pie_handle->pie_xy_c_scale->REV1 = src_pie_handle->pie_xy_c_scale->REV1;
}

void pie_xy_c_scale_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE XY C SCALER regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_xy_c_scale);

    print("CSR=0x%X\n",  pie_handle->pie_xy_c_scale->CSR);
    print("CLCR=0x%X\n",  pie_handle->pie_xy_c_scale->CLCR);
    print("CCR=0x%X\n",  pie_handle->pie_xy_c_scale->CCR);    
    print("CFR1=0x%X\n", pie_handle->pie_xy_c_scale->CFR1);
    print("CFR2=0x%X\n",  pie_handle->pie_xy_c_scale->CFR2);
    print("HSR=0x%X\n",   pie_handle->pie_xy_c_scale->HSR);
    print("VSR=0x%X\n",   pie_handle->pie_xy_c_scale->VSR);
    print("CXR=0x%X\n",   pie_handle->pie_xy_c_scale->CXR);
    print("CYR=0x%X\n",   pie_handle->pie_xy_c_scale->CYR);
    print("XOR=0x%X\n",   pie_handle->pie_xy_c_scale->XOR);
    print("SR=0x%X\n",   pie_handle->pie_xy_c_scale->SR);    
    print("REV0=0x%X\n",  pie_handle->pie_xy_c_scale->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_xy_c_scale->REV1);
}
EXPORT_SYMBOL(pie_xy_c_scale_dump_handle_regs);

void pie_xy_c_scale_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);
    
    if (bypass == true)
        pie_handle->pie_xy_c_scale->CSR = XYCSCALE_CSR_B_REPLACE_VAL(pie_handle->pie_xy_c_scale->CSR, PIE_SCALE_BYPASS);
    else
        pie_handle->pie_xy_c_scale->CSR = XYCSCALE_CSR_B_REPLACE_VAL(pie_handle->pie_xy_c_scale->CSR, PIE_SCALE_ENABLE);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_bypass);

bool pie_xy_c_scale_get_bypass(struct pie_handle_t *pie_handle)
{
    PIE_PTRCHECK(pie_xy_c_scale, true);
    
    if (XYCSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xy_c_scale->CSR) == PIE_SCALE_BYPASS)
        return true;
    else
        return false;
}
EXPORT_SYMBOL(pie_xy_c_scale_get_bypass);

void pie_xy_c_scale_set_replicate_mode(struct pie_handle_t *pie_handle, uint8_t rm)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);

    pie_handle->pie_xy_c_scale->CCR = XYCSCALE_CCR_RM_REPLACE_VAL(pie_handle->pie_xy_c_scale->CCR, rm);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_replicate_mode);

void pie_xy_c_scale_set_scale(struct pie_handle_t *pie_handle, uint32_t x_scale, uint32_t y_scale)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);

    pie_handle->pie_xy_c_scale->HSR = XYCSCALE_HSR_S_REPLACE_VAL(pie_handle->pie_xy_c_scale->HSR, x_scale);
    pie_handle->pie_xy_c_scale->VSR = XYCSCALE_VSR_S_REPLACE_VAL(pie_handle->pie_xy_c_scale->VSR, y_scale);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_scale);

void pie_xy_c_scale_calculate_and_set_scale(struct pie_handle_t *pie_handle, float x_factor, float y_factor)
{
#ifndef __KERNEL__
    unsigned int x_scale, y_scale;
    unsigned int x_edge_pixels = 0, y_edge_pixels = 0;
    int x_transpose = 0, y_transpose = 0;

    PIE_PTRCHECK_NR(pie_xy_c_scale);

    x_scale = 0x10000 / x_factor;
    y_scale = 0x10000 / y_factor;

    if (x_factor > 1.0)
    {
        x_edge_pixels = (unsigned int)((x_factor-1)/2);
        if ((float)x_edge_pixels < ((x_factor-1)/2))
        {
            x_edge_pixels++;
        }
    }
    if (y_factor > 1.0)
    {
        y_edge_pixels = (unsigned int)((y_factor-1)/2);
        if ((float)y_edge_pixels < ((y_factor-1)/2))
        {
            y_edge_pixels++;
        }
    }

    x_transpose = 0;
    y_transpose = 0;

    pie_handle->pie_xy_c_scale->HSR =  XYCSCALE_HSR_S_REPLACE_VAL(pie_handle->pie_xyscale->HSR,
                                                                  x_scale);
    pie_handle->pie_xy_c_scale->VSR =  XYCSCALE_VSR_S_REPLACE_VAL(pie_handle->pie_xyscale->VSR,
                                                                  y_scale);

    pie_handle->pie_xy_c_scale->XTR =  XYCSCALE_XTR_DX_REPLACE_VAL(pie_handle->pie_xyscale->XTR,
                                                                   x_transpose);
    pie_handle->pie_xy_c_scale->YTR =  XYCSCALE_YTR_DY_REPLACE_VAL(pie_handle->pie_xyscale->YTR,
                                                                   y_transpose);

    pie_handle->pie_xy_c_scale->CXR =  XYCSCALE_CXR_DX_REPLACE_VAL(pie_handle->pie_xyscale->CXR,
                                                                   x_transpose);
    pie_handle->pie_xy_c_scale->CYR =  XYCSCALE_CYR_DY_REPLACE_VAL(pie_handle->pie_xyscale->CYR,
                                                                   y_transpose);
#endif
}
EXPORT_SYMBOL(pie_xy_c_scale_calculate_and_set_scale);

void pie_xy_c_scale_set_output_column_cnt(struct pie_handle_t *pie_handle, uint32_t pixels_out)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);
    
    pie_handle->pie_xy_c_scale->CCR = XYCSCALE_CCR_CC_REPLACE_VAL(pie_handle->pie_xy_c_scale->CCR, pixels_out);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_output_column_cnt);

void pie_xy_c_scale_set_fill_color(struct pie_handle_t *pie_handle, uint8_t fillcolor0, uint8_t fillcolor1,
                                   uint8_t fillcolor2)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);
    
    pie_handle->pie_xy_c_scale->CFR1 = XYCSCALE_CFR1_FILL2_REPLACE_VAL(pie_handle->pie_xy_c_scale->CFR1, fillcolor2);
    pie_handle->pie_xy_c_scale->CFR1 = XYCSCALE_CFR1_FILL1_REPLACE_VAL(pie_handle->pie_xy_c_scale->CFR1, fillcolor1);    
    pie_handle->pie_xy_c_scale->CFR2 = XYCSCALE_CFR2_FILL0_REPLACE_VAL(pie_handle->pie_xy_c_scale->CFR2, fillcolor0);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_fill_color);

void pie_xy_c_scale_set_transpose(struct pie_handle_t *pie_handle, uint32_t x_trans, uint32_t y_trans)
{
    PIE_PTRCHECK_NR(pie_xy_c_scale);
    
    pie_handle->pie_xy_c_scale->XTR =  XYCSCALE_XTR_DX_REPLACE_VAL(pie_handle->pie_xy_c_scale->XTR,
                                                                   x_trans);
    pie_handle->pie_xy_c_scale->YTR =  XYCSCALE_YTR_DY_REPLACE_VAL(pie_handle->pie_xy_c_scale->YTR,
                                                                   y_trans);
}
EXPORT_SYMBOL(pie_xy_c_scale_set_transpose);

void pie_xy_c_scale_update_write_only_regs(struct pie_handle_t *pie_handle)
{
    pie_handle->pie_xy_c_scale->YTR = pie_handle->pie_xy_c_scale->CYR;
}

int pie_xy_c_scale_get_output_lines(struct pie_handle_t *pie_handle, unsigned int *out_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;
    unsigned int scale = 0x10000;
    unsigned int percent = 0;

    PIE_PTRCHECK(pie_xy_c_scale, -1);

    lines_in = *out_lines;

    if (cap_at_max && (lines_in > PIE_XYCSCALE_MAX_INPUT_LINES))
    {
        // Cap the input lines
        lines_in = PIE_XYCSCALE_MAX_INPUT_LINES;
        retVal = -1;
    }

    if (pie_xy_c_scale_get_bypass(pie_handle) == true)
    {
        lines_out = lines_in;
    }
    else
    {
        scale = XYCSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->VSR);
        percent = XYCSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xy_c_scale->CYR);

        /*
         * Usable range yielded by input lines:
         *
         *    UR = (Number of lines in) - ( 3 lines lost to bicubic algo) = (lines_in-3)
         *
         * Space that will be made into output lines
         *
         *    InSpace = UR * NORMAL - ( percent of the first line already used )
         *
         * Number of output lines
         *
         *    OutLines = ceiling ( InSpace / scale )
         */
        lines_out = (0x10000 * (lines_in-3) - percent + scale - 1) / scale;

        lines_in = (scale * (lines_out) + percent)/ 0x10000 + 3;
        if (cap_at_max && (lines_in != *out_lines))
        {
            if (lines_in > *out_lines) 
            {
               lines_out--;
            }
            retVal = -1;
        }
    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_out > PIE_XYCSCALE_MAX_OUTPUT_LINES))
    {
        lines_out = PIE_XYCSCALE_MAX_OUTPUT_LINES;
        retVal = -1;
    }
    else if (lines_out < 1)
    {
        retVal = -1;
    }

    *out_lines = lines_out;
    //print("XYC GetOut Lines IN:%d Out:%d Perc:%d VSR:%x ret:%d\n", lines_in, lines_out, percent, scale, retVal);

    return retVal;
}
EXPORT_SYMBOL(pie_xy_c_scale_get_output_lines);

int pie_xy_c_scale_get_input_lines(struct pie_handle_t *pie_handle, unsigned int *in_lines, unsigned int cap_at_max)
{
    int retVal = 0;
    unsigned int lines_in;
    unsigned int lines_out;
    unsigned int scale=0x10000;
    unsigned int percent=0;

    PIE_PTRCHECK(pie_xy_c_scale, -1);

    lines_out = *in_lines;

    if (cap_at_max && (lines_out > PIE_XYCSCALE_MAX_OUTPUT_LINES))
    {
        // Cap the input lines
        lines_out = PIE_XYCSCALE_MAX_OUTPUT_LINES;
        retVal = -1;
    }

    if (pie_xy_c_scale_get_bypass(pie_handle) == true)
    {
        // Bypass is active, input width matches output width
        lines_in = lines_out;
    }
    else
    {
        scale = XYCSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->VSR);
        percent = XYCSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xy_c_scale->CYR);

        // Bicubic mode will cause three lines to be lost
        lines_in = (scale * (lines_out) + percent)/ 0x10000 + 3;

        lines_out = (0x10000 * (lines_in-3) - percent + scale - 1) / scale;
        if (cap_at_max && (lines_out != *in_lines))
        {
            if (lines_out > *in_lines)
            {
               lines_in--;
            }
            retVal = -1;
        }
    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    if (cap_at_max && (lines_in > PIE_XYCSCALE_MAX_INPUT_LINES))
    {
        lines_in = PIE_XYCSCALE_MAX_INPUT_LINES;
        retVal = -1;
    }
    else if (lines_in < 1)
    {
        retVal = -1;
    }

    *in_lines = lines_in;
    //print("XYC GetIn Lines IN:%d Out:%d Perc:%d VSR:%x ret:%d\n", lines_in, lines_out, percent, scale, retVal);

    return retVal;
}
EXPORT_SYMBOL(pie_xy_c_scale_get_input_lines);

int pie_xy_c_scale_get_discard_lines(struct pie_handle_t *pie_handle, unsigned int *discard_lines)
{
    int retVal = 0;
    unsigned int discard = 0;
    unsigned int previous_discard;
    unsigned int first_line_weight;
    unsigned int scale=0x10000;
    unsigned int percent=0;
    unsigned int leftovers=0;

    PIE_PTRCHECK(pie_xy_c_scale, -1);

    /*
     * This function is intended to be called to determine the number of lines that can be discarded based upon what
     * the SUBSEQUENT sub-block has discarded.  We will overwrite the discard lines and send back a return code.
     */

    previous_discard = *discard_lines;

    if (pie_xy_c_scale_get_bypass(pie_handle) == true)
    {
        discard = previous_discard;
    }
    else
    {
        scale = XYCSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->VSR);
        percent = XYCSCALE_CYR_DY_MASK_SHIFT(pie_handle->pie_xy_c_scale->CYR);

        first_line_weight = percent;

        discard = (scale * (previous_discard) + percent);

        leftovers = discard & 0xFFFF;
        discard /= 0x10000;

        // Store the partial pixel position for use during pie_get_current
        pie_handle->pie_xy_c_scale_partial_pixel_pos = leftovers;
    }

    // Generate the return code
    // Negative = this block cannot perform this operation in a single run
    // Zero = everything is ok
    *discard_lines = discard;

    //print("XYC GetDiscard Prev:%d Scale:%x Percent:%d Disc:%d Leftovers:%d ret:%d\n",
    //      previous_discard, scale, percent, discard, leftovers, retVal);

    return retVal;
}
EXPORT_SYMBOL(pie_xy_c_scale_get_discard_lines);

int pie_xy_c_scale_get_output_width(struct pie_handle_t *pie_handle, unsigned int *out_w)
{
    unsigned int in_w;
    unsigned int column_width;

    PIE_PTRCHECK(pie_xy_c_scale, -1);

    in_w = *out_w;

    if (pie_xy_c_scale_get_bypass(pie_handle) == true)
    {
        // Bypass is active, input width matches output width
        return 0;
    }

    *out_w = 0x10000 * (in_w-3) / XYCSCALE_HSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->HSR);

    column_width = XYCSCALE_CCR_CC_MASK_SHIFT(pie_handle->pie_xy_c_scale->CCR);

    if (column_width)
    {
        *out_w = column_width;
    }

    return 0;
}
EXPORT_SYMBOL(pie_xy_c_scale_get_output_width);

////////////////////////////////////////////////////////////////////////
// pie Colorshift 
////////////////////////////////////////////////////////////////////////

static void get_colorshift_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;
    
    // All values from the 88PA6270RA-Registers doc - September 3, 2014
    regval = 0;
    regval = COLORSHIFT_CCSCR_BYPASS_REPLACE_VAL(regval, 1);
    pie_handle->pie_colorshift->CCSCR   = regval;
    
    regval = 0;
    regval = COLORSHIFT_CCSMMR_BLUE_REPLACE_VAL(regval, 1);
    regval = COLORSHIFT_CCSMMR_GREEN_REPLACE_VAL(regval, 1);
    regval = COLORSHIFT_CCSMMR_RED_REPLACE_VAL(regval, 1);
    pie_handle->pie_colorshift->CCSMMR  = regval;

    pie_handle->pie_colorshift->CCSMWR  = 0;
    pie_handle->pie_colorshift->CCSLSSR = 0;

    regval = 0;
    regval = COLORSHIFT_CCSLSR_BLUELFSR_REPLACE_VAL(regval, 2);
    regval = COLORSHIFT_CCSLSR_GREENLFSR_REPLACE_VAL(regval, 4);
    regval = COLORSHIFT_CCSLSR_REDLFSR_REPLACE_VAL(regval, 8);
    pie_handle->pie_colorshift->CCSLSR  = regval;

    pie_handle->pie_colorshift->REV0 = PIE_COLORSHIFT_MAJ_REV0;
    pie_handle->pie_colorshift->REV1 = 0;
}

static void pie_colorshift_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle,  pie_colorshift);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_colorshift);
    
    dest_pie_handle->pie_colorshift->CCSCR   = src_pie_handle->pie_colorshift->CCSCR;
    dest_pie_handle->pie_colorshift->CCSMMR  = src_pie_handle->pie_colorshift->CCSMMR;
    dest_pie_handle->pie_colorshift->CCSMWR  = src_pie_handle->pie_colorshift->CCSMWR;
    dest_pie_handle->pie_colorshift->CCSLSSR = src_pie_handle->pie_colorshift->CCSLSSR;
    dest_pie_handle->pie_colorshift->CCSLSR  = src_pie_handle->pie_colorshift->CCSLSR;
    dest_pie_handle->pie_colorshift->REV0    = src_pie_handle->pie_colorshift->REV0;
    dest_pie_handle->pie_colorshift->REV1    = src_pie_handle->pie_colorshift->REV1;
}

void pie_colorshift_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE COLORSHIFT regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_colorshift);

    print("CCSCR   = 0x%X\n",  pie_handle->pie_colorshift->CCSCR);
    print("CCSMMR  = 0x%X\n",  pie_handle->pie_colorshift->CCSMMR);
    print("CCSMWR  = 0x%X\n",  pie_handle->pie_colorshift->CCSMWR);
    print("CCSLSSR = 0x%X\n",  pie_handle->pie_colorshift->CCSLSSR);
    print("CCSLSR  = 0x%X\n",  pie_handle->pie_colorshift->CCSLSR);
    print("REV0    = 0x%X\n",  pie_handle->pie_colorshift->REV0);
    print("REV1    = 0x%X\n",  pie_handle->pie_colorshift->REV1);
}
EXPORT_SYMBOL(pie_colorshift_dump_handle_regs);

void pie_colorshift_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_colorshift);
    
    if (bypass == true)
    {
        pie_handle->pie_colorshift->CCSCR = 
               COLORSHIFT_CCSCR_BYPASS_REPLACE_VAL(pie_handle->pie_colorshift->CCSCR, 1);
    }
    else
    {
        pie_handle->pie_colorshift->CCSCR = 
               COLORSHIFT_CCSCR_BYPASS_REPLACE_VAL(pie_handle->pie_colorshift->CCSCR, 0);
    }
}
EXPORT_SYMBOL(pie_colorshift_set_bypass);

void pie_colorshift_setup(struct pie_handle_t *pie_handle,
                          uint8_t round_mode, uint8_t line_mode,
                          uint8_t red_multmode,  uint8_t green_multmode,  uint8_t blue_multmode,
                          uint8_t red_shiftmult, uint8_t green_shiftmult, uint8_t blue_shiftmult)
{
    PIE_PTRCHECK_NR(pie_colorshift);

    // Verbose debug (don't leave enabled)
    //print("%s rmode=0x%x lmode=0x%x\n", __FUNCTION__, round_mode, line_mode);
    //print("%s rmult=0x%x gmult=0x%x bmult=0x%x\n", __FUNCTION__, red_multmode, green_multmode, blue_multmode);
    //print("%s rshft=0x%x gshft=0x%x bshft=0x%x\n", __FUNCTION__, red_shiftmult, green_shiftmult, blue_shiftmult);

    // Config register settings
    pie_handle->pie_colorshift->CCSCR = 
           COLORSHIFT_CCSCR_ROUND_REPLACE_VAL(pie_handle->pie_colorshift->CCSCR, round_mode);
    pie_handle->pie_colorshift->CCSCR = 
           COLORSHIFT_CCSCR_TWOLINE_REPLACE_VAL(pie_handle->pie_colorshift->CCSCR, line_mode);

    // Multiplier mode settings
    pie_handle->pie_colorshift->CCSMMR = 
           COLORSHIFT_CCSMMR_BLUE_REPLACE_VAL(pie_handle->pie_colorshift->CCSMMR, blue_multmode);
    pie_handle->pie_colorshift->CCSMMR = 
           COLORSHIFT_CCSMMR_GREEN_REPLACE_VAL(pie_handle->pie_colorshift->CCSMMR, green_multmode);
    pie_handle->pie_colorshift->CCSMMR = 
           COLORSHIFT_CCSMMR_RED_REPLACE_VAL(pie_handle->pie_colorshift->CCSMMR, red_multmode);

    // Multiplier weight settings
    pie_handle->pie_colorshift->CCSMWR = 
           COLORSHIFT_CCSMWR_BLUE_REPLACE_VAL(pie_handle->pie_colorshift->CCSMWR, blue_shiftmult);
    pie_handle->pie_colorshift->CCSMWR = 
           COLORSHIFT_CCSMWR_GREEN_REPLACE_VAL(pie_handle->pie_colorshift->CCSMWR, green_shiftmult);
    pie_handle->pie_colorshift->CCSMWR = 
           COLORSHIFT_CCSMWR_RED_REPLACE_VAL(pie_handle->pie_colorshift->CCSMWR, red_shiftmult);
}
EXPORT_SYMBOL(pie_colorshift_setup);

////////////////////////////////////////////////////////////////////////
// OTMarb - OT merge and replace background
////////////////////////////////////////////////////////////////////////
static void get_otmarb_default(struct pie_handle_t *pie_handle)
{
    uint32_t regval;

    // all values from the 88PA6270RA-Registers doc - July 11, 2014
    regval = OTMARB_CONFIG_BYPASS_REPLACE_VAL(0, 1);
    pie_handle->pie_otmarb->Config = regval;
    pie_handle->pie_otmarb->BGColor = 0;
    pie_handle->pie_otmarb->IntEnable = 0;
    pie_handle->pie_otmarb->IntAck = 0;
    pie_handle->pie_otmarb->IntStat = 0;
    pie_handle->pie_otmarb->IntTest = 0;    
    pie_handle->pie_otmarb->REV0 = PIE_OTMARB_MAJ_REV0;
    pie_handle->pie_otmarb->REV1 = 0;
}

static void pie_otmarb_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle)
{
    PIE_PTRCHECK_NAME_NR(src_pie_handle, pie_otmarb);
    PIE_PTRCHECK_NAME_NR(dest_pie_handle, pie_otmarb);

    // clone the callbacks
    dest_pie_handle->pie_otmarb_callback = src_pie_handle->pie_otmarb_callback;
    dest_pie_handle->pie_otmarb_callback_data = src_pie_handle->pie_otmarb_callback_data;

    // clone the registers
    dest_pie_handle->pie_otmarb->Config = src_pie_handle->pie_otmarb->Config;
    dest_pie_handle->pie_otmarb->BGColor = src_pie_handle->pie_otmarb->BGColor;
    dest_pie_handle->pie_otmarb->IntEnable = src_pie_handle->pie_otmarb->IntEnable;
    dest_pie_handle->pie_otmarb->IntAck = src_pie_handle->pie_otmarb->IntAck;
    dest_pie_handle->pie_otmarb->IntStat = src_pie_handle->pie_otmarb->IntStat;
    dest_pie_handle->pie_otmarb->IntTest = src_pie_handle->pie_otmarb->IntTest;    
    dest_pie_handle->pie_otmarb->REV0 = src_pie_handle->pie_otmarb->REV0;
    dest_pie_handle->pie_otmarb->REV1 = src_pie_handle->pie_otmarb->REV1;
}

void pie_otmarb_dump_handle_regs(struct pie_handle_t *pie_handle)
{
    print("PIE otmarb regs (pie_handle)\n");
    PIE_PTRCHECK_NR(pie_otmarb);

    print("Config=0x%X\n",  pie_handle->pie_otmarb->Config);
    print("BGColor=0x%X\n",  pie_handle->pie_otmarb->BGColor);
    print("IntEnable=0x%X\n",  pie_handle->pie_otmarb->IntEnable);
    print("IntAck=0x%X\n",  pie_handle->pie_otmarb->IntAck);
    print("IntStat=0x%X\n",  pie_handle->pie_otmarb->IntStat);
    print("IntTest=0x%X\n",  pie_handle->pie_otmarb->IntTest);    
    print("REV0=0x%X\n",  pie_handle->pie_otmarb->REV0);
    print("REV1=0x%X\n",  pie_handle->pie_otmarb->REV1);
}
EXPORT_SYMBOL(pie_otmarb_dump_handle_regs);

void pie_otmarb_set_bypass(struct pie_handle_t *pie_handle, bool bypass)
{
    PIE_PTRCHECK_NR(pie_otmarb);
    
    if (bypass == true)
    {
        pie_handle->pie_otmarb->Config = 
               OTMARB_CONFIG_BYPASS_REPLACE_VAL(pie_handle->pie_otmarb->Config, 1);
    }
    else
    {
        pie_handle->pie_otmarb->Config = 
               OTMARB_CONFIG_BYPASS_REPLACE_VAL(pie_handle->pie_otmarb->Config, 0);
    }
}
EXPORT_SYMBOL(pie_otmarb_set_bypass);

// enable background replacement for specific OT types - bit field
void pie_otmarb_set_background_enable(struct pie_handle_t *pie_handle, uint16_t ot_bitmask)
{
    PIE_PTRCHECK_NR(pie_otmarb);

    pie_handle->pie_otmarb->Config =
        OTMARB_CONFIG_BRENABLE_REPLACE_VAL(pie_handle->pie_otmarb->Config, ot_bitmask);
}
EXPORT_SYMBOL(pie_otmarb_set_background_enable);

// specify the 3 color channel replacement values that matched the OT enable (10 bits per channel)
void pie_otmarb_set_background_colors(struct pie_handle_t *pie_handle,
                                      uint16_t ch0, uint16_t ch1, uint16_t ch2)
{
    PIE_PTRCHECK_NR(pie_otmarb);

    pie_handle->pie_otmarb->BGColor =
        OTMARB_BGCOLOR_CH0_REPLACE_VAL(pie_handle->pie_otmarb->BGColor, ch0);
    pie_handle->pie_otmarb->BGColor =
        OTMARB_BGCOLOR_CH1_REPLACE_VAL(pie_handle->pie_otmarb->BGColor, ch1);
    pie_handle->pie_otmarb->BGColor =
        OTMARB_BGCOLOR_CH2_REPLACE_VAL(pie_handle->pie_otmarb->BGColor, ch2);
}
EXPORT_SYMBOL(pie_otmarb_set_background_colors);

//
// pie OTMarb interrupt functions
//

void pie_enable_otmarb_irqs(struct pie_handle_t *pie_handle,
                            struct otmarb_interrupt_info *irqstruct,
                            bool enable)
{
    struct otmarb_interrupt_info irq;

    PIE_PTRCHECK_NR(pie_otmarb);
    PIE_PTRCHECK_NR(pie_otmarb);

    if (irqstruct == NULL)
    {
        pie_otmarb_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    pie_handle->pie_otmarb->IntEnable =
        pie_otmarb_convert_irqstruct_to_update_reg(irqstruct,
                                                   pie_handle->pie_otmarb->IntEnable,
                                                   enable);
}
EXPORT_SYMBOL(pie_enable_otmarb_irqs);

void pie_register_otmarb_callback(struct pie_handle_t *pie_handle, void *callback_fcn, void *callback_data)
{
    BUG_ON(pie_handle == NULL);
    
    pie_handle->pie_otmarb_callback = callback_fcn;
    pie_handle->pie_otmarb_callback_data = callback_data;    
}
EXPORT_SYMBOL(pie_register_otmarb_callback);

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// end of subblock functions
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// pie high level functions
// These are the functions that are calling all of the subblock
// functions above.
////////////////////////////////////////////////////////////////////////

// get defaults for pogo (all the idmas, odma, pogoizer and depogoizer)
static void get_pogo_default(struct pie_handle_t *pie_handle)
{
    int i;

    get_pogoizer_default(pie_handle);
    get_depogoizer_default(pie_handle);
    for (i=0;i<NUM_PIE_POGO_IDMA_CHANNELS;i++)
    {
        get_pogo_idma_udma_default(pie_handle, i);
        get_pogo_idma_core_default(pie_handle, i);
    }
    get_pogo_odma_udma_default(pie_handle);
    get_pogo_odma_core_default(pie_handle);

    get_otpogoizer_default(pie_handle);
    get_otdepogoizer_default(pie_handle);    
    get_pogo_otidma_udma_default(pie_handle);
    get_pogo_otidma_core_default(pie_handle);
    get_pogo_otodma_udma_default(pie_handle);
    get_pogo_otodma_core_default(pie_handle);
}

static void pie_pogo_clone_regs(struct pie_handle_t *src_pie_handle,
                                struct pie_handle_t *dest_pie_handle, int num_idmas)
{
    int i;

    pie_pogoizer_clone_regs(src_pie_handle, dest_pie_handle);
    pie_depogoizer_clone_regs(src_pie_handle, dest_pie_handle);
    for (i=0;i<num_idmas;i++)
    {
        pie_input_dma_clone_regs(src_pie_handle, dest_pie_handle, i);
    }
    pie_output_dma_clone_regs(src_pie_handle, dest_pie_handle);

    pie_otpogoizer_clone_regs(src_pie_handle, dest_pie_handle);
    pie_otdepogoizer_clone_regs(src_pie_handle, dest_pie_handle);
    pie_ot_input_dma_clone_regs(src_pie_handle, dest_pie_handle);
    pie_ot_output_dma_clone_regs(src_pie_handle, dest_pie_handle);    
}

static void pie_pogo_dump_handle_regs(struct pie_handle_t *pie_handle, int num_idmas)
{
    int i;
    
    pie_pogoizer_dump_handle_regs(pie_handle);
    pie_depogoizer_dump_handle_regs(pie_handle);
    for (i=0;i<num_idmas;i++)
    {
        pie_input_dma_dump_handle_regs(pie_handle, i);
    }
    pie_output_dma_dump_handle_regs(pie_handle);
    pie_otpogoizer_dump_handle_regs(pie_handle);
    pie_otdepogoizer_dump_handle_regs(pie_handle);    
    pie_ot_input_dma_dump_handle_regs(pie_handle);
    pie_ot_output_dma_dump_handle_regs(pie_handle);
}    

struct pie_handle_t *pie_do_get_default(struct pie_handle_t *pie_handle)
{
    int i;
    
    get_pie_common_default(pie_handle);
    get_pogo_default(pie_handle);
    get_otmarb_default(pie_handle);
    get_xyscale_default(pie_handle);
    get_xy_c_scale_default(pie_handle);        
    get_colorshift_default(pie_handle);        
    get_rgb2ycc_default(pie_handle);        
    get_ycc2rgb_default(pie_handle);
    get_rgb2esrgb_default(pie_handle);            
    get_dsmf_default(pie_handle);            
    get_tcns_default(pie_handle);            
    get_sccsc_default(pie_handle);            
    get_cstats_default(pie_handle);            
    get_ngadjust_default(pie_handle);
    get_distort_default(pie_handle);
    // note: get_distort_default_ddma() is called from get_distort_default()
    get_antifcor_default(pie_handle);
    get_decim_default(pie_handle);
    get_clippad_default(pie_handle);
    for (i=0;i<NUM_BDE_CHANNELS;i++)
        get_bde_default(pie_handle, i);
    
    // as new subblocks are added, create their get_default function
    // and add the call here

    // init all of the interrupt callbacks and callback data
    // if a new callback is added, be sure to add space in the
    // pie_handle for it (pie_create_new_default_handle), as well
    // as adding it to the clone function.
    pie_handle->pie_common_callback = NULL;
    pie_handle->pie_common_callback_data = NULL;
    
    pie_handle->pie_odma_udma_callback = NULL;
    pie_handle->pie_odma_udma_callback_data = NULL;
    
    pie_handle->pie_odma_core_callback = NULL;
    pie_handle->pie_odma_core_callback_data = NULL;

    pie_handle->pie_idma_udma_callback = NULL;
    pie_handle->pie_idma_udma_callback_data = NULL;
    
    pie_handle->pie_idma_core_callback = NULL;
    pie_handle->pie_idma_core_callback_data = NULL;
    
    pie_handle->pie_otidma_udma_callback = NULL;
    pie_handle->pie_otidma_udma_callback_data = NULL;
    
    pie_handle->pie_otidma_core_callback = NULL;
    pie_handle->pie_otidma_core_callback_data = NULL;

    pie_handle->pie_otodma_udma_callback = NULL;
    pie_handle->pie_otodma_udma_callback_data = NULL;
    
    pie_handle->pie_otodma_core_callback = NULL;
    pie_handle->pie_otodma_core_callback_data = NULL;

    pie_handle->pie_otmarb_callback = NULL;
    pie_handle->pie_otmarb_callback_data = NULL;
    
    return pie_handle;
}
EXPORT_SYMBOL(pie_do_get_default);

// This function allocates the pie handle and returns a pointer to it
// When the caller is done with the pie handle in kernel space,
// calling pie_do_free_handle() must be done to avoid a memory leak.
// 
// For user space, just freeing the memory is all that needs to be done

struct pie_handle_t *pie_create_new_default_handle(void)
{
    struct pie_handle_t *pie_handle = NULL;
    int pie_handle_size;
    uint32_t *subblock_size_table;
    enum pie_full_subblock_list i;

    // create a table of the size of every subblock
    subblock_size_table = get_subblock_sizes();

    // We need to compute the size of the pie_handle to allocate it.
    // Start out with the size of the pie_handle structure
    pie_handle_size = sizeof(struct pie_handle_t);

    // Now add the size of the shadow registers
    // The shadow registers are in the pie_handle
    // (any non-existent ones will have a 0 entry)
    for (i=common_index; i<max_subblock_val; i++)
        pie_handle_size += subblock_size_table[i];

    pie_handle = allocate_memory(pie_handle_size, GFP_KERNEL);

    if (pie_handle == NULL)
    {
        error_print("Error - Could not allocate memory for pie handle (%d)\n", pie_handle_size);
        return NULL;
    }

    // write the number of bytes in the whole pie_handle table to the pie_handle
    pie_handle->pie_handle_size = pie_handle_size;
    
    // copy the subblock_size_table into the pie_handle - it's needed
    // to create the pointer list (required when moving to kernel space)
    for (i=common_index; i<max_subblock_val; i++)
        pie_handle->subblock_size_table[i] = subblock_size_table[i];

    fixup_shadow_pointers(pie_handle);

    free_memory(subblock_size_table);

    pie_do_get_default(pie_handle); // now put the default values in
    
    return pie_handle;
}
EXPORT_SYMBOL(pie_create_new_default_handle);

// pie_clone_regs will copy each subblock set of registers from
// the src pie_handle to the dest
void pie_clone_regs(struct pie_handle_t *src_pie_handle, struct pie_handle_t *dest_pie_handle, int num_idmas)
{
    int i;

    pie_common_clone_regs(src_pie_handle, dest_pie_handle);
    pie_pogo_clone_regs(src_pie_handle, dest_pie_handle, num_idmas);
    pie_otmarb_clone_regs(src_pie_handle, dest_pie_handle);
    pie_rgb2esrgb_clone_regs(src_pie_handle, dest_pie_handle);
    pie_esrgb2ycc_clone_regs(src_pie_handle, dest_pie_handle);
    pie_ycc2esrgb_clone_regs(src_pie_handle, dest_pie_handle);
    pie_tcns_clone_regs(src_pie_handle, dest_pie_handle);
    pie_dsmf_clone_regs(src_pie_handle, dest_pie_handle);
    pie_xyscale_clone_regs(src_pie_handle, dest_pie_handle);
    pie_xy_c_scale_clone_regs(src_pie_handle, dest_pie_handle);
    pie_colorshift_clone_regs(src_pie_handle, dest_pie_handle);
    pie_ngadjust_clone_regs(src_pie_handle, dest_pie_handle);
    pie_sccsc_clone_regs(src_pie_handle, dest_pie_handle);
    pie_cstats_clone_regs(src_pie_handle, dest_pie_handle);
    pie_distort_clone_regs(src_pie_handle, dest_pie_handle);
    pie_distort_dma_clone_regs(src_pie_handle, dest_pie_handle);
    pie_antifcor_clone_regs(src_pie_handle, dest_pie_handle);
    pie_decim_clone_regs(src_pie_handle, dest_pie_handle);
    pie_clippad_clone_regs(src_pie_handle, dest_pie_handle);
    for (i=0; i<num_idmas; i++)
        pie_bde_clone_regs(src_pie_handle, dest_pie_handle, i);

    dest_pie_handle->pie_xyscale_partial_pixel_pos = src_pie_handle->pie_xyscale_partial_pixel_pos;
    dest_pie_handle->pie_xyscale_partial_pixel_pos = src_pie_handle->pie_xyscale_partial_pixel_pos;
}
EXPORT_SYMBOL(pie_clone_regs);

// pie_dump_handle_regs will dump all the regs in the pie_handle (does
// NOT read the registers from hardware).
// NOTE: the caller should call pie_do_get_current on the pie_handle
// before calling this function if you want to get the latest hardware 
// register values.

void pie_dump_handle_regs(struct pie_handle_t *pie_handle, int num_idmas)
{
    int i;
    
    pie_common_dump_handle_regs(pie_handle);
    pie_pogo_dump_handle_regs(pie_handle, num_idmas);
    pie_otmarb_dump_handle_regs(pie_handle);
    pie_rgb2esrgb_dump_handle_regs(pie_handle);
    pie_esrgb2ycc_dump_handle_regs(pie_handle);
    pie_ycc2esrgb_dump_handle_regs(pie_handle);
    pie_tcns_dump_handle_regs(pie_handle);
    pie_sccsc_dump_handle_regs(pie_handle);
    pie_ngadjust_dump_handle_regs(pie_handle);
    pie_dsmf_dump_handle_regs(pie_handle);
    pie_xyscale_dump_handle_regs(pie_handle);
    pie_xy_c_scale_dump_handle_regs(pie_handle);
    pie_colorshift_dump_handle_regs(pie_handle);
    pie_cstats_dump_handle_regs(pie_handle);
    pie_decim_dump_handle_regs(pie_handle);
    pie_clippad_dump_handle_regs(pie_handle);
    for (i=0;i<NUM_BDE_CHANNELS;i++)
        pie_bde_dump_handle_regs(pie_handle, i);
    // note that we still need to call dump functions for
    // the distort, distort idma, and anitfcor
}
EXPORT_SYMBOL(pie_dump_handle_regs);


// set the bypass bits for various subblocks

void pie_bypass_all_pie_subblocks(struct pie_handle_t *pie_handle)
{
    // no bypass for pie_common
    // no real bypass for pogo blocks - some have enables...
    pie_otmarb_set_bypass(pie_handle, true);
    pie_xyscale_set_bypass(pie_handle, true);
    pie_xy_c_scale_set_bypass(pie_handle, true);
    pie_dsmf_set_bypass(pie_handle, true);
    pie_ycc2esrgb_set_bypass(pie_handle, true);
    pie_tcns_set_bypass(pie_handle, true);
    pie_esrgb2ycc_set_bypass(pie_handle, true);
    pie_rgb2esrgb_set_bypass(pie_handle, true);
    pie_ngadjust_set_bypass(pie_handle, true);
    pie_sccsc_set_bypass(pie_handle, true);
    pie_colorshift_set_bypass(pie_handle, true);
    // no real bypass for distort ddma - it has an enable bit
    // no real bypass for cstats - it has an enable bit instead.
    pie_antifcor_set_bypass(pie_handle, true);
    pie_decim_set_bypass(pie_handle, true);
    pie_clippad_set_bypass(pie_handle, true);
    pie_bde_set_bypass(pie_handle, true);
}
EXPORT_SYMBOL(pie_bypass_all_pie_subblocks);


////////////////////////////////////////////////////////////////////////
// pie helper functions
////////////////////////////////////////////////////////////////////////

// copy the read registers to their corresponding write registers
// Do this for all subblocks with write-only regs that are updated
// by read registers
void pie_update_write_only_regs(struct pie_handle_t *pie_handle)
{
    pie_xyscale_update_write_only_regs(pie_handle);
    pie_xy_c_scale_update_write_only_regs(pie_handle);
}
EXPORT_SYMBOL(pie_update_write_only_regs);

/**
 * Calculates output height and discard lines (if needed) from 
 * input height. 
 * 
 * This function will calculate the output height for a given 
 * number of input lines, taking into account each sub-blocks 
 * constraints for number of lines in and out. Both of these 
 * values are passed by reference, as either could be changed. 
 * Discard lines may be requested as well. 
 * 
 * Generally, we calculate number of output lines from an input 
 * line count that is passed in.  If return value is 0, we are 
 * done.  If an error has occurred, something in the pipe had to 
 * cap the output lines.  If this happens, we must work from the 
 * back to the front calculating input lines from output lines. 
 * This can fail again, so we must try calculating input lines 
 * from output lines one last time.  Once we know output lines, 
 * we can calculate discard lines. 
 * 
 * \return Error status.  Less than 0 indicates an error has 
 *         occurred.  Will also write values into in_lines, 
 *         out_lines, and discard_lines.
 */
int pie_get_max_output_lines(struct pie_handle_t *pie_handle, unsigned int *in_lines, unsigned int *out_lines, unsigned int *discard_lines)
{
    int retVal = 0;
    unsigned int max_out_lines = *out_lines;
    //unsigned int max_in_lines = *in_lines;
    retVal = pie_get_output_lines_from_input(pie_handle, *in_lines, out_lines, 1);

    //print("PieLines MaxIn:%d MaxOut:%d\n", max_in_lines, max_out_lines);
    if (*out_lines > max_out_lines)
    {
        *out_lines = max_out_lines;
        retVal = -1;
    }
    if (retVal != 0)
    {
        // Input lines generated too many output lines
        retVal = pie_get_input_lines_from_output(pie_handle, in_lines, *out_lines, 1);
        
        if (retVal != 0)
        {
            // Output lines generated too many input lines
            retVal = pie_get_output_lines_from_input(pie_handle, *in_lines, out_lines, 0);
            //if (retVal < 0) return retVal;
        }
    }
    if (discard_lines)
    {
        retVal += pie_get_discard_lines_from_output(pie_handle, *out_lines, discard_lines);
    }
    
    //print("PieLines MaxIn:%d MaxOut:%d In:%d Out:%d Disc:%d ret:%d\n\n", max_in_lines, max_out_lines, *in_lines, *out_lines, (discard_lines?*discard_lines:-1), retVal);
    return retVal;
}
EXPORT_SYMBOL(pie_get_max_output_lines);

/**
 * Calculates output height from input height 
 *  
 * This function will calculate the output height for a given 
 * number of input lines, taking into account each sub-blocks 
 * constraints for number of lines in and out. We work from the 
 * front of the pipe to the back, passing the output line count 
 * from each stage to be used as the input line count of the 
 * next. 
 *
 * This function can cap at maximum for each sub-block, or it
 * can not.  This is useful for calculating the whole page
 * height from a given number of input lines.
 *
 * \return Error status.  Less than 0 indicates an error has
 *         occurred.  Will also write value into out_lines.
 */
int pie_get_output_lines_from_input(struct pie_handle_t *pie_handle, const unsigned int in_lines, unsigned int *out_lines, 
                                    unsigned int cap_at_max)
{
    int retVal = 0;
    *out_lines = in_lines;

    //print("FIXME - partial stub %s: add rest of bypass functions\n", __func__);
    if (pie_common_get_distort_dma_enabled(pie_handle))
    {
        // Distort
        //retVal += pie_distort_dma_get_output_lines(pie_handle, out_lines, cap_at_max);
        //retVal += pie_bde_get_output_lines(pie_handle, out_lines, cap_at_max);
    }
    else
    {
        //retVal += pie_pogo_idma_get_output_lines(pie_handle, out_lines, cap_at_max);
        //retVal += pie_bde_get_output_lines(pie_handle, out_lines, cap_at_max);
        //retVal += pie_colorshift_get_output_lines(pie_handle, out_lines, cap_at_max);
    }
    //retVal += pie_rgb2esrgb_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_sccsc_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_esrgb2ycc_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_ngadjust_get_output_lines(pie_handle, out_lines, cap_at_max);

    // Config 2
    retVal += pie_xyscale_get_output_lines(pie_handle, out_lines, cap_at_max);
    retVal += pie_xy_c_scale_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_dsmf_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_tcns_get_output_lines(pie_handle, out_lines, cap_at_max);
    //retVal += pie_ycc2esrgb_get_output_lines(pie_handle, out_lines, cap_at_max);

    retVal += pie_depogoizer_get_output_lines(pie_handle, out_lines, cap_at_max);

    //print("PIE OutFromIn In:%d Out:%d Cap:%d Ret:%d\n", in_lines, *out_lines, cap_at_max, retVal);

    return retVal;
}
EXPORT_SYMBOL(pie_get_output_lines_from_input);

/**
 * Calculates input height from output height 
 *  
 * This function will calculate the input height for a given 
 * number of output lines, taking into account each sub-blocks 
 * constraints for number of lines in and out. We work from the 
 * back of the pipe to the front, passing the input line count 
 * from each stage to be used as the output line count of the 
 * next. 
 *
 * This function can cap at maximum for each sub-block.
 *
 * \return Error status.  Less than 0 indicates an error has
 *         occurred.  Will also write value into in_lines.
 */
int pie_get_input_lines_from_output(struct pie_handle_t *pie_handle, unsigned int *in_lines, const unsigned int out_lines, 
                                    unsigned int cap_at_max)
{
    int retVal = 0;
    *in_lines = out_lines;

    //print("FIXME - partial stub %s: add rest of bypass functions\n", __func__);
    retVal += pie_depogoizer_get_input_lines(pie_handle, in_lines, cap_at_max);

    //retVal += pie_ycc2esrgb_get_input_lines(pie_handle, in_lines, cap_at_max);
    //retVal += pie_tcns_get_input_lines(pie_handle, in_lines, cap_at_max);
    //retVal += pie_dsmf_get_input_lines(pie_handle, in_lines, cap_at_max);
    retVal += pie_xy_c_scale_get_input_lines(pie_handle, in_lines, cap_at_max);
    retVal += pie_xyscale_get_input_lines(pie_handle, in_lines, cap_at_max);
    // Config 2
    
    //retVal += pie_ngadjust_get_input_lines(pie_handle, in_lines, cap_at_max);
    //retVal += pie_esrgb2ycc_get_input_lines(pie_handle, in_lines, cap_at_max);
    //retVal += pie_sccsc_get_input_lines(pie_handle, in_lines, cap_at_max);
    //retVal += pie_rgb2esrgb_get_input_lines(pie_handle, in_lines, cap_at_max);

    if (pie_common_get_distort_dma_enabled(pie_handle))
    {
        // Distort
        //retVal += pie_bde_get_input_lines(pie_handle, in_lines, cap_at_max);
        //retVal += pie_distort_idma_get_input_lines(pie_handle, in_lines, cap_at_max);
    }
    else
    {
        //retVal += pie_colorshift_get_input_lines(pie_handle, in_lines, cap_at_max);
        //retVal += pie_bde_get_input_lines(pie_handle, in_lines, cap_at_max);
        //retVal += pie_pogo_idma_get_input_lines(pie_handle, in_lines, cap_at_max);
    }

    //print("PIE InFromOut In:%d Out:%d Cap:%d Ret:%d\n", *in_lines, out_lines, cap_at_max, retVal);
    return retVal;
}
EXPORT_SYMBOL(pie_get_input_lines_from_output);

/**
 * Calculates discard lines from output height 
 *  
 * This function will calculate the number of lines to discard 
 * for a given number of output lines.  This may vary based on 
 * partial pixel position of some of the sub-blocks. 
 *
 * \return Error status.  Less than 0 indicates an error has
 *         occurred.  Will also write value into discard_lines.
 */
int pie_get_discard_lines_from_output(struct pie_handle_t *pie_handle, const unsigned int out_lines, 
                                      unsigned int *discard_lines)
{
    int retVal = 0;
    *discard_lines = out_lines;

    //print("FIXME - partial stub %s: add rest of bypass functions\n", __func__);
    retVal += pie_depogoizer_get_discard_lines(pie_handle, discard_lines);

    //retVal += pie_ycc2esrgb_get_discard_lines(pie_handle, discard_lines);
    //retVal += pie_tcns_get_discard_lines(pie_handle, discard_lines);
    //retVal += pie_dsmf_get_discard_lines(pie_handle, discard_lines);
    retVal += pie_xy_c_scale_get_discard_lines(pie_handle, discard_lines);
    retVal += pie_xyscale_get_discard_lines(pie_handle, discard_lines);
    // Config 2

    //retVal += pie_ngadjust_get_discard_lines(pie_handle, discard_lines);
    //retVal += pie_esrgb2ycc_get_discard_lines(pie_handle, discard_lines);
    //retVal += pie_sccsc_get_discard_lines(pie_handle, discard_lines);
    //retVal += pie_rgb2esrgb_get_discard_lines(pie_handle, discard_lines);

    if (pie_common_get_distort_dma_enabled(pie_handle))
    {
        // Distort
        //retVal += pie_bde_get_discard_lines(pie_handle, discard_lines);
        //retVal += pie_distort_idma_get_discard_lines(pie_handle, discard_lines);
    }
    else
    {
        //retVal += pie_colorshift_get_discard_lines(pie_handle, discard_lines);
        //retVal += pie_bde_get_discard_lines(pie_handle, discard_lines);
        //retVal += pie_pogo_idma_get_discard_lines(pie_handle, discard_lines);
    }

    return retVal;
}
EXPORT_SYMBOL(pie_get_discard_lines_from_output);

/**
 * Calculates output width from input width 
 *  
 * This function will calculate the output width in pixels for a
 * given number of input pixels.  Scaling, clipping, and padding 
 * could change these values. 
 *
 * \return Error status.  Less than 0 indicates an error has
 *         occurred.  Will also write value into out_w.
 */
int pie_get_output_width(struct pie_handle_t *pie_handle, const unsigned int in_w, unsigned int *out_w)
{
    int retVal = 0;

    *out_w = in_w;
    //print("FIXME - partial stub %s: add rest of bypass functions\n", __func__);
    if (pie_common_get_distort_dma_enabled(pie_handle))
    {
        // Distort
        //retVal += pie_distort_dma_get_output_width(pie_handle, out_w);
        //retVal += pie_bde_get_output_width(pie_handle, out_w);
    }
    else
    {
        //retVal += pie_pogo_idma_get_output_width(pie_handle, out_w);
        //retVal += pie_bde_get_output_width(pie_handle, out_w);
        //retVal += pie_colorshift_get_output_width(pie_handle, out_w);
    }
    //retVal += pie_rgb2esrgb_get_output_width(pie_handle, out_w);
    //retVal += pie_sccsc_get_output_width(pie_handle, out_w);
    //retVal += pie_esrgb2ycc_get_output_width(pie_handle, out_w);
    //retVal += pie_ngadjust_get_output_width(pie_handle, out_w);

    // Config 2
    retVal += pie_xyscale_get_output_width(pie_handle, out_w);
    retVal += pie_xy_c_scale_get_output_width(pie_handle, out_w);
    //retVal += pie_dsmf_get_output_width(pie_handle, out_w);
    //retVal += pie_tcns_get_output_width(pie_handle, out_w);
    //retVal += pie_ycc2esrgb_get_output_width(pie_handle, out_w);

    //retVal += pie_depogoizer_get_output_width(pie_handle, out_w);

    return retVal;
}
EXPORT_SYMBOL(pie_get_output_width);

void pie_dump_conf(struct pie_handle_t *pie_handle, int instanceId)
{
#ifndef __KERNEL__
    int configure_xyscale = 0;
    int configure_xy_c_scale = 0;
    FILE *fout;
    char conf_filename[80];

    snprintf(conf_filename, 80, "/var/tmp/pie%d.conf", instanceId);
    fout = fopen(conf_filename,"w");

    if (pie_handle->pie_xyscale)
    {
        configure_xyscale = (XYSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xyscale->CSR) == PIE_SCALE_ENABLE);
    }
    if (pie_handle->pie_xy_c_scale)
    {
        configure_xy_c_scale = (XYCSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xy_c_scale->CSR) == PIE_SCALE_ENABLE);
    }
    fprintf(fout, "dripipe {\n  pipeline= {\"pogo\", ");
    if (configure_xyscale)
    {
        fprintf(fout, "\"XYScale\", ");
    }
    if (configure_xy_c_scale)
    {
        fprintf(fout, "\"XYCScale\", ");
    }
    fprintf(fout, "\"unpogo\"}\n}\n\n");

    if (configure_xyscale)
    {
        fprintf(fout, "XYScale {\n");
        fprintf(fout, "  bypass = %d\n", XYSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xyscale->CSR) == PIE_SCALE_BYPASS);
        fprintf(fout, "  linear-mode = %d\n", XYSCALE_CSR_LM_MASK_SHIFT(pie_handle->pie_xyscale->CSR));
        fprintf(fout, "  edge-pixel-count = %d\n", XYSCALE_HSR_E_MASK_SHIFT(pie_handle->pie_xyscale->HSR));
        fprintf(fout, "  output-column-count = %d\n", XYSCALE_CCR_CC_MASK_SHIFT(pie_handle->pie_xyscale->CCR));
        fprintf(fout, "  replicate-mode = %d\n", XYSCALE_CCR_RM_MASK_SHIFT(pie_handle->pie_xyscale->CCR));
        fprintf(fout, "  fill-data0 = %d\n", XYSCALE_CFR2_FILL0_MASK_SHIFT(pie_handle->pie_xyscale->CFR2));
        fprintf(fout, "  fill-data1 = %d\n", XYSCALE_CFR1_FILL1_MASK_SHIFT(pie_handle->pie_xyscale->CFR1));
        fprintf(fout, "  fill-data2 = %d\n", XYSCALE_CFR1_FILL2_MASK_SHIFT(pie_handle->pie_xyscale->CFR1));
        fprintf(fout, "  round-up = %d\n", XYSCALE_CSR_RU_MASK_SHIFT(pie_handle->pie_xyscale->CSR));
        fprintf(fout, "  scale-factorX = 0x%x\n", XYSCALE_HSR_S_MASK_SHIFT(pie_handle->pie_xyscale->HSR));
        fprintf(fout, "  scale-factorY = 0x%x\n", XYSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSR));
        fprintf(fout, "  transpose-dx = 0x%x\n", XYSCALE_XTR_DX_MASK_SHIFT(pie_handle->pie_xyscale->XTR));
        fprintf(fout, "  transpose-dy = 0x%x\n", XYSCALE_YTR_DY_MASK_SHIFT(pie_handle->pie_xyscale->YTR));
        fprintf(fout, "  scale-factor-minus-normalX = 0x%x\n", XYSCALE_HSMR_S_MASK_SHIFT(pie_handle->pie_xyscale->HSMR));
        fprintf(fout, "  scale-factor-minus-normalY = 0x%x\n", XYSCALE_VSMR_S_MASK_SHIFT(pie_handle->pie_xyscale->VSMR));
        fprintf(fout, "}\n");
    }
    if (configure_xy_c_scale)
    {
        fprintf(fout, "XYCScale {\n");
        fprintf(fout, "  bypass = %d\n", XYCSCALE_CSR_B_MASK_SHIFT(pie_handle->pie_xy_c_scale->CSR) == PIE_SCALE_BYPASS);
        fprintf(fout, "  output_width = %d\n", XYCSCALE_CCR_CC_MASK_SHIFT(pie_handle->pie_xy_c_scale->CCR));
        fprintf(fout, "  replicate_mode = %d\n", XYCSCALE_CCR_RM_MASK_SHIFT(pie_handle->pie_xy_c_scale->CCR));
        fprintf(fout, "  c0_fill = %d\n", XYCSCALE_CFR2_FILL0_MASK_SHIFT(pie_handle->pie_xy_c_scale->CFR2));
        fprintf(fout, "  c1_fill = %d\n", XYCSCALE_CFR1_FILL1_MASK_SHIFT(pie_handle->pie_xy_c_scale->CFR1));
        fprintf(fout, "  c2_fill = %d\n", XYCSCALE_CFR1_FILL2_MASK_SHIFT(pie_handle->pie_xy_c_scale->CFR1));
        fprintf(fout, "  x_scale_factor = 0x%x\n", XYCSCALE_HSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->HSR));
        fprintf(fout, "  y_scale_factor = 0x%x\n", XYCSCALE_VSR_S_MASK_SHIFT(pie_handle->pie_xy_c_scale->VSR));
        fprintf(fout, "  dx_initial = 0x%x\n", XYCSCALE_XTR_DX_MASK_SHIFT(pie_handle->pie_xy_c_scale->XTR));
        fprintf(fout, "  dy_initial = 0x%x\n", XYCSCALE_YTR_DY_MASK_SHIFT(pie_handle->pie_xy_c_scale->YTR));
        fprintf(fout, "}\n");
    }

    fclose(fout);
#endif
}
EXPORT_SYMBOL(pie_dump_conf);

//------------------------------------------------------------------------------
// info functions
//------------------------------------------------------------------------------
// specify the format of the output data, and get back the number
// of bytes per pixel

int pie_pogo_get_output_bytes_per_pixel(int pogo_fmt_type_out)
{
    switch(pogo_fmt_type_out)
    {
    case POGO_FMT_MONO:
        return 1;   // reminder - the Bytes per pixel is per channel for planar
        break;
    case POGO_FMT_RGB:
        return 3;
        break;
    case POGO_FMT_RGBX:
        return 4;
        break;
    case POGO_FMT_XRGB:
        return 4;
        break;
    case POGO_FMT_PLANAR:
        error_print("ERROR: output fmt type can not be Color Planar\n");
        return -1;
        break;
    default:
        error_print("ERROR!! BAD pogo_fmt_type %d\n",pogo_fmt_type_out);
        return -EINVAL;
    }
}
EXPORT_SYMBOL(pie_pogo_get_output_bytes_per_pixel);

// specify the format of input and output data, the number of bits per pixel
// for the input data, and get back the idma/odma bytes per pixel and number
// of idma channels you need to feed
// Bpp is bytes per pixel
// Note that Mono and Planar will return 2 or 1 - bytes per pixel per channel
// while RGB returns 3 - since all the bytes for a pixel are in the same channel
//  negative value returned in case of error
int pie_pogo_fmt_type_parms(int pogo_fmt_type_in, int pogo_fmt_type_out, int pogo_bpp_input,
                            int *idma_Bpp, int *odma_Bpp, int *num_idma_channels)
{
    *num_idma_channels = 1;
    
    switch(pogo_fmt_type_in)
    {
    case POGO_FMT_MONO:
        if (pogo_bpp_input == POGO_8BPP)
            *idma_Bpp = 1; // reminder - the Bytes per pixel is per channel for planar
        else
            *idma_Bpp = 2;
        break;
    case POGO_FMT_RGB:
        *idma_Bpp = 3;
        break;
    case POGO_FMT_RGBX:
        *idma_Bpp = 4;
        break;
    case POGO_FMT_XRGB:
        *idma_Bpp = 4;
        break;
    case POGO_FMT_PLANAR:
        if (pogo_bpp_input == POGO_8BPP)
            *idma_Bpp = 1;
        else
            *idma_Bpp = 2;  // reminder - the Bytes per pixel is per channel for planar
        *num_idma_channels = 3;
        break;
    default:
        error_print("ERROR!! BAD pogo_fmt_type %d\n",pogo_fmt_type_in);
        return -EINVAL;
    }
    *odma_Bpp = pie_pogo_get_output_bytes_per_pixel(pogo_fmt_type_out);

    return 0;
}
EXPORT_SYMBOL(pie_pogo_fmt_type_parms);

MODULE_LICENSE("GPL");
