/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/irqreturn.h> // needed for irqreturn_t and IRQ_HANDLED
#include <linux/spinlock.h>  // for spinlock_t
#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/delay.h>     // for msleep

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_if.h"
#include "pie_data.h"
#include "pie_driver.h"
#include "pie_common.h"
#include "pie_common_if.h"
#include "pie_driverlib_if.h"  // for print macros


//  NOTE THERE IS NO FUNCTION clear_pie_common_int - since the interrupt
//  register is READ only.  Each bit must be cleared at the source (odma, etc)

// enable each field that is set to true
void enable_pie_common_irq(pieCommonData *device_data,
                           struct pie_common_ints *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pie_common_ints irq;

    if (irqstruct == NULL)
    {
        pie_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = pieCommonRead(IENR);
    reg = pie_common_convert_irqstruct_to_update_reg(irqstruct, reg, true);
    pieCommonWrite(IENR, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

// disable each field that is set to true
void disable_pie_common_irq(pieCommonData *device_data,
                            struct pie_common_ints *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pie_common_ints irq;

    if (irqstruct == NULL)
    {
        pie_common_set_all_irqstruct( &irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = pieCommonRead(IENR);
    reg = pie_common_convert_irqstruct_to_update_reg(irqstruct, reg, false);
    pieCommonWrite(IENR, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

/*************************************
 * pie_platform_irq: interrupt handler
 * @irq:  irq number
 * @pdev: interrupt source
 *
 * This function returns IRQ_HANDLED if the IRQ has been handled
 * This is an ISR don't trace, use attribute interface instead
 **************************************/

///  NOTE - This function runs in interrupt context - no long operations allowed
irqreturn_t pie_platform_irq(int irq, void *pdev)
{
    uint32_t reg, intval;
    pieCommonData *device_data;
    struct pie_common_ints irqstruct;
    unsigned long flags;  // for IRQ protection macro    

    device_data = pdev;
    // check for the test register - f/w can use to cause an interrupt - clear it
    PROTECT_INTREG_ACCESS_IRQ;
    reg = pieCommonRead(IPR);
    UNPROTECT_INTREG_ACCESS_IRQ;    

    // Check to see if POGO Write DMA or OT Write DMA is causing the interrupt
    intval = PIE_COMMON_IPR_WDMAINTPEND_MASK_SHIFT(reg);
    intval |= PIE_COMMON_IPR_OT_WDMAINTPEND_MASK_SHIFT(reg);
    if (intval != 0)
    {
        // clear any pie ODMA interrupts 
        pie_handle_odma_irqs();
    }

    intval = POGO_RDMA_INT_MASK(reg);
    if (intval != 0)
    {
        // clear any pie IDMA interrupts
        pie_handle_idma_irqs();
    }

    if (reg & PIE_COMMON_IPR_DISTORTINTPEND_MASK)
    {
        pie_handle_distort_irqs();
    }

    if (reg & PIE_COMMON_IPR_AC_DATA_RDMA0INTPEND_MASK)
    {
        pie_handle_distort_idma_irqs(0);
    }

    if (reg & PIE_COMMON_IPR_AC_DATA_RDMA1INTPEND_MASK)
    {
        pie_handle_distort_idma_irqs(1);
    }

    if (reg & PIE_COMMON_IPR_AC_DATA_RDMA2INTPEND_MASK)
    {
        pie_handle_distort_idma_irqs(2);
    }

    intval = PIE_COMMON_IPR_OTMARBINTPEND_MASK_SHIFT(reg);
    if (intval != 0)
    {
        pie_handle_otmarb_irqs();
    }
    
    if (reg != 0)
    {
        pie_common_convert_uintarray_to_irqstruct(&irqstruct, reg);
        // now that we have a list of interrupts that were set,
        // execute the registered callback function
        if (device_data->interrupt_callback != NULL)
        {
            debug_print("calling callback with struct (reg=0x%X)\n",reg);
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
    return IRQ_HANDLED;
}

// caller defines which part of the a debug bus registers to read,
// select that, then read and return the debug value
uint32_t get_pie_common_a_debug(pieCommonData *device_data, int sel)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = pieCommonRead(CTL);
    reg = PIE_COMMON_CTL_DB_DEBUG_SEL_REPLACE_VAL(reg, sel);
    pieCommonWrite(CTL, reg);
    reg = pieCommonRead(DBG);
    UNPROTECT_REG_ACCESS;
    return reg;
}

////////////////////////////////////////////////
// dump all read and read/write regs

void dump_pie_common_debug_regs(pieCommonData *device_data, int sel)
{
    int upperbit, lowerbit;
    uint32_t reg;

    // use a_debug to select which debug registers we wish to print

    reg = get_pie_common_a_debug(device_data, sel);
    upperbit = 32*(sel+1) - 1;
    lowerbit = 32*(sel);
    print("DBG[%d:%d]=0x%X\n", upperbit, lowerbit, reg);
}

void dump_pie_common_regs(pieCommonData *device_data)
{
    print("IENR=0x%X\n",pieCommonRead(IENR));
    print("IPR=0x%X\n",pieCommonRead(IPR));
    print("CBIR=0x%X\n",pieCommonRead(CBIR));
    print("CTL=0x%X\n",pieCommonRead(CTL));
    print("REV0=0x%X\n",pieCommonRead(REV0));
    print("REV1=0x%X\n",pieCommonRead(REV1));
    dump_pie_common_debug_regs(device_data, 0); //print debug bus 31:0
    dump_pie_common_debug_regs(device_data, 1); //print debug bus 63:32
    dump_pie_common_debug_regs(device_data, 2); //print debug bus 95:64
    dump_pie_common_debug_regs(device_data, 3); //print debug bus 127:96
}

// END: dump all read and read/write regs
////////////////////////////////////////////////

static int pie_revcheck(pieCommonData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = PIE_COMMON_REV0_MAJ_MASK_SHIFT(pieCommonRead(REV0));
    pieh_rev = PIE_COMMON_REV0_MAJ_MASK_SHIFT(pie_handle->pie_common->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}


// set, then clear the reset bit
static void pie_reset(pieCommonData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = pieCommonRead(CBIR);
    reg = PIE_COMMON_CBIR_SOFTRESET_REPLACE_VAL(reg, 1);
    pieCommonWrite(CBIR, reg);
    UNPROTECT_REG_ACCESS;
    
    msleep(5);  // wait for the reset to occur

    PROTECT_REG_ACCESS;
    reg = pieCommonRead(CBIR);
    reg = PIE_COMMON_CBIR_SOFTRESET_REPLACE_VAL(reg, 0);
    pieCommonWrite(CBIR, reg);
    UNPROTECT_REG_ACCESS;
    
}

static void pie_configure(pieCommonData *device_data, struct pie_handle_t *pie_handle)
{
    device_data->interrupt_callback = pie_handle->pie_common_callback;
    device_data->interrupt_callback_data = pie_handle->pie_common_callback_data;
    pieCommonWrite(IENR, pie_handle->pie_common->IENR);
    // IPR is Read only
    pieCommonWrite(CBIR,  pie_handle->pie_common->CBIR);
    pieCommonWrite(CTL,  pie_handle->pie_common->CTL);
    // DBG, REV0 and REV1 are read only
}

static void pie_get_current(pieCommonData *device_data, struct pie_handle_t *pie_handle)
{
    pie_handle->pie_common->IENR = pieCommonRead(IENR);
    pie_handle->pie_common->IPR = pieCommonRead(IPR);
    pie_handle->pie_common->CBIR = pieCommonRead(CBIR);
    pie_handle->pie_common->CTL = pieCommonRead(CTL);
    pie_handle->pie_common->DBG = pieCommonRead(DBG);
    pie_handle->pie_common->REV0 = pieCommonRead(REV0);
    pie_handle->pie_common->REV1 = pieCommonRead(REV1);
}

struct pie_common_function_struct pie_common_functions =
{
    .pie_reset = pie_reset,
    .pie_configure = pie_configure,
    .pie_get_current = pie_get_current,
    .pie_revcheck = pie_revcheck,

    .dump_pie_common_regs = dump_pie_common_regs,
    .dump_pie_common_debug_regs = dump_pie_common_debug_regs,
    .get_pie_common_a_debug = get_pie_common_a_debug,

    .enable_pie_common_irq = enable_pie_common_irq,
    .disable_pie_common_irq = disable_pie_common_irq,
};

void pie_common_init(pieCommonData *device_data)
{
    pieCommonDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(pieCommonDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_common_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(common, pie_device_handle);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pie_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));    
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pie_common_init);

void pie_common_exit(pieCommonData *device_data)
{
    pieCommonDeviceHandle *pie_device_handle;
    
    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(common);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_common_exit);
