/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_colorshift.h"
#include "pie_colorshift_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_colorshift_regs(colorshiftData *device_data)
{
    print("PIE COLORSHIFT Registers\n");
    print("CCSCR=0x%X\n",  colorshiftRead(CCSCR));
    print("CCSMMR=0x%X\n", colorshiftRead(CCSMMR));
    print("CCSMWR=0x%X\n", colorshiftRead(CCSMWR));
    // CCSLSSR is write only
    print("CCSLSR=0x%X\n", colorshiftRead(CCSLSR));
    print("REV0=0x%X\n",   colorshiftRead(REV0));
    print("REV1=0x%X\n",   colorshiftRead(REV1));
}

static int pie_revcheck(colorshiftData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    
    rev0 = COLORSHIFT_REV0_MAJ_MASK_SHIFT(colorshiftRead(REV0));
    pieh_rev = COLORSHIFT_REV0_MAJ_MASK_SHIFT(pie_handle->pie_colorshift->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(colorshiftData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = colorshiftRead(CCSCR);
    reg = COLORSHIFT_CCSCR_BYPASS_REPLACE_VAL(reg, 1);  // bypass to do a reset
    colorshiftWrite(CCSCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(colorshiftData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for colorshift

    PROTECT_REG_ACCESS;
    colorshiftWrite(CCSCR, pie_handle->pie_colorshift->CCSCR);
    colorshiftWrite(CCSMMR, pie_handle->pie_colorshift->CCSMMR);
    colorshiftWrite(CCSMWR, pie_handle->pie_colorshift->CCSMWR);
    colorshiftWrite(CCSLSSR, pie_handle->pie_colorshift->CCSLSSR);
    // CCSLSR, REV0, and REV1 are Read only
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(colorshiftData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_colorshift->CCSCR  = colorshiftRead(CCSCR);
    pie_handle->pie_colorshift->CCSMMR = colorshiftRead(CCSMMR);
    pie_handle->pie_colorshift->CCSMWR = colorshiftRead(CCSMWR);
    pie_handle->pie_colorshift->CCSLSR = colorshiftRead(CCSLSR);
    // CCSLSSR is write only
    pie_handle->pie_colorshift->REV0   = colorshiftRead(REV0);
    pie_handle->pie_colorshift->REV1   = colorshiftRead(REV1);
}

struct pie_colorshift_function_struct pie_colorshift_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_colorshift_regs  = dump_colorshift_regs,
};

void pie_colorshift_init(colorshiftData *device_data)
{
    colorshiftDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(colorshiftDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_colorshift_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(colorshift, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // colorshift has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE COLORSHIFT LOADED .....\n");
}
EXPORT_SYMBOL(pie_colorshift_init);


void pie_colorshift_exit(colorshiftData *device_data)
{
    colorshiftDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(colorshift);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_colorshift_exit);

