/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"  // for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_clippad.h"
#include "pie_clippad_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_clippad_regs(clippadData *device_data)
{
    print("PIE CLIPPAD Registers\n");
    print("Cfg=0x%X\n",   clippadRead(Cfg));
    print("LPad=0x%X\n", clippadRead(LPad));
    print("RPad=0x%X\n", clippadRead(RPad));
    print("LVal0=0x%X\n", clippadRead(LVal0));
    print("LVal1=0x%X\n", clippadRead(LVal1));
    print("LVal2=0x%X\n", clippadRead(LVal2));
    print("LVal3=0x%X\n", clippadRead(LVal3));
    print("RVal0=0x%X\n", clippadRead(RVal0));
    print("RVal1=0x%X\n", clippadRead(RVal1));
    print("RVal2=0x%X\n", clippadRead(RVal2));
    print("RVal3=0x%X\n", clippadRead(RVal3));
    print("LTrunc=0x%X\n", clippadRead(LTrunc));
    print("RTrunc=0x%X\n", clippadRead(RTrunc));
    print("LineWidth=0x%X\n", clippadRead(LineWidth));
    print("CLIPPADPR=0x%X\n", clippadRead(CLIPPADPR));
    print("REV0=0x%X\n", clippadRead(REV0));
    print("REV1=0x%X\n", clippadRead(REV1));
}

static int pie_revcheck(clippadData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;

    rev0 = CLIPPAD_REV0_MAJ_MASK_SHIFT(clippadRead(REV0));
    pieh_rev = CLIPPAD_REV0_MAJ_MASK_SHIFT(pie_handle->pie_clippad->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(clippadData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = clippadRead(Cfg);
    reg = CLIPPAD_CFG_BYPASS_REPLACE_VAL(reg, 1);
    clippadWrite(Cfg, 0);
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(clippadData *device_data,
                          struct pie_handle_t *pie_handle)
{
    // There is no callback to configure for clippad

    PROTECT_REG_ACCESS;
    clippadWrite(Cfg, pie_handle->pie_clippad->Cfg);
    clippadWrite(LPad, pie_handle->pie_clippad->LPad);
    clippadWrite(RPad, pie_handle->pie_clippad->RPad);
    clippadWrite(LVal0, pie_handle->pie_clippad->LVal0);
    clippadWrite(LVal1, pie_handle->pie_clippad->LVal1);
    clippadWrite(LVal2, pie_handle->pie_clippad->LVal2);
    clippadWrite(LVal3, pie_handle->pie_clippad->LVal3);
    clippadWrite(RVal0, pie_handle->pie_clippad->RVal0);
    clippadWrite(RVal1, pie_handle->pie_clippad->RVal1);
    clippadWrite(RVal2, pie_handle->pie_clippad->RVal2);
    clippadWrite(RVal3, pie_handle->pie_clippad->RVal3);
    clippadWrite(LTrunc, pie_handle->pie_clippad->LTrunc);
    clippadWrite(RTrunc, pie_handle->pie_clippad->RTrunc);
    clippadWrite(LineWidth, pie_handle->pie_clippad->LineWidth);
    // CLIPPADPR, REV0,  and REV1 are Read only
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(clippadData *device_data,
                            struct pie_handle_t *pie_handle)
{
    pie_handle->pie_clippad->Cfg = clippadRead(Cfg);
    pie_handle->pie_clippad->LPad = clippadRead(LPad);
    pie_handle->pie_clippad->RPad = clippadRead(RPad);
    pie_handle->pie_clippad->LVal0 = clippadRead(LVal0);
    pie_handle->pie_clippad->LVal1 = clippadRead(LVal1);
    pie_handle->pie_clippad->LVal2 = clippadRead(LVal2);
    pie_handle->pie_clippad->LVal3 = clippadRead(LVal3);
    pie_handle->pie_clippad->RVal0 = clippadRead(RVal0);
    pie_handle->pie_clippad->RVal1 = clippadRead(RVal1);
    pie_handle->pie_clippad->RVal2 = clippadRead(RVal2);
    pie_handle->pie_clippad->RVal3 = clippadRead(RVal3);
    pie_handle->pie_clippad->LTrunc = clippadRead(LTrunc);
    pie_handle->pie_clippad->RTrunc = clippadRead(RTrunc);
    pie_handle->pie_clippad->LineWidth = clippadRead(LineWidth);
    pie_handle->pie_clippad->CLIPPADPR = clippadRead(CLIPPADPR);
    pie_handle->pie_clippad->REV0 = clippadRead(REV0);
    pie_handle->pie_clippad->REV1 = clippadRead(REV1);
}

struct pie_clippad_function_struct pie_clippad_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_clippad_regs  = dump_clippad_regs,
};

void pie_clippad_init(clippadData *device_data)
{
    clippadDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(clippadDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_clippad_functions;
    pie_device_handle->device_data = device_data;
    register_pie_subblock(clippad, pie_device_handle);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // clippad has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE CLIPPAD LOADED .....\n");
}
EXPORT_SYMBOL(pie_clippad_init);


void pie_clippad_exit(clippadData *device_data)
{
    clippadDeviceHandle *pie_device_handle;

    // unregister with the parent
    pie_device_handle = unregister_pie_subblock(clippad);
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_clippad_exit);

