/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/spinlock.h>  // for spinlock_t

#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h" // needed for detailed pie_handle
#include "pie_handle.h"
#include "pie_data.h"
#include "pie_if.h"
#include "pie_driver.h"
#include "pie_bde.h"
#include "pie_bde_if.h"

#include "pie_driverlib_if.h"  // for print macros

void dump_bde_regs(bdeData *device_data)
{
    print("PIE BDE Registers, channe %d\n", device_data->submodinstance);
    print("REV0=0x%X\n", bdeRead(REV0));
    print("REV1=0x%X\n", bdeRead(REV1));
    print("LUT_CR=0x%X\n", bdeRead(LUT_CR));
}

void dump_bde_lut(bdeData *device_data)
{
    int i;
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg = bdeRead(LUT_CR);
    bdeWrite(LUT_CR, (reg | DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_CPU_ACCESS_MASK));
    
    print("PIE BDE LUT entries, channel %d", device_data->submodinstance);
    for (i=0;i<PIE_BDE_MAX_LUT_ENTRIES;i++)
    {
        if (i%16 == 0)
            print("\n%d-%d: ", i, i+15);            
        print("0x%X ", bdeReadArray(LUT0, i));
    }
    print("\n");

    // reset CPU access bit to original
    bdeWrite(LUT_CR, reg);
    UNPROTECT_REG_ACCESS;
}

static int pie_revcheck(bdeData *device_data,
                        struct pie_handle_t *pie_handle)
{
    uint32_t rev0;
    uint32_t pieh_rev;
    int channel;

    channel = device_data->submodinstance;
    
    rev0 = DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_REV0_MAJ_MASK_SHIFT(bdeRead(REV0));
    pieh_rev =
        DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_REV0_MAJ_MASK_SHIFT(pie_handle->pie_bde[channel]->REV0);
    
    if (rev0 != pieh_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pieh_rev);
        return -1;
    }
    else
        return 0;
}

static void pie_reset(bdeData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = bdeRead(LUT_CR);
    reg = DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_BYPASS_ENABLE_REPLACE_VAL(reg, PIE_BDE_BYPASS);  // bypass to do a reset
    bdeWrite(LUT_CR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static void pie_configure(bdeData *device_data,
                          struct pie_handle_t *pie_handle)
{
    int i;
    uint32_t reg;
    int channel;
    
    channel = device_data->submodinstance;
    // There is no callback to configure for bde
    
    PROTECT_REG_ACCESS;

    // NOTE: to program the LUTs the CPU access bit must
    // be set first.  Then all the LUTs can be written,
    // then the CPU access bit will be cleared.

    // set CPU access bit before programming LUTs
    reg = pie_handle->pie_bde[channel]->LUT_CR;
    reg |= DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_CPU_ACCESS_MASK;
    bdeWrite(LUT_CR, reg);
    // REV0 and REV1 are Read only
    for (i=0;i<PIE_BDE_MAX_LUT_ENTRIES;i++)
    {
        bdeWriteArray(LUT0, pie_handle->pie_bde[channel]->LUT0[i], i);
    }
    
    // now write the LUT_CR (with the cpu_access normal)
    bdeWrite(LUT_CR, pie_handle->pie_bde[channel]->LUT_CR);
    UNPROTECT_REG_ACCESS;
}

static void pie_get_current(bdeData *device_data,
                            struct pie_handle_t *pie_handle)
{
    int i;
    uint32_t reg;
    int channel;

    channel = device_data->submodinstance;
    
    PROTECT_REG_ACCESS;
    // set CPU access bit before reading LUTs
    reg = bdeRead(LUT_CR);
    pie_handle->pie_bde[channel]->LUT_CR = reg;

    bdeWrite(LUT_CR, (reg | DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_LUT_CR_CPU_ACCESS_MASK));
    
    pie_handle->pie_bde[channel]->REV0 = bdeRead(REV0);
    pie_handle->pie_bde[channel]->REV1 = bdeRead(REV1);
    for (i=0;i<PIE_BDE_MAX_LUT_ENTRIES;i++)
    {
        pie_handle->pie_bde[channel]->LUT0[i] = bdeReadArray(LUT0, i);
    }
    // now write the LUT_CR (with the cpu_access normal)
    bdeWrite(LUT_CR, reg);
    UNPROTECT_REG_ACCESS;
}

struct pie_bde_function_struct pie_bde_functions =
{
    .pie_reset             = pie_reset,
    .pie_configure         = pie_configure,
    .pie_get_current       = pie_get_current,
    .pie_revcheck          = pie_revcheck,
    
    .dump_bde_regs  = dump_bde_regs,
    .dump_bde_lut  = dump_bde_lut,
};

void pie_bde_init(bdeData *device_data)
{
    bdeDeviceHandle *pie_device_handle;

    pie_device_handle = allocate_memory(sizeof(bdeDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pie_device_handle->fcn_tbl = &pie_bde_functions;
    pie_device_handle->device_data = device_data;
    switch (device_data->submodinstance)
    {
    case 0:
        register_pie_subblock(bde0, pie_device_handle);
        break;
    case 1:
        register_pie_subblock(bde1, pie_device_handle);
        break;
    case 2:
        register_pie_subblock(bde2, pie_device_handle);
        break;
    }
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pie_device_handle->device_data->reg_spinlock));

    // bde has no interrupts associated with it, so no callback
    pie_device_handle->device_data->interrupt_callback = NULL;
    pie_device_handle->device_data->interrupt_callback_data = NULL;    
    print("PIE BDE LOADED .....\n");
}
EXPORT_SYMBOL(pie_bde_init);


void pie_bde_exit(bdeData *device_data)
{
    bdeDeviceHandle *pie_device_handle;

    pie_device_handle = NULL;
    // unregister with the parent
    switch (device_data->submodinstance)
    {
    case 0:
        pie_device_handle = unregister_pie_subblock(bde0);
        break;
    case 1:
        pie_device_handle = unregister_pie_subblock(bde1);
        break;
    case 2:
        pie_device_handle = unregister_pie_subblock(bde2);
        break;
    }
    
    free_memory(pie_device_handle);
}
EXPORT_SYMBOL(pie_bde_exit);

