/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/kernel.h>    // for printk and uint32_t
#include <linux/bug.h>       // for BUG_ON/BUG
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/vmalloc.h>

#include "pie_handle_if.h"   // for opaque pie_handle definition
#include "pie_if.h"          // for pie irq structures
#include "pie.h"             // for opaque (mostly) subblock handles
#include "pie_driver.h"      // pie.c enums and exports

// for subblock register sizes, include the pie headers
#include "PIE_regheaders.h"
#include "pie_full_subblock_list.h"
// these include files are for the subblock function_structs
#include "pie_common_if.h"
#include "pogo_dma_if.h"
#include "pie_xyscaler_if.h"
#include "pie_xycscaler_if.h"
#include "pie_colorshift_if.h"
#include "pie_rgb2ycc_if.h"
#include "pie_ycc2rgb_if.h"
#include "pie_rgb2esrgb_if.h"
#include "pie_dsmf_if.h"
#include "pie_tcns_if.h"
#include "pie_sccsc_if.h"
#include "pie_cstats_if.h"
#include "pie_ngadjust_if.h"
#include "pie_distort_if.h"
#include "pie_decim_if.h"
#include "pie_clippad_if.h"
#include "pie_bde_if.h"
#include "pie_otmarb_if.h"
// end of include files for the subblock function_structs

#include "pie_driverlib_if.h"  // for the print macros


struct pie_object_struct
{
    pieCommonDeviceHandle  *pie_common_handle;
    pogoDeviceHandle       *pogo_dma_handle;
    xyscaleDeviceHandle    *xyscaler_handle;
    xy_c_scaleDeviceHandle *xy_c_scaler_handle;
    colorshiftDeviceHandle *colorshift_handle;
    rgb2yccDeviceHandle    *rgb2ycc_handle;
    ycc2rgbDeviceHandle    *ycc2rgb_handle;
    rgb2esrgbDeviceHandle  *rgb2esrgb_handle;
    dsmfDeviceHandle       *dsmf_handle;
    tcnsDeviceHandle       *tcns_handle;    
    sccscDeviceHandle      *sccsc_handle;    
    cstatsDeviceHandle     *cstats_handle;    
    ngadjustDeviceHandle   *ngadjust_handle;    
    istoneDeviceHandle     *istone_handle;
    denoiseDeviceHandle    *denoise_handle;
    decimDeviceHandle      *decim_handle;
    clippadDeviceHandle    *clippad_handle;
    antifcorDeviceHandle   *antifcor_handle;
    distortDeviceHandle    *distort_handle;
    dismapDeviceHandle     *dismap_handle;
    bdeDeviceHandle        *bde_handle[NUM_BDE_CHANNELS];
    ihiDeviceHandle        *ihi_handle;
    ihiDepogoDeviceHandle  *ihi_depogo_handle;                 
    pogoIhiOdmaUdmaDeviceHandle  *pogo_ihi_odma_udma_handle;
    pogoIhiOdmaCoreDeviceHandle  *pogo_ihi_odma_core_handle;
    ddmaAcDataDeviceHandle       *ddma_ac_data_handle[NUM_PIE_DDMA_CHANNELS];
    ddmaAcCorrDeviceHandle       *ddma_ac_corr_handle[NUM_PIE_DDMA_CHANNELS];
    otPogoIdmaUdmaDeviceHandle   *otpogoidma_udma_handle;
    otPogoIdmaCoreDeviceHandle   *otpogoidma_core_handle;
    otPogoOdmaUdmaDeviceHandle   *otpogoodma_udma_handle;
    otPogoOdmaCoreDeviceHandle   *otpogoodma_core_handle;
    otPogoizerDeviceHandle       *otpogoizer_handle;
    otDepogoizerDeviceHandle     *otdepogoizer_handle;
    otmarbDeviceHandle           *otmarb_handle;                                
    
    // as more handles are added, add corresponding inits down in the
    // init_pie and register/unregister_pie_subblock
};

struct pie_object_struct pie_object;

// called by lo level driver initialization - before any chance
// of the subblock drivers registering
void init_pie(void)
{
    int x;
    debug_print("%s: initializing pie\n", __func__);
    pie_object.pie_common_handle = NULL;
    pie_object.pogo_dma_handle = NULL;    
    pie_object.xyscaler_handle = NULL;
    pie_object.xy_c_scaler_handle = NULL;
    pie_object.colorshift_handle = NULL;
    pie_object.rgb2ycc_handle = NULL;
    pie_object.ycc2rgb_handle = NULL;
    pie_object.rgb2esrgb_handle = NULL;
    pie_object.dsmf_handle = NULL;
    pie_object.tcns_handle = NULL;    
    pie_object.sccsc_handle = NULL;    
    pie_object.cstats_handle = NULL;    
    pie_object.ngadjust_handle = NULL;
    pie_object.istone_handle = NULL;
    pie_object.denoise_handle = NULL;
    pie_object.decim_handle = NULL;
    pie_object.clippad_handle = NULL;
    pie_object.antifcor_handle = NULL;
    pie_object.distort_handle = NULL;
    pie_object.dismap_handle = NULL;

    for (x=0;x<NUM_BDE_CHANNELS;x++)
        pie_object.bde_handle[x] = NULL;

    pie_object.ihi_handle = NULL;
    pie_object.ihi_depogo_handle = NULL;
    pie_object.pogo_ihi_odma_udma_handle = NULL;
    pie_object.pogo_ihi_odma_core_handle = NULL;

    for (x=0;x<NUM_PIE_DDMA_CHANNELS;x++)
    {
        pie_object.ddma_ac_data_handle[x] = NULL;
        pie_object.ddma_ac_corr_handle[x] = NULL;
    }

    pie_object.otpogoidma_udma_handle = NULL;
    pie_object.otpogoidma_core_handle = NULL;
    pie_object.otpogoodma_udma_handle = NULL;
    pie_object.otpogoodma_core_handle = NULL;
    pie_object.otpogoizer_handle = NULL;
    pie_object.otdepogoizer_handle = NULL;
    pie_object.otmarb_handle = NULL;
    
    // initialize any subsystems that need it
    pogo_dma_init();
}

void exit_pie(void)
{
    // we don't need to NULL out the pie_object, since the unregister takes care of that.
    
    // call cleanup of any subsystems that need it
    pogo_dma_exit();
}

void register_pie_subblock(enum pie_internal_subblock piesubblock, void *pie_subblock_data)
{
    if (piesubblock == common)
    {
        debug_print("registered pie_common with pie.c\n");
        pie_object.pie_common_handle = pie_subblock_data;
    }
    else if (piesubblock == pogo_dma)
    {
        debug_print("registered pogo dma with pie.c\n");        
        pie_object.pogo_dma_handle = pie_subblock_data;
    }
    else if (piesubblock == xyscaler)
    {
    	debug_print("registered xyscaler with pie.c\n");
    	pie_object.xyscaler_handle = pie_subblock_data;
    }
    else if (piesubblock == xycscaler)
    {
    	debug_print("registered xycscaler with pie.c\n");
    	pie_object.xy_c_scaler_handle = pie_subblock_data;
    }
    else if (piesubblock == colorshift)
    {
    	debug_print("registered colorshift with pie.c\n");
    	pie_object.colorshift_handle = pie_subblock_data;
    }
    else if (piesubblock == rgb2ycc)
    {
    	debug_print("registered rgb2ycc with pie.c\n");
    	pie_object.rgb2ycc_handle = pie_subblock_data;
    }
    else if (piesubblock == ycc2rgb)
    {
    	debug_print("registered ycc2rgb with pie.c\n");
    	pie_object.ycc2rgb_handle = pie_subblock_data;
    }
    else if (piesubblock == rgb2esrgb)
    {
    	debug_print("registered rgb2esrgb with pie.c\n");
    	pie_object.rgb2esrgb_handle = pie_subblock_data;
    }
    else if (piesubblock == dsmf)
    {
    	debug_print("registered dsmf with pie.c\n");
    	pie_object.dsmf_handle = pie_subblock_data;
    }
    else if (piesubblock == tcns)
    {
    	debug_print("registered tcns with pie.c\n");
    	pie_object.tcns_handle = pie_subblock_data;
    }
    else if (piesubblock == sccsc)
    {
    	debug_print("registered sccsc with pie.c\n");
    	pie_object.sccsc_handle = pie_subblock_data;
    }
    else if (piesubblock == cstats)
    {
    	debug_print("registered cstats with pie.c\n");
    	pie_object.cstats_handle = pie_subblock_data;
    }
    else if (piesubblock == ngadjust)
    {
    	debug_print("registered ngadjust with pie.c\n");
    	pie_object.ngadjust_handle = pie_subblock_data;
    }
    else if (piesubblock == antifcor)
    {
    	debug_print("registered antifcor with pie.c\n");
    	pie_object.antifcor_handle = pie_subblock_data;
    }
    else if (piesubblock == distort)
    {
    	debug_print("registered distort with pie.c\n");
    	pie_object.distort_handle = pie_subblock_data;
    }
    else if (piesubblock == ddma0)
    {
    	debug_print("registered ddma0 with pie.c\n");
    	pie_object.ddma_ac_data_handle[0] = pie_subblock_data;
    }
    else if (piesubblock == ddma1)
    {
    	debug_print("registered ddma1 with pie.c\n");
    	pie_object.ddma_ac_data_handle[1] = pie_subblock_data;
    }
    else if (piesubblock == ddma2)
    {
    	debug_print("registered ddma2 with pie.c\n");
    	pie_object.ddma_ac_data_handle[2] = pie_subblock_data;
    }
    else if (piesubblock == decim)
    {
    	debug_print("registered decim with pie.c\n");
    	pie_object.decim_handle = pie_subblock_data;
    }
    else if (piesubblock == clippad)
    {
    	debug_print("registered clippad with pie.c\n");
    	pie_object.clippad_handle = pie_subblock_data;
    }
    else if (piesubblock == bde0)
    {
    	debug_print("registered bde with pie.c\n");
    	pie_object.bde_handle[0] = pie_subblock_data;
    }
    else if (piesubblock == bde1)
    {
    	debug_print("registered bde with pie.c\n");
    	pie_object.bde_handle[1] = pie_subblock_data;
    }
    else if (piesubblock == bde2)
    {
    	debug_print("registered bde with pie.c\n");
    	pie_object.bde_handle[2] = pie_subblock_data;
    }
    else if (piesubblock == otmarb)
    {
    	debug_print("registered otmarb with pie.c\n");
    	pie_object.otmarb_handle = pie_subblock_data;
    }        
    
    else
    {
        error_print("Error!!! no support yet for blocks %d\n", piesubblock);
    }
}

void *unregister_pie_subblock(enum pie_internal_subblock piesubblock)
{
    void *subblock_data;
    
    if (piesubblock == common)
    {
        subblock_data = pie_object.pie_common_handle;
        pie_object.pie_common_handle = NULL;
        debug_print("unregistered pie_common with pie.c\n");
    }
    else if (piesubblock == pogo_dma)
    {
        subblock_data = pie_object.pogo_dma_handle;
        pie_object.pogo_dma_handle = NULL;
        debug_print("unregistered pogo dma with pie.c\n");        
    }
    else if (piesubblock == xyscaler)
    {
        subblock_data = pie_object.xyscaler_handle;
        pie_object.xyscaler_handle = NULL;
    	debug_print("unregistered xyscaler with pie.c\n");
    }
    else if (piesubblock == xycscaler)
    {
        subblock_data = pie_object.xy_c_scaler_handle;
        pie_object.xy_c_scaler_handle = NULL;
    	debug_print("unregistered xycscaler with pie.c\n");
    }
    else if (piesubblock == colorshift)
    {
        subblock_data = pie_object.colorshift_handle;
        pie_object.colorshift_handle = NULL;
    	debug_print("unregistered colorshift with pie.c\n");
    }
    else if (piesubblock == rgb2ycc)
    {
        subblock_data = pie_object.rgb2ycc_handle;
        pie_object.rgb2ycc_handle = NULL;
    	debug_print("unregistered rgb2ycc with pie.c\n");
    }
    else if (piesubblock == ycc2rgb)
    {
        subblock_data = pie_object.ycc2rgb_handle;
        pie_object.ycc2rgb_handle = NULL;
    	debug_print("unregistered ycc2rgb with pie.c\n");
    }
    else if (piesubblock == rgb2esrgb)
    {
        subblock_data = pie_object.rgb2esrgb_handle;
        pie_object.rgb2esrgb_handle = NULL;
    	debug_print("unregistered rgb2esrgb with pie.c\n");
    }
    else if (piesubblock == dsmf)
    {
        subblock_data = pie_object.dsmf_handle;
        pie_object.dsmf_handle = NULL;
    	debug_print("unregistered dsmf with pie.c\n");
    }
    else if (piesubblock == tcns)
    {
        subblock_data = pie_object.tcns_handle;
        pie_object.tcns_handle = NULL;
    	debug_print("unregistered tcns with pie.c\n");
    }
    else if (piesubblock == sccsc)
    {
        subblock_data = pie_object.sccsc_handle;
        pie_object.sccsc_handle = NULL;
    	debug_print("unregistered sccsc with pie.c\n");
    }
    else if (piesubblock == cstats)
    {
        subblock_data = pie_object.cstats_handle;
        pie_object.cstats_handle = NULL;
    	debug_print("unregistered cstats with pie.c\n");
    }
    else if (piesubblock == ngadjust)
    {
        subblock_data = pie_object.ngadjust_handle;
        pie_object.ngadjust_handle = NULL;
    	debug_print("unregistered ngadjust with pie.c\n");
    }
    else if (piesubblock == antifcor)
    {
        subblock_data = pie_object.antifcor_handle;
        pie_object.antifcor_handle = NULL;
    	debug_print("unregistered antifcor with pie.c\n");
    }
    else if (piesubblock == distort)
    {
        subblock_data = pie_object.distort_handle;
        pie_object.distort_handle = NULL;
    	debug_print("unregistered distort with pie.c\n");
    }
    else if (piesubblock == ddma0)
    {
        subblock_data = pie_object.ddma_ac_data_handle[0];
        pie_object.ddma_ac_data_handle[0] = NULL;
    	debug_print("unregistered distort with pie.c\n");
    }
    else if (piesubblock == ddma1)
    {
        subblock_data = pie_object.ddma_ac_data_handle[1];
        pie_object.ddma_ac_data_handle[1] = NULL;
    	debug_print("unregistered distort with pie.c\n");
    }
    else if (piesubblock == ddma2)
    {
        subblock_data = pie_object.ddma_ac_data_handle[2];
        pie_object.ddma_ac_data_handle[2] = NULL;
    	debug_print("unregistered distort with pie.c\n");
    }
    else if (piesubblock == decim)
    {
        subblock_data = pie_object.decim_handle;
        pie_object.decim_handle = NULL;
    	debug_print("unregistered decim with pie.c\n");
    }
    else if (piesubblock == clippad)
    {
        subblock_data = pie_object.clippad_handle;
        pie_object.clippad_handle = NULL;
    	debug_print("unregistered clippad with pie.c\n");
    }
    else if (piesubblock == bde0)
    {
        subblock_data = pie_object.bde_handle[0];
        pie_object.bde_handle[0] = NULL;
    	debug_print("unregistered bde0 with pie.c\n");
    }
    else if (piesubblock == bde1)
    {
        subblock_data = pie_object.bde_handle[1];
        pie_object.bde_handle[1] = NULL;
    	debug_print("unregistered bde1 with pie.c\n");
    }
    else if (piesubblock == bde2)
    {
        subblock_data = pie_object.bde_handle[2];
        pie_object.bde_handle[2] = NULL;
    	debug_print("unregistered bde2 with pie.c\n");
    }
    else if (piesubblock == otmarb)
    {
        subblock_data = pie_object.otmarb_handle;
        pie_object.otmarb_handle = NULL;
    	debug_print("unregistered otmarb with pie.c\n");
    }
    else
    {
        error_print("Error!!! no support yet for blocks %d\n", piesubblock);
        return NULL;
    }
    return subblock_data;
}

// NOTE! This is only for use by pogo_dma.c - when registering or
// unregistering subblocks
pogoDeviceHandle *get_pogo_device(void)
{
    return pie_object.pogo_dma_handle;
}


#define CHECK4NULL(pointer)                                             \
    if (pointer == NULL)                                                \
    {                                                                   \
        print("ERROR %s, pointer %s is NULL\n", __func__,               \
              #pointer);                                                \
        BUG();                                                          \
    }

// macro to find the function table and subblock data
#define PIE_RETRIEVE_function_table(data_type, function_struct, pie_object_field) \
    struct function_struct *ft;                                         \
    data_type *pie_subblock_data;                                       \
                                                                        \
    if (pie_object.pie_object_field == NULL)                            \
        return;  /* subblock doesn't exist */                           \
    pie_subblock_data = pie_object.pie_object_field->device_data;       \
    CHECK4NULL(pie_subblock_data);                                      \
    ft = pie_object.pie_object_field->fcn_tbl;                          \
    CHECK4NULL(ft);

#define PIE_RETRIEVE_function_table_retval(data_type, function_struct, pie_object_field, ret_on_null) \
    struct function_struct *ft;                                         \
    data_type *pie_subblock_data;                                       \
                                                                        \
    if (pie_object.pie_object_field == NULL)                            \
        return ret_on_null;  /* subblock doesn't exist */               \
    pie_subblock_data = pie_object.pie_object_field->device_data;       \
    CHECK4NULL(pie_subblock_data);                                      \
    ft = pie_object.pie_object_field->fcn_tbl;                          \
    CHECK4NULL(ft);

// ok, now that we have our function table in hand (the ft from the macro), figure out
// which standard function to call
// Note that retval is only set if the requested function returns a value
#define EXECUTE_FUNCTION(the_function, retval)                          \
    if (the_function == do_reset)                                       \
        ft->pie_reset(pie_subblock_data);                               \
    else if (the_function == do_configure)                              \
        ft->pie_configure(pie_subblock_data, pie_handle);               \
    else if (the_function == do_get_current)                            \
        ft->pie_get_current(pie_subblock_data, pie_handle);             \
    else if (the_function == do_revcheck)                               \
        *retval = ft->pie_revcheck(pie_subblock_data, pie_handle);      \
    else                                                                \
        error_print("%s:Error!! no support for function %d\n", __func__, the_function);


static int do_function_subblock(enum pie_internal_subblock piesubblock,
                                enum function_type do_function,
                                struct pie_handle_t *pie_handle)
{
    int retval = -1;

    if (piesubblock == common)
    {
        PIE_RETRIEVE_function_table_retval(pieCommonData, pie_common_function_struct,
                                           pie_common_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == pogo_dma)
    {
        PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                           pogo_dma_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == xyscaler)
    {
        PIE_RETRIEVE_function_table_retval(xyscaleData, pie_xyscaler_function_struct,
                                           xyscaler_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == xycscaler)
    {
        PIE_RETRIEVE_function_table_retval(xy_c_scaleData, pie_xycscaler_function_struct,
                                           xy_c_scaler_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == colorshift)
    {
        PIE_RETRIEVE_function_table_retval(colorshiftData, pie_colorshift_function_struct,
                                           colorshift_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == rgb2ycc)
    {
        PIE_RETRIEVE_function_table_retval(rgb2yccData, pie_rgb2ycc_function_struct,
                                           rgb2ycc_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == ycc2rgb)
    {
        PIE_RETRIEVE_function_table_retval(ycc2rgbData, pie_ycc2rgb_function_struct,
                                           ycc2rgb_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == rgb2esrgb)
    {
        PIE_RETRIEVE_function_table_retval(rgb2esrgbData, pie_rgb2esrgb_function_struct,
                                           rgb2esrgb_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == dsmf)
    {
        PIE_RETRIEVE_function_table_retval(dsmfData, pie_dsmf_function_struct,
                                           dsmf_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == tcns)
    {
        PIE_RETRIEVE_function_table_retval(tcnsData, pie_tcns_function_struct,
                                          tcns_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == sccsc)
    {
        PIE_RETRIEVE_function_table_retval(sccscData, pie_sccsc_function_struct,
                                           sccsc_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == cstats)
    {
        PIE_RETRIEVE_function_table_retval(cstatsData, pie_cstats_function_struct,
                                           cstats_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == ngadjust)
    {
        PIE_RETRIEVE_function_table_retval(ngadjustData, pie_ngadjust_function_struct,
                                           ngadjust_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == decim)
    {
        PIE_RETRIEVE_function_table_retval(decimData, pie_decim_function_struct,
                                           decim_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == clippad)
    {
        PIE_RETRIEVE_function_table_retval(clippadData, pie_clippad_function_struct,
                                           clippad_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == antifcor)
    {
        PIE_RETRIEVE_function_table_retval(antifcorData, pie_antifcor_function_struct, antifcor_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == distort)
    {
        PIE_RETRIEVE_function_table_retval(distortData, pie_distort_function_struct, distort_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == ddma0)
    {
        PIE_RETRIEVE_function_table_retval(ddmaAcDataData, pie_ddma_ac_data_function_struct, ddma_ac_data_handle[0], -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == ddma1)
    {
        PIE_RETRIEVE_function_table_retval(ddmaAcDataData, pie_ddma_ac_data_function_struct, ddma_ac_data_handle[1], -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == ddma2)
    {
        PIE_RETRIEVE_function_table_retval(ddmaAcDataData, pie_ddma_ac_data_function_struct, ddma_ac_data_handle[2], -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else if (piesubblock == bde0)
    {
        PIE_RETRIEVE_function_table_retval(bdeData, pie_bde_function_struct,
                                           bde_handle[0], -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == bde1)
    {
        PIE_RETRIEVE_function_table_retval(bdeData, pie_bde_function_struct,
                                           bde_handle[1], -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == bde2)
    {
        PIE_RETRIEVE_function_table_retval(bdeData, pie_bde_function_struct,
                                           bde_handle[2], -1);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (piesubblock == otmarb)
    {
        PIE_RETRIEVE_function_table_retval(pieOTMarbData, pie_otmarb_function_struct, otmarb_handle, -1);
        EXECUTE_FUNCTION(do_function, &retval);        
    }
    else
    {
        error_print("%s:Error!!! no support yet for blocks %d\n", __func__, piesubblock);
    }
    return retval;
}
    
//////////////////////// start of public API functions ///////////////////////
int pie_do_revcheck(struct pie_handle_t *pie_handle)
{
    if (do_function_subblock(common, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(pogo_dma, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(xyscaler, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(xycscaler, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(colorshift, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(rgb2ycc, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(ycc2rgb, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(rgb2esrgb, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(dsmf, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(tcns, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(sccsc, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(cstats, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(ngadjust, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(antifcor, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(distort, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(ddma0, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(ddma1, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(ddma2, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(decim, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(clippad, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(bde0, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(bde1, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(bde2, do_revcheck, pie_handle) != 0)
        return -1;
    if (do_function_subblock(otmarb, do_revcheck, pie_handle) != 0)
        return -1;
//    print("FIXME %s - add subblocks \n", __func__);
    return 0;
}
EXPORT_SYMBOL(pie_do_revcheck);

// soft reset - This is sets whatever bits necessary to
// reset pie's subblocks
void pie_do_reset(void)
{
    do_function_subblock(common, do_reset, NULL);
    do_function_subblock(pogo_dma, do_reset, NULL);
    do_function_subblock(xyscaler, do_reset, NULL);
    do_function_subblock(xycscaler, do_reset, NULL);
    do_function_subblock(colorshift, do_reset, NULL);
    do_function_subblock(rgb2ycc, do_reset, NULL);    
    do_function_subblock(ycc2rgb, do_reset, NULL);
    do_function_subblock(rgb2esrgb, do_reset, NULL);
    do_function_subblock(dsmf, do_reset, NULL);
    do_function_subblock(tcns, do_reset, NULL);    
    do_function_subblock(sccsc, do_reset, NULL);
    do_function_subblock(cstats, do_reset, NULL);
    do_function_subblock(ngadjust, do_reset, NULL);
    do_function_subblock(antifcor, do_reset, NULL);        
    do_function_subblock(distort, do_reset, NULL);        
    do_function_subblock(ddma0, do_reset, NULL);        
    do_function_subblock(ddma1, do_reset, NULL);        
    do_function_subblock(ddma2, do_reset, NULL);        
    do_function_subblock(decim, do_reset, NULL);
    do_function_subblock(clippad, do_reset, NULL);
    do_function_subblock(bde0, do_reset, NULL);
    do_function_subblock(bde1, do_reset, NULL);
    do_function_subblock(bde2, do_reset, NULL);
    do_function_subblock(otmarb, do_reset, NULL);

//    print("FIXME %s - add subblocks\n", __func__);
}
EXPORT_SYMBOL(pie_do_reset);

// NOTE: pie_do_configure writes all of the configuration registers,
// but not the side effects registers.  e.g.  we do not write the UDR
// for the UDMA, since that is not a configuration register, but rather
// starts the DMA.  To start the DMAs for a pogo transfer (the dma code only!)
// be sure to call the functions pie_start_pogo_input_dma() and
// pie_start_pogo_output_dma().
void pie_do_configure(struct pie_handle_t *pie_handle)
{
    do_function_subblock(common, do_configure, pie_handle);
    do_function_subblock(pogo_dma, do_configure, pie_handle);
    do_function_subblock(xyscaler, do_configure, pie_handle);
    do_function_subblock(xycscaler, do_configure, pie_handle);
    do_function_subblock(colorshift, do_configure, pie_handle);
    do_function_subblock(rgb2ycc, do_configure, pie_handle);    
    do_function_subblock(ycc2rgb, do_configure, pie_handle);
    do_function_subblock(rgb2esrgb, do_configure, pie_handle);
    do_function_subblock(dsmf, do_configure, pie_handle);
    do_function_subblock(tcns, do_configure, pie_handle);    
    do_function_subblock(sccsc, do_configure, pie_handle);
    do_function_subblock(cstats, do_configure, pie_handle);
    do_function_subblock(ngadjust, do_configure, pie_handle);        
    do_function_subblock(antifcor, do_configure, pie_handle);        
    do_function_subblock(distort, do_configure, pie_handle);        
    do_function_subblock(ddma0, do_configure, pie_handle);        
    do_function_subblock(ddma1, do_configure, pie_handle);        
    do_function_subblock(ddma2, do_configure, pie_handle);        
    do_function_subblock(decim, do_configure, pie_handle);
    do_function_subblock(clippad, do_configure, pie_handle);    
    do_function_subblock(bde0, do_configure, pie_handle);
    do_function_subblock(bde1, do_configure, pie_handle);
    do_function_subblock(bde2, do_configure, pie_handle);
    do_function_subblock(otmarb, do_configure, pie_handle);
//    print("FIXME %s - add subblocks \n", __func__);
}
EXPORT_SYMBOL(pie_do_configure);

void pie_do_free_handle(struct pie_handle_t *pie_handle)
{
    vfree(pie_handle);
}
EXPORT_SYMBOL(pie_do_free_handle);

// Send the pie_handle down to each subblock to fill in the
// current values of all the registers
void pie_do_get_current(struct pie_handle_t *pie_handle)
{
    do_function_subblock(common, do_get_current, pie_handle);
    do_function_subblock(pogo_dma, do_get_current, pie_handle);
    do_function_subblock(xyscaler, do_get_current, pie_handle);
    do_function_subblock(xycscaler, do_get_current, pie_handle);
    do_function_subblock(colorshift, do_get_current, pie_handle);
    do_function_subblock(rgb2ycc, do_get_current, pie_handle);    
    do_function_subblock(ycc2rgb, do_get_current, pie_handle);
    do_function_subblock(rgb2esrgb, do_get_current, pie_handle);
    do_function_subblock(dsmf, do_get_current, pie_handle);
    do_function_subblock(tcns, do_get_current, pie_handle);    
    do_function_subblock(sccsc, do_get_current, pie_handle);
    do_function_subblock(cstats, do_get_current, pie_handle);
    do_function_subblock(ngadjust, do_get_current, pie_handle);        
    do_function_subblock(antifcor, do_get_current, pie_handle);        
    do_function_subblock(distort, do_get_current, pie_handle);        
    do_function_subblock(ddma0, do_get_current, pie_handle);        
    do_function_subblock(ddma1, do_get_current, pie_handle);        
    do_function_subblock(ddma2, do_get_current, pie_handle);        
    do_function_subblock(decim, do_get_current, pie_handle);        
    do_function_subblock(clippad, do_get_current, pie_handle);
    do_function_subblock(bde0, do_get_current, pie_handle);
    do_function_subblock(bde1, do_get_current, pie_handle);
    do_function_subblock(bde2, do_get_current, pie_handle);    
    do_function_subblock(otmarb, do_get_current, pie_handle);    
//    print("FIXME %s - add subblocks \n", __func__);
}
EXPORT_SYMBOL(pie_do_get_current);

////////////////////////////
// register dump functions  
////////////////////////////

void pie_dump_common_regs(void)
{
    PIE_RETRIEVE_function_table(pieCommonData, pie_common_function_struct, pie_common_handle);
    ft->dump_pie_common_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_dump_common_regs);

void pie_dump_pogo_regs(void)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->pogo_dma_dump_all_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_dump_pogo_regs);

void pie_xy_scale_dump(void)
{
    PIE_RETRIEVE_function_table(xyscaleData, pie_xyscaler_function_struct, xyscaler_handle);
    ft->dump_xyscaler_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_xy_scale_dump);

void pie_xy_c_scale_dump()
{
    PIE_RETRIEVE_function_table(xy_c_scaleData, pie_xycscaler_function_struct, xy_c_scaler_handle);
    ft->dump_xycscaler_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_xy_c_scale_dump);

void pie_colorshift_dump(void)
{
    PIE_RETRIEVE_function_table(colorshiftData, pie_colorshift_function_struct, colorshift_handle);
    ft->dump_colorshift_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_colorshift_dump);

void pie_rgb2ycc_dump(void)
{
    PIE_RETRIEVE_function_table(rgb2yccData, pie_rgb2ycc_function_struct, rgb2ycc_handle);
    ft->dump_rgb2ycc_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_rgb2ycc_dump);

void pie_ycc2rgb_dump(void)
{
    PIE_RETRIEVE_function_table(ycc2rgbData, pie_ycc2rgb_function_struct, ycc2rgb_handle);
    ft->dump_ycc2rgb_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_ycc2rgb_dump);

void pie_rgb2esrgb_dump(void)
{
    PIE_RETRIEVE_function_table(rgb2esrgbData, pie_rgb2esrgb_function_struct, rgb2esrgb_handle);
    ft->dump_rgb2esrgb_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_rgb2esrgb_dump);

void pie_rgb2esrgb_dump_luts(void)
{
    PIE_RETRIEVE_function_table(rgb2esrgbData, pie_rgb2esrgb_function_struct, rgb2esrgb_handle);
    ft->dump_rgb2esrgb_luts(pie_subblock_data);
}
EXPORT_SYMBOL(pie_rgb2esrgb_dump_luts);

void pie_dsmf_dump(void)
{
    PIE_RETRIEVE_function_table(dsmfData, pie_dsmf_function_struct, dsmf_handle);
    ft->dump_dsmf_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_dsmf_dump);

void pie_tcns_dump(void)
{
    PIE_RETRIEVE_function_table(tcnsData, pie_tcns_function_struct, tcns_handle);
    ft->dump_tcns_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_tcns_dump);

void pie_tcns_dump_luts(void)
{
    PIE_RETRIEVE_function_table(tcnsData, pie_tcns_function_struct, tcns_handle);
    ft->dump_tcns_luts(pie_subblock_data, 1);
    ft->dump_tcns_luts(pie_subblock_data, 2);    
}
EXPORT_SYMBOL(pie_tcns_dump_luts);

void pie_sccsc_dump(void)
{
    PIE_RETRIEVE_function_table(sccscData, pie_sccsc_function_struct, sccsc_handle);
    ft->dump_sccsc_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_sccsc_dump);

void pie_sccsc_dump_ilut(void)
{
    PIE_RETRIEVE_function_table(sccscData, pie_sccsc_function_struct, sccsc_handle);
    ft->dump_sccsc_ilut(pie_subblock_data);
}
EXPORT_SYMBOL(pie_sccsc_dump_ilut);

void pie_cstats_dump(void)
{
    PIE_RETRIEVE_function_table(cstatsData, pie_cstats_function_struct, cstats_handle);
    ft->dump_cstats_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_cstats_dump);

void pie_cstats_dump_ram(void)
{
    PIE_RETRIEVE_function_table(cstatsData, pie_cstats_function_struct, cstats_handle);
    ft->dump_cstats_ram(pie_subblock_data, 0);
    ft->dump_cstats_ram(pie_subblock_data, 1);
    ft->dump_cstats_ram(pie_subblock_data, 2);
    ft->dump_cstats_ram(pie_subblock_data, 3);
}
EXPORT_SYMBOL(pie_cstats_dump_ram);

void pie_ngadjust_dump(void)
{
    PIE_RETRIEVE_function_table(ngadjustData, pie_ngadjust_function_struct, ngadjust_handle);
    ft->dump_ngadjust_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_ngadjust_dump);

void pie_ngadjust_dump_lut(void)
{
    PIE_RETRIEVE_function_table(ngadjustData, pie_ngadjust_function_struct, ngadjust_handle);
    ft->dump_ngadjust_lut(pie_subblock_data);
}
EXPORT_SYMBOL(pie_ngadjust_dump_lut);

void pie_decim_dump(void)
{
    PIE_RETRIEVE_function_table(decimData, pie_decim_function_struct, decim_handle);
    ft->dump_decim_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_decim_dump);

void pie_clippad_dump(void)
{
    PIE_RETRIEVE_function_table(clippadData, pie_clippad_function_struct, clippad_handle);
    ft->dump_clippad_regs(pie_subblock_data);
}
EXPORT_SYMBOL(pie_clippad_dump);

void pie_bde_dump(void)
{
    int i;

    // dump all bde's channels
    for (i=0;i<NUM_BDE_CHANNELS;i++)
    {
        PIE_RETRIEVE_function_table(bdeData, pie_bde_function_struct, bde_handle[i]);
        ft->dump_bde_regs(pie_subblock_data);
    }
}
EXPORT_SYMBOL(pie_bde_dump);

void pie_bde_lut_dump(void)
{
    int i;

    // dump all bde's channels
    for (i=0;i<NUM_BDE_CHANNELS;i++)
    {
        PIE_RETRIEVE_function_table(bdeData, pie_bde_function_struct, bde_handle[i]);
        ft->dump_bde_lut(pie_subblock_data);
    }
}
EXPORT_SYMBOL(pie_bde_lut_dump);

void pie_otmarb_dump(void)
{
    PIE_RETRIEVE_function_table(pieOTMarbData, pie_otmarb_function_struct, otmarb_handle);
    ft->pie_dump_regs(pie_subblock_data);
}    

void pie_dump(void)
{
    pie_dump_common_regs();
    pie_dump_pogo_regs();
    pie_xy_scale_dump();
    pie_xy_c_scale_dump();
    pie_colorshift_dump();
    pie_rgb2ycc_dump();
    pie_ycc2rgb_dump();
    pie_rgb2esrgb_dump();
    pie_dsmf_dump();    
    pie_tcns_dump();
    pie_sccsc_dump();
    pie_cstats_dump();
    pie_ngadjust_dump();
    pie_decim_dump();
    pie_clippad_dump();    
    pie_bde_dump();
    pie_otmarb_dump();
}
EXPORT_SYMBOL(pie_dump);

//////////////////////
// clear irq function
//////////////////////
void pie_do_clear_all_irqs(void)
{
    // can't clear pie common, have to clear pogo to clear common irqs
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);

    ft->clear_pogo_idma_irqs(pie_subblock_data);
    ft->clear_pogo_odma_irqs(pie_subblock_data);

    do
    {
        PIE_RETRIEVE_function_table(pieOTMarbData, pie_otmarb_function_struct, otmarb_handle);
        ft->clear_otmarb_irqs(pie_subblock_data, NULL);
    } while(0);
}
EXPORT_SYMBOL(pie_do_clear_all_irqs);


// PIE SC CSC APIs that interface to SRAM (not register based), so
// they don't look like the typical interfaces.
void pie_sccsc_load_ilut(const uint32_t lut_data[], uint32_t lut_entries)
{
    PIE_RETRIEVE_function_table(sccscData, pie_sccsc_function_struct, sccsc_handle);
    ft->load_sccsc_ilut(pie_subblock_data, lut_data, lut_entries);
}
EXPORT_SYMBOL(pie_sccsc_load_ilut);

int pie_sccsc_dbg_verify_ilut()
{
    PIE_RETRIEVE_function_table_retval(sccscData, pie_sccsc_function_struct, sccsc_handle, 0);
    return ft->dbg_verify_sccsc_ilut(pie_subblock_data);
}
EXPORT_SYMBOL(pie_sccsc_dbg_verify_ilut);


///////////////////////////////////////////////////////
// ONLY FOR USE BY THE DMA CODE!!!! - NOT A GENERAL PUBLIC API
///////////////////////////////////////////////////////
// enable pie common interrupts
// caller sets the irqstruct field bit to true for each interrupt to enable
// irqstruct of NULL means to enable all interrupts
void pie_enable_pie_common_irq(struct pie_common_ints *irqstruct)
{
    PIE_RETRIEVE_function_table(pieCommonData, pie_common_function_struct, pie_common_handle);
    ft->enable_pie_common_irq(pie_subblock_data, irqstruct);
}
EXPORT_SYMBOL(pie_enable_pie_common_irq);

// disable pie common interrupts
// caller sets the irqstruct field bit to true for each interrupt to disable
// irqstruct of NULL means to disable all interrupts
void pie_disable_pie_common_irq(struct pie_common_ints *irqstruct)
{
    PIE_RETRIEVE_function_table(pieCommonData, pie_common_function_struct, pie_common_handle);
    ft->disable_pie_common_irq(pie_subblock_data, irqstruct);
}
EXPORT_SYMBOL(pie_disable_pie_common_irq);

bool pie_odma_is_enabled(void)
{
    PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                       pogo_dma_handle, false);
    return(ft->odma_enabled(pie_subblock_data));
}
EXPORT_SYMBOL(pie_odma_is_enabled);

bool pie_idma_is_enabled(uint8_t channel_num)
{
    PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                       pogo_dma_handle, false);
    return(ft->idma_enabled(pie_subblock_data, channel_num));
}
EXPORT_SYMBOL(pie_idma_is_enabled);

bool pie_odma_is_idle(void)
{
    PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                       pogo_dma_handle, false);
    return(ft->odma_idle(pie_subblock_data));
}
EXPORT_SYMBOL(pie_odma_is_idle);

bool pie_idma_is_idle(uint8_t channel_num)
{
    PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                       pogo_dma_handle, false);
    return(ft->idma_idle(pie_subblock_data, channel_num));
}
EXPORT_SYMBOL(pie_idma_is_idle);

// Caller passes in the physical address of the odma descriptor.
// The odma will be started
void pie_start_pogo_output_dma(dma_addr_t phys_desc)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct,
                                pogo_dma_handle);
    ft->start_pie_pogo_odma(pie_subblock_data, phys_desc);
}
EXPORT_SYMBOL(pie_start_pogo_output_dma);

// Caller passes in the physical address of the idma descriptor
// For a given channel.  The idma for that channel will be started
void pie_start_pogo_input_dma(dma_addr_t phys_desc, uint8_t channum)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->start_pie_pogo_idma(pie_subblock_data, phys_desc, channum);
}
EXPORT_SYMBOL(pie_start_pogo_input_dma);

void pie_start_distort_input_dma(dma_addr_t phys_desc, uint8_t channum)
{
    PIE_RETRIEVE_function_table(ddmaAcDataData, pie_ddma_ac_data_function_struct, ddma_ac_data_handle[channum]);
    ft->start_pie_distort_idma(pie_subblock_data, phys_desc);
}
EXPORT_SYMBOL(pie_start_distort_input_dma);

void pie_start_distort(struct pie_handle_t *pie_handle)
{
    PIE_RETRIEVE_function_table(distortData, pie_distort_function_struct, distort_handle);
    ft->start_pie_distort(pie_subblock_data, pie_handle);
}
EXPORT_SYMBOL(pie_start_distort);

// Caller passes in the physical address of the idma descriptor
// For a given channel.  The idma for that channel will be started
void pie_start_pogo_ot_idma(dma_addr_t phys_desc)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->start_pie_pogo_ot_idma(pie_subblock_data, phys_desc);
}
EXPORT_SYMBOL(pie_start_pogo_ot_idma);

// Caller passes in the physical address of the odma descriptor
// For a given channel.  The idma for that channel will be started
void pie_start_pogo_ot_odma(dma_addr_t phys_desc)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->start_pie_pogo_ot_odma(pie_subblock_data, phys_desc);
}
EXPORT_SYMBOL(pie_start_pogo_ot_odma);

///////////////////////////////////////////////////////
// only for use by the pie_common interrupt handler - not a public API
///////////////////////////////////////////////////////

void pie_handle_odma_irqs(void)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->handle_pogo_odma_irqs(pie_subblock_data);
}

void pie_handle_idma_irqs(void)
{
    PIE_RETRIEVE_function_table(piePogoData, pie_pogo_dma_function_struct, pogo_dma_handle);
    ft->handle_pogo_idma_irqs(pie_subblock_data);
}

void pie_handle_distort_irqs()
{
    PIE_RETRIEVE_function_table(distortData, pie_distort_function_struct, distort_handle);
    ft->handle_distort_irqs(pie_subblock_data);
}

void pie_handle_distort_idma_irqs(int instance)
{
    PIE_RETRIEVE_function_table(ddmaAcDataData, pie_ddma_ac_data_function_struct, ddma_ac_data_handle[instance]);
    ft->handle_distort_idma_irqs(pie_subblock_data);
}

void pie_handle_otmarb_irqs(void)
{
    PIE_RETRIEVE_function_table(pieOTMarbData, pie_otmarb_function_struct, otmarb_handle);
    ft->handle_otmarb_irqs(pie_subblock_data);
}

///////////////////////////////////////////////////////
// only for use by the pie driverlib - not a public API
///////////////////////////////////////////////////////
// we allocate - caller must deallocate.
// we return size of table (in bytes) or error
int pie_get_subblock_sizes_array(uint32_t **subblock_parm)
{
    enum pie_full_subblock_list i;
    int x;
    int allocsize;
    uint32_t *subblock_array;

    PIE_RETRIEVE_function_table_retval(piePogoData, pie_pogo_dma_function_struct,
                                       pogo_dma_handle, -ENODEV);

    allocsize = max_subblock_val * sizeof(uint32_t);
    subblock_array = vmalloc(allocsize);
    *subblock_parm = subblock_array;

    if (subblock_array == NULL)
    {
        error_print("%s:ERROR, could not allocate 0x%X bytes of kernel memory\n", __func__, allocsize);
        return -ENOMEM;
    }
    
    debug_print("%s: max_subblock=%d\n", __func__, max_subblock_val);

    // 0 out all subblocks as default
    for (i=common_index;i<max_subblock_val;i++)
        subblock_array[i] = 0;
    
    if (pie_object.pie_common_handle != NULL)
        subblock_array[common_index] = sizeof(PIE_COMMON_REGS_t);
        
    if (pie_object.pogo_dma_handle  != NULL)
        ft->pie_get_pogo_sizes_array(subblock_array, pie_subblock_data);
    
    if (pie_object.xyscaler_handle != NULL)
        subblock_array[xyscale_index] = sizeof(XYSCALE_REGS_t);

    if (pie_object.xy_c_scaler_handle != NULL)
        subblock_array[xy_c_scale_index] = sizeof(XYCSCALE_REGS_t);

    if (pie_object.colorshift_handle != NULL)
        subblock_array[colorshift_index] = sizeof(COLORSHIFT_REGS_t);

    if (pie_object.rgb2ycc_handle != NULL)
        subblock_array[es2y_index] = sizeof(ES2Y_REGS_t);

    if (pie_object.ycc2rgb_handle != NULL)
        subblock_array[y2es_index] = sizeof(Y2ES_REGS_t);

    if (pie_object.rgb2esrgb_handle != NULL)
        subblock_array[lut1d_rgb2esrgb_index] = sizeof(LUT1D_RGB2ESRGB_REGS_t);

    if (pie_object.dsmf_handle != NULL)
        subblock_array[dsmf_index] = sizeof(DSMF_REGS_t);
    
    if (pie_object.tcns_handle != NULL)
        subblock_array[tcnsense_index] = sizeof(TCNSENSE_REGS_t);

    if (pie_object.sccsc_handle != NULL)
        subblock_array[sccsc_index] = sizeof(SCCSC_REGS_t);

    if (pie_object.cstats_handle != NULL)
        subblock_array[cstats_index] = sizeof(CSTATS_REGS_t);
    
    if (pie_object.ngadjust_handle != NULL)
        subblock_array[ngadjust_index] = sizeof(NGADJUST_REGS_t);

    if (pie_object.istone_handle != NULL)
        subblock_array[istone_index] = sizeof(ISTONE_REGS_t);

    if (pie_object.denoise_handle != NULL)
        subblock_array[denoise_index] = sizeof(DENOISE_REGS_t);

    if (pie_object.decim_handle != NULL)
        subblock_array[decim_index] = sizeof(DECIM_REGS_t);
    
    if (pie_object.clippad_handle != NULL)
        subblock_array[clippad_index] = sizeof(CLIPPAD_REGS_t);
    
    if (pie_object.antifcor_handle != NULL)
        subblock_array[antifcor_index] = sizeof(DISTORT_TOP_ANTIFCOR_TOP_ANTIFCOR_REGS_t);
    
    if (pie_object.distort_handle != NULL)
        subblock_array[distort_index] = sizeof(DISTORT_TOP_DISTORT_REGS_t);
    
    if (pie_object.dismap_handle != NULL)
        subblock_array[dismap_index] = sizeof(DISTORT_TOP_DISTORT_MAP_REGS_t);

    for (x=0;x<NUM_BDE_CHANNELS;x++)
    {
        i = lut1d_bde0_index + x;
        if (pie_object.bde_handle[x] != NULL)
            subblock_array[i] = sizeof(DISTORT_TOP_ANTIFCOR_TOP_LUT1D_BDE0_REGS_t);
    }

    if (pie_object.ihi_handle != NULL)
        subblock_array[ihi_index] = sizeof(IHI_REGS_t);

    if (pie_object.ihi_depogo_handle != NULL)
        subblock_array[ihidepogoizer_index] = sizeof(IHIDEPOGOIZER_REGS_t);

    if (pie_object.pogo_ihi_odma_udma_handle != NULL)
        subblock_array[pogo_ihi_odma_udma_index] = sizeof(POGO_IDMA0_POGO_IDMA_UDMA_REGS_t);

    if (pie_object.pogo_ihi_odma_core_handle != NULL)
        subblock_array[pogo_ihi_odma_core_index] = sizeof(POGO_IDMA0_POGO_IDMA_CORE_REGS_t);
    
    for (x=0;x<NUM_PIE_DDMA_CHANNELS;x++)
    {
        i = ddma_ac_data0_index + x;
        if (pie_object.ddma_ac_data_handle[x] != NULL)
            subblock_array[i] = sizeof(DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_DATA0_REGS_t);

        i = ddma_ac_corr0_index + x;
        if (pie_object.ddma_ac_corr_handle[x] != NULL)
            subblock_array[i] = sizeof(DISTORT_TOP_ANTIFCOR_TOP_DDMA_AC_CORR0_REGS_t);
    }
    
    if (pie_object.otpogoidma_udma_handle != NULL)
        subblock_array[ot_pogo_idma_udma_index] = sizeof(OT_POGO_IDMA_POGO_IDMA_UDMA_REGS_t);

    if (pie_object.otpogoidma_core_handle != NULL)
        subblock_array[ot_pogo_idma_core_index] = sizeof(OT_POGO_IDMA_POGO_IDMA_CORE_REGS_t);

    if (pie_object.otpogoodma_udma_handle != NULL)
        subblock_array[pogo_otodma_udma_index] = sizeof(POGO_OTODMA_POGO_ODMA_UDMA_REGS_t);

    if (pie_object.otpogoodma_core_handle != NULL)
        subblock_array[pogo_otodma_core_index] = sizeof(POGO_OTODMA_POGO_ODMA_UDMA_REGS_t);
    
    if (pie_object.otpogoizer_handle != NULL)
        subblock_array[otpogoizer_index] = sizeof(OTPOGOIZER_REGS_t);

    if (pie_object.otdepogoizer_handle != NULL)
        subblock_array[otdepogoizer_index] = sizeof(OTDEPOGOIZER_REGS_t);

    if (pie_object.otmarb_handle != NULL)
        subblock_array[otmarb_index] = sizeof(OTMARB_REGS_t);

    return allocsize;
}
EXPORT_SYMBOL(pie_get_subblock_sizes_array);
