/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/of_device.h>
#include <linux/dma-mapping.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/string.h>  // for strcpy
#include <linux/vmalloc.h>

#include "pie_data.h"
#include "pie_pltfm.h"
#include "pie_full_subblock_list.h"

//#define DURING_DEBUG 1

#ifdef DURING_DEBUG
  #define debug_print(args...) printk(args)
#else
#define debug_print(args...)
#endif

struct device *global_dev;

// structure to store state of block during suspend/resume
struct device_suspend_context_struct
{
    int placeholder;  // what needs to be stored in suspend for this block?  FIXME TODO
};

extern void init_pie(void); // pie.c needs to know about the start of a probe
extern void exit_pie(void); // pie.c needs to know about the exit
extern int pie_get_subblock_sizes_array(uint32_t **copybuf);
extern void fixup_shadow_pointers(void *pie_handle);
extern int pie_do_revcheck(void *pie_handle);
        
static char *revcheck_status="unknown";   // revcheck hasn't run yet

module_param(revcheck_status, charp, S_IRUSR);

// binary sysfs entry to allow user space to give the pie handle to kernel space
// and have the kernel check the revision
static ssize_t write_pie_handle(struct file *filp, struct kobject *kobj,
                                struct bin_attribute *attr,
                                char *buf, loff_t offset, size_t count)
{
    static uint32_t *pie_handle_ptr;
    static int first_write = 1;
    static int pie_handle_size;
    uint8_t *curr_pie_handle_byte_ptr;
    uint32_t *tmp_ptr;
    
    if (first_write)
    {
        strcpy(revcheck_status, "pending");  // revcheck has started
        tmp_ptr = (uint32_t *) buf;
        pie_handle_size = tmp_ptr[0];

        // create a pie_handle to pass to the driver for a revision check
        pie_handle_ptr = (uint32_t *) vmalloc(pie_handle_size+1); // +1 for margin
        first_write = 0;
    }

    curr_pie_handle_byte_ptr = (uint8_t *) pie_handle_ptr;
    curr_pie_handle_byte_ptr += offset;  // move to the current location in our handl
    
    memcpy(curr_pie_handle_byte_ptr, buf, count);

    if (offset+count >= pie_handle_size)
    {

        // we're done copying in data from the user, set up for next time
        // the user writes the pie_handle
        first_write = 1;
        
        // when the pie handle is passed in, the pointers are invalid (user space),
        // so we fix them for kernel space
        fixup_shadow_pointers(pie_handle_ptr);

        if (pie_do_revcheck(pie_handle_ptr) == -1)
        {
            printk("Error!!  revcheck failed\n");
            
            strcpy(revcheck_status, "error");
        }
        else
            strcpy(revcheck_status, "passed");
    }

    return count;
}

// binary sysfs entry to allow user space to give the pie handle to kernel space
static ssize_t read_pie_handle(struct file *filp, struct kobject *kobj,
                                struct bin_attribute *attr,
                                char *buf, loff_t offset, size_t count)
{
#if 0    
    static uint32_t *pie_handle_ptr = NULL;
    static int first_read = 1;
    static int all_data_returned = 0;
    static int pie_handle_size = 0;
    uint8_t *curr_pie_handle_byte_ptr;
    int i;
#endif    
    
#if 1
    printk("FIXME, need to have a function to call to retrieve the pie_handle\n");
    printk("How to figure out which pie_handle you want, and how to get the ptr\n");
    return 0;
#else
// commented out until we have a function to call to return the requested pie_handle
    if (all_data_returned)
    {
        all_data_returned = 0;
        first_read = 1; // we're done, set up for next time a read is called
        return 0;  // tell caller we finished the read
    }
    
    if (first_read)
    {
        pie_handle_ptr = get_pie_handle_function_must_be_written();
        pie_handle_size = pie_handle_ptr[0];
        first_read = 0;
    }
    // When we are called, the system tells us what offset from the start of our buffer we are at.
    // First time, offset=0, count=4096, next time, offset=4096, count 4096, next time offset=8192....
    curr_pie_handle_byte_ptr = (uint8_t *) pie_handle_ptr;
    curr_pie_handle_byte_ptr += offset;
    
    if (offset+count >= pie_handle_size)
    {
        int holdcount;
        // we've sent back all the data except this last buffer 
        all_data_returned = 1;

        // calculate how many bytes we're really returning this time
        holdcount = count;
        count = pie_handle_size % count;

        // handle the case where our pie_handle_size is the exact multiple of system count
        if (count == 0)
            count = holdcount;
    }

    for (i=0;i<count;i++)
    {
        // copy only as much as they have allocated
        // (or as much as we have left)
        memcpy(buf, curr_pie_handle_byte_ptr, count);
    }

    return count;  // tell caller we have return 
    
#endif    

}

// binary sysfs entry to allow user space to ask for the subblock size array
static ssize_t retrieve_subblock_sizes_array(struct file *filp, struct kobject *kobj,
                                             struct bin_attribute *attr,
                                             char *buf, loff_t offset, size_t size)
{
    uint32_t *copybuf;
    int count;

    if (offset >= max_subblock_val * sizeof(uint32_t))
        return 0;

    // the size of the data to send to user space is well under the 4K page size,
    // so no worries about having to do multiple copies to user space
    count = pie_get_subblock_sizes_array(&copybuf);

    if (count < 0)
    {
        printk("ERROR, pie_get_subblock_sizes_array returned %d\n", count);
        return 0;
    }
    
    memcpy(buf, copybuf, count); 

    vfree(copybuf);  // pie_get_subblock_sizes_array allocates memory, we must deallocate
    return count;
}
// reading from pie_handle gives the subblock sizes array, writing to it sends down
// the completed pie_handle
BIN_ATTR(pie_handle_revcheck, S_IRUGO | S_IWUSR, read_pie_handle, write_pie_handle, 0);
BIN_ATTR(subblock_sizes_array, S_IRUGO, retrieve_subblock_sizes_array, NULL, max_subblock_val * sizeof(uint32_t));

// functions/struct only used for debug and bringup - not needed in normal driver use

extern void pie_dump(void);
ssize_t debug_test_write1(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
    pie_dump();
    return count;
}

DEVICE_ATTR(pietest, S_IWUSR | S_IRUGO, NULL, debug_test_write1);

const struct attribute *pietest_attrs[] = {
    &dev_attr_pietest.attr,
    NULL,
};

// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
static struct device_data *init_subblock(char *subblock_name,
                                         void (*init_func)(struct device_data *),
                                         void (*exit_func)(struct device_data *),
                                         int instance,
                                         int irq, 
                                         struct device_node *parent_node,
                                         struct list_head *device_data_list)
{
    struct resource res;
    struct device_node *child_node;
    struct device_data *device_data;
    int retval;

    // now this is for each child
    child_node = of_get_child_by_name(parent_node, subblock_name);
    if (child_node == NULL)
    {
        debug_print("no child by name of %s\n", subblock_name);
        return NULL; // no child by that name, just return
    }

    retval = of_address_to_resource(child_node, 0, &res);
    if (retval < 0)
    {
        dev_err(global_dev, "%s can't get resource for node, returned 0x%X\n", subblock_name, retval);
        goto early_failure;
    }

    device_data = (struct device_data *)kzalloc(sizeof(struct device_data), GFP_KERNEL);
    if (!device_data)
    {
        dev_err(global_dev, "kzalloc failed\n");
        goto early_failure;
    }

    if (!request_mem_region(res.start, resource_size(&res), subblock_name))
    {
        dev_err(global_dev, "%s request_mem_region failed: %s\n", subblock_name, __func__);
        goto request_mem_region_failed;
    } 

    device_data->virt_addr = of_iomap(child_node, 0); // get virtual address
    if (device_data->virt_addr == NULL)
    {
        dev_err(global_dev, "%s of_iomap failed: %s\n", subblock_name, __func__);
        device_data->virt_addr = 0;
        goto iomap_failed;
    }

    device_data->submodinstance = instance;
    device_data->phys_addr = res.start;
    device_data->addr_size = resource_size(&res); 
    device_data->irq = irq;  
    device_data->exit_func = exit_func;
    debug_print("%s: virt_addr=0x%p, phys_addr=0x%X, submod=%d\n", subblock_name,
                device_data->virt_addr, device_data->phys_addr,
                 device_data->submodinstance);
    
    // if we are a subblock that has its own IRQ (the calling function knows this), 
    // then map the interrupt service routine
    if (irq > -1)
    {
        retval = request_threaded_irq(irq, NULL,
                                      pie_platform_irq,
                                      IRQF_ONESHOT, "PIE", device_data);
        if (retval)
        {
            dev_err(global_dev, "request_irq failed\n");
            goto request_irq_failed;
        }
    }

    // add the device_data to the linked list
    list_add((struct list_head *)&device_data->linked_list, device_data_list);

    // call the initialization function (which should register with the top level pie driver)
    init_func(device_data);

    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return device_data;

request_irq_failed:
    device_data->irq = 0;

    iounmap(device_data->virt_addr); 
iomap_failed:
    release_mem_region(device_data->phys_addr, device_data->addr_size); 
request_mem_region_failed:
    kfree(device_data);
early_failure:
    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return NULL;
}

static int pie_platform_probe(struct platform_device *pdev)
{
    int retval;
    int irq;
    struct device_node *parent_node = pdev->dev.of_node;

    int ignore;
    struct device_data *device_data;
    struct list_head *device_data_list;

    // initialize the head ptr for the whole linked list of device_data
    global_dev = &pdev->dev; // for error print statements (dev_err)
    
    if (!parent_node)
    {
        dev_err(global_dev, "Could not find device node %s\n", "pie");
        return -ENODEV;
    }

    device_data_list = kmalloc(sizeof(struct list_head), GFP_KERNEL);
    if (device_data_list == NULL)
    {
        dev_err(global_dev, "Could not allocate list_head\n");
        return -ENOMEM;
    }

    INIT_LIST_HEAD(device_data_list);
    
    // get the irq from the parent node (only 1 for all of pie)
    irq = irq_of_parse_and_map(parent_node, 0);
    if (irq < 0)
    {
        dev_err(global_dev, "platform_get_irq failed\n");
        retval = -ENXIO;
        goto early_failure;
    }
    debug_print("Probe in %s: irq = %d\n", __func__, irq);

    // tell pie.c we're starting up now.  
    init_pie();
    
    // pie has many subblocks - parse them all out of the device tree, allocate device_data
    // for each of them, put that data on the device_data_list, and call their
    // respective init functions in the main driver code

    device_data = init_subblock("pie-common", pie_common_init, pie_common_exit, -1, irq,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma0-udma", pie_pogo_idma_udma_init, pie_pogo_idma_udma_exit, 0, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma0-core", pie_pogo_idma_core_init, pie_pogo_idma_core_exit, 0, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma1-udma", pie_pogo_idma_udma_init, pie_pogo_idma_udma_exit, 1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma1-core", pie_pogo_idma_core_init, pie_pogo_idma_core_exit, 1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma2-udma", pie_pogo_idma_udma_init, pie_pogo_idma_udma_exit, 2, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-idma2-core", pie_pogo_idma_core_init, pie_pogo_idma_core_exit, 2, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogoizer", pie_pogoizer_init, pie_pogoizer_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-ot-pogo-idma-udma", pie_pogo_otidma_udma_init, pie_pogo_otidma_udma_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-ot-pogo-idma-core", pie_pogo_otidma_core_init, pie_pogo_otidma_udma_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-otpogoizer", pie_otpogoizer_init, pie_otpogoizer_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-colorshift", pie_colorshift_init, pie_colorshift_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-lut1d-rgb2esrgb", pie_rgb2esrgb_init, pie_rgb2esrgb_exit, -1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-sccsc", pie_sccsc_init, pie_sccsc_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-es2y", pie_rgb2ycc_init, pie_rgb2ycc_exit, -1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-cstats", pie_cstats_init, pie_cstats_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-otmarb", pie_otmarb_init, pie_otmarb_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-ngadjust", pie_ngadjust_init, pie_ngadjust_exit, -1, -1,
                                parent_node, device_data_list);
    //device_data = init_subblock("pie-istone", pie_istone_init, pie_istone_exit, -1, -1,
    //                            parent_node, device_data_list);
    device_data = init_subblock("pie-tcnsense", pie_tcns_init, pie_tcns_exit, -1, -1,
                                parent_node, device_data_list);
    //device_data = init_subblock("pie-denoise", pie_denoise_init, pie_denoise_exit, -1, -1,
    //                            parent_node, device_data_list);
    device_data = init_subblock("pie-dsmf", pie_dsmf_init, pie_dsmf_exit, -1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-decim", pie_decim_init, pie_decim_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-y2es", pie_ycc2rgb_init, pie_ycc2rgb_exit, -1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-xyscale", pie_xyscaler_init, pie_xyscaler_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-clippad", pie_clippad_init, pie_clippad_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-depogoizer", pie_depogoizer_init, pie_depogoizer_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-odma-udma", pie_pogo_odma_udma_init, pie_pogo_odma_udma_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-odma-core", pie_pogo_odma_core_init, pie_pogo_odma_core_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-otdepogoizer", pie_otdepogoizer_init, pie_otdepogoizer_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-otodma-udma", pie_pogo_otodma_udma_init, pie_pogo_otodma_udma_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-pogo-otodma-core", pie_pogo_otodma_core_init, pie_pogo_otodma_core_exit, -1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-lut1d-bde0", pie_bde_init, pie_bde_exit, 0, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-lut1d-bde1", pie_bde_init, pie_bde_exit, 1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-lut1d-bde2", pie_bde_init, pie_bde_exit, 2, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-antifcor", pie_antifcor_init, pie_antifcor_exit, -1, -1,
                                parent_node, device_data_list);    
    device_data = init_subblock("pie-ddma-ac-data0", pie_ddma_ac_data_init, pie_ddma_ac_data_exit, 0, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-ddma-ac-data1", pie_ddma_ac_data_init, pie_ddma_ac_data_exit, 1, -1,
                                parent_node, device_data_list);
    device_data = init_subblock("pie-ddma-ac-data2", pie_ddma_ac_data_init, pie_ddma_ac_data_exit, 2, -1,
                                parent_node, device_data_list);
    //device_data = init_subblock("pie-ddma-ac-corr0", pie_ddma_ac_corr0_init, pie_ddma_ac_corr0_exit, 0, -1,
    //                            parent_node, device_data_list);
    //device_data = init_subblock("pie-ddma-ac-corr1", pie_ddma_ac_corr1_init, pie_ddma_ac_corr1_exit, 1, -1,
    //                            parent_node, device_data_list);
    //device_data = init_subblock("pie-ddma-ac-corr2", pie_ddma_ac_corr2_init, pie_ddma_ac_corr2_exit, 2, -1,
    //                            parent_node, device_data_list);
    device_data = init_subblock("pie-distort", pie_distort_init, pie_distort_exit, -1, -1,
                                parent_node, device_data_list);
    //device_data = init_subblock("pie-dis-map", pie_dis_map_init, pie_dis_map_exit, -1, -1,
    //                            parent_node, device_data_list);
    device_data = init_subblock("pie-xycscale", pie_xycscaler_init, pie_xycscaler_exit, -1, -1,
                                parent_node, device_data_list);

    //device_data = init_subblock("pie-ihi", pie_ihi_init, pie_ihi_exit, -1,
    //                            -1, parent_node, device_data_list);
    //device_data = init_subblock("pie-ihidepogoizer", pie_ihidepogoizer_init, pie_ihidepogoizer_exit, -1,
    //                            -1, parent_node, device_data_list);
    //device_data = init_subblock("pie-pogo-ihi-odma-udma", pie_pogo_ihi_odma_udma_init,
    //                            pie_pogo_ihi_odma_udma_exit, -1, -1, parent_node, device_data_list);
    //device_data = init_subblock("pie-pogo-ihi-odma-core", pie_pogo_ihi_odma_core_init,
    //                            pie_pogo_ihi_odma_core_exit, -1, -1, parent_node, device_data_list);

    platform_set_drvdata(pdev, device_data_list); // store our data in the device struct
    // for debugging and bringup to read registers- not needed for everyday driver work
    ignore = sysfs_create_files(&(pdev->dev.kobj), pietest_attrs);
    // allow userspace to access the pie subblock sizes array as a *binary* value
    ignore = sysfs_create_bin_file(&(pdev->dev.kobj), &bin_attr_pie_handle_revcheck);
    ignore = sysfs_create_bin_file(&(pdev->dev.kobj), &bin_attr_subblock_sizes_array);    
    
    return 0;
    
early_failure:    
    return retval;
}


static int pie_platform_remove(struct platform_device *pdev)
{
    // get linked list of device_datas
    struct list_head *device_data_list_ptr = platform_get_drvdata(pdev); 
    struct device_data *device_data, *dd_next; // dd_next is used by list function/macro

    list_for_each_entry_safe(device_data, dd_next, device_data_list_ptr, linked_list)
    {
        // items were added to the list LIFO, and we remove FIFO
        list_del(&device_data->linked_list);
        device_data->exit_func(device_data);  // call exit function for driver subblock
        iounmap(device_data->virt_addr);
        release_mem_region(device_data->phys_addr, device_data->addr_size);
        if (device_data->irq > 0)
            free_irq(device_data->irq, device_data); 
        kfree(device_data);
    }
    // tell pie.c we're closing down now.
    exit_pie();

    kfree(device_data_list_ptr);

    platform_set_drvdata(pdev, NULL);
    sysfs_remove_files(&(pdev->dev.kobj), pietest_attrs); 
    sysfs_remove_bin_file(&(pdev->dev.kobj), &bin_attr_pie_handle_revcheck);
    sysfs_remove_bin_file(&(pdev->dev.kobj), &bin_attr_subblock_sizes_array);    
    debug_print("%s: complete\n",__func__);
    return 0;
}

static int pie_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    printk(KERN_WARNING "%s event = %d\n",__func__, state.event);

    // allocate suspend context structure
    device_data->device_suspend_context = kmalloc(sizeof(struct device_suspend_context_struct), GFP_KERNEL);
    if (device_data->device_suspend_context == NULL)
    {
        return -1;
    }
    
    return (0);
}

static int pie_platform_resume(struct platform_device *pdev)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    printk(KERN_WARNING "%s\n",__func__);

    if (device_data->device_suspend_context == NULL)
    {
        printk(KERN_ERR "%s: invalid suspend context\n",__func__);
        return -1;
    }
    
    // restore block context here TODO FIXME what to do?

    // free the suspend context
    kfree(device_data->device_suspend_context);
    device_data->device_suspend_context = NULL;
    return(0);
}



#ifdef CONFIG_OF
static const struct of_device_id mrvl_pie_dt_match[] = {

    { .compatible = "marvell,g2-imgpipe-pie"},
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_pie_dt_match);
#endif

static struct platform_driver pie_platform_driver =
{
    .probe   = pie_platform_probe,
    .remove  = pie_platform_remove,
    .suspend = pie_platform_suspend,
    .resume = pie_platform_resume,
//    .id_table = mrvl_pie_driver_ids, ???
    .driver  = {
        .name  = "pie_block",
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_pie_dt_match),
    }
};

module_platform_driver(pie_platform_driver);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell PIE subblock driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_Jul_15");

