/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t
#include <linux/io.h> // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>  // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h" // needed for detailed pic_handle
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_wdma_channel.h"
#include "pic_wdma_channel_if.h"
#include "pic_output_dma_driver.h"
#include "pic_driverlib_if.h"

// SANDRA, CAN WE REMOVE THESE FUNCTIONS?
static uint32_t clear_wdma_channel_irqs_uint32(picWdmaChannelData *device_data,
                                               uint32_t int_val)
{
    unsigned long flags;  
    uint32_t reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = picWDMAChannelRead(int_pend);
    picWDMAChannelWrite(int_ack, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// clear requested (or all set) interrupts - return the ones that
// were set 
// 
void clear_wdma_channel_irqs(picWdmaChannelData *device_data,
                             struct pic_output_dma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_wdma_channel_irqs_uint32(device_data, 0xFFFFFFFF);  // just clear everything and get out
        return; // ignoring the reg return
    }
    irqs_to_clear = pic_output_convert_irqstruct_to_update_reg(irqstruct, 0, true);
    reg = clear_wdma_channel_irqs_uint32(device_data, irqs_to_clear);

    // set fields for all irqs that were set
    pic_output_convert_uintarray_to_irqstruct(irqstruct, reg);
    irqstruct->pic_instance = device_data->instance;
    irqstruct->chan_num = device_data->submodinstance;    
    irqstruct->int_array = reg; // for debug only
}

// enable each field that is set
void enable_wdma_channel_irqs(picWdmaChannelData *device_data,
                              struct pic_output_dma_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pic_output_dma_interrupt_info irq;

    if (irqstruct == NULL)
    {
        pic_output_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    } 
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = picWDMAChannelRead(int_en);
    reg = pic_output_convert_irqstruct_to_update_reg(irqstruct, reg, true);
    picWDMAChannelWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

// disable each field that is set
void disable_wdma_channel_irqs(picWdmaChannelData *device_data,
                               struct pic_output_dma_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pic_output_dma_interrupt_info irq;

    if (irqstruct == NULL)
    {
        pic_output_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    } 
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = picWDMAChannelRead(int_en);
    reg = pic_output_convert_irqstruct_to_update_reg(irqstruct, reg, false);
    picWDMAChannelWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}


///  NOTE - This function runs in interrupt thread context - try to be brief
void handle_wdma_channel_irqs(picWdmaChannelData *device_data)
{
    uint32_t int_val;
    struct pic_output_dma_interrupt_info irqstruct;
        
    int_val = clear_wdma_channel_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    if (int_val != 0)
    {
        pic_output_convert_uintarray_to_irqstruct(&irqstruct, int_val);
        irqstruct.pic_instance = device_data->instance;
        irqstruct.chan_num = device_data->submodinstance;
        irqstruct.int_array = int_val;
        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

bool is_enabled_wdma_channel(picWdmaChannelData *device_data)
{
    bool rc = false;
    uint32_t reg;

    reg= picWDMAChannelRead(cfg);
    if (PIC_WDMA_DMA0_CFG_ENABLE_MASK_SHIFT(reg)) rc = true;
    return rc;
}

// get the wdma_channel status register
void get_status_wdma_channel(picWdmaChannelData *device_data, struct pic_wdma_status_info *info)
{
    uint32_t reg;

    reg = picWDMAChannelRead(status);
    info->bytes_waiting = PIC_WDMA_DMA0_STATUS_BYTESWAITING_MASK_SHIFT(reg);
    info->force_burst_busy = PIC_WDMA_DMA0_STATUS_FORCE_BURST_BUSY_MASK_SHIFT(reg);
    info->dma_paused = PIC_WDMA_DMA0_STATUS_DMAPAUSED_MASK_SHIFT(reg);
    info->pause_after = PIC_WDMA_DMA0_STATUS_PAUSEAFTER_MASK_SHIFT(reg);
    info->pause_before = PIC_WDMA_DMA0_STATUS_PAUSEBEFORE_MASK_SHIFT(reg);
    info->desc_own = PIC_WDMA_DMA0_STATUS_DESC_OWN_MASK_SHIFT(reg);
    info->desc_eoi = PIC_WDMA_DMA0_STATUS_DESC_EOI_MASK_SHIFT(reg);
    info->desc_soi = PIC_WDMA_DMA0_STATUS_DESC_SOI_MASK_SHIFT(reg);
    info->reverse = PIC_WDMA_DMA0_STATUS_REVERSE_MASK_SHIFT(reg);
    info->softreset = PIC_WDMA_DMA0_STATUS_SOFTRESET_MASK_SHIFT(reg);
    info->full_cbuf = PIC_WDMA_DMA0_STATUS_FULL_CBUF_MASK_SHIFT(reg);
    info->empty_cbuf = PIC_WDMA_DMA0_STATUS_EMPTY_CBUF_MASK_SHIFT(reg);
    info->full_dbuf = PIC_WDMA_DMA0_STATUS_FULL_DBUF_MASK_SHIFT(reg);
    info->empty_dbuf = PIC_WDMA_DMA0_STATUS_EMPTY_DBUF_MASK_SHIFT(reg);
    info->packer_empty = PIC_WDMA_DMA0_STATUS_PACKER_EMPTY_MASK_SHIFT(reg);
    info->dma_busy = PIC_WDMA_DMA0_STATUS_DMA_BUSY_MASK_SHIFT(reg);
    info->debug_array = reg;
}

void get_trans_len_wdma_channel(picWdmaChannelData *device_data, uint32_t *trans_len)
{
    *trans_len = picWDMAChannelRead(xfer_length);
}

// start the pic wdma
int set_desc_write_wdma_channel(picWdmaChannelData *device_data, dma_addr_t phys_addr)
{
    // write-only register, only 1 field to set
    if (phys_addr & 0x3)
    {
        print("%s bad address - must be a word address, not 0x%X\n", __func__, phys_addr);
        return -1;
    }
    picWDMAChannelWrite(desc_write, phys_addr);
    return 0;
}

static void wdma_channel_dump_regs(picWdmaChannelData *device_data)
{
    print("WDMA regs for pic %d channel %d\n",
                 device_data->instance, device_data->submodinstance);
    print("cfg=0x%08X    ",picWDMAChannelRead(cfg));
    print("status=0x%08X ",picWDMAChannelRead(status));
    print("line_size=0x%08X\n",picWDMAChannelRead(line_size));
    print("int_en=0x%08X  ",picWDMAChannelRead(int_en));
    print("int_pend=0x%08X ",picWDMAChannelRead(int_pend));
    print("desc_read=0x%08X\n",picWDMAChannelRead(desc_read));
    print("xfer_length=0x%08X  ",picWDMAChannelRead(xfer_length));
    print("xfer_addr=0x%08X    ",picWDMAChannelRead(xfer_addr));
    print("xfer_burst=0x%08X\n",picWDMAChannelRead(xfer_burst));
    print("REV0=0x%X\n",picWDMAChannelRead(REV0));
    print("REV1=0x%X\n",picWDMAChannelRead(REV1));
}


static void pic_configure(picWdmaChannelData *device_data, struct pic_handle_t *pic_handle)
{
    device_data->interrupt_callback = pic_handle->pic_odma_callback[device_data->submodinstance];
    device_data->interrupt_callback_data = pic_handle->pic_odma_callback_data[device_data->submodinstance];
    
    PROTECT_REG_ACCESS;
    picWDMAChannelWrite(cfg, pic_handle->pic_wdma_channel[device_data->submodinstance]->cfg);
    // status is read only
    picWDMAChannelWrite(line_size, pic_handle->pic_wdma_channel[device_data->submodinstance]->line_size);
    // ack interrupts before enabling....
    picWDMAChannelWrite(int_ack, pic_handle->pic_wdma_channel[device_data->submodinstance]->int_ack);
    picWDMAChannelWrite(int_en, pic_handle->pic_wdma_channel[device_data->submodinstance]->int_en);
    // int_pend is read only
    picWDMAChannelWrite(int_fo, pic_handle->pic_wdma_channel[device_data->submodinstance]->int_fo);
    // configure does NOT write to desc_write - it would start the dma going
    // desc_read, xfer_length, xfer_addr, xfer_burst are read only
    picWDMAChannelWrite(reset, pic_handle->pic_wdma_channel[device_data->submodinstance]->reset);
    // REV0 and REV1 are read only

    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picWdmaChannelData *device_data, struct pic_handle_t *pic_handle)
{
    uint8_t chan;
    
    chan = device_data->submodinstance;

    PROTECT_REG_ACCESS;
    pic_handle->pic_wdma_channel[chan]->cfg = picWDMAChannelRead(cfg);
    pic_handle->pic_wdma_channel[chan]->status = picWDMAChannelRead(status);
    pic_handle->pic_wdma_channel[chan]->line_size = picWDMAChannelRead(line_size);
    pic_handle->pic_wdma_channel[chan]->int_en = picWDMAChannelRead(int_en);
    pic_handle->pic_wdma_channel[chan]->int_pend = picWDMAChannelRead(int_pend);
    pic_handle->pic_wdma_channel[chan]->desc_read = picWDMAChannelRead(desc_read);
    pic_handle->pic_wdma_channel[chan]->xfer_length = picWDMAChannelRead(xfer_length);
    pic_handle->pic_wdma_channel[chan]->xfer_addr = picWDMAChannelRead(xfer_addr);
    pic_handle->pic_wdma_channel[chan]->xfer_burst = picWDMAChannelRead(xfer_burst);
    pic_handle->pic_wdma_channel[chan]->REV0 = picWDMAChannelRead(REV0);
    pic_handle->pic_wdma_channel[chan]->REV1 = picWDMAChannelRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picWdmaChannelData *device_data)
{
    uint32_t reg = 0; // RESET is a write only register. assume 0 by default
    
    PROTECT_REG_ACCESS;
    reg = PIC_WDMA_DMA0_RESET_SOFT_RESET_REPLACE_VAL(reg, 1);
    reg = PIC_WDMA_DMA0_RESET_FORCE_LAST_BURST_REPLACE_VAL(reg, 1);
    picWDMAChannelWrite(reset, reg);
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picWdmaChannelData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;
    uint8_t chan;

    chan = device_data->submodinstance;

    rev0 = PIC_WDMA_DMA0_REV0_MAJ_MASK_SHIFT(picWDMAChannelRead(REV0));
    pich_rev = PIC_WDMA_DMA0_REV0_MAJ_MASK_SHIFT(pic_handle->pic_wdma_channel[chan]->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_wdma_channel_function_struct pic_wdma_channel_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,

    .pic_dump_regs = wdma_channel_dump_regs,

    .clear_wdma_channel_irqs = clear_wdma_channel_irqs,
    .enable_wdma_channel_irqs = enable_wdma_channel_irqs,
    .disable_wdma_channel_irqs = disable_wdma_channel_irqs,    
    .handle_wdma_channel_irqs = handle_wdma_channel_irqs,
    .start_wdma_channel = set_desc_write_wdma_channel,
    .get_status_wdma_channel = get_status_wdma_channel,
    .is_enabled_wdma_channel = is_enabled_wdma_channel,
    .get_trans_len_wdma_channel = get_trans_len_wdma_channel
};

void pic_wdma_channel_init(picWdmaChannelData *device_data)
{
    picWdmaChannelDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(picWdmaChannelDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_wdma_channel_functions;
    pic_device_handle->device_data = device_data;
    register_pic_output_dma_device_data(wdma_channel, pic_device_handle,
                                        device_data->instance, device_data->submodinstance);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pic_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));    
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_wdma_channel_init);

void pic_wdma_channel_exit(picWdmaChannelData *device_data)
{
    picWdmaChannelDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_output_dma_device_data(wdma_channel,
                                                              device_data->instance,
                                                              device_data->submodinstance);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_wdma_channel_exit);
