/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t,
#include <linux/io.h> // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h>  // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h" // needed for detailed pic_handle
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_wdma.h"
#include "pic_wdma_if.h"
#include "pic_output_dma_driver.h"
#include "pic_driverlib_if.h"

void dump_wdma_regs(picWdmaData *device_data)
{
    print("PIC WDMA Dump for pic %d\n", device_data->instance);
    print("DWOCR=0x%X\n",picWDMARead(DWOCR)); 
    print("DWRTBL=0x%X\n",picWDMARead(DWRTBL));
    print("DWCIS=0x%X\n",picWDMARead(DWCIS)); 
    print("REV0=0x%X\n",picWDMARead(REV0)); 
    print("REV1=0x%X\n",picWDMARead(REV1)); 
}

static void pic_configure(picWdmaData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    picWDMAWrite(DWOCR, pic_handle->pic_wdma->DWOCR);
    picWDMAWrite(DWRTBL, pic_handle->pic_wdma->DWRTBL);
    picWDMAWrite(DWCIS, pic_handle->pic_wdma->DWCIS);
    // REV0 and REV1 are read only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picWdmaData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_wdma->DWOCR = picWDMARead(DWOCR);
    pic_handle->pic_wdma->DWRTBL = picWDMARead(DWRTBL);
    pic_handle->pic_wdma->DWCIS = picWDMARead(DWCIS);
    pic_handle->pic_wdma->REV0 = picWDMARead(REV0);
    pic_handle->pic_wdma->REV1 = picWDMARead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picWdmaData *device_data)
{
    // nothing to do for resetting wdma
}

static int pic_revcheck(picWdmaData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = PIC_WDMA_REV0_MAJ_MASK_SHIFT(picWDMARead(REV0));
    pich_rev = PIC_WDMA_REV0_MAJ_MASK_SHIFT(pic_handle->pic_wdma->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}


struct pic_wdma_function_struct pic_wdma_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck    = pic_revcheck,
    
    .pic_dump_regs = dump_wdma_regs,
};

void pic_wdma_init(picWdmaData *device_data)
{
    picWdmaDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(picWdmaDeviceHandle), GFP_KERNEL);

    // register with the parent
    pic_device_handle->fcn_tbl = &pic_wdma_functions;
    pic_device_handle->device_data = device_data;
    register_pic_output_dma_device_data(wdma, pic_device_handle, device_data->instance, 0);

    spin_lock_init(&(device_data->reg_spinlock));
    // wdma has no interrupts associated with it, so no callback
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_wdma_init);

void pic_wdma_exit(picWdmaData *device_data)
{
    picWdmaDeviceHandle *pic_device_handle;
    
    // unregister with the parent
    pic_device_handle = unregister_pic_output_dma_device_data(wdma, device_data->instance, 0);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_wdma_exit);
