/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t
#include <linux/io.h>         // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_pd.h"
#include "pic_pd_if.h"
#include "pic_driverlib_if.h"

void dump_pic_regs(picPrnuDsnuData *device_data)
{
    print("PIC PD Dump for pic %d\n", device_data->instance);
    print("RLSCR  = 0x%X\n",  pic_pdRead(RLSCR)); 
    print("LSVCR  = 0x%X\n",  pic_pdRead(LSVCR)); 
    print("LMPCR0 = 0x%X\n",  pic_pdRead(LMPCR0)); 
    print("LMPCR1 = 0x%X\n",  pic_pdRead(LMPCR1)); 
    print("LMPCR2 = 0x%X\n",  pic_pdRead(LMPCR2)); 
    print("RSVCR  = 0x%X\n",  pic_pdRead(RSVCR)); 
    print("RSPCR0 = 0x%X\n",  pic_pdRead(RSPCR0)); 
    print("RSPCR1 = 0x%X\n",  pic_pdRead(RSPCR1)); 
    print("RSPCR2 = 0x%X\n",  pic_pdRead(RSPCR2)); 
    print("PDCR1  = 0x%X\n",  pic_pdRead(PDCR1)); 
    print("PDCR2  = 0x%X\n",  pic_pdRead(PDCR2));
    print("PONCR  = 0x%X\n",  pic_pdRead(PONCR));     
    print("QUAD   = 0x%X\n",  pic_pdRead(QUAD)); 
    print("DOFF0  = 0x%X\n",  pic_pdRead(DOFF0)); 
    print("DOFF1  = 0x%X\n",  pic_pdRead(DOFF1)); 
    print("DOFF2  = 0x%X\n",  pic_pdRead(DOFF2)); 
    print("DMULT0 = 0x%X\n",  pic_pdRead(DMULT0)); 
    print("DMULT1 = 0x%X\n",  pic_pdRead(DMULT1)); 
    print("DMULT2 = 0x%X\n",  pic_pdRead(DMULT2)); 
    print("POFF0  = 0x%X\n",  pic_pdRead(POFF0)); 
    print("POFF1  = 0x%X\n",  pic_pdRead(POFF1)); 
    print("POFF2  = 0x%X\n",  pic_pdRead(POFF2)); 
    print("PMULT0 = 0x%X\n",  pic_pdRead(PMULT0)); 
    print("PMULT1 = 0x%X\n",  pic_pdRead(PMULT1)); 
    print("PMULT2 = 0x%X\n",  pic_pdRead(PMULT2)); 
    print("REV0   = 0x%X\n",  pic_pdRead(REV0)); 
    print("REV1   = 0x%X\n",  pic_pdRead(REV1)); 
}

static void pic_configure(picPrnuDsnuData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_pdWrite(RLSCR, pic_handle->pic_pd->RLSCR);
    pic_pdWrite(LSVCR, pic_handle->pic_pd->LSVCR);
    pic_pdWrite(LMPCR0, pic_handle->pic_pd->LMPCR0);
    pic_pdWrite(LMPCR1, pic_handle->pic_pd->LMPCR1);
    pic_pdWrite(LMPCR2, pic_handle->pic_pd->LMPCR2);
    pic_pdWrite(RSVCR, pic_handle->pic_pd->RSVCR);
    pic_pdWrite(RSPCR0, pic_handle->pic_pd->RSPCR0);
    pic_pdWrite(RSPCR1, pic_handle->pic_pd->RSPCR1);
    pic_pdWrite(RSPCR2, pic_handle->pic_pd->RSPCR2);
    pic_pdWrite(PDCR1, pic_handle->pic_pd->PDCR1);
    pic_pdWrite(PDCR2, pic_handle->pic_pd->PDCR2);
    pic_pdWrite(PONCR, pic_handle->pic_pd->PONCR);
    pic_pdWrite(QUAD, pic_handle->pic_pd->QUAD);
    pic_pdWrite(DOFF0, pic_handle->pic_pd->DOFF0);
    pic_pdWrite(DOFF1, pic_handle->pic_pd->DOFF1);
    pic_pdWrite(DOFF2, pic_handle->pic_pd->DOFF2);
    pic_pdWrite(DMULT0, pic_handle->pic_pd->DMULT0);
    pic_pdWrite(DMULT1, pic_handle->pic_pd->DMULT1);
    pic_pdWrite(DMULT2, pic_handle->pic_pd->DMULT2);
    pic_pdWrite(POFF0, pic_handle->pic_pd->POFF0);
    pic_pdWrite(POFF1, pic_handle->pic_pd->POFF1);
    pic_pdWrite(POFF2, pic_handle->pic_pd->POFF2);
    pic_pdWrite(PMULT0, pic_handle->pic_pd->PMULT0);
    pic_pdWrite(PMULT1, pic_handle->pic_pd->PMULT1);
    pic_pdWrite(PMULT2, pic_handle->pic_pd->PMULT2);
    // REV0, REV1 are read-only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picPrnuDsnuData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_pd->RLSCR = pic_pdRead(RLSCR);
    pic_handle->pic_pd->LSVCR = pic_pdRead(LSVCR);
    pic_handle->pic_pd->LMPCR0 = pic_pdRead(LMPCR0);
    pic_handle->pic_pd->LMPCR1 = pic_pdRead(LMPCR1);
    pic_handle->pic_pd->LMPCR2 = pic_pdRead(LMPCR2);
    pic_handle->pic_pd->RSVCR = pic_pdRead(RSVCR);
    pic_handle->pic_pd->RSPCR0 = pic_pdRead(RSPCR0);
    pic_handle->pic_pd->RSPCR1 = pic_pdRead(RSPCR1);
    pic_handle->pic_pd->RSPCR2 = pic_pdRead(RSPCR2);
    pic_handle->pic_pd->PDCR1 = pic_pdRead(PDCR1);
    pic_handle->pic_pd->PDCR2 = pic_pdRead(PDCR2);
    pic_handle->pic_pd->PONCR = pic_pdRead(PONCR);
    pic_handle->pic_pd->QUAD = pic_pdRead(QUAD);
    pic_handle->pic_pd->DOFF0 = pic_pdRead(DOFF0);
    pic_handle->pic_pd->DOFF1 = pic_pdRead(DOFF1);
    pic_handle->pic_pd->DOFF2 = pic_pdRead(DOFF2);
    pic_handle->pic_pd->DMULT0 = pic_pdRead(DMULT0);
    pic_handle->pic_pd->DMULT1 = pic_pdRead(DMULT1);
    pic_handle->pic_pd->DMULT2 = pic_pdRead(DMULT2);
    pic_handle->pic_pd->POFF0 = pic_pdRead(POFF0);
    pic_handle->pic_pd->POFF1 = pic_pdRead(POFF1);
    pic_handle->pic_pd->POFF2 = pic_pdRead(POFF2);
    pic_handle->pic_pd->PMULT0 = pic_pdRead(PMULT0);
    pic_handle->pic_pd->PMULT1 = pic_pdRead(PMULT1);
    pic_handle->pic_pd->PMULT2 = pic_pdRead(PMULT2);
    pic_handle->pic_pd->REV0 = pic_pdRead(REV0);
    pic_handle->pic_pd->REV1 = pic_pdRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picPrnuDsnuData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    /*
     * Due to a soft reset issue, two pixels of PRNU DSNU data get left
     * in the PNU2D block for PIC 1.  Putting the PRNU DSNU functions in
     * bypass causes the data from PNU2D to be discarded. This has to be
     * done in a separate write from a block bypass.
     */
    reg = 0;
    reg = PRNUDSNU_PDCR2_BYPASSDSNU_REPLACE_VAL(reg, 1);
    reg = PRNUDSNU_PDCR2_BYPASSPRNU_REPLACE_VAL(reg, 1);
    reg = PRNUDSNU_PDCR2_BYPASSQUAD_REPLACE_VAL(reg, 1);
    pic_pdWrite(PDCR2, reg);
    /* Now set it to block bypass. */
    reg = 0;
    reg = PRNUDSNU_PDCR2_BYPASS_REPLACE_VAL(reg, 1);
    pic_pdWrite(PDCR2, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picPrnuDsnuData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = PRNUDSNU_REV0_MAJ_MASK_SHIFT(pic_pdRead(REV0));
    pich_rev = PRNUDSNU_REV0_MAJ_MASK_SHIFT(pic_handle->pic_pd->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_pd_function_struct pic_pd_functions =
{
    .pic_reset       = pic_reset,
    .pic_configure   = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck    = pic_revcheck,
    
    .pic_dump_regs   = dump_pic_regs,
};

void pic_pd_init(picPrnuDsnuData *device_data)
{
    prnuDsnuDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(prnuDsnuDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_pd_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(pd, pic_device_handle, device_data->instance, 0);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
    // pd has no interrupts associated with it, so no callback
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_pd_init);


void pic_pd_exit(picPrnuDsnuData *device_data)
{
    prnuDsnuDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(pd, device_data->instance, 0);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_pd_exit);

