/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/slab.h>  // for memory allocation
#include <linux/export.h>  // for EXPORT_SYMBOL

#include "PIC_regheaders.h"

#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_driver.h"
#include "pic_data.h"
#include "pic_output_dma.h"
#include "pic_output_dma_if.h"
#include "pic_output_dma_driver.h"
#include "pic_wdma_if.h"
#include "pic_wdma_channel_if.h"
#include "pic_convenience_if.h"
#include "pic_driverlib_if.h"

// NOTE:  when creating new subblocks for outputdma, each must register with us
// using register_wdma_data()
// pic_output_dma consists of one pic_wdma instance and 3 pic_wdma_channel's instances.
// output_dma_data has private_data which contains 4 device handles, one for wdma and 3
// for wdma_channels.

#define CHECK4NULL(pointer)                                             \
    if (pointer == NULL)                                                \
    {                                                                   \
        print("ERROR %s, pointer %s is NULL\n", __func__,               \
              #pointer);                                                \
        BUG();                                                          \
    }

#define PIC_ODMA_RETRIEVE_function_table_data(dma_object, dmatype, instance, dev_data, ft, dh) \
    dh = dma_object->priv_data[dmatype+instance];                       \
    CHECK4NULL(dh);                                                     \
    dev_data = dh->device_data;                                         \
    ft = dh->fcn_tbl;                                                   \
    BUG_ON(dev_data == NULL);                                           \
    BUG_ON(ft == NULL);

// implementation of functions exported in function table

// clear all requested interrupts for the requested dma channel, and
// return the set ones in the structure
void clear_output_dma_irqs(outputDmaData *output_dma_object,
                           struct pic_output_dma_interrupt_info *irqstruct,
                           int dma_channel)
{
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, dma_channel,
                                          device_data_channel, ft_channel, dh_channel);
    ft_channel->clear_wdma_channel_irqs(device_data_channel, irqstruct);
}

void dump_channel_regs(outputDmaData *output_dma_object, char channel)
{
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaChannelDeviceHandle *dh_channel;
    struct pic_wdma_channel_function_struct *ft_channel;

    // dump requested channel regs

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);
    ft_channel->pic_dump_regs(device_data_channel);
}

void dump_all_regs(outputDmaData *output_dma_object)
{
    int i;
    picWdmaData *device_data_wdma = NULL;
    picWdmaDeviceHandle *dh_wdma;
    struct pic_wdma_function_struct *ft_wdma;

    // dump all the wdma regs
    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma, 0, device_data_wdma,
                                          ft_wdma, dh_wdma);
    ft_wdma->pic_dump_regs(device_data_wdma);

    // dump all 3 channels of wdma_channel regs
// FIXME - query how many channels instead of using constant
    for (i=0;i<MAX_OUTPUT_DMAS;i++)
    {
        dump_channel_regs(output_dma_object, i);
    }
}

int start_output_dma(outputDmaData *output_dma_object,
                     uint8_t channel, dma_addr_t phys_desc)
{
    picWdmaChannelData *device_data_channel = NULL;
    struct pic_wdma_channel_function_struct *ft_channel;
    picWdmaChannelDeviceHandle *dh_channel;    

    // start the output dma
    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);
    if (ft_channel->start_wdma_channel(device_data_channel, phys_desc) != 0)
    {
        error_print("ERROR!!! bad phys_addr of 0x%X\n", phys_desc);
        return -1;
    }
//    debug_print("Started odma %d at descriptor phys addr 0x%X\n", channel, phys_desc);
    return 0;
}

// do a soft reset on all channels and wdma block
void pic_reset(outputDmaData *output_dma_object)
{
    int channel;
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaData *device_data_wdma = NULL;    
    picWdmaDeviceHandle *dh_wdma;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;
    struct pic_wdma_function_struct *ft_wdma;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma, 0, device_data_wdma,
                                          ft_wdma, dh_wdma);
    ft_wdma->pic_reset(device_data_wdma);
    
    for (channel=0;channel<MAX_OUTPUT_DMAS;channel++)
    {
        PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                              device_data_channel, ft_channel, dh_channel);
        ft_channel->pic_reset(device_data_channel);
    }
}

// send the pic_handle down to configure
void pic_configure(outputDmaData *output_dma_object, struct pic_handle_t *pic_handle)
{
    int channel;
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaData *device_data_wdma = NULL;
    picWdmaDeviceHandle *dh_wdma;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;
    struct pic_wdma_function_struct *ft_wdma;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma, 0, device_data_wdma,
                                          ft_wdma, dh_wdma);
    ft_wdma->pic_configure(device_data_wdma, pic_handle);

    for (channel=0;channel<MAX_OUTPUT_DMAS;channel++)
    {
        PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                              device_data_channel, ft_channel, dh_channel);
        ft_channel->pic_configure(device_data_channel, pic_handle);
    }
}

// send the pic_handle down to fill with the current reg values
void pic_get_current(outputDmaData *output_dma_object, struct pic_handle_t *pic_handle)
{
    int channel;
    picWdmaData *device_data_wdma = NULL;
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaDeviceHandle *dh_wdma;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;
    struct pic_wdma_function_struct *ft_wdma;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma, 0, device_data_wdma,
                                          ft_wdma, dh_wdma);
    ft_wdma->pic_get_current(device_data_wdma, pic_handle);

    for (channel=0;channel<MAX_OUTPUT_DMAS;channel++)
    {
        PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                              device_data_channel, ft_channel, dh_channel);
        ft_channel->pic_get_current(device_data_channel, pic_handle);
    }
}

// send the pic_handle down to do a rev check against the real asic revs.
// if any of them fail, return the error immediately
int  pic_revcheck(outputDmaData *output_dma_object, struct pic_handle_t *pic_handle)
{
    int channel, retval;
    picWdmaData *device_data_wdma = NULL;
    picWdmaChannelData *device_data_channel = NULL;    
    picWdmaDeviceHandle *dh_wdma;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;
    struct pic_wdma_function_struct *ft_wdma;


    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma, 0, device_data_wdma,
                                          ft_wdma, dh_wdma);
    retval = ft_wdma->pic_revcheck(device_data_wdma, pic_handle);
    
    if (retval != 0)
        return retval;

    for (channel=0;channel<MAX_OUTPUT_DMAS;channel++)
    {
        PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                              device_data_channel, ft_channel, dh_channel);
        retval = ft_channel->pic_revcheck(device_data_channel, pic_handle);
        if (retval != 0)
            return retval;
    }
    return retval;
}

void output_dma_get_channel_status(outputDmaData *output_dma_object,
                                   struct pic_wdma_status_info *info,
                                   uint8_t channel)
{
    struct pic_wdma_channel_function_struct *ft_channel;
    picWdmaChannelDeviceHandle *dh_channel;    
    picWdmaChannelData *device_data_channel = NULL;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);

    ft_channel->get_status_wdma_channel(device_data_channel, info);
}

// return true if requested channel is busy, false if idle
bool output_dma_channel_status_busy(outputDmaData *output_dma_object,
                                    uint8_t channel)
{
    struct pic_wdma_status_info info;
    
    output_dma_get_channel_status(output_dma_object, &info, channel);
    if (info.dma_busy)
        return true;
    else
        return false;
}

bool output_dma_channel_is_enabled(outputDmaData *output_dma_object, uint8_t channel)
{
    struct pic_wdma_channel_function_struct *ft_channel;
    picWdmaChannelDeviceHandle *dh_channel;    
    picWdmaChannelData *device_data_channel = NULL;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);
    return ft_channel->is_enabled_wdma_channel(device_data_channel);
}

// enable requested wdma interrupts on the channel requested
// If structure pointer is NULL, enable or disable all interrupts
void enable_output_dma_irqs(outputDmaData *output_dma_object,
                            struct pic_output_dma_interrupt_info *irqstruct,
                            uint8_t channel, bool enable)
{
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);

    if (enable == false)
    {
        // The caller wants to disable interrupts.
        ft_channel->disable_wdma_channel_irqs(device_data_channel, irqstruct);
    }
    else
    {
        // the caller wants to enable interrupts
        ft_channel->enable_wdma_channel_irqs(device_data_channel, irqstruct);
    }
}

void handle_dma_channels_irqs(outputDmaData *output_dma_object)
{
    int i;
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;

    for (i=0;i<MAX_OUTPUT_DMAS;i++)
    {
        PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, i,
                                              device_data_channel, ft_channel, dh_channel);
        ft_channel->handle_wdma_channel_irqs(device_data_channel);
    }
}

void output_dma_channel_get_trans_len(outputDmaData *output_dma_object,
                                      uint32_t *trans_len, uint8_t channel)
{
    picWdmaChannelData *device_data_channel = NULL;
    picWdmaChannelDeviceHandle *dh_channel;    
    struct pic_wdma_channel_function_struct *ft_channel;

    PIC_ODMA_RETRIEVE_function_table_data(output_dma_object, wdma_channel, channel,
                                          device_data_channel, ft_channel, dh_channel);
    ft_channel->get_trans_len_wdma_channel(device_data_channel, trans_len);
}

void pic_get_output_dma_sizes_array(uint32_t *subblock_array,
                                    void *input_object)
{
    int x;
    outputDmaData *output_dma_object;

    output_dma_object = (outputDmaData *) input_object;

    if (output_dma_object->priv_data[wdma] != NULL)
        subblock_array[wdma_top_index] = sizeof(PIC_WDMA_REGS_t);

    for (x=0;x<MAX_OUTPUT_DMAS;x++)
    {
        if (output_dma_object->priv_data[wdma_channel + x] != NULL)
            subblock_array[wdma_dma0_index + x] = sizeof(PIC_WDMA_DMA0_REGS_t);
    }
}

struct output_dma_function_struct output_dma_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,
    
    .dump_channel_regs = dump_channel_regs,
    .dump_all_regs = dump_all_regs,

    .clear_output_dma_irqs = clear_output_dma_irqs,
    .start_output_dma = start_output_dma,
    .output_dma_get_channel_status = output_dma_get_channel_status,
    .output_dma_channel_status_busy = output_dma_channel_status_busy,
    .output_dma_channel_is_enabled = output_dma_channel_is_enabled,
    .output_dma_channel_get_trans_len = output_dma_channel_get_trans_len,
    .enable_output_dma_irqs = enable_output_dma_irqs,
    .handle_dma_channels_irqs = handle_dma_channels_irqs,
    .pic_get_output_dma_sizes_array = pic_get_output_dma_sizes_array,
// SANDRA, can we remove any of these?    
};

static struct output_dma_data *allocate_dma_object(int pic_instance)
{
    int i;
    struct output_dma_data *output_dma_object;

    output_dma_object = kmalloc(sizeof(struct output_dma_data), GFP_KERNEL);
    if (output_dma_object == NULL)
        return NULL;
    output_dma_object->device_data = kmalloc(sizeof(outputDmaData), GFP_KERNEL);
    if (output_dma_object->device_data == NULL)
        return NULL;
    output_dma_object->device_data->num_output_dma_instances = 0;
    output_dma_object->fcn_tbl = &output_dma_functions;
    // NULL out the pointers in the device data array to know who has properly registered
    for (i=0;i<MAX_OUTPUT_DMAS + 1;i++)  // 3 WDMA channels, and 1 WDMA block
        output_dma_object->device_data->priv_data[i] = NULL;

    return output_dma_object;
}

void pic_output_dma_init(int pic_instance)
{
    struct output_dma_data *output_dma_object;

    // create and register this pic instance of the object
    output_dma_object = allocate_dma_object(pic_instance);
    if (output_dma_object == NULL)
    {
        error_print("Error return from allocate_dma_object\n");
        return;
    }

    register_pic_subblock(output_dma, output_dma_object, pic_instance, 0);
}

void pic_output_dma_exit(int pic_instance)
{
    struct output_dma_data *output_dma_object;

    // undo what pic_output_dma_init did
    output_dma_object = unregister_pic_subblock(output_dma, pic_instance, 0);
    if (output_dma_object != NULL)
    {
        kfree(output_dma_object->device_data);
        kfree(output_dma_object);
    }
}

void register_pic_output_dma_device_data(enum output_dma_subblock wdmasubblock,
                                         void *device_data,
                                         int pic_instance, int WDMA_instance)
{
    struct output_dma_data *output_dma_object;

    output_dma_object = get_output_dma_device(pic_instance);
    // wdma is at index 0, wdma_channel at 1+WDMAinstance (0, 1, 2)
    output_dma_object->device_data->priv_data[wdmasubblock+WDMA_instance] = device_data;

#ifdef DURING_DEBUG
    debug_print("     output dma detail:registered wdma ");
    if (wdmasubblock == wdma_channel)
        debug_print("channel %d ",WDMA_instance);
    debug_print("instance %d with pic_output_dma.c\n", pic_instance);
#endif
}
EXPORT_SYMBOL(register_pic_output_dma_device_data);

void *unregister_pic_output_dma_device_data(enum output_dma_subblock wdmasubblock,
                                           int pic_instance, int WDMA_instance)
{
    struct output_dma_data *output_dma_object;
    void *device_private;
    
    output_dma_object = get_output_dma_device(pic_instance);
    device_private = output_dma_object->device_data->priv_data[wdmasubblock+WDMA_instance];
    output_dma_object->device_data->priv_data[wdmasubblock+WDMA_instance] = NULL;

#ifdef DURING_DEBUG
    debug_print("     output dma detail:unregistered wdma ");
    if (wdmasubblock == wdma_channel)
        debug_print("channel %d ",WDMA_instance);
    debug_print("instance %d with pic_output_dma.c\n", pic_instance);
#endif
    return device_private;
    
}
EXPORT_SYMBOL(unregister_pic_output_dma_device_data);
