/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t
#include <linux/io.h> // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>  // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_lrmargin.h"
#include "pic_lrmargin_if.h"
#include "pic_driverlib_if.h"

void dump_lrmargin_regs(picLrmarginData *device_data)
{
    print("LRMargin regs for pic %d lrmargin_instance %d\n",
           device_data->instance, device_data->submodinstance);
    print("LRCR=0x%08X    ", picLRmargRead(LRCR));
    print("LRMR0=0x%08X    ", picLRmargRead(LRMR0));
    print("LRMR1=0x%08X    ", picLRmargRead(LRMR1));
    print("LRMR2=0x%08X    ", picLRmargRead(LRMR2));    
    print("REV0=0x%X\n", picLRmargRead(REV0));
    print("REV1=0x%X\n", picLRmargRead(REV1));
}
EXPORT_SYMBOL(dump_lrmargin_regs);  // allow outsiders just for debug

static void pic_configure(picLrmarginData *device_data, struct pic_handle_t *pic_handle)
{
    
    PROTECT_REG_ACCESS;
    picLRmargWrite(LRCR, pic_handle->pic_lrmargin[device_data->submodinstance]->LRCR);
    picLRmargWrite(LRMR0, pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR0);
    picLRmargWrite(LRMR1, pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR1);
    picLRmargWrite(LRMR2, pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR2);
    // REV0, REV1 are read only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picLrmarginData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_lrmargin[device_data->submodinstance]->LRCR = picLRmargRead(LRCR);
    pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR0 = picLRmargRead(LRMR0);
    pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR1 = picLRmargRead(LRMR1);
    pic_handle->pic_lrmargin[device_data->submodinstance]->LRMR2 = picLRmargRead(LRMR2);
    pic_handle->pic_lrmargin[device_data->submodinstance]->REV0 = picLRmargRead(REV0);
    pic_handle->pic_lrmargin[device_data->submodinstance]->REV1 = picLRmargRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picLrmarginData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = picLRmargRead(LRCR);
    reg = LRMARGIN0_LRCR_BYPASS_REPLACE_VAL(reg, 1); // bypass to do a reset
    picLRmargWrite(LRCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picLrmarginData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = LRMARGIN0_REV0_MAJ_MASK_SHIFT(picLRmargRead(REV0));  // NOTE: margin0 and 1 are the same
    pich_rev = LRMARGIN0_REV0_MAJ_MASK_SHIFT(pic_handle->pic_lrmargin[device_data->submodinstance]->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_lrmargin_function_struct pic_lrmargin_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,
    
    .pic_dump_regs = dump_lrmargin_regs, 
};

void pic_lrmargin_init(picLrmarginData *device_data)
{
    lrMarginDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(lrMarginDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_lrmargin_functions;
    pic_device_handle->device_data = device_data;
    
    register_pic_subblock(lrmargin, pic_device_handle,
                          device_data->instance, device_data->submodinstance);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pic_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));    
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_lrmargin_init);

void pic_lrmargin_exit(picLrmarginData *device_data)
{
    lrMarginDeviceHandle *pic_device_handle;
    
    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(lrmargin, device_data->instance,
                                                device_data->submodinstance);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_lrmargin_exit);
