/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/**
 * \file pic_if.h
 *
 * \brief The main interface for the PIC module.
 *
 * A typical use of PIC API is through \link picPublicApi the public API \endlink. Once a pic_handle
 * is created, the values can be set either directly or with the help of
 * \link picConvenience pic convenience functions \endlink.
 * A call to configure will transfer the values in pic_handle to the pic block registers.
 */

#ifndef _PIC_IF_H_
#define _PIC_IF_H_


#ifndef __KERNEL__
#define dma_addr_t uint32_t
#endif

// pic WDMA DWOCR modes
#define PIC_OUT_DMA_DISALLOW_DATA 0
#define PIC_OUT_DMA_DISCARD_DATA 1  // for debugging - throw out data
#define PIC_OUT_DMA_PROCESS_DATA 2  // normal mode

// pic WDMA cfg input data width values
#define PIC_WDMA_CFG_IN_WIDTH_1_BIT 0
#define PIC_WDMA_CFG_IN_WIDTH_2_BIT 1
#define PIC_WDMA_CFG_IN_WIDTH_4_BIT 2
#define PIC_WDMA_CFG_IN_WIDTH_8_BIT 3
#define PIC_WDMA_CFG_IN_WIDTH_16_BIT 4
#define PIC_WDMA_CFG_IN_WIDTH_32_BIT 5
#define PIC_WDMA_CFG_IN_WIDTH_24_BIT 6

// pic WDMA cfg line_rev values
#define PIC_WDMA_CFG_LINE_NORM 0
#define PIC_WDMA_CFG_LINE_REV  1

// pic WDMA cfg enable dma values
#define PIC_WDMA_CFG_DISABLE 0
#define PIC_WDMA_CFG_ENABLE 1

// pic WDMA output routing
#define PIC_DMA_CHANNEL_CEVEN_0  0
#define PIC_DMA_CHANNEL_CEVEN_1  2
#define PIC_DMA_CHANNEL_CEVEN_2  4
#define PIC_DMA_CHANNEL_MEVEN   0
#define PIC_DMA_CHANNEL_CODD_0  1
#define PIC_DMA_CHANNEL_CODD_1  3
#define PIC_DMA_CHANNEL_CODD_2  5
#define PIC_DMA_CHANNEL_MODD    1

// pic WDMA channel burst length
#define PIC_WDMA_CFG_BURST_LEN_4_WORD 0
#define PIC_WDMA_CFG_BURST_LEN_8_WORD 1
#define PIC_WDMA_CFG_BURST_LEN_16_WORD 2

// interrupt and register structures - exported so that calling modules can
// interpret the parameter passed to the callback function (for ints) or
// interface to other functions


/* Number of 32-bit fields in PIC BDR LUT (1025 is NOT a typo -- the LUT
 * covers values from 0 to 1024 since part of its algorithm involves fetching
 * entry x+1).
 * LUT contains this many 8-bit values stored as 32-bit registers.
 */
#define PIC_BDR_LUT_SIZE   1025
#define PIC_BDR_BDRCR_NUMLUTS_ONE   0
#define PIC_BDR_BDRCR_NUMLUTS_THREE 1
#define PIC_BDR_BDRCR_LUTSEL_COLOR0MONO 0
#define PIC_BDR_BDRCR_LUTSEL_COLOR1 1
#define PIC_BDR_BDRCR_LUTSEL_COLOR2 2
// FIXME - do these need to move to pic_driver.h?

// PIC currently has a hardware limit of 16 chipgaps
#define PIC_CHIPGAP_MAX_GAPS       16

// Each chipgap can be a maximum of 15 pixels (4 bits)
#define PIC_CHIPGAP_MAX_GAP_PIXELS 15

// PIC LR Margin defines
#define PIC_LRMARGIN_NUM_BLOCKS 2
#define PIC_LRMARGIN_TOP        0
#define PIC_LRMARGIN_BOTTOM     1

// PRNU/DSNU (PD) values
#define PD_PON_NORM_DEFAULT 0x38
#define PD_PON_N_DEFAULT    0x0
#define PD_PON_N_10         0x0
#define PD_PON_N_11         0x1
#define PD_PON_N_12         0x2
#define PD_PON_N_13         0x3
#define PD_PON_N_14         0x4
#define PD_PON_N_15         0x5


struct pic_common_ints
{
    int instance;
    bool skew;
    bool skew_odma;
    bool af_wdma_common;
    bool af_dma2;
    bool af_dma1;
    bool af_dma0;
    bool wdma_common;
    bool wdma2;
    bool wdma1;
    bool wdma0;
    bool ps_esd;
    bool idma_2d;
    // NOTE: this is for debug only - use the fields in the structure to see if a bit is set!!!!
    uint32_t int_array;  
};

struct pic_output_dma_interrupt_info
{
    uint8_t pic_instance;
    uint8_t chan_num;
    bool bad_rresp;
    bool bad_bresp;
    bool soft_rst_cmpl;
    bool eoi;
    bool dir_err;
    bool chg_line_align_err;
    bool eol_align_err;
    bool eoi_align_err;
    bool eoi_err;
    bool xfer_end;
    bool own;
    // NOTE: this is for debug only - use the fields in the structure to see if a bit is set!!!!
    uint32_t int_array;  
};

// wdma_channel's status register
struct pic_wdma_status_info
{
    uint8_t bytes_waiting;
    uint8_t force_burst_busy;
    uint8_t dma_paused;
    uint8_t pause_after;
    uint8_t pause_before;
    uint8_t desc_own;
    uint8_t desc_eoi;
    uint8_t desc_soi;
    uint8_t reverse;
    uint8_t softreset;
    uint8_t full_cbuf;
    uint8_t empty_cbuf;
    uint8_t full_dbuf;
    uint8_t empty_dbuf;
    uint8_t packer_empty;
    uint8_t dma_busy;
    // NOTE: this is for debug only - use the field in the structure to see if a bit is set!!!!
    uint32_t debug_array;
};


// ****************************************************************************
// PRNU/DSNU IDMA_2D virtual register structures.
//    Note that these virtual register structures MAY or MAY NOT match physical
//    hardware. If not, the PIC driver will translate to the format needed by
//    the sub-block driver.
// ****************************************************************************

struct pic_idma2d_interrupt_info
{
    uint8_t  pic_instance;
    bool     eol;
    bool     rst;
    bool     eoi;
    bool     fin;
    bool     who;
    uint32_t irq_array;    // NOTE: this is for debug only!!!!
};

/**
 * \defgroup picDmaFuncs PIC DMA Related Functions
 * These functions are to facilitate DMA transfers and do not belong to the general public API.
 *  @{
 */

/**
 * \brief Clear all of pic's interrupts
 * \param pic_instance the pic instance
 */
void pic_do_clear_all_irqs(int pic_instance);

/**
 * \brief To trigger the transfer for the prnu/dsnu DMA
 * \param pic_instance
 * \param descriptor_phys the address of the dma descriptor
 */
void pic_start_idma2d_dma(uint8_t pic_instance, dma_addr_t descriptor_phys);

/**
 * \brief To trigger the DMA transfer
 * \param pic_instance
 * \param channel the dma channel
 * \param descriptor_phys the address of the dma descriptor
 */
void pic_start_output_dma(uint8_t pic_instance, uint8_t channel,
                          dma_addr_t descriptor_phys);

/**
 * \brief To check if a DMA channel is busy
 * \param pic_instance the pic instance
 * \param channel the dma channel
 * \return true if the channel is busy
 */
bool pic_output_dma_channel_busy(uint8_t pic_instance, uint8_t channel);

/**
 * \brief To check if an output DMA channel is enabled
 * \param pic_instance
 * \param channel the DMA channel
 * \return true if the channel is enabled
 */
bool pic_output_dma_channel_is_enabled(uint8_t pic_instance, uint8_t channel);

/**
 * \brief To get the output byte count of the dma channel
 * \param pic_instance
 * \param trans_len on return, store the transfer length
 * \param channel
 */
void pic_get_trans_len_output_dma_channel(uint8_t pic_instance, uint32_t *trans_len,
                                          uint8_t channel);

/**
 * \brief Do a reset on the output dma - legacy function - not for new code
 * \param pic_instance the pic instance
 */
void pic_odma_soft_reset(uint8_t pic_instance);

/**
 * \brief To enable pic common interrupts - legacy function - not for new code
 * \param pic_instance the pic instance
 * \param irqstruct defines which interrupt(s) to turn on
 */
void pic_enable_pic_common_irqs(uint8_t pic_instance,
                               struct pic_common_ints *irqstruct);

/**
 * \brief To disable pic common interrupts  - legacy function - not for new code
 * \param pic_instance the pic instance
 * \param irqstruct defines which interrupt(s) to turn off
 */
void pic_disable_pic_common_irqs(uint8_t pic_instance, struct pic_common_ints *irqstruct);

/**
 * \brief To enable the interrupts for the output dma  - legacy function - not for new code
 * \param pic_instance
 * \param ints defines what interrupts to enable
 * \param channel the dma channel
 */
void pic_enable_pic_output_dma_channel_irqs(uint8_t pic_instance,
                                            struct pic_output_dma_interrupt_info *irqstruct,
                                            uint8_t channel);

/**
 * \brief To disable the interrupts for the output dma - legacy function - not for new code
 * \param pic_instance
 * \param irqstruct defines what interrupts to disable
 * \param channel the dma channel
 */
void pic_disable_pic_output_dma_channel_irqs(uint8_t pic_instance,
                                             struct pic_output_dma_interrupt_info *irqstruct,
                                             uint8_t channel);

/** @} */

/**
 *  \defgroup picPublicApi PIC Public API Functions
 *  @{
 */


/**
 * \brief To set the pic block register values with what are defined in pic_handle
 *
 * \param pic_handle the values to be copied to the registers
 * \param instance the pic instance
 */
void pic_do_configure(struct pic_handle_t *pic_handle, int instance);


/**
 * \brief Retrieve the current register values and store them in pic_handle
 *
 * \param pic_handle the values to be copied to
 * \param instance the pic instance
 */
void pic_do_get_current(struct pic_handle_t *pic_handle, int instance);

/**
 * \brief free the pic_handle (created by pic_create_new_default_handle)
 *
 * \param pic_handle the handle to free
 */
void pic_do_free_handle(struct pic_handle_t *pic_handle);

/**
 * \brief reset the pic registers
 *
 * \param instance the pic instance
 */
void pic_do_reset(int instance);
/** @} */

typedef size_t (*pic_external_pd_lut_provider)(int pic_instance, dma_addr_t *lut);
void pic_set_external_pd_lut_provider(pic_external_pd_lut_provider pd_lut_provider);

/**
 * \brief Disable reset in pic common 
 * \param pic_instance the pic instance
 */
void pic_common_disable_soft_reset(int pic_instance);

/**
 * @defgroup picRegDump PIC Print Register Values
 * @{
 */

/**
 * \brief print out all register values
 * \param pic_instance the pic instance
 */
void pic_dump(uint8_t pic_instance);

/**
 * \brief print out common block register values
 * \param pic_instance the pic instance
 */
void pic_dump_common_regs(uint8_t pic_instance);

/**
 * \brief print out PRNU DSNU register values
 * \param pic_instance the pic instance
 */
void pic_pd_dump(uint8_t pic_instance);

/**
 * \brief print out input DMA register values
 * \param pic_instance the pic instance
 */
void pic_pd_idma_dump(uint8_t pic_instance);

/**
 * \brief print the left/right margin register values
 * \param pic_instance the pic_instance
 * \param lrmargin_instance the submodule instance
 */
void pic_lrmargin_dump(uint8_t pic_instance, uint8_t lrmargin_instance);

/**
 * \brief print out the chip gap register values
 * \param pic_instance the pic instance
 */
void pic_chipgap_drv_dump(uint8_t pic_instance);

/**
 * \brief print out the bit depth reduction register values
 * \param pic_instance the pic instance
 * \param dump_lut true to dump the LUT values
 */
void pic_bdr_dump(uint8_t pic_instance, bool dump_lut);

/**
 * \brief print out the output dma register values
 * \param pic_instance the pic instance
 */
void pic_dump_output_dma_regs(uint8_t pic_instance);

/** @} */

/**
 * \defgroup picIrqHandling PIC interrupt handling Related Functions
 * These functions are to allow the main irq (in pic_common.c) to request
 * that each subblock deal with their interrupts.  So these function do
 * not belong to the general public API.
 *  @{
 */
void pic_output_dma_channel_handle_irqs(uint8_t pic_instance);
void pic_pd_idma2d_handle_irqs(uint8_t pic_instance);


/** @} */

// Not a public function - used by pic_driverlib.c
int pic_get_subblock_sizes_array(uint32_t **subblock_parm);

#endif // _PIC_IF_H_

