/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t
#include <linux/io.h>         // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_idma2d.h"
#include "pic_idma2d_if.h"
#include "pic_driverlib_if.h"

static uint32_t clear_idma2d_irqs_uint32(picIdma2dData *device_data, uint32_t int_val)
{
    unsigned long flags;  
    uint32_t      reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = pic_idma2dRead(int_st);
    pic_idma2dWrite(int_cl, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;

    return reg;
}

// NOTE: This function runs in interrupt thread context - try to be brief
void pic_idma2d_handle_irqs(picIdma2dData *device_data)
{
    uint32_t int_val;
    struct   pic_idma2d_interrupt_info irqstruct;
        
    // Grab the pending irqs then ack them all
    int_val = clear_idma2d_irqs_uint32(device_data, 0xFFFFFFFF);

    // Anything pending?
    if (int_val != 0)
    {
        pic_idma2d_convert_uintarray_to_irqstruct(&irqstruct, int_val);
        irqstruct.pic_instance = device_data->instance;
        irqstruct.irq_array    = int_val;

        // if someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
}

void dump_idma2d_regs(picIdma2dData *device_data)
{
    print("PIC IDMA_2D dump for pic %d\n", device_data->instance);
    print("cfg        = 0x%X\n",  pic_idma2dRead(cfg)); 
    print("status     = 0x%X\n",  pic_idma2dRead(status)); 
    print("line_width = 0x%X\n",  pic_idma2dRead(line_width)); 
    print("int_en     = 0x%X\n",  pic_idma2dRead(int_en)); 
    print("int_st     = 0x%X\n",  pic_idma2dRead(int_st)); 
    print("desc_read  = 0x%X\n",  pic_idma2dRead(desc_read)); 
    print("xfer_len   = 0x%X\n",  pic_idma2dRead(xfer_length)); 
    print("xfer_addr  = 0x%X\n",  pic_idma2dRead(xfer_addr)); 
    print("ctrl_word  = 0x%X\n",  pic_idma2dRead(ctrl_word)); 
    print("REV0       = 0x%X\n",  pic_idma2dRead(REV0)); 
    print("REV1       = 0x%X\n",  pic_idma2dRead(REV1)); 
}
EXPORT_SYMBOL(dump_idma2d_regs);

static void pic_configure(picIdma2dData *device_data, struct pic_handle_t *pic_handle)
{
    // write the values except those related to interrupt (side effect)
    
    device_data->interrupt_callback = pic_handle->pic_idma2d_callback;
    device_data->interrupt_callback_data = pic_handle->pic_idma2d_callback_data;
    PROTECT_REG_ACCESS;
    pic_idma2dWrite(cfg, pic_handle->pic_idma_2d->cfg);
    // status is a read-only register
    pic_idma2dWrite(line_width, pic_handle->pic_idma_2d->line_width);
    pic_idma2dWrite(int_en, pic_handle->pic_idma_2d->int_en);
    // int_st is a read-only register
    pic_idma2dWrite(int_cl, pic_handle->pic_idma_2d->int_cl);
    pic_idma2dWrite(int_fo, pic_handle->pic_idma_2d->int_fo);
    // configure does NOT write to desc_write - it would start the dma going
    // Desc_read, xfer_length, xfer_addr, ctrl_word are read-only registers
    pic_idma2dWrite(reset, pic_handle->pic_idma_2d->reset);
    // REV0, REV1 are read-only registers
    UNPROTECT_REG_ACCESS;
}


static void pic_get_current(picIdma2dData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_idma_2d->cfg = pic_idma2dRead(cfg);
    pic_handle->pic_idma_2d->status = pic_idma2dRead(status);
    pic_handle->pic_idma_2d->line_width = pic_idma2dRead(line_width);
    pic_handle->pic_idma_2d->int_en = pic_idma2dRead(int_en);
    pic_handle->pic_idma_2d->int_st = pic_idma2dRead(int_st);
    // int_cl, int_fo, desc_write are write-only registers
    pic_handle->pic_idma_2d->desc_read = pic_idma2dRead(desc_read);
    pic_handle->pic_idma_2d->xfer_length = pic_idma2dRead(xfer_length);
    pic_handle->pic_idma_2d->xfer_addr = pic_idma2dRead(xfer_addr);
    pic_handle->pic_idma_2d->ctrl_word = pic_idma2dRead(ctrl_word);
    // reset is a write-only register
    pic_handle->pic_idma_2d->REV0 = pic_idma2dRead(REV0);
    pic_handle->pic_idma_2d->REV1 = pic_idma2dRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picIdma2dData *device_data)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    // reset is a write-only register
    reg = 0;
    reg = IDMA_2D_RESET_SOFT_RESET_REPLACE_VAL(reg, 1);
    pic_idma2dWrite(reset, reg);
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picIdma2dData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = IDMA_2D_REV0_MAJ_MASK_SHIFT(pic_idma2dRead(REV0));
    pich_rev = IDMA_2D_REV0_MAJ_MASK_SHIFT(pic_handle->pic_idma_2d->REV0);

    // pic_idma2d driver code should work with older ASIC revs
    if (rev0 > pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

// start the idma2d dma (prnu/dsnu's data in dma)
static void pic_idma2d_start_dma(picIdma2dData *device_data, dma_addr_t desc_write)
{
    PROTECT_REG_ACCESS;
    // desc_write is a write-only register
    pic_idma2dWrite(desc_write, desc_write);
    UNPROTECT_REG_ACCESS;
}

struct pic_idma2d_function_struct pic_idma2d_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,

    .pic_dump_regs = dump_idma2d_regs,

    .pic_start_idma2d_dma = pic_idma2d_start_dma,

    .pic_idma2d_handle_irqs    = pic_idma2d_handle_irqs,
};

void pic_idma2d_init(picIdma2dData *device_data)
{
    picIdma2dDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(picIdma2dDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_idma2d_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(idma2d, pic_device_handle, device_data->instance, 0);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
	// NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(pic_device_handle->device_data->int_spinlock));
    // pd has no interrupts associated with it, so no callback
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_idma2d_init);

void pic_idma2d_exit(picIdma2dData *device_data)
{
    picIdma2dDeviceHandle *pic_device_handle;
    
    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(idma2d, device_data->instance, 0);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_idma2d_exit);

